/*
 * The function here are used to increase "detail" in an image.  I do very
 * simple shading where every polygon is a solid color.  Thus the smaller
 * the polygons, the better the shading.  Each level of detailing splits
 * each individual triangle into 4 smaller ones.  Detail 0 is the base
 * image as read in from the data file, and thus it has a special function.
 *
 * My detailing algorithm lets me skip any number of intermediate detail
 * levels, but it creates duplicate points.  To remove these I hash all
 * points into a hash table as they are created (discarding duplicates),
 * and read them back out into an array when done.
 */

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <math.h>

#define	BUCKETS	10000
#define _ABS(x)         ((x) < 0 ? -(x) : (x))

#define dfactor 10

extern void doTriangles();
extern void replaceText();


extern Widget xscrollbar, yscrollbar, zscrollbar, dscrollbar;
extern Widget detailcommand[];
extern float view_dist;
extern float distance;
extern float Old_distance;
extern int	OrigNpts, OrigNpolys;
extern int	Npts, Npolys;
extern int	*OrigPolys, *AllPolys, *iptr;
extern int	*OrigPixels, *AllPixels;
extern float	*OrigProbs, *AllProbs;
extern float	*OrigPts, *AllPts, *fptr;
extern int	LowMem;
extern int	save_xt, save_yt, save_zt;
extern int	save_iter;
extern int	Changes;
extern double	Total_Rot[3][3];

int Iter = 0;


struct hash_rec {
	float x, y, z;
	int indx;
	struct hash_rec *next;
	struct hash_rec *hptr;
};

struct hash_rec *Hash[BUCKETS];
struct hash_rec Head;
struct hash_rec *Curr_ptr;
int Pt_cnt = 0;


/*
 * add_pt() adds a point to a hash table of all points.
 * no duplicate entries are allowed.
 * Returns the count of this point in the total count of all points.
 */
int
add_pt(x, y, z)
	float x, y, z;
{
	int i;
	int ret;
	int bucket, dup;
	struct hash_rec *hptr;
	struct hash_rec *t_hptr;

	bucket = (int)(x * (BUCKETS - 1));
	bucket = _ABS(bucket);
	if (bucket < 0)
	{
		bucket = 0;
	}
	if (bucket >= BUCKETS)
	{
		bucket = bucket % BUCKETS;
	}
	t_hptr = Hash[bucket];
	if (t_hptr == NULL)
	{
		Hash[bucket] = (struct hash_rec *)XtMalloc(sizeof(struct hash_rec));
		Hash[bucket]->x = x;
		Hash[bucket]->y = y;
		Hash[bucket]->z = z;
		Hash[bucket]->indx = Pt_cnt;
		Hash[bucket]->next = NULL;
		Hash[bucket]->hptr = NULL;
		Curr_ptr->next = Hash[bucket];
		Curr_ptr = Curr_ptr->next;
		ret = Pt_cnt;
		Pt_cnt++;
	}
	else
	{
		dup = 0;
		while(t_hptr->hptr != NULL)
		{
			if ((t_hptr->x == x)&&
				(t_hptr->y == y)&&
				(t_hptr->z == z))
			{
				dup = 1;
				break;
			}
			t_hptr = t_hptr->hptr;
		}
		if ((t_hptr->x == x)&&
			(t_hptr->y == y)&&
			(t_hptr->z == z))
		{
			dup = 1;
		}
		if (dup == 0)
		{
			t_hptr->hptr = (struct hash_rec *)
				XtMalloc(sizeof(struct hash_rec));
			t_hptr = t_hptr->hptr;
			t_hptr->x = x;
			t_hptr->y = y;
			t_hptr->z = z;
			t_hptr->indx = Pt_cnt;
			t_hptr->next = NULL;
			t_hptr->hptr = NULL;
			Curr_ptr->next = t_hptr;
			Curr_ptr = Curr_ptr->next;
			ret = Pt_cnt;
			Pt_cnt++;
		}
		else
		{
			ret = t_hptr->indx;
		}
	}
	return(ret);
}


/*
 * Detail the image by splitting triangles into 4 smaller triangles.
 * This itteratively details up or down to the selected level.
 * Exception:  There is a seperate routine for detailing to 0;
 */
void
Detail(w, client_data, call_data)
	Widget w;
	caddr_t client_data, call_data;
{
	char buf[128];
	int i, j, k;
	int xt, yt, zt, val, junk1, junk2, junk3;
	int scale, npts, ntri, tcnt;
	int Polycount;
	int a, b, c;
	int *index, *base;
	int *polys;
	int *pixels;
	float *probs;
	float ax, ay, az, bx, by, bz, cx, cy, cz;
	float px1, py1, pz1, px2, py2, pz2, px, py, pz;
	struct hash_rec *t_hptr;
	int iter;

	iter = (int)client_data;
/*
 * If we are already at this detail level, there is no reason to re-do the work
 */
if((save_iter != Iter)||(iter != Iter))
{
	XtFree(AllPts);
	XtFree(AllPolys);
	XtFree(AllProbs);
	XtFree(AllPixels);
	Head.next = NULL;
	Curr_ptr = &Head;
	Pt_cnt = 0;
	for (i=0; i<BUCKETS; i++)
	{
		Hash[i] = NULL;
	}
	scale = 2;
	ntri = 4;
	for (i=2; i<=iter; i++)
	{
		scale = scale * 2;
		ntri = ntri * 4;
	}
	npts = (scale + 1) * ((scale / 2) + 1);

	Polycount = OrigNpolys * ntri;
	polys = (int *)XtMalloc(Polycount * 3 * sizeof(int));
	probs = (float *)XtMalloc(Polycount * sizeof(float));
	pixels = (int *)XtMalloc(Polycount * sizeof(int));
	index = (int *)XtMalloc(npts * sizeof(int));

	for (i=0; i<OrigNpolys; i++)
	{
		iptr = (int *)(OrigPolys + (i * 3));
		/*
		 * a, b, and c are indecies into a list of floats to the
		 * x, y, and z triples of each corner of the triangle.
		 */
		a = *iptr;
		b = *(iptr + 1);
		c = *(iptr + 2);
		/*
		 * This is the first corner
		 */
		ax = *(OrigPts + (a * 3));
		ay = *(OrigPts + (a * 3) + 1);
		az = *(OrigPts + (a * 3) + 2);
		/*
		 * This is the second corner
		 */
		bx = *(OrigPts + (b * 3));
		by = *(OrigPts + (b * 3) + 1);
		bz = *(OrigPts + (b * 3) + 2);
		/*
		 * This is the third corner
		 */
		cx = *(OrigPts + (c * 3));
		cy = *(OrigPts + (c * 3) + 1);
		cz = *(OrigPts + (c * 3) + 2);
		/*
		 * add all points for iterating this triangle
		 * store their locations in the index array.
		 */
		iptr = index;
		for (j=0; j<=scale; j++)
		{
			px1 = ax + ((cx - ax) * j / scale);
			py1 = ay + ((cy - ay) * j / scale);
			pz1 = az + ((cz - az) * j / scale);
			px2 = bx + ((cx - bx) * j / scale);
			py2 = by + ((cy - by) * j / scale);
			pz2 = bz + ((cz - bz) * j / scale);
			for (k=0; k<=(scale - j); k++)
			{
				if ((scale - j) == 0)
				{
					px = px1;
					py = py1;
					pz = pz1;
				}
				else
				{
					px = px1 + ((px2 - px1) *
						k / (scale - j));
					py = py1 + ((py2 - py1) *
						k / (scale - j));
					pz = pz1 + ((pz2 - pz1) *
						k / (scale - j));
				}
				*iptr++ = add_pt(px, py, pz);
			}
		}
		/*
		 * Now we need to add all the new triangles.
		 */
		tcnt = 0;
		base = index;
		for (j=0; j<scale; j++)
		{
			for (k=0; k<(scale - j); k++)
			{
				fptr = (float *)(probs + (ntri * i) + tcnt);
				*fptr = *((float *)(OrigProbs + i));
				iptr = (int *)(pixels + (ntri * i) + tcnt);
				*iptr = *((int *)(OrigPixels + i));
				iptr = (int *)(polys +
					(((ntri * i) + tcnt) * 3));
				*iptr = *(base + k);
				*(iptr + 1) = *(base + k + 1);
				*(iptr + 2) = *(base + k + scale - j + 1);
				tcnt++;
				/*
				 * i this is not the last triangle in a row
				 * there is a second triangle
				 */
				if (k != (scale - j - 1))
				{
					fptr = (float *)(probs +
						(ntri * i) + tcnt);
					*fptr = *((float *)(OrigProbs + i));
					iptr = (int *)(pixels +
						(ntri * i) + tcnt);
					*iptr = *((int *)(OrigPixels + i));
					iptr = (int *)(polys +
						(((ntri * i) + tcnt) * 3));
					*iptr = *(base + k + scale - j + 1);
					*(iptr + 1) = *(base + k + 1);
					*(iptr + 2) = *(base + k + scale - j + 2);
					tcnt++;
				}
			}
			base = base + (scale + 1 - j);
		}
	}

	XtFree(index);
	AllPts = (float *)XtMalloc(Pt_cnt * 3 * sizeof(float));
	fptr = AllPts;
	Curr_ptr = Head.next;
	while (Curr_ptr != NULL)
	{
		*fptr++ = Curr_ptr->x;
		*fptr++ = Curr_ptr->y;
		*fptr++ = Curr_ptr->z;
		t_hptr = Curr_ptr;
		Curr_ptr = Curr_ptr->next;
		XtFree(t_hptr);
	}
	Npts = Pt_cnt;

	/*
	 * The new detailed points were generated from the original
	 * unrotated data.  Apply the cumulative rotation matrix to all
	 * points to get back to the current rotation.
	 */
	fptr = AllPts;
	for (i=0; i<Npts; i++)
	{
		float x, y, z;

		x = *fptr;
		y = *(fptr + 1);
		z = *(fptr + 2);
		*fptr = (float)((Total_Rot[0][0] * x) +
			 (Total_Rot[1][0] * y) +
			 (Total_Rot[2][0] * z));
		fptr++;
		*fptr = (float)((Total_Rot[0][1] * x) +
			 (Total_Rot[1][1] * y) +
			 (Total_Rot[2][1] * z));
		fptr++;
		*fptr = (float)((Total_Rot[0][2] * x) +
			 (Total_Rot[1][2] * y) +
			 (Total_Rot[2][2] * z));
		fptr++;
	}

	AllPixels = pixels;
	AllProbs = probs;
	AllPolys = polys;
	Npolys = Polycount;
}

	sprintf(buf, "  %1d  ", Iter);
	replaceText(detailcommand[Iter], buf);
	Iter = iter;
	sprintf(buf, " *%1d  ", Iter);
	replaceText(detailcommand[Iter], buf);

	/*
	 * Because Detail is the first thing called upon mapping
	 * if there have been changes, we need to read the scroll
	 * bars to find and applyu those changes.
	 */
	XmScrollBarGetValues(dscrollbar, &val, &junk1, &junk2, &junk3);
	distance = (((float)val / 100) * (dfactor + view_dist)) - 
		(view_dist - 0.00001);
	if (Changes)
	{
		XmScrollBarGetValues(xscrollbar, &xt, &junk1, &junk2, &junk3);
		xt = -xt;
		XmScrollBarGetValues(yscrollbar, &yt, &junk1, &junk2, &junk3);
		XmScrollBarGetValues(zscrollbar, &zt, &junk1, &junk2, &junk3);
		zt = -zt;

		xt = xt - save_xt;
		yt = yt - save_yt;
		zt = zt - save_zt;
	}
	else
	{
		xt = yt = zt = 0;
	}
	save_iter = -1;
	Old_distance = 100.0;
	doTriangles(xt, yt, zt, distance);
}


/*
 * This detail function restores the original polygon mesh.
 */
void
DetailZero(w, client_data, call_data)
	Widget w;
	caddr_t client_data, call_data;
{
	char buf[128];
	int i;
	int norefresh;
	int xt, yt, zt, val, junk1, junk2, junk3;
	int *iptr1, *iptr2;
	float *fptr1, *fptr2;

	norefresh = (int)client_data;
/*
 * If we are already at detail 0, we don't need to do this.
 * If we have low memory, there is only one set of points kept
 * so there is no original to copy from.
 */
if ((!LowMem)&&(save_iter != Iter))
{
	/*
	 * free current mesh
	 */
	XtFree(AllPts);
	XtFree(AllPolys);
	XtFree(AllProbs);
	/*
	 * Reset points to the original points list, but apply the
	 * cumulative rotattion matrix to rotate them to the current
	 * position.
	 */
	Npts = OrigNpts;
	Npolys = OrigNpolys;
	AllPts = (float *)XtMalloc(Npts * 3 * sizeof(float));
	fptr1 = AllPts;
	fptr2 = OrigPts;
	for (i=0; i<Npts; i++)
	{
		float x, y, z;

		x = *fptr2++;
		y = *fptr2++;
		z = *fptr2++;
		*fptr1 = (float)((Total_Rot[0][0] * x) +
			 (Total_Rot[1][0] * y) +
			 (Total_Rot[2][0] * z));
		fptr1++;
		*fptr1 = (float)((Total_Rot[0][1] * x) +
			 (Total_Rot[1][1] * y) +
			 (Total_Rot[2][1] * z));
		fptr1++;
		*fptr1 = (float)((Total_Rot[0][2] * x) +
			 (Total_Rot[1][2] * y) +
			 (Total_Rot[2][2] * z));
		fptr1++;
	}
	/*
	 * Reset polygons and their probrobilities to their original values
	 */
	AllProbs = (float *)XtMalloc(Npolys * sizeof(float));
	fptr1 = AllProbs;
	fptr2 = OrigProbs;
	for (i=0; i<Npolys; i++)
	{
		*fptr1++ = *fptr2++;
	}
	AllPixels = (int *)XtMalloc(Npolys * sizeof(int));
	iptr1 = AllPixels;
	iptr2 = OrigPixels;
	for (i=0; i<Npolys; i++)
	{
		*iptr1++ = *iptr2++;
	}
	AllPolys = (int *)XtMalloc(Npolys * 3 * sizeof(int));
	iptr1 = AllPolys;
	iptr2 = OrigPolys;
	for (i=0; i<Npolys; i++)
	{
		*iptr1++ = *iptr2++;
		*iptr1++ = *iptr2++;
		*iptr1++ = *iptr2++;
	}
}
	sprintf(buf, "  %1d  ", Iter);
	replaceText(detailcommand[Iter], buf);
	Iter = 0;
	sprintf(buf, " *%1d  ", Iter);
	replaceText(detailcommand[Iter], buf);

	/*
	 * Because Detail is the first thing called upon mapping
	 * if there have been changes, we need to read the scroll
	 * bars to find and applyu those changes.
	 */
	XmScrollBarGetValues(dscrollbar, &val, &junk1, &junk2, &junk3);
	distance = (((float)val / 100) * (dfactor + view_dist)) - 
		(view_dist - 0.00001);
	if (Changes)
	{
		XmScrollBarGetValues(xscrollbar, &xt, &junk1, &junk2, &junk3);
		xt = -xt;
		XmScrollBarGetValues(yscrollbar, &yt, &junk1, &junk2, &junk3);
		XmScrollBarGetValues(zscrollbar, &zt, &junk1, &junk2, &junk3);
		zt = -zt;

		xt = xt - save_xt;
		yt = yt - save_yt;
		zt = zt - save_zt;
	}
	else
	{
		xt = yt = zt = 0;
	}
	save_iter = -1;
	Old_distance = 100.0;
	if (norefresh != 1)
	{
		doTriangles(xt, yt, zt, distance);
	}
}


#ifdef notdef
void
NewPts()
{
	int i;
	float *fptr1;
	float *fptr2;

	fptr1 = AllPts;
	fptr2 = OrigPts;
	for (i=0; i<Npts; i++)
	{
		float x, y, z;

		x = *fptr2++;
		y = *fptr2++;
		z = *fptr2++;
		*fptr1 = (float)((Total_Rot[0][0] * x) +
			 (Total_Rot[1][0] * y) +
			 (Total_Rot[2][0] * z));
		fptr1++;
		*fptr1 = (float)((Total_Rot[0][1] * x) +
			 (Total_Rot[1][1] * y) +
			 (Total_Rot[2][1] * z));
		fptr1++;
		*fptr1 = (float)((Total_Rot[0][2] * x) +
			 (Total_Rot[1][2] * y) +
			 (Total_Rot[2][2] * z));
		fptr1++;
	}
}
#endif

