;; Functions for extending the character set and dealing with case tables.
;; Copyright (C) 1987, 1990 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.


;; Written by:
;; Howard Gayle
;; TN/ETX/TT/HL
;; Ericsson Telecom AB
;; S-126 25 Stockholm
;; Sweden
;; howard@ericsson.se
;; uunet!ericsson.se!howard
;; Phone: +46 8 719 5565
;; FAX  : +46 8 719 8439

(require 'text-mode)

(defun case-of (ch ct)
   "Return 'nocase if character CH is marked as caseless in
case table CT, 'lowercase for lower case, and 'uppercase for
upper case."
   (cond
      ((nocase-p ch ct) 'nocase)
      ((lower-p  ch ct) 'lowercase)
      (t                'uppercase)
   )
)

(defun describe-buffer-case-table ()
   "Describe the case table of the current buffer."
   (interactive)
   (describe-case-table (case-table))
)

(defun describe-case-table (ct)
   "Describe the given case table in a help buffer."
   (let* (
      	 (i 0) 	     	     ; First character in range.
	 (ic (case-of 0 ct)) ; Case of i.
	 (j 0) 	     	     ; Last character in range.
	 (jc ic)     	     ; Case of j.
	 (k 1) 	     	     ; Current character.
	 kc    	     	     ; Case of k.
      	 )
      (with-output-to-temp-buffer "*Help*"
	 (while (<= k 255)
	    (setq kc (case-of k ct))
	    (if (not (eq jc kc))
	       (progn
		  (describe-character i)
		  (if (not (= i j))
		     (progn
		     	(princ "..")
			(describe-character j)
		     )
		  )
		  (princ "\t")
		  (princ (symbol-name jc))
		  (princ "\n")
		  (setq i k)
		  (setq ic kc)
	       )
	    )
	    (if (= k 255)
	       (progn
		  (describe-character i)
		  (if (not (= i k))
		     (progn
		     	(princ "..")
			(describe-character k)
		     )
		  )
		  (princ "\t")
		  (princ (symbol-name kc))
		  (princ "\n")
	       )
	    )
	    (setq j k)
	    (setq jc kc)
	    (setq k (1+ k))
	 )
	 (print-help-return-message)
      )
   )
)

(defun describe-character (c)
   "Print character C readably."
   (cond
      ((= c ?\t) (princ "\\t"))
      ((= c ?\n) (princ "\\n"))
      (t (princ (char-to-string c)))
   )
)

(defun invert-case ()
   "Change the case of the character just after point."
   (interactive "*")
   (let  (
      	 (oc (following-char)) ; Old character.
      	 )
      (cond
      	 ((lower-p oc) (replace-char (upcase   oc)))
      	 ((upper-p oc) (replace-char (downcase oc)))
      )
   )
   (forward-char)
)

(defun standard-case-syntax-delims (l r)
   "Set the entries for characters L and R in standard-case-table,
standard-downcase-table, standard-upcase-table,
standard-syntax-table, and text-mode-syntax-table to indicate
left and right delimiters."
   (set-case-table-nocase l (standard-case-table))
   (set-case-table-nocase r (standard-case-table))
   (set-trans-table-to l l (standard-downcase-table))
   (set-trans-table-to r r (standard-downcase-table))
   (set-trans-table-to l l (standard-upcase-table))
   (set-trans-table-to r r (standard-upcase-table))
   (modify-syntax-entry l
      (concat "(" (char-to-string r) "  ") (standard-syntax-table))
   (modify-syntax-entry l
      (concat "(" (char-to-string r) "  ") text-mode-syntax-table)
   (modify-syntax-entry r
      (concat ")" (char-to-string l) "  ") (standard-syntax-table))
   (modify-syntax-entry r
      (concat ")" (char-to-string l) "  ") text-mode-syntax-table)
)

(defun standard-case-syntax-pair (uc lc)
   "Set the entries for characters UC and LC in
standard-case-table, standard-downcase-table,
standard-upcase-table, standard-case-fold-table, standard-syntax-table, and
text-mode-syntax-table to indicate an (uppercase, lowercase)
pair of letters."
   (set-case-table-pair lc uc (standard-case-table))
   (set-trans-table-to lc lc (standard-downcase-table))
   (set-trans-table-to uc lc (standard-downcase-table))
   (set-trans-table-to lc uc (standard-upcase-table))
   (set-trans-table-to uc uc (standard-upcase-table))
   (modify-syntax-entry lc "w   " (standard-syntax-table))
   (modify-syntax-entry lc "w   " text-mode-syntax-table)
   (modify-syntax-entry uc "w   " (standard-syntax-table))
   (modify-syntax-entry uc "w   " text-mode-syntax-table)
)

(defun standard-case-syntax-punct (c)
   "Set the entries for character C in standard-case-table,
standard-downcase-table, standard-upcase-table,
standard-syntax-table, and text-mode-syntax-table to indicate
punctuation."
   (set-case-table-nocase c (standard-case-table))
   (set-trans-table-to c c (standard-downcase-table))
   (set-trans-table-to c c (standard-upcase-table))
   (modify-syntax-entry c ".   " (standard-syntax-table))
   (modify-syntax-entry c ".   " text-mode-syntax-table)
)

(defun standard-case-syntax-symb (c)
   "Set the entries for character C in standard-case-table,
standard-downcase-table, standard-upcase-table,
standard-syntax-table, and text-mode-syntax-table to indicate a
symbol."
   (set-case-table-nocase c (standard-case-table))
   (set-trans-table-to c c (standard-downcase-table))
   (set-trans-table-to c c (standard-upcase-table))
   (modify-syntax-entry c "_   " (standard-syntax-table))
   (modify-syntax-entry c "_   " text-mode-syntax-table)
)

(defun standard-case-syntax-white (c)
   "Set the entries for character C in standard-case-table,
standard-downcase-table, standard-upcase-table,
standard-syntax-table, and text-mode-syntax-table to indicate
white space."
   (set-case-table-nocase c (standard-case-table))
   (set-trans-table-to c c (standard-downcase-table))
   (set-trans-table-to c c (standard-upcase-table))
   (modify-syntax-entry c "    " (standard-syntax-table))
   (modify-syntax-entry c "    " text-mode-syntax-table)
)

(defun standard-case-syntax-word (c)
   "Set the entries for character C in standard-case-table,
standard-downcase-table, standard-upcase-table,
standard-syntax-table, and text-mode-syntax-table to indicate a
word component."
   (set-case-table-nocase c (standard-case-table))
   (set-trans-table-to c c (standard-downcase-table))
   (set-trans-table-to c c (standard-upcase-table))
   (modify-syntax-entry c "w   " (standard-syntax-table))
   (modify-syntax-entry c "w   " text-mode-syntax-table)
)

(provide 'case-table)
