;; Functions for dealing with char tables.
;; Copyright (C) 1987 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.


;; Written by Howard Gayle.  See case-table.el for details.

(require 'case-table)

(defun buffer-ctl-arrow-off ()
   "Display control characters as \\ number in curent buffer.
Does not change existing windows."
   (interactive)
   (setq buffer-char-table (backslash-char-table))
)

(defun buffer-ctl-arrow-on ()
   "Display control characters as ^ character in curent buffer.
Does not change existing windows."
   (interactive)
   (setq buffer-char-table (ctl-arrow-char-table))
)

(defun ctl-arrow-off ()
   "Display control characters as \\ number in selected window."
   (interactive)
   (set-window-char-table (backslash-char-table))
)

(defun ctl-arrow-on ()
   "Display control characters as ^ character in selected window."
   (interactive)
   (set-window-char-table (ctl-arrow-char-table))
)

(defun default-ctl-arrow-off ()
   "By default, display control characters as \\ number."
   (interactive)
   (setq-default buffer-char-table (backslash-char-table))
)

(defun default-ctl-arrow-on ()
   "By default, display control characters as ^ character."
   (interactive)
   (setq-default buffer-char-table (ctl-arrow-char-table))
)

(defun describe-char-table (ct)
   "Describe the given char table in a help buffer."
   (let  (
      	 (i 0) ; Current character.
	 j     ; Rope index.
	 r     ; Rope.
	 )
      (with-output-to-temp-buffer "*Help*"
	 (princ "Frame glyf: ")
	 (prin1 (glyf-to-string (get-char-table-frameg ct)))
	 (princ "\nTruncation glyf: ")
	 (prin1 (glyf-to-string (get-char-table-truncg ct)))
	 (princ "\nWrap glyf: ")
	 (prin1 (glyf-to-string (get-char-table-wrapg ct)))
      	 (princ "\nSelective display character: ")
	 (describe-character (get-char-table-invisc ct))
	 (princ "\nSelective display rope: ")
	 (setq r (get-char-table-invisr ct))
	 (setq j 0)
	 (while (< j (length r))
	    (aset r j (glyf-to-string (aref r j)))
	    (setq j (1+ j))
	 )
	 (prin1 r)
	 (princ "\n\nCharacter ropes:\n")
	 (while (<= i 255)
	    (describe-character i)
	    (princ "\t")
	    (setq r (get-char-table-dispr ct i))
	    (setq j 0)
	    (while (< j (length r))
	       (aset r j (glyf-to-string (aref r j)))
	       (setq j (1+ j))
	    )
	    (prin1 r)
	    (princ "\n")
	    (setq i (1+ i))
	 )
	 (print-help-return-message)
      )
   )
)

(defun describe-window-char-table ()
   "Describe the char table of the selected window."
   (interactive)
   (describe-char-table (window-char-table (selected-window)))
)

(defun standard-chars-8bit (l h)
   "Display characters in the range [L, H] with their actual
values in backslash-char-table and ctl-arrow-char-table."
   (let	 (r)
      (while (<= l h)
      	 (setq r (vector (new-glyf (char-to-string l))))
	 (put-char-table-dispr (backslash-char-table) l r)
	 (put-char-table-dispr (ctl-arrow-char-table) l r)
	 (setq l (1+ l))
      )
      r
   )
)

(defun standard-char-ascii (c s)
   "Display character C with string S in
   backslash-char-table and ctl-arrow-char-table."
   (let	 ((r (string-to-rope s)))
      (put-char-table-dispr (backslash-char-table) c r)
      (put-char-table-dispr (ctl-arrow-char-table) c r)
   )
c
)

(defun standard-char-g1 (c sc)
   "Display character C as G1 character SC in
   backslash-char-table and ctl-arrow-char-table."
   (let	 ((r (vector (new-glyf (concat "\016" (char-to-string sc) "\017")))))
      (put-char-table-dispr (backslash-char-table) c r)
      (put-char-table-dispr (ctl-arrow-char-table) c r)
      r
   )
)

(defun string-to-rope (s)
   "Convert string S to a rope with 1 glyf for each character."
   (let* (
         (l (length s))
         (r (make-vector l nil)) ; The rope.
         (i 0)                   ; Index.
         )
      (while (/= i l)
         (aset r i (get-glyf (char-to-string (aref s i))))
         (setq i (1+ i))
      )
      r
   )
)

(defun toggle-ctl-arrow ()
   "Toggle display of control characters in selected window."
   (interactive)
   (if (eq (window-char-table) (ctl-arrow-char-table))
      (ctl-arrow-off)
      (ctl-arrow-on)
   )
)

(defun toggle-default-ctl-arrow ()
   "Toggle default display of control characters."
   (interactive)
   (if (eq (default-value 'buffer-char-table) (ctl-arrow-char-table))
      (default-ctl-arrow-off)
      (default-ctl-arrow-on)
   )
)

(provide 'char-table)
