;; Mail sending commands for Emacs.
;; Copyright (C) 1985, 1986, 1990 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.


;; Modified 1990 for 8-bit character support by Howard Gayle.
;; See case-table.el for details.


(provide 'sendmail)

(defvar vms-inet-prefix "MX%"
  "This is the prefix in VMS for Internet addresses.
This may be overriden if you define a variable vms-inet-prefix
with a string containing the name of your local prefix, like
\"INET%\" for example.")

(defvar vms-mail-fix-MX-bug t
  "MX has, in its current implementation (ver. 3.1A) a bug, which makes
if barf on addresses of the form '\"foo bar\" <foobar@domain.se>'.
The fix it to reduce such lines to 'foobar@domain.se'.
 When vms-mail-fix-MX-bug is non-nil, this done. If it is nil, the address
is left as is.")

;(defconst mail-self-blind nil
;  "Non-nil means insert BCC to self in messages to be sent.
;This is done when the message is initialized,
;so you can remove or alter the BCC field to override the default.")

;(defconst mail-interactive nil
;  "Non-nil means when sending a message wait for and display errors.
;nil means let mailer mail back a message to report errors.")

;(defconst mail-yank-ignored-headers
;   "^via:\\|^mail-from:\\|^origin:\\|^status:\\|^remailed\\|^received:\\|^message-id:\\|^summary-line:\\|^to:\\|^subject:\\|^in-reply-to:\\|^return-path:"
;   "Delete these headers from old message when it's inserted in a reply.")
;(defvar send-mail-function 'sendmail-send-it
;  "Function to call to send the current buffer as mail.
;The headers are be delimited by a line which is mail-header-separator"")

; really defined in loaddefs for emacs 17.17+
;(defvar mail-header-separator "--text follows this line--"
;  "*Line used to separate headers from text in messages being composed.")
; really defined in loaddefs for emacs 17.17+
;(defvar mail-archive-file-name nil
;  "*Name of file to write all outgoing messages in, or nil for none.")
; really defined in loaddefs for emacs 17.17+
(defvar mail-aliases t
  "Alias of mail address aliases,
or t meaning should be initialized from .mailrc.")

(defvar mail-default-reply-to nil
  "*Address to insert as default Reply-to field of outgoing messages.")

(defvar mail-abbrevs-loaded nil)
(defvar mail-mode-map nil)

(autoload 'build-mail-aliases "mailalias"
  "Read mail aliases from ~/.mailrc and set mail-aliases."
  nil)

(autoload 'expand-mail-aliases "mailalias"
  "Expand all mail aliases in suitable header fields found between BEG and END.
Suitable header fields are To, CC and BCC."
  nil)

(defun mail-setup (to subject in-reply-to cc replybuffer)
  (if (eq mail-aliases t)
      (progn
	(setq mail-aliases nil)
	(if (file-exists-p "~/.mailrc")
	    (build-mail-aliases))))
  (setq mail-reply-buffer replybuffer)
  (goto-char (point-min))
  (insert "To: ")
  (save-excursion
    (if to
	(progn
	  (insert to "\n")
	  ;;; Here removed code to extract names from within <...>
	  ;;; on the assumption that mail-strip-quoted-names
	  ;;; has been called and has done so.
	  (let ((fill-prefix "\t"))
	    (fill-region (point-min) (point-max))))
      (newline))
    (if cc
	(let ((opos (point))
	      (fill-prefix "\t"))
	  (insert "CC: " cc "\n")
	  (fill-region-as-paragraph opos (point-max))))
    (if in-reply-to
	(insert "In-reply-to: " in-reply-to "\n"))
    (insert "Subject: " (or subject "") "\n")
    (if mail-default-reply-to
	(insert "Reply-to: " mail-default-reply-to "\n"))
    (if mail-self-blind
	(insert "BCC: " (user-login-name) "\n"))
    (if mail-archive-file-name
	(insert "FCC: " mail-archive-file-name "\n"))
    (insert mail-header-separator "\n"))
  (if to (goto-char (point-max)))
  (or to subject in-reply-to
      (set-buffer-modified-p nil))
  (run-hooks 'mail-setup-hook))

(defun mail-mode ()
  "Major mode for editing mail to be sent.
Separate names of recipients (in To: and Cc: fields) with commas.
Like Text Mode but with these additional commands:
C-c C-s  mail-send (send the message)    C-c C-c  mail-send-and-exit
C-c C-f  move to a header field (and create it if there isn't):
	 C-c C-f C-t  move to To:	C-c C-f C-s  move to Subj:
	 C-c C-f C-b  move to BCC:	C-c C-f C-c  move to CC:
C-c C-w  mail-signature (insert ~/.signature at end).
C-c C-y  mail-yank-original (insert current message, in Rmail).
C-c C-q  mail-fill-yanked-message (fill what was yanked)."
  (interactive)
  (kill-all-local-variables)
  (make-local-variable 'mail-reply-buffer)
  (setq mail-reply-buffer nil)
  (set-syntax-table text-mode-syntax-table)
  (use-local-map mail-mode-map)
  (setq local-abbrev-table text-mode-abbrev-table)
  (setq major-mode 'mail-mode)
  (setq mode-name "Mail")
  (setq buffer-offer-save t)
  (make-local-variable 'paragraph-separate)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^" mail-header-separator
				"$\\|^[ \t]*[-_][-_][-_]+$\\|"
				paragraph-start))
  (setq paragraph-separate (concat "^" mail-header-separator
				   "$\\|^[ \t]*[-_][-_][-_]+$\\|"
				   paragraph-separate))
  (run-hooks 'text-mode-hook 'mail-mode-hook))

(if mail-mode-map
    nil
  (setq mail-mode-map (make-sparse-keymap))
  (define-key mail-mode-map "\C-c?" 'describe-mode)
  (define-key mail-mode-map "\C-c\C-f\C-t" 'mail-to)
  (define-key mail-mode-map "\C-c\C-f\C-b" 'mail-bcc)
  (define-key mail-mode-map "\C-c\C-f\C-c" 'mail-cc)
  (define-key mail-mode-map "\C-c\C-f\C-s" 'mail-subject)
  (define-key mail-mode-map "\C-c\C-w" 'mail-signature)		; who
  (define-key mail-mode-map "\C-c\C-y" 'mail-yank-original)
  (define-key mail-mode-map "\C-c\C-q" 'mail-fill-yanked-message)
  (define-key mail-mode-map "\C-c\C-c" 'mail-send-and-exit)
  (define-key mail-mode-map "\C-c\C-s" 'mail-send))

(defun mail-send-and-exit (arg)
  "Send message like mail-send, then, if no errors, exit from mail buffer.
Prefix arg means don't delete this window."
  (interactive "P")
  (mail-send)
  (bury-buffer (current-buffer))
  (if (and (not arg)
	   (not (one-window-p))
	   (save-excursion
	     (set-buffer (window-buffer (next-window (selected-window) 'not)))
	     (eq major-mode 'rmail-mode)))
      (delete-window)
    (switch-to-buffer (other-buffer (current-buffer)))))

(defun mail-send ()
  "Send the message in the current buffer.
If  mail-interactive  is non-nil, wait for success indication
or error messages, and inform user.
Otherwise any failure is reported in a message back to
the user from the mailer."
  (interactive)
  (run-hooks 'mail-send-hook)
  (message "Sending...")
  (funcall send-mail-function)
  (set-buffer-modified-p nil)
  (delete-auto-save-file-if-necessary)
  (message "Sending...done"))

(defun sendmail-send-it (&optional process)
  (let ((errbuf (if (or mail-interactive
			(eq system-type 'vax-vms))
		    (get-buffer-create "*sendmail errors*")
		  0))
	(tembuf (generate-new-buffer " sendmail temp"))
	(case-fold-search nil)
	delimline
	(there-were-errors nil)
	(mailbuf (current-buffer)))
    (unwind-protect
	(save-excursion
	  (set-buffer errbuf)
	  (setq buffer-undo-list t)
	  (erase-buffer)
	  (set-buffer tembuf)
	  (setq buffer-undo-list t)
	  (erase-buffer)
	  (insert-buffer-substring mailbuf)
	  (goto-char (point-max))
	  ;; require one newline at the end.
	  (or (= (preceding-char) ?\n)
	      (insert ?\n))
	  ;; Change header-delimiter to be what sendmail expects.
	  (goto-char (point-min))
	  (re-search-forward
	   (concat "^" (regexp-quote mail-header-separator) "\n"))
	  (replace-match "\n")
	  (backward-char 1)
	  (setq delimline (point-marker))
	  (if mail-aliases
	      (expand-mail-aliases (point-min) delimline))
	  (goto-char (point-min))
	  ;; ignore any blank lines in the header
	  (while (and (re-search-forward "\n\n\n*" delimline t)
		      (< (point) delimline))
	    (replace-match "\n"))
	  (let ((case-fold-search t))
	    ;; Find and handle any FCC fields.
	    (goto-char (point-min))
	    (if (re-search-forward "^FCC:" delimline t)
		(mail-do-fcc delimline))
	    (if (eq system-type 'vax-vms)
		(let (to-line
		      cc-line
		      subject-line)
		  (setq to-line (vms-mail-do-header "To" delimline t))
		  (setq cc-line (vms-mail-do-header "CC" delimline t))
		  (setq subject-line (vms-mail-do-header "Subject" delimline nil))
		  (delete-region (point-min)
				 (progn delimline (forward-line 1) (point)))
		  (goto-char (point-min))
		  ;; Do not be surprised over the DECK command. If you don't
		  ;; have that, the process will barf with a bad DCL command
		  ;; error. Instead, we do it the Un*x way... we simply end
		  ;; the buffer with a single dot, and tell DCL about it!
		  (insert ;; "$ DECK/DOLLARS=\".\"\n"
			  "SEND/NOEDIT/CC\n"
			  to-line ?\n
			  cc-line ?\n
			  subject-line ?\n)
		  (goto-char (point-max))
		  ;; (insert ".\n")
		  )
	      (progn
		;; If there is a From and no Sender, put it a Sender.
		(goto-char (point-min))
		(and (re-search-forward "^From:"  delimline t)
		     (not (save-excursion
			    (goto-char (point-min))
			    (re-search-forward "^Sender:" delimline t)))
		     (progn
		       (forward-line 1)
		       (insert "Sender: " (user-login-name) "\n")))
		;; don't send out a blank subject line
		(goto-char (point-min))
		(if (re-search-forward "^Subject:[ \t]*\n" delimline t)
		    (replace-match ""))
		(if mail-interactive
		    (save-excursion
		      (set-buffer errbuf)
		      (erase-buffer))))))
	  (if (eq system-type 'vax-vms)
	      (let ((process-connection-type t)
		    (process-tmp nil)
		    cnt tmp-marker
		    cmd-line to-line cc-line subj-line)
		(catch 'vms-mail-tag
		  (progn
		    (goto-char (point-min))
		    (setq cmd-line
			  (buffer-substring (point)
					    (progn (forward-line 1) (point))))
		    (setq to-line
			  (buffer-substring (point)
					    (progn (forward-line 1) (point))))
		    (setq cc-line
			  (buffer-substring (point)
					    (progn (forward-line 1) (point))))
		    (setq subj-line
			  (buffer-substring (point)
					    (progn (forward-line 1) (point))))
		    ;; The following test is to see if the file is really
		    ;; to be sent anywhere. Maybe the user only specified
		    ;; some FCC: lines and no To: or CC: lines?
		    (if (and (string-match "^[ \t]*\n" to-line)
			     (string-match "^[ \t]*\n" cc-line))
			(throw 'vms-mail-tag 1))
		    (setq tmp-marker (point))
		    (setq process-tmp
			  (if process
			      process
			    (start-process "mail-sender" errbuf "MAIL")))
		    (set-buffer errbuf)
		    (vms-mail-wait-for-prompt "MAIL> $")
		    (process-send-string process-tmp cmd-line)
		    (if (not (vms-mail-wait-for-prompt "To:[ \t]*$" "MAIL> $"))
			(throw 'vms-mail-tag nil))
		    (process-send-string process-tmp to-line)
		    (if (not (vms-mail-wait-for-prompt "CC:[ \t]*$" "MAIL> $"))
			(throw 'vms-mail-tag nil))
		    (process-send-string process-tmp cc-line)
		    (if (not (vms-mail-wait-for-prompt "Subj:[ \t]*$"
						       "MAIL> $"))
			(throw 'vms-mail-tag nil))
		    (process-send-string process-tmp subj-line)
		    (vms-mail-wait-for-prompt "Enter.*quit:$")
		    (set-buffer tembuf)
		    (process-send-region process-tmp tmp-marker (point-max))
		    (set-buffer errbuf)
		    (process-send-eof process-tmp)
		    (vms-mail-wait-for-prompt "MAIL> $")
		    ;; The following lines seem unnecessary at first, but
		    ;; you must keep in mind that VMS MAIL cleans up at exit,
		    ;; for example SYS$SCRATCH:MAIL_*_SEND.TMP
		    ;; And oh, waiting for the next line is no good either,
		    ;; Because you might get a message telling you that disk
		    ;; space is reclaimed, so we turn of translation mode and
		    ;; wait for the NUL which always precedes a prompt.
		    (set-process-translation-mode process-tmp nil)
		    (process-send-string  process-tmp "EXIT\r")
		    (vms-mail-wait-for-prompt (concat ".*" (char-to-string 0)))
		    ;; Now, I'm sure the DCL prompt is back...
		    ))
		(if process-tmp (kill-process process-tmp)))
	    (apply 'call-process-region
		   (append (list (point-min) (point-max)
				 (if (boundp 'sendmail-program)
				     sendmail-program
				   "/usr/lib/sendmail")
				 nil errbuf nil
				 "-oi" "-t")
			   ;; Always specify who from,
			   ;; since some systems have broken sendmails.
			   (list "-f" (user-login-name))
;;;			 ;; Don't say "from root" if running under su.
;;;			 (and (equal (user-real-login-name) "root")
;;;			      (list "-f" (user-login-name)))
			   ;; These mean "report errors by mail"
			   ;; and "deliver in background".
			   (if (null mail-interactive) '("-oem" "-odb")))))
	  (if (or mail-interactive
		  (eq system-type 'vax-vms))
	      (save-excursion
		(set-buffer errbuf)
		(if (eq system-type 'vax-vms)
		    (progn
		      (goto-char (point-min))
		      (while (search-forward "\n\n" nil t)
			(replace-match "\n"))
		      (goto-char (point-min))
		      (if (re-search-forward "^%" nil t)
			  (let ((win (get-buffer-window errbuf)))
			    (if win
				(delete-window win))
			    (delete-region (point-min) (match-beginning 0))
			    (goto-char (point-min))
			    (re-search-forward "^MAIL> $" nil t)
			    (delete-region (match-beginning 0) (point-max))
			    (goto-char (1- (point-max)))
			    (delete-char 1)
			    (setq mode-line-format
				  "-------------------------- Mail Errors %-")
			    (split-and-display errbuf)
			    (setq there-were-errors t)
			    (error "Errors while sending mail"))))
		  (progn
		    (goto-char (point-min))
		    (while (re-search-forward "\n\n* *" nil t)
		      (replace-match "; "))
		    (if (not (zerop (buffer-size)))
			(error "Sending...failed to %s"
			       (buffer-substring (point-min)
						 (point-max)))))))))
      (kill-buffer tembuf)
      (if (eq system-type 'vax-vms)
	  (if (not there-were-errors)
	      (let ((win (get-buffer-window errbuf)))
		(if win
		    (delete-window win))
		(kill-buffer errbuf)))
	(if (bufferp errbuf)
	    (kill-buffer errbuf))))))

;; i think i cribbed this from somewhere...  appt.el maybe?
(defun split-and-display (buf)
  "Splits the selected window in two and displays BUF in the lower half.
The new window will be no larger than is needed to display BUF, but not
larger than half of the size of the selected window.  If the selected window
is too small to split, BUF will be displayed in the selected buffer."
  (require 'electric)
  (condition-case nil
      (let* ((window-min-height 2)
	     (newwindow (split-window))
	     (orig-height (window-height newwindow)))
	(progn
	  (set-window-buffer newwindow errbuf)
	  (save-window-excursion
	    (shrink-window-if-larger-than-buffer newwindow)
	    (setq new-height (window-height newwindow)))
	  (enlarge-window (- orig-height new-height))))
    (args-out-of-range (set-window-buffer (selected-window) errbuf))))

(defun vms-process-quotes (line)
  (let ((result "")
	(tempend 0)
	tempstr)
    (while (string-match "\"" line tempend)
      (progn
	(setq tempstr (substring line tempend (match-beginning 0)))
	(setq tempend (match-end 0))
	(setq result (concat result tempstr "\"\""))))
    (concat result (substring line tempend))))
	
(defun vms-find-address-end (line &optional start)
  (let ((tempend start)
	(l (length line))
	(realend nil))
    (if (= start l)
	nil
      (while (string-match "[,\"(]" line tempend)
	(progn
	  (setq tempend (match-end 0))
	  (setq tempchr (elt line (match-beginning 0)))
	  (cond ((char-equal tempchr ?\")
		 (if (string-match "\\\"" line tempend)
		     (setq tempend (match-end 0))
		   (setq tempend l)))
		((char-equal tempchr ?\()
		 (if (string-match ")" line tempend)
		     (setq tempend (match-end 0))
		   (setq tempend l)))
		(t			; There's only the "," left, now!
		 (setq realend (match-beginning 0))
		 (setq tempend l)))))
      (if realend
	  (cons realend (1+ realend))
	(cons l l)))))

(defun vms-fix-MX-bug (addr)
    ;; The following piece of code takes away the unneeded parts from addresses
    ;; of the form 'phrase "<" local-user "@" domain ">"', because MX (which
    ;; we use at ttt.kth.se) barfs at such addresses (A bug, I mean, that's
    ;; RFC822-compliant addresses!!!)
    (if (string-match "^\\(\"[^\"]*\"\\|([^)]*)\\|[^<]\\)*<\\([^>]*\\)>[ \t]*$" addr)
	(substring addr (match-beginning 2) (match-end 2))
      addr))

(defun vms-process-one-address (addr)
  "Takes one address, and munges it until it is usable."
  (let (tempaddr)
    (cond ((string-match "^[ \t]*_?[A-Za-z]*%" addr)
	   ;; This is an address starting with foo% (whatever foo is)
	   addr)
	  ((string-match "^[ \t]*_?[A-Za-z0-9]*::" addr)
	   ;; This is mail to another DECnet node. Do not munge!
	   addr)
	  ((string-match "^[ \t]*@" addr)
	   ;; This most likely is a distribution file, so DO NOT MUNGE!
	   addr)
	  (t
	   ;; Now, we have probably found al the VMS specific cases, so now,
	   ;; we only have the Internet addresses to take care of
	   (if (string-match "@" addr)
	       (progn
		 (if vms-mail-fix-MX-bug (setq addr (vms-fix-MX-bug addr)))
		 (concat vms-inet-prefix "\"" (vms-process-quotes addr) "\""))
	     addr)))))

(defun vms-append-address (prefix addr)
  (if prefix (concat prefix "," addr) addr))

(defun vms-process-addresses (line)
  (let ((result nil)
	(tempstart 0)
	tempend)
    (while (setq tempend (vms-find-address-end line tempstart))
      (progn
	(setq result
	      (vms-append-address
	       result
	       (vms-process-one-address (substring line
						   tempstart
						   (car tempend)))))
	(setq tempstart (cdr tempend))))
    (if result result "")))
				       
(defun vms-mail-do-header (header header-end p)
  (let ((line "")
	(case-fold-search t))
    (goto-char (point-min))
    (while (re-search-forward (concat "^" header ":[ \t]*") header-end t)
      (setq line
	    (concat line (if (and p (not (string= line ""))) "," "")
		    (buffer-substring (point)
				      (progn
					(end-of-line)
					(skip-chars-backward " \t")
					(point)))))
      (while (progn
	       (end-of-line)
	       (and (looking-at "[ \t\n]*[ \t]+")
		    (< (match-end 0) header-end)))
	(goto-char (match-end 0))
	(setq line
	      (concat line
		      (buffer-substring (point)
					(progn
					  (end-of-line)
					  (skip-chars-backward " \t")
					  (point)))))))
    (if p (vms-process-addresses line) line)))

(defun mail-do-fcc (header-end)
  (let (fcc-list
	(rmailbuf (current-buffer))
	(timezone "")
	(tembuf (generate-new-buffer " rmail output"))
	(case-fold-search t))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward "^FCC:[ \t]*" header-end t)
	(setq fcc-list (cons (buffer-substring (point)
					       (progn
						 (end-of-line)
						 (skip-chars-backward " \t")
						 (point)))
			     fcc-list))
	(delete-region (match-beginning 0)
		       (progn (forward-line 1) (point))))
      (set-buffer tembuf)
      (erase-buffer)
      (if (not (eq system-type 'vax-vms))
	  (progn
	    (call-process "date" nil t nil)
	    (goto-char (point-min))
	    (re-search-forward 
	     "[0-9] \\([A-Za-z][A-Za-z ]*[A-Za-z]\\)[0-9 ]*$")
	    (setq timezone (buffer-substring (match-beginning 1) (match-end 1)))
	    (erase-buffer)))
      (insert "\nFrom " (user-login-name) " "
	      (current-time-string) "\n")
      ;; Insert the time zone before the year.
      (forward-char -1)
      (forward-word -1)
      (insert timezone " ")
      (goto-char (point-max))
      (insert-buffer-substring rmailbuf)
      ;; Make sure messages are separated.
      (goto-char (point-max))
      (insert ?\n)
      (goto-char 2)
      ;; ``Quote'' "^From " as ">From "
      ;;  (note that this isn't really quoting, as there is no requirement
      ;;   that "^[>]+From " be quoted in the same transparent way.)
      (let ((case-fold-search nil))
	(while (search-forward "\nFrom " nil t)
	  (forward-char -5)
	  (insert ?>)))
      (while fcc-list
	(let ((buffer (get-file-buffer (car fcc-list))))
	  (if buffer
	      ;; File is present in a buffer => append to that buffer.
	      (let ((curbuf (current-buffer))
		    (beg (point-min)) (end (point-max)))
		(save-excursion
		  (set-buffer buffer)
		  (goto-char (point-max))
		  (insert-buffer-substring curbuf beg end)))
	    ;; Else append to the file directly.
	    (write-region (point-min) (point-max) (car fcc-list) t)))
	(setq fcc-list (cdr fcc-list))))
    (kill-buffer tembuf)))

(defun mail-to ()
  "Move point to end of To-field."
  (interactive)
  (expand-abbrev)
  (mail-position-on-field "To"))

(defun mail-subject ()
  "Move point to end of Subject-field."
  (interactive)
  (expand-abbrev)
  (mail-position-on-field "Subject"))

(defun mail-cc ()
  "Move point to end of CC-field.  Create a CC field if none."
  (interactive)
  (expand-abbrev)
  (or (mail-position-on-field "cc" t)
      (progn (mail-position-on-field "to")
	     (insert "\nCC: "))))

(defun mail-bcc ()
  "Move point to end of BCC-field.  Create a BCC field if none."
  (interactive)
  (expand-abbrev)
  (or (mail-position-on-field "bcc" t)
      (progn (mail-position-on-field "to")
	     (insert "\nBCC: "))))

(defun mail-position-on-field (field &optional soft)
  (let (end
	(case-fold-search t))
    (goto-char (point-min))
    (re-search-forward
     (concat "^" (regexp-quote mail-header-separator) "\n"))
    (setq end (match-beginning 0))
    (goto-char (point-min))
    (if (re-search-forward (concat "^" (regexp-quote field) ":") end t)
	(progn
	  (re-search-forward "^[^ \t]" nil 'move)
	  (beginning-of-line)
	  (skip-chars-backward "\n")
	  t)
      (or soft
	  (progn (goto-char end)
		 (insert field ": \n")
		 (skip-chars-backward "\n")))
      nil)))

(defun mail-signature ()
  "Sign letter with contents of ~/.signature file."
  (interactive)
  (save-excursion
    (goto-char (point-max))
    (insert-file-contents (expand-file-name "~/.signature"))))

(defun mail-fill-yanked-message (&optional justifyp)
  "Fill the paragraphs of a message yanked into this one.
Numeric argument means justify as well."
  (interactive "P")
  (save-excursion
    (goto-char (point-min))
    (search-forward (concat "\n" mail-header-separator "\n") nil t)
    (fill-individual-paragraphs (point)
				(point-max)
				justifyp
				t)))
(defun mail-yank-original (arg)
  "Insert the message being replied to, if any (in rmail).
Puts point before the text and mark after.
Indents each nonblank line ARG spaces (default 3).
Just \\[universal-argument] as argument means don't indent
and don't delete any header fields."
  (interactive "P")
  (if mail-reply-buffer
      (let ((start (point)))
	(delete-windows-on mail-reply-buffer)
	(insert-buffer mail-reply-buffer)
	(if (consp arg)
	    nil
	  (mail-yank-clear-headers start (mark))
	  (indent-rigidly start (mark)
			  (if arg (prefix-numeric-value arg) 3)))
	(exchange-point-and-mark)
	(if (not (eolp)) (insert ?\n)))))

(defun mail-yank-clear-headers (start end)
  (save-excursion
    (goto-char start)
    (if (search-forward "\n\n" end t)
	(save-restriction
	  (narrow-to-region start (point))
	  (goto-char start)
	  (while (let ((case-fold-search t))
		   (re-search-forward mail-yank-ignored-headers nil t))
	    (beginning-of-line)
	    (delete-region (point)
			   (progn (re-search-forward "\n[^ \t]")
				  (forward-char -1)
				  (point))))))))

;;; Here are some VMS specific parts.
;;;
;;; Low-level routines
(defun vms-mail-stop-process (proc)
  (if proc
      (kill-process proc)))

(defun vms-mail-wait-for-prompt (prompt &optional prompt2 other)
  "Waits until the PROMPT (a regular expression) shows up at the bottom of the
current buffer.
If optional argument PROMPT2 is non-NIL, it is a regular expression, which is
searched for at the same time as PROMPT.
If optional argument OTHER is non-NIL, it is a regular expression, which is
searched for from the current point in the current buffer. This search is done
before the check for PROMPT and PROMPT2.
vms-mail-wait-for-prompt returns nil if PROMPT2 or OTHER is found, otherwise t.
This function is somewhat cryptic."
  (let ((p (point)))
    (not (or (if other
		 (progn
		   (goto-char p)
		   (re-search-forward other nil t)))
	     (let ((cnt1 0)
		   (cnt2 0)
		   (result nil))	; result will during execution contain
					; a list starting with the actual
					; result
	       (while
		   (progn
		     (goto-char (point-max))
		     (beginning-of-line)
		     (if (= cnt2 (point))
			 (if (or (looking-at prompt)
				 (and prompt2
				      (looking-at prompt2)))
			     (if (> (setq cnt1 (1+ cnt1)) 10)
				 (progn
				   (setq result
					 (list (and prompt2
						    (looking-at prompt2))))
				   nil)
			       t)
			   (setq cnt1 0) 
			   (sleep-for 1)
			   t)
		       (setq cnt2 (point))
		       (sleep-for 1)
		       t)))
	       (car result))))))

(defun vms-mail-end-of-output-p (proc)
  (goto-char (point-max))
  (beginning-of-line)
  (looking-at "MAIL> $"))

;; Put these last, to reduce chance of lossage from quitting in middle of loading the file.

(defun mail (&optional noerase to subject in-reply-to cc replybuffer)
  "Edit a message to be sent.  Argument means resume editing (don't erase).
Returns with message buffer selected; value t if message freshly initialized.
While editing message, type C-c C-c to send the message and exit.

Separate names of recipients with commas.

Various special commands starting with C-c are available in sendmail mode
to move to message header fields:
\\{mail-mode-map}

If mail-self-blind is non-nil, a BCC to yourself is inserted
when the message is initialized.

If mail-default-reply-to is non-nil, it should be an address (a string);
a Reply-to: field with that address is inserted.

If mail-archive-file-name is non-nil, an FCC field with that file name
is inserted.

If mail-setup-hook is bound, its value is run by means of run-hooks
after the message is initialized.  It can add more default fields.
See the documentation of run-hooks.

When calling from a program, the second through fifth arguments
 TO, SUBJECT, IN-REPLY-TO and CC specify if non-nil
 the initial contents of those header fields.
 These arguments should not have final newlines.
The sixth argument REPLYBUFFER is a buffer whose contents
 should be yanked if the user types C-c C-y."
  (interactive "P")
  (switch-to-buffer "*mail*")
  (setq default-directory (expand-file-name "~/"))
  (auto-save-mode auto-save-default)
  (mail-mode)
  (and (not noerase)
       (or (not (buffer-modified-p))
	   (y-or-n-p "Unsent message being composed; erase it? "))
       (progn (erase-buffer)
	      (mail-setup to subject in-reply-to cc replybuffer)
	      t)))

(defun mail-other-window (&optional noerase to subject in-reply-to cc replybuffer)
  "Like `mail' command, but display mail buffer in another window."
  (interactive "P")
  (let ((pop-up-windows t))
    (pop-to-buffer "*mail*"))
  (mail noerase to subject in-reply-to cc replybuffer))

;;; Do not add anything but external entries on this page.
