;; Functions for dealing with sort tables.
;; Copyright (C) 1987 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.


;; Written by Howard Gayle.  See case-table.el for details.

(require 'case-table)

(defun describe-case-distinct-table ()
   "Describe the case-distinct sort table of the current buffer."
   (interactive)
   (describe-sort-table (case-distinct-table))
)

(defun describe-case-fold-table ()
   "Describe the case-fold sort table of the current buffer."
   (interactive)
   (describe-sort-table (case-fold-table))
)

(defun describe-sort-table (st)
   "Describe the given sort table in a help buffer.  The
equivalence classes are listed one per line in increasing order."
   (let	 (
      	 e
      	 (i 0) 	     	      	   ; Current character.
	 j     	     	      	   ; Steps through EC.
      	 (v (make-vector 256 nil)) ; v[i] is EC containing char i.
      	 )
      (with-output-to-temp-buffer "*Help*"
	 (while (<= i 255)
	    (setq e (get-sort-table-ec-num i st))
	    (aset v e (get-sort-table-ec i st))
	    (setq i (1+ i))
	 )
	 (setq i 0)
	 (setq e (aref v i))
	 (while e
	    (setq j 0)
	    (while (< j (length e))
	       (describe-character (aref e j))
	       (setq j (1+ j))
	    )
	    (princ "\n")
	    (setq i (1+ i))
	    (setq e (aref v i))
	 )
	 (print-help-return-message)
      )
   )
)

(defun expand-sort-table-list (lst)
   "One argument: a list of elements in increasing order.  Each
element is either a single character, which represents a
singleton equivalence class, or a pair (lo . hi), which is
short for all single elements in the range lo .. hi, or a list
of characters, all in the same equivalence class.  Returns a
list suitable for make-sort-table.  Checks for errors."
   (let	 (
      	 c1 ; Current character.
      	 c2 ; Last character in dotted pair.
      	 ce ; Current element in lst.
	 (cvr (make-vector 256 nil)) ; Flag set when each char covered.
	 (p lst) ; Steps through lst.
	 q1 ; Steps through sublist.
      	 z ; Result.
      	 )
      (while p
      	 (setq ce (car p))
	 (cond
	    ((numberp ce)
	       (if (aref cvr ce)
	       	  (message "Attempt to redefine %c (%d)" ce ce)
		  (setq z (cons (list ce) z))
		  (aset cvr ce t)
	       ))
	    ((numberp (cdr ce))
	       (setq c1 (car ce))
	       (setq c2 (cdr ce))
	       (while (<= c1 c2)
		  (if (aref cvr c1)
		     (message "Attempt to redefine %c (%d)" c1 c1)
		     (setq z (cons (list c1) z))
		     (aset cvr c1 t)
		     (setq c1 (1+ c1))
		  )
	       ))
	    (t
	       (setq q1 ce)
	       (while q1
		  (setq c1 (car q1))
		  (if (aref cvr c1)
		     (message "Attempt to redefine %c (%d)" c1 c1)
		     (aset cvr c1 t)
		  )
		  (setq q1 (cdr q1))
	       )
	       (setq z (cons ce z))
	    )
	 )
      	 (setq p (cdr p))
      )
      (setq c1 0)
      (while (<= c1 255)
      	 (if (null (aref cvr c1))
	    (progn
	       (message "Character %c (%d) uncovered" c1 c1)
	       (sit-for 1)
	    )
	 )
      	 (setq c1 (1+ c1))
      )
      (reverse z)
   )
)

(defun new-sort-table (lst)
   "Return a new sort table.  Argument same as for
expand-sort-table-list."
   (make-sort-table (expand-sort-table-list lst))
)

(provide 'sort-table)
