;;; Set up Facit 4440 (Twist) terminal.

;; Map Twist function key escape sequences
;; into the standard slots in function-keymap.

(require 'keypad)

(keypad-default "p" 'redraw-screen-72-lines)
(keypad-default "q" 'redraw-screen-24-lines)

(defvar CSI-map nil
  "The CSI-map maps the CSI function keys on the Twist keyboard.
The CSI keys are the arrow keys.")

(if (not CSI-map)
    (progn
     (setq CSI-map (lookup-key global-map "\e["))
     (if (not (keymapp CSI-map))
	 (setq CSI-map (make-sparse-keymap)))  ;; <ESC>[ commands
     (setup-terminal-keymap CSI-map '(
        ("A"   . ?u) ; up arrow
	("B"   . ?d) ; down-arrow
	("C"   . ?r) ; right-arrow
	("D"   . ?l) ; left-arrow
	("H"   . ?h) ; home
	("J"   . ?C) ; shift-erase = clear screen
	("K"   . ?c) ; erase
	("L"   . ?A) ; insert line
	("M"   . ?L) ; delete line
	("P"   . ?D) ; delete character
	("U"   . ?N) ; shift-down-arrow = next page
	("V"   . ?P) ; shift-up-arrow = previous page
	("X"   . ?H) ; shift-home = home-down
	("Z"   . ?b) ; tabulation backward
	("4h"  . ?I) ; insert character
	("?Ln" . ?q) ; landscape mode
	("?Pn" . ?p) ; portrait mode
))))

(defun enable-arrow-keys ()
  "Enable the use of the Twist arrow keys for cursor motion.
Because of the nature of the Twist, this unavoidably breaks
the standard Emacs command ESC [; therefore, it is not done by default,
but only if you give this command."
  (interactive)
  (global-set-key "\e[" CSI-map)
  (send-string-to-terminal "\e[?1n") ; Landscape or portrait?
)

(defvar SS3a-map nil
  "SS3a-map maps the SS3 function keys on the Twist keyboard.
The SS3 keys are the numeric keypad keys in keypad application mode
\(DECKPAM).  SS3 is DEC's name for the sequence <ESC>O which is
the common prefix of what these keys transmit.")

(if (not SS3a-map)
    (progn
     (setq SS3a-map (lookup-key global-map "\eO"))
     (if (not (keymapp SS3a-map))
	 (setq SS3a-map (make-keymap)))  ;; <ESC>O commands
     (setup-terminal-keymap SS3a-map
	    '(("A" . ?u)	   ; up arrow
	      ("B" . ?d)	   ; down-arrow
	      ("C" . ?r)	   ; right-arrow
	      ("D" . ?l)	   ; left-arrow
	      ("M" . ?e)	   ; Enter
	      ("P" . ?\C-a)	   ; PF1
	      ("Q" . ?\C-b)	   ; PF2
	      ("R" . ?\C-c)	   ; PF3
	      ("S" . ?\C-d)	   ; PF4
	      ("l" . ?,)	   ; ,
	      ("m" . ?-)	   ; -
	      ("n" . ?.)	   ; .
	      ("p" . ?0)	   ; 0
	      ("q" . ?1)	   ; 1
	      ("r" . ?2)	   ; 2
	      ("s" . ?3)	   ; 3
	      ("t" . ?4)	   ; 4
	      ("u" . ?5)	   ; 5
	      ("v" . ?6)	   ; 6
	      ("w" . ?7)	   ; 7
	      ("x" . ?8)	   ; 8
	      ("y" . ?9)))))       ; 9

(defun keypad-application-mode ()
  "Switch on keypad application mode."
  (interactive)
  (send-string-to-terminal "\e=")
  (global-set-key "\eO" SS3a-map))

(defvar SS3n-map nil
  "SS3n-map maps the SS3 function keys on the Twist keyboard.
The SS3 keys are the numeric keypad keys in keypad numeric mode
\(DECKPAM).  SS3 is DEC's name for the sequence <ESC>O which is
the common prefix of what these keys transmit.")

(if (not SS3n-map)
    (progn
     (setq SS3n-map (lookup-key global-map "\eO"))
     (if (not (keymapp SS3n-map))
	 (setq SS3n-map (make-sparse-keymap)))  ;; <ESC>O commands
     (setup-terminal-keymap SS3n-map '(
	      ("P" . ?\C-a)	   ; PF1
	      ("Q" . ?\C-b)	   ; PF2
	      ("R" . ?\C-c)	   ; PF3
	      ("S" . ?\C-d)	   ; PF4
     ))
    (global-set-key "\eO" SS3n-map)
))

(if (fboundp 'get-glyf)
   (progn
      (require 'iso8859-1-swedish)
      (require 'char-table-vt100)
      (send-string-to-terminal "\e)B\e)1") ; Select Swedish letters as G1 set.
      (standard-char-underline 170 ?a) ; ordinal indicator, feminine
      (standard-char-graphic   176 125); degree sign
      (standard-char-graphic   177 ?~) ; plus or minus sign
      (standard-char-graphic   183 ?g) ; middle dot
      (standard-char-underline 186 ?o) ; ordinal indicator, masculine
      (standard-frameg-graphic ?x)     ; Vertical bar.
      (standard-truncg-graphic ?t)     ; Left T.
      (standard-wrapg-graphic  ?k)     ; Upper right corner.
   )
)

(defun redraw-screen-24-lines ()
   "This function is intended for use with Facit Twist terminals.
It should be bound to \"C-[[?Ln\", which is what the terminal transmits
when it is twisted into landscape mode.  The terminal must also have
the Display Report switch (custom setup 4, group 1, switch 3) set to
Auto."
   (interactive)
   (set-screen-height 24))
   
(defun redraw-screen-72-lines ()
   "This function is intended for use with Facit Twist terminals.
It should be bound to \"C-[[?Pn\", which is what the terminal transmits
when it is twisted into portrait mode.  The terminal must also have
the Display Report switch (custom setup 4, group 1, switch 3) set to
Auto."
   (interactive)
   (send-string-to-terminal "\e[r")
   (set-screen-height 72))

(setq start-emphasis "\e[4m") ; Underline on.
(setq stop-emphasis  "\e[m")  ; Underline off.
