/* GNU Emacs routines to deal with case tables.
   Copyright (C) 1987 Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU Emacs General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU Emacs, but only under the conditions described in the
GNU Emacs General Public License.   A copy of this license is
supposed to have been given to you along with GNU Emacs so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/* Written by Howard Gayle.  See chartab.c for details. */

#include "config.h"
#include "lisp.h"
#include "buffer.h"
#include "casetab.h"
#include "etctab.h"

Lisp_Object Qcase_table_p;
DEFUN ("case-table-p", Fcase_table_p, Scase_table_p, 1, 1, 0,
   "Return t iff ARG is a case table.")
(obj)
Lisp_Object obj;
{
return ((XTYPE (obj) == Lisp_Casetab) ? Qt : Qnil);
}

static Lisp_Object
check_case_table (obj)
Lisp_Object obj;
{
register Lisp_Object tem;

while (tem = Fcase_table_p (obj), NULL (tem))
   obj = wrong_type_argument (Qcase_table_p, obj, 0);
return (obj);
}   

/* Convert the given Lisp_Casetab to a Lisp_Object. */
static Lisp_Object
enlisp_case_table (sp)
struct Lisp_Casetab *sp;
{
register Lisp_Object z; /* Return. */

XSET (z, Lisp_Casetab, sp);
return (z);
}

DEFUN ("case-table", Fcase_table, Scase_table, 0, 0, 0,
   "Return the case table of the current buffer.")
()
{
return (enlisp_case_table (current_buffer->case_table_v));
}

DEFUN ("standard-case-table", Fstandard_case_table,
   Sstandard_case_table, 0, 0, 0,
   "Return the standard case table.\n\
This is the one used for new buffers.")
()
{
return (enlisp_case_table (buffer_defaults.case_table_v));
}

/* Extract the case table from the given Lisp object.  Check for errors. */
static struct Lisp_Casetab *
get_case_table_arg (obj)
register Lisp_Object obj;
{
if (NULL (obj)) return (current_buffer->case_table_v);
obj = check_case_table (obj);
return (XCASETAB (obj));
}

/* Store a case table.  Check for errors. */
static Lisp_Object
set_case_table (p, t)
struct Lisp_Casetab **p; /* Points to where to store the case table. */
register Lisp_Object  t; /* The case table as a Lisp object. */
{
t = check_case_table (t);
*p = XCASETAB (t);
return (t);
}

DEFUN ("set-case-table", Fset_case_table, Sset_case_table, 1, 1, 0,
   "Select a new case table for the current buffer.\n\
One argument, a case table.")
(table)
Lisp_Object table;
{
return (set_case_table (&current_buffer->case_table_v, table));
}

DEFUN ("set-standard-case-table",
   Fset_standard_case_table, Sset_standard_case_table, 1, 1, 0,
   "Select a new standard case table.  This does not change the\n\
case tables of any existing buffers.  One argument, a case table.")
(table)
Lisp_Object table;
{
return (set_case_table (&buffer_defaults.case_table_v, table));
}

DEFUN ("make-case-table", Fmake_case_table, Smake_case_table, 0, 0, 0,
   "Make a new case table.  All characters are caseless.")
()
{
register struct Lisp_Casetab *nt; /* New case table. */
register int   	     	      i;
register Lisp_Object 	      z;  /* Return. */

z = make_etc_table (sizeof (struct Lisp_Casetab), Lisp_Casetab);
nt = XCASETAB (z);
for (i = 0; i <= 255; ++i)
   nt->cas_case[i] = nocase_e;
return (z);
}

DEFUN ("nocase-p", Fnocase_p, Snocase_p, 1, 2, 0,
   "Return t iff character CHAR is caseless, according to case\n\
table TABLE.")
(ch, table)
Lisp_Object ch;
Lisp_Object table;
{
return (CASETAB_ISNOCASE (get_char_arg (ch), get_case_table_arg (table))
        ? Qt : Qnil);
}

DEFUN ("lower-p", Flower_p, Slower_p, 1, 2, 0,
   "Return t iff character CHAR is lower case, according to case\n\
table TABLE (default (case-table)).")
(ch, table)
Lisp_Object ch;
Lisp_Object table;
{
return (CASETAB_ISLOWER (get_char_arg (ch), get_case_table_arg (table))
        ? Qt : Qnil);
}

DEFUN ("upper-p", Fupper_p, Supper_p, 1, 2, 0,
   "Return t iff character CHAR is upper case, according to case\n\
table TABLE (default (case-table)).")
(ch, table)
Lisp_Object ch;
Lisp_Object table;
{
return (CASETAB_ISUPPER (get_char_arg (ch), get_case_table_arg (table))
        ? Qt : Qnil);
}

DEFUN ("set-case-table-nocase",
   Fset_case_table_nocase, Sset_case_table_nocase, 1, 2, 0,
   "Mark character CHAR as caseless in case table TABLE\n\
(default (case-table)).")
(ch, table)
Lisp_Object ch;
Lisp_Object table;
{
get_case_table_arg (table)->cas_case[get_char_arg (ch)] = nocase_e;
return (ch);
}

DEFUN ("set-case-table-pair",
   Fset_case_table_pair, Sset_case_table_pair, 2, 3, 0,
   "Mark characters LC and UC as an (upper case, lower case)\n\
pair in case table TABLE (default (case-table)).")
(lc, uc, table)
Lisp_Object lc;
Lisp_Object uc;
Lisp_Object table;
{
register struct Lisp_Casetab *cp = get_case_table_arg (table);
register char_t lch = get_char_arg (lc);
register char_t uch = get_char_arg (uc);

cp->cas_case[lch] = lowercase_e;
cp->cas_case[uch] = uppercase_e;
return (lc);
}

init_case_table_once ()
{
register int i;
register case_t *p;

Fset_standard_case_table (Fmake_case_table ());
p = buffer_defaults.case_table_v->cas_case;
for (i = 'A'; i <= 'Z'; ++i)
   p[i] = uppercase_e;
for (i = 'a'; i <= 'z'; ++i)
   p[i] = lowercase_e;
}

syms_of_case_table ()
{
Qcase_table_p = intern ("case-table-p");
staticpro (&Qcase_table_p);

defsubr (&Scase_table_p);
defsubr (&Scase_table);
defsubr (&Sstandard_case_table);
defsubr (&Sset_case_table);
defsubr (&Sset_standard_case_table);
defsubr (&Smake_case_table);
defsubr (&Snocase_p);
defsubr (&Slower_p);
defsubr (&Supper_p);
defsubr (&Sset_case_table_nocase);
defsubr (&Sset_case_table_pair);
}
