/* Lisp functions for making directory listings.
   Copyright (C) 1985, 1986, 1991 Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


/* Modified 1991 for 8-bit character support by Howard Gayle.
 *  See chartab.c for details. */

/* Modified 1992 to complete directories on VMS ---- Richard Levitte */


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef VMS
#include <ssdef.h>
#include <rmsdef.h>
#include <stdlib.h>
#endif

#include "config.h"

#ifdef SYSV_SYSTEM_DIR

#include <dirent.h>
#define DIRENTRY struct dirent
#define NAMLEN(p) strlen (p->d_name)

#else

#ifdef NONSYSTEM_DIR_LIBRARY
#include "ndir.h"
#else /* not NONSYSTEM_DIR_LIBRARY */
#include <sys/dir.h>
#endif /* not NONSYSTEM_DIR_LIBRARY */

#define DIRENTRY struct direct
#define NAMLEN(p) p->d_namlen

extern DIR *opendir ();
#include "filetypes.h"
extern struct direct *readdir ();

#endif

#ifdef NULL
#undef NULL
#endif

#include "lisp.h"
#include "buffer.h"
#include "commands.h"

#include "regex.h"
#include "sorttab.h"

#define min(a, b) ((a) < (b) ? (a) : (b))

/* if system does not have symbolic links, it does not have lstat.
   In that case, use ordinary stat instead.  */

#ifndef S_IFLNK
#define lstat stat
#endif
#ifdef VMS
#include <descrip.h>
#define DESCRIPTOR(name,string,size) \
    struct dsc$descriptor name = { size, DSC$K_DTYPE_T, DSC$K_CLASS_S, string }
#endif /* VMS */

extern int completion_ignore_case;

Lisp_Object Vcompletion_ignored_extensions;

Lisp_Object Qcompletion_ignore_case;
#ifdef VMS
int vms_write_file_where_read_from;
#endif /* VMS */

DEFUN ("directory-files", Fdirectory_files, Sdirectory_files, 1, 3, 0,
  "Return a list of names of files in DIRECTORY.\n\
If FULL is non-NIL, absolute pathnames of the files are returned.\n\
If MATCH is non-NIL, only pathnames containing that regexp are returned.")
  (dirname, full, match)
     Lisp_Object dirname, full, match;
{
  DIR *d;
  char slashfilename[MAXNAMLEN+2];
  char *filename = slashfilename;
  int length;
  Lisp_Object list, name;

  /* In search.c */
  extern struct re_pattern_buffer searchbuf;

  if (!NULL (match))
    {
      CHECK_STRING (match, 3);
      /* Compile it now so we don't get an error after opendir */
#ifdef VMS
      compile_pattern (match, &searchbuf, buffer_defaults.case_fold_table_v);
#else
      compile_pattern (match, &searchbuf, NULL_SORT_TABLE);
#endif /* VMS */
    }

  dirname = Fexpand_file_name (dirname, Qnil);
  if (!(d = opendir (XSTRING (Fdirectory_file_name (dirname))->data)))
    report_file_error ("Opening directory", Fcons (dirname, Qnil));

  list = Qnil;
  length = XSTRING (dirname)->size;
#ifndef VMS
  if (length == 0   ||  XSTRING (dirname)->data[length - 1] != '/')
    *filename++ = '/';
#endif /* !VMS */

  /* Loop reading blocks */
  while (1)
    {
      DIRENTRY *dp = readdir (d);
      int len;

      if (!dp) break;
      len = NAMLEN (dp);
      if (dp->d_ino)
	{
	  strncpy (filename, dp->d_name, len);
	  filename[len] = 0;
	  if (NULL (match) ||
	      (0 <= re_search (&searchbuf, filename, len, 0, len, 0)))
	    {
	      if (!NULL (full))
		name = concat2 (dirname, build_string (slashfilename));
	      else
		name = build_string (filename);
	      list = Fcons (name, list);
	    }
	}
    }
  closedir (d);
  return Fsort (Fnreverse (list), Qstring_lessp);
}

Lisp_Object file_name_completion ();

DEFUN ("file-name-completion", Ffile_name_completion, Sfile_name_completion,
  2, 2, 0,
  "Complete file name FILE in directory DIR.\n\
Returns the longest string common to all filenames in DIR\n\
that start with FILE.\n\
If there is only one and FILE matches it exactly, returns t.\n\
Returns nil if DIR contains no name starting with FILE.")
  (file, dirname)
     Lisp_Object file, dirname;
{
  /* Don't waste time trying to complete a null string.
     Besides, this case happens when user is being asked for
     a directory name and has supplied one ending in a /.
     We would not want to add anything in that case
     even if there are some unique characters in that directory.  */
  if (XTYPE (file) == Lisp_String && XSTRING (file)->size == 0)
    return file;
  return file_name_completion (file, dirname, 0, 0);
}

DEFUN ("file-name-all-completions", Ffile_name_all_completions,
  Sfile_name_all_completions, 2, 2, 0,
  "Return a list of all completions of file name FILE in directory DIR.")
  (file, dirname)
     Lisp_Object file, dirname;
{
  return file_name_completion (file, dirname, 1, 0);
}

#ifdef VMS

DEFUN ("file-name-all-versions", Ffile_name_all_versions,
  Sfile_name_all_versions, 2, 2, 0,
  "Return a list of all versions of file name FILE in directory DIR.")
  (file, dirname)
     Lisp_Object file, dirname;
{
  return file_name_completion (file, dirname, 1, 1);
}

/* This function is used to deal with vms searth paths.			*/
/* jdeifik@isi.edu, bryan howard formerly at isi.edu			*/
DEFUN ("vms-expand-search-paths", Fvms_expand_search_paths, 
       Svms_expand_search_paths, 1, 1, 0,
       "Expands any vms paths to find which entry contains the first\n\
occurrence of the specified file.  The path returned is unambiguous\n\
with regard to VMS paths.  I.e. a file written to the returned\n\
filename will appear in the same directory it was read from.")
(filename)
Lisp_Object filename;
{
  char buf[256];

  CHECK_STRING(filename, 0);
  if(NULL(filename)) return(Qnil);
  return(expand_search_paths(filename));
}

Lisp_Object
expand_search_paths(file)
Lisp_Object file;
{
  Lisp_Object tem, dev, result = file;
  char buf[256];
  int i,n,new;
  char *cp, *bp;
  int skip_0_dir = 0;

/* This piece of code is repeated twice, and is a mess to maintain if
   not done this way. I know this is a kludge! ---- Richard Levitte  */
#define expand_search_paths_kludge \
  if(!strchr(buf,':')) \
    strcat(buf,":"); \
  if ((bp = strchr(buf,']'))) \
    { \
      switch(skip_0_dir) \
	{ \
	case -1: \
	  strcat(buf,"["); \
	  skip_0_dir = 8; \
	  break; \
	case 8: \
	  if (*--bp == '.') \
	    strcpy(bp,"]"); \
	  break; \
	} \
      strcat(buf,cp + skip_0_dir); \
    } \
  else \
    strcat(buf,cp)
/* End of kludge */

  new = access(XSTRING(file)->data,0);		/* does the file exist? */
  tem = build_string(XSTRING(file)->data);
  while(1)
    {
      dev = build_string(XSTRING(tem)->data);
      cp = strchr(XSTRING(dev)->data,':');
      if(!cp) return(result);
      *cp = 0;
      cp++;
      skip_0_dir = (strncmp(cp,"[000000]",8) == 0
		    || strncmp(cp,"<000000>",8) == 0 ? 8
		    : (strncmp(cp,"[000000.",8) == 0
		       || strcmp(cp,"<000000.",8) == 0 ? -1 : 0));
      n = trnlnm(XSTRING(dev)->data,0,buf);
      if(n <= 0) return(result);	/* no more translations or error */

      expand_search_paths_kludge;

      tem = Fexpand_file_name(build_string(buf));
      if(n == 1) continue;
      if(new || !access(XSTRING(tem)->data,0))
        {
	  result = tem;		/* change the string which we will return */
	  continue;
	}
      for(i = 1 ; i < n ; i++)
	{
	  trnlnm(XSTRING(dev)->data,i,buf);

	  expand_search_paths_kludge;

	  if(!access(buf,0))
	    {			/* found the translation we want */
	      tem = Fexpand_file_name(build_string(buf));
	      result = tem;
	      break;
	    }
	}
    }
}
#endif /* VMS */

Lisp_Object
file_name_completion (file, dirname, all_flag, ver_flag)
     Lisp_Object file, dirname;
     int all_flag, ver_flag;
{
  DIR *d;
  int bestmatchsize, skip;
  register int compare, matchsize;
  unsigned char *p1, *p2;
  int matchcount = 0;
  Lisp_Object bestmatch, tem, elt, name;
  struct stat st;
  int directoryp;
  int passcount;
  int count = specpdl_ptr - specpdl;
#ifdef VMS
/* This stuff is needed to call lib$find_file, which is needed to find  */
/* files in a vms search path that aren't on the first element of the path.*/
  extern DIRENTRY * readdirver ();
  DIRENTRY *((* readfunc) ());
  unsigned long context = 0, uflags = 2;
  char fbuf[256], dbuf[256], rbuf[256];
  DESCRIPTOR(rdsc,rbuf,256);
  DESCRIPTOR(fdsc,fbuf,0);
  DESCRIPTOR(ddsc,dbuf,0);
  int status;

  int VMSdirectoryp = 0;
  Lisp_Object VMSbestmatch;
  int VMS_2dotindex = -1;
  int VMSdir_no_starting_dot = 0;
#endif /* VMS */

  CHECK_STRING (file, 0);

  dirname = Fexpand_file_name (dirname, Qnil);

#ifdef VMS
#ifndef TRUE
#define TRUE 1
#endif
  file = Fupcase (file);
  p1 = XSTRING(file)->data;

  /* Filename completion on VMS ignores case, since VMS filesys does.  */
  specbind (Qcompletion_ignore_case, Qt);

  strcat(strcpy(fbuf,XSTRING(file)->data),"*");
  strcpy(dbuf,XSTRING(dirname)->data);

  /* Check for special case of VMS Directory completion. */
  if (fbuf[0]=='[' || fbuf[0] == '<')
    {
      char start_d = fbuf[0], end_d = (fbuf[0] == '[' ? ']' : '>');
      Lisp_Object new_file;
      char temp[256];
      char *last_dot;
      char *start_fname;
      int l;

      strcpy(temp,fbuf);
      last_dot = strrchr(temp,'.');
      if (last_dot)
        {
	  *last_dot++ = 0;
          start_fname = last_dot;
        }
      else
        {
          start_fname = temp;
          *start_fname++ = 0;
        }
      strcpy(fbuf,start_fname);
      l = strlen(dbuf) - 1;
      /* The following test is to cover the case where the user has
	 entered FOO:[BAR.BAZ][.COOKIE.MONKEY and then pushes Tab */
      if (dbuf[l] == ']' || dbuf[l] == '>')
	{
	  char tem[2] = ">";
	  *tem = dbuf[l];

	  if (temp[0]=='[' || temp[0]=='<')
	    {
	      /* if the user entered FOO:[BAR][BAZ.COOKIE + Tab, convert to
		 FOO:[BAR][.BAZ.COOKIE */
	      if (temp[1] != '.' && temp[1] != '\0'
		  && dbuf[l-1] != '.')
		{
		  VMSdir_no_starting_dot = 1;
		  if (temp[1] != '\0')
		    dbuf[l++] = '.';
		}
	      strcpy(dbuf+l, temp+1);
	      strcat(dbuf+l, tem);
	    }
	  else
	    /* if the user entered FOO:[BAR][BAZ + Tab, convert to
	       FOO:[BAR][.BAZ */
	    if (dbuf[l-1] != '.')
	      VMSdir_no_starting_dot = 1;
	}
      else
	{
	  strcat(temp,
		 (temp[0]=='[' ? "]" : (temp[0]=='<' ? ">" : "[000000]")));
	  strcat(dbuf,temp);
	}
      VMSdirectoryp = 1;
    }
  else
    VMSdirectoryp = 0;

  strcat(dbuf, VMSdirectoryp ? ".DIR" : ".*");
  if(ver_flag) strcat(dbuf,".*");
  fdsc.dsc$w_length = strlen(fbuf);
  ddsc.dsc$w_length = strlen(dbuf);

  if (!ver_flag && !VMSdirectoryp)
    {
      unsigned char * VMStmp = index (XSTRING (file)->data, '.');
      if (VMStmp)
	{
	  unsigned char *VMStmp2 = index (VMStmp+1, '.');
	  if (VMStmp2)
	    VMStmp = VMStmp2;
	  else
	    VMStmp = index (VMStmp+1, ';');
	}
      if (VMStmp)
	{
	  VMS_2dotindex = VMStmp - XSTRING(file)->data;
	  file = Fsubstring (file,
			     make_number (0), make_number (VMS_2dotindex));
	  if (VMS_2dotindex < 3 || strncmp(VMStmp - 4, ".DIR", 4) != 0)
	    ver_flag = TRUE;	/* We may be interested in version numbers,
				   but not for directory files */
	}
    }
#endif /* VMS */

  bestmatch = Qnil;

  /* passcount = 0, ignore files that end in an ignored extension.
     If nothing found then try again with passcount = 1, don't ignore them.
     If looking for all completions, start with passcount = 1,
     so always take even the ignored ones.  */
  for (passcount = !!all_flag; NULL (bestmatch) && passcount < 2; passcount++)
    {
#ifndef VMS
      if (!(d = opendir (XSTRING (Fdirectory_file_name (dirname))->data)))
	report_file_error ("Opening directory", Fcons (dirname, Qnil));
#endif

      /* Loop reading blocks */
      /* (att3b compiler bug requires do a null comparison this way) */
      while (1)
	{
	  DIRENTRY *dp;
	  int len;

#ifdef VMS
          char *cp;
          DIRENTRY buf;

          status = lib$find_file(&fdsc,&rdsc,&context,&ddsc,0,0,&uflags);
          if (status==RMS$_NMF || status==RMS$_FNF) break; /* No more files */
          if (status!=RMS$_NORMAL)
            {
              Lisp_Object dname = make_string(dbuf,strlen(dbuf));

              report_file_error ("Opening directory", Fcons (dname, Qnil));
            }
          if(cp = strchr(rbuf,' ')) *cp = 0; /* terminate filename with '\0' */
          if(!ver_flag && (cp = strchr(rbuf,';'))) *cp = 0;
          cp = strchr(rbuf,']');
	  if (cp == 0) cp = strchr(rbuf,'>');
	  cp = cp + 1;
	  
/* Make dp point to some allocated memory i.e. buf.*/
          dp = &buf;

          /* Must reformat VMS directories */
          if (VMSdirectoryp)
            {
              /* Form VMS Directory name */
              char *dpname = dp->d_name;
              char *dot;
              int tlen;

	      if (VMSdir_no_starting_dot)
		{
		  dpname[0] = *(XSTRING(file)->data);
		  dpname[1] = '.';
		  strcpy(dpname+2,XSTRING(file)->data + 1);
		}
	      else
		strcpy(dpname,XSTRING(file)->data);
              dot = strrchr(dpname,'.');
              if (dot)
                strcpy(++dot,cp);
              else
                strcpy(&dpname[1],cp);
              dot = strrchr(dpname,'.');
              *dot = 0;
            }
          else
            strcpy(dp->d_name,cp);
/* Set d_namlen so that NAMLEN(dp) will work.*/
          dp->d_namlen = strlen(dp->d_name);
/* Set this to something other than 0.*/
          dp->d_ino = 1;
#else /* not VMS */
	  dp = readdir (d);
	  if (!dp) break;
#endif /* not VMS */

	  len = NAMLEN (dp);

	  if (!NULL (Vquit_flag) && NULL (Vinhibit_quit))
	    goto quit;
	  if (!dp->d_ino
	      || len < XSTRING (file)->size
#ifdef VMS
	      /* If the given file name has a second dot, or a semicolon,
		 and ver_flag == 0, then we are asking for an exact match */
	      || (!ver_flag
		  && len > XSTRING (file)->size && VMS_2dotindex != -1)
	      || (VMSdir_no_starting_dot
		  ? 0 <= scmp (dp->d_name + 2, XSTRING (file)->data + 1,
			       XSTRING (file)->size - 1)
		  : 0 <= scmp (dp->d_name, XSTRING (file)->data,
			       XSTRING (file)->size))
#else /* not VMS */
	      || 0 <= scmp (dp->d_name, XSTRING (file)->data,
			    XSTRING (file)->size)
#endif /* not VMS */
	      )
	    continue;

#ifdef VMS
	  if (stat(rbuf, &st) < 0)
#else
          if (file_name_completion_stat (dirname, dp, &st) < 0)
#endif
            continue;

          directoryp = ((st.st_mode & S_IFMT) == S_IFDIR);
#ifdef VMS
	  /* If we are looking for directories, we only want that! */
	  if (!directoryp && VMSdirectoryp)
	    continue;
#endif
	  tem = Qnil;
          if (!directoryp)
            {
	      /* Compare extensions-to-be-ignored against end of this file name */
	      /* if name is not an exact match against specified string */
	      if (!passcount && len > XSTRING (file)->size)
		/* and exit this for loop if a match is found */
		for (tem = Vcompletion_ignored_extensions;
		     CONSP (tem); tem = XCONS (tem)->cdr)
		  {
		    elt = XCONS (tem)->car;
		    if (XTYPE (elt) != Lisp_String) continue;
		    skip = len - XSTRING (elt)->size;
		    if (skip < 0) continue;

		    if (0 <= scmp (dp->d_name + skip,
				   XSTRING (elt)->data,
				   XSTRING (elt)->size))
		      continue;
		    break;
		  }
	    }

	  /* Unless an ignored-extensions match was found,
             process this name as a completion */
	  if (passcount || !CONSP (tem))
	    {
	      /* Update computation of how much all possible completions match */

	      matchcount++;

	      if (all_flag || NULL (bestmatch))
		{
		  /* This is a possible completion */
		  if (directoryp && !VMSdirectoryp)
		    {
		      /* This completion is a directory; make it end with '/' */
#ifdef VMS
		      VMSbestmatch = make_string (dp->d_name, len);
		      name = Ffile_name_as_directory (VMSbestmatch);
#else /* not VMS */
		      name = Ffile_name_as_directory (make_string (dp->d_name, len));
#endif /* not VMS */
		    }
		  else
#ifdef VMS
		    VMSbestmatch =
#endif /* VMS */
		      name = make_string (dp->d_name, len);
		  if (all_flag)
		    {
		      bestmatch = Fcons (name, bestmatch);
		    }
		  else
		    {
		      bestmatch = name;
#ifdef VMS
		      bestmatchsize = XSTRING (VMSbestmatch)->size;
#else /* not VMS */
		      bestmatchsize = XSTRING (name)->size;
#endif /* not VMS */
		    }
		}
	      else
		{
		  compare = min (bestmatchsize, len);
#ifdef VMS
		  p1 = XSTRING (VMSbestmatch)->data;
#else /* not VMS */
		  p1 = XSTRING (bestmatch)->data;
#endif /* not VMS */
		  p2 = (unsigned char *) dp->d_name;
		  matchsize = scmp(p1, p2, compare);
		  if (matchsize < 0)
		    matchsize = compare;
		  if (completion_ignore_case)
		    {
		      /* If this is an exact match except for case,
			 use it as the best match rather than one that is not
			 an exact match.  This way, we get the case pattern
			 of the actual match.  */
		      if ((matchsize == len
#ifdef VMS
			   && matchsize < XSTRING (VMSbestmatch)->size)
#else /* not VMS */
			   && matchsize < XSTRING (bestmatch)->size)
#endif /* not VMS */
			  ||
			  /* If there is no exact match ignoring case,
			     prefer a match that does not change the case
			     of the input.  */
			  (((matchsize == len)
			    ==
#ifdef VMS
			    (matchsize == XSTRING (VMSbestmatch)->size))
#else /* not VMS */
			    (matchsize == XSTRING (bestmatch)->size))
#endif /* not VMS */
			   /* If there is more than one exact match aside from
			      case, and one of them is exact including case,
			      prefer that one.  */
			   && !bcmp (p2, XSTRING (file)->data, XSTRING (file)->size)
			   && bcmp (p1, XSTRING (file)->data, XSTRING (file)->size)))
			{
#ifdef VMS
			  VMSbestmatch =
#endif /* VMS */
			    bestmatch = make_string (dp->d_name, len);
			  if (directoryp)
			    bestmatch = Ffile_name_as_directory (bestmatch);
			}
		    }
		  /* If this dirname all matches,
		     see if implicit following slash does too.  */
		  if (directoryp
		      && compare == matchsize
		      && bestmatchsize > matchsize
		      && p1[matchsize] == '/')
		    matchsize++;
		  bestmatchsize = matchsize;
		}
	    }
	}
#ifndef VMS
      closedir (d);
#else
      status = lib$find_file_end(&context);
#endif /* not VMS */
    }

  unbind_to (count);

  if (all_flag || NULL (bestmatch))
    return bestmatch;

  /* If we are ignoring case, and there is no exact match,
     and no additional text was supplied,
     don't change the case of what the user typed.  */
  if (completion_ignore_case && bestmatchsize == XSTRING (file)->size
#ifdef VMS
      && XSTRING (VMSbestmatch)->size > bestmatchsize)
#else /* not VMS */
      && XSTRING (bestmatch)->size > bestmatchsize)
#endif /* not VMS */
    return file;

#ifdef VMS
  /* If the supplied string (minus version number) has exactly one match, and
     it really is a directory, then return the [.x] form of it */
  if (matchcount == 1
      && (* (XSTRING (bestmatch)->data) == '['
	  || * (XSTRING (bestmatch)->data) == '<'))
    return bestmatch;
#endif /* VMS */

  /* Return t if the supplied string is an exact match (counting case);
     it does not require any change to be made.  */
  if (matchcount == 1 && bestmatchsize == XSTRING (file)->size
#ifdef VMS
      && !bcmp (XSTRING (VMSbestmatch)->data, XSTRING (file)->data,
#else /* not VMS */
      && !bcmp (XSTRING (bestmatch)->data, XSTRING (file)->data,
#endif /* not VMS */
		bestmatchsize))
    return Qt;
#ifdef VMS
  return Fsubstring (VMSbestmatch, make_number (0), make_number (bestmatchsize));
#else /* not VMS */
  return Fsubstring (bestmatch, make_number (0), make_number (bestmatchsize));
#endif /* not VMS */
 quit:
#ifndef VMS
  if (d) closedir (d);
#endif
  Vquit_flag = Qnil;
  return Fsignal (Qquit, Qnil);
}

file_name_completion_stat (dirname, dp, st_addr)
     Lisp_Object dirname;
     DIRENTRY *dp;
     struct stat *st_addr;
{
  int len = NAMLEN (dp);
  int pos = XSTRING (dirname)->size;
  char *fullname = (char *) alloca (len + pos + 2);

  bcopy (XSTRING (dirname)->data, fullname, pos);
#ifndef VMS
  if (fullname[pos - 1] != '/')
    fullname[pos++] = '/';
#endif /* VMS */

  bcopy (dp->d_name, fullname + pos, len);
  fullname[pos + len] = 0;

#ifdef S_IFLNK
  /* Use ordinary stat first, if that succeeds,
     so that a symlink pointing to a directory is considered a directory.
     Afterward, use lstat, so that a link pointing to nowhere
     at least seems to exist.  */
  if (stat (fullname, st_addr) >= 0)
    return 0;
  return lstat (fullname, st_addr);
#else
  return stat (fullname, st_addr);
#endif
}

Lisp_Object
make_time (time)
     int time;
{
  return Fcons (make_number (time >> 16),
		Fcons (make_number (time & 0177777), Qnil));
}

/* --------Added by Joshua Marantz, Viewlogic Systems Inc, 11/1/88-------- */
DEFUN ("time-string", Ftime_string, Stime_string, 1, 1, 0,
  "Convert TIME-LIST, which is a list of the high-order and\n\
low-order bytes of a Unix time value, to a string.")
  (time_list)
    Lisp_Object time_list;
{
    Lisp_Object s;
    long time_val, high, low;
    char *temp;

    s = Fcar (time_list);           CHECK_NUMBER (s, 3);  high = XFASTINT (s);
    s = Fcar (Fcdr (time_list));    CHECK_NUMBER (s, 3);  low =  XFASTINT (s);
    time_val = (high << 16) | low;
    temp = (char *) ctime (&time_val);
    return (build_string (temp));
}


DEFUN ("file-attributes", Ffile_attributes, Sfile_attributes, 1, 1, 0,
  "Return a list of attributes of file FILENAME.\n\
Value is nil if specified file cannot be opened.\n\
Otherwise, list elements are:\n\
 0. t for directory, string (name linked to) for symbolic link, or nil.\n\
 1. Number of links to file.\n\
 2. File uid.\n\
 3. File gid.\n\
 4. Last access time, as a list of two integers.\n\
  First integer has high-order 16 bits of time, second has low 16 bits.\n\
 5. Last modification time, likewise.\n\
 6. Last status change time, likewise.\n\
 7. Size in bytes.\n\
 8. File modes, as a string of ten letters or dashes as in ls -l.\n\
 9. t iff file's gid would change if file were deleted and recreated.\n\
10. inode number.\n\
\n\
If file does not exists, returns nil.")
  (filename)
     Lisp_Object filename;
{
  Lisp_Object values[11];
  Lisp_Object dirname;
  struct stat s;
  struct stat sdir;
  char modes[10];

  filename = Fexpand_file_name (filename, Qnil);
  if (lstat (XSTRING (filename)->data, &s) < 0)
    return Qnil;

  switch (s.st_mode & S_IFMT)
    {
    default:
      values[0] = Qnil; break;
    case S_IFDIR:
      values[0] = Qt; break;
#ifdef S_IFLNK
    case S_IFLNK:
      values[0] = Ffile_symlink_p (filename); break;
#endif
    }
  values[1] = make_number (s.st_nlink);
  values[2] = make_number (s.st_uid);
  values[3] = make_number (s.st_gid);
  values[4] = make_time (s.st_atime);
  values[5] = make_time (s.st_mtime);
  values[6] = make_time (s.st_ctime);
  /* perhaps we should set this to most-positive-fixnum if it is too large? */
  values[7] = make_number (s.st_size);
  filemodestring (&s, modes);
  values[8] = make_string (modes, 10);
#ifdef BSD4_3 /* Gross kludge to avoid lack of "#if defined(...)" in VMS */
#define BSD4_2 /* A new meaning to the term `backwards compatability' */
#endif
#ifdef BSD4_2			/* file gid will be dir gid */
  dirname = Ffile_name_directory (filename);
  if (dirname != Qnil && stat (XSTRING (dirname)->data, &sdir) == 0)
    values[9] = (sdir.st_gid != s.st_gid) ? Qt : Qnil;
  else					/* if we can't tell, assume worst */
    values[9] = Qt;
#else					/* file gid will be egid */
  values[9] = (s.st_gid != getegid ()) ? Qt : Qnil;
#endif	/* BSD4_2 (or BSD4_3) */
#ifdef BSD4_3
#ifdef BSD4_2
#undef BSD4_2 /* ok, you can look again without throwing up */
#endif /* ok, you can look again without throwing up */
#endif
  values[10] = make_number (s.st_ino);
  return Flist (11, values);
}

syms_of_dired ()
{
  defsubr (&Sdirectory_files);
  defsubr (&Sfile_name_completion);
#ifdef VMS
  defsubr (&Sfile_name_all_versions);
  defsubr (&Svms_expand_search_paths);
  
  DEFVAR_BOOL ("vms-write-file-where-read-from",
	       &vms_write_file_where_read_from,
	       "*Non-nil means to not allow vms path names in buffer filenames.\n\
This is done by calling vms-expand-search-paths which picks the\n\
proper component out of any vms path in the full filespec.  This\n\
variable is only meaningful on VMS.");
  vms_write_file_where_read_from = 1;
#endif /* VMS */
  defsubr (&Sfile_name_all_completions);
  defsubr (&Sfile_attributes);

/* --------Added by Joshua Marantz, Viewlogic Systems Inc, 11/1/88-------- */
  defsubr (&Stime_string);

#ifdef VMS
  Qcompletion_ignore_case = intern ("completion-ignore-case");
  staticpro (&Qcompletion_ignore_case);
#endif /* VMS */

  DEFVAR_LISP ("completion-ignored-extensions", &Vcompletion_ignored_extensions,
    "*Completion ignores filenames ending in any string in this list.");
  Vcompletion_ignored_extensions = Qnil;
}
