/* GNU Emacs routines to deal with trans tables.
   Copyright (C) 1987 Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU Emacs General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU Emacs, but only under the conditions described in the
GNU Emacs General Public License.   A copy of this license is
supposed to have been given to you along with GNU Emacs so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/* Written by Howard Gayle.  See chartab.c for details. */

#include "config.h"
#include "lisp.h"
#include "buffer.h"
#include "etctab.h"
#include "transtab.h"

Lisp_Object Qtrans_table_p;
DEFUN ("trans-table-p", Ftrans_table_p, Strans_table_p, 1, 1, 0,
   "Return t iff ARG is a trans table.")
(obj)
Lisp_Object obj;
{
return ((XTYPE (obj) == Lisp_Transtab) ? Qt : Qnil);
}

static Lisp_Object
check_trans_table (obj)
Lisp_Object obj;
{
register Lisp_Object tem;

while (tem = Ftrans_table_p (obj), NULL (tem))
   obj = wrong_type_argument (Qtrans_table_p, obj, 0);
return (obj);
}   

/* Convert the given Lisp_Transtab to a Lisp_Object. */
static Lisp_Object
enlisp_trans_table (sp)
struct Lisp_Transtab *sp;
{
register Lisp_Object z; /* Return. */

XSET (z, Lisp_Transtab, sp);
return (z);
}

DEFUN ("downcase-table", Fdowncase_table, Sdowncase_table, 0, 0, 0,
   "Return the lower case conversion trans table of the current buffer.")
()
{
return (enlisp_trans_table (current_buffer->downcase_table_v));
}

DEFUN ("upcase-table", Fupcase_table, Supcase_table, 0, 0, 0,
   "Return the upper case conversion trans table of the current buffer.")
()
{
return (enlisp_trans_table (current_buffer->upcase_table_v));
}

DEFUN ("standard-downcase-table", Fstandard_downcase_table,
   Sstandard_downcase_table, 0, 0, 0,
   "Return the standard lower case conversion trans table.\n\
This is the one used for new buffers.")
()
{
return (enlisp_trans_table (buffer_defaults.downcase_table_v));
}

DEFUN ("standard-upcase-table", Fstandard_upcase_table,
   Sstandard_upcase_table, 0, 0, 0,
   "Return the standard upper case conversion trans table.\n\
This is the one used for new buffers.")
()
{
return (enlisp_trans_table (buffer_defaults.upcase_table_v));
}

/* Store a trans table.  Check for errors. */
static Lisp_Object
set_trans_table (p, t)
struct Lisp_Transtab **p; /* Points to where to store the trans table. */
register Lisp_Object t;   /* The trans table as a Lisp object. */
{
t = check_trans_table (t);
*p = XTRANSTAB (t);
return (t);
}

DEFUN ("set-downcase-table", Fset_downcase_table, Sset_downcase_table, 1, 1, 0,
   "Select a new lower case conversion trans table for the current buffer.\n\
One argument, a trans table.")
(table)
Lisp_Object table;
{
return (set_trans_table (&current_buffer->downcase_table_v, table));
}

DEFUN ("set-upcase-table", Fset_upcase_table, Sset_upcase_table, 1, 1, 0,
   "Select a new lower case conversion trans table for the current buffer.\n\
One argument, a trans table.")
(table)
Lisp_Object table;
{
return (set_trans_table (&current_buffer->upcase_table_v, table));
}

DEFUN ("set-standard-downcase-table",
   Fset_standard_downcase_table, Sset_standard_downcase_table, 1, 1, 0,
   "Select a new standard lower case conversion trans table.\n\
This does not change the trans tables of any existing buffers.\n\
One argument, a trans table.")
(table)
Lisp_Object table;
{
return (set_trans_table (&buffer_defaults.downcase_table_v, table));
}

DEFUN ("set-standard-upcase-table",
   Fset_standard_upcase_table, Sset_standard_upcase_table, 1, 1, 0,
   "Select a new standard upper case conversion trans table.\n\
This does not change the trans tables of any existing buffers.\n\
One argument, a trans table.")
(table)
Lisp_Object table;
{
return (set_trans_table (&buffer_defaults.upcase_table_v, table));
}

DEFUN ("translate-region",
   Ftranslate_region, Stranslate_region, 3, 4, 0,
   "From START to END, translate characters according to trans\n\
table TABLE.  If optional arg NOUNDO is non-nil, don't record\n\
this change for undo and don't mark the buffer as really\n\
changed.  Returns the number of characters changed.")
(start, end, table, noundo)
Lisp_Object start;
Lisp_Object end;
register Lisp_Object table;
Lisp_Object noundo;
{
register int pos, stop; /* Limits of the region. */
register int rec; /* Flag set iff noundo is nil. */
register char_t *tt; /* Trans table. */
register char_t oc; /* Old character. */
register char_t nc; /* New character. */
register int cnt; /* Number of changes made. */
register Lisp_Object z; /* Return. */

validate_region (&start, &end);
table = check_trans_table (table);
tt = XTRANSTAB (table)->trt_to;
pos = XINT (start);
stop = XINT (end);
modify_region (pos, stop);
rec = NULL (noundo);
#if 0				/* Removed for Gayle's patches */
// *  if (!rec) bf_modified--;
#endif
cnt = 0;
for (; pos < stop; ++pos)
   {
   oc = FETCH_CHAR (pos);
   nc = tt[oc];
   if (nc != oc)
      {
      if (rec) record_change (pos, 1);
      FETCH_CHAR (pos) = nc;
      ++cnt;
      }
   }
XFASTINT (z) = cnt;
return (z);
}

DEFUN ("make-trans-table", Fmake_trans_table, Smake_trans_table, 0, 0, 0,
   "Return a new identity trans table.")
()
{
register struct Lisp_Transtab *nt; /* New trans table. */
register int   	     	      i;
register Lisp_Object 	      z;  /* Return. */

z = make_etc_table (sizeof (struct Lisp_Transtab), Lisp_Transtab);
nt = XTRANSTAB (z);
for (i = 0; i <= 255; ++i)
   nt->trt_to[i] = (char_t) i;
return (z);
}

DEFUN ("get-trans-table-to",
   Fget_trans_table_to, Sget_trans_table_to, 2, 2, 0,
   "Return the character to which character FROM is translated\n\
in trans table TABLE.")
(fromc, table)
Lisp_Object fromc;
register Lisp_Object table;
{
register Lisp_Object z;

table = check_trans_table (table);
XFASTINT (z) = XTRANSTAB (table)->trt_to[get_char_arg (fromc)];
return (z);
}

DEFUN ("set-trans-table-to",
   Fset_trans_table_to, Sset_trans_table_to, 3, 3, 0,
   "Set the translation from character FROM to character TO in\n\
trans table TABLE.")
(fromc, toc, table)
Lisp_Object fromc;
Lisp_Object toc;
register Lisp_Object table;
{
register char_t f = get_char_arg (fromc);
register char_t t = get_char_arg (toc);

table = check_trans_table (table);
XTRANSTAB (table)->trt_to[f] = t;
return (table);
}

init_trans_table_once ()
{
register int i;
register char_t *p;

Fset_standard_downcase_table (Fmake_trans_table ());
p = buffer_defaults.downcase_table_v->trt_to;
for (i = 'A'; i <= 'Z'; ++i)
   p[i] = (char_t) (i + 'a' - 'A');

Fset_standard_upcase_table (Fmake_trans_table ());
p = buffer_defaults.upcase_table_v->trt_to;
for (i = 'a'; i <= 'z'; ++i)
   p[i] = (char_t) (i - 'a' + 'A');
}

syms_of_trans_table ()
{
Qtrans_table_p = intern ("trans-table-p");
staticpro (&Qtrans_table_p);

defsubr (&Strans_table_p);
defsubr (&Sdowncase_table);
defsubr (&Supcase_table);
defsubr (&Sstandard_downcase_table);
defsubr (&Sstandard_upcase_table);
defsubr (&Sset_downcase_table);
defsubr (&Sset_upcase_table);
defsubr (&Sset_standard_downcase_table);
defsubr (&Sset_standard_upcase_table);
defsubr (&Stranslate_region);
defsubr (&Smake_trans_table);
defsubr (&Sget_trans_table_to);
defsubr (&Sset_trans_table_to);
}
