/*******************************************************************************
*									       *
* search.c -- Nirvana Editor search and replace functions		       *
*									       *
* Copyright (c) 1991 Universities Research Association, Inc.		       *
* All rights reserved.							       *
* 									       *
* This material resulted from work developed under a Government Contract and   *
* is subject to the following license:  The Government retains a paid-up,      *
* nonexclusive, irrevocable worldwide license to reproduce, prepare derivative *
* works, perform publicly and display publicly by or for the Government,       *
* including the right to distribute to other Government contractors.  Neither  *
* the United States nor the United States Department of Energy, nor any of     *
* their employees, makes any warrenty, express or implied, or assumes any      *
* legal liability or responsibility for the accuracy, completeness, or         *
* usefulness of any information, apparatus, product, or process disclosed, or  *
* represents that its use would not infringe privately owned rights.           *
*                                        				       *
* Fermilab Nirvana GUI Library						       *
* May 10, 1991								       *
*									       *
* Written by Mark Edel							       *
*									       *
*******************************************************************************/
static char SCCSID[] = "@(#)search.c	1.7     8/12/93";
#include <stdio.h>
#include <ctype.h>
#ifdef VMS
#include "../util/VMSparam.h"
#else
#include <sys/param.h>
#endif /*VMS*/
#include <Xm/Xm.h>
#include <X11/Shell.h>
#include <Xm/XmP.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#include <X11/Xatom.h>		/* for getting selection */
#ifdef MOTIF10
#include <X11/Selection.h>	/* " " */
#endif
#include <X11/X.h>		/* " " */
#include "../util/DialogF.h"
#include "../util/misc.h"
#include "nedit.h"
#include "search.h"
#include "window.h" 

/*
** Module Global Variables
*/
/* the current search string */
static char TheSearchString[SEARCHMAX] = "";
/* current replace-with string */
static char TheReplaceString[SEARCHMAX] = "";
/* current selected search type, literal, case sensitive, or regular exp. */
static int TheSearchType = SEARCH_LITERAL;
/* sleeze for passing search direction to callback routines */
static int SearchDirection;

#ifndef VMS
#ifndef SUN
extern char *regcmp();
extern char *regex();
extern char *__loc1;
#endif
#endif /*VMS*/

static void createReplaceDlog(Widget parent, WindowInfo *window);
static void createFindDlog(Widget parent, WindowInfo *window);
static void fFocusCB(Widget w, WindowInfo *window, caddr_t *callData);
static void rFocusCB(Widget w, WindowInfo *window, caddr_t *callData);
static void replaceCB(Widget w, WindowInfo *window,
		XmAnyCallbackStruct *callData); 
static void replaceAllCB(Widget w, WindowInfo *window,
		XmAnyCallbackStruct *callData);
static void rInSelCB(Widget w, WindowInfo *window,
		XmAnyCallbackStruct *callData); 
static void rCancelCB(Widget w, WindowInfo *window, caddr_t callData);
static void fCancelCB(Widget w, WindowInfo *window, caddr_t callData);
static void rFindCB(Widget w,WindowInfo *window,XmAnyCallbackStruct *callData);
static void findCB(Widget w, WindowInfo *window,XmAnyCallbackStruct *callData); 
static void literalCB(Widget w, WindowInfo *window, caddr_t callData);
static void caseCB(Widget w, WindowInfo *window, caddr_t callData);
static void regExpCB(Widget w, WindowInfo *window, caddr_t callData);
static void storeReplaceStrings(WindowInfo *window);
static void selectedSearchCB(Widget w, WindowInfo *window, Atom *selection,
			     Atom *type, char *value, int *length, int *format);
static int searchLiteral(char *string, char *searchString, int caseSense, 
	      int direction, int wrap, int beginPos, int *startPos, int *endPos);
static int searchRegex(char *string, char *searchString, int direction,
		int wrap, int beginPos, int *startPos, int *endPos);
static int forwardRegexSearch(char *string, char *searchString,
		int wrap, int beginPos, int *startPos, int *endPos);
static int backwardRegexSearch(char *string, char *searchString,
		int wrap, int beginPos, int *startPos, int *endPos);
static void upCaseString(char *outString, char *inString);
static void downCaseString(char *outString, char *inString);
static void resetFindTabGroup(WindowInfo *window);
static void resetReplaceTabGroup(WindowInfo *window);
static int searchMatchesSelection(WindowInfo *window);
static int FindMatchingChar(char *string, int charPos,
		int *startPos, int *endPos);

typedef struct _charMatchTable {
    char c;
    char match;
    char direction;
} charMatchTable;

#define N_MATCH_CHARS 13
static charMatchTable MatchingChars[N_MATCH_CHARS] = {
    {'{', '}', SEARCH_FORWARD},
    {'(', ')', SEARCH_FORWARD},
    {'<', '>', SEARCH_FORWARD},
    {'[', ']', SEARCH_FORWARD},
    {'/', '/', SEARCH_FORWARD},
    {'"', '"', SEARCH_FORWARD},
    {'\'', '\'', SEARCH_FORWARD},
    {'`', '`', SEARCH_FORWARD},
    {'\\', '\\', SEARCH_FORWARD},
    {'}', '{', SEARCH_BACKWARD},
    {')', '(', SEARCH_BACKWARD},
    {'>', '<', SEARCH_BACKWARD},
    {']', '[', SEARCH_BACKWARD}
};
    
void DoReplaceDlog(WindowInfo *window, int direction)
{
    Widget button;
    int selStart, selEnd;

    /* Create the dialog if it doesn't already exist */
    if (window->replaceDlog == NULL)
    	createReplaceDlog(window->shell, window);
    
    /* Set the initial search type based on user preference */
    TheSearchType = GetPrefSearch();
    
    /* Set the buttons with the selected search type */
    switch (TheSearchType) {
      case SEARCH_LITERAL:
      	button = window->replaceLiteralBtn;
	break;
      case SEARCH_CASE_SENSE:
      	button = window->replaceCaseBtn;
	break;
      case SEARCH_REGEX:
      	button = window->replaceRegExpBtn;
	break;
    }
    XmToggleButtonSetState(button, True, True);
    
    /* Blank the text fields */
    XmTextSetString(window->replaceText, "");
    XmTextSetString(window->replaceWithText, "");
    
    /* Dim the "Replace in Selection" button if there is no selection	*/
    XtSetSensitive(window->replaceInSelBtn,
		   GetSelection(window->textArea, &selStart, &selEnd));
		   
    /* Pass the search direction to the callback routines whose single	*/
    /* parameter is already used up with the window structure.  We use	*/
    /* this global variable, so as not to muck up the window structure	*/
    SearchDirection = direction;
    
    /* Display the dialog */
    XtManageChild(window->replaceDlog);
    while (XtIsManaged(window->replaceDlog))
	XtAppProcessEvent(XtWidgetToApplicationContext(window->replaceDlog),
		XtIMAll);
}

void DoFindDlog(WindowInfo *window, int direction)
{
    Widget button;

    /* Create the dialog if it doesn't already exist */
    if (window->findDlog == NULL)
    	createFindDlog(window->shell, window);
    
    /* Set the initial search type based on user preference */
    TheSearchType = GetPrefSearch();
    
    /* Set the buttons with the currently selected search type */
    switch (TheSearchType) {
      case SEARCH_LITERAL:
      	button = window->findLiteralBtn;
	break;
      case SEARCH_CASE_SENSE:
      	button = window->findCaseBtn;
	break;
      case SEARCH_REGEX:
      	button = window->findRegExpBtn;
	break;
    }
    XmToggleButtonSetState(button, True, True);
    
    /* Blank the text field */
    XmTextSetString(window->findText, "");

    /* Pass the search direction to the callback routines whose single
       parameter is already used up with the window structure.  We use
       this global variable, so as not to muck up the window structure */
    SearchDirection = direction;
    
    /* Display the dialog */
    XtManageChild(window->findDlog);
    while (XtIsManaged(window->findDlog))
	XtAppProcessEvent(XtWidgetToApplicationContext(window->findDlog),
		XtIMAll);
}

static void createReplaceDlog(Widget parent, WindowInfo *window)
{
    Arg    	args[50];
    int    	argcnt;
    XmString	st1;
    Widget	form, btnForm;
    Widget	searchTypeBox, literalBtn, caseBtn, regExpBtn;
    Widget	label1, label, replaceText, findText;
    Widget	findBtn, replaceAllBtn, rInSelBtn, cancelBtn, replaceBtn;
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNautoUnmanage, False); argcnt++;
    XtSetArg(args[argcnt], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL);
    	    argcnt++;
    form = XmCreateFormDialog(parent, "replaceDialog", args, argcnt);
    XtVaSetValues(form, XmNshadowThickness, 0, 0);
    SET_ONE_RSRC(XtParent(form), XmNtitle, "Replace");
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNtopOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNalignment, XmALIGNMENT_BEGINNING); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString,
    	     st1=MKSTRING("String to Find:")); argcnt++;
    label1 = XmCreateLabel(form, "label1", args, argcnt);
    XmStringFree(st1);
    XtManageChild(label1);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_WIDGET); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNtopWidget, label1); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNrightOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNmaxLength, SEARCHMAX); argcnt++;
    findText = XmCreateText(form, "replaceString", args, argcnt);
    XtAddCallback(findText, XmNfocusCallback, rFocusCB, window);
    RemapDeleteKey(findText);
    XtManageChild(findText);
    XmAddTabGroup(findText);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_WIDGET); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNtopWidget, findText); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNtopOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNalignment, XmALIGNMENT_BEGINNING); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString,
    	     st1=MKSTRING("Replace With:")); argcnt++;
    label = XmCreateLabel(form, "label", args, argcnt);
    XmStringFree(st1);
    XtManageChild(label);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_WIDGET); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNtopWidget, label); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNrightOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNmaxLength, SEARCHMAX); argcnt++;
    replaceText = XmCreateText(form, "replaceWithString", args, argcnt);
    RemapDeleteKey(replaceText);
    XtManageChild(replaceText);
    XmAddTabGroup(replaceText);

    argcnt = 0;
    XtSetArg(args[argcnt], XmNorientation, XmHORIZONTAL); argcnt++;
    XtSetArg(args[argcnt], XmNpacking, XmPACK_TIGHT); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_WIDGET); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNtopWidget, replaceText); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 2); argcnt++;
    XtSetArg(args[argcnt], XmNrightOffset, 4); argcnt++;
    XtSetArg(args[argcnt], XmNradioBehavior, True); argcnt++;
    XtSetArg(args[argcnt], XmNradioAlwaysOne, True); argcnt++;
    searchTypeBox = XmCreateRowColumn(form, "searchTypeBox", args, argcnt);
    XtManageChild(searchTypeBox);
    XmAddTabGroup(searchTypeBox);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString,
    	     st1=MKSTRING("Literal")); argcnt++;
    literalBtn = XmCreateToggleButton(searchTypeBox, "literal", args, argcnt);
    XtAddCallback (literalBtn, XmNvalueChangedCallback,literalCB,window);
    XmStringFree(st1);
    XtManageChild(literalBtn);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString,
    	     st1=MKSTRING("Case Sensitive Literal")); argcnt++;
    caseBtn = XmCreateToggleButton(searchTypeBox, "caseSenseLiteral", args, argcnt);
    XtAddCallback (caseBtn, XmNvalueChangedCallback, caseCB, window);
    XmStringFree(st1);
    XtManageChild(caseBtn);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString,
    	     st1=MKSTRING("Regular Expression")); argcnt++;
#if defined(SUN) || defined(VMS)
    XtSetArg(args[argcnt], XmNsensitive, False); argcnt++;
#endif
    regExpBtn = XmCreateToggleButton(searchTypeBox, "regExp", args, argcnt);
    XtAddCallback (regExpBtn, XmNvalueChangedCallback, regExpCB, window);
    XmStringFree(st1);
    XtManageChild(regExpBtn);

    argcnt = 0;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_WIDGET); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNtopWidget, searchTypeBox); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNrightOffset, 6); argcnt++;
    btnForm = XmCreateForm(form, "buttons", args, argcnt);
    XtManageChild(btnForm);
    XmAddTabGroup(btnForm);

    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, st1=MKSTRING("Replace")); argcnt++;
    XtSetArg(args[argcnt], XmNshowAsDefault, (short)1); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_POSITION); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftPosition, 0); argcnt++;
    XtSetArg(args[argcnt], XmNbottomOffset, 6); argcnt++;
    replaceBtn = XmCreatePushButton(btnForm, "replace", args, argcnt);
    XtAddCallback (replaceBtn, XmNactivateCallback, replaceCB, window);
    XmStringFree(st1);
    XtManageChild(replaceBtn);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString,
    	     st1=MKSTRING("Replace All")); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_POSITION); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftPosition, 19); argcnt++;
    XtSetArg(args[argcnt], XmNtopOffset, 6); argcnt++;
    replaceAllBtn = XmCreatePushButton(btnForm, "all", args, argcnt);
    XtAddCallback (replaceAllBtn, XmNactivateCallback, replaceAllCB, window);
    XmStringFree(st1);
    XtManageChild(replaceAllBtn);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString,
    	     st1=MKSTRING("R. In Selection")); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_POSITION); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftPosition, 43); argcnt++;
    XtSetArg(args[argcnt], XmNtopOffset, 6); argcnt++;
    rInSelBtn = XmCreatePushButton(btnForm, "inSel", args, argcnt);
    XtAddCallback (rInSelBtn, XmNactivateCallback, rInSelCB, window);
    XmStringFree(st1);
    XtManageChild(rInSelBtn);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, st1=MKSTRING("Find")); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_POSITION); argcnt++;
    XtSetArg(args[argcnt], XmNrightPosition, 84); argcnt++;
    XtSetArg(args[argcnt], XmNtopOffset, 6); argcnt++;
    findBtn = XmCreatePushButton(btnForm, "find", args, argcnt);
    XtAddCallback (findBtn, XmNactivateCallback, rFindCB, window);
    XmStringFree(st1);
    XtManageChild(findBtn);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, st1=MKSTRING("Cancel")); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_POSITION); argcnt++;
    XtSetArg(args[argcnt], XmNrightPosition, 100); argcnt++;
    XtSetArg(args[argcnt], XmNtopOffset, 6); argcnt++;
    cancelBtn = XmCreatePushButton(btnForm, "cancel", args, argcnt);
    XtAddCallback (cancelBtn, XmNactivateCallback, rCancelCB, window);
    XtVaSetValues(form, XmNcancelButton, cancelBtn, 0);
    XmStringFree(st1);
    XtManageChild(cancelBtn);
    
    window->replaceDlog = form;
    window->replaceText = findText;
    window->replaceWithText = replaceText;
    window->replaceLiteralBtn = literalBtn;
    window->replaceCaseBtn = caseBtn;
    window->replaceRegExpBtn = regExpBtn;
    window->replaceBtns = btnForm;
    window->replaceBtn = replaceBtn;
    window->replaceInSelBtn = rInSelBtn;
    window->replaceSearchTypeBox = searchTypeBox;
}

static void createFindDlog(Widget parent, WindowInfo *window)
{
    Arg    	args[50];
    int    	argcnt;
    XmString	st1;
    Widget	form, btnForm, searchTypeBox, literalBtn, caseBtn, regExpBtn;
    Widget	findText, label1, cancelBtn, findBtn;
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNautoUnmanage, False); argcnt++;
    XtSetArg(args[argcnt], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL);
    	    argcnt++;
    form = XmCreateFormDialog(parent, "findDialog", args, argcnt);
    XtVaSetValues(form, XmNshadowThickness, 0, 0);
    SET_ONE_RSRC(XtParent(form), XmNtitle, "Find");
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNtopOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNalignment, XmALIGNMENT_BEGINNING); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, 
    	     st1=MKSTRING("String to Find:")); argcnt++;
    label1 = XmCreateLabel(form, "label1", args, argcnt);
    XmStringFree(st1);
    XtManageChild(label1);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_WIDGET); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNtopWidget, label1); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNrightOffset, 6); argcnt++;
    XtSetArg(args[argcnt], XmNmaxLength, SEARCHMAX); argcnt++;
    findText = XmCreateText(form, "searchString", args, argcnt);
    XtAddCallback(findText, XmNfocusCallback, fFocusCB, window);
    RemapDeleteKey(findText);
    XtManageChild(findText);
    XmAddTabGroup(findText);

    argcnt = 0;
    XtSetArg(args[argcnt], XmNorientation, XmHORIZONTAL); argcnt++;
    XtSetArg(args[argcnt], XmNpacking, XmPACK_TIGHT); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_WIDGET); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNtopWidget, findText); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 2); argcnt++;
    XtSetArg(args[argcnt], XmNrightOffset, 4); argcnt++;
    XtSetArg(args[argcnt], XmNradioBehavior, True); argcnt++;
    XtSetArg(args[argcnt], XmNradioAlwaysOne, True); argcnt++;
    searchTypeBox = XmCreateRowColumn(form, "searchTypeBox", args, argcnt);
    XtManageChild(searchTypeBox);
    XmAddTabGroup(searchTypeBox);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, st1=MKSTRING("Literal")); argcnt++;
    literalBtn = XmCreateToggleButton(searchTypeBox, "literal", args, argcnt);
    XtAddCallback (literalBtn, XmNvalueChangedCallback,literalCB,window);
    XmStringFree(st1);
    XtManageChild(literalBtn);
    
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, 
    	     st1=MKSTRING("Case Sensitive Literal")); argcnt++;
    caseBtn = XmCreateToggleButton(searchTypeBox, "caseSenseLiteral", args, argcnt);
    XtAddCallback (caseBtn, XmNvalueChangedCallback, caseCB, window);
    XmStringFree(st1);
    XtManageChild(caseBtn);
 
    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, 
    	     st1=MKSTRING("Regular Expression")); argcnt++;
#if defined(SUN) || defined(VMS)
    XtSetArg(args[argcnt], XmNsensitive, False); argcnt++;
#endif
    regExpBtn = XmCreateToggleButton(searchTypeBox, "regExp", args, argcnt);
    XtAddCallback (regExpBtn, XmNvalueChangedCallback, regExpCB, window);
    XmStringFree(st1);
    XtManageChild(regExpBtn);

    argcnt = 0;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_WIDGET); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNtopWidget, searchTypeBox); argcnt++;
    XtSetArg(args[argcnt], XmNleftOffset, 2); argcnt++;
    XtSetArg(args[argcnt], XmNrightOffset, 4); argcnt++;
    btnForm = XmCreateForm(form, "buttons", args, argcnt);
    XtManageChild(btnForm);
    XmAddTabGroup(btnForm);

    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, st1=MKSTRING("Find")); argcnt++;
    XtSetArg(args[argcnt], XmNshowAsDefault, (short)1); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_POSITION); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftPosition, 20); argcnt++;
    XtSetArg(args[argcnt], XmNbottomOffset, 6); argcnt++;
    findBtn = XmCreatePushButton(btnForm, "find", args, argcnt);
    XtAddCallback (findBtn, XmNactivateCallback, findCB, window);
    XmStringFree(st1);
    XtManageChild(findBtn);

    argcnt = 0;
    XtSetArg(args[argcnt], XmNtraversalOn, True); argcnt++;
    XtSetArg(args[argcnt], XmNhighlightThickness, 2); argcnt++;
    XtSetArg(args[argcnt], XmNlabelString, st1=MKSTRING("Cancel")); argcnt++;
    XtSetArg(args[argcnt], XmNtopAttachment, XmATTACH_FORM); argcnt++;
    XtSetArg(args[argcnt], XmNbottomAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNleftAttachment, XmATTACH_NONE); argcnt++;
    XtSetArg(args[argcnt], XmNrightAttachment, XmATTACH_POSITION); argcnt++;
    XtSetArg(args[argcnt], XmNrightPosition, 80); argcnt++;
    XtSetArg(args[argcnt], XmNtopOffset, 6); argcnt++;
    cancelBtn = XmCreatePushButton(btnForm, "cancel", args, argcnt);
    XtAddCallback (cancelBtn, XmNactivateCallback, fCancelCB, window);
    XmStringFree(st1);
    XtManageChild(cancelBtn);
    XtVaSetValues(form, XmNcancelButton, cancelBtn, 0);
    
    window->findDlog = form;
    window->findText = findText;
    window->findLiteralBtn = literalBtn;
    window->findCaseBtn = caseBtn;
    window->findRegExpBtn = regExpBtn;
    window->findBtns = btnForm;
    window->findBtn = findBtn;
    window->findSearchTypeBox = searchTypeBox;
}

/*
** These callbacks fix a Motif 1.1 problem that the default button gets the
** keyboard focus when a dialog is created.  We want the first text field
** to get the focus, so we don't set the default button until the text field
** has the focus for sure.  I have tried many other ways and this is by far
** the least nasty.
*/
static void fFocusCB(Widget w, WindowInfo *window, caddr_t *callData) 
{
    SET_ONE_RSRC(window->findDlog, XmNdefaultButton, window->findBtn);
}
static void rFocusCB(Widget w, WindowInfo *window, caddr_t *callData) 
{
    SET_ONE_RSRC(window->replaceDlog, XmNdefaultButton, window->replaceBtn);
}

static void literalCB(Widget w, WindowInfo *window, caddr_t callData) 
{
    if (XmToggleButtonGetState(w))
    	TheSearchType = SEARCH_LITERAL;
}

static void caseCB(Widget w, WindowInfo *window, caddr_t callData) 
{
    if (XmToggleButtonGetState(w))
    	TheSearchType = SEARCH_CASE_SENSE;
}

static void regExpCB(Widget w, WindowInfo *window, caddr_t callData) 
{
    if (XmToggleButtonGetState(w))
    	TheSearchType = SEARCH_REGEX;
}

static void replaceCB(Widget w, WindowInfo *window,
		      XmAnyCallbackStruct *callData) 
{
    /* put find and replace strings from the dialog into the window struct */
    storeReplaceStrings(window);

    /* Set the initial focus of the dialog back to the search string	*/
    resetReplaceTabGroup(window);
    
    /* find the text and mark it */
    SearchAndReplace(window, SearchDirection);
    
    /* pop down the dialog */
    XtUnmanageChild(window->replaceDlog);
}

static void replaceAllCB(Widget w, WindowInfo *window,
			 XmAnyCallbackStruct *callData) 
{
    /* put find and replace strings from the dialog into the window struct */
    storeReplaceStrings(window);

    /* Set the initial focus of the dialog back to the search string	*/
    resetReplaceTabGroup(window);

    /* do replacement based on strings stored in window structure */
    ReplaceAll(window);
    
    /* pop down the dialog */
    XtUnmanageChild(window->replaceDlog);
}

static void rInSelCB(Widget w, WindowInfo *window,
			 XmAnyCallbackStruct *callData) 
{
    /* put find and replace strings from the dialog into the window struct */
    storeReplaceStrings(window);

    /* Set the initial focus of the dialog back to the search string	*/
    resetReplaceTabGroup(window);

    /* do replacement based on strings stored in window structure */
    ReplaceInSelection(window);
    
    /* pop down the dialog */
    XtUnmanageChild(window->replaceDlog);
}

static void rCancelCB(Widget w, WindowInfo *window, caddr_t callData) 
{
    /* Set the initial focus of the dialog back to the search string	*/
    resetReplaceTabGroup(window);

    /* pop down the dialog */
    XtUnmanageChild(window->replaceDlog);
}

static void fCancelCB(Widget w, WindowInfo *window, caddr_t callData) 
{
    /* Set the initial focus of the dialog back to the search string	*/
    resetFindTabGroup(window);
    
    /* pop down the dialog */
    XtUnmanageChild(window->findDlog);
}

static void rFindCB(Widget w, WindowInfo *window,XmAnyCallbackStruct *callData) 
{
    /* put find and replace strings from the dialog into the window struct */
    storeReplaceStrings(window);

    /* Set the initial focus of the dialog back to the search string	*/
    resetReplaceTabGroup(window);
    
    /* find the text and mark it */
    SearchAndSelect(window, SearchDirection, CurrentTime);

    /* pop down the dialog */
    XtUnmanageChild(window->replaceDlog);
}

static void findCB(Widget w, WindowInfo *window,XmAnyCallbackStruct *callData) 
{
    char *findText;
    
    /* save find string from the dialog in the global search string */
    findText = XmTextGetString(window->findText);
    strcpy(TheSearchString, findText);
    XtFree(findText);

    /* Set the initial focus of the dialog back to the search string	*/
    resetFindTabGroup(window);
    
    /* find the text and mark it */
    SearchAndSelect(window, SearchDirection, CurrentTime);    

    /* pop down the dialog */
    XtUnmanageChild(window->findDlog);
}

/* save the search and replace strings from the Replace dialog */
static void storeReplaceStrings(WindowInfo *window)
{
    char *replaceText, *replaceWithText;
    
    replaceText = XmTextGetString(window->replaceText);
    replaceWithText = XmTextGetString(window->replaceWithText);
    strcpy(TheSearchString, replaceText);
    strcpy(TheReplaceString, replaceWithText);
    XtFree(replaceText);
    XtFree(replaceWithText);
}

int SearchAndSelect(WindowInfo *window, int direction, Time time)
{
    int startPos, endPos;
    int found;
    int beginPos, cursorPos, selStart, selEnd;
        
    /* set the position to start the search so we don't find the same	*/
    /* string that was found on the last search				*/
    if (searchMatchesSelection(window)) {
    	/* selection matches search string, start before or after sel.	*/
    	GetSelection(window->textArea, &selStart, &selEnd);
	if (direction == SEARCH_BACKWARD) {
	    /* use the insert position - 1 for backward searches */
	    beginPos = selStart-1;
	} else {
	    /* use the insert position for forward searches */
	    beginPos = selEnd;
	}
    } else {
    	selStart = -1; selEnd = -1;
    	/* no selection, or no match, search relative cursor */
    	cursorPos = XmTextGetInsertionPosition(window->lastFocus);
	if (direction == SEARCH_BACKWARD) {
	    /* use the insert position - 1 for backward searches */
	    beginPos = cursorPos-1;
	} else {
	    /* use the insert position for forward searches */
	    beginPos = cursorPos;
	}
    }

    /* do the search, SearchWindow does appropriate dialogs and beeps */
    found = SearchWindow(window, direction, beginPos, &startPos, &endPos);
    if (!found)
    	return FALSE;
    
    /* if matched text is already selected, just beep */
    if (selStart==startPos && selEnd==endPos) {
    	XBell(TheDisplay, 100);
    	return FALSE;
    }

    /* select the text found string */
    XmTextSetSelection(window->lastFocus, startPos, endPos, time);

#ifdef MOTIF10 /* Motif 1.1 does this automatically, with less redraws	*/
    /* set the cursor position so pending delete will work and so 	*/
    /* autoShowCursorPosition will show the selected string		*/
    XmTextSetInsertionPosition(window->lastFocus,
    	(direction == SEARCH_FORWARD) ? endPos : startPos);
#endif

    return TRUE;
}

void SearchForSelected(WindowInfo *window, int direction, Time time)
{
    SearchDirection = direction;
    XtGetSelectionValue(window->textArea, XA_PRIMARY, XA_STRING,
    			selectedSearchCB, window, time);
}

static void selectedSearchCB(Widget w, WindowInfo *window, Atom *selection,
			     Atom *type, char *value, int *length, int *format)
{
    /* skip if we can't get the selection data or it's too long */
    if (*type == XT_CONVERT_FAIL || value == NULL) {
    	if (GetPrefSearchDlogs())
   	    DialogF(DF_WARN, window->shell, 1,
   	    	    "Selection not appropriate for searching", "OK");
    	else
    	    XBell(TheDisplay, 100);
	return;
    }
    if (*length > SEARCHMAX) {
    	if (GetPrefSearchDlogs())
   	    DialogF(DF_WARN, window->shell, 1, "Selection too long", "OK");
    	else
    	    XBell(TheDisplay, 100);
	XtFree(value);
	return;
    }
    /* should be of type text??? */
    if (*format != 8) {
    	fprintf(stderr, "NEdit: can't handle non 8-bit text\n");
    	XBell(TheDisplay, 100);
	XtFree(value);
	return;
    }
    /* make the selection the current search string */
    strncpy(TheSearchString, value, *length);
    TheSearchString[*length] = '\0';
    XtFree(value);
    
    /* this kind of search is by definition a literal search and not a	*/
    /* regular expression search, so reset the search type from REGEX	*/
    if (TheSearchType == SEARCH_REGEX)
    	TheSearchType = SEARCH_LITERAL;

    /* search for it in the window */
    SearchAndSelect(window, SearchDirection, CurrentTime);
}

void MatchSelectedCharacter(WindowInfo *window, Time time)
{
    char *findText;
    char *fileString;
    int selStart, selEnd;
    int startPos, endPos;
    int found;

    /*
    ** get the character to match from the text widget selection, or
    ** the character before the insert point if nothing is selected
    ** give up if too many characters are selected
    */
    if (!GetSelection(window->textArea, &selStart, &selEnd)) {
        selEnd = XmTextGetInsertionPosition(window->lastFocus);
	selStart = selEnd - 1;
	if (selStart < 0) {
	    XBell(TheDisplay, 100);
	    return;
	}
    }
    if ((selEnd - selStart) != 1) {
    	XBell(TheDisplay, 100);
	return;
    }
    findText = GetTextRange(window->textArea, selStart, selEnd);
    if (findText == NULL) {
    	XBell(TheDisplay, 100);
	return;
    }
    XtFree(findText);
    
    /*
    ** Search for it in the text widget
    */
    /* get the entire (sigh) text buffer from the text area widget */
    fileString = XmTextGetString(window->textArea);
    if (fileString == NULL) {
        DialogF(DF_ERR, window->shell, 1,
		"Out of memory!\nTry closing some windows.\nSave your files!",
		"OK");
	return;
    }
    /* lookup the matching character and search for it */
    found = FindMatchingChar(fileString, selStart, &startPos, &endPos);
    /* Free the text buffer copy returned from XmTextGetString */
    XtFree(fileString);
    if (!found) {
    	XBell(TheDisplay, 100);
	return;
    }

    /*
    ** select the text between the matching characters
    */
    XmTextSetSelection(window->lastFocus, startPos, endPos, time);
}

static int FindMatchingChar(char *string, int charPos,
			    int *startPos, int *endPos)
{
    int nestDepth, matchIndex, direction, beginPos;
    char *stringPtr;
    char c, matchC;
    
    /* get the character to match */
    c = string[charPos];
    
    /* Look up the matching character */
    for (matchIndex = 0; matchIndex<N_MATCH_CHARS; matchIndex++) {
        if (MatchingChars[matchIndex].c == c)
	    break;
    }
    if (matchIndex == N_MATCH_CHARS)
	return FALSE;
    matchC = MatchingChars[matchIndex].match;
    
    /* find it in string */
    direction = MatchingChars[matchIndex].direction;
    beginPos = (direction==SEARCH_FORWARD) ? charPos+1 : charPos-1;
    nestDepth = 1;
    if (direction == SEARCH_FORWARD) {
    	for (stringPtr= &string[beginPos]; *stringPtr!='\0'; stringPtr++) {
	    if (*stringPtr == matchC) {
	    	nestDepth--;
		if (nestDepth == 0) {
		    *startPos = charPos;
		    *endPos = stringPtr - string + 1;
		    return TRUE;
		}
	    } else if (*stringPtr == c) {
	    	nestDepth++;
	    }
	}
    } else {
    	/* SEARCH_BACKWARD */
	for (stringPtr= &string[beginPos]; stringPtr>=string; stringPtr--) {
	    if (*stringPtr == matchC) {
	    	nestDepth--;
		if (nestDepth == 0) {
		    *startPos = stringPtr - string;
		    *endPos = charPos + 1;
		    return TRUE;
		}
	    } else if (*stringPtr == c) {
	    	nestDepth++;
	    }
	}
    }
    return FALSE;
}

int SearchAndReplace(WindowInfo *window, int direction)
{
    int startPos, endPos;
    int found;
    int beginPos, cursorPos;
    int selectionMatches;
    
    /* If the text selected in the window matches the search string, 	*/
    /* the user is probably using search then replace method, so	*/
    /* replace the selected text regardless of where the cursor is	*/
    selectionMatches = searchMatchesSelection(window);
    if (selectionMatches) {
    	GetSelection(window->textArea, &startPos, &endPos);
	
    /* Otherwise, search for the string */
    } else {
	/* get the position to start the search */
	cursorPos = XmTextGetInsertionPosition(window->lastFocus);
	if (direction == SEARCH_BACKWARD) {
	    /* use the insert position - 1 for backward searches */
	    beginPos = cursorPos-1;
	} else {
	    /* use the insert position for forward searches */
	    beginPos = cursorPos;
	}
	/* do the search */
	found = SearchWindow(window, direction, beginPos, &startPos, &endPos);
	if (!found)
	    return FALSE;
    }
    
    /* replace the text */
    XmTextReplace(window->lastFocus, startPos, endPos, TheReplaceString);

    /* set the cursor position so autoShowCursorPosition will show the	*/
    /* selected string		*/
    XmTextSetInsertionPosition(window->lastFocus, startPos +
    	((direction == SEARCH_FORWARD) ? strlen(TheReplaceString) : 0));
    
    /* after successfully completing a replace, selected text attracts	*/
    /* attention away from the area of the replacement, particularly	*/
    /* when the selection represents a previous search. so deselect	*/
    XmTextClearSelection(window->lastFocus, CurrentTime);

    return TRUE;
}

int ReplaceInSelection(WindowInfo *window)
{
    int selStart, selEnd, beginPos, startPos, endPos, realOffset, replaceLen;
    int found, anyFound;
    char *fileString;
    
    /* find out where the selection is */
    if (!GetSelection(window->textArea, &selStart, &selEnd))
    	return FALSE;
	
    /* get the ENTIRE selected text */
    fileString = XmTextGetSelection(window->textArea);
    if (fileString == NULL)
    	return FALSE;
	
    replaceLen = strlen(TheReplaceString);
    found = TRUE;
    anyFound = FALSE;
    beginPos = 0;
    realOffset = selStart;
    while (found) {
	found = SearchString(fileString, TheSearchString, SEARCH_FORWARD,
			TheSearchType, FALSE, beginPos, &startPos, &endPos);
	if (found) {
	    /* replace the string and compensate for length change */
	    XmTextReplace(window->lastFocus, startPos+realOffset,
	    		  endPos+realOffset, TheReplaceString);
    	    realOffset += replaceLen - (endPos - startPos);
    	    beginPos = endPos;
	    anyFound = TRUE;
	}
    }
    XtFree(fileString);
    if (!anyFound) {
    	if (GetPrefSearchDlogs()) {
    	    /* Avoid bug in Motif by putting away search dialog before DialogF */
    	    if (window->findDlog)
    		XtUnmanageChild(window->findDlog);
    	    if (window->replaceDlog)
    		XtUnmanageChild(window->replaceDlog);
   	    DialogF(DF_INF, window->shell, 1, "String was not found", "OK");
    	} else
    	    XBell(TheDisplay, 100);
 	return FALSE;
    }
    
    /* set the insert point at the end of the last replacement */
    XmTextSetInsertionPosition(window->lastFocus, endPos + realOffset);

    return TRUE;
}

int ReplaceAll(WindowInfo *window)
{
    int beginPos, startPos, endPos;
    int found, nFound, removeLen, replaceLen, copyLen;
    int copyStart, copyEnd;
    char *fileString, *newFileString, *fillPtr;
    
    /* reject empty string */
    if (*TheSearchString == '\0')
    	return FALSE;
	
    /* get the entire text buffer from the text area widget */
    fileString = XmTextGetString(window->textArea);
    if (fileString == NULL)
	return FALSE;		/* rely on XtMalloc to print error */
    
    /* rehearse the search first to determine the size of the buffer needed
       to hold the substituted text.  No substitution done here yet */
    found = TRUE;
    nFound = 0;
    removeLen = 0;
    beginPos = 0;
    copyStart = -1;
    while (found) {
    	found = SearchString(fileString, TheSearchString, SEARCH_FORWARD,
    			 TheSearchType, FALSE, beginPos, &startPos, &endPos);
	if (found) {
	    if (copyStart < 0)
	    	copyStart = startPos;
    	    copyEnd = endPos;
    	    beginPos = endPos;
	    nFound++;
	    removeLen += endPos - startPos;
	}
    }
    
    /* If no matches were found, beep to tell the user, and return */
    if (nFound == 0) {
    	if (GetPrefSearchDlogs()) {
    	    /* Avoid bug in Motif by putting away search dialog before DialogF */
    	    if (window->findDlog)
    		XtUnmanageChild(window->findDlog);
    	    if (window->replaceDlog)
    		XtUnmanageChild(window->replaceDlog);
   	    DialogF(DF_INF, window->shell, 1, "String was not found", "OK");
    	} else
    	    XBell(TheDisplay, 100);
	return FALSE;
    }
    
    /* Allocate a new buffer to hold all of the new text between the first
       and last substitutions */
    replaceLen = strlen(TheReplaceString);
    copyLen = copyEnd - copyStart;
    newFileString = XtMalloc(copyLen - removeLen + nFound*replaceLen + 1);
    if (newFileString == NULL)
	return FALSE;		/* rely on XtMalloc to print error */
    
    /* Scan through the text buffer again, substituting the replace string
       and copying the part between replaced text to the new buffer  */
    found = TRUE;
    beginPos = 0;
    fillPtr = newFileString;
    while (found) {
    	found = SearchString(fileString, TheSearchString, SEARCH_FORWARD,
    			 TheSearchType, FALSE, beginPos, &startPos, &endPos);
	if (found) {
	    if (beginPos != 0) {
#ifdef VMS
		memcpy(fillPtr, &fileString[beginPos], startPos - beginPos);
#else
		bcopy(&fileString[beginPos], fillPtr, startPos - beginPos);
#endif /*VMS*/
		fillPtr += startPos - beginPos;
	    }
#ifdef VMS
	    memcpy(fillPtr, TheReplaceString, replaceLen);
#else
	    bcopy(TheReplaceString, fillPtr, replaceLen);
#endif /*VMS*/
	    fillPtr += replaceLen;
	    beginPos = endPos;
	}
    }
    *fillPtr = 0;
    XtFree(fileString);
    
    /* replace the contents of the text widget with the substituted text */
    XmTextReplace(window->lastFocus, copyStart, copyEnd, newFileString);
    XtFree(newFileString);
    
    /* Move the cursor to the end of the last replacement */
    XmTextSetInsertionPosition(window->lastFocus,
	   copyStart + fillPtr - newFileString);

    return TRUE;
}

/*
** searches through the text in window, attempting to match the
** current search string
*/
int SearchWindow(WindowInfo *window, int direction,
	int beginPos, int *startPos, int *endPos)
{
    char *fileString;
    int found, resp, fileEnd;
    
    /* reject empty string */
    if (*TheSearchString == '\0')
    	return FALSE;
	
    /* get the entire (sigh) text buffer from the text area widget */
    fileString = XmTextGetString(window->textArea);
    if (fileString == NULL) {
        DialogF(DF_ERR, window->shell, 1,
		"Out of memory!\nTry closing some windows.\nSave your files!",
		"OK");
	return FALSE;
    }
    
    /* search the string copied from the text area widget, and present
       dialogs, or just beep */
    if (GetPrefSearchDlogs()) {
    	found = SearchString(fileString, TheSearchString, direction,
    		TheSearchType, FALSE, beginPos, startPos, endPos);
    	/* Avoid bug in Motif by putting away search dialog before DialogF */
    	if (window->findDlog)
    	    XtUnmanageChild(window->findDlog);
    	if (window->replaceDlog)
    	    XtUnmanageChild(window->replaceDlog);
    	if (!found) {
    	    fileEnd = TextLength(window->textArea) - 1;
    	    if (direction == SEARCH_FORWARD && beginPos != 0) {
    		resp = DialogF(DF_QUES, window->shell, 2,
    			"Continue search from\nbeginning of file?", "Continue",
    			"Cancel");
    		if (resp == 2) {
 		    XtFree(fileString);
		    return False;
		}
   	    	found = SearchString(fileString, TheSearchString, direction,
    			TheSearchType, FALSE, 0, startPos, endPos);
	    } else if (direction == SEARCH_BACKWARD && beginPos != fileEnd) {
    		resp = DialogF(DF_QUES, window->shell, 2,
    			"Continue search\nfrom end of file?", "Continue",
    			"Cancel");
    		if (resp == 2) {
 		    XtFree(fileString);
		    return False;
		}
    	    	found = SearchString(fileString, TheSearchString, direction,
    			TheSearchType, FALSE, fileEnd, startPos, endPos);
	    }
	    if (!found)
    	    	DialogF(DF_INF, window->shell,1,"String was not found","OK");
    	}
    } else { /* no dialogs */
    	found = SearchString(fileString, TheSearchString, direction,
    		TheSearchType, TRUE, beginPos, startPos, endPos);
    	if (!found)
    	    XBell(TheDisplay, 100);
    }
    
    /* Free the text buffer copy returned from XmTextGetString */
    XtFree(fileString);

    return found;
}

int SearchString(char *string, char *searchString, int direction,
	   int searchType, int wrap, int beginPos, int *startPos, int *endPos)
{
    switch (searchType) {
      case SEARCH_CASE_SENSE:
      	 return searchLiteral(string, searchString, TRUE, direction, wrap,
	 		       beginPos, startPos, endPos);
      case SEARCH_LITERAL:
      	 return  searchLiteral(string, searchString, FALSE, direction, wrap,
	 		       beginPos, startPos, endPos);
      case SEARCH_REGEX:
      	 return  searchRegex(string, searchString, direction, wrap,
      	 		     beginPos, startPos, endPos);
    }
    return FALSE; /* never reached, just makes compilers happy */
}

static int searchLiteral(char *string, char *searchString, int caseSense, 
	int direction, int wrap, int beginPos, int *startPos, int *endPos)
{
/* This is critical code for the speed of searches.			    */
/* For efficiency, we define the macro DOSEARCH with the guts of the search */
/* routine and repeat it, changing the parameters of the outer loop for the */
/* searching, forwards, backwards, and before and after the begin point	    */
#define DOSEARCH() \
    if (*filePtr == *ucString || *filePtr == *lcString) { \
	/* matched first character */ \
	ucPtr = ucString; \
	lcPtr = lcString; \
	tempPtr = filePtr; \
	while (*tempPtr == *ucPtr || *tempPtr == *lcPtr) { \
	    tempPtr++; ucPtr++; lcPtr++; \
	    if (*ucPtr == 0) { \
		/* matched whole string */ \
		*startPos = filePtr - string; \
		*endPos = tempPtr - string; \
		return TRUE; \
	    } \
	} \
    } \

    register char *filePtr, *tempPtr, *ucPtr, *lcPtr;
    char lcString[SEARCHMAX], ucString[SEARCHMAX];

    if (caseSense) {
        strcpy(ucString, searchString);
        strcpy(lcString, searchString);
    } else {
    	upCaseString(ucString, searchString);
    	downCaseString(lcString, searchString);
    }

    if (direction == SEARCH_FORWARD) {
	/* search from beginPos to end of string */
	for (filePtr=string+beginPos; *filePtr!=0; filePtr++) {
	    DOSEARCH()
	}
	if (!wrap)
	    return FALSE;
	/* search from start of file to beginPos	*/
	for (filePtr=string; filePtr<=string+beginPos; filePtr++) {
	    DOSEARCH()
	}
	return FALSE;
    } else {
    	/* SEARCH_BACKWARD */
	/* search from beginPos to start of file.  A negative begin pos	*/
	/* says begin searching from the far end of the file		*/
	if (beginPos >= 0) {
	    for (filePtr=string+beginPos; filePtr>=string; filePtr--) {
		DOSEARCH()
	    }
	}
	if (!wrap)
	    return FALSE;
	/* search from end of file to beginPos */
	/*... this strlen call is extreme inefficiency, but it's not obvious */
	/* how to get the text string length from the text widget (under 1.1)*/
	for (filePtr=string+strlen(string);
		filePtr>=string+beginPos; filePtr--) {
	    DOSEARCH()
	}
	return FALSE;
    }
}

static int searchRegex(char *string, char *searchString, int direction,
		int wrap, int beginPos, int *startPos, int *endPos)
{
    if (direction == SEARCH_FORWARD)
	return forwardRegexSearch(string, searchString, wrap, 
				  beginPos, startPos, endPos);
    else
    	return backwardRegexSearch(string, searchString, wrap, 
				   beginPos, startPos, endPos);
}

static int forwardRegexSearch(char *string, char *searchString,
		int wrap, int beginPos, int *startPos, int *endPos)
{
    char *compiledRE = NULL;
    char *matchStart, *matchEnd;   
    
    /* search from beginPos to end of string */

#ifndef VMS
#ifndef SUN
    compiledRE = regcmp(searchString, (char *)NULL);
    if (compiledRE == NULL)
	return FALSE;
    matchEnd = regex(compiledRE, string + beginPos);
    matchStart = __loc1;
    if (matchEnd != NULL && matchStart != matchEnd) {
	*startPos = matchStart - string;
	*endPos = matchEnd - string;
	return TRUE;
    }
    if (!wrap)
	return FALSE;
    matchEnd = regex(compiledRE, string);
    matchStart = __loc1;
    if (   matchEnd != NULL && matchStart != matchEnd &&
	   (int)(matchEnd - string) <= beginPos) {
	*startPos = matchStart - string;
	*endPos = matchEnd - string;
	return TRUE;
    }
#endif
#endif /*VMS*/

    return FALSE;
}

static int backwardRegexSearch(char *string, char *searchString,
		int wrap, int beginPos, int *startPos, int *endPos)
{
    int matchStart, matchEnd, fwdBegin;
    int found, anyFound = FALSE;

    /* search from beginPos to start of file.  A negative begin pos	*/
    /* says begin searching from the far end of the file.		*/
    if (beginPos >= 0) {
	fwdBegin = 0;
	while (TRUE) {
	    found = forwardRegexSearch(string, searchString, FALSE,
      	 		 	       fwdBegin, &matchStart, &matchEnd);
      	    if (!found || matchEnd > beginPos)
      	 	break;
      	    *startPos = matchStart;
      	    *endPos = matchEnd;
      	    anyFound = TRUE;
      	    fwdBegin = matchEnd;
      	}
      	if (anyFound)
      	    return TRUE;
    }
    if (!wrap)
	return FALSE;

    /* search from end of file to beginPos */
    fwdBegin = (beginPos == -1) ? 0 : beginPos;
    while (TRUE) {
	found = forwardRegexSearch(string, searchString, FALSE,
		 		   fwdBegin, &matchStart, &matchEnd);
      	if (!found)
      	    break;
      	anyFound = TRUE;
      	fwdBegin = matchEnd;
    }
    if (anyFound) {
      	*startPos = matchStart;
      	*endPos = matchEnd;
      	return TRUE;
    }
    return FALSE;
}

static void upCaseString(char *outString, char *inString)
{
    char *outPtr, *inPtr;
    
    for (outPtr=outString, inPtr=inString; *inPtr!=0; inPtr++, outPtr++) {
    	*outPtr = toupper(*inPtr);
    }
    *outPtr = 0;
}

static void downCaseString(char *outString, char *inString)
{
    char *outPtr, *inPtr;
    
    for (outPtr=outString, inPtr=inString; *inPtr!=0; inPtr++, outPtr++) {
    	*outPtr = tolower(*inPtr);
    }
    *outPtr = 0;
}

/*
** resetFindTabGroup & resetReplaceTabGroup are really gruesome kludges to
** set the keyboard traversal.  XmProcessTraversal does not work at
** all on these dialogs.  ...It seems to have started working around
** Motif 1.1.2
*/
static void resetFindTabGroup(WindowInfo *window)
{
#ifdef MOTIF10
    XmRemoveTabGroup(window->findText);
    XmRemoveTabGroup(window->findSearchTypeBox);
    XmRemoveTabGroup(window->findBtns);
    XmAddTabGroup(window->findText);
    XmAddTabGroup(window->findSearchTypeBox);
    XmAddTabGroup(window->findBtns);
#endif
    XmProcessTraversal(window->findText, XmTRAVERSE_CURRENT);
}
static void resetReplaceTabGroup(WindowInfo *window)
{
#ifdef MOTIF10
    XmRemoveTabGroup(window->replaceText);
    XmRemoveTabGroup(window->replaceWithText);
    XmRemoveTabGroup(window->replaceSearchTypeBox);
    XmRemoveTabGroup(window->replaceBtns);
    XmAddTabGroup(window->replaceText);
    XmAddTabGroup(window->replaceWithText);
    XmAddTabGroup(window->replaceSearchTypeBox);
    XmAddTabGroup(window->replaceBtns);
#endif
    XmProcessTraversal(window->replaceText, XmTRAVERSE_CURRENT);
}

static int searchMatchesSelection(WindowInfo *window)
{
    int selLen, selStart, selEnd, startPos, endPos;
    char *string;
    int found;
    
    /* find length of selection, give up on no selection or too long */
    if (!GetSelection(window->textArea, &selStart, &selEnd))
	return FALSE;
    selLen = selEnd - selStart;
    if (selLen > SEARCHMAX)
	return FALSE;
    
    /* get the selected text */
    string = XmTextGetSelection(window->textArea);
    if (string == NULL)
    	return FALSE;
	
    /* search for the string in the selection (we are only interested 	*/
    /* in an exact match, but the procedure searchString does important */
    /* stuff like applying the correct matching algorithm)		*/
    found = SearchString(string, TheSearchString, SEARCH_FORWARD,
    			 TheSearchType, FALSE, 0, &startPos, &endPos);
    XtFree(string);
    
    /* decide if it is an exact match */
    if (!found)
    	return FALSE;
    if (startPos != 0 || endPos != selLen)
    	return FALSE;
    return TRUE;
}
