/*******************************************************************************
*									       *
* shift.c -- Nirvana Editor shift right, shift left functions		       *
*									       *
* Copyright (c) 1991 Universities Research Association, Inc.		       *
* All rights reserved.							       *
* 									       *
* This material resulted from work developed under a Government Contract and   *
* is subject to the following license:  The Government retains a paid-up,      *
* nonexclusive, irrevocable worldwide license to reproduce, prepare derivative *
* works, perform publicly and display publicly by or for the Government,       *
* including the right to distribute to other Government contractors.  Neither  *
* the United States nor the United States Department of Energy, nor any of     *
* their employees, makes any warrenty, express or implied, or assumes any      *
* legal liability or responsibility for the accuracy, completeness, or         *
* usefulness of any information, apparatus, product, or process disclosed, or  *
* represents that its use would not infringe privately owned rights.           *
*                                        				       *
* Fermilab Nirvana GUI Library						       *
* June 18, 1991								       *
*									       *
* Written by Mark Edel							       *
*									       *
*******************************************************************************/
static char SCCSID[] = "@(#)shift.c	1.4     8/12/93";
#ifdef VMS
#include "../util/VMSparam.h"
#else
#include <sys/param.h>
#endif /*VMS*/
#include <Xm/Xm.h>
#include <Xm/Text.h>
#include "../util/DialogF.h"
#include "nedit.h"
#include "shift.h"
#include "window.h"

static char *shiftText(char *text, int direction, int tabsAllowed, int *newLen);
static void shiftLineRight(char *line, int tabsAllowed);
static void shiftLineLeft(char *line);
static int atTabStop(int pos);
static int nextTab(int pos);
static int countLines(char *text);

void ShiftSelection(WindowInfo *window, int direction, Time time)
{
    Widget tw = window->lastFocus;
    int startPos, endPos, shiftedLen, newEndPos;
    char *text, *shiftedText;

    if (!FindSelectedLines(tw, &startPos, &endPos))
    	return;
	
    text = GetTextRange(tw, startPos, endPos);
    shiftedText = shiftText(text, direction, TRUE, &shiftedLen);
    XtFree(text);
    XmTextReplace(tw, startPos, endPos, shiftedText);
    XtFree(shiftedText);
    
    newEndPos = startPos + shiftedLen;
    XmTextSetSelection(tw, startPos, newEndPos, time);
#ifdef MOTIF10	/* Motif 1.1 and later set the insert pt at end of selection */
    SetAndShowInsertPosition(window, newEndPos);
#endif
}

/*
** Extend the selection to encompass complete lines.  If there is no selection,
** return the line that the insert position is on.  Will return an error
** if it can't find a newline within MAX_LINE_LENGTH of the ends of the
** selection
*/
int FindSelectedLines(Widget widget, int *startPos, int *endPos)
{
    int selStart, selEnd, rangeStart, rangeEnd;
    int searchLen, searchPos;
    char *searchText;
    
    /* get selection, if no text selected, use current insert position */
    if (!GetSelection(widget, &selStart, &selEnd)) {
    	GET_ONE_RSRC(widget, XmNcursorPosition, &selStart);
	selEnd = selStart;
    }
    
    /* if the selection is empty, extend it so the search code below
       will find a whole line rather than a single newline character */
    if (selEnd == selStart)
    	selEnd++;

    /* get a copy of the text at least one line further in each direction
       from the selected text or the insertion point */
    rangeStart = MAX(0, selStart-MAX_LINE_LENGTH);
    rangeEnd = MIN(TextLength(widget), selEnd+MAX_LINE_LENGTH);
    searchText = GetTextRange(widget, rangeStart, rangeEnd);
    searchLen = rangeEnd - rangeStart;
    
    /* search backward from the start of the selection for a newline */
    for (searchPos=selStart-rangeStart-1; searchPos>=0; searchPos--) {
	if (searchText[searchPos] == '\n') {
	    *startPos = searchPos + rangeStart + 1;
	    break;
	}
    }
    if (searchPos < 0) {
    	if (rangeStart == 0) {
	    /* didn't find a newline, because this is start of text */
	    *startPos = 0;
	} else {
	    DialogF(DF_ERR, widget, 1, "Line too long", "OK");
	    XtFree(searchText);
	    return FALSE;
	}
    }
    
    /* search forward from the end of the selection for a newline */
    for (searchPos=selEnd-rangeStart-1; searchPos<=searchLen; searchPos++) {
	if (searchText[searchPos] == '\n') {
	    *endPos = searchPos + rangeStart + 1;
	    break;
	}
    }
    if (searchPos > searchLen) {
    	if (rangeEnd == TextLength(widget)) {
	    /* didn't find a newline, because this is the end of the text */
	    *endPos = rangeEnd;
	} else {
	    DialogF(DF_ERR, widget, 1, "Line too long", "OK");
	    XtFree(searchText);
	    return FALSE;
	}
    }
    XtFree(searchText);
    return TRUE;
}

/*
** shift lines left and right in a multi-line text string.  Returns the
** shifted text in memory that must be freed by the caller with XtFree.
*/
static char *shiftText(char *text, int direction, int tabsAllowed, int *newLen)
{
    char line[MAX_LINE_LENGTH];
    char *shiftedText;
    char *textPtr, *linePtr, *shiftedPtr;
    int bufLen;
    
    /*
    ** Allocate memory for shifted string.  Shift left adds a maximum
    ** of 6 characters per line (remove one tab, add 7 spaces).  Shift right
    ** adds a maximum of 1 character per line.
    */
    if (direction == SHIFT_RIGHT)
        bufLen = strlen(text) + countLines(text) + 1;
    else
        bufLen = strlen(text) + countLines(text) * 6 + 1;
    shiftedText = (char *)XtMalloc(bufLen);
    
    /*
    ** break into lines and call shiftLine on each
    */
    linePtr = line;
    textPtr = text;
    shiftedPtr = shiftedText;
    while (TRUE) {
	if (*textPtr=='\n' || *textPtr=='\0') {
	    *linePtr = '\0';
	    if (direction == SHIFT_RIGHT)
		shiftLineRight(line, tabsAllowed);
	    else
		shiftLineLeft(line);
	    strcpy(shiftedPtr, line);
	    shiftedPtr += strlen(line);
	    if (*textPtr == '\0') {
	        /* terminate string & exit loop at end of text */
	    	*shiftedPtr = '\0';
		break;
	    } else {
	    	/* move the newline from text to shifted text */
		*shiftedPtr++ = *textPtr++;
	    }
	    /* start line over */
	    linePtr = line;
	} else {
	    /* build up line */
	    *linePtr++ = *textPtr++;
	}
    }
    *newLen = shiftedPtr - shiftedText;
    return shiftedText;
}

static void shiftLineRight(char *line, int tabsAllowed)
{
    char lineIn[MAX_LINE_LENGTH];
    char *lineInPtr, *lineOutPtr;
    int whiteWidth;
    
    strcpy(lineIn, line);
    lineInPtr = lineIn;
    lineOutPtr = line;
    whiteWidth = 0;
    while (TRUE) {
        if (*lineInPtr == '\0') {
	    /* nothing on line, wipe it out */
	    *line = '\0';
	    return;
        } else if (*lineInPtr == ' ') {
	    /* white space continues with tab, advaance to next tab stop */
	    whiteWidth++;
	    *lineOutPtr++ = *lineInPtr++;
	} else if (*lineInPtr == '\t') {
	    /* white space continues with tab, advaance to next tab stop */
	    whiteWidth = nextTab(whiteWidth);
	    *lineOutPtr++ = *lineInPtr++;
	} else {
	    /* end of white space, add a space */
	    *lineOutPtr++ = ' ';
	    whiteWidth++;
	    /* if that puts us at a tab stop, change last 8 spaces into a tab */
	    if (tabsAllowed && atTabStop(whiteWidth)) {
		lineOutPtr -= TAB_DISTANCE;
		*lineOutPtr++ = '\t';
	    }
	    /* move remainder of line */
    	    while (*lineInPtr!='\0')
		*lineOutPtr++ = *lineInPtr++;
	    *lineOutPtr = '\0';
	    return;
	}
    }
}

static void shiftLineLeft(char *line)
{
    char lineIn[MAX_LINE_LENGTH];
    int whiteWidth, lastWhiteWidth, whiteGoal;
    char *lineInPtr, *lineOutPtr;
    
    strcpy(lineIn, line);
    lineInPtr = lineIn;
    lineOutPtr = line;
    whiteWidth = 0;
    lastWhiteWidth = 0;
    while (TRUE) {
        if (*lineInPtr == '\0') {
	    /* nothing on line, wipe it out */
	    *line = '\0';
	    return;
        } else if (*lineInPtr == ' ') {
	    /* white space continues with space, advance one character */
	    whiteWidth++;
	    *lineOutPtr++ = *lineInPtr++;
	} else if (*lineInPtr == '\t') {
	    /* white space continues with tab, advance to next tab stop	    */
	    /* save the position, though, in case we need to remove the tab */
	    lastWhiteWidth = whiteWidth;
	    whiteWidth = nextTab(whiteWidth);
	    *lineOutPtr++ = *lineInPtr++;
	} else {
	    /* end of white space, remove a character, (if there is one) */
	    if (lineOutPtr != line) {
		if (*(lineOutPtr-1) == ' ') {
		    /* end of white space is a space, just remove it */
		    lineOutPtr--;
		} else {
	    	    /* end of white space is a tab, remove it and add spaces */
		    lineOutPtr--;
		    whiteGoal = whiteWidth - 1;
		    whiteWidth = lastWhiteWidth;
		    while (whiteWidth < whiteGoal) {
			*lineOutPtr++ = ' ';
			whiteWidth++;
		    }
		}
	    }
	    /* move remainder of line */
    	    while (*lineInPtr!='\0')
		*lineOutPtr++ = *lineInPtr++;
	    /* add a null */
	    *lineOutPtr = '\0';
	    return;
	}
    }
}
       
static int atTabStop(int pos)
{
    return (pos%TAB_DISTANCE == 0);
}

static int nextTab(int pos)
{
    return (pos/TAB_DISTANCE)*TAB_DISTANCE + TAB_DISTANCE;
}

static int countLines(char *text)
{
    int count = 1;
    
    while(*text != '\0') {
    	if (*text++ == '\n') {
	    count++;
	}
    }
    return count;
}
