/*******************************************************************************
*									       *
* undo.c -- Nirvana Editor undo command					       *
*									       *
* Copyright (c) 1991 Universities Research Association, Inc.		       *
* All rights reserved.							       *
* 									       *
* This material resulted from work developed under a Government Contract and   *
* is subject to the following license:  The Government retains a paid-up,      *
* nonexclusive, irrevocable worldwide license to reproduce, prepare derivative *
* works, perform publicly and display publicly by or for the Government,       *
* including the right to distribute to other Government contractors.  Neither  *
* the United States nor the United States Department of Energy, nor any of     *
* their employees, makes any warrenty, express or implied, or assumes any      *
* legal liability or responsibility for the accuracy, completeness, or         *
* usefulness of any information, apparatus, product, or process disclosed, or  *
* represents that its use would not infringe privately owned rights.           *
*                                        				       *
* Fermilab Nirvana GUI Library						       *
* May 10, 1991								       *
*									       *
* Written by Mark Edel							       *
*									       *
*******************************************************************************/
static char SCCSID[] = "@(#)undo.c	1.3     8/12/93";
#ifdef VMS
#include "../util/VMSparam.h"
#else
#include <sys/param.h>
#endif /*VMS*/
#include <Xm/Xm.h>
#include <Xm/Text.h>
#include "nedit.h"
#include "undo.h"
#include "search.h"
#include "window.h"

#define FORWARD 1
#define REVERSE 2

static void appendDeletedText(WindowInfo *window, XmTextVerifyPtr mod,
				    int direction);
static int determineUndoType(XmTextVerifyPtr mod);

void Undo(WindowInfo *window)
{
    char *replaceString = NULL;
    
    /*
    ** XmTextReplace will eventually call SaveUndoInformation.  This is mostly
    ** good because it automatically makes undo operations undoable, but we
    ** need to be prepared for it.  1) Don't allow this undo operation to be
    ** accumulated with the operation that is now being undone (unlikely,
    ** but it could happen) by faking the last undo operation to something that
    ** doesn't accumulate.  2) Copy the saved text into a temporary string,
    ** because it gets reallocated in SaveUndoInformation.
    */
    window->undo.type = UNDO_NOOP;
    if (window->undo.oldText != NULL) {
    	replaceString = (char *)XtMalloc(window->undo.oldLen);
    	strcpy(replaceString, window->undo.oldText);
    }
    
    /* use the saved undo information to reverse changes */
    XmTextReplace(window->textArea, window->undo.startPos, window->undo.endPos,
    		  replaceString);
    
    /* free the temporary string */
    if (replaceString != NULL)
    	XtFree(replaceString);
}

/*
** SaveUndoInformation stores away the contents of the text widget's modify
** verify information in a form useable by Undo.  As a side effect, it also
** increments the autoSave operation and character counts since it needs to
** do the classification anyhow.
*/
/*... note, this assumes the format of the text returned in mod is FMT8BIT  */
/* It would be strange if it wasn't, since you give the text widget strings */
/* in standard c format, I can't imagine it converting them to FMT16BIT to  */
/* pass them back to you */
void SaveUndoInformation(WindowInfo *window, XmTextVerifyPtr mod)
{
    int newType, oldType, nInserted, nDeleted;
    UndoInfo *undo = &window->undo;
    
    newType = determineUndoType(mod);
    oldType = undo->type;
    /*
    ** Check for continuations of single character operations.  These are
    ** accumulated so a whole insertion or deletion can be undone, rather
    ** than just the last character that the user typed.
    */ 
    if (  ((oldType == ONE_CHAR_INSERT || oldType == ONE_CHAR_REPLACE)
    	   && newType == ONE_CHAR_INSERT) && (mod->startPos == undo->endPos)) {
	/* normal sequential character insertion */
	undo->endPos++;
	window->autoSaveCharCount++;
    } else if ((oldType == ONE_CHAR_REPLACE && newType == ONE_CHAR_REPLACE) &&
    	       (mod->startPos == undo->endPos)) {
    	/* overstrike mode replacement */
    	appendDeletedText(window, mod, FORWARD);
	undo->endPos++;
	window->autoSaveCharCount++;
    } else if ((oldType==ONE_CHAR_DELETE && newType==ONE_CHAR_DELETE) &&
    	       (mod->startPos==undo->startPos)) {
    	/* forward delete */
    	appendDeletedText(window, mod, FORWARD);
    } else if ((oldType==ONE_CHAR_DELETE && newType==ONE_CHAR_DELETE) &&
    	       (mod->startPos == undo->startPos-1)) {
    	/* reverse delete */
    	appendDeletedText(window, mod, REVERSE);
	undo->startPos--;
	undo->endPos--;
    /*
    ** Check for delete followed immediatly by insert at same point.  Lump
    ** these together to form a replace.  This happens alot in Motif 1.0,
    ** not as much in 1.1 (single character typed in pending delete selection
    ** still produces two modify verify callbacks in 1.1)
    */
    } else if ((oldType==ONE_CHAR_DELETE || oldType==BLOCK_DELETE) &&
    	       (newType==ONE_CHAR_INSERT || newType==BLOCK_INSERT) &&
    	       (mod->startPos == undo->startPos)) {
    	undo->endPos += mod->text->length;
    	undo->type = (newType == ONE_CHAR_INSERT) ?
    	    ONE_CHAR_REPLACE : BLOCK_REPLACE;
    } else {
    	/*
	** The user has started a new operation, throw out the old undo data
	** and save the new undo data
	*/
	if (undo->oldText != NULL)
	    XtFree(undo->oldText);
	undo->oldLen = 0;
	undo->oldText = NULL;
	undo->type = newType;
	if (newType == UNDO_NOOP)
	    return;
	nInserted = mod->text->length;
    	nDeleted = mod->endPos - mod->startPos;
	undo->startPos = mod->startPos;
	undo->endPos = mod->startPos + nInserted;
	if (nDeleted > 0) {
	    /* text was deleted, so save it */
	    undo->oldLen = nDeleted + 1;	/* +1 is for null at end */
	    undo->oldText = 
	    	GetTextRange(window->textArea, mod->startPos, mod->endPos);
	}
	window->autoSaveOpCount++;
    }
}

/*
** get the text that is about to be deleted and add to the beginning or end
** of the text saved for undoing the last operation.  This routine is intended
** for continuing of a string of one character deletes or replaces, but will
** work with more than one character.
*/
static void appendDeletedText(WindowInfo *window, XmTextVerifyPtr mod,
				    int direction)
{
    UndoInfo *undo = &window->undo;
    char *newText, *comboText;
    int newLen;

    /* get the text that is about to be consumed */
    newText = GetTextRange(window->textArea, mod->startPos,  mod->endPos);
    newLen = mod->endPos - mod->startPos;
    /* re-allocate space for saved text, adding 1 for the new character */
    comboText = XtMalloc(undo->oldLen + newLen);
    /* copy the new character and the already deleted text to the new memory */
    if (direction == FORWARD) {
    	strcpy(comboText, undo->oldText);
    	strcat(comboText, newText);
    } else {
	strcpy(comboText, newText);
	strcat(comboText, undo->oldText);
    }
    /* free the old saved text and attach the new */
    XtFree(undo->oldText);
    undo->oldText = comboText;
    undo->oldLen += newLen;
    /* dispose of the memory allocated by GetTextRange */
    XtFree(newText);
}

static int determineUndoType(XmTextVerifyPtr mod)
{
    int textDeleted, textInserted, nInserted, nDeleted;
    
    nInserted = mod->text->length;
    nDeleted = mod->endPos - mod->startPos;
    textDeleted = (nDeleted > 0);
    textInserted = (nInserted > 0);
    
    if (textInserted && !textDeleted) {
    	/* Insert */
	if (nInserted == 1)
	    return ONE_CHAR_INSERT;
	else
	    return BLOCK_INSERT;
    } else if (textInserted && textDeleted) {
    	/* Replace */
	if (nInserted == 1)
	    return ONE_CHAR_REPLACE;
	else
	    return BLOCK_REPLACE;
    } else if (!textInserted && textDeleted) {
    	/* Delete */
	if (nDeleted == 1)
	    return ONE_CHAR_DELETE;
	else
	    return BLOCK_DELETE;
    } else {
    	/* Nothing deleted or inserted */
	return UNDO_NOOP;
    }
}
