/*---------------------------------------------------------------------------
 *
 *  Module: mswin.c
 *
 *--------------------------------------------------------------------------*/

#define WIN31 
#define STRICT

#include <windows.h>
#include <commdlg.h>
#include <toolhelp.h>

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <signal.h>
#include <setjmp.h>
/*#include <conio.h>*/
#include <time.h>
/*#include <signal.h>*/
#include <fcntl.h>

#define	termdef	1			/* don't define "term" external */

#include "osdep.h"
#include "pico.h"
#include "estruct.h"
#include "efunc.h"
#include "edef.h"



/* Windows only version and resource defines. */
#include "resource.h"




/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *			Defines
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


#define VER_MAJOR 1
#define VER_MINOR 0
#define VER_BUILD 15


/* For debugging, export locals so debugger can see them. */
#ifdef DEBUG
#define LOCAL
#else
#define LOCAL		static
#endif



#define GWL_PTTYINFO		0	/* Offset in Window extra storage. */

#define ABOUTDLG_USEBITMAP	1



/* Max size permitted for the screen.  Larger than ever expec, but small
 * enough to prevent errors.  And small enough that the size of the
 * screen strucure is less than 64K. */
#define MAXNROW			180
#define MAXNCOLUMN		256

#define MINNROW			10	/* Minimum screen size */
#define MINNCOLUMN		32

#define WIN_MIN_X_SIZE		190	/* Minimum window size. */
#define WIN_MIN_Y_SIZE		180

#define WIN_X_BORDER_SIZE	8	/* Space taked by window frame. */
#define WIN_Y_BORDER_SIZE	65



/* Max size for temp storage. */
#define MAXLEN_TEMPSTR		256



/* Length of keyboard input queue. */
#define CHARACTER_QUEUE_LENGTH	32
#define MOUSE_QUEUE_LENGTH	32


/* Number of resize callback functions we can keep track of. */
#define RESIZE_CALLBACK_ARRAY_SIZE	3


/* Number of bytes held in the write accumulator. */
#define WRITE_ACCUM_SIZE		200



/* Cursor states. */
#define CS_SHOW         0x01		/* Cursor is not hidden. */
#define CS_FOCUSED	0x02		/* Window is focused. */
#define CS_VISIBLE	0x03		/* When above two bits set, cursor is
					 * visible. */

/* Auto Wrap States. */
#define WRAP_OFF	0		/* Never wrap to next line. */
#define WRAP_ON		1		/* Wrap to next line. */
#define WRAP_NO_SCROLL	2		/* Wrap to next line but DON'T scroll
					   screen to do it. */

/* Speicial keys in the Character Queue. */
#define CQ_FLAG_DOWN		0x01
#define CQ_FLAG_EXTENDED	0x02
#define CQ_FLAG_ALT		0x04



/* Special ASCII characters. */
#define ASCII_BEL       0x07
#define ASCII_BS        0x08
#define ASCII_LF        0x0A
#define ASCII_CR        0x0D
#define ASCII_XON       0x11
#define ASCII_XOFF      0x13




/* Character Attributes. */
#define CHAR_ATTR_NORM	0x00		/* Normal. */
#define CHAR_ATTR_REV	0x01		/* Reverse Video. */



/* My Timer Message */
#define MY_TIMER_ID	33
#define MY_TIMER_PERIOD (UINT)60000		/* timeout period in miliseconds. */

#define TIMER_FAIL_MESSAGE "Failed to get all necessary Windows resoruce (timers).  Pine will run, but may not be able to keep the connection to the server alive.  Quiting other applications and restarting Pine may solve the problem."




/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *			Typedefs
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

/* Type that the screen array is made up of. */
typedef unsigned char		CHAR;

/* Type that the attribute array is made up of. */
typedef BYTE			CharAttrib;
typedef int			(*ResizeCallBackProc)();

/* NOTE:  There is currently code that assumes that CHAR and CharAttrib
 *	are one byte in size.  All this cold is flaged with a preceeding
 *	assert () */


/* General info. */
typedef struct tagTTYINFO {
    CHAR	*pScreen;	/* Screen. */
    BYTE	*pAttrib;	/* Attributes. */
    BOOL	screenDirty;	/* TRUE if screen needs update. */
    BOOL	eraseScreen;	/* TRUE if need to erase whole screen */
#ifdef OWNRECT
    RECT	rPaintRect;	/* Rectangle that needs to be repainted. */
#endif
    CHAR	writeAccum[WRITE_ACCUM_SIZE];
    int		writeAccumCount;
    WORD	wCursorState;	/* Is cursor displayed? */
    HFONT	hTTYFont;
    LOGFONT	lfTTYFont;
    DWORD	rgbFGColor;	/* Normal forground color. */
    DWORD	rgbBGColor;	/* Normal background color. */
    DWORD	rgbRFGColor;	/* Reverse forground color. */
    DWORD	rgbRBGColor;	/* Reverse background color */
    BOOL	fMinimized;	/* True when window is minimized. */
    BOOL	fFocused;	/* True when we have focus. */
    BOOL	fNewLine;	/* Auto LF on CR. */
    BOOL	fMassiveUpdate;	/* True when in Massive screen update. */
    ResizeCallBackProc  resizer[RESIZE_CALLBACK_ARRAY_SIZE];
    int		autoWrap;	/* Auto wrap to next line. */
    CharAttrib	curAttrib;	/* Current character attributes. */
    int		actNRow, actNColumn;	/* Actual number of rows and comumns
					 * displayed. */
    int		xSize, ySize;		/* Size of screen in pixels */
    int		xScroll, yScroll;	/* ?? */
    int		xOffset, yOffset;	/* Amount of scroll offset 
				         * (always zero) in pixels */
    int		nColumn, nRow;		/* Current position of cursor in 
				         * cells. */
    int		xChar, yChar;		/* Width of a char in pixels. */
} TTYINFO, *PTTYINFO ;


typedef struct MSWINColor {
    char		*colorName;
    COLORREF		colorRef;
} MSWINColor;



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *			Forward function declarations.
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/



#define GETHINST( hWnd )  ((HINSTANCE) GetWindowWord( hWnd, GWW_HINSTANCE ))

#define MIN(x,y)	((x) < (y) ? (x) : (y))
#define MAX(x,y)	((x) > (y) ? (x) : (y))


// function prototypes (private)

LOCAL BOOL	InitApplication (HANDLE);
LOCAL HWND	InitInstance (HANDLE, int);
LOCAL void	MakeArgv (HINSTANCE hInstance, LPSTR cmdLine, int *pargc, 
				char ***pargv);
LOCAL LRESULT NEAR	CreateTTYInfo (HWND hWnd);
LOCAL BOOL NEAR	DestroyTTYInfo (HWND hWnd);
LOCAL int	ResizeTTYScreen (HWND hWnd, PTTYINFO pTTYInfo, 
					int newNRow, int newNColumn);
LOCAL BOOL	ResetTTYScreen (HWND, PTTYINFO);
#ifdef OWNRECT
void		MyInvalidateRect (PTTYINFO pTTYInfo, HWND hWnd, RECT *r);
#endif
LOCAL BOOL	PaintTTY (HWND);
LOCAL BOOL	GetMinMaxInfoTTY (HWND hWnd, MINMAXINFO __far *lpmmi);
LOCAL BOOL	SizeTTY (HWND, int, WORD, WORD);
LOCAL BOOL	SetTTYFocus (HWND);
LOCAL BOOL	KillTTYFocus (HWND);
LOCAL BOOL	MoveTTYCursor (HWND);
LOCAL BOOL	ProcessTTYKeyDown (HWND hWnd, WORD bOut, DWORD keyData);
LOCAL BOOL	ProcessTTYKeyUp (HWND hWnd, WORD key, DWORD keyData);
LOCAL BOOL	ProcessTTYCharacter (HWND hWnd, WORD bOut, DWORD keyData);
LOCAL BOOL	ProcessTTYMouse (int mevent, int button, int xPos, 
					int yPos, WORD keys);
LOCAL void	ScrollTTY (HWND hWnd, int scroll, int maxRows);
LOCAL BOOL	WriteTTYBlock (HWND, LPSTR, int);
LOCAL BOOL	WriteTTYText (HWND, LPSTR, int);
LOCAL BOOL	WriteTTYChar (HWND, char);
LOCAL VOID	GoModalDialogBoxParam (HINSTANCE, LPCSTR, HWND, 
					DLGPROC, LPARAM);
LOCAL BOOL	SelectTTYFont (HWND);
LOCAL void	SetColorAttribute (COLORREF *cf, char *colorName);
LOCAL BOOL	ConvertRGBString (char *colorName, COLORREF *cf);
LOCAL void	FlushWriteAccum (void);


LOCAL void	CQInit (void);
LOCAL BOOL	CQAvailable (void);
LOCAL BOOL	CQAdd (WORD c, DWORD keyData);
LOCAL WORD	CQGet ();

LOCAL void	MQInit (void);
LOCAL BOOL	MQAvailable (void);
LOCAL BOOL	MQAdd (int mevent, int button, int nRow, int nColumn, 
				unsigned int keys);
LOCAL BOOL	MQGet (MEvent * pmouse);

LOCAL int	MapVKtoMS (WORD c, WORD flags);



/* Functions exported to MS Windows. */

LRESULT FAR PASCAL __export PWndProc (HWND, UINT, WPARAM, LPARAM);
BOOL FAR PASCAL __export AboutDlgProc (HWND, UINT, WPARAM, LPARAM);







/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *			Exported Globals
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *			Module globals.
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


LOCAL PTTYINFO		gpTTYInfo;
LOCAL HWND		ghTTYWnd;
LOCAL char		TempBuf [MAXLEN_TEMPSTR];

LOCAL char		gszTTYClass[] = "TTYWndClass";
LOCAL char		gszAppName[45];

LOCAL HANDLE		ghAccel;

LOCAL BOOL		gfInitPhase;

LOCAL BOOL		KeyControlDown = FALSE;


LOCAL MSWINColor  MSWINColorTable[] =  {
	"black",	RGB(0,0,0),
	"blue",		RGB(0,0,255),
	"green",	RGB(0,255,0),
	"cyan",		RGB(0,255,255),
	"red",		RGB(255,0,0),
	"magenta",	RGB(255,0,255),
	"yellow",	RGB(255,255,0),
	"white",	RGB(255,255,255),
	"ltgray",	RGB(192,192,192),
	"gray",		RGB(128,128,128),
	"dkgray",	RGB(64,64,64),
	NULL,		0,
};







/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *			Windows Functions.
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

/*---------------------------------------------------------------------------
 *  int PASCAL WinMain( HANDLE hInstance, HANDLE hPrevInstance,
 *                      LPSTR lpszCmdLine, int nCmdShow )
 *
 *  Description:
 *     This is the main window loop!
 *
 *  Parameters:
 *     As documented for all WinMain() functions.
 *
 *--------------------------------------------------------------------------*/

int PASCAL __export
WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow )
{
    char		**argv;
    int			argc;


    gfInitPhase = TRUE;
    if (!hPrevInstance)
	if (!InitApplication( hInstance ))
	    return ( FALSE ) ;

    if (NULL == (ghTTYWnd = InitInstance( hInstance, nCmdShow )))
	return ( FALSE ) ;


    MakeArgv (hInstance, lpszCmdLine, &argc, &argv);
    
    gfInitPhase = FALSE;
    app_main (argc, argv);

    return (TRUE);
}


/*---------------------------------------------------------------------------
 *  BOOL  InitApplication( HANDLE hInstance )
 *
 *  Description:
 *     First time initialization stuff.  This registers information
 *     such as window classes.
 *
 *  Parameters:
 *     HANDLE hInstance
 *        Handle to this instance of the application.
 *
 *--------------------------------------------------------------------------*/

LOCAL BOOL  
InitApplication (HANDLE hInstance)
{
   WNDCLASS  wndclass;

   /* 
	* Register tty window class.
	*/

   wndclass.style =         CS_NOCLOSE;
   wndclass.lpfnWndProc =   PWndProc;
   wndclass.cbClsExtra =    0;
   wndclass.cbWndExtra =    sizeof (LONG);
   wndclass.hInstance =     hInstance ;
   wndclass.hIcon =         LoadIcon (hInstance, MAKEINTRESOURCE( PINEICON));
   wndclass.hCursor =       LoadCursor (NULL, IDC_ARROW);
   wndclass.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
   wndclass.lpszMenuName =  MAKEINTRESOURCE (PINEMENU);
   wndclass.lpszClassName = gszTTYClass ;

   return (RegisterClass (&wndclass));
}

/*---------------------------------------------------------------------------
 *  HWND  InitInstance( HANDLE hInstance, int nCmdShow )
 *
 *  Description:
 *     Initializes instance specific information.
 *
 *  Parameters:
 *     HANDLE hInstance
 *        Handle to instance
 *
 *     int nCmdShow
 *        How do we show the window?
 *
/*--------------------------------------------------------------------------*/

LOCAL HWND  
InitInstance (HANDLE hInstance, int nCmdShow)
{
    HWND  hTTYWnd;


    /* load accelerators */
    ghAccel = LoadAccelerators (hInstance, MAKEINTRESOURCE (PINEACCELL));

    LoadString (hInstance, IDS_APPNAME, gszAppName, sizeof (gszAppName));

    /* create the TTY window */
    hTTYWnd = CreateWindow (gszTTYClass, gszAppName,
		       WS_OVERLAPPEDWINDOW,
		       CW_USEDEFAULT, CW_USEDEFAULT,
		       CW_USEDEFAULT, CW_USEDEFAULT,
		       NULL, NULL, hInstance, NULL);


    if (NULL == hTTYWnd) 
	    return (NULL);

    ghTTYWnd = hTTYWnd;

    ShowWindow (hTTYWnd, nCmdShow);
    UpdateWindow (hTTYWnd);

    CQInit ();
    MQInit ();

    return (hTTYWnd);
}


/*---------------------------------------------------------------------------
 *  void MakeArgv ()
 *
 *  Description:
 *	Build a standard C argc, argv pointers into the command line string.
 *
 *
 *  Parameters:
 *	cmdLine		- Command line.
 *	*argc		- Count of words.
 *	***argc		- Pointer to Pointer to array of pointers to 
 *			  characters.
 *
 *--------------------------------------------------------------------------*/
LOCAL void
MakeArgv (HINSTANCE hInstance, LPSTR cmdLine, int *pargc, char ***pargv)
{
    int			argc;
    char		**argv;
    LPSTR		c;
    BOOL		inWord;
    int			wordCount;
#define CMD_PATH_LEN    128
    char		*modPath;
    char		mpLen;
    

    /* Count words in cmdLine. */
    wordCount = 0;
    inWord = FALSE;
    for (c = cmdLine; *c != '\0'; ++c) {
	if (inWord) {
	    if (*c == ' ' || *c == '\t') 
		inWord = FALSE;
	}
	else {
	    if (*c != ' ' && *c != '\t') {
		inWord = TRUE;
		++wordCount;
	    }
        }
    }
    
    ++wordCount;				/* One for program name. */
    argv = (char **) MemAlloc (sizeof (char _far *) * wordCount);
    *pargv = argv;
    *pargc = wordCount;

    modPath = (char *) MemAlloc (CMD_PATH_LEN);
    mpLen = GetModuleFileName (hInstance, modPath, CMD_PATH_LEN);
    if (mpLen > 0) {
	*(modPath + mpLen) = '\0';
        *(argv++) = modPath;
    }
    else {
	MemFree (modPath);
	*(argv++) = "Pine/Pico";
    }
    
    /* Now break up command line. */
    inWord = FALSE;
    for (c = cmdLine; *c != '\0'; ++c) {
	if (inWord) {
	    if (*c == ' ' || *c == '\t') {
		inWord = FALSE;
		*c = '\0';
	    }
	}
	else {
	    if (*c != ' ' && *c != '\t') {
		inWord = TRUE;
		*(argv++) = c;
	    }
        }
    }
}




/*---------------------------------------------------------------------------
 *  LRESULT FAR PASCAL __export TTYWndProc( HWND hWnd, UINT uMsg,
 *                                 WPARAM wParam, LPARAM lParam )
 *
 *  Description:
 *     This is the TTY Window Proc.  This handles ALL messages
 *     to the tty window.
 *
 *  Parameters:
 *     As documented for Window procedures.
 *
/*--------------------------------------------------------------------------*/

LRESULT FAR PASCAL __export 
PWndProc (HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    case WM_CREATE:
	 if (SetTimer (hWnd, MY_TIMER_ID, MY_TIMER_PERIOD, NULL) == 0) 
	     MessageBox (hWnd, TIMER_FAIL_MESSAGE, NULL, 
			     MB_OK | MB_ICONINFORMATION);
         return (CreateTTYInfo (hWnd));

    case WM_COMMAND: {
	switch ((WORD) wParam) {
        case IDM_SETFONT:
	    SelectTTYFont (hWnd);
	    break ;

        case IDM_ABOUT:
            GoModalDialogBoxParam ( GETHINST( hWnd ),
                                       MAKEINTRESOURCE( ABOUTDLGBOX ),
                                       hWnd,
                                       AboutDlgProc, NULL ) ;
            break;
#if 0	/* No exit menu item. */
        case IDM_EXIT:
            PostMessage( hWnd, WM_CLOSE, NULL, 0L ) ;
            break;
#endif
        }
        }
	break ;

    case WM_PAINT:
	PaintTTY( hWnd ) ;
        break ;
	 
    case WM_GETMINMAXINFO:
	GetMinMaxInfoTTY (hWnd, (MINMAXINFO __far *)lParam);
	break;

    case WM_SIZE:
	SizeTTY (hWnd, wParam, HIWORD(lParam), LOWORD(lParam));
        break ;
		 
#ifdef SCROLL
   case WM_HSCROLL:
	ScrollTTYHorz( hWnd, (WORD) wParam, LOWORD( lParam ) ) ;
        break ;

    case WM_VSCROLL:
        ScrollTTYVert( hWnd, (WORD) wParam, LOWORD( lParam ) ) ;
        break ;
#endif

    /*
     * WM_KEYDOWN is sent for every "key press" and reports on they
     * keyboard key, with out processing shift and control keys.
     * WM_CHAR is a synthetic event, created from KEYDOWN and KEYUP
     * events.  It includes processing or control and shift characters.
     * But does not get generated for extended keys suchs as arrow
     * keys. 
     * I'm going to try to use KEYDOWN for processing just extended keys
     * and let CHAR handle the the rest.
     *
     * The only key combo that is special is ^-space.  For that, I'll use
     * WM_KEYDOWN and WM_KEYUP to track the state of the control key.
     */
    case WM_CHAR:
        ProcessTTYCharacter (hWnd, LOBYTE (wParam), (DWORD)lParam);
        break ;
	 
    case WM_KEYDOWN:
	if (ProcessTTYKeyDown (hWnd, LOBYTE (wParam), (DWORD)lParam))
	    return (0);
        return( DefWindowProc( hWnd, uMsg, wParam, lParam ) ) ;

    case WM_KEYUP:
	if (ProcessTTYKeyUp (hWnd, LOBYTE (wParam), (DWORD)lParam))
	    return (0);
        return( DefWindowProc( hWnd, uMsg, wParam, lParam ) ) ;

    case WM_LBUTTONDOWN:
	ProcessTTYMouse (MEVENT_MOUSEDOWN, 1, LOWORD (lParam), 
			 HIWORD (lParam), wParam);
	break;

    case WM_LBUTTONUP:
	ProcessTTYMouse (MEVENT_MOUSEUP, 1, LOWORD (lParam), 
			 HIWORD (lParam), wParam);
	break;

    case WM_MBUTTONDOWN:
	ProcessTTYMouse (MEVENT_MOUSEDOWN, 2, LOWORD (lParam), 
		 HIWORD (lParam), wParam);
	break;

    case WM_MBUTTONUP:
	ProcessTTYMouse (MEVENT_MOUSEUP, 2, LOWORD (lParam), 
			 HIWORD (lParam), wParam);
	break;

    case WM_RBUTTONDOWN:
	ProcessTTYMouse (MEVENT_MOUSEDOWN, 3, LOWORD (lParam), 
			 HIWORD (lParam), wParam);
	break;

    case WM_RBUTTONUP:
	ProcessTTYMouse (MEVENT_MOUSEUP, 3, LOWORD (lParam), 
			 HIWORD (lParam), wParam);
	break;


    case WM_SETFOCUS:
        SetTTYFocus (hWnd);
        break;

    case WM_KILLFOCUS:
        KillTTYFocus (hWnd);
        break;
	
    case WM_TIMER:
	/* Really just used so that we continue to receive messages even while
	 * in background.  Causes mswin_getc() to process message and return
	 * to caller so that it can get some periodic processing in. */
	break;

    case WM_DESTROY:
	KillTimer (hWnd, MY_TIMER_ID);
        DestroyTTYInfo (hWnd);
        PostQuitMessage (0);
        break;

    case WM_CLOSE:
        if (IDOK != MessageBox( hWnd, "OK to close window?", "TTY Sample",
                                 MB_ICONQUESTION | MB_OKCANCEL ))
        break;

         // fall through

    default:
        return( DefWindowProc( hWnd, uMsg, wParam, lParam ) ) ;
    }
    return 0L ;

} // end of TTYWndProc()




/*---------------------------------------------------------------------------
 *  LRESULT NEAR CreateTTYInfo( HWND hWnd )
 *
 *  Description:
 *     Creates the tty information structure and sets
 *     menu option availability.  Returns -1 if unsuccessful.
 *
 *  Parameters:
 *     HWND  hWnd
 *        Handle to main window.
 *
 *-------------------------------------------------------------------------*/

LOCAL LRESULT NEAR 
CreateTTYInfo (HWND hWnd)
{
    HMENU		hMenu;
    PTTYINFO		pTTYInfo;
    int			i;
    

    pTTYInfo = (PTTYINFO) MemAlloc (sizeof (TTYINFO));
    if (pTTYInfo == NULL)
	return ((LRESULT) - 1);
    gpTTYInfo = pTTYInfo;

    /* initialize TTY info structure */

    pTTYInfo->wCursorState		= CS_SHOW;/* Shown but not focused. */
    pTTYInfo->fMinimized		= FALSE;
    pTTYInfo->fFocused			= FALSE;
    pTTYInfo->fNewLine			= FALSE;
    pTTYInfo->fMassiveUpdate		= FALSE;
    pTTYInfo->autoWrap			= WRAP_NO_SCROLL;
    pTTYInfo->writeAccumCount		= 0;
    pTTYInfo->actNRow			= 0;
    pTTYInfo->actNColumn		= 0;
    pTTYInfo->xSize			= 0;
    pTTYInfo->ySize			= 0;
    pTTYInfo->xScroll			= 0;
    pTTYInfo->yScroll			= 0;
    pTTYInfo->xOffset			= 0;
    pTTYInfo->yOffset			= 0;
    pTTYInfo->nColumn			= 0;
    pTTYInfo->nRow			= 0;
    pTTYInfo->xChar			= 0;
    pTTYInfo->yChar			= 0;
    pTTYInfo->hTTYFont			= NULL;
    pTTYInfo->rgbFGColor		= GetSysColor (COLOR_WINDOWTEXT);
    pTTYInfo->rgbBGColor		= GetSysColor (COLOR_WINDOW);
#if 1
    pTTYInfo->rgbRFGColor		= GetSysColor (COLOR_HIGHLIGHTTEXT);
    pTTYInfo->rgbRBGColor		= GetSysColor (COLOR_HIGHLIGHT);
#else
    pTTYInfo->rgbRFGColor		= pTTYInfo->rgbBGColor;
    pTTYInfo->rgbRBGColor		= pTTYInfo->rgbFGColor;
#endif

    /* Clear resize callback procs. */
    for (i = 0; i < RESIZE_CALLBACK_ARRAY_SIZE; ++i) 
	pTTYInfo->resizer[i] = NULL;

#ifdef OWNRECT
    /* set empty paint rect. */
    SetRectEmpty (&pTTYInfo->rPaintRect);
#endif
	    

    /* clear screen space */
    pTTYInfo->pScreen = NULL;
    pTTYInfo->pAttrib = NULL;

    /* setup default font information */

    pTTYInfo->lfTTYFont.lfHeight =         12;
    pTTYInfo->lfTTYFont.lfWidth =          0;
    pTTYInfo->lfTTYFont.lfEscapement =     0;
    pTTYInfo->lfTTYFont.lfOrientation =    0;
    pTTYInfo->lfTTYFont.lfWeight =         0;
    pTTYInfo->lfTTYFont.lfItalic =         0;
    pTTYInfo->lfTTYFont.lfUnderline =      0;
    pTTYInfo->lfTTYFont.lfStrikeOut =      0;
    pTTYInfo->lfTTYFont.lfCharSet =        OEM_CHARSET;
    pTTYInfo->lfTTYFont.lfOutPrecision =   OUT_DEFAULT_PRECIS;
    pTTYInfo->lfTTYFont.lfClipPrecision =  CLIP_DEFAULT_PRECIS;
    pTTYInfo->lfTTYFont.lfQuality =        DEFAULT_QUALITY;
    pTTYInfo->lfTTYFont.lfPitchAndFamily = FIXED_PITCH | FF_MODERN;
    pTTYInfo->lfTTYFont.lfFaceName[0] =    NULL;

    /* set TTYInfo handle before any further message processing. */

    SetWindowLong (hWnd, GWL_PTTYINFO, (LPARAM) pTTYInfo);

    /* reset the character information, etc. */

    ResetTTYScreen (hWnd, pTTYInfo);

    hMenu = GetMenu (hWnd);
    return ((LRESULT) TRUE);
}



/*---------------------------------------------------------------------------
 *  BOOL NEAR DestroyTTYInfo( HWND hWnd )
 *
 *  Description:
 *     Destroys block associated with TTY window handle.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *-------------------------------------------------------------------------*/

LOCAL BOOL NEAR 
DestroyTTYInfo (HWND hWnd)
{
	PTTYINFO			pTTYInfo;

	pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
	if (pTTYInfo == NULL)
		return (FALSE);


	DeleteObject (pTTYInfo->hTTYFont);

	MemFree (pTTYInfo);
	return (TRUE);
}



/*---------------------------------------------------------------------------
 *  void  ResizeTTYScreen( HWND hWnd, PTTYINFO pTTYInfo, 
 *					int newNrow, int newNColumn);
 *
 *  Description:
 *		Resize the screen to new size, copying data.
 *
 *  Parameters:
 *     PTTYINFO  pTTYInfo
 *        pointer to TTY info structure
 *		newNCo.umn, newNRow
 *			new size of screen.
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL
ResizeTTYScreen (HWND hWnd, PTTYINFO pTTYInfo, int newNRow, int newNColumn)
{
    BYTE		*	pNewAttrib;
    CHAR		*	pNewScreen;
    void		*	source;
    void		*	dest;
    size_t			len;
    int				cells;
    int				r;
    extern TERM			term;

    cells = newNColumn * newNRow;
    pNewScreen = (CHAR *)MemAlloc (cells * sizeof (CHAR));
    if (pNewScreen == NULL)
	return (FALSE);
    pNewAttrib = (CharAttrib *)MemAlloc (cells * sizeof (CharAttrib));
    if (pNewAttrib == NULL) {
	MemFree ((void *)pNewScreen);
	return (FALSE);
    }


    /* 
     * Clear new screen. 
     */

    assert (sizeof (CHAR) == 1);
    assert (sizeof (CharAttrib) == 1);
    _fmemset ((void *)pNewScreen, ' ', cells);
    _fmemset ((void *)pNewAttrib, 0, cells);


    /* 
     * Copy old screen onto new screen. 
     */
    if (pTTYInfo->pScreen != NULL) {

	for (r = 1; r <= newNRow && r <= pTTYInfo->actNRow; ++r) {

		
	    source = pTTYInfo->pScreen + ((pTTYInfo->actNRow - r) * 
						    pTTYInfo->actNColumn);
	    dest = pNewScreen + ((newNRow - r) * newNColumn);
	    len = MIN (newNColumn, pTTYInfo->actNColumn) * sizeof (CHAR);
	    _fmemcpy (dest, source, len);

	    
	    source = pTTYInfo->pAttrib + ((pTTYInfo->actNRow - r) * 
						    pTTYInfo->actNColumn);
	    dest = pNewAttrib + ((newNRow - r) * newNColumn);
	    len = MIN (newNColumn, pTTYInfo->actNColumn) * sizeof(CharAttrib);
	    _fmemcpy (dest, source, len);

	}
	pTTYInfo->nColumn = MIN (pTTYInfo->nColumn, newNColumn);
	pTTYInfo->nRow = MAX (0, 
			pTTYInfo->nRow + (newNRow - pTTYInfo->actNRow));
	MemFree (pTTYInfo->pScreen);
	MemFree (pTTYInfo->pAttrib);
    }
    else {
	pTTYInfo->nColumn = MIN (pTTYInfo->nColumn, newNColumn);
	pTTYInfo->nRow = MIN (pTTYInfo->nRow, newNRow);
    }
    pTTYInfo->pScreen = pNewScreen;
    pTTYInfo->pAttrib = pNewAttrib;
    pTTYInfo->actNColumn = newNColumn;
    pTTYInfo->actNRow = newNRow;
    
    
    /* Repaint whole screen. */
    pTTYInfo->screenDirty = TRUE;
    pTTYInfo->eraseScreen = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (pTTYInfo, hWnd, NULL);
#else
    InvalidateRect (hWnd, NULL, FALSE);
#endif

    
    
    /* xxx pico specific. */
    if (term.t_nrow == 0) {
	term.t_nrow = newNRow - 1;
	term.t_ncol = newNColumn;
    }
    

    return (TRUE);
}
		
		




/*---------------------------------------------------------------------------
 *  BOOL  ResetTTYScreen( HWND hWnd, PTTYINFO pTTYInfo )
 *
 *  Description:
 *     Resets the TTY character information and causes the
 *     screen to resize to update the scroll information.
 *
 *  Parameters:
 *     PTTYINFO  pTTYInfo
 *        pointer to TTY info structure
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
ResetTTYScreen (HWND hWnd, PTTYINFO pTTYInfo)
{
    HDC			hDC;
    TEXTMETRIC		tm;
    int			newNRow;
    int			newNColumn;
    BOOL		newsize;
    int			i;
/*	RECT		rcWindow;*/


    if (NULL == pTTYInfo)
	    return (FALSE);

    if (NULL != pTTYInfo->hTTYFont)
	    DeleteObject (pTTYInfo->hTTYFont);

    pTTYInfo->hTTYFont = CreateFontIndirect (&pTTYInfo->lfTTYFont);

    hDC = GetDC (hWnd);
    SelectObject (hDC, pTTYInfo->hTTYFont);
    GetTextMetrics (hDC, &tm);
    ReleaseDC (hWnd, hDC);

    pTTYInfo->xChar = tm.tmAveCharWidth;
    pTTYInfo->yChar = tm.tmHeight + tm.tmExternalLeading;

    /* Update the current number of rows and cols. */
    newNRow = MIN (MAXNROW, pTTYInfo->ySize / pTTYInfo->yChar);
    newNColumn = MIN (MAXNCOLUMN, pTTYInfo->xSize / pTTYInfo->xChar);

    newsize = newNRow != pTTYInfo->actNRow || 
		    newNColumn != pTTYInfo->actNColumn;
    if (newsize)
	    ResizeTTYScreen (hWnd, pTTYInfo, newNRow, newNColumn);

    /* Resize the carrot as well. */
    if (pTTYInfo->wCursorState == CS_VISIBLE) {
	    HideCaret (hWnd);
	    DestroyCaret();
	    CreateCaret (hWnd, NULL, pTTYInfo->xChar, pTTYInfo->yChar);
	    SetCaretPos ((pTTYInfo->nColumn * pTTYInfo->xChar) - pTTYInfo->xOffset,
		    (pTTYInfo->nRow * pTTYInfo->yChar) - pTTYInfo->yOffset);
	    ShowCaret (hWnd);
    }

    /* Redraw screen and, if the "size" changed, tell the upper layers. */
    pTTYInfo->screenDirty = TRUE;
    pTTYInfo->eraseScreen = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (pTTYInfo, hWnd, NULL);
#else
    InvalidateRect (hWnd, NULL, FALSE);
#endif
    if (newsize) {
	for (i = 0; i < RESIZE_CALLBACK_ARRAY_SIZE; ++i) {
	    if (pTTYInfo->resizer[i] != NULL) 
		pTTYInfo->resizer[i](pTTYInfo->actNRow, pTTYInfo->actNColumn);
	}
    }



#ifdef SCROLL
    /* a slimy hack to force the scroll position, region to
     * be recalculated based on the new character sizes. */

    GetWindowRect( hWnd, &rcWindow );
    SendMessage( hWnd, WM_SIZE, SIZENORMAL,
	    (LPARAM) MAKELONG( rcWindow.right - rcWindow.left,
	    rcWindow.bottom - rcWindow.top ) );
#endif

    return (TRUE);
}



#ifdef OWNRECT
void
MyInvalidateRect (PTTYINFO pTTYInfo, HWND hWnd, RECT *r)
{
    RECT	u;
    
    if (r == NULL) {
	SetRect (&pTTYInfo->rPaintRect, 0, 0, pTTYInfo->xSize, pTTYInfo->ySize);
    }
    else {
	UnionRect (&u, &pTTYInfo->rPaintRect, r);
	CopyRect (&pTTYInfo->rPaintRect, &u);
    }
}
#endif





/*---------------------------------------------------------------------------
 *  BOOL  PaintTTY( HWND hWnd )
 *
 *  Description:
 *     Paints the rectangle determined by the paint struct of
 *     the DC.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window (as always)
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL
PaintTTY (HWND hWnd) 
{
    int          nRow, nCol;		/* Top right corner of update. */
    int		 nEndRow, nEndCol;	/* lower right corner of update. */
    int		 nCount;		/* Number of columns in each row. */
    int		 nHorzPos, nVertPos;	/* Position of each text write. */
    int		 col;			/* start col of run of similar attr */
    int		 count;			/* count of run of similar attrib. */
    int		 endCount;		/* How far to count. */
    CharAttrib	*pAttrib;
    HDC          hDC ;
    HFONT        hOldFont;
    PTTYINFO    pTTYInfo;
    PAINTSTRUCT  ps;
    RECT         rect;
    RECT	 erect;
    HBRUSH	 hBrush;
    long	 offset;		/* Offset into screen and attrib */
    CharAttrib	 lastAttrib;		/* Attributes of last text write. */
    CharAttrib	 newAttrib;		/* Attributes of this text write. */

    pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
    if (pTTYInfo == NULL)
	return (FALSE);

    hDC = BeginPaint (hWnd, &ps);
    hOldFont = SelectObject (hDC, pTTYInfo->hTTYFont);
    SetTextColor (hDC, pTTYInfo->rgbFGColor);
    SetBkColor (hDC, pTTYInfo->rgbBGColor);
    SetBkMode (hDC, OPAQUE);
    rect = ps.rcPaint;

    nRow = min (pTTYInfo->actNRow - 1,
	   max (0, (rect.top + pTTYInfo->yOffset) / pTTYInfo->yChar));
    nEndRow = min (pTTYInfo->actNRow - 1,
	    ((rect.bottom + pTTYInfo->yOffset - 1) / pTTYInfo->yChar));
    nCol = min (pTTYInfo->actNColumn - 1,
	    max (0, (rect.left + pTTYInfo->xOffset) / pTTYInfo->xChar));
    nEndCol = min (pTTYInfo->actNColumn - 1,
	    ((rect.right + pTTYInfo->xOffset - 1) / pTTYInfo->xChar));

    nCount = nEndCol - nCol + 1;
    lastAttrib = CHAR_ATTR_NORM;
    
    
    /* Erase screen if necessary. */
    if (pTTYInfo->eraseScreen) {
	erect.top = 0;
	erect.left = 0;
	erect.bottom = pTTYInfo->ySize;
	erect.right = pTTYInfo->xSize;
	hBrush = CreateSolidBrush (pTTYInfo->rgbBGColor);
	if (hBrush != NULL) {
	    FillRect (hDC, &erect, hBrush);
	    DeleteObject (hBrush);
	}
	pTTYInfo->eraseScreen = FALSE;
    }
    else {
	/* If not erasing entire screen, we may need to hand paint the
	 * left over area to the right and below the characters.  Otherwise
	 * Windows fills it with the default background color. */
	if (pTTYInfo->actNColumn * pTTYInfo->xChar < rect.right) {
	    erect.top = rect.top;
	    erect.left = pTTYInfo->actNColumn * pTTYInfo->xChar;
	    erect.bottom = rect.bottom;
	    erect.right = rect.right;
	    hBrush = CreateSolidBrush (pTTYInfo->rgbBGColor);
	    if (hBrush != NULL) {
		FillRect (hDC, &erect, hBrush);
		DeleteObject (hBrush);
	    }
	}
	if (pTTYInfo->actNRow * pTTYInfo->yChar < rect.bottom) {
	    erect.top = pTTYInfo->actNRow * pTTYInfo->yChar;
	    erect.left = rect.left;
	    erect.bottom = rect.bottom;
	    erect.right = rect.right;
	    hBrush = CreateSolidBrush (pTTYInfo->rgbBGColor);
	    if (hBrush != NULL) {
		FillRect (hDC, &erect, hBrush);
		DeleteObject (hBrush);
	    }
	}
    }
    
    
    /* Paint rows of text. */
    for (; nRow <= nEndRow; nRow++)    {
	nVertPos = (nRow * pTTYInfo->yChar) - pTTYInfo->yOffset;
	rect.top = nVertPos;
	rect.bottom = nVertPos + pTTYInfo->yChar;
	
	/* Paint runs of similar attributes. */
	col = nCol;				/* Start at left. */
	while (col <= nEndCol) {		/* While not past right. */

	    /* Starting with Character at nRow, col, what is its attribute? */
	    offset = (nRow * pTTYInfo->actNColumn) + col;
	    newAttrib = *(pTTYInfo->pAttrib + offset);
	    
	    if (newAttrib != lastAttrib) {
		/* Set new attributes. */
		if (newAttrib & CHAR_ATTR_REV) {
		    SetTextColor (hDC, pTTYInfo->rgbRFGColor);
		    SetBkColor (hDC, pTTYInfo->rgbRBGColor);
		}
		else {
		    SetTextColor (hDC, pTTYInfo->rgbFGColor);
		    SetBkColor (hDC, pTTYInfo->rgbBGColor);
		}
	    }
	    
	    /* Find run of similar attributes. */
	    count = 1;
	    pAttrib = pTTYInfo->pAttrib + offset + 1;
	    endCount = nEndCol - col;
	    while (count <= endCount && *pAttrib++ == newAttrib) 
		++count;
	
	    /* Paint run of characters from nRow, col to nRow, col + count 
	     * rect.top and rect.bottom have already been calculated. */
	    nHorzPos = (col * pTTYInfo->xChar) - pTTYInfo->xOffset;
	    rect.left = nHorzPos;
	    rect.right = nHorzPos + pTTYInfo->xChar * count;
	    ExtTextOut (hDC, nHorzPos, nVertPos, ETO_OPAQUE | ETO_CLIPPED, &rect,
		(LPSTR)(pTTYInfo->pScreen + offset),
		count, NULL);
	
	    /* Move pointer to end of this span of characters. */
	    col += count;
	    lastAttrib = newAttrib;
        }
    }

    SelectObject (hDC, hOldFont);
    EndPaint (hWnd, &ps);
    MoveTTYCursor (hWnd);
    pTTYInfo->screenDirty = FALSE;
#ifdef OWNRECT
    SetRectEmpty (&pTTYInfo->rPaintRect);
#endif
    return (TRUE);
}

/*---------------------------------------------------------------------------
 *  BOOL  GetMinMaxInfoTTY (HWND hWnd, (MINMAXINFO __far *)lParam)
 *
 *  Description:
 *     Return the min and max size that the window can be.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     MINMAXINFO
 *	  Info structure that Windows would like us to fill.
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
GetMinMaxInfoTTY (HWND hWnd, MINMAXINFO __far *lpmmi)
{
    PTTYINFO		pTTYInfo;
    
    
    pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
    if (pTTYInfo == NULL)
	return (FALSE);

    lpmmi->ptMaxTrackSize.x = lpmmi->ptMaxSize.x = MIN (lpmmi->ptMaxSize.x,
			    pTTYInfo->xChar * MAXNCOLUMN + WIN_X_BORDER_SIZE);
    lpmmi->ptMaxTrackSize.y = lpmmi->ptMaxSize.y = MIN (lpmmi->ptMaxSize.y, 
			    pTTYInfo->yChar * MAXNROW + WIN_Y_BORDER_SIZE);

    lpmmi->ptMinTrackSize.x = MAX (WIN_MIN_X_SIZE, 
		    pTTYInfo->xChar * MINNCOLUMN + WIN_X_BORDER_SIZE);
    lpmmi->ptMinTrackSize.y = MAX (WIN_MIN_Y_SIZE,
		    pTTYInfo->yChar * MINNROW + WIN_Y_BORDER_SIZE);
    return (TRUE);
}
	


/*---------------------------------------------------------------------------
 *  BOOL  SizeTTY( HWND hWnd, int fwSizeType, WORD wVertSize, WORD wHorzSize )
 *
 *  Description:
 *     Sizes TTY and sets up scrolling regions.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     WORD wVertSize
 *        new vertical size
 *
 *     WORD wHorzSize
 *        new horizontal size
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
SizeTTY (HWND hWnd, int fwSizeType, WORD wVertSize, WORD wHorzSize)
{
/*  int		nScrollAmt ;*/
    PTTYINFO	pTTYInfo;
    int		newNColumn;
    int		newNRow;
    int		i;



    pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
    if (pTTYInfo == NULL)
	    return ( FALSE );

    
    /*
     * Is the window being minimized?
     */
    if (fwSizeType == SIZE_MINIMIZED) {
	pTTYInfo->fMinimized = TRUE;
	return (TRUE);
    }
    
    pTTYInfo->fMinimized = FALSE;
	    
	    
    pTTYInfo->ySize = (int) wVertSize;
    newNRow = min (MAXNROW, pTTYInfo->ySize / pTTYInfo->yChar);


#ifdef SCROLL
    pTTYInfo->yScroll = max (0, (MAXROWS * pTTYInfo->yChar)) -
	    pTTYInfo->ySize);
    nScrollAmt = min (pTTYInfo->yScroll, pTTYInfo->yOffset) -
	    pTTYInfo->yOffset);
    ScrollWindow (hWnd, 0, -nScrollAmt, NULL, NULL);

    pTTYInfo->yOffset = pTTYInfo->yOffset + nScrollAmt;
    SetScrollPos (hWnd, SB_VERT, pTTYInfo->yOffset, FALSE);
    SetScrollRange (hWnd, SB_VERT, 0, pTTYInfo->yScroll, TRUE);
#else
    pTTYInfo->yOffset = 0;
#endif	

    pTTYInfo->xSize = (int) wHorzSize;
    newNColumn = min (MAXNCOLUMN, pTTYInfo->xSize / pTTYInfo->xChar);

#ifdef SCROLL
    pTTYInfo->xScroll = max (0, (MAXCOLS * pTTYInfo->xChar) -
	    pTTYInfo->xSize);
    nScrollAmt = min (pTTYInfo->xScroll, pTTYInfo->xOffset) -
	    pTTYInfo->xOffset);
    ScrollWindow (hWnd, 0, -nScrollAmt, NULL, NULL);
    pTTYInfo->xOffset = pTTYInfo->xOffset + nScrollAmt;
    SetScrollPos (hWnd, SB_HORZ, pTTYInfo->xOffset, FALSE);
    SetScrollRange (hWnd, SB_HORZ, 0, pTTYInfo->xScroll, TRUE);
#else
    pTTYInfo->xOffset = 0;
#endif


    ResizeTTYScreen (hWnd, pTTYInfo, newNRow, newNColumn);
    pTTYInfo->screenDirty = TRUE;
    pTTYInfo->eraseScreen = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (pTTYInfo, hWnd, NULL);
#else
    InvalidateRect (hWnd, NULL, FALSE);
#endif

    for (i = 0; i < RESIZE_CALLBACK_ARRAY_SIZE; ++i) {
	if (pTTYInfo->resizer[i] != NULL) 
		pTTYInfo->resizer[i](pTTYInfo->actNRow, pTTYInfo->actNColumn);
    }

    return (TRUE);
}



/*---------------------------------------------------------------------------
 *  BOOL  SetTTYFocus( HWND hWnd )
 *
 *  Description:
 *     Sets the focus to the TTY window also creates caret.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
SetTTYFocus (HWND hWnd)
{
	PTTYINFO  pTTYInfo;

	pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
	if (pTTYInfo == NULL)
		return (FALSE);

	pTTYInfo->fFocused = TRUE;
	pTTYInfo->wCursorState |= CS_FOCUSED;
	
	if (pTTYInfo->wCursorState == CS_VISIBLE) {
		CreateCaret (hWnd, NULL, pTTYInfo->xChar, pTTYInfo->yChar);
		ShowCaret (hWnd);
	}
	
	MoveTTYCursor (hWnd);
	return (TRUE);
}




/*---------------------------------------------------------------------------
 *  BOOL  KillTTYFocus( HWND hWnd )
 *
 *  Description:
 *     Kills TTY focus and destroys the caret.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
KillTTYFocus (HWND hWnd)
{
	PTTYINFO  pTTYInfo;

	pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
	if (pTTYInfo == NULL)
		return (FALSE);

	if (pTTYInfo->wCursorState == CS_VISIBLE) {
		HideCaret (hWnd);
		DestroyCaret();
	}
	
	pTTYInfo->wCursorState &= ~CS_FOCUSED;
	pTTYInfo->fFocused = FALSE;

	return (TRUE);
}




/*---------------------------------------------------------------------------
 *  BOOL  MoveTTYCursor( HWND hWnd )
 *
 *  Description:
 *     Moves caret to current position.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
MoveTTYCursor (HWND hWnd)
{
    PTTYINFO  pTTYInfo;

    pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
    if (pTTYInfo == NULL)
	    return (FALSE);

    if (pTTYInfo->wCursorState == CS_VISIBLE && 
	    !pTTYInfo->fMassiveUpdate) {
	HideCaret (hWnd);
	SetCaretPos ((pTTYInfo->nColumn * pTTYInfo->xChar) - pTTYInfo->xOffset,
		(pTTYInfo->nRow * pTTYInfo->yChar) - pTTYInfo->yOffset);
	ShowCaret (hWnd);
    }

    return (TRUE);
}




/*---------------------------------------------------------------------------
 *  BOOL  ProcessTTYKeyDown ( HWND hWnd, WORD bOut, DWORD keyData )
 *
 *  Description:
 *	Called to process MW_KEYDOWN message.  We are only interested in 
 *	virtual keys that pico/pine use.  All others get passed on to 
 *	the default message handler.  Regular key presses will return 
 *	latter as a WM_CHAR message, with SHIFT and CONTROL processing
 *	already done.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     BYTE key
 *	  Virtual key code.
 *
 *     DWORD keyData
 *	  Additional flags passed in lParam for WM_KEYDOWN
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
ProcessTTYKeyDown (HWND hWnd, WORD key, DWORD keyData)
{
    WORD		myKey;
    
    
    /* Special keys. */
    if (keyData & 0X20000000)
	return (FALSE);			/* Message NOT handled. */

    switch (key) {
	case VK_UP:		myKey = MSWIN_KEY_UP;		break;
	case VK_DOWN:		myKey = MSWIN_KEY_DOWN;		break;
	case VK_RIGHT:		myKey = MSWIN_KEY_RIGHT;	break;
	case VK_LEFT:		myKey = MSWIN_KEY_LEFT;		break;
	case VK_PRIOR:		myKey = MSWIN_KEY_PREVPAGE;	break;
	case VK_NEXT:		myKey = MSWIN_KEY_NEXTPAGE;	break;
	case VK_HOME:		myKey = MSWIN_KEY_HOME;		break;
	case VK_END:		myKey = MSWIN_KEY_END;		break;
	case VK_DELETE:		myKey = MSWIN_KEY_DELETE;	break;
	case VK_F1:		myKey = MSWIN_KEY_F1;		break;
	case VK_F2:		myKey = MSWIN_KEY_F2;		break;
	case VK_F3:		myKey = MSWIN_KEY_F3;		break;
	case VK_F4:		myKey = MSWIN_KEY_F4;		break;
	case VK_F5:		myKey = MSWIN_KEY_F5;		break;
	case VK_F6:		myKey = MSWIN_KEY_F6;		break;
	case VK_F7:		myKey = MSWIN_KEY_F7;		break;
	case VK_F8:		myKey = MSWIN_KEY_F8;		break;
	case VK_F9:		myKey = MSWIN_KEY_F9;		break;
	case VK_F10:		myKey = MSWIN_KEY_F10;		break;
	case VK_F11:		myKey = MSWIN_KEY_F11;		break;
	case VK_F12:		myKey = MSWIN_KEY_F12;		break;
	
	/* Control is special - I keep track, but do not claim to handle. */
	case VK_CONTROL:	KeyControlDown = TRUE;
				return (FALSE);
				
	default:		return (FALSE);	/* Message NOT handled.*/
    }

    CQAdd (myKey, 0);
    return (TRUE);			/* Message handled .*/
}



/*---------------------------------------------------------------------------
 *  BOOL  ProcessTTYKeyUp ( HWND hWnd, WORD bOut, DWORD keyData )
 *
 *  Description:
 *	Called to process MW_KEYDOWN message. 
 *	Used only to detect when the control key goes up.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     BYTE key
 *	  Virtual key code.
 *
 *     DWORD keyData
 *	  Additional flags passed in lParam for WM_KEYDOWN
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
ProcessTTYKeyUp (HWND hWnd, WORD key, DWORD keyData)
{
    WORD		myKey;
    
    
    /* Special keys. */
    if (keyData & 0X20000000)
	return (FALSE);			/* Message NOT handled. */

    if (key == VK_CONTROL) 
	KeyControlDown = FALSE;
				
    return (FALSE);	/* Message NOT handled.*/
}




/*---------------------------------------------------------------------------
 *  BOOL  ProcessTTYCharacter( HWND hWnd, WORD bOut, DWORD keyData )
 *
 *  Description:
 *		Place the character into a queue.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     BYTE bOut
 *        byte from keyboard
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
ProcessTTYCharacter (HWND hWnd, WORD bOut, DWORD keyData)
{
    if (bOut == ' ' && KeyControlDown)
	    bOut = '\0';
    CQAdd (bOut, keyData);
    return (TRUE);		/* Message handled. */
}





/*---------------------------------------------------------------------------
 *  BOOL  ProcessTTYMouse( HWND hWnd, BYTE bOut )
 *
 *  Description:
 *		Place the Mouse into a queue.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     BYTE bOut
 *        byte from keyboard
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
ProcessTTYMouse (int mevent, int button, int xPos, int yPos, WORD keys)
{
	int		nRow;
	int		nColumn;
	
	nColumn = xPos / gpTTYInfo->xChar;
	nRow = yPos / gpTTYInfo->yChar;
	
	MQAdd (mevent, button, nRow, nColumn, keys);
	return (0);		/* Message handled. */
}





#if 0
/* Old function used when screen was fixed array. */

/*---------------------------------------------------------------------------
 *  void ScrollTTY (HWND hWnd, int scroll)
 *
 *  Description:
 *		Scroll the screen up (negative) or down (positive) by 'scroll' rows.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     int scroll
 *        number of rows to scroll
 *
/*--------------------------------------------------------------------------*/

LOCAL void
ScrollTTY (HWND hWnd, int scroll, int maxRows)

{
	PTTYINFO	pTTYInfo;
	int			ascroll;
	LPSTR		source, dest;
	size_t		len;
	
	if (scroll == 0) 
		return;
	if (scroll >= MAXROWS)
		return;

	pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
	if (pTTYInfo == NULL)
		return;
	
	ascroll = 0;
	if (scroll < 0) {
		ascroll = -scroll;
		source = (LPSTR) (pTTYInfo->abScreen + (MAXCOLS * ascroll));
		dest = (LPSTR)pTTYInfo->abScreen;
		len = MAXCOLS * (maxRows - ascroll);
		_fmemmove (dest, source, len);
		_fmemset ((LPSTR) (pTTYInfo->abScreen + (MAXCOLS * (maxRows - ascroll))),
			' ', MAXCOLS * ascroll);
		pTTYInfo->nRow = max (0, pTTYInfo->nRow - ascroll);
	}
	else {
		source = (LPSTR)pTTYInfo->abScreen;
		dest = (LPSTR) (pTTYInfo->abScreen + (MAXCOLS * scroll));
		len = MAXCOLS * (maxRows - scroll);
		_fmemmove (dest, source, len);
		_fmemset ((LPSTR)pTTYInfo->abScreen, ' ', MAXCOLS * scroll);
		pTTYInfo->nRow = min (maxRows - 1, pTTYInfo->nRow + scroll);
	}

	MoveTTYCursor (hWnd);
	pTTYInfo->screenDirty = TRUE;
	pTTYInfo->eraseScreen = TRUE;
#ifdef OWNRECT
	MyInvalidateRect (pTTYInfo, hWnd, NULL);
#else
	InvalidateRect (hWnd, NULL, FALSE);
#endif
}
#endif /* 0 */



/*---------------------------------------------------------------------------
 *  BOOL  WriteTTYBlock( HWND hWnd, LPSTR lpBlock, int nLength )
 *
 *  Description:
 *     Writes block to TTY screen.  Nothing fancy - just
 *     straight TTY.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     LPSTR lpBlock
 *        far pointer to block of data
 *
 *     int nLength
 *        length of block
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
WriteTTYBlock (HWND hWnd, LPSTR lpBlock, int nLength)
{
    int				i;
    PTTYINFO			pTTYInfo;
    RECT			rect;
    BOOL			fNewLine;
    long			offset;

    pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
    if (pTTYInfo == NULL)
	    return (FALSE);

    for (i = 0 ; i < nLength; i++) {
	switch (lpBlock[i]) {
	case ASCII_BEL:
	    // Bell
	    MessageBeep (0) ;
	    break ;

	case ASCII_BS:
	    // Backspace
	    if (pTTYInfo->nColumn > 0)
		    --pTTYInfo->nColumn;
	    MoveTTYCursor (hWnd);
	    break;

	case ASCII_CR:
	    // Carriage return
	    pTTYInfo->nColumn = 0 ;
	    MoveTTYCursor (hWnd);
	    if (!pTTYInfo->fNewLine)
		    break;

	    // fall through

	case ASCII_LF:
	    // Line feed
	    if (++pTTYInfo->nRow == pTTYInfo->actNRow) {
		/* Scroll the Screen. */
		_fmemmove ((LPSTR)pTTYInfo->pScreen,
			(LPSTR) (pTTYInfo->pScreen + pTTYInfo->actNColumn),
			((pTTYInfo->actNRow - 1) * pTTYInfo->actNColumn) *
				sizeof (CHAR));
		assert (sizeof (CHAR) == 1);
		_fmemset ((LPSTR) (pTTYInfo->pScreen + 
			    (pTTYInfo->actNRow - 1) * pTTYInfo->actNColumn),
			' ', pTTYInfo->actNColumn);

		/* Scroll the Attributes. */
		_fmemmove ((LPSTR)pTTYInfo->pAttrib,
			(LPSTR) (pTTYInfo->pAttrib + pTTYInfo->actNColumn),
			((pTTYInfo->actNRow - 1) * pTTYInfo->actNColumn) *
				sizeof (CharAttrib));
		assert (sizeof (CharAttrib) == 1);
		_fmemset ((LPSTR) (pTTYInfo->pScreen + 
			    (pTTYInfo->actNRow - 1) * pTTYInfo->actNColumn),
			0, pTTYInfo->actNColumn);


		pTTYInfo->screenDirty = TRUE;
		pTTYInfo->eraseScreen = TRUE;
#ifdef OWNRECT
		MyInvalidateRect (pTTYInfo, hWnd, NULL);
#else
		InvalidateRect (hWnd, NULL, FALSE);
#endif
		--pTTYInfo->nRow;
	    }
	    MoveTTYCursor (hWnd);
	    break;



	default:
	    offset = (pTTYInfo->nRow * pTTYInfo->actNColumn) +
		    pTTYInfo->nColumn;
	    *(pTTYInfo->pScreen + offset) = lpBlock[i];
	    *(pTTYInfo->pAttrib + offset) = pTTYInfo->curAttrib;
	    rect.left = (pTTYInfo->nColumn * pTTYInfo->xChar) -
		    pTTYInfo->xOffset;
	    rect.right = rect.left + pTTYInfo->xChar;
	    rect.top = (pTTYInfo->nRow * pTTYInfo->yChar) -
		    pTTYInfo->yOffset;
	    rect.bottom = rect.top + pTTYInfo->yChar;
	    pTTYInfo->screenDirty = TRUE;
#ifdef OWNRECT
	    MyInvalidateRect (pTTYInfo, hWnd, &rect);
#else
	    InvalidateRect (hWnd, &rect, FALSE);
#endif

	    /* Line Wrap. */
	    if (pTTYInfo->nColumn < pTTYInfo->actNColumn - 1)
		    pTTYInfo->nColumn++ ;
	    else if (pTTYInfo->autoWrap == WRAP_ON || 
		    (pTTYInfo->autoWrap == WRAP_NO_SCROLL && 
			    pTTYInfo->nRow < pTTYInfo->actNRow - 1)) {
		    fNewLine = pTTYInfo->fNewLine;
		    pTTYInfo->fNewLine = FALSE;
		    WriteTTYBlock (hWnd, "\r\n", 2);
		    pTTYInfo->fNewLine = fNewLine;
	    }
	    break;
	}
    }
    return (TRUE);
}




/*---------------------------------------------------------------------------
 *  BOOL  WriteTTYText ( HWND hWnd, LPSTR lpBlock, int nLength )
 *
 *  Description:
 *	Like WriteTTYBlock but optimized for strings that are text only,
 *	no carrage control characters. 
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     LPSTR lpBlock
 *        far pointer to block of data
 *
 *     int nLength
 *        length of block
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
WriteTTYText (HWND hWnd, LPSTR lpText, int nLength)
{
    int				i;
    PTTYINFO			pTTYInfo;
    RECT			rect;
    BOOL			fNewLine;
    long			offset;
    long			colEnd;
    long			screenEnd;
    BOOL			wraper;

    
    pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
    if (pTTYInfo == NULL)
	    return ( FALSE );

    
    /* Calculate offset of cursor, end of current column, and end of screen */
    offset = (pTTYInfo->nRow * pTTYInfo->actNColumn) + pTTYInfo->nColumn;
    colEnd = (pTTYInfo->nRow + 1) * pTTYInfo->actNColumn;
    screenEnd = pTTYInfo->actNRow * pTTYInfo->actNColumn;
    
    
    /* Text is allowed to wrap around to subsequent lines, but not past end
     * of screen */
    if (offset + nLength > screenEnd)
	nLength = screenEnd - offset;


    /* Calculate bounding rectangle. */
    if (offset + nLength <= colEnd) {
	/* Single line. */
	rect.left = (pTTYInfo->nColumn * pTTYInfo->xChar) -pTTYInfo->xOffset;
	rect.right = rect.left + (pTTYInfo->xChar * nLength);
	rect.top = (pTTYInfo->nRow * pTTYInfo->yChar) - pTTYInfo->yOffset;
	rect.bottom = rect.top + pTTYInfo->yChar;
    }
    else {
	/* Wraps across multiple lines.  Calculate one rect to cover all 
	 * lines. */
	rect.left = 0;
	rect.right = pTTYInfo->ySize;
	rect.top = (pTTYInfo->nRow * pTTYInfo->yChar) - pTTYInfo->yOffset;
	rect.bottom = ((((offset + nLength) / pTTYInfo->actNColumn) + 1) * 
			pTTYInfo->yChar) - pTTYInfo->yOffset;
    }
    
    
    /* Apply text and attributes to screen in one smooth motion. */
    _fmemcpy (pTTYInfo->pScreen + offset, lpText, nLength);
    _fmemset (pTTYInfo->pAttrib + offset, pTTYInfo->curAttrib, nLength);
    
    
    /* New cursor position. */
    offset += nLength;
    if (offset == screenEnd) {
	pTTYInfo->nRow = pTTYInfo->actNRow - 1;
	pTTYInfo->nColumn = pTTYInfo->actNColumn - 1;
    }
    else {
	pTTYInfo->nRow = offset / pTTYInfo->actNColumn;
	pTTYInfo->nColumn = offset % pTTYInfo->actNColumn;
    }
    
 
    /* Invalidate rectangle */
    pTTYInfo->screenDirty = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (pTTYInfo, hWnd, &rect);
#else
    InvalidateRect (hWnd, &rect, FALSE);
#endif
    return (TRUE);
}


/*---------------------------------------------------------------------------
 *  BOOL  WriteTTYChar (HWND hWnd, char ch)
 *
 *  Description:
 *	Write a single character to the cursor position and advance the
 *	cursor.  Does not handle carage control.
 *
 *  Parameters:
 *     HWND hWnd
 *        handle to TTY window
 *
 *     char ch
 *	  character being written.
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL  
WriteTTYChar (HWND hWnd, char ch)
{
    int				i;
    PTTYINFO			pTTYInfo;
    RECT			rect;
    BOOL			fNewLine;
    long			offset;
    long			colEnd;
    long			screenEnd;
    BOOL			wraper;

    pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
    if (pTTYInfo == NULL)
	return (FALSE);

#if 0
   /* xxxx for debugging only to provide a bread point for updates to status
    * xxxx line. */
    if (pTTYInfo->nRow == pTTYInfo->actNRow - 3)
	wraper = TRUE;
#endif

    offset = (pTTYInfo->nRow * pTTYInfo->actNColumn) +
	    pTTYInfo->nColumn;
    
    *(pTTYInfo->pScreen + offset) = ch;
    *(pTTYInfo->pAttrib + offset) = pTTYInfo->curAttrib;

    rect.left = (pTTYInfo->nColumn * pTTYInfo->xChar) - pTTYInfo->xOffset;
    rect.right = rect.left + pTTYInfo->xChar;
    rect.top = (pTTYInfo->nRow * pTTYInfo->yChar) - pTTYInfo->yOffset;
    rect.bottom = rect.top + pTTYInfo->yChar;
    pTTYInfo->screenDirty = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (pTTYInfo, hWnd, &rect);
#else
    InvalidateRect (hWnd, &rect, FALSE);
#endif
    
    

    /* Line Wrap. */
    if (pTTYInfo->nColumn < pTTYInfo->actNColumn - 1)
	pTTYInfo->nColumn++ ;
    else if ((pTTYInfo->autoWrap == WRAP_ON || 
	      pTTYInfo->autoWrap == WRAP_NO_SCROLL) && 
		    pTTYInfo->nRow < pTTYInfo->actNRow - 1) {
       pTTYInfo->nRow++;
       pTTYInfo->nColumn = 0;
    }
    return (TRUE);
}

/*---------------------------------------------------------------------------
 *  VOID  GoModalDialogBoxParam( HINSTANCE hInstance,
 *                                   LPCSTR lpszTemplate, HWND hWnd,
 *                                   DLGPROC lpDlgProc, LPARAM lParam )
 *
 *  Description:
 *     It is a simple utility function that simply performs the
 *     MPI and invokes the dialog box with a DWORD paramter.
 *
 *  Parameters:
 *     similar to that of DialogBoxParam() with the exception
 *     that the lpDlgProc is not a procedure instance
 *
/*--------------------------------------------------------------------------*/

LOCAL VOID  
GoModalDialogBoxParam( HINSTANCE hInstance, LPCSTR lpszTemplate,
                                 HWND hWnd, DLGPROC lpDlgProc, LPARAM lParam )
{
   DLGPROC  lpProcInstance ;

   lpProcInstance = (DLGPROC) MakeProcInstance( (FARPROC) lpDlgProc,
                                                hInstance ) ;
   DialogBoxParam( hInstance, lpszTemplate, hWnd, lpProcInstance, lParam ) ;
   FreeProcInstance( (FARPROC) lpProcInstance ) ;
}





/*---------------------------------------------------------------------------
 *  BOOL FAR PASCAL __export AboutDlgProc( HWND hDlg, UINT uMsg,
 *                                WPARAM wParam, LPARAM lParam )
 *
 *  Description:
 *     Simulates the Windows System Dialog Box.
 *
 *  Parameters:
 *     Same as standard dialog procedures.
 *
/*--------------------------------------------------------------------------*/

BOOL FAR PASCAL __export 
AboutDlgProc (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
   switch (uMsg) {
      case WM_INITDIALOG:
      {
         int          idModeString ;
         char         szTemp [81];
         DWORD        dwFreeMemory, dwWinFlags ;
         WORD         wFreeResources, wRevision, wVersion ;

#ifdef ABOUTDLG_USEBITMAP
         // if we are using the bitmap, hide the icon

         ShowWindow( GetDlgItem( hDlg, IDD_ABOUTICON ), SW_HIDE ) ;
#endif
         // sets up the version number for Windows

         wVersion = LOWORD (GetVersion());
         switch (HIBYTE (wVersion)) {
            case 10:
               wRevision = 1;
               break;

            default:
               wRevision = 0;
               break;
         }
         wVersion &= 0xFF;

	 
         /* sets up version number for PINE */

         GetDlgItemText (hDlg, IDD_VERSION, szTemp, sizeof (szTemp));
         wsprintf (TempBuf, szTemp, VER_MAJOR, VER_MINOR, VER_BUILD);
         SetDlgItemText (hDlg, IDD_VERSION, (LPSTR) TempBuf);

         // get by-line

         LoadString (GETHINST (hDlg), IDS_BYLINE, TempBuf,
                     sizeof (TempBuf));
         SetDlgItemText (hDlg, IDD_BYLINE, TempBuf);

      }
      return ( TRUE ) ;

#ifdef ABOUTDLG_USEBITMAP
      // used to paint the bitmap

      case WM_PAINT:
      {
         HBITMAP      hBitMap ;
         HDC          hDC, hMemDC ;
         PAINTSTRUCT  ps ;

         // load bitmap and display it

         hDC = BeginPaint( hDlg, &ps ) ;
         if (NULL != (hMemDC = CreateCompatibleDC( hDC )))
         {
            hBitMap = LoadBitmap( GETHINST( hDlg ),
                                  MAKEINTRESOURCE( PINEBITMAP ) ) ;
            hBitMap = SelectObject( hMemDC, hBitMap ) ;
            BitBlt( hDC, 10, 10, 64, 64, hMemDC, 0, 0, SRCCOPY ) ;
            DeleteObject( SelectObject( hMemDC, hBitMap ) ) ;
            DeleteDC( hMemDC ) ;
         }
         EndPaint( hDlg, &ps ) ;
      }
      break ;
#endif

      case WM_COMMAND:
         if ((WORD) wParam == IDD_OK)
         {
            EndDialog( hDlg, TRUE ) ;
            return ( TRUE ) ;
         }
         break;
   }
   return ( FALSE ) ;

} // end of AboutDlgProc()









/*---------------------------------------------------------------------------
 *  BOOL  SelectTTYFont( HWND hDlg )
 *
 *  Description:
 *     Selects the current font for the TTY screen.
 *     Uses the Common Dialog ChooseFont() API.
 *
 *  Parameters:
 *     HWND hDlg
 *        handle to settings dialog
 *
/*--------------------------------------------------------------------------*/

BOOL  
LOCAL SelectTTYFont (HWND hWnd)
{
	CHOOSEFONT		cfTTYFont;
	PTTYINFO		pTTYInfo;

	pTTYInfo = (PTTYINFO) GetWindowLong (hWnd, GWL_PTTYINFO);
	if (pTTYInfo == NULL)
		return (FALSE);

	cfTTYFont.lStructSize    = sizeof (CHOOSEFONT);
	cfTTYFont.hwndOwner      = hWnd ;
	cfTTYFont.hDC            = NULL ;
	cfTTYFont.rgbColors      = pTTYInfo->rgbFGColor;
	cfTTYFont.lpLogFont      = &pTTYInfo->lfTTYFont;
	cfTTYFont.Flags          = CF_SCREENFONTS | CF_FIXEDPITCHONLY |
		CF_EFFECTS | CF_INITTOLOGFONTSTRUCT | CF_ANSIONLY | 
		CF_FORCEFONTEXIST | CF_LIMITSIZE;
	cfTTYFont.nSizeMin		 = 5;
	cfTTYFont.nSizeMax		 = 21;
	cfTTYFont.lCustData      = NULL ;
	cfTTYFont.lpfnHook       = NULL ;
	cfTTYFont.lpTemplateName = NULL ;
	cfTTYFont.hInstance      = GETHINST (hWnd);
	
	

	if (ChooseFont (&cfTTYFont)) {
		pTTYInfo->rgbFGColor = cfTTYFont.rgbColors;
		ResetTTYScreen (hWnd, pTTYInfo);
	}

	return (TRUE);

}



/*
 * Set a specific color (forground, background, reverse, normal) to
 * the color specified by name. 
 */
LOCAL void
SetColorAttribute (COLORREF *cf, char *colorName)
{
    MSWINColor		*ct;
    int			i;
    
    for (ct = MSWINColorTable; ct->colorName != NULL; ++ct) {
	if (strcmpi (colorName, ct->colorName) == 0) goto FoundColor;
    }
    
    /* color name not in table.  Try converting RGB string. */
    ConvertRGBString (colorName, cf);
    return;
    
FoundColor:
    *cf = ct->colorRef;

    /* Redraw screen. */
    gpTTYInfo->screenDirty = TRUE;
    gpTTYInfo->eraseScreen = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (gpTTYInfo, hWnd, NULL);
#else
    InvalidateRect (ghTTYWnd, NULL, FALSE);
#endif
}




/*
 * Convert a RGB string to a color ref.  The string should look like:
 *    rrr,ggg,bbb
 * where rrr, ggg, and bbb are numbers between 0 and 255 that represent
 * red, gree, and blue values.  Must be comma seperated.
 * Returns:
 *	TRUE	- Successfully converted string.
 *	FALSE	- Bad format, 'cf' unchanged.
 */

LOCAL BOOL
ConvertRGBString (char *colorName, COLORREF *cf)
{
    int		cv;
    char	*c;
    char	*s;
    char	*p;
    char	cpy[16];
    int		rgb[3];
   
    /* Some basic tests. */
    if (colorName == NULL) return (FALSE);		/* Not Null? */
    if (strlen (colorName) > 11) return (FALSE);	/* Not too long? */
    for (s = colorName; *s != '\0'; ++s) {
	if (!isdigit (*s) && *s != ',') return (FALSE); /* Valid characters?*/
    }
    
    /* Work with a copy of string. */
    strcpy (cpy, colorName);
    s = cpy;				/* Start at beginning. */
    for (cv = 0; cv < 3; ++cv) {	/* Get three digits. */
	if (cv < 2) {			/* Expect only two commas. */
	    c = strchr (s, ',');		/* Find next comma. */
	    if (c == NULL) return (FALSE);
	    *c = '\0';
	}
	if (*s == ',' || *s == '\0') return (FALSE);
	if (strlen (s) > 3) return (FALSE);
	rgb[cv] = atoi (s);
	if (rgb[cv] < 0 || rgb[cv] > 255) return (FALSE);
	s = c + 1;
    }

    *cf = RGB (rgb[0], rgb[1], rgb[2]);
    printf ("%d, %d, %d  ", rgb[0], rgb[1], rgb[2]);
    return (TRUE);
}
	   
    

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *                  Upper Layer Screen routines.
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

/*
 * Flush the write accumulator buffer.
 */
LOCAL void
FlushWriteAccum (void)
{
    if (gpTTYInfo->writeAccumCount > 0) {
	WriteTTYText (ghTTYWnd, gpTTYInfo->writeAccum, 
					gpTTYInfo->writeAccumCount);
	gpTTYInfo->writeAccumCount = 0;
    }
}



/*
 *	Called to get mouse event.
 */
int
WinGetMEvent (MEvent * pMouse)
{
	return (MQGet (pMouse));
}


/*
 *	Called to see if there is a character available.
 */
int
mswin_charavail (void)
{
    MSG		msg;
    
    if (GetMessage (&msg, NULL, 0, 0)) {
	if (ghAccel == NULL || 
		!TranslateAccelerator (ghTTYWnd, ghAccel, &msg)) {
	    TranslateMessage (&msg);
	    DispatchMessage (&msg);
	}
    }

    return (CQAvailable ());
}


/*
 *	Called in idle loops.   Dispatch messages but do not yeild control
 */
void
WinIdle (void)
{
    MSG		msg;
    
    if (PeekMessage (&msg, NULL, 0, 0, PM_NOYIELD | PM_REMOVE)) {
	if (ghAccel == NULL || 
		!TranslateAccelerator (ghTTYWnd, ghAccel, &msg)) {
	    TranslateMessage (&msg);
	    DispatchMessage (&msg);
	}
    }
}


/*
 * Set a callback for function 'ch'
 */
int
mswin_setresizecallback (int (*cb)())
{
    int		i;
    
    for (i = 0; i < RESIZE_CALLBACK_ARRAY_SIZE; ++i) {
	if (gpTTYInfo->resizer[i] == NULL) {
	    gpTTYInfo->resizer[i] = cb;
	    return (0);
	}
    }
    return (-1);
}

/*
 * Clear the entry for callback function 'ch'
 */
int
mswin_clearresizecallback (int (*cb)())
{
    int		i;
    
    for (i = 0; i < RESIZE_CALLBACK_ARRAY_SIZE; ++i) {
	if (gpTTYInfo->resizer[i] == cb
		) {
	    gpTTYInfo->resizer[i] = NULL;
	    return (0);
	}
    }
    return (-1);
}


void
mswin_beginupdate (void)
{
    gpTTYInfo->fMassiveUpdate = TRUE;
}



void
mswin_endupdate (void)
{
    gpTTYInfo->fMassiveUpdate = FALSE;
    MoveTTYCursor (ghTTYWnd);
}
	
	

/*
 *	Call to get next character.  Dispatch one message.
 */
int
mswin_getc (void)
{
    BOOL	appDone = FALSE;
    MSG		msg;
    
    mswin_flush();
    if (GetMessage (&msg, NULL, 0, 0)) {
	if (ghAccel == NULL || 
		!TranslateAccelerator (ghTTYWnd, ghAccel, &msg)) {
	    TranslateMessage (&msg);
	    DispatchMessage (&msg);
	}
    }
    else 
       appDone = TRUE;

    /* Should never be told to exit like this. */
    assert (appDone == FALSE);

    if (CQAvailable ()) 
	return (CQGet ());
    else
	return (MSWIN_KEY_NODATA);
}







/*
 * ibmopen - open termnal
 */
int
mswin_open (void)
{
    /* This is a no-op.  All of this should have been set up already. */
    return (0);
}


/*
 * ibmclsoe - Close terminal
 */
int
mswin_close (void)
{
    /* Another no-op. */
    return (0);
}




/*
 * ibmmove - Move cursor to...
 */
int
mswin_move (int row, int column)

{
    FlushWriteAccum ();
    if (row < 0 || row >= gpTTYInfo->actNRow)
	return (-1);
    if (column < 0 || column >= gpTTYInfo->actNColumn)
	return (-1);
    gpTTYInfo->nRow = row;
    gpTTYInfo->nColumn = column;
    MoveTTYCursor (ghTTYWnd);
    return (0);
}



int
mswin_getpos (int *row, int *column)
{
    FlushWriteAccum ();
    if (row == NULL || column == NULL)
	return (-1);
    *row = gpTTYInfo->nRow;
    *column = gpTTYInfo->nColumn;
    return (0);
}



int
mswin_getscreensize (int *row, int *column)
{
    if (row == NULL || column == NULL)
	return (-1);
    *row = gpTTYInfo->actNRow;
    *column = gpTTYInfo->actNColumn;
    return (0);
}




int
mswin_showcursor (int show)
{
    int		wasShow;
    
    
    wasShow = (gpTTYInfo->wCursorState & CS_SHOW) != 0;
	
    if (show) {
	
	/* if the cursor was not already show, show it now. */
	if (!wasShow) {
	    gpTTYInfo->wCursorState |= CS_SHOW;

	    if (gpTTYInfo->wCursorState == CS_VISIBLE) {
		    CreateCaret (ghTTYWnd, NULL, gpTTYInfo->xChar, 
			    gpTTYInfo->yChar);
		    ShowCaret (ghTTYWnd);
	    }
	    MoveTTYCursor (ghTTYWnd);
        }
    }
    else {
	
	/* If the cursor is shown, hide it. */
	if (wasShow) {
	    if (gpTTYInfo->wCursorState == CS_VISIBLE) {
		HideCaret (ghTTYWnd);
		DestroyCaret();
	    }

	    gpTTYInfo->wCursorState &= ~CS_SHOW;
	}
    }

    return (wasShow);
}



int
mswin_puts (char *str)
{
    int			strLen;

    FlushWriteAccum ();
    if (str == NULL)
	return (-1);
    strLen = strlen (str);
    if (strLen > 0)
	WriteTTYText (ghTTYWnd, str, strLen);
    return (0);
}



int
mswin_putblock (char *str, int strLen)
{
    FlushWriteAccum ();
    if (str == NULL)
	return (-1);
    if (strLen > 0)
	WriteTTYText (ghTTYWnd, str, strLen);
    return (0);
}



/*
 * mswin_putc - put a character at the current position in the
 *	     current colors
 */

int
mswin_putc (int c)
{
    BYTE		cc;

    cc = (char)(c & 0xff);

    gpTTYInfo->writeAccum[gpTTYInfo->writeAccumCount++] = cc;
    if (gpTTYInfo->writeAccumCount == WRITE_ACCUM_SIZE)
	    FlushWriteAccum ();
#if 0
/*	WriteTTYText (ghTTYWnd, &cc, 1);*/
    WriteTTYChar (ghTTYWnd, (char)(cc&0xff));
#endif
    return (0);
}



/* 
 * ibmoutc - output a single character with the right attributes, but
 *           don't advance the cursor
 */
int
mswin_outc (char c)
{
    RECT	rect;
    long	offset;
    
    FlushWriteAccum ();
    
    switch (c) {
    case ASCII_BEL:
	MessageBeep (0);
	break;
	
    case ASCII_BS:
    case ASCII_CR:
    case ASCII_LF:
	/* Do nothing for these screen motion characters. */
	break;
	
	
    default:
	/* Paint character to screen. */
	offset = (gpTTYInfo->nRow * gpTTYInfo->actNColumn) + gpTTYInfo->nColumn;
	*(gpTTYInfo->pScreen + offset) = c;
	*(gpTTYInfo->pAttrib + offset) = gpTTYInfo->curAttrib;
	
	/* Invalidate rectange covering singel character. */
	rect.left = (gpTTYInfo->nColumn * gpTTYInfo->xChar) -
		gpTTYInfo->xOffset;
	rect.right = rect.left + gpTTYInfo->xChar;
	rect.top = (gpTTYInfo->nRow * gpTTYInfo->yChar) -
		gpTTYInfo->yOffset;
	rect.bottom = rect.top + gpTTYInfo->yChar;
	gpTTYInfo->screenDirty = TRUE;
#ifdef OWNRECT
	MyInvalidateRect (gpTTYInfo, ghTTYWnd, &rect);
#else
	InvalidateRect (ghTTYWnd, &rect, FALSE);
#endif
	break;
    }
    return (0);
}



/*
 * mswin_delchar - delete character at cursor.  Shift end of line left.
 *	Cursor stays in position.
 */
int
mswin_delchar (void)
{
    CHAR	*cStart;
    CharAttrib	*aStart;
    long	length;
    RECT	rect;
    
    FlushWriteAccum ();
    
    /* From position right of cursor to end of line. */
    length = gpTTYInfo->actNColumn - (gpTTYInfo->nColumn + 1);

    /* Shift end of line. */
    if (length > 0) {
        cStart = gpTTYInfo->pScreen + (gpTTYInfo->nRow *gpTTYInfo->actNColumn)
				+ gpTTYInfo->nColumn;
	_fmemmove (cStart, cStart+1, (size_t) length);
        aStart = gpTTYInfo->pAttrib + (gpTTYInfo->nRow *gpTTYInfo->actNColumn)
			+ gpTTYInfo->nColumn;
        _fmemmove (aStart, aStart + 1, (size_t) length);
    }
    
    /* Clear last char in line. */
    *(cStart + length) = ' ';
    *(aStart + length) = CHAR_ATTR_NORM;

    /* Invalidate from cursor to end of line. */
    rect.left = (gpTTYInfo->nColumn * gpTTYInfo->xChar) - gpTTYInfo->xOffset;
    rect.right = gpTTYInfo->xSize;
    rect.top = (gpTTYInfo->nRow * gpTTYInfo->yChar) - gpTTYInfo->yOffset;
    rect.bottom = rect.top + gpTTYInfo->yChar;
    gpTTYInfo->screenDirty = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (gpTTYInfo, ghTTYWnd, &rect);
#else
    InvalidateRect (ghTTYWnd, &rect, FALSE);
#endif

    return (0);
}



/*
 * mswin_inschar - insert character in current position.  Shift line
 *	to right and drop last char on line. 
 *	Cursor advances one 
 */
int
mswin_inschar (int c)
{
    CHAR	*cStart;
    CharAttrib	*aStart;
    long	length;
    RECT	rect;
    
    FlushWriteAccum ();
    
    /* From cursor to end of line - 1
    length = (gpTTYInfo->actNColumn - gpTTYInfo->nColumn) - 1;

    /* Shift end of line. */
    if (length > 0) {
        cStart = gpTTYInfo->pScreen + (gpTTYInfo->nRow *gpTTYInfo->actNColumn)
				+ gpTTYInfo->nColumn;
	_fmemmove (cStart+1, cStart, (size_t) length);
        aStart = gpTTYInfo->pAttrib + (gpTTYInfo->nRow *gpTTYInfo->actNColumn)
			+ gpTTYInfo->nColumn;
        _fmemmove (aStart+1, aStart, (size_t) length);
    }
    
    /* Insert new char. */
    *(cStart + length) = (char)(c & 0x00ff);
    *(aStart + length) = gpTTYInfo->curAttrib;

    /* Invalidate from cursor to end of line. */
    rect.left = (gpTTYInfo->nColumn * gpTTYInfo->xChar) - gpTTYInfo->xOffset;
    rect.right = gpTTYInfo->xSize;
    rect.top = (gpTTYInfo->nRow * gpTTYInfo->yChar) - gpTTYInfo->yOffset;
    rect.bottom = rect.top + gpTTYInfo->yChar;
    gpTTYInfo->screenDirty = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (gpTTYInfo, ghTTYWnd, &rect);
#else
    InvalidateRect (ghTTYWnd, &rect, FALSE);
#endif
    
    /* move cursor forward one space. */
    if (gpTTYInfo->nColumn < gpTTYInfo->actNColumn - 1) {
	++gpTTYInfo->nColumn;
	MoveTTYCursor (ghTTYWnd);
    }

    return (0);
}



/*
 * ibmrev - change reverse video state
 */
int
mswin_rev (int state)
{
    int		curState;
    
    curState = (gpTTYInfo->curAttrib & CHAR_ATTR_REV);
    if ((state && !curState) || (!state && curState)) {
	FlushWriteAccum ();
	if (state) 
	    gpTTYInfo->curAttrib |= CHAR_ATTR_REV;
	else
	    gpTTYInfo->curAttrib &= ~CHAR_ATTR_REV;
    }
    return (0);
}
	


/*
 * Get current reverse video state. 
 */
int
mswin_getrevstate (void)
{
    return ((gpTTYInfo->curAttrib & CHAR_ATTR_REV) != 0);
}



/*
 * ibmeeol - erase to the end of the line
 */
int
mswin_eeol (void)
{
    CHAR	*cStart;
    CharAttrib	*aStart;
    long	length;
    RECT	rect;
    
    FlushWriteAccum ();
    
    /* From current position to end of line. */
    length = gpTTYInfo->actNColumn - gpTTYInfo->nColumn;		

    cStart = gpTTYInfo->pScreen + (gpTTYInfo->nRow * gpTTYInfo->actNColumn)
			+ gpTTYInfo->nColumn;
    _fmemset (cStart, ' ', (size_t) length);
    
    aStart = gpTTYInfo->pAttrib + (gpTTYInfo->nRow * gpTTYInfo->actNColumn)
			+ gpTTYInfo->nColumn;
    _fmemset (aStart, CHAR_ATTR_NORM, (size_t) length);

    rect.left = (gpTTYInfo->nColumn * gpTTYInfo->xChar) -
	    gpTTYInfo->xOffset;
    rect.right = gpTTYInfo->xSize;
    rect.top = (gpTTYInfo->nRow * gpTTYInfo->yChar) -
	    gpTTYInfo->yOffset;
    rect.bottom = rect.top + gpTTYInfo->yChar;
    gpTTYInfo->screenDirty = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (gpTTYInfo, ghTTYWnd, &rect);
#else
    InvalidateRect (ghTTYWnd, &rect, FALSE);
#endif

    return (0);
}



/*
 * ibmeeop - clear from cursor to end of page
 */
int
mswin_eeop (void)
{
    CHAR	*cStart;
    CharAttrib	*aStart;
    long	length;
    RECT	rect;
    
    FlushWriteAccum ();
    /* From current position to end of screen. */

    cStart = gpTTYInfo->pScreen + (gpTTYInfo->nRow * gpTTYInfo->actNColumn)
			+ gpTTYInfo->nColumn;
    length = (gpTTYInfo->pScreen + (gpTTYInfo->actNColumn * gpTTYInfo->actNRow))
		    - cStart;
    _fmemset (cStart, ' ', (size_t) length);
    
    aStart = gpTTYInfo->pAttrib + (gpTTYInfo->nRow * gpTTYInfo->actNColumn)
			+ gpTTYInfo->nColumn;
    _fmemset (aStart, CHAR_ATTR_NORM, (size_t) length);

    
    /* Invalidate a rectangle that includes all of the current line down
     * to the bottom of the window. */
    rect.left = 0;
    rect.right = gpTTYInfo->xSize;
    rect.top = (gpTTYInfo->nRow * gpTTYInfo->yChar) -
	    gpTTYInfo->yOffset;
    rect.bottom = gpTTYInfo->ySize;
    gpTTYInfo->screenDirty = TRUE;
#ifdef OWNRECT
    MyInvalidateRect (gpTTYInfo, ghTTYWnd, &rect);
#else
    InvalidateRect (ghTTYWnd, &rect, FALSE);
#endif

    return (0);
}



/*
 * ibmbeep - system beep...
 */
int
mswin_beep (void)
{
    MessageBeep (MB_OK);
    return (0);
}



/*
 * ibmflush - Flush output to screen.
 *
 */
int
mswin_flush (void)
{
    FlushWriteAccum ();
#ifdef OWNRECT
    if (gpTTYInfo->screenDirty) 
	InvalidateRect (ghTTYWnd, &gpTTYInfo->rPaintRect, FALSE);
#endif

    /* 
     * While the screen is dirty, Yeild to windows.
     * BUT, don't to this if the screen is minimized.
     */
    if (!gpTTYInfo->fMinimized) {
	while (gpTTYInfo->screenDirty) 
	    WinIdle ();
    }
    return (0);
}


/*
 * A replacement for fflush
 * relies on #define fflush mswin_fflush
 */
#undef fflush
int
mswin_fflush (FILE *f)
{
    if (f == stdout) {
	mswin_flush();
    }
    else
	fflush (f);
}




/*
 * pico_XXcolor() - each function sets a particular attribute
 */
pico_nfcolor(s)
char *s;
{
    SetColorAttribute (&gpTTYInfo->rgbFGColor, s);
}

pico_nbcolor(s)
char *s;
{
    SetColorAttribute (&gpTTYInfo->rgbBGColor, s);
}

pico_rfcolor(s)
char *s;
{
    SetColorAttribute (&gpTTYInfo->rgbRFGColor, s);
}

pico_rbcolor(s)
char *s;
{
    SetColorAttribute (&gpTTYInfo->rgbRBGColor, s);
}





/*
 * Signal 
 */
void (__cdecl * __cdecl signal (int sig,void (__cdecl *hndlr)(int)))(int)

{
    return (SIG_IGN);
}



int
printf (const char *fmt, ...)
{
    va_list	marker;
    
    va_start (marker, fmt);
    _vsnprintf (TempBuf, MAXLEN_TEMPSTR, fmt, marker);
    
    mswin_puts (TempBuf);
    return (1);
}
    
    




/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *                  Character Queue
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


typedef struct {
	WORD	flags;
	WORD	c;
} CQEntry;

LOCAL CQEntry CQBuffer [CHARACTER_QUEUE_LENGTH];
LOCAL int CQHead;
LOCAL int CQTail;
LOCAL int CQCount;


/*---------------------------------------------------------------------------
 *  BOOL  CQInit ()
 *
 *  Description:
 *		Initialize the Character queue.
 *
 *  Parameters:
 *
 *
/*--------------------------------------------------------------------------*/
LOCAL void
CQInit (void)
{
	CQHead = 0;
	CQTail = 0;
	CQCount = 0;
}
 
 
/*---------------------------------------------------------------------------
 *  BOOL  CQAvailable (void)
 *
 *  Description:
 *		Return TRUE if there are characters in the queue.
 *
 *  Parameters:
 *
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL
CQAvailable (void)
{
	return (CQCount > 0);
}
 

 
/*---------------------------------------------------------------------------
 *  BOOL  CQAdd (WORD c, DWORC keyData)
 *
 *  Description:
 *		Add 'c' to the end of the character queue.
 *
 *  Parameters:
 *		return true if successfull.
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL
CQAdd (WORD c, DWORD keyData)
{
	if (CQCount == CHARACTER_QUEUE_LENGTH)
		return (FALSE);
	
	
	CQBuffer[CQTail].flags = 0;
	if ((keyData & 0x80000000) == 0)
	    CQBuffer[CQTail].flags |= CQ_FLAG_DOWN;
	if (keyData & 0x01000000)
	    CQBuffer[CQTail].flags |= CQ_FLAG_EXTENDED;
	if (keyData & 0x20000000)
	    CQBuffer[CQTail].flags |= CQ_FLAG_ALT;
	CQBuffer[CQTail].c = c;
	CQTail = (CQTail + 1) % CHARACTER_QUEUE_LENGTH;
	++CQCount;
	return (TRUE);
}
 
 
/*---------------------------------------------------------------------------
 *  int  CQGet ()
 *
 *  Description:
 *		Return the next byte from the head of the queue.  If there is
 *		no byte available, returns 0, which is indistinquishable from 
 *		'\0'.  So it is a good idea to call CQAvailable first.
 *
 *  Parameters:
 *		none.
 *
/*--------------------------------------------------------------------------*/

LOCAL WORD
CQGet ()
{
    WORD	c;

    if (CQCount == 0)
	return (0);

    c = CQBuffer[CQHead].c;
    CQHead = (CQHead + 1) % CHARACTER_QUEUE_LENGTH;
    --CQCount;
    return (c);
}


#if 0
/*---------------------------------------------------------------------------
 *
 *  LOCAL int MapVKtoMS (WORD c, WORD flags);
 *
 * Description:
 *	Map key received in WM_CHAR message to intermediate character code.
 *      which latter gets maped to a pico or pine character code.
 */
LOCAL int
MapVKtoMS (WORD c, WORD flags)
{
    /* Special keys. */
    if (flags & CQ_FLAG_ALT)
	return (MSWIN_KEY_NODATA);

    if (flags & CQ_FLAG_EXTENDED) {
	switch (c) {
	    case VK_UP:			return (MSWIN_KEY_UP);
	    case VK_DOWN:		return (MSWIN_KEY_DOWN);
	    case VK_RIGHT:		return (MSWIN_KEY_RIGHT);
	    case VK_LEFT:		return (MSWIN_KEY_LEFT);
	    case VK_PRIOR:		return (MSWIN_KEY_PREVPAGE);
	    case VK_NEXT:		return (MSWIN_KEY_NEXTPAGE);
	    case VK_HOME:		return (MSWIN_KEY_HOME);
	    case VK_END:		return (MSWIN_KEY_END);
	    case VK_DELETE:		return (MSWIN_KEY_DELETE);
	    case VK_F1:			return (MSWIN_KEY_F1);
	    case VK_F2:			return (MSWIN_KEY_F2);
	    case VK_F3:			return (MSWIN_KEY_F3);
	    case VK_F4:			return (MSWIN_KEY_F4);
	    case VK_F5:			return (MSWIN_KEY_F5);
	    case VK_F6:			return (MSWIN_KEY_F6);
	    case VK_F7:			return (MSWIN_KEY_F7);
	    case VK_F8:			return (MSWIN_KEY_F8);
	    case VK_F9:			return (MSWIN_KEY_F9);
	    case VK_F10:		return (MSWIN_KEY_F10);
	    case VK_F11:		return (MSWIN_KEY_F11);
	    case VK_F12:		return (MSWIN_KEY_F12);
	    default:			return (MSWIN_KEY_NODATA);
        }
    }
    
    /* xxx Also need to map control-space to something but don't know
     * xxx how to tell if the control key was pressed. */

    /* Normal keys. */
    return (c);
}
#endif




/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *                  Mouse Event Queue
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/



LOCAL MEvent MQBuffer [MOUSE_QUEUE_LENGTH];
LOCAL int MQHead;
LOCAL int MQTail;
LOCAL int MQCount;


/*---------------------------------------------------------------------------
 *  BOOL  MQInit ()
 *
 *  Description:
 *		Initialize the Character queue.
 *
 *  Parameters:
 *
 *
/*--------------------------------------------------------------------------*/
LOCAL void
MQInit (void)
{
	MQHead = 0;
	MQTail = 0;
	MQCount = 0;
}
 
 
/*---------------------------------------------------------------------------
 *  BOOL  MQAvailable (void)
 *
 *  Description:
 *		Return TRUE if there are characters in the queue.
 *
 *  Parameters:
 *
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL
MQAvailable (void)
{
	return (MQCount > 0);
}
 

 
/*---------------------------------------------------------------------------
 *  BOOL  MQAdd ()
 *
 *  Description:
 *		Add 'c' to the end of the character queue.
 *
 *  Parameters:
 *		return true if successfull.
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL
MQAdd (int mevent, int button, int nRow, int nColumn, unsigned int keys)
{
	if (MQCount == MOUSE_QUEUE_LENGTH)
		return (FALSE);
	
	MQBuffer[MQTail].event = mevent;
	MQBuffer[MQTail].button = button;
	MQBuffer[MQTail].nRow = nRow;
	MQBuffer[MQTail].nColumn = nColumn;
	MQBuffer[MQTail].flags = (unsigned long) keys;
	MQTail = (MQTail + 1) % MOUSE_QUEUE_LENGTH;
	++MQCount;
	return (TRUE);
}
 
 
/*---------------------------------------------------------------------------
 *  BOOL  MQGet ()
 *
 *  Description:
 *		Return the next byte from the head of the queue.  If there is
 *		no byte available, returns 0, which is indistinquishable from 
 *		'\0'.  So it is a good idea to call MQAvailable first.
 *
 *  Parameters:
 *		none.
 *
/*--------------------------------------------------------------------------*/

LOCAL BOOL
MQGet (MEvent * pMouse)
{

    if (MQCount == 0)
	return (FALSE);

    *pMouse = MQBuffer[MQHead];
    pMouse->flags = MQBuffer[MQHead].flags;
    MQHead = (MQHead + 1) % MOUSE_QUEUE_LENGTH;
    --MQCount;
    return (TRUE);
}





/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *                  Memory allocation routines.
 *
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

/*
 * Notes:
 *
 *	Current implemtation allocates a Windows global memory block for 
 *	every allocation.  Simplistic, but it provides nice hardware memory
 *	protection.
 *
 *	Each memory block is assigned a unique id.  This is only used
 *	to match allocations with frees in the debug log.
 */

#define MEM_DEBUG

#define GUARD_LOW		0x9999
#define GUARD_HIGH		0xaaaaaaaa

#define SMALL_MEM_BOUNDRY	96

#define HEAP_SIZE		32000


/* Memory block header.  Placed at beginning of each allocated block. */
typedef struct {			/*size  len */
    HGLOBAL		handle;		/* 00 - 02 */
    short		globalBlk;	/* 02 - 02 */
    MemSize		size;		/* 04 - 04 */
    WORD		id;		/* 08 - 02 */
    WORD		guard1;		/* 0A - 02 */
} MemBlk;				/* Total size:  0C */

typedef MemBlk __far *	MemBlkPtr;


/* Memory high guard tailer.  Placed at end of each allocated block. */
typedef struct {
    unsigned long	guard1;
} MemHighGuard;

typedef MemHighGuard __far *MemHighGuardPtr;


/*
 * Memory allocation globals. 
 */
static unsigned long	MemLocalFails = 0;
#ifdef MEM_DEBUG
static BOOL		MemDoDebug = FALSE;	/* Doing debugging. */
static BOOL		MemLocalFull = FALSE;	/* True when local heap full*/
static FILE		*MemDebugFile = NULL;	/* File to write to. */
static WORD		MemID = 0;		/* Keep track of ID. */
static unsigned long	MemInUse = 0;		/* Total bytes in use. */
static unsigned long	MemInUseMax = 0;	/* max in use at one time. */
static unsigned long    SmallMemInUse = 0;
static unsigned long	SmallMemInUseMax = 0;
#endif /* MEM_DEBUG */

#define GET_SEGMENT(x)		(0xffff & ((DWORD)(x) >> 16))
#define GET_OFFSET(x)		(0xffff & (DWORD)(x))


#undef malloc
#undef realloc
#undef free

void		MemATest (void);


void *
malloc (size_t size)
{
	return (MemAlloc (size));
}


void __far *
_fmalloc (size_t size)
{
	return (MemAlloc (size));
}


void __far *
realloc (void *memblock, size_t size)
{
	return (MemRealloc (memblock, size));
}


void __far *
_frealloc (void *memblock, size_t size)
{
	return (MemRealloc (memblock, size));
}


void 
free (void *memblock)
{
	MemFree (memblock);
}

void 
_ffree (void *memblock)
{
	MemFree (memblock);
}



void
MemDebug (int debug, FILE *debugFile)
{
#ifdef MEM_DEBUG
    if (debugFile == NULL) {
	MemDoDebug = FALSE;
	MemDebugFile = NULL;
    }
    else {
	MemDoDebug = debug;
	MemDebugFile = debugFile;
    }
    if (debug) 
	MemATest ();
#endif /* MEM_DEBUG */
}





MemPtr
_MemAlloc (MemSize size, char __far * file, int line)
{
    MemBlkPtr			pBlk;
    MemHighGuardPtr		pHG;
    HGLOBAL			hBlk;
    HLOCAL			hLBlk;
    UINT			s;
    BYTE		__far *	pb;

    assert (size <= MEM_BLOCK_SIZE_MAX);

    /*
     * Calculate total size we need to allocate.
     */
    s = (UINT)size + sizeof (MemBlk) + sizeof (MemHighGuard);
    
    
    pBlk = NULL;
    if (s <= SMALL_MEM_BOUNDRY && !MemLocalFull) {
	    
	/* Allocate block from local storage. */
	hLBlk = LocalAlloc (LMEM_MOVEABLE, s);
	if (hLBlk != NULL) {

	    /* Lock block and dereference. */
	    pBlk = (MemBlkPtr) LocalLock (hLBlk);
	    if (pBlk != NULL) {
		pBlk->handle = hLBlk;
		pBlk->globalBlk = FALSE;
	    }
	    else 
	        LocalFree (hLBlk);
	}
	else {
	    ++MemLocalFails;
	    MemLocalFull = TRUE;
	    if (MemDoDebug)
		fprintf (MemDebugFile, "Local Memory alloc failed, %lu fails, %lu bytes in use\n",
			MemLocalFails, SmallMemInUse);
	}
    }
    
    /* If it is a large block, or local alloc failed, we allocate from
     * global space. */
    if (pBlk == NULL) {
	    
	/* Allocate block from global storage. */
	hBlk = GlobalAlloc (GMEM_MOVEABLE, s);
	if (hBlk == NULL) 
	    return (NULL);


	/* Lock block and dereference. */
	pBlk = (MemBlkPtr) GlobalLock (hBlk);
	if (pBlk == NULL) {
	    GlobalFree (hBlk);
	    return (NULL);
	}
	pBlk->handle = hBlk;
	pBlk->globalBlk = TRUE;
    }


    
    /* Fill rest of header. */
    pBlk->size = size;
    pBlk->id = ++MemID;
    pBlk->guard1 = GUARD_LOW;

    
    /* Find address that will be returned to caller. */
    pb = (BYTE __far *) (pBlk + 1);

    
    /* Find high guard and fill. */
    pHG = (MemHighGuardPtr) (pb + size);
    pHG->guard1 = GUARD_HIGH;
    

    /* Debugging info... */
#ifdef MEM_DEBUG
    if (MemDoDebug) {
	if( !file ) file = "??";
	fprintf (MemDebugFile, "MemAlloc: addr(%lx) id(%u) size(%ld) global(%d)\n",
		pb, pBlk->id, (long)s, pBlk->globalBlk);
	fflush (MemDebugFile);
    }
    MemInUse += s;
    if (MemInUse > MemInUseMax)
	MemInUseMax = MemInUse;
    if (s < SMALL_MEM_BOUNDRY) {
	SmallMemInUse += s;
	if (SmallMemInUse > SmallMemInUseMax)
	    SmallMemInUseMax = SmallMemInUse;
    }
#endif /* MEM_DEBUG */
	

    return ((MemPtr) pb);
}




	
int
_MemFree (MemPtr block, char __far *file, int line)
{
    MemBlkPtr		pBlk;
    MemHighGuardPtr	pHG;
    HGLOBAL		hBlk;
    HLOCAL		hLBlk;
    BOOL		brc;
    
    if (block == NULL)
        return (0);


    /* Find header and high guard and check them. */
    pBlk = ((MemBlkPtr)block) - 1;
    pHG = (MemHighGuardPtr) ((char __far *)block + pBlk->size);

    assert (pBlk->guard1 == GUARD_LOW);
    assert (pHG->guard1 == GUARD_HIGH);


    
#ifdef MEM_DEBUG    
    /* Deubgging info... */
    if (MemDoDebug) {
	if (pBlk->size < SMALL_MEM_BOUNDRY && 
		SmallMemInUse == SmallMemInUseMax) 
	   fprintf (MemDebugFile, "Small memory usage is up to %lu\n", SmallMemInUseMax);
	if (MemInUse == MemInUseMax) 
	   fprintf (MemDebugFile, "Memory usage is up to %lu\n", MemInUseMax);
    }
    MemInUse -= pBlk->size + sizeof (MemBlk) + sizeof (MemHighGuard);
    if (pBlk->size+sizeof(MemBlk)+sizeof(MemHighGuard) < SMALL_MEM_BOUNDRY)
        SmallMemInUse -= pBlk->size + sizeof (MemBlk) + sizeof (MemHighGuard);
    if (MemDoDebug) {
	fprintf (MemDebugFile, "MemFree: addr(%lx) id(%u)\n", 
		block, pBlk->id);
	fflush (MemDebugFile);
    }
#endif /* MEM_DEBUG */



    if (!pBlk->globalBlk) {
	/* Unlock block */
	hLBlk = pBlk->handle;
	brc = LocalUnlock (hLBlk);
	assert (!brc);

	/* And free block. */
	hLBlk = LocalFree (hLBlk);
	assert (hLBlk == NULL);
	MemLocalFull = FALSE;
    }
    else {
	/* Unlock block */
	hBlk = pBlk->handle;
	brc = GlobalUnlock (hBlk);
	assert (!brc);

	/* And free block. */
	hBlk = GlobalFree (hBlk);
	assert (hBlk == NULL);
    }
    return (0);
}





MemPtr
_MemRealloc (MemPtr block, MemSize size, char __far * file, int line)
{
    MemPtr	newBlock;
    
    
    newBlock = MemAlloc (size);
    if (newBlock == NULL)
	return (NULL);

    _fmemcpy (newBlock, block , (size_t)MIN (size, MemBlkSize (block)));
    
    MemFree (block);
	
    return (newBlock);
}
	
	

	
MemSize
MemBlkSize (MemPtr block)
{
    MemBlkPtr			pBlk;
    
    pBlk = ((MemBlkPtr)block) - 1;
    assert (pBlk->guard1 == GUARD_LOW);

    return (pBlk->size);
}


#ifdef MEM_DEBUG
struct testblock {
    struct testblock	__far * prev;
    HLOCAL						h;
};



void
MemATest (void)
{
    void    __near		*n;
    struct testblock __far	*p;
    struct testblock __far	*pnew;
    HLOCAL			hl;
    int				bcount;
    UINT			seg, start, end;
    void    __far		*f;
    HGLOBAL			hg;
    DWORD			dw;
    LOCALINFO			li;
    
#if 0    
    hg = GlobalAlloc (GMEM_FIXED, HEAP_SIZE);
    if (hg == NULL)
	return;
    f = GlobalLock (hg);
    if (f == NULL) 
	goto Fail1;
    seg = (UINT) GET_SEGMENT (f);
    start = (UINT) GET_OFFSET (f);
    end = start + HEAP_SIZE - 1;
    start += 16;
    if (!LocalInit (seg, start, end))
	goto Fail2;
#endif
#if 0
    hl = LocalAlloc (0, SMALL_MEM_BOUNDRY);
    if (hl == NULL) 
	    return;
    n = LocalLock (hl);
    f = (void __far *)n;
    seg = GET_SEGMENT(f);
    dw = GlobalHandle (seg);
    hg = (HGLOBAL) (dw & 0xffff);
    if (hg == NULL)
	    return;
    
    li.dwSize = sizeof (li);
    if (!LocalInfo (&li, hg))
	    return;
    
    dw = GlobalSize (hg);
    f = GlobalLock (hg);
    GlobalUnlock (hg);
    
    LocalUnlock (hl);
    LocalFree (hl);

    
#endif
    
    
	
	
    p = NULL;
    pnew = NULL;
    bcount = 0;
	
    do {
	hl = LocalAlloc (0, SMALL_MEM_BOUNDRY);
	if (hl != NULL) {
	    ++bcount;
	    n = LocalLock (hl);
	    pnew = (struct testblock __far*) n;
	    pnew->h = hl;
	    pnew->prev = p;
	    p = pnew;
        }
    } while (hl != NULL);
    
    
    if (MemDebugFile != NULL)
	fprintf (MemDebugFile, "Allocated %d blocks of size %d\n",
	    bcount, SMALL_MEM_BOUNDRY);
			
    while (p != NULL) {
	pnew = p->prev;
	hl = p->h;
	LocalUnlock (hl);
	LocalFree (hl);
	p = pnew;
    }
#if 0    
Fail2:	GlobalUnlock (hg);
Fail1:	GlobalFree (hg);
#endif
	return;
}
#endif /* MEM_DEBUG */



void 
AssertFail	(char *str, char *file, int line, int msgbox)
{
    /* static to avoid risk of stack overflow */
    static int	inAssert	= 0;
    static int	doBreak		= 1;
    char	*	filename;
    int			ret;



    /* for now, strip off path from filename */
    filename = _fstrrchr(file, '\\');
    if (filename)
	    filename++;
    else
	    filename = file;

    wsprintf(TempBuf, "Assert(%s) failed!  %s:%d\n", 
			    str, filename, line);

    inAssert++;


retry:

    /*	Unfortunately, there are certain times when attempting to
     *	display a message box will cause the system to crash (e.g.,
     *	when handling certain messages in a WndProc).
     *
     *	Message box was previously TaskModal, but this didn't work
     *	well because the client AIM could keep sending requests.
     *	SystemModal shuts the whole system down but at least it
     *	guarantees some degree of consistency, and makes debugging
     *	a bit easier.
     */
    ret = MessageBox (NULL, TempBuf, NULL, 
				    MB_ABORTRETRYIGNORE | MB_SYSTEMMODAL );


    if (ret == IDABORT) {
	    ret = MessageBox(NULL, 
			    "Aborting may not properly free resources.  Are you sure?",
			    "Assertion Abort",
			    MB_YESNO | MB_TASKMODAL | MB_ICONSTOP);

	    if (ret == IDYES) {
		    /*
		     *	Cause a GPF in case DrWatson is running
		     */
		    char *p = NULL;
		    *p = 1;
	    }
	    else 
		    goto retry;
    }

    /* retry is not (and will never be) hooked up, but it's
     * easier to use the standard dialog box than create
     * our own.  if retry is selected, just report the
     * error again.
     */
    if (ret == IDRETRY)
	    goto retry;
    inAssert--;
}
