#if !defined(lint) && !defined(DOS)
static char rcsid[] = "$Id: addrbook.c,v 4.89 1994/06/21 19:36:05 hubert Exp $";
#endif
/*----------------------------------------------------------------------

            T H E    P I N E    M A I L   S Y S T E M

   Laurence Lundblade and Mike Seibel
   Networks and Distributed Computing
   Computing and Communications
   University of Washington
   Administration Builiding, AG-44
   Seattle, Washington, 98195, USA
   Internet: lgl@CAC.Washington.EDU
             mikes@CAC.Washington.EDU

   Please address all bugs and comments to "pine-bugs@cac.washington.edu"

   Copyright 1989-1994  University of Washington

    Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee to the University of
   Washington is hereby granted, provided that the above copyright notice
   appears in all copies and that both the above copyright notice and this
   permission notice appear in supporting documentation, and that the name
   of the University of Washington not be used in advertising or publicity
   pertaining to distribution of the software without specific, written
   prior permission.  This software is made available "as is", and
   THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
   WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
   WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
   NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
   LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
   (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  
   Pine and Pico are trademarks of the University of Washington.
   No commercial use of these trademarks may be made without prior
   written permission of the University of Washington.

   Pine is in part based on The Elm Mail System:
    ***********************************************************************
    *  The Elm Mail System  -  Revision: 2.13                             *
    *                                                                     *
    * 			Copyright (c) 1986, 1987 Dave Taylor              *
    * 			Copyright (c) 1988, 1989 USENET Community Trust   *
    ***********************************************************************
 

  ----------------------------------------------------------------------*/

/*====================================================================== 
    addrbook.c
    display, browse and edit the address book.

    Support routines are in adrbklib.c.

The policy for changing the address book is to write it immediately 
after the change is made, so there is no idea of having to save the 
address book.  It was determined that this will be fast even for 
large address books and is implemented in the adrbklib.  Each call to
these functions can then expect write errors. 
 ====*/


#include "headers.h"
#include "adrbklib.h"


/*
 * Data structures for the display of the address book.  There's one
 * entry in this array of structures per line on the screen.  The list 
 * here is re-sorted so the distribution lists are at the end of the
 * display.  The strings returned by the adrbklib routines stay around
 * so only pointers to them are maintained, and not the actual strings.
 *
 * Types: Title -- The title line for the different address books.  It has
 *		   a ptr to the text of the Title line.
 *    ClickHere -- This is the line that says to click here to
 *                 expand.  It changes types into the individual expanded
 *                 components once it is expanded.  It doesn't have any data
 *                 other than an implicit title.
 *        Empty -- Line that says this is an empty addressbook.  No data.
 *       Single -- A single addressbook entry.  It has a ptr to an AdrBk_Entry.
 *                 When it is displayed, the fields are:
 *                 <nickname>       <fullname>       <address or another nic>
 *     ListHead -- The head of an address list.  This has a ptr to an
 *		   AdrBk_Entry.
 *                 <blank line> followed by
 *                 <nickname>       <fullname>       "DISTRIBUTION LIST:"
 *      ListEnt -- The rest of an address list.  It has a pointer to its
 *		   ListHead element and a ptr (other) to this specific address
 *		   (not a ptr to another AdrBk_Entry).
 *                 <blank>          <blank>          <address or another nic>
 *         Text -- A ptr to text.  For example, the ----- lines and
 *		   whitespace lines.
 *          End -- The last (imaginary) element in the array.
 */
typedef enum {Title, ClickHere, Empty, Single, ListHead, ListEnt,
							Text, End} LineType;
/* each line is one of these structures */
typedef struct addrscrn_disp {
    LineType       type;
    union {
      AdrBk_Entry *abe;
      char        *txt;
    }a;
    char          *other;
} AddrScrn_Disp;

/*
 * There is one of these per addressbook.  Each addressbook manages its
 * own memory, including its part of the display list.
 */
typedef enum {LocalFile} AddrBookType;  /* only one type so far */
typedef enum {ReadOnly, ReadWrite, NoAccess, NoExists} AccessType;
typedef enum {Closed,     /* data not read in, no display list             */
	      NoDisplay,  /* data is read in to memory, no display list    */
	      HalfOpen,   /* data not read in, initial display list is set */
	      Open,       /* data is read in and display list is complete  */
	      Same        /* leave it the same as it already was           */
	     } OpenStatus;
typedef struct peraddrbook {
    AddrBookType        type;
    AccessType          access;
    OpenStatus          ostatus;
    char               *nickname,
		       *filename;
    AdrBk              *address_book; /* the address book handle             */
    AddrScrn_Disp      *disp_list;    /* array, one entry per displayed line */
    int                 dl_used,      /* display lines used                  */
                        dl_allocated; /* display lines alloced               */
    int                 gave_parse_warnings;
} PerAddrBook;

/*
 * Just one of these.  This keeps track of the current state of
 * the screen and which addressbook we're looking at.
 */
typedef struct addrscreenstate {
    PerAddrBook   *adrbks;       /* array of addrbooks                    */
    int           *dl_ends,      /* array of sums of dl_used's            */
		   initialized,  /* have we done at least simple init?    */
                   n_addrbk,     /* how many addrbooks are there          */
                   tot_used,     /* sum of dl_used's                      */
                   cur,          /* current addrbook                      */
                   cur_fld,      /* currently selected field (0, 1, or 2) */
                   cur_row,      /* currently selected line (1st is 0)    */
                   old_cur_row,  /* previously selected line              */
                   top_ent,      /* index in disp_list of top entry on screen */
                   l_p_page;	 /* lines per (screen) page               */
} AddrScrState;

static AddrScrState as;

typedef enum {AddrBookScreen, SelectAddr, SelectNick} AddrBookArg;

/*
 * Information used to paint and maintain a line on the TakeAddr screen
 */
typedef struct takeaddr_line {
    int		      is_blank; /* addr is selected                     */
    char	     *strvalue; /* alloc'd value string                 */
    int		      checked;  /* addr is selected                     */
    ADDRESS	     *addr;     /* original ADDRESS this line came from */
    struct takeaddr_line *next, *prev;
} TA_S;

typedef struct takeaddress_screen {
    TA_S  *current,
          *top_line;
} TA_SCREEN_S;

typedef enum {AddToList, CreateList, Either} CreateOrAdd;

static TA_SCREEN_S *ta_screen;


#ifdef ANSI
void           ab_compose_to_addr();
void           ab_goto_folder();
void           ab_print();
void           ab_resize(int);
int            ab_whereis();
void           add_abook_entry(ADDRESS *, char *, int);
char          *addr_book(AddrBookArg);
int            addr_book_delete(AdrBk *, int, int);
char          *addr_book_takeaddr();
AdrBk_Entry   *addr_to_ae(ADDRESS *);
int            addr_to_list(AdrBk *, int, AddrScrn_Disp *);
int            addresses_match(ADDRESS *, AdrBk_Entry *);
int            adrbk_access(char *, int);
AdrBk_Entry   *adrbk_lookup_with_opens(char *, int, int *);
void           alloc_more_ab_display(PerAddrBook *);
int            any_addrs_avail(int);
int            build_address_internal(char *, char **, char **, char **, int);
int            calculate_field_widths(int *);
int            change_address_entry(AdrBk *, int, int, int);
int            change_address(AdrBk *, AddrScrn_Disp *, AdrBk_Entry **,
								char **, int);
int            change_comment(AdrBk *, AddrScrn_Disp *, AdrBk_Entry **, int);
int            change_fcc(AdrBk *, AddrScrn_Disp *, AdrBk_Entry **, int);
int            change_fullname(AdrBk *, AddrScrn_Disp *, AdrBk_Entry **, int);
int            change_nickname(AdrBk *, AddrScrn_Disp *, AdrBk_Entry **, int);
PerAddrBook   *check_for_addrbook(char *);
int            cmp_ints(const QSType *, const QSType *);
int            convert_dl(int, int *, int *);
void           create_abook_entry(char *, char *, char *, char *, int);
void           create_abook_list(char **, CreateOrAdd, int);
int            create_list(AdrBk *, int);
int            cur_addr_book();
AddrScrn_Disp *dlist(int);
void           display_book(int, int, int, int, int, int, int, int, int *);
int            do_edit_extras(int, AdrBk *, int, AddrScrn_Disp *);
int            edit_address(int, int, char *, char *, HelpType);
int            edit_comment(int, char *, char *, HelpType);
int            edit_extras(AdrBk *, int, int);
int            edit_extras_menu(int);
int            edit_fcc(int, char *, char *, HelpType);
int            edit_fullname(int, char *, char *, HelpType);
int            edit_nickname(AdrBk *, AddrScrn_Disp *, int, char *, char *,
							HelpType, int, int);
void           end_adrbks();
int            entry_is_clickable(int);
int            est_size(ADDRESS *);
ADDRESS       *expand_address(char *, char *, char *, int *, char **);
int            find_in_book(int, int, char *, int *, int *);
TA_S          *first_checked(TA_S *);
int            first_line(int);
TA_S          *first_taline(TA_S *);
void           free_disp_list(PerAddrBook *);
void           free_list_of_checked(char ***);
void           free_taline(TA_S **);
int            illegal_chars(char *);
int            init_ab_if_needed();
int            init_addrbooks(OpenStatus, int, int, int);
int            init_disp_list(PerAddrBook *, OpenStatus, AdrBk_Entry *,
								char *, int);
int            is_addr(int);
int            is_empty(int);
int            line_is_active(int);
char         **list_of_checked(TA_S *);
TA_S          *new_taline(TA_S **);
int            next_field(int, int, int *, int *);
int            next_line(int, int, int *, int *);
TA_S          *next_nonblank_taline(TA_S *);
int            nickname_check(char *);
int            our_build_address(char *, char **, char **, char **, int);
void           paint_line(int, AddrScrn_Disp *, int, int, int *);
int            prev_field(int, int, int *, int *);
int            prev_line(int, int, int *, int *);
TA_S          *prev_nonblank_taline(TA_S *);
void           redraw_addr_screen();
void           reset_dl_ends();
void           restore_state(AddrScrState *, OpenStatus *);
void           save_state(AddrScrState **, OpenStatus **);
int            search_book(int, int, int, int *, int *);
int            search_in_one_line(AddrScrn_Disp *, int, char *);
int            set_disp_list(PerAddrBook *, AdrBk_Entry *, char *, int, int);
PerAddrBook   *setup_for_addrbook_add(AddrScrState **, OpenStatus **, int);
int            simple_add(AdrBk *, int);
int            ta_take_marked_addrs(int, TA_S *);
int            ta_add_to_list(int, TA_S *);
int            ta_mark_all(TA_S *);
int            ta_unmark_all(TA_S *);
void           takeaddr_screen_redrawer();
int            update_takeaddr_screen(struct pine *, TA_S *, TA_SCREEN_S *);
PerAddrBook   *use_this_addrbook(int);
int            warn_bad_addr(char *, int);

#else  /* !ANSI */

void           ab_compose_to_addr();
void           ab_goto_folder();
void           ab_print();
void           ab_resize(int);
int            ab_whereis();
void           add_abook_entry();
char          *addr_book();
int            addr_book_delete();
char          *addr_book_takeaddr();
AdrBk_Entry   *addr_to_ae();
int            addr_to_list();
int            addresses_match();
int            adrbk_access();
AdrBk_Entry   *adrbk_lookup_with_opens();
void           alloc_more_ab_display();
int            any_addrs_avail();
int            build_address_internal();
int            calculate_field_widths();
int            change_address_entry();
int            change_address();
int            change_comment();
int            change_fcc();
int            change_fullname();
int            change_nickname();
PerAddrBook   *check_for_addrbook();
int            cmp_ints();
int            convert_dl();
void           create_abook_entry();
void           create_abook_list();
int            create_list();
int            cur_addr_book();
AddrScrn_Disp *dlist();
void           display_book();
int            do_edit_extras();
int            edit_address();
int            edit_comment();
int            edit_extras();
int            edit_extras_menu();
int            edit_fcc();
int            edit_fullname();
int            edit_nickname();
void           end_adrbks();
int            entry_is_clickable();
int            est_size();
ADDRESS       *expand_address();
int            find_in_book();
TA_S          *first_checked();
int            first_line();
TA_S          *first_taline();
void           free_disp_list();
void           free_list_of_checked();
void           free_taline();
int            illegal_chars();
int            init_ab_if_needed();
int            init_addrbooks();
int            init_disp_list();
int            is_addr();
int            is_empty();
int            line_is_active();
char         **list_of_checked();
TA_S          *new_taline();
int            next_field();
int            next_line();
TA_S          *next_nonblank_taline();
int            nickname_check();
int            our_build_address();
void           paint_line();
int            prev_field();
int            prev_line();
TA_S          *prev_nonblank_taline();
void           redraw_addr_screen();
void           reset_dl_ends();
void           restore_state();
void           save_state();
int            search_book();
int            search_in_one_line();
int            set_disp_list();
PerAddrBook   *setup_for_addrbook_add();
int            simple_add();
int            ta_take_marked_addrs();
int            ta_add_to_list();
int            ta_mark_all();
int            ta_unmark_all();
void           takeaddr_screen_redrawer();
int            update_takeaddr_screen();
PerAddrBook   *use_this_addrbook();
int            warn_bad_addr();

#endif /* !ANSI */


/*
  HEADER_LINES is the number of lines in the header
  FOOTER_LINES is the number of lines in the footer
 */
#define HEADER_LINES   2
#define FOOTER_LINES   3

#define CLICKHERE       "[ Select Here to See Expanded List ]"
#define NO_PERMISSION   "[ Permission Denied - No Read Access to File ]"
#define EMPTY           "[ Empty ]"
#define READONLY        "(ReadOnly)"
#define NOACCESS        "(Un-readable)"
#define DISTLIST        "DISTRIBUTION LIST:"

#define INIT_DISPLIST_SIZE 8

#define MAX_FCC     MAX_ADDRESS
#define MAX_COMMENT (10000)


/*
 * Make sure addrbooks are minimally initialized.
 *
 * Results:  0 -- ok
 *          -1 -- failed
 */
int
init_ab_if_needed()
{
    dprint(9, (debugfile, "- init_ab_if_needed -\n"));

    if(!as.initialized && !init_addrbooks(Closed, 0, 0, 0))
	return -1;
    
    return 0;
}


/*
 * Sets everything up to get started.
 *
 * Args: want_status      -- The desired OpenStatus for all addrbooks.
 *       reset_to_top     -- Forget about the old location and put cursor
 *                           at top.
 *       open_if_only_one -- If want_status is HalfOpen and there is only
 *                           one addressbook, then promote want_status to Open
 *       no_ro_warning    -- Don't display a ReadOnly warning.
 *
 * Return: number of addrbooks.
 */
int
init_addrbooks(want_status, reset_to_top, open_if_only_one, no_ro_warning)
OpenStatus want_status;
int        reset_to_top,
           open_if_only_one,
	   no_ro_warning;
{
    register PerAddrBook *pab;
    char *p, *q, **t;
#ifdef DOS
    char book_path[MAXPATH];
#endif /* DOS */

    dprint(6, (debugfile, "-- init_addrbooks(%s, %d, %d, %d) --\n",
		    want_status==Open ?
				"Open" :
				want_status==HalfOpen ?
					"HalfOpen" :
					want_status==NoDisplay ?
						"NoDisplay" :
						want_status==Same ?
						    "Same" :
						    "Closed",
		    reset_to_top, open_if_only_one, no_ro_warning));

    /* already been initialized */
    if(as.n_addrbk){
	int i;

	/*
	 * Special case.  If there is only one addressbook we start the
	 * user out with that open, just like we did when there was always
	 * only one addressbook.
	 */
	if(open_if_only_one && as.n_addrbk == 1 && want_status == HalfOpen)
	    want_status = Open;

	/* open to correct state */
	for(i = 0; i < as.n_addrbk; i++)
	    (void)init_disp_list(&as.adrbks[i], want_status,
		(AdrBk_Entry *)NULL, NULL, no_ro_warning);

	if(reset_to_top){
	    as.top_ent                      = 0;
	    as.cur_row                      = first_line(0);
	    as.old_cur_row                  = as.cur_row;
	    as.cur_fld                      = 0;
	    as.cur                          = cur_addr_book();
	}

	dprint(9, (debugfile, "init_addrbooks: already initialized: %d books\n",
				    as.n_addrbk));
        return(as.n_addrbk);
    }

    /* there are no addrbooks */
    if(as.initialized && !as.n_addrbk)
	return 0;

    as.initialized = 1;


    if(!ps_global->VAR_ADDRESSBOOK ||
       !ps_global->VAR_ADDRESSBOOK[0] ||
       !ps_global->VAR_ADDRESSBOOK[0][0])
	return 0;
    
    /*
     * The rest of this initialization only happens once, the first time
     * the addressbooks are accessed.  After that, the as structure remains
     * and the peraddrbook pab structures stay initialized.  However, the
     * memory used to read in the addrbooks with adrbk_open is freed and
     * re-allocated as we go in and out of the addrbooks, and the display
     * memory for each addrbook is freed and re-allocated frequently.
     */

    /* count addressbooks */
    for(t = ps_global->VAR_ADDRESSBOOK; *t != NULL; t++)
	as.n_addrbk++;

    if(open_if_only_one && as.n_addrbk == 1 && want_status == HalfOpen)
	want_status = Open;

    as.cur          = 0;
    as.l_p_page     = ps_global->ttyo->screen_rows - FOOTER_LINES
						   - HEADER_LINES;

    /*
     * allocate array of PerAddrBooks
     * (we don't give this up until we exit Pine, but it's small)
     */
    as.adrbks       = (PerAddrBook *)fs_get(as.n_addrbk * sizeof(PerAddrBook));
    memset((void *)as.adrbks, 0, as.n_addrbk * sizeof(PerAddrBook));

    /*
     * Array of ending indices of disp_lists.
     * This is used to make the combination of all the disp_lists seem
     * sort of like one big disp_list.
     * (don't free this until we exit, either)
     */
    as.dl_ends      = (int *)fs_get((as.n_addrbk + 1) * sizeof(int));

    /* init PerAddrBook data */
    for(t = ps_global->VAR_ADDRESSBOOK; (q = *t) != NULL; t++, as.cur++){
	int quoted = 0;

	pab = &as.adrbks[as.cur];
	
	/* Parse entry for optional nickname and filename */
	tmp_20k_buf[0] = '\0';
	for(p = q; *p; p++){
	    if(*p == '"')                   /* quoted label? */
	      quoted = (quoted) ? 0 : 1;

	    if(*p == '\\' && *(p+1) == '"') /* escaped quote? */
	      p++;                          /* skip it...     */

	    if(isspace(*p) && !quoted){     /* space delimits nic/filename */
		char *p1 = tmp_20k_buf;

		for(; q < p ; q++){
		    if(*q == '\\' && *(q+1) == '"')
		      *p1++ = *++q;
		    else if(*q != '"')
		      *p1++ = *q;
		}

		*p1 = '\0';                 /* tie off nickname */
		while(isspace(*p))          /* find start of filename */
		  p++;

		q = p;          	    /* point to filename start */
		break;                      /* pop out of here */
	    }
	}

	if(tmp_20k_buf[0] == '\0')
	    pab->nickname = cpystr(q);
	else
	    pab->nickname = cpystr(tmp_20k_buf);

        strcpy(tmp_20k_buf, q);
        q = tmp_20k_buf;
	if(*q == '~')
	    expand_foldername(q);
	/* q points to the filename */

#ifdef DOS
	if(q[0] == '\\' || (isalpha(q[0]) && q[1] == ':')){
	    strcpy(book_path, q);
	}
	else{
	    int l = last_cmpnt(ps_global->pinerc) - ps_global->pinerc;

	    strncpy(book_path, ps_global->pinerc, l);
	    book_path[l] = '\0';
	    strcat(book_path, q);
	}

	pab->filename = cpystr(book_path);
#else /* !DOS */
	pab->filename = cpystr(q);
#endif /* !DOS */

	pab->type  = LocalFile;

	if(adrbk_access(pab->filename, ACCESS_EXISTS) == 0){
	    if(adrbk_access(pab->filename, EDIT_ACCESS) == 0)
		pab->access = ReadWrite;
	    else if(adrbk_access(pab->filename, READ_ACCESS) == 0)
		pab->access = ReadOnly;
	    else
		pab->access = NoAccess;
	}
	else
	    pab->access = NoExists;

	pab->ostatus  = Closed;

	if(!no_ro_warning &&
	    open_if_only_one &&
	    as.n_addrbk == 1 &&
	    want_status == Open){

	    if(pab->access == ReadOnly)
		q_status_message(0, 1, 3, "AddressBook is Read Only");
	    else if(pab->access == NoAccess)
		q_status_message(0, 1, 3,
		    "AddressBook not accessible, permission denied");
	}

	/*
	 * and remember that the memset above initializes everything
	 * else to 0
	 */

	(void)init_disp_list(pab, want_status,
	    (AdrBk_Entry *)NULL, NULL, no_ro_warning);
    }

    /*
     * Have to reset_to_top in this case since this is the first open,
     * regardless of the value of the argument, since these values haven't been
     * set before here.
     */
    reset_dl_ends();
    as.top_ent                      = 0;
    as.cur_row                      = first_line(0);
    as.old_cur_row                  = as.cur_row;
    as.cur_fld                      = 0;
    as.cur                          = cur_addr_book();

    return(as.n_addrbk);
}


/*
 * VAR_ADDRESSBOOK was changed in options screen, so need to start over.
 */
void
addrbook_reset()
{
    dprint(8, (debugfile, "- addrbook_reset -\n"));

    completely_done_with_adrbks();
}


/*
 * Returns 0 if access mode is ok, -1 if not.
 */
int
adrbk_access(filename, mode)
char *filename;
int   mode;
{
    char fbuf[501];

#ifdef DOS
    if(filename[0] == '\\' || (isalpha(filename[0]) && filename[1] == ':'))
#else /* !DOS */
    if(*filename == '/')
#endif /* !DOS */
	build_path(fbuf, NULL, filename);
    else
	build_path(fbuf, ps_global->home_dir, filename);
    
    return(can_access(fbuf, mode));
}


/*
 * Returns the index of the current address book.
 *
 * dl_ends, top_ent, and cur_row must be up-to-date for this to work.
 */
int
cur_addr_book()
{
    register int i, addrbks, cur_line;

    addrbks   = as.n_addrbk;
    cur_line  = as.top_ent + as.cur_row; 

    for(i = 0; i < addrbks && cur_line > as.dl_ends[i+1]; i++)
	;/* do nothing */
    
    return(i);
}


/*
 * dl_ends is an array one longer than the number of address books.
 * dl_ends[i] is the index of the display list row that comes right before
 * the start of the i'th address book's display list.  That is also the
 * index of the last display list element of the i-1'th list.
 *
 *  Example: 2 addrbooks (n_addrbk = 2)
 *           tot_used = 14
 *
 *  dl_ends[0] = -1 
 *                     --- row 0  addrbook 0 start
 *                     --- row 1                8 disp_list rows for book 0
 *                     ---  ...
 *  dl_ends[1] =  7    --- row 7  addrbook 0 end
 *                     --- row 8  addrbook 1 start
 *                     ---  ...                 6 disp_list rows for book 1
 *  dl_ends[2] = 13    --- row 13 addrbook 1 end
 *  (There is also an imaginary row 14 which has type "End", which is
 *   what dlist(14) returns.)
 */
void
reset_dl_ends()
{
    register int i, addrbks, cntr;

    dprint(9, (debugfile, "- reset_dl_ends -\n"));

    as.dl_ends[0] = -1;
    addrbks       = as.n_addrbk;
    cntr          = 0;

    for(i = 0; i < addrbks; i++){
	cntr += as.adrbks[i].dl_used;
	as.dl_ends[i+1] = cntr - 1;
    }

    as.tot_used = cntr;
}


/*
 * Free most of what we've allocated.  The rest we keep for the duration.
 */
void
end_adrbks()
{
    int i;

    dprint(8, (debugfile, "- end_adrbks -\n"));

    for(i = 0; i < as.n_addrbk; i++)
	(void)init_disp_list(&as.adrbks[i], Closed, (AdrBk_Entry *)NULL,
	    NULL, 1);
}


/*
 * Free and close everything.
 */
void
completely_done_with_adrbks()
{
    register PerAddrBook *pab;
    int i;

    dprint(8, (debugfile, "- completely_done_with_adrbks -\n"));

    for(i = 0; i < as.n_addrbk; i++){
	pab = &as.adrbks[i];
	(void)init_disp_list(pab, Closed, (AdrBk_Entry *)NULL, NULL, 1);

	if(pab->filename)
	    fs_give((void **)&pab->filename);

	if(pab->nickname)
	    fs_give((void **)&pab->nickname);
    }

    if(as.adrbks)
	fs_give((void **)&as.adrbks);
    if(as.dl_ends)
	fs_give((void **)&as.dl_ends);
    as.n_addrbk    = 0;
    as.initialized = 0;
}


/*
 * Save the screen state and the Open or Closed status of the addrbooks.
 */
void
save_state(savep, stp)
AddrScrState **savep;
OpenStatus   **stp;
{
    static AddrScrState save;
    static OpenStatus  *st;
    int                 i;

    dprint(9, (debugfile, "- save_state -\n"));

    save   = as;    /* copy as struct to save static struct       */
    *savep = &save; /* set the passed in pointer to point to save */

    /* allocate space for saving the ostatus for each addrbook */
    st   = (OpenStatus *)fs_get(as.n_addrbk * sizeof(OpenStatus));
    *stp = st;

    for(i = 0; i < as.n_addrbk; i++)
	st[i] = as.adrbks[i].ostatus;
}


/*
 * Restore the state.
 */
void
restore_state(savep, stp)
AddrScrState *savep;
OpenStatus   *stp;
{
    int i;

    dprint(9, (debugfile, "- restore_state -\n"));

    /* restore addressbook OpenStatus to what it was before */
    for(i = 0; i < as.n_addrbk; i++)
	(void)init_disp_list(&as.adrbks[i], stp[i], (AdrBk_Entry *)NULL,
	    NULL, 1);

    as = *savep;  /* put back cur_row and all that */

    fs_give((void **)&stp);
}


void
free_disp_list(pab)
PerAddrBook *pab;
{
    register AddrScrn_Disp *dl;
    register int i;

    dprint(9, (debugfile, "- free_disp_list -\n"));

    if((dl = pab->disp_list) == NULL)
	return;

    for(i=0; i < pab->dl_used && dl; i++,dl++){
	switch(dl->type){
	  case Text:
	  case Title:
	    if(dl->a.txt)
		fs_give((void **)&dl->a.txt);
	    break;
	}
    }
    fs_give((void **)&pab->disp_list);
    pab->dl_used = 0;
    pab->dl_allocated = 0;
}


/*
 * Initialize or re-initialize the display list structure for this
 * addressbook.  Read in the data, and initialize the
 * display structure with the individual addresses if no_open isn't set.
 *
 *  Args: pab           -- the PerAddrBook ptr
 *        want_status   -- desired OpenStatus
 *        match_ent     -- see set_disp_list()
 *        match_addr    -- see set_disp_list()
 *        no_ro_warning -- No ReadOnly warnings
 */
int
init_disp_list(pab, want_status, match_ent, match_addr, no_ro_warning)
PerAddrBook *pab;
OpenStatus   want_status;
AdrBk_Entry *match_ent;
char        *match_addr;
int          no_ro_warning;
{
    register OpenStatus new_status;

    dprint(7, (debugfile, "- init_disp_list -\n"));
    dprint(7, (debugfile,
	    "    addrbook nickname = %s filename = %s want ostatus %s\n",
		pab->nickname ? pab->nickname : "<null>",
		pab->filename ? pab->filename : "<null>",
		want_status==Open ?
			    "Open" :
			    want_status==HalfOpen ?
				    "HalfOpen" :
				    want_status==NoDisplay ?
					    "NoDisplay" :
					    want_status==Same ?
						"Same" :
						"Closed"));
    dprint(7, (debugfile, "    ostatus was %s, want %s\n",
		pab->ostatus==Open ?
			    "Open" :
			    pab->ostatus==HalfOpen ?
				    "HalfOpen" :
				    pab->ostatus==NoDisplay ?
					    "NoDisplay" :
					    "Closed",
		want_status==Open ?
			    "Open" :
			    want_status==HalfOpen ?
				    "HalfOpen" :
				    want_status==NoDisplay ?
					    "NoDisplay" :
					    want_status==Same ?
						"Same" :
						"Closed"));

    if(want_status == Same)
	want_status = pab->ostatus;
    new_status = want_status;  /* optimistic default */

    /* free display memory */
    if(pab->disp_list != NULL)
	free_disp_list(pab);

    /*
     * If we don't need it, release the addrbook memory by calling
     * adrbk_close().
     */
    if((want_status == Closed || want_status == HalfOpen) &&
	pab->address_book != NULL){
	adrbk_close(pab->address_book);
	pab->address_book = NULL;
    }
    /* If we want the addrbook read in and it hasn't been, do so */
    else if((want_status == Open || want_status == NoDisplay) &&
	pab->address_book == NULL){
	if(pab->access != NoAccess){
	    char warning[800]; /* place to put a warning */

	    warning[0] = '\0';
	    pab->address_book = adrbk_open(pab->filename,
					   ps_global->home_dir, warning);
	    if(pab->address_book == NULL){
		pab->access = NoAccess;
		if(want_status == Open){
		    new_status = HalfOpen;  /* best we can do */
		    q_status_message2(1, 2, 4,
			  "\007Error opening/creating address book %s: %s",
			  pab->nickname, error_description(errno));
		}
		else
		    new_status  = Closed;
		dprint(1, (debugfile, "Error opening address book %s: %s\n",
			  pab->nickname, error_description(errno)));
	    }
	    else{
		if(adrbk_access(pab->filename, EDIT_ACCESS) == 0)
		    pab->access = ReadWrite;
		else
		    pab->access = ReadOnly;
		new_status = want_status;
		dprint(1,
		      (debugfile, "Address book %s (%s) opened with %d items\n",
		       pab->nickname, pab->filename,
		       adrbk_count(pab->address_book)));
		if(*warning){
		    dprint(1, (debugfile,
				 "Addressbook parse error in %s (%s): %s\n",
				 pab->nickname, pab->filename, warning));
		    if(!pab->gave_parse_warnings && want_status == Open){
			pab->gave_parse_warnings++;
			q_status_message2(1, 2, 4, "%s: %s",
			    as.n_addrbk > 1 ? pab->nickname : "addressbook",
			    warning);
		    }
		}
	    }
	}
	else{
	    if(want_status == Open){
		new_status = HalfOpen;  /* best we can do */
		q_status_message1(1, 2, 4,
	        "\007Insufficient file permissions for opening address book %s",
	        pab->nickname);
	    }
	    else
		new_status = Closed;
	}
    }

    pab->ostatus = new_status;

    if(new_status == Open || new_status == HalfOpen)
	return(set_disp_list(pab, match_ent, match_addr, as.n_addrbk==1,
							    no_ro_warning));
    else
	return -1;
}


/*
 * Set the display list structure for this addressbook.
 *
 *  Args: pab        -- the PerAddrBook ptr
 *        match_ent  -- address of entry to return line number of address
 *                      which matches.  That is, match_ent is an address
 *                      and we return the disp_list line number of that
 *			address.  Note, the line number we return is the
 *			global display list line number, not the number
 *			relative to this particular address book.
 *        match_addr -- if this is non-null, we return the line number of
 *                      the list entry corresponding to it.  This is only used
 *			if match_ent is a list and this is an entry within
 *			that list.
 *        no_title   -- if set, don't put a title in for this addrbook
 *        no_ro_warning -- if set, don't put ReadOnly in title
 *
 *  Result: The disp_list structure is initialized or re-initialized.
 *
 * A number of the address book editing functions need to know what line
 * in the disp_list the just edited item is on so they can paint the screen
 * more efficiently, so they pass in pointers to the entry just edited
 * and this returns the line number in the new disp_list where they were found. 
 * 
 * This sorts the list with the distribution lists at the end.
 */
int
set_disp_list(pab, match_ent, match_addr, no_title, no_ro_warning)
PerAddrBook *pab;
AdrBk_Entry *match_ent;
char        *match_addr;
int          no_title,
             no_ro_warning;
{
    int             n, n_addrs, match_n = -1;
    int             are_lists = 0;
    int             are_non_lists = 0;
    int             len;
    int             screen_width = ps_global->ttyo->screen_cols;
    AdrBk_Entry    *ae;
    register char **p;

    dprint(7, (debugfile, "- set_disp_list -\n"));
    dprint(7, (debugfile,
	    "addrbook nickname = %s filename = %s\n",
		pab->nickname ? pab->nickname : "<null>",
		pab->filename ? pab->filename : "<null>"));

    if(pab->ostatus == Open){
	n_addrs = adrbk_count(pab->address_book);
        pab->dl_allocated = n_addrs + INIT_DISPLIST_SIZE;
    }
    else if(pab->ostatus == HalfOpen)
        pab->dl_allocated = INIT_DISPLIST_SIZE;
    else
	panic("programmer botch in set_disp_list");

    /* get new memory */
    pab->disp_list = (AddrScrn_Disp *)
			     fs_get(pab->dl_allocated * sizeof(AddrScrn_Disp));
    memset((void *)pab->disp_list, 0, pab->dl_allocated*sizeof(AddrScrn_Disp));

    if(!no_title){
	/* line of dashes */
	if(++pab->dl_used > pab->dl_allocated)
	    alloc_more_ab_display(pab);
	memset((void *)tmp_20k_buf, '-', screen_width * sizeof(char));
	tmp_20k_buf[screen_width] = '\0';
	pab->disp_list[pab->dl_used - 1].a.txt = cpystr(tmp_20k_buf);
	pab->disp_list[pab->dl_used - 1].type  = Text;

	/* title for this addrbook */
	if(++pab->dl_used > pab->dl_allocated)
	    alloc_more_ab_display(pab);
	memset((void *)tmp_20k_buf, ' ', screen_width * sizeof(char));
	tmp_20k_buf[screen_width] = '\0';
	sprintf(tmp_20k_buf+screen_width+2, "AddressBook <%s>",
		    pab->nickname ? pab->nickname : pab->filename);
	len = strlen(tmp_20k_buf+screen_width+2);
	strncpy(tmp_20k_buf, tmp_20k_buf+screen_width+2, len);
	if(!no_ro_warning){
	    char *q;

	    if(pab->access == ReadOnly){
		if(screen_width - len - strlen(READONLY) > 3){
		    q = tmp_20k_buf + screen_width - strlen(READONLY);
		    strcpy(q, READONLY);
		}
	    }
	    else if(pab->access == NoAccess){
		if(screen_width - len - strlen(NOACCESS) > 3){
		    q = tmp_20k_buf + screen_width - strlen(NOACCESS);
		    strcpy(q, NOACCESS);
		}
	    }
	}
	pab->disp_list[pab->dl_used - 1].a.txt = cpystr(tmp_20k_buf);
	pab->disp_list[pab->dl_used - 1].type  = Title;

	/* line of dashes */
	if(++pab->dl_used > pab->dl_allocated)
	    alloc_more_ab_display(pab);
	memset((void *)tmp_20k_buf, '-', screen_width * sizeof(char));
	tmp_20k_buf[screen_width] = '\0';
	pab->disp_list[pab->dl_used - 1].a.txt = cpystr(tmp_20k_buf);
	pab->disp_list[pab->dl_used - 1].type  = Text;

	/* blank line */
	if(++pab->dl_used > pab->dl_allocated)
	    alloc_more_ab_display(pab);
	pab->disp_list[pab->dl_used - 1].a.txt = cpystr("");
	pab->disp_list[pab->dl_used - 1].type  = Text;
    }

    /* read in all the addresses and put them in the display list */
    if(pab->ostatus == Open){
      /* first pass for non-lists */
      for(n = 0; n < n_addrs; n++){

        ae = adrbk_get(pab->address_book, (unsigned)n);

        if(ae->tag == List){
            are_lists++;
            continue;
        }
	else
            are_non_lists++;

	if(++pab->dl_used > pab->dl_allocated)
	    alloc_more_ab_display(pab);

        if(ae == match_ent)
          match_n = pab->dl_used - 1;

        pab->disp_list[pab->dl_used - 1].a.abe = ae;
        pab->disp_list[pab->dl_used - 1].other = NULL;
        pab->disp_list[pab->dl_used - 1].type  = Single;
    }

    if(are_lists){

	/* Second pass for lists */
        for(n = 0; n < n_addrs; n++){

            ae = adrbk_get(pab->address_book, (unsigned)n);

            if(ae->tag == Atom)
                continue;

	    /* if no simple addrs, skip the first blank */
	    if(are_non_lists){
		/*----- First a blank line ------*/
		if(++pab->dl_used > pab->dl_allocated)
		    alloc_more_ab_display(pab);
		pab->disp_list[pab->dl_used - 1].a.txt = cpystr("");
		pab->disp_list[pab->dl_used - 1].type  = Text;
	    }
	    else
		are_non_lists++;

            /*----- The head of the list -----*/
	    if(++pab->dl_used > pab->dl_allocated)
		alloc_more_ab_display(pab);
            pab->disp_list[pab->dl_used - 1].a.abe = ae;
            pab->disp_list[pab->dl_used - 1].other = NULL;
            pab->disp_list[pab->dl_used - 1].type  = ListHead;
    
            if(ae == match_ent && match_addr == NULL)
              match_n = pab->dl_used - 1;

            if(ae->addr.list != NULL){
              for(p = ae->addr.list; *p != NULL; p++){
                  /*----- each entry in the list ------*/
		  if(++pab->dl_used > pab->dl_allocated)
		    alloc_more_ab_display(pab);
                  pab->disp_list[pab->dl_used - 1].a.abe = ae;
                  pab->disp_list[pab->dl_used - 1].other = *p;
                  pab->disp_list[pab->dl_used - 1].type  = ListEnt;
  
                  if(ae == match_ent && match_addr == *p)
                    match_n = pab->dl_used - 1;
              }
	    }
        }
      }
      if(!are_lists && !are_non_lists){
	/* Empty line */
	if(++pab->dl_used > pab->dl_allocated)
	    alloc_more_ab_display(pab);
	pab->disp_list[pab->dl_used - 1].type  = Empty;
      }
    }
    else if(pab->access == NoAccess){
	/* line of dashes */
	if(++pab->dl_used > pab->dl_allocated)
	    alloc_more_ab_display(pab);
	pab->disp_list[pab->dl_used - 1].a.txt = cpystr(NO_PERMISSION);
	pab->disp_list[pab->dl_used - 1].type  = Text;
    }
    else{
	/* Click Here line */
	if(++pab->dl_used > pab->dl_allocated)
	    alloc_more_ab_display(pab);
	pab->disp_list[pab->dl_used - 1].type  = ClickHere;
    }

    /* blank line */
    if(++pab->dl_used > pab->dl_allocated)
	alloc_more_ab_display(pab);
    pab->disp_list[pab->dl_used - 1].a.txt = cpystr("");
    pab->disp_list[pab->dl_used - 1].type  = Text;

    reset_dl_ends();

    /* convert return value to global coordinates */
    if(match_n >= 0)
	match_n += (as.dl_ends[as.cur] + 1);
    dprint(9, (debugfile, "set_disp_list: return match_n = %d\n", match_n));
    return(match_n);
}


/*
 * Increase the amount of memory allocated for the addressbook screen.
 */
void
alloc_more_ab_display(pab)
PerAddrBook *pab;
{

    pab->dl_allocated *= 2;
    fs_resize((void **)&pab->disp_list,
			    pab->dl_allocated * sizeof(AddrScrn_Disp));

    dprint(9,
	(debugfile,
	    "- alloc_more_ab_display -\n  for \"%s\" dl_allocated now %d\n",
	     pab->nickname, pab->dl_allocated));
}


/* return this when we get past the end of the list */
static AddrScrn_Disp end = {End};
/*
 * Return a ptr to the row'th line of the global disp_list.  The first row
 * at the top of the available space on the first screen is row 0.
 * Returns &end if row is at the end.
 * Returns NULL if row is out of range.
 */
AddrScrn_Disp *
dlist(row)
int row;
{
    int addrbk_index, local_row;

    if(row == as.tot_used)
	return(&end);

    if(convert_dl(row, &addrbk_index, &local_row) < 0)
	return NULL;

    return(&as.adrbks[addrbk_index].disp_list[local_row]);
}


/*
 * Given the global_row number that applies to all of the addressbooks,
 * return the addressbook index in adrbk_index and the row number within
 * that addressbook in local_row.
 *
 * Returns -1 if out of range, 0 otherwise.
 */
int
convert_dl(global_row, adrbk_index, local_row)
int  global_row,
    *adrbk_index,
    *local_row;
{
    register int j = 0;
    register int *ends;

    if(global_row > as.tot_used - 1 || global_row < 0)
	return -1;

    ends = as.dl_ends;

    while(global_row > ends[j])
	j++;
    
    *adrbk_index = j - 1;

    *local_row   = global_row - ends[*adrbk_index] - 1;

    return 0;
}


/*
 *
 * Args: top_of_screen  --  index in disp_list of 1st line on the screen
 *       lines_per_page --  number of lines available to us
 *       start_disp     --  line to start displaying on when redrawing, 0 is
 *	 		    the top_of_screen
 *       cur_line       --  current line number (0 is 1st line we display)
 *       cur_fld        --  current field number: 0, 1, or 2
 *       old_line       --  old line number
 *       old_fld        --  old field number
 *       redraw         --  flag requesting redraw as opposed to update of
 *			    current line
 *
 * Result: lines painted on the screen
 *
 * It either redraws the screen from line "start_disp" down or
 * moves the cursor from one field to another.
 */
void
display_book(top_of_screen, lines_per_page, start_disp, cur_line,
             cur_fld, old_line, old_fld, redraw, fld_width)
int top_of_screen,
    lines_per_page,
    start_disp,
    cur_line,
    cur_fld,
    old_line,
    old_fld,
    redraw;
int fld_width[3];
{
    int i, j, cursor;

    dprint(9, (debugfile,
    "- display_book() -\n   top %d start %d cur_line %d cur_fld %d redraw %d\n",
	 top_of_screen, start_disp, cur_line, cur_fld, redraw));

    if(lines_per_page <= 0)
      return;

    if(redraw){
	AddrScrn_Disp *dl;

        /*------ Repaint all of the screen or bottom part of screen ---*/
        j = top_of_screen + start_disp;
        for(i = start_disp; i < lines_per_page; i++){
            cursor = (i == cur_line) ? cur_fld : -1;
            MoveCursor(i + HEADER_LINES, 0);
            CleartoEOLN();
            paint_line(i + HEADER_LINES, (dl=dlist(j)), -1, cursor, fld_width);
            if(dl && dl->type != End)
              j++;
        }
    }
    else{

        /*----- Only update current, or move the cursor -----*/
        if(cur_line != old_line || cur_fld != old_fld){

            /*---- Repaint old position to erase "cursor" ----*/
            paint_line(old_line + HEADER_LINES,
                       dlist(top_of_screen + old_line),
                       old_fld, -1, fld_width);
        }

        /*---- paint the position with the cursor ---*/
        paint_line(cur_line + HEADER_LINES,
		   dlist(top_of_screen + cur_line),
                   cur_fld, cur_fld, fld_width);
    }
    fflush(stdout);
}


/*
 * Paint a line on the screen
 *
 * Args: line    --  Line on screen to paint, 0 is first available line
 *       dl      --  The display list ptr for this line
 *       field   --  The field in the line to paint (see below)
 *       cursor  --  The field the cursor is on (see below)
 *
 * Result: Line is painted
 *
 * The field and cursor arguments have special meaning:
 *   -1 means the whole line, n means field n.  There are 3 fields, 0, 1, and 2.
 *   If the cursor is -1 (or anything other than 0, 1, or 2), then
 *   there is no highlighting in this line.
 *
 * The three field widths for the formatting are passed in.  There is an
 * implicit 2 spaces between the fields.
 *
 *    | fld_width[0] chars |__| fld_width[1] |__| fld_width[2] |
 */
void
paint_line(line, dl, field, cursor, fld_width)
int            line;
AddrScrn_Disp *dl;
int            field,
	       cursor;
int	       fld_width[3];
{
    int   fld1_pos,
	  fld2_pos,
	  col,
	  screen_width;
    char  fld0_control[11],
	  fld1_control[11],
	  fld2_control[11],
	  full_control[11];

    dprint(10, (debugfile, "- paint_line -\n"));

    screen_width = ps_global->ttyo->screen_cols;

    fld1_pos = min(fld_width[0] + 2, screen_width - 1);
    fld2_pos = min(fld1_pos + fld_width[1] + 2, screen_width - 1);

    /* so we don't have to pass funky args to Putlinex() */
    sprintf(fld0_control, "%%-%d.%ds", fld_width[0], fld_width[0]);
    sprintf(fld1_control, "%%-%d.%ds", fld_width[1], fld_width[1]);
    sprintf(fld2_control, "%%-%d.%ds", fld_width[2], fld_width[2]);
    sprintf(full_control, "%%-%d.%ds", screen_width, screen_width);


    switch(dl->type){

      case Text:
      case Title:
	/* center it */
	col = (screen_width - strlen(dl->a.txt))/2;
	if(col >= 0)
	    PutLine0(line, col, dl->a.txt);
	else
	    PutLine1(line, 0, full_control, dl->a.txt);
	break;

      case ClickHere:
	if(cursor != -1)
	  StartInverse();
	col = (screen_width - strlen(CLICKHERE))/2;
	if(col >= 0)
	    PutLine0(line, col, CLICKHERE);
	else
	    PutLine1(line, 0, full_control, CLICKHERE);
	if(cursor != -1)
	  EndInverse();
	break;
    
      case Empty:
	if(cursor != -1)
	  StartInverse();
	col = (screen_width - strlen(EMPTY))/2;
	if(col >= 0)
	    PutLine0(line, col, EMPTY);
	else
	    PutLine1(line, 0, full_control, EMPTY);
	if(cursor != -1)
	  EndInverse();
	break;
    
      case Single:
	if(field == -1 || field == 0){
	    if(cursor == 0)
	      StartInverse();
	    PutLine1(line, 0, fld0_control, dl->a.abe->nickname);
	    if(cursor == 0)
	      EndInverse();
	}
	if((field == -1 || field == 1) && fld_width[1] != 0){
	    if(cursor == 1)
	      StartInverse();
	    PutLine1(line, fld1_pos, fld1_control, dl->a.abe->fullname);
	    if(cursor == 1)
	      EndInverse();
	}
	if((field == -1 || field == 2) && fld_width[2] != 0){
	    if(cursor == 2)
	      StartInverse();
	    PutLine1(line, fld2_pos, fld2_control, dl->a.abe->addr.addr);
	    if(cursor == 2)
	      EndInverse();
	}
	break;
    
      case ListHead:
	if(field == -1 || field == 0){
	    if(cursor == 0)
	      StartInverse();
	    PutLine1(line, 0, fld0_control, dl->a.abe->nickname);
	    if(cursor == 0)
	      EndInverse();
	}
	if((field == -1 || field == 1) && fld_width[1] != 0){
	    if(cursor == 1)
	      StartInverse();
	    PutLine1(line, fld1_pos, fld1_control, dl->a.abe->fullname);
	    if(cursor == 1)
	      EndInverse();
	}
	if(field == -1 && fld_width[2] != 0)
	    PutLine1(line, fld2_pos, fld2_control, DISTLIST);
	break;
        
      case ListEnt:
	if(field == -1){
	    if(line == HEADER_LINES){
	      char temp[50];
	      sprintf(temp,"%.18s (continued)", dl->a.abe->fullname);
	      PutLine1(line, fld1_pos, fld1_control, temp);
	    }
	}
	if(fld_width[2] != 0){
	    if(cursor == 0 || cursor == 1 || cursor == 2)
	      StartInverse();
	    PutLine1(line, fld2_pos, fld2_control, dl->other);
	    if(cursor == 0 || cursor == 1 || cursor == 2)
	      EndInverse();
	}
	break;

      case End:
        break;
    }
}


/*
 * Comparison function for qsorting array of ints.
 *
 * Args: pointers to two ints.
 */
int
cmp_ints(i1, i2)
const QSType *i1, *i2;
{
    int a = *(int *)i1, b = *(int *)i2;

    return(a - b);
}


static int old_fld_width[3];
int percentiles[] = {100, 98, 95, 90, 80, 70, 60, 50, 40, 25};
#define N_PERCENTILES 10
/*
 * Set field widths for the three columns of the display
 *
 * Args: fld_width -- a 3 element array in which the answer is returned
 *
 * Col1 and col2 are arrays which contain some widths useful for
 * formatting the screen.  The 0th element is the max
 * width in that column.  The 1st element is the 98th percentile, ...
 * The idea is that we'd like to get as much of the useful information on the
 * screen as possible.
 *
 *   0     100th percentile (max)
 *   1     98th percentile
 *   2     95th percentile
 *   3     90th
 *   4     80th
 *   5     70th
 *   6     60th
 *   7     50th
 *   8     40th
 *   9     25th
 *
* BUG -- Should base the calculation on 
* a sample of the lines instead of all of them.  That way it will
* still perform quickly enough so that it can be called frequently.  The idea
* is that we want to avoid huge sorts.
 * Returns 1 if the widths changed since the last time called.
 */
int
calculate_field_widths(fld_width)
int fld_width[3];
{
    int col1[N_PERCENTILES + 3], col2[N_PERCENTILES + 3];
    int *col1_widths, *col2_widths;
    register AddrScrn_Disp *dl;
    int lineno = 0;
    int max_nickname;
    int space_left;
    int ret = 0;
    int i = 0;

    dprint(9, (debugfile, "- calculate_field_widths -\n"));

    col1_widths = (int *)fs_get(as.tot_used * sizeof(int));
    col2_widths = (int *)fs_get(as.tot_used * sizeof(int));
    memset((void *)col1_widths, 0, as.tot_used * sizeof(int));
    memset((void *)col2_widths, 0, as.tot_used * sizeof(int));
    max_nickname = 0;

    dprint(10, (debugfile,
	"calculate_field_widths: about to figure out all widths\n"));

    for(dl=dlist(lineno); dl->type != End; dl = dlist(++lineno)){
	/* fill in all the col1 and col2 widths */
      switch(dl->type){
	case Single:
	  col2_widths[lineno] = strlen(dl->a.abe->addr.addr);
	  /* fall through */

	case ListHead:
	  max_nickname        = max(max_nickname, strlen(dl->a.abe->nickname));
	  col1_widths[lineno] = strlen(dl->a.abe->fullname);
	  break;

	case ListEnt:
	  col2_widths[lineno] = strlen(dl->other);
	  break;
	
	/* Otherwise, leave it alone */
      }
    }

    dprint(10, (debugfile,
	"calculate_field_widths: sorting %d col1_widths\n", as.tot_used));
    qsort((QSType *)col1_widths, (size_t)as.tot_used, sizeof(int), cmp_ints);
    dprint(10, (debugfile,
	"calculate_field_widths: sorting %d col2_widths\n", as.tot_used));
    qsort((QSType *)col2_widths, (size_t)as.tot_used, sizeof(int), cmp_ints);

    /*
     * initialize the percentiles arrays
     * For example, col2[1] is the width col2 needs to be in order that
     * 98% of the entries will fit.
     *
     * Restrict them to be at least two columns wide, if possible.
     */
    dprint(10, (debugfile,
	"calculate_field_widths: initialize percentile arrays\n"));
    for(i = 0; i < N_PERCENTILES; i++){
	col1[i] = col1_widths[(percentiles[i]*as.tot_used)/100 - 1];
	col1[i] = max(2, col1[i]);
	col2[i] = col2_widths[(percentiles[i]*as.tot_used)/100 - 1];
	col2[i] = max(2, col2[i]);
    }
    col1[N_PERCENTILES] = 3;
    col2[N_PERCENTILES] = 3;
    col1[N_PERCENTILES + 1] = 2;
    col2[N_PERCENTILES + 1] = 2;
    col1[N_PERCENTILES + 1] = 1;  /* try hard to fit all columns on screen */
    col2[N_PERCENTILES + 1] = 1;

    space_left   = ps_global->ttyo->screen_cols;

    /* All of the nickname field should be visible. */
    fld_width[0] = min(max_nickname, space_left);

    /*
     * The 2 is for two blank columns between nickname and next field.
     * Those blank columns are taken automatically in paint_line().
     */
    space_left  -= (fld_width[0] + 2);

    if(space_left > 0){
	for(i = 0; i < N_PERCENTILES + 3; i++){
	    /* try fitting most of each field in */
	    if(col1[i]+2+col2[i] <= space_left){
		int extra;

		extra = space_left - col1[i] - 2 - col2[i];
		fld_width[2] = col2[i] + extra/2;
		fld_width[1] = space_left - 2 - fld_width[2];
		break;
	    }
	}
	/*
	 * None of them would fit.  Toss fld2.
	 */
	if(i == N_PERCENTILES + 3){
	    fld_width[1] = space_left;
	    fld_width[2] = 0;
	}
    }
    else{
	fld_width[1] = 0;
	fld_width[2] = 0;
    }

    for(i = 0; i < 3; i++){
	if(fld_width[i] != old_fld_width[i]){
	    ret++;  /* Tell the caller the screen changed */
	    old_fld_width[i] = fld_width[i];
	}
    }

    fs_give((void **)&col1_widths);
    fs_give((void **)&col2_widths);
    dprint(10, (debugfile, "calculate_field_widths: returning\n"));

    return(ret);
}


void
redraw_addr_screen()
{
    int i;
    int fld_width[3];

    dprint(9, (debugfile, "- redraw_addr_screen -\n"));

    as.l_p_page     = ps_global->ttyo->screen_rows -FOOTER_LINES -HEADER_LINES;
    if(as.l_p_page <= 0)
        return;
    i               = as.cur_row + as.top_ent;
    as.top_ent      = (i/as.l_p_page) * as.l_p_page;
    as.cur_row      = i - as.top_ent;
    as.old_cur_row  = as.cur_row;

    (void)calculate_field_widths(fld_width);

    display_book(as.top_ent, as.l_p_page, 0, as.cur_row, as.cur_fld, -1, -1, 1,
		 fld_width);
}


/*
 * Little front end for address book screen so it can be called out
 * of the main command loop in pine.c
 */
void
addr_book_screen(pine_state)
struct pine *pine_state;
{
    dprint(1, (debugfile, "--- addr_book_screen ---\n"));

    (void)addr_book(AddrBookScreen);
    end_adrbks();
    pine_state->prev_screen = addr_book_screen;
}


/*ARGSUSED*/
/*
 * Call address book from message composer
 *
 * Args: error_mess -- pointer to return error messages in (unused here)
 *
 * Returns: pointer to returned address, or NULL if nothing returned
 */
char *
addr_book_compose(error_mess)
char **error_mess;
{
    char *p;

    dprint(1, (debugfile, "--- addr_book_compose ---\n"));

    p = addr_book(SelectAddr);

    end_adrbks();

    return(p);
}


/*
 * Call address book from take address screen
 *
 * Returns: pointer to returned nickname, or NULL if nothing returned
 *
 * The caller is assumed to handle the closing of the addrbooks, so we
 * don't call end_adrbks().
 */
char *
addr_book_takeaddr()
{
    char *p;

    dprint(1, (debugfile, "- addr_book_takeaddr -\n"));

    p = addr_book(SelectNick);

    return(p);
}


static struct key ab_keys[] =
     {{"?","Help",0},       {"O","OTHER CMDS",0},   {NULL,NULL,0},
      {NULL,NULL,0},        {"P","PrevField",0},    {"N","NextField",0},
      {"-","PrevPage",0},   {"Spc","NextPage",0},   {"D","Delete",0},
      {"A","Add",0},        {"S","CreateList",0},   {NULL,NULL,0},
      {"?","Help",0},       {"O","OTHER CMDS",0},   {"Q","Quit",0},
      {"C","ComposeTo",0},  {"L","ListFldrs",0},    {"G","GotoFldr",0},
      {"I","Index",0},      {"W","WhereIs",0},      {"Y","prYnt",0},
      {"F","FCC...",0},     {NULL,NULL,0},          {NULL,NULL,0}};
static struct key_menu ab_keymenu =
	{sizeof(ab_keys)/(sizeof(ab_keys[0])*12), 0, 0,0,0,0, ab_keys};
#define OTHER_KEY  1
#define MAIN_KEY   2
#define EDIT_KEY   3
#define DELETE_KEY 8
#define ADD_KEY    9
#define CREATE_KEY 10
#define ADDTO_KEY  11
#define FCC_KEY    21


/*
 *  Main address book screen 
 *
 * Control loop for address book.  Commands are executed out of a big
 * switch and screen painting is done.
 * The argument controls whether or not it is
 * called to return an address to the composer, a nickname to the TakeAddr
 * screen, or just for address book maintenance.
 *
 * Args: style -- how we were called
 *
 * Return: might return a string for the composer to edit (if SelectAddr style)
 *         or a nickname (if SelectNick).
 */
char *
addr_book(style)
AddrBookArg style;
{
    int		     i, r, c, orig_c, fl,
		     command_line,
		     new_ent,
		     did_delete_flag,
                     quit,           /* loop control                         */
		     current_changed_flag,  /* only current row needs update */
		     was_clickable_last_time, /* on CLICKHERE last time thru */
		     old_cur_fld,    /* cur_fld last time through            */
		     new_line,       /* new line number after operation      */
		     new_fld,        /* new field number after operation     */
		     new_top_ent,    /* entry on top of screen after oper    */
		     mangled_screen, /*                                      */
		     mangled_header, /* These track whether or not we        */
		     mangled_footer, /* need to repaint the various parts    */
		     mangled_body,   /* of the screen.                       */
		     start_disp,     /* Paint from this line down (0 is top) */
		     readonly,       /* cur addrbook read only               */
		     empty,          /* cur addrbook empty                   */
		     noaccess,       /* cur addrbook permission denied       */
		     are_selecting;  /* called as ^T selector                */
    char            *title,
                    *addr;
    bitmap_t         bitmap;
    struct key_menu *km;
    OtherMenu        what;
    int              fld_width[3];
    PerAddrBook     *pab;

    km = &ab_keymenu;

    are_selecting = (style == SelectAddr || style == SelectNick);

    dprint(1, (debugfile, "--- addr_book ---  (are_selecting %d)\n",
			    are_selecting));

    if(style == SelectAddr){
        /* Coming in from the composer, need to reset this stuff */
        get_windsize(ps_global->ttyo);
        init_signals();
        clear_cursor_pos();
        mark_status_dirty();
    }

    command_line = -3;  /* third line from the bottom */
    title        = (style == AddrBookScreen) ? "ADDRESS BOOK" :
		    (style == SelectAddr)     ? "COMPOSER: SELECT ADDRESS" :
		     (style == SelectNick)     ? "TAKEADDR: SELECT NICKNAME" :
						  "UNKNOWN SCREEN STYLE!";
    what         = FirstMenu;

    if(!init_addrbooks(HalfOpen, 1, 1, are_selecting)){
	q_status_message(1, 1, 3, "\007No Address Book Configured");
	if(!are_selecting)
	    ps_global->next_screen = ps_global->prev_screen;
        return NULL;
    }

    old_cur_fld              = 0;
    quit                     = 0;
    mangled_screen           = 1;
    mangled_header           = 1;
    mangled_body             = 1;
    mangled_footer           = 1;
    start_disp               = 0;
    current_changed_flag     = 0;
    was_clickable_last_time  = 0; /* doesn't matter what it is */

    (void)calculate_field_widths(fld_width);

    c = 'x'; /* For display_message the first time through */


    do{
	ps_global->redrawer = redraw_addr_screen;

        if(new_mail(NULL, 0, c == NO_OP_IDLE ? 0 : 2) >= 0)
	    mangled_header++;

        if(streams_died())
            mangled_header++;

	if(mangled_screen){
	    ClearScreen();
	    mangled_header++;
	    mangled_body++;
	    mangled_footer++;
	    start_disp     = 0;
	    mangled_screen = 0;
	}

	if(mangled_header){
            set_titlebar(title, ps_global->context_current,
			 ps_global->cur_folder, ps_global->msgmap, 1,
			 are_selecting ? FolderName : MessageNumber, 0, 0);
	    mangled_header = 0;
	}

	if(mangled_body){
	    int old_cur;

	    if(as.tot_used != 0){

		if(calculate_field_widths(fld_width))
		    start_disp = 0;

                display_book(as.top_ent,
                             as.l_p_page,
                             start_disp,
                             as.cur_row,  
                             as.cur_fld,  
                             as.old_cur_row,  
                             old_cur_fld,
                             1,
			     fld_width);
	    }
	    else{
		for(i = HEADER_LINES;
		    i < ps_global->ttyo->screen_rows - FOOTER_LINES + 1;
		    i++){
		    MoveCursor(i, 0);
		    CleartoEOLN(); 
		}
	        q_status_message(1, 1, 3, "\007No Addresses in Address Book.");
                as.cur_row = 0;
                as.cur_fld = 0;
	    }
	    as.old_cur_row = as.cur_row;
	    old_cur_fld    = as.cur_fld;
	    mangled_body   = 0;
            start_disp     = 0;
	    old_cur        = as.cur;
	    as.cur         = cur_addr_book();
	    pab            = &as.adrbks[as.cur];
	    if(as.cur != old_cur)
	        q_status_message1(0, 0, 2, "Now in addressbook %s",
						pab->nickname);
	}
	/* current entry has been changed */
	else if(current_changed_flag){
	    int old_cur;
	    int need_redraw;

	    need_redraw = calculate_field_widths(fld_width);

	    /*---------- Update the current entry, (move or change) -------*/
	    display_book(as.top_ent,
			 as.l_p_page,
			 as.cur_row,
			 need_redraw ? 0 : as.cur_row,  
			 as.cur_fld,  
			 as.old_cur_row,  
			 old_cur_fld,
			 need_redraw,
			 fld_width);
	    as.old_cur_row       = as.cur_row;
	    old_cur_fld          = as.cur_fld;
	    current_changed_flag = 0;
	    old_cur              = as.cur;
	    as.cur               = cur_addr_book();
	    pab                  = &as.adrbks[as.cur];
	    if(as.cur != old_cur)
	        q_status_message1(0, 0, 2, "Now in addressbook %s",
						pab->nickname);
        }

	dprint(9, (debugfile,
	 "addr_book: top of loop, addrbk %d top_ent %d cur_row %d cur_fld %d\n",
	  as.cur, as.top_ent, as.cur_row, as.cur_fld));

	/*
	 * This is check to catch the case where we move from a non-
	 * clickable row into a clickable row or vice versa.
	 */
	if(!mangled_footer && ((was_clickable_last_time &&
		!entry_is_clickable(as.top_ent+as.cur_row)) ||
	   (!was_clickable_last_time &&
		entry_is_clickable(as.top_ent+as.cur_row))))
	    mangled_footer++;
	was_clickable_last_time = entry_is_clickable(as.top_ent+as.cur_row);

        if(mangled_footer){

	    setbitmap(bitmap);
	    if(are_selecting){
	      km->how_many = 1;
	      ab_keys[MAIN_KEY].name   = "E";
	      ab_keys[MAIN_KEY].label  = "ExitSelect";
	      ab_keys[EDIT_KEY].name   = "S";
	      ab_keys[EDIT_KEY].label  = "[Select]";
	      ab_keys[ADDTO_KEY].name  = "W";
	      ab_keys[ADDTO_KEY].label = "WhereIs";
	      clrbitn(OTHER_KEY, bitmap);
	      clrbitn(DELETE_KEY, bitmap);
	      clrbitn(ADD_KEY, bitmap);
	      clrbitn(CREATE_KEY, bitmap);
	      clrbitn(FCC_KEY, bitmap);
	    }
	    else{
	      km->how_many = 2;
	      ab_keys[MAIN_KEY].name   = "M";
	      ab_keys[MAIN_KEY].label  = "MainMenu";
	      if(entry_is_clickable(as.top_ent+as.cur_row)){
		  ab_keys[EDIT_KEY].name   = "S";
		  ab_keys[EDIT_KEY].label  = "[Select]";
		  clrbitn(CREATE_KEY, bitmap);
	      }
	      else{
		  ab_keys[EDIT_KEY].name   = "E";
		  ab_keys[EDIT_KEY].label  = "[Edit]";
	      }
	      ab_keys[ADDTO_KEY].name  = "Z";
	      ab_keys[ADDTO_KEY].label = "AddToList";
	    }

	    draw_keymenu(km, bitmap, ps_global->ttyo->screen_cols,
						    -2, 0, what, 0);
	    mangled_footer = 0;
	    what           = SameTwelve;
	}

	readonly = (pab->access == ReadOnly);
	noaccess = (pab->access == NoAccess);
	empty    = is_empty(as.cur_row+as.top_ent);

	/*------------ display any status messages ------------------*/
	display_message(c);
	MoveCursor(max(0, ps_global->ttyo->screen_rows - 3), 0);
	fflush(stdout);


	/*---------------- Get command and validate -------------------*/
	c = read_command();
        orig_c = c;

	if(c == ctrl('M') || c == ctrl('J')) /* set up default */
	  if(F_ON(F_USE_FK,ps_global))
	    c = PF4;
	  else
	    c = (are_selecting || entry_is_clickable(as.top_ent+as.cur_row))
									? 's'
									: 'e';
	if(c < 'z' && isupper(c))
	    c = tolower(c);

	if(km->which == 1 && c >= PF1 && c <= PF12)
            c = PF2OPF(c);

        c = validatekeys(c); 

        dprint(9, (debugfile, "Addrbook command :'%c' (%d)\n", c, c));


	/*------------- execute command ----------------*/
	switch(c){

            /*------------ Noop   (new mail check) --------------*/
          case NO_OP_IDLE:
	  case NO_OP_COMMAND: 
	    break;


            /*----------- Help -------------------*/
	  case PF1:
	  case OPF1:
	  case '?':
	  case ctrl('G'):
	    /*
	     * We need this next_screen test in order that helper() can
	     * have a return to Main Menu key.  If helper is called with
	     * a third argument of 1, that isn't one of the possibilities.
	     */
	    if(!are_selecting)
		ps_global->next_screen = SCREEN_FUN_NULL;
	    if(style == SelectAddr)
                helper(h_use_address_book, "HELP ON ADDRESS BOOK", 1);
	    else if(style == SelectNick)
                helper(h_select_nickname, "HELP ON ADDRESS BOOK", 1);
            else
                helper(h_address_book, "HELP ON ADDRESS BOOK", 0);
	    if(!are_selecting && ps_global->next_screen != SCREEN_FUN_NULL)
                quit = 1;
	    mangled_screen++;
	    break;

             
            /*---------- display other key bindings ------*/
          case PF2:
          case OPF2:
          case 'o' :
            if(are_selecting)
                goto bleep;
            if(c == 'o')
	        warn_other_cmds();
            what = NextTwelve;
            mangled_footer++;
            break;


            /*------------- Back to main menu or exit to caller -------*/
	  case PF3:
	  case 'm':
	  case 'e':
	    if(c == 'm' && are_selecting)
	        goto bleep;
	    if(c == 'e' && !are_selecting)
	        goto edit;
	    if(!are_selecting)
                ps_global->next_screen = main_menu_screen;
	    quit = 1;
	    break;


            /*------------ Edit field or select -------------*/
	  case PF4:
select:
	    if(entry_is_clickable(as.top_ent+as.cur_row)){
	        /* open this addrbook and fill in display list */
                as.cur_fld = 0;
		(void)init_disp_list(pab, Open, (AdrBk_Entry *)NULL, NULL,
		    are_selecting);
		if(!are_selecting && pab->access == ReadOnly)
	            q_status_message1(0, 1, 3, "AddressBook %s is Read Only",
				    pab->nickname);
                mangled_body++;
		break;
	    }
	    if(are_selecting){
              /*---- Select an entry to mail to or a nickname to add to ---*/
	      if(!any_addrs_avail(0)){
	          q_status_message(0, 1, 3,
        "\007No entries in address book. Use ExitSelect to leave address book");
	          break;
	      }
	      if(is_addr(as.top_ent+as.cur_row)){
		  char *error = "";
		  AddrScrn_Disp *dl;

		  dl = dlist(as.top_ent+as.cur_row);

		  if(style == SelectAddr){
		      if(dl->type == ListEnt)
			  our_build_address(
				  dl->other,
				  &addr,
				  &error,
				  NULL,
				  1);
		      else
			  /*
			   * This may get us the wrong lookup, but we're willing
			   * to live with that.  That is, if nickname is an
			   * entry in two of our addrbooks and we are in the
			   * second now, this lookup will get the first anyway,
			   * because our_build_address() just goes through all
			   * the addrbooks and takes the first.  We could
			   * call adrbk_lookup on the current addrbook first
			   * and that would give us the correct address or
			   * another nickname to expand on further, but we would
			   * lose the fullname that way.  We could have
			   * our_build_address() just look in the current
			   * addrbook, but then the recursive expands would
			   * also be limited to that one addrbook.  This is
			   * probably the best.
			   */
			  our_build_address(
				  dl->a.abe->nickname,
				  &addr,
				  &error,
				  NULL,
				  1);
		      if(*error){
			  q_status_message1(0, 2, 4, "%s", error);
			  display_message(NO_OP_COMMAND);
			  sleep(1);
		      }
		      return(addr);  /* Note, composer frees this */
		  }
		  else{
		      static char selected_nickname[MAX_NICKNAME + 1];

		      strncpy(selected_nickname, dl->a.abe->nickname,
			  MAX_NICKNAME);
		      return(selected_nickname);
		  }
	      }
	      else{
	          q_status_message1(0, 1, 3, "\007No %s selected",
		      (style == SelectAddr) ? "address" : "nickname");
	          break;
	      }
	    }
edit:
	    if(!any_addrs_avail(0)){
                q_status_message(1, 2, 4, "\007No entries to edit");
                break;
            }
	    if(readonly){
                q_status_message(1, 2, 4, "\007AddressBook is ReadOnly");
                break;
	    }
	    if(empty){
                q_status_message(1, 2, 4, "\007AddressBook is Empty");
                break;
	    }
	    new_ent = (c == 'f' || c == OPF10) ?
			edit_extras(pab->address_book,
				       command_line,
				       as.cur_row+as.top_ent)
		    :
			change_address_entry(pab->address_book,
				       command_line,
				       as.cur_row+as.top_ent,
				       as.cur_fld);
            mangled_footer++;
            if(new_ent == -1 || new_ent == -3 || new_ent == -4)
	        break;
            else if(new_ent == -2){
                /* changed only current entry */
                current_changed_flag++;
                break;
            }

	    /*----- Edit succeeded, now update the screen -----*/
            new_top_ent = as.l_p_page > 0 ? 
				   (new_ent/as.l_p_page)*as.l_p_page :
				   new_ent;
	    as.cur_row  = new_ent - new_top_ent;
	    dprint(9, (debugfile,
		"Edit succeeded, new_ent %d new_top_ent %d new cur_row %d\n",
		 new_ent, new_top_ent, as.cur_row));
	    if(new_top_ent != as.top_ent){
	        as.top_ent  = new_top_ent;
		start_disp = 0;
	    }
	    else
	        start_disp = min(as.old_cur_row, as.cur_row);
	    as.old_cur_row = as.cur_row;
	    mangled_body++;
	    break;


            /*-------------- Move Right, or Down --------------*/
          case PF6:
          case ctrl('N'):
          case KEY_DOWN:
	  case ctrl('F'): 
	  case KEY_RIGHT:
          case '\t':
          case 'n':
	    if(any_addrs_avail(0)){
                if(c == ctrl('N') || c == KEY_DOWN){
                    r = next_line(as.cur_row+as.top_ent, as.cur_fld,
                                &new_line, &new_fld);
		    if(r == 0){
			q_status_message(0,0,1,"\007Already on last line.");
			break;
		    }
                }
                else{
                    r = next_field(as.cur_row+as.top_ent, as.cur_fld,
				&new_line, &new_fld);
		    if(r == 0){
			q_status_message(0,0,1,"\007Already on last entry.");
			break;
		    }
                }
                as.cur_row = new_line - as.top_ent;
                as.cur_fld  = new_fld;
                if(as.cur_row >=  as.l_p_page){
                    /*-- Changed pages --*/
                    as.top_ent += as.l_p_page;
                    as.cur_row -= as.l_p_page;
                    mangled_body++;
                }
		else
                    current_changed_flag++;
	    }
	    else
		q_status_message(0, 1, 3, "\007Address book is empty");
	    break;


            /*------------------- Move Left, or Up ----------------*/
          case PF5:
          case ctrl('P'):
          case KEY_UP:
	  case ctrl('B'):
	  case KEY_LEFT:
	  case 'p':
	    if(any_addrs_avail(0)){
                if(c == ctrl('P') || c == KEY_UP){
                    r = prev_line(as.cur_row+as.top_ent, as.cur_fld,
				 &new_line, &new_fld);
		    if(r == 0){
			q_status_message(0,0,1,"\007Already on first line.");
			break;
		    }
		}
                else{
                    r = prev_field(as.cur_row+as.top_ent, as.cur_fld,
				 &new_line, &new_fld);
		    if(r == 0){
			q_status_message(0,0,1,"\007Already on first entry.");
			break;
		    }
                }
                as.cur_row = new_line - as.top_ent;
                as.cur_fld = new_fld;
                if(as.cur_row < 0){
                    as.top_ent -= as.l_p_page;
                    as.cur_row += as.l_p_page;
                    mangled_body++;
                }
		else
                    current_changed_flag++;
	    }
	    else
	        q_status_message(0, 1, 3, "\007Address book is empty");
	    break;


            /*------------- Page Up ----------------*/
	  case '-':
          case ctrl('Y'): 
	  case PF7:
	  case KEY_PGUP:
            /*------------- Page Down --------------*/
          case ' ':
          case ctrl('V'): 
          case '+':		    
	  case PF8:
	  case KEY_PGDN:
	    /* if Up */
	    if(c == '-' || c == ctrl('Y') || c == PF7 || c == KEY_PGUP){
		if(as.top_ent <= 0){
		    fl = first_line(0);
		    if(fl != -1){
			if(as.cur_row == fl){ /* no change */
			    q_status_message(0, 1, 1,
				"\007Already on first page.");
			    break;
			}
		    }
		}
		else
		    as.top_ent -= max(as.l_p_page, 1);
		as.cur_row  = 0;
		/* find first active line */
		fl = first_line(as.top_ent);
	    }
	    /* else Down */
	    else{
		if(as.top_ent + as.l_p_page < as.tot_used){
		    as.top_ent += max(as.l_p_page, 1); /* go to next page */
		    as.cur_row  = 0;
		    /* find first active line */
		    fl = first_line(as.top_ent); /* at top of page */
		}
		else{
		    /* find last active line, already on last page */
		    fl = first_line(as.tot_used - 1); /* go to last line */
		    if(fl != -1){
			if(as.cur_row == (fl - as.top_ent)){ /* no change */
			    q_status_message(0, 1, 1,
				"\007Already on last page.");
			    break;
			}
		    }
		}
	    }

	    start_disp  = 0;
	    as.cur_fld  = 0;
	    if(fl != -1)
		as.cur_row = fl - as.top_ent;
	    as.old_cur_row = as.cur_row;
	    mangled_body++;
	    break;


	    /*------------- Delete item from addrbook ---------*/
	  case PF9:
	  case 'd': 
	    if(are_selecting)
	        goto bleep;
	    if(!any_addrs_avail(0)){
                q_status_message(1, 2, 4, "\007No entries to delete");
                break;
	    }
	    if(entry_is_clickable(as.top_ent+as.cur_row)){
                q_status_message(1, 2, 4,
		    "\007AddressBook not expanded, use \"S\" to expand");
		break;
	    }
	    if(readonly){
                q_status_message(1, 2, 4, "\007AddressBook is ReadOnly");
		break;
	    }
	    if(empty){
                q_status_message(1, 2, 4, "\007AddressBook is Empty");
                break;
	    }
            did_delete_flag = addr_book_delete(pab->address_book,
					       command_line,
                                               as.cur_row+as.top_ent);
	    mangled_footer++;
	    if(did_delete_flag){
		/*
		 * In case the line we're now at is not an active field (i.e.,
		 * it is a Text or Title field).
		 */
		new_line = first_line(as.cur_row+as.top_ent);
		if(new_line != -1 && new_line != as.cur_row+as.top_ent){
                    as.cur_row = new_line - as.top_ent;
                    if(as.cur_row < 0){
                        as.top_ent -= as.l_p_page;
                        as.cur_row += as.l_p_page;
                    }
		    else if(as.cur_row > as.l_p_page-1){
                        as.top_ent += as.l_p_page;
                        as.cur_row -= as.l_p_page;
		    }
		}
		start_disp = min(as.cur_row, as.old_cur_row);
	        as.old_cur_row = as.cur_row;
	        mangled_body++;
	    }
            break;


            /*-------- Add a single (Atom) entry to addrbook ------*/
          case PF10:
 	  case 'a':
            /*--------------- Create a new list -------------------*/
          case PF11:
          case 's':
            /*--------------- Add entry to existing list ----------*/
          case PF12:
          case 'z':
	    if(c == 's' &&
	        (are_selecting || entry_is_clickable(as.top_ent+as.cur_row)))
	        goto select;
	    /*------ Whereis (when selecting PF12 is Whereis)------*/
	    if(are_selecting && c == PF12){
		new_top_ent = ab_whereis();

		if(new_top_ent >= 0){
		    if(new_top_ent != as.top_ent){
			as.top_ent     = new_top_ent;
			start_disp     = 0;
			as.old_cur_row = as.cur_row;
			mangled_body++;
		    }
		    else
			current_changed_flag++;
		}
		mangled_footer++;
		break;
	    }
	    if(are_selecting)
		goto bleep;
	    if(entry_is_clickable(as.top_ent+as.cur_row)){
                q_status_message(1, 2, 4,
		    "\007AddressBook not expanded, use \"S\" to expand");
		break;
	    }
	    if(readonly){
                q_status_message(1, 2, 4, "\007AddressBook is ReadOnly");
		break;
	    }
	    if(noaccess){
                q_status_message(1, 2, 4, "\007Permission Denied");
		break;
	    }
	    new_ent = (c == 's' || c == PF11) ?
                           create_list(pab->address_book,
				       command_line)
                    : (c == 'z' || c == PF12) ?
                           addr_to_list(pab->address_book,
                                        command_line,
                                        dlist(as.top_ent+as.cur_row))
                    :
                           simple_add(pab->address_book, command_line);
            if(new_ent < 0){
                /*---- didn't succeed ----*/
		mangled_footer++;
		break;
	    }

	    /*----- Addition succeeded, now update the screen -----*/
            new_top_ent = as.l_p_page > 0 ? 
                        (new_ent/as.l_p_page) * as.l_p_page :
                        new_ent;
	    as.cur_row  = new_ent - new_top_ent;
	    if(new_top_ent != as.top_ent){
	        as.top_ent  = new_top_ent;
		start_disp = 0;
	    }
	    else
	        start_disp  = min(as.old_cur_row,as.cur_row);
            as.cur_fld      = 0;
	    as.old_cur_row  = as.cur_row;
	    mangled_body++;
	    mangled_footer++;
	    break;


            /*--------- QUIT pine -----------*/
          case OPF3:
	  case 'q':
            if(are_selecting)
                goto bleep;
            ps_global->next_screen = quit_screen;
	    quit = 1;
            break;


            /*--------- Compose -----------*/
          case OPF4:
	  case 'c':
            if(are_selecting)
                goto bleep;
	    ab_compose_to_addr();
	    mangled_screen++;
	    /* window size may have changed in composer */
	    ab_resize(0);
            break;


            /*--------- Folders -----------*/
          case OPF5:
	  case 'l':
            if(are_selecting)
                goto bleep;
            ps_global->next_screen = folder_screen;
	    quit = 1;
            break;


            /*---------- Open specific new folder ----------*/
          case OPF6:
	  case 'g':
            if(are_selecting || ps_global->nr_mode)
                goto bleep;
	    ab_goto_folder();
	    quit = 1;
            break;


            /*--------- Index -----------*/
          case OPF7:
	  case 'i':
            if(are_selecting)
                goto bleep;
            ps_global->next_screen = mail_index_screen;
	    quit = 1;
            break;


            /*----------- Where is (search) ----------------*/
	  case OPF8:
	  case 'w':
	  case ctrl('W'):
	    if(c == OPF8 && are_selecting)
	        goto bleep;

	    new_top_ent = ab_whereis();

	    if(new_top_ent >= 0){
		if(new_top_ent != as.top_ent){
		    as.top_ent     = new_top_ent;
		    start_disp     = 0;
		    as.old_cur_row = as.cur_row;
		    mangled_body++;
		}
		else
		    current_changed_flag++;
	    }
	    mangled_footer++;
	    break;


	    /*----------------- Print --------------------*/
	  case OPF9: 
	  case 'y':
	    ab_print();
            break;


            /*---------- Edit fcc and annotate -----------*/
	  case OPF10: 
	  case 'f':
	    goto edit;
	    /*NOTREACHED*/
	    break;


            /*------------ Suspend pine  (^Z) -------------*/
          case ctrl('Z'):
            if(!have_job_control())
                goto bleep;
            if(F_OFF(F_CAN_SUSPEND,ps_global)){
                q_status_message(1, 2, 4, 
                          "\007Pine suspension not enabled - see help text");
                break;
            }
	    else
                do_suspend(ps_global);
	    ab_resize(0);
            mangled_screen++;
	    break;


          case KEY_RESIZE:
          case ctrl('L'):
	    ab_resize(c == KEY_RESIZE);
            mangled_screen++;
	    break;


          case ctrl('C'):
	    if(!are_selecting)
	        goto bleep;
	    quit = 1;
	    break;


	  default:
          bleep:
	    bogus_command(orig_c, F_ON(F_USE_FK,ps_global) ? "F1" : "?");
	    break;
	}
    }while(!quit);
    
    return NULL;
}


/*
 * Validate selected address with build_address, save addrbook state,
 * call composer, restore addrbook state.
 */
void
ab_compose_to_addr()
{
    AddrScrState  *savep;   /* For saving state of addrbooks temporarily */
    OpenStatus    *stp;     /* For saving state of addrbooks temporarily */
    int		   good_addr;
    char          *addr,
		  *error,
		  *fcc;
    AddrScrn_Disp *dl;

    dprint(9, (debugfile, "- ab_compose_to_addr -\n"));

    save_state(&savep, &stp);

    if(is_addr(as.top_ent+as.cur_row)){

	dl = dlist(as.top_ent+as.cur_row);

	error = "";

	if(dl->type == ListEnt)
	    good_addr = (our_build_address(
		      dl->other,
		      &addr,
		      &error,
		      &fcc,
		      0) >= 0);
	else
	    good_addr = (our_build_address(
		      dl->a.abe->nickname,
		      &addr,
		      &error,
		      &fcc,
		      0) >= 0);
	if(*error){
	    q_status_message1(0, 2, 4, "%s", error);
	    display_message(NO_OP_COMMAND);
	}
	if(!good_addr)
	    fs_give((void **)&addr); /* relying on fs_give setting
					addr to NULL */
    }
    else
	addr = NULL ;

    compose_mail(addr, fcc);

    restore_state(savep, stp);

    if(addr)
	fs_give((void **)&addr);
    if(fcc)
	fs_give((void **)&fcc);
}


/*
 * Go to folder.
 */
void
ab_goto_folder()
{
    char *go_folder;
    CONTEXT_S *tc;

    tc = (ps_global->context_last
	      && !(ps_global->context_current->type & FTYPE_BBOARD)) 
	       ? ps_global->context_last : ps_global->context_current;

    go_folder = broach_folder(-3, 1, &tc);
#ifdef DOS
    if(go_folder && *go_folder == '{' && coreleft() < 20000){
	q_status_message(0, 0, 2, "\007Not enough memory to open IMAP folder");
	go_folder = NULL;
    }
#endif /* !DOS */
    if(go_folder != NULL){
	dprint(9, (debugfile, "do_broach_folder (%s, %s)\n",
		 go_folder, tc->context));
	if(do_broach_folder(go_folder, tc) > 0)
	    /* so we go out and come back into the index */
	    ps_global->next_screen = mail_index_screen;
    }
}


/*
 * Execute whereis command.
 *
 * Returns value of the new top entry, or -1 if cancelled.
 */
int
ab_whereis()
{
    int rc, new_line, new_fld, new_top_ent;

    rc = search_book(as.top_ent+as.cur_row, as.cur_fld, -3,
		    &new_line, &new_fld);

    new_top_ent = -1;

    if(rc == -2)
	q_status_message(0, 0, 2,"\007Addressbook search cancelled");

    if(rc == -1)
	q_status_message(0, 0, 2,"Word not found");

    if(rc == 0){  /* search succeeded */
	if((new_line < as.cur_row+as.top_ent) ||
	      (new_line == as.cur_row+as.top_ent && new_fld < as.cur_fld))
	    q_status_message(0, 0, 2, "Search wrapped to beginning");

	/*-- place cursor on new address ----*/
	new_top_ent = as.l_p_page > 0 ? 
			 (new_line / as.l_p_page) * as.l_p_page :
			 new_line;
	as.cur_row  = new_line - new_top_ent;
	as.cur_fld  = new_fld;
    }

    return(new_top_ent);
}


/*
 * Print out the display list.
 */
void
ab_print()
{
    AddrScrn_Disp *dl; 
    int lineno = 0;

    if(open_printer("address books ") == 0){

	for(dl = dlist(lineno); dl->type != End; dl = dlist(++lineno)){
	    switch(dl->type){
	      case Single:
		print_text3("%-10.10s %-35.35s %s\n",
			    dl->a.abe->nickname, dl->a.abe->fullname,
			    dl->a.abe->addr.addr);
		break;

	      case ListHead:
		print_text3("%-10.10s %-35.35s %s\n",
		    dl->a.abe->nickname, dl->a.abe->fullname, DISTLIST);
		break;

	      case ListEnt:
print_text1("                                               %s\n", dl->other);
		break;

	      case ClickHere:
		print_text1("%s\n", CLICKHERE);
		break;

	      case Empty:
		print_text1("%s\n", EMPTY);
		break;

	      case Text:
	      case Title:
		print_text1("%s\n", dl->a.txt);
		break;
	    }
	}
	close_printer();
    }
}


/*
 * recalculate display parameters for window size change
 *
 * Args: clear_i_cache -- call clear_index_cache() if set
 */
void
ab_resize(clear_i_cache)
int clear_i_cache;
{
    int new_line;

    as.l_p_page = ps_global->ttyo->screen_rows - FOOTER_LINES - HEADER_LINES;

    if(as.l_p_page <= 0)
        return;

    new_line   = as.top_ent + as.cur_row;
    as.top_ent = as.l_p_page > 0 ? 
		     (new_line / as.l_p_page) * as.l_p_page :
		     new_line;
    as.cur_row     = new_line - as.top_ent;
    as.old_cur_row = as.cur_row;

    /* need this to re-initialize Text and Title lines in display */
    (void)init_addrbooks(Same, 0, 0, 0);
    if(clear_i_cache)
	clear_index_cache();
}


/*
 * Returns 0 if we know for sure that no line of the display list greater than
 * or equal to lineno contains an address.
 */
int
any_addrs_avail(lineno)
register int lineno;
{
    register AddrScrn_Disp *dl;

    for(dl=dlist(lineno); dl->type != End; dl = dlist(++lineno)){
	switch(dl->type){
	  case Single:
	  case ListEnt:
	  case ListHead:
	  case ClickHere:
	    return 1;
	}
    }

    return 0;
}


/*
 * Returns 1 if this line is a clickable line.
 */
int
entry_is_clickable(lineno)
register int lineno;
{
    register AddrScrn_Disp *dl;

    if((dl = dlist(lineno)) && dl->type == ClickHere)
	return 1;

    return 0;
}


/*
 * Returns 1 if an address or list is selected.
 */
int
is_addr(lineno)
int lineno;
{
    register AddrScrn_Disp *dl;

    if((dl = dlist(lineno)) && (dl->type == ListHead ||
				dl->type == ListEnt  ||
				dl->type == Single))
	return 1;

    return 0;
}


/*
 * Returns 1 if type of line is Empty.
 */
int
is_empty(lineno)
int lineno;
{
    register AddrScrn_Disp *dl;

    if((dl = dlist(lineno)) && dl->type == Empty)
	return 1;

    return 0;
}


/*
 * Returns 1 if this line is of a type that can have a cursor on it.
 */
int
line_is_active(lineno)
int lineno;
{
    register AddrScrn_Disp *dl;

    if((dl = dlist(lineno)) && (dl->type == Text  ||
				dl->type == Title ||
				dl->type == End))
	return 0;

    return 1;
}


/*
 * Find the first "active" line greater than or equal to line.  That is, the
 * first line the cursor is allowed to start on.  (Cursor never points to
 * Text or Title lines.)
 *
 * Returns the line number of the first line or -1 if there is none.
 */
int
first_line(line)
int line;
{
    register int lineno;

    for(lineno=line;
       dlist(lineno)->type != End && !line_is_active(lineno);
       lineno++)
	;/* do nothing */

    if(line_is_active(lineno))
	return(lineno);

    /*
     * There were no active lines from lineno on down.  Trying looking
     * back up the list.
     */
    for(lineno=line-1;
       lineno >= 0 && !line_is_active(lineno);
       lineno--)
	;/* do nothing */

    if(lineno >= 0 && line_is_active(lineno))
	return(lineno);

    /* no active lines at all */
    return -1;
}


/*
 * Find the line and field number of the next field
 *
 * Args: cur_line     -- The current line position (in global display list)
 *			 of cursor
 *       cur_fld      -- The current field position of cursor
 *       new_line     -- Return value: new line position
 *       new_fld      -- Return value: new fld position
 *
 * Result: The new line and field number are set.
 *       The value 1 is returned if OK or 0 if there is no next field.
 */
int
next_field(cur_line, cur_fld, new_line, new_fld)
int  cur_line,
     cur_fld,
    *new_line,
    *new_fld;
{
    register AddrScrn_Disp *dl;

    dl = dlist(cur_line);

    switch(dl->type){

      case Single:
        cur_fld = (cur_fld + 1) % 3;
        if(cur_fld == 0)
            cur_line++;
        break;

      case ListHead:
        if(cur_fld == 0)
            cur_fld++;
	else{
            cur_line++;
            cur_fld = 0;
        }
        break;

      case ListEnt:
      case ClickHere:
      case Empty:
        cur_line++;
	cur_fld = 0;
        break;
    }

    /* go past non-active lines */
    while((dl=dlist(cur_line))->type != End && !line_is_active(cur_line))
	cur_line++;

    if(dl->type != End){
        *new_line = cur_line;
        *new_fld  = cur_fld;
        return 1;
    }
    else
	return 0;
}


/*
 * Find the line and field number of the previous field
 *
 * Args: cur_line     -- The current line position (in global display list)
 *			 of cursor
 *       cur_fld      -- The current field position of cursor
 *       new_line     -- Return value: new line position
 *       new_fld      -- Return value: new fld position
 *
 * Result: The new line and field number are set.
 *       The value 1 is returned if OK or 0 if there is no previous field.
 */
int
prev_field(cur_line, cur_fld, new_line, new_fld)
int  cur_line,
     cur_fld,
    *new_line,
    *new_fld;
{
    register AddrScrn_Disp *dl;

    if(cur_fld == 0 && cur_line == 0)
	return 0;

    dl = dlist(cur_line);

    switch(dl->type){

      case Single:
      case ListHead:
	if(cur_fld > 0)
	    cur_fld--;
	else{
	    cur_fld = 2;
            cur_line--;
	}
        break;

      case ListEnt:
      case ClickHere:
      case Empty:
        cur_line--;
	cur_fld = 2;
        break;
    }

    /* back over non-active lines */
    while(cur_line >= 0 &&
         ((dl=dlist(cur_line))->type == Text || dl->type == Title))
	cur_line--;

    /*
     * If this is true, we backed over all the non-active lines but
     * never managed to find an active one, so we were already at the
     * first entry.
     */
    if(cur_line < 0)
	return 0;
    else{
        if(dl->type == ListHead)
	    cur_fld = min(1, cur_fld);
        *new_line = cur_line;
        *new_fld  = cur_fld;
        return 1;
    }
}


/*
 * Find the line and field number of the next line, keeping the field
 * about the same.
 *
 * Args: cur_line     -- The current line position (in global display list)
 *			 of cursor
 *       cur_fld      -- The current field position of cursor
 *       new_line     -- Return value: new line position
 *       new_fld      -- Return value: new fld position
 *
 * Result: The new line and field number are set.
 *       The value 1 is returned if OK or 0 if there is no next line.
 */
int
next_line(cur_line, cur_fld, new_line, new_fld)
int  cur_line,
     cur_fld,
    *new_line,
    *new_fld;
{
    register AddrScrn_Disp *dl;

    /* skip over non-active lines */
    for(cur_line++;
        (dl=dlist(cur_line))->type == Text || dl->type == Title;
	cur_line++)
	;/* do nothing */

    switch(dl->type){

      case Single:
      case ListEnt:
      case ClickHere:
      case Empty:
        break;

      case ListHead:
	cur_fld = min(1, cur_fld); /* field 2 is not a data item for ListHead */
        break;

      case End:
        return 0;
    }

    *new_line = cur_line;
    *new_fld  = cur_fld;
    return 1;
}


/*
 * Find the line and field number of the previous line, keeping the field
 * about the same.
 *
 * Args: cur_line     -- The current line position (in global display list)
 *			 of cursor
 *       cur_fld      -- The current field position of cursor
 *       new_line     -- Return value: new line position
 *       new_fld      -- Return value: new fld position
 *
 * Result: The new line and field number are set.
 *       The value 1 is returned if OK or 0 if there is no previous line.
 */
int
prev_line(cur_line, cur_fld, new_line, new_fld)
int  cur_line,
     cur_fld,
    *new_line,
    *new_fld;
{
    register AddrScrn_Disp *dl;

    /* skip backwards over non-active lines */
    for(cur_line--;
	cur_line >= 0 &&
              ((dl=dlist(cur_line))->type == Text || dl->type == Title);
	cur_line--)
	;/* do nothing */

    if(cur_line < 0)
        return 0;

    if(dl->type == ListHead)
	cur_fld = min(1, cur_fld);

    *new_line = cur_line;
    *new_fld  = cur_fld;

    return 1;
}


/*
 * Delete an entry from the address book
 *
 * Args: abook        -- The addrbook handle into access library
 *       command_line -- The screen line on which to prompt
 *       cur_line     -- The entry number in the display list
 *
 * Result: returns 1 if an entry was deleted, 0 if not.
 *
 * The main routine above knows what to repaint because it's always the
 * current entry that's deleted.  Here confirmation is asked of the user
 * and the appropriate adrbklib functions are called.
 */
int
addr_book_delete(abook, command_line, cur_line)
AdrBk *abook;
int    command_line,
       cur_line;
{
    char   ch, *cmd, *dname;
    char   prompt[40+MAX_ADDRESS+1]; /* 40 is len of string constants below */
    int    rc = command_line; /* nuke warning about command_line unused */
    register AddrScrn_Disp *dl;
    PerAddrBook     *pab;

    dprint(8, (debugfile, "- addr_book_delete -\n"));

    dl = dlist(cur_line);

    dname = ""; /* So compilers won't complain this used before it's set */
    cmd   = ""; /* So compilers won't complain this used before it's set */

    switch(dl->type){
      case Single:
        dname = dl->a.abe->fullname;
        cmd   = "Really delete \"%.50s\"";
        break;

      case ListHead:
        dname = dl->a.abe->fullname;
	cmd   = "Really delete ENTIRE list \"%.50s\"";
        break;

      case ListEnt:
        dname = dl->other;
	cmd   = "Really delete \"%.100s\" from list";
        break;
    } 

    sprintf(prompt, cmd, dname);
    ch = want_to(prompt, 'n', 'n', NO_HELP, 0, 0);
    if(ch == 'y'){
	if(dl->type == Single || dl->type == ListHead){
	    /*--- Kill a single entry or an entire list ---*/
            rc = adrbk_delete(abook, dl->a.abe);
	    dl->a.abe = NULL;
	}
	else{
            /*---- Kill an entry out of a list ----*/
            rc = adrbk_listdel(abook, dl->a.abe, dl->other);
	    as.cur_fld = 2;
	}

	pab = &as.adrbks[as.cur];

	if(rc == 0){
	    q_status_message(0, 1, 3, "Entry deleted, address book updated");
            dprint(2, (debugfile, "\"%s\" deleted from addr book\n", dname));
	    (void)init_disp_list(pab, Same, (AdrBk_Entry *)NULL, NULL, 0);
            return 1;
        }
	else{
            q_status_message1(0, 2, 5, "\007Error writing address book: %s",
                                                   error_description(errno));
            dprint(1, (debugfile, "Error deleting \"%s\" from %s (%s): %s\n",
		dname, pab->nickname, pab->filename, error_description(errno)));
        }
	return 0;
    }
    else{
	q_status_message(0, 0, 2, "\007Entry not deleted");
	return 0;
    }
}


/*
 *   Prompt for name, address, etc for a simple addition
 *
 * Args: abook          -- The address book handle for the addrbook access lib
 *       command_line   -- The screen line to prompt on
 *
 * Result: address book possibly updated.  If address selection screen
 *       is called up returns fact that screen needs repainting.  Returns
 *       less than zero on failure or cancellation and line number of entry
 *       that was just added on success.
 *
 * This is only for adding a plain address book entry, and does nothing with
 * lists.
 */
int
simple_add(abook, command_line)
AdrBk *abook;
int    command_line;
{
    char         new_fullname[MAX_FULLNAME + 1],
		 new_address[MAX_ADDRESS + 1],
                 new_nickname[MAX_NICKNAME + 1];
    char        *prompt,
		*fname;
    int          where, rc, which_addrbook;
    AdrBk_Entry *new;


    dprint(8, (debugfile, "- simple_add -\n"));

    new_address[0]   = '\0';
    new_fullname[0]  = '\0';
    new_nickname[0]  = '\0';

    /*------ full name ------*/
    prompt = "New full name (Last, First): ";
    rc     = edit_fullname(command_line, new_fullname, prompt, h_oe_add_full);
    if(rc != 0)
	goto add_cancel;
            

    /*----- nickname ------*/
    prompt = "Enter new nickname (one word and easy to remember): ";
    rc     = edit_nickname(abook, (AddrScrn_Disp *)NULL, command_line,
	new_nickname, prompt, h_oe_add_nick, 0, 0);
    if(rc != 0)
	goto add_cancel;


    /*---- address ------*/
    prompt = "Enter new e-mail address: ";
    rc     = edit_address(command_line, 0, new_address, prompt, h_oe_add_addr);
    if(rc != 0)
	goto add_cancel;

            
    if(fname=addr_lookup(new_nickname, &which_addrbook)){
	q_status_message3(0, 7, 7, "Warning! %s exists in %s as %s",
	    new_nickname, as.adrbks[which_addrbook].nickname, fname);
    }

    /*---- write it into the file ----*/
    rc = adrbk_add(as.adrbks[as.cur].address_book, &new, new_nickname,
		   new_fullname, new_address, NULL, NULL, Atom);

    if(rc == -2 || rc == -3){
        q_status_message1(1, 2, 4, "\007Error updating address book: %s",
              rc == -2 ?  error_description(errno) : "Pine bug");
        dprint(1, (debugfile, "Error adding \"%s\": %s\n", new_nickname,
              rc == -2 ?  error_description(errno) : "Pine bug"));
        return -2;
    }
    else if(rc == -4){
        q_status_message(0, 2, 3, "\007Tabs not allowed in address book");
        return -4;
    }

    where = init_disp_list(&as.adrbks[as.cur], Same, new, NULL, 0);

    q_status_message(0, 0, 2, "Addition complete. Address book updated.");
    dprint(2, (debugfile, "\"%s\" added to address book\n", new_nickname));
    return(where);
 
 add_cancel:
    q_status_message(0, 0, 1, "\007Address book addition cancelled");
    return -1;
}


/*
 *   Create a distribution list
 *
 * Args: abook          -- Handle into address book access library
 *       command_line   -- screen line to prompt on
 *
 * Result: Distribution list possibly created, 
 *         returns flag if screen was painted by the address grabber
 *         return value:  >=0: new entry added at line n
 *                         -1: creation cancelled
 *                         -2: error writing address book
 *                         -4: Tabs in fields (tabs are not allowed).
 *                                (should never happen)
 *
 * Prompt for the description, then nickname, then up to MAX_ADDRESS
 * list members, which can be addresses or nicknames of other addresses or
 * lists.
 */
int
create_list(abook, command_line)
AdrBk *abook;
int    command_line;
{
    char         list_name[MAX_FULLNAME+1],
		 new_nickname[MAX_NICKNAME+1],
                 new_address[MAX_ADDRESS+1],
		 prompt2[80],
                *temp_list[MAX_NEW_LIST+1],
               **p,
		*q,
	        *prompt,
		*fname;
    AdrBk_Entry *new;
    int          rc, where, which_addrbook;
    PerAddrBook *pab;


    dprint(8, (debugfile, "- create_list -\n"));

    list_name[0]     = '\0';
    new_nickname[0]  = '\0';
    new_address[0]   = '\0';

    /*------ name for list ------*/
    prompt = "Long name/description of new list: ";
    rc     = edit_fullname(command_line, list_name, prompt, h_oe_crlst_full);
    if(rc != 0)
	goto create_cancel;


    /*----- nickname ------*/
    prompt = "Enter list nickname (one word and easy to remember): ";
    rc     = edit_nickname(abook, (AddrScrn_Disp *)NULL, command_line,
	new_nickname, prompt, h_oe_crlst_nick, 0, 0);
    if(rc != 0)
	goto create_cancel;


    /*---- addresses ------*/
    for(p = temp_list; p < &temp_list[MAX_NEW_LIST]; p++){

        sprintf(prompt2, "Enter %s address or blank when done: ",
						enth_string(p - temp_list + 1));
        new_address[0] = '\0';
	rc = edit_address(command_line, 1, new_address, prompt2,
							       h_oe_crlst_addr);
	for(q = new_address; isspace(*q); ++q)
	    ;/* do nothing */

	if(*q == '\0') 
	    goto done_with_addrs;

	if(rc != 0)
	    goto create_cancel;

        *p = cpystr(new_address);
    }

done_with_addrs:

    if(fname=addr_lookup(new_nickname, &which_addrbook)){
	q_status_message3(0, 7, 7, "Warning! %s exists in %s as %s",
	    new_nickname, as.adrbks[which_addrbook].nickname, fname);
    }

    *p = NULL;
    pab = &as.adrbks[as.cur];

    rc = adrbk_add(pab->address_book, &new, new_nickname,
				      list_name, NULL, NULL, NULL, List);

    if(rc == 0){
        for(p = temp_list; *p != NULL; p++) 
            if((rc=adrbk_listadd(pab->address_book, new, *p, (char **)NULL)))
                break;
    }
    else if(rc == -2 || rc == -3){
        q_status_message1(1, 2, 4,"\007Error updating address book: %s",
                   rc == -2 ?  error_description(errno) : "Pine bug");
        dprint(1, (debugfile, "Error creating list \"%s\" in addrbook: %s\n",
               new_nickname, rc == -2 ? error_description(errno) :
               "pine bug"));
        return -2;
    }
    else if(rc == -4){
        q_status_message(0, 2,2, "\007Tabs not allowed in address book");
        return -4;
    }

    where = init_disp_list(pab, Same, new, NULL, 0);

    q_status_message1(0, 1, 2,
        "Addition of list %s complete. Address book updated.", new_nickname);
    dprint(2, (debugfile, "List addition to address book \"%s\"\n",
                                                          new_nickname));
    return(where);
 
create_cancel:
    q_status_message(0, 0, 1, "\007Address book list creation cancelled");
    return -1;
}


/*
 * Add an entry to a distribution list
 *
 * Args: abook          -- Handle to address book for adrbklib access library
 *       command-line   -- Screen line number to prompt on 
 *       dl             -- Display list line for current line
 *
 * Result: flag set if screen was mangled, possible addition to list
 *         Returns: >= 0: line in disp list of new entry
 *                    -1: addition cancelled
 *                    -2: error writing address book
 */
int
addr_to_list(abook, command_line, dl)
AdrBk         *abook;
int            command_line;
AddrScrn_Disp *dl;
{
    char     new_address[MAX_ADDRESS+1],
             prompt[80],
            *new_addr,
	    *edited_nick;
    int      rc, where;

    dprint(8, (debugfile, "- addr_to_list -\n"));

    if(abook->book_used == 0){
        q_status_message(0, 1, 3,
			"\007No distribution lists. To create, use \"S\"");
        return -1;
    }

    if(dl->type != ListHead && dl->type != ListEnt){
        q_status_message(1, 1, 3,
"\007Move cursor to list you wish to add to. Use \"A\" to create plain entry");
        return -1;
    }

    new_address[0] = '\0';
    edited_nick    = dl->a.abe->nickname;

    sprintf(prompt, "Address to add to \"%s\" list: ", edited_nick);
    rc = edit_address(command_line, 1, new_address, prompt, h_oe_adlst_addr);
    if(rc != 0)
	goto alist_cancel;


    rc = adrbk_listadd(abook, dl->a.abe, new_address, &new_addr);

    if(rc == -2){
        q_status_message1(1, 1, 3, "\007Error updating address book: %s",
                                                    error_description(errno));
        dprint(1, (debugfile, "Error adding to list \"%s\": %s\n",
                       edited_nick, error_description(errno)));
        return -2;
    }

    where = init_disp_list(&as.adrbks[as.cur], Same, dl->a.abe, new_addr, 0);

    q_status_message1(0, 0, 2,
        "Addition to \"%s\" list complete. Address book updated.", edited_nick);
    dprint(2, (debugfile, "Addition to list \"%s\"\n", edited_nick));
    return(where);

alist_cancel:
    q_status_message(0, 0, 1, "\007Address book addition cancelled");
    return -1;
}


/*
 * Edit a nickname field.
 *
 * Args: abook     -- the addressbook handle
 *       dl        -- display list line (NULL if new entry)
 *    command_line -- line to prompt on
 *       orig      -- nickname to edit
 *       prompt    -- prompt
 *      this_help  -- help
 * return_existing -- changes the behavior when a user types in a nickname
 *                    which already exists in this abook.  If not set, it
 *                    will just keep looping until the user changes; if set,
 *                    it will return -8 to the caller and orig will be set
 *                    to the matching nickname.
 *    allow_ctrl_t -- allow the user to escape to the address books to
 *                    select a nickname
 *
 * Returns: -10 to cancel
 *          -9  no change
 *          -8  existing nickname chosen (only happens if return_existing set)
 *           0  new value copied into orig
 */
int
edit_nickname(abook, dl, command_line, orig, prompt, this_help,
						return_existing, allow_ctrl_t)
AdrBk         *abook;
AddrScrn_Disp *dl;
int            command_line;
char          *orig,
              *prompt;
HelpType       this_help;
int            return_existing,
               allow_ctrl_t;
{
    char         edit_buf[MAX_NICKNAME + 1];
    HelpType     help;
    int          rc;
    AdrBk_Entry *check;
    ESCKEY_S     ekey[2];

    if(allow_ctrl_t){
	ekey[0].ch    = ctrl('T');
	ekey[0].rval  = 2;
	ekey[0].name  = "^T";
	ekey[0].label = "To AddrBk";

	ekey[1].ch    = -1;
    }
    else
	ekey[0].ch    = -1;

    edit_buf[MAX_NICKNAME] = '\0';
    strncpy(edit_buf, orig, MAX_NICKNAME);
    help  = NO_HELP;
    rc    = 0;
    check = NULL;
    do{
	/* display a message because adrbk_lookup returned positive */
	if(check){
	    if(return_existing){
		strcpy(orig, edit_buf);
		return -8;
	    }

            q_status_message1(0, 1, 1,
		    "\007Already an entry with nickname \"%s\"", edit_buf);
            display_message(NO_OP_COMMAND);
            sleep(2);
	}

	if(rc == 3)
            help = (help == NO_HELP ? this_help : NO_HELP);

	rc = optionally_enter(edit_buf, command_line, 0, MAX_NICKNAME, 1,
                                  0, prompt, ekey, help, 0);

	if(rc == 1)  /* ^C */
	    break;

	if(rc == 2){ /* ^T */
	    void (*redraw) () = ps_global->redrawer;
	    char *returned_nickname;

	    push_titlebar_state();
	    returned_nickname = addr_book_takeaddr();
	    if(returned_nickname)
		strncpy(edit_buf, returned_nickname, MAX_NICKNAME);
	    ClearScreen();
	    pop_titlebar_state();
	    redraw_titlebar();
	    if(ps_global->redrawer = redraw) /* reset old value, and test */
	        (*ps_global->redrawer)();
	}
            
    }while(rc == 2 ||
	   rc == 3 ||
	   rc == 4 ||
	   nickname_check(edit_buf) ||
           ((check = adrbk_lookup(abook, edit_buf)) &&
	    (!dl || (check != dl->a.abe))));

    if(rc != 0)
	return -10;

    if(strcmp(edit_buf, orig) == 0) /* no change */
	return -9;
    
    strcpy(orig, edit_buf);
    return 0;
}


/*
 * Let user change a nickname field.
 *
 * Args: abook     -- the addressbook handle
 *       dl        -- display list line
 *       new       -- points to new entry on return
 *    command_line -- line to prompt on
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
change_nickname(abook, dl, new, command_line)
AdrBk         *abook;
AddrScrn_Disp *dl;
AdrBk_Entry  **new;
int            command_line;
{
    char     edit_buf[MAX_NICKNAME + 1];
    char    *prompt, **p, *fname;
    int      rc, which_addrbook;

    dprint(8, (debugfile, "- change_nickname -\n"));

    if(dl->a.abe->nickname){
	edit_buf[MAX_NICKNAME] = '\0';
	strncpy(edit_buf, dl->a.abe->nickname, MAX_NICKNAME);
    }
    else
	edit_buf[0] = '\0';
    prompt = "Edit Nickname: ";

    rc     = edit_nickname(abook, dl, command_line, edit_buf, prompt,
							h_oe_editab_nick, 0, 0);

    if(rc == 0){

	if(fname=addr_lookup(edit_buf, &which_addrbook)){
	    q_status_message3(0, 7, 7, "Warning! %s exists in %s as %s",
		edit_buf, as.adrbks[which_addrbook].nickname, fname);
	}

	rc = adrbk_add(abook, new, edit_buf, dl->a.abe->fullname,
		       dl->a.abe->tag == List ?
			   (char *)dl->a.abe->addr.list:
			   dl->a.abe->addr.addr,
		       dl->a.abe->fcc, dl->a.abe->extra, dl->a.abe->tag);

	if(dl->a.abe->tag == List && dl->a.abe->addr.list)
	    /* Add list back in if it's a list */
	    for(p = dl->a.abe->addr.list; rc == 0 && *p; p++)
		rc = adrbk_listadd(abook, *new, *p, (char **)NULL);

	if(rc == 0 && strucmp(dl->a.abe->nickname, edit_buf) != 0){
	    /*
	     * if only change was case in nickname, then above was an edit
	     * and no delete is required
	     */
	    rc = adrbk_delete(abook, dl->a.abe);
	    dl->a.abe = NULL;
	    (void)init_disp_list(&as.adrbks[as.cur], Same, (AdrBk_Entry *)NULL,
		NULL, 0);
	} 
    }

    return(rc);
}


/*
 * Edit a fullname field.
 *
 * Args: command_line -- line to prompt on
 *          orig      -- fullname to edit
 *          prompt    -- prompt
 *         this_help  -- help
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
edit_fullname(command_line, orig, prompt, this_help)
int      command_line;
char    *orig,
        *prompt;
HelpType this_help;
{
    char     edit_buf[MAX_FULLNAME + 1];
    HelpType help;
    int      rc;

    edit_buf[MAX_FULLNAME] = '\0';
    strncpy(edit_buf, orig, MAX_FULLNAME);
    help = NO_HELP;
    rc   = 0;
    do{
	if(rc == 3)
	    help = (help == NO_HELP ? this_help : NO_HELP);

	rc = optionally_enter(edit_buf, command_line, 0, MAX_FULLNAME, 1,
                                  0, prompt, (ESCKEY_S *)NULL, help, 0);

    }while(rc == 3 || rc == 4);

    if(rc != 0)
	return -10;

    if(strcmp(edit_buf, orig) == 0) /* no change */
	return -9;
    
    strcpy(orig, edit_buf);
    return 0;
}


/*
 * Let user change a fullname field.  Can be a list or a simple address.
 *
 * Args: abook     -- the addressbook handle
 *       dl        -- display list line
 *       new       -- points to new entry on return
 *    command_line -- line to prompt on
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
change_fullname(abook, dl, new, command_line)
AdrBk         *abook;
AddrScrn_Disp *dl;
AdrBk_Entry  **new;
int            command_line;
{
    char  edit_buf[MAX_FULLNAME + 1];
    char *prompt;
    int   rc;

    dprint(8, (debugfile, "- change_fullname -\n"));

    if(dl->a.abe->fullname){
	edit_buf[MAX_FULLNAME] = '\0';
	strncpy(edit_buf, dl->a.abe->fullname, MAX_FULLNAME);
    }
    else
	edit_buf[0] = '\0';
    if(dl->type == ListHead || dl->type == ListEnt)
	prompt = "Edit Full name of list: ";
    else
	prompt = "Edit Full name: ";

    rc = edit_fullname(command_line, edit_buf, prompt, h_oe_editab_full);

    if(rc == 0)
	rc = adrbk_add(abook, new, dl->a.abe->nickname, edit_buf,
		   dl->a.abe->tag == List ?
		       (char *)dl->a.abe->addr.list:
		       dl->a.abe->addr.addr,
		   dl->a.abe->fcc, dl->a.abe->extra, dl->a.abe->tag);

    return(rc);
}


/*
 * Edit an address field.
 *
 * Args: command_line -- line to prompt on
 *          spaces_ok -- spaces are allowed in address (because it is a full
 *                       address in a list, not just the actual address part of
 *                       a simple addrbook entry)
 *          orig      -- fullname to edit
 *          prompt    -- prompt
 *         this_help  -- help
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
edit_address(command_line, spaces_ok, orig, prompt, this_help)
int      command_line,
	 spaces_ok;
char    *orig,
        *prompt;
HelpType this_help;
{
    char     edit_buf[MAX_ADDRESS + 1];
    HelpType help;
    int      rc;

    edit_buf[MAX_ADDRESS] = '\0';
    strncpy(edit_buf, orig, MAX_ADDRESS);
    help = NO_HELP;
    rc   = 0;
    do{
	if(rc == 3)
	    help = (help == NO_HELP ? this_help : NO_HELP);

	rc = optionally_enter(edit_buf, command_line, 0, MAX_ADDRESS, 1,
                                  0, prompt, (ESCKEY_S *)NULL, help, 0);
            
	if(rc == 1) /* ^C */
	    break;
            
    }while(rc == 3 ||
	   rc == 4 ||
	   warn_bad_addr(edit_buf, spaces_ok));

    if(rc != 0)
	return -10;

    if(strcmp(edit_buf, orig) == 0) /* no change */
	return -9;

    strcpy(orig, edit_buf);
    return 0;
}


/*
 * Change an address field, either a single address or a list member.
 *
 * Args: abook     -- the addressbook handle
 *       dl        -- display list line
 *       new       -- points to new entry on return
 *     addr_match  -- points to new entry within list on return
 *    command_line -- line to prompt on
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
change_address(abook, dl, new, addr_match, command_line)
AdrBk         *abook;
AddrScrn_Disp *dl;
AdrBk_Entry  **new;
char         **addr_match;
int            command_line;
{
    char     edit_buf[MAX_ADDRESS + 1];
    char    *prompt;
    HelpType this_help;
    int      spaces_ok;
    int      rc;

    dprint(8, (debugfile, "- change_address -\n"));

    edit_buf[MAX_ADDRESS] = '\0';
    if(dl->type == ListEnt){
	if(dl->other)
	    strncpy(edit_buf, dl->other, MAX_ADDRESS);
	else
	    edit_buf[0] = '\0';
	prompt    = "Edit address in list: ";
	this_help = h_oe_editab_al;
	spaces_ok = 1;
    }
    else{
	if(dl->a.abe->addr.addr)
	    strncpy(edit_buf, dl->a.abe->addr.addr, MAX_ADDRESS);
	else
	    edit_buf[0] = '\0';
	prompt    = "Edit address: ";
	this_help = h_oe_editab_addr;
	spaces_ok = 0;
    }

    rc = edit_address(command_line, spaces_ok, edit_buf, prompt, this_help);

    if(rc == 0){
	if(dl->type == ListEnt){
	    rc = adrbk_listdel(abook, dl->a.abe, dl->other);
	    if(rc == 0)
		rc = adrbk_listadd(abook, dl->a.abe, edit_buf, addr_match);
	    *new = dl->a.abe;
	}
	else{
	    rc = adrbk_add(abook, new, dl->a.abe->nickname, dl->a.abe->fullname,
			     edit_buf, dl->a.abe->fcc, dl->a.abe->extra,
			     dl->a.abe->tag);
	}
    }
    return(rc);
}


/*
 * Edit the addressbook comment field of an address or list.
 *
 * Args: command_line -- line to prompt on
 *          orig      -- fullname to edit
 *          prompt    -- prompt
 *         this_help  -- help
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
edit_comment(command_line, orig, prompt, this_help)
int      command_line;
char    *orig,
        *prompt;
HelpType this_help;
{
    char     edit_buf[MAX_COMMENT + 1];
    HelpType help;
    int      rc;

    edit_buf[MAX_COMMENT] = '\0';
    strncpy(edit_buf, orig, MAX_COMMENT);
    help = NO_HELP;
    rc   = 0;
    do{
	if(rc == 3)
	    help = (help == NO_HELP ? this_help : NO_HELP);

	rc = optionally_enter(edit_buf, command_line, 0, MAX_COMMENT, 1,
                                  0, prompt, (ESCKEY_S *)NULL, help, 0);

    }while(rc == 3 || rc == 4);

    if(rc != 0)
	return -10;

    if(strcmp(edit_buf, orig) == 0) /* no change */
	return -9;

    strcpy(orig, edit_buf);
    return 0;
}


/*
 * Edit the addressbook comment field of an address or list.
 *
 * Args: abook     -- the addressbook handle
 *       dl        -- display list line
 *       new       -- points to new entry on return
 *    command_line -- line to prompt on
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
change_comment(abook, dl, new, command_line)
AdrBk         *abook;
AddrScrn_Disp *dl;
AdrBk_Entry  **new;
int            command_line;
{
    char     edit_buf[MAX_COMMENT + 1];
    char    *prompt;
    int      rc;

    dprint(8, (debugfile, "- change_comment -\n"));

    if(dl->a.abe->extra){
	edit_buf[MAX_COMMENT] = '\0';
	strncpy(edit_buf, dl->a.abe->extra, MAX_COMMENT);
    }
    else
	edit_buf[0] = '\0';
    prompt = "Comment: ";

    rc = edit_comment(command_line, edit_buf, prompt, h_oe_editab_comment);

    if(rc == 0)
	rc = adrbk_add(abook, new, dl->a.abe->nickname, dl->a.abe->fullname,
		   dl->a.abe->tag == List ?
		       (char *)dl->a.abe->addr.list:
		       dl->a.abe->addr.addr,
		   dl->a.abe->fcc, edit_buf, dl->a.abe->tag);

    return(rc);
}


/*
 * Edit the fcc field.
 *
 * Args: command_line -- line to prompt on
 *          orig      -- fullname to edit
 *          prompt    -- prompt
 *         this_help  -- help
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
edit_fcc(command_line, orig, prompt, this_help)
int      command_line;
char    *orig,
        *prompt;
HelpType this_help;
{
    char     edit_buf[MAX_FCC + 1],
             tmp[MAX_FCC + 1];
    HelpType help;
    int      rc, slp;

    edit_buf[MAX_FCC] = '\0';
    strncpy(edit_buf, orig, MAX_FCC);
    help = NO_HELP;
    rc   = 0;
    slp  = 0;
    do{
	/* display the message queued by expand_foldername() */
	if(slp){
            display_message(NO_OP_COMMAND);
	    sleep(1);
	}

	if(rc == 3)
	    help = (help == NO_HELP ? this_help : NO_HELP);

	rc = optionally_enter(edit_buf, command_line, 0, MAX_FCC, 1,
                                  0, prompt, (ESCKEY_S *)NULL, help, 0);

	if(rc == 1) /* ^C */
	    break;
            
	tmp[MAX_FCC] = '\0';
	strncpy(tmp, edit_buf, MAX_FCC);

    }while(rc == 3 ||
	   rc == 4 ||
	   (*tmp != '{' && (slp = !expand_foldername(tmp))));

    if(rc != 0)
	return -10;

    if(strcmp(edit_buf, orig) == 0) /* no change */
	return -9;

    strcpy(orig, edit_buf);
    return 0;
}


/*
 * Change the fcc field of an address or list.
 *
 * Args: abook     -- the addressbook handle
 *       dl        -- display list line
 *       new       -- points to new entry on return
 *    command_line -- line to prompt on
 *
 * Returns: -10 to cancel
 *          -9  no change
 *              else regular adrbk_add return values
 */
int
change_fcc(abook, dl, new, command_line)
AdrBk         *abook;
AddrScrn_Disp *dl;
AdrBk_Entry  **new;
int            command_line;
{
    char  edit_buf[MAX_FCC + 1];
    char *prompt;
    int   rc;

    dprint(8, (debugfile, "- change_fcc -\n"));

    if(dl->a.abe->fcc){
	edit_buf[MAX_FCC] = '\0';
	strncpy(edit_buf, dl->a.abe->fcc, MAX_FCC);
    }
    else
	edit_buf[0] = '\0';
    if(dl->type == ListHead || dl->type == ListEnt)
	prompt = "Edit Fcc for list: ";
    else
	prompt = "Edit Fcc: ";

    rc = edit_fcc(command_line, edit_buf, prompt, h_oe_editab_fcc);

    if(rc == 0)
	rc = adrbk_add(abook, new, dl->a.abe->nickname, dl->a.abe->fullname,
		   dl->a.abe->tag == List ?
		       (char *)dl->a.abe->addr.list:
		       dl->a.abe->addr.addr,
		   edit_buf, dl->a.abe->extra, dl->a.abe->tag);

    return(rc);
}


/*
 *  Edit some individual field in the address book
 *
 *  Args: abook          -- Handle into access library for open addrbook
 *        command_line   -- Screen line number to prompt on
 *        cur_line       -- Current line in display list
 *        cur_fld        -- Current field cursor is on (to be edited)
 *
 *  Result: Returns -1 - addition was cancelled
 *                  -2 - Simple change to current, no re-sorting required
 *                  -3 - No change made
 *                >= 0 - Change made and address book re-sorted
 *
 * This can edit any of the regularly displayed fields in the address book.
 * Re-sorting of the address book may be required because of an edit.
 * The adrbklib routines do their part, the display is reinitialized, and
 * the display line number of the position of the new entry is returned
 * so the cursor can be positioned on it.
 */
int
change_address_entry(abook, command_line, cur_line, cur_fld)
AdrBk *abook;
int    command_line,
       cur_line,
       cur_fld;
{
    register AddrScrn_Disp *dl;
    AdrBk_Entry            *new         = NULL;
    char                   *addr_match  = NULL;
    int                     rc,
			    where;

    dl = dlist(cur_line);

    dprint(7, (debugfile,
	"- change_address_entry -\n   type=%d cur_line=%d cur_fld=%d\n",
	dl->type, cur_line, cur_fld));

    if((dl->type == ListHead || dl->type == Single) && cur_fld == 0)
	rc = change_nickname(abook, dl, &new, command_line);
    else if((dl->type == ListHead || dl->type == Single) && cur_fld == 1)
	rc = change_fullname(abook, dl, &new, command_line);
    else if((dl->type == Single && cur_fld == 2) || dl->type == ListEnt)
	rc = change_address(abook, dl, &new, &addr_match, command_line);
    else
	rc = -9;  /* can't happen */
        
    if(rc == -9)  /* no change */
	return -3;

    if(rc == -10){ /* cancel */
	q_status_message(0, 0, 2, "\007Address book change cancelled");
	return -1;
    }

    if(rc == -2 || rc == -3){
        q_status_message1(1, 2, 3, "\007Error updating address book: %s",
                          rc == -2 ? error_description(errno) : "Pine bug");
        dprint(1, (debugfile, "Error editing address book %s\n",
                   rc == -2 ? error_description(errno) : "Pine bug"));
        return -2;
    }

    if(new != NULL)
	where = init_disp_list(&as.adrbks[as.cur], Same, new, addr_match, 0);
    else
        where = -2; /* -2 means no re-sorting happened */

    q_status_message(0, 1, 3, "Address book edited and updated");
    dprint(2, (debugfile, "Address book edited\n"));

    return(where);
}


/*
 *  Edit fields that don't normally show on the screen (and others, too).
 *
 *  Args: abook        -- Handle into access library for open addrbook
 *        command_line -- Screen line number to prompt on
 *        cur_line     -- Current line in display list
 *
 *  Result: Returns -1 - addition was cancelled
 *                  -2 - Simple change to current, no re-sorting required
 *                  -3 - No change made
 *                  -4 - Fcc or comment, so no visible screen change
 *                >= 0 - Change made and address book re-sorted
 *
 * This can edit any of the fields in the address book.
 * Re-sorting of the address book may be required because of an edit.
 * The adrbklib routines do their part, the display is reinitialized, and
 * the display line number of the position of the new entry is returned
 * so the cursor can be positioned on it.
 */
int
edit_extras(abook, command_line, cur_line)
AdrBk *abook;
int    command_line,
       cur_line;
{
    int extras_cmd;
    int rc;
    register AddrScrn_Disp *dl;

    dprint(9, (debugfile, "- edit_extras -\n"));

    dl         = dlist(cur_line);

    extras_cmd = edit_extras_menu(command_line);
    rc         = do_edit_extras(extras_cmd, abook, command_line, dl);

    return(rc);
}

static ESCKEY_S choices[] = {
    {'c', 'c', "C", "Comment"},
    {'f', 'f', "F", "FCC"},
    {'n', 'n', "N", "Nickname"},
    {'d', 'd', "D", "FullName"},
    {'a', 'a', "A", "Address"},
    {-1, 0, NULL, NULL}
};


/*
 * User selects which field to edit.
 *
 * Args: command_line -- line to draw on
 *
 * Returns: a letter from the menu
 */
int
edit_extras_menu(command_line)
int command_line;
{
    static char *prompt = "Edit FCC, Comment, or other ? ";
    HelpType     help;
    int s;

    help = h_edit_ab_extras;

    s = radio_buttons(prompt, command_line, 0, choices, 'f', 'x', 0, help);

    /* ^C */
    if(s == 'x')
	Writechar('\007', 0);

    return(s);
}

/*
 * Fire up the selected edit in the status line.
 */
int
do_edit_extras(extras_cmd, abook, command_line, dl)
int            extras_cmd;
AdrBk         *abook;
int            command_line;
AddrScrn_Disp *dl;
{
    AdrBk_Entry *new         = NULL;
    char        *addr_match  = NULL;
    int          rc,
	         where;

    switch(extras_cmd){
      case 'f':  /* fcc */
	rc = change_fcc(abook, dl, &new, command_line);
	break;

      case 'c':  /* comment field */
	rc = change_comment(abook, dl, &new, command_line);
	break;

      case 'n':  /* nickname */
	rc = change_nickname(abook, dl, &new, command_line);
	break;

      case 'd':  /* fullname */
	rc = change_fullname(abook, dl, &new, command_line);
	break;

      case 'a':  /* address */
	rc = change_address(abook, dl, &new, &addr_match, command_line);
	break;

      case 'x':  /* ^C */
	rc = -9;
	break;
      
      default:
        q_status_message1(0, 0, 2,
	    "Can't happen, do_edit_extras(extras_cmd=%c)", (void *)extras_cmd);
	rc = -9;
	break;
    }

    if(rc == -9)  /* no change */
	return -3;

    if(rc == -10){ /* cancel */
	q_status_message(0, 0, 2, "\007Address book change cancelled");
	return -1;
    }

    if(rc == -2 || rc == -3){
        q_status_message1(1, 2, 3, "\007Error updating address book: %s",
                          rc == -2 ? error_description(errno) : "Pine bug");
        dprint(1, (debugfile, "Error editing address book %s\n",
                   rc == -2 ? error_description(errno) : "Pine bug"));
        return -2;
    }

    if(new != NULL)
	where = init_disp_list(&as.adrbks[as.cur], Same, new, addr_match, 0);
    else
        where = -2; /* -2 means no re-sorting happened */

    q_status_message(0, 1, 3, "Address book edited and updated");
    dprint(2, (debugfile, "Address book edited\n"));

    if(extras_cmd == 'f' || extras_cmd == 'c')
	return -4;

    return(where);
}


/*
 * Prompt user for search string and call find_in_book.
 * Find the line and field number of the previous line, keeping the field
 * about the same.
 *
 * Args: cur_line     -- The current line position (in global display list)
 *			 of cursor
 *       cur_fld      -- The current field position of cursor
 *       command_line -- The screen line to prompt on
 *       new_line     -- Return value: new line position
 *       new_fld      -- Return value: new fld position
 *
 * Result: The new line and field number are set if the search is successful.
 *         Returns 0 if found, -1 if not, -2 if cancelled.
 *       
 */
int
search_book(cur_line, cur_fld, command_line, new_line, new_fld)
int  cur_line,
     cur_fld,
     command_line,
     *new_line,
     *new_fld;
{
    int         x, rc;
    static char search_string[MAX_SEARCH + 1] = { '\0' };
    char        prompt[MAX_SEARCH + 50], nsearch_string[MAX_SEARCH+1];
    HelpType	help;
    ESCKEY_S    ekey[3];

    dprint(9, (debugfile, "- search_book -\n"));

    sprintf(prompt, "Word to search for [%s]: ", search_string);
    help              = NO_HELP;
    nsearch_string[0] = '\0';

    ekey[0].ch    = ctrl('Y');
    ekey[0].rval  = 10;
    ekey[0].name  = "^Y";
    ekey[0].label = "First Adr";

    ekey[1].ch    = ctrl('V');
    ekey[1].rval  = 11;
    ekey[1].name  = "^V";
    ekey[1].label = "Last Adr";

    ekey[2].ch    = -1;

    while(1){
        rc = optionally_enter(nsearch_string, command_line, 0, MAX_SEARCH,
                              1, 0, prompt, ekey, help, 0);
        if(rc == 3){
            help = help == NO_HELP ? h_oe_searchab : NO_HELP;
            continue;
        }
	else if(rc == 10){
	    int nl;
	    if((nl=first_line(0)) != -1){
		*new_fld  = 0;
		*new_line = nl;
		q_status_message(0, 0, 2, "Searched to first entry");
		return 0;
	    }
	    else{
		q_status_message(0, 0, 2, "No entries");
		return -1;
	    }
	}
	else if(rc == 11){
	    int nl;
	    if((nl=first_line(as.tot_used - 1)) != -1){
		*new_fld  = 0;
		*new_line = nl;
		q_status_message(0, 0, 2, "Searched to last entry");
		return 0;
	    }
	    else{
		q_status_message(0, 0, 2, "No entries");
		return -1;
	    }
	}

        if(rc != 4)
            break;
    }

        
    if(rc == 1 || (search_string[0] == '\0' && nsearch_string[0] == '\0'))
        return -2;

    if(nsearch_string[0] != '\0'){
        search_string[MAX_SEARCH] = '\0';
        strncpy(search_string, nsearch_string, MAX_SEARCH);
    }

    x = find_in_book(cur_line, cur_fld, search_string, new_line, new_fld);

    return(x);
}


/*
 * Search the display list for the given string.
 *
 * Args: cur_line     -- The current line position (in global display list)
 *			 of cursor
 *       cur_fld      -- The current field position of cursor
 *       string       -- String to search for
 *       new_line     -- Return value: new line position
 *       new_fld      -- Return value: new fld position
 *
 * Result: The new line and field number are set if the search is successful.
 *         Returns 0 if found, -1 if not.
 */
int
find_in_book(cur_line, cur_fld, string, new_line, new_fld)
int   cur_line,
      cur_fld;
char *string;
int  *new_line,
     *new_fld;
{
    register AddrScrn_Disp *dl,
			   *dlchk;
    int                     nl = cur_line,
			    nf = cur_fld,
			    found_field;

    dprint(9, (debugfile, "- find_in_book -\n"));

    /* search from here to the end of the disp_list */
    if(next_field(cur_line, cur_fld, &nl, &nf)){
        dlchk = dlist(nl);
        for(dl=dlist(nl); dl->type != End; dl=dlist(++nl)){
	    if((found_field=search_in_one_line(dl, nf, string)) >= 0)
	        goto found;
	    nf = 0;
        }
    }

    nl = 0;
    /* now wrap back to the start of the addressbook */
    for(dl=dlist(nl); dl != dlchk && dl->type != End; dl=dlist(++nl)){
	if((found_field=search_in_one_line(dl, nf, string)) >= 0)
	    goto found;
	nf = 0;
    }

    /* the last few entries if any in the line we're in */
    dl = dlist(cur_line);
    if((found_field=search_in_one_line(dl, nf, string)) >= 0)
	goto found;

    return -1;

found:
    /* skip to an active field */
    if(!line_is_active(nl)){
        if(!next_field(nl, found_field, &nl, &nf))
	    prev_field(nl, found_field, &nl, &nf);
	*new_line = nl;
	*new_fld  = nf;
    }
    else if(dl->type == ClickHere || dl->type == Empty){
	*new_line = nl;
	*new_fld  = 0;
    }
    else{
	*new_line = nl;
	if(dl->type == ListHead)
	    *new_fld  = min(found_field, 1);
	else
	    *new_fld  = found_field;
    }
    return 0;
}

/*
 * Look in line dl beginning with field nf for string.
 *
 * Args: dl     -- the display list for this line
 *       nf     -- start looking at this field
 *       string -- look for this string
 *
 * Result:  -1   -- string not found
 *           0-2 -- string found in field 0, 1, or 2
 */
int
search_in_one_line(dl, nf, string)
AddrScrn_Disp *dl;
int            nf;
char          *string;
{
    register int c;

    for(c = nf; c < 3; c++){
      switch(c){
	case 0:
	  switch(dl->type){
	    case Single:
	    case ListHead:
	      if(srchstr(dl->a.abe->nickname, string))
		return(c);
	      break;

	    case Text:
	    case Title:
	      if(srchstr(dl->a.txt, string))
		return(c);
	  }
	  break;

	case 1:
	  switch(dl->type){
	    case Single:
	    case ListHead:
	      if(srchstr(dl->a.abe->fullname, string))
		return(c);
	  }
	  break;

	case 2:
	  switch(dl->type){
	    case Single:
	      if(srchstr(dl->a.abe->addr.addr, string))
		return(c);
	      break;

	    case ListEnt:
	      if(srchstr(dl->other, string))
		return(c);
	      break;

	    case ListHead:
	      if(srchstr(DISTLIST, string))
		return(c);
	      break;

	    case ClickHere:
	      if(srchstr(CLICKHERE, string))
		return(c);
	      break;

	    case Empty:
	      if(srchstr(EMPTY, string))
		return(c);
	      break;
	  }
	  break;
      }
    }
    return -1;
}


/*
 * Add an entry to address book, called from outside address book.
 *
 * Args: nickname -- new nickname (can be null)
 *       fullname -- new fullname (can be null)
 *       address  -- new address (can be null)
 *       strvalue -- the address in string form, not parsed as above.  (this
 *                   may be useful for adding to a list)
 *   command_line -- line to prompt on
 *
 * Result: item is added to address book after editing by user
 *       error message queued if appropriate
 */
void
create_abook_entry(nickname, fullname, address, strvalue, command_line)
char *nickname,
     *fullname,
     *address,
     *strvalue;
int   command_line;
{
    char         new_nickname[MAX_NICKNAME + 1],
                 new_fullname[MAX_FULLNAME + 1],
		 new_address[MAX_ADDRESS + 1];
    char         prompt[200],
		*fname;
    int          rc,
		 list_add = 0,
		 changing_existing = 0,
		 which_addrbook,
		 ans;
    AdrBk       *abook;
    AddrScrState *savep;
    OpenStatus   *stp;
    PerAddrBook  *pab;
    AdrBk_Entry  *ae = (AdrBk_Entry *)NULL;

    dprint(8, (debugfile, "-- create_abook_entry --\n"));

    pab = setup_for_addrbook_add(&savep, &stp, command_line);

    /* check we got it opened ok */
    if(pab == NULL || pab->address_book == NULL)
        goto create_abook_entry_cancel;

    abook = pab->address_book;

    /*----- nickname ------*/
    new_nickname[0] = '\0';
    if(nickname){
	strncpy(new_nickname, nickname, MAX_NICKNAME);
	new_nickname[MAX_NICKNAME] = '\0';
    }
    sprintf(prompt, "%s new nickname (one word and easy to remember): ",
	new_nickname[0] ? "Edit" : "Enter");
    rc = edit_nickname(abook, (AddrScrn_Disp *)NULL, command_line,
		new_nickname, prompt, h_oe_takenick, 1, 1);
    if(rc == -8){  /* this means an existing nickname was entered */

	ae = adrbk_lookup(abook, new_nickname);
	if(!ae){  /* this shouldn't happen */
	    q_status_message1(0, 1, 3, "Already an entry %s in address book!",
		new_nickname);
	    goto create_abook_entry_cancel;
	}

	if(ae->tag == Atom){  /* single address entry */
	    sprintf(prompt, "Entry %s (%s) exists, replace", new_nickname,
		ae->fullname);
	    ans = want_to(prompt, 'n', 'n', NO_HELP, 0, 0);
	    if(ans != 'y')
		goto create_abook_entry_cancel;
	    changing_existing++;
	}
	else{  /* List */
	    sprintf(prompt, "List %s (%s) exists, add address to it",
		new_nickname, ae->fullname);
	    ans = want_to(prompt, 'n', 'n', NO_HELP, 0, 0);
	    if(ans == 'y' && address){
		changing_existing++;
		list_add++;
	    }
	    else
		goto create_abook_entry_cancel;
	}
    }
    else if(rc != 0)
	goto create_abook_entry_cancel;

    if(!list_add){  /* don't bother with full name for list additions */
	/*------ full name ------*/
	new_fullname[0] = '\0';
	if(fullname){
	    strncpy(new_fullname, fullname, MAX_FULLNAME);
	    new_fullname[MAX_FULLNAME] = '\0';
	}
	sprintf(prompt, "%s new Full Name%s: ",
	    new_fullname[0] ? "Edit" : "Enter",
	    new_fullname[0] ? "" : " (Last, First)");
	rc = edit_fullname(command_line, new_fullname, prompt, h_oe_takename);
	if(rc != 0 && rc != -9)
	    goto create_abook_entry_cancel;
    }

    /*------ address ------*/
    new_address[0] = '\0';
    if(list_add && strvalue){
	strncpy(new_address, strvalue, MAX_ADDRESS);
	new_address[MAX_ADDRESS] = '\0';
    }
    else if(address){
	strncpy(new_address, address, MAX_ADDRESS);
	new_address[MAX_ADDRESS] = '\0';
    }
    if(list_add)
	sprintf(prompt, "Edit address to add to \"%s\" list: ", new_nickname);
    else
	sprintf(prompt, "%s new e-mail address: ",
	    new_address[0] ? "Edit" : "Enter");
    rc = edit_address(command_line, list_add, new_address, prompt,
	    h_oe_takeaddr);
    if(rc != 0 && rc != -9)
	goto create_abook_entry_cancel;

    /*
     * We may also want to check when changing_existing is set.  In that
     * case, we would look only in the other addrbooks, not the one we
     * know it is in.  We don't have a routine to do that right now and it
     * may not be worth it.  It isn't as serious since the duplicates already
     * exist in that case, whereas we're adding a new duplicate in the
     * case below.
     */
    if(!changing_existing &&
	(fname = addr_lookup(new_nickname, &which_addrbook))){
	q_status_message3(0, 7, 7, "Warning! %s exists in %s as %s",
	    new_nickname, as.adrbks[which_addrbook].nickname, fname);
    }

    
    /*---- write it into the file ----*/
    if(list_add)
	rc = adrbk_listadd(abook, ae, new_address, (char **)NULL);
    else
	rc = adrbk_add(abook, (AdrBk_Entry **)NULL, new_nickname,
		   new_fullname, new_address, NULL, NULL, Atom);

    restore_state(savep, stp);

    switch(rc){
      case 0:
        q_status_message3(0, 1, 3, "%s added. Address book %s%supdated",
	    new_nickname, (as.n_addrbk>1)?pab->nickname:"",
	    (as.n_addrbk>1)?" ":"");
        dprint(2, (debugfile, "Added \"%s\",\"%s\",\"%s\" to \"%s\"\n",
	    new_nickname, new_fullname, new_address, pab->filename));
	    break;

      case -3:
      case -2:
        q_status_message1(1, 2, 4, "\007Error updating address book: %s",
               rc == -2 ? error_description(errno) : "Pine bug");
        dprint(1, (debugfile, "Error adding \"%s\",\"%s\",\"%s\" to %s: %s\n",
	    new_nickname, new_fullname, new_address, pab->filename,
		rc == -2 ? error_description(errno) : "Pine bug"));
        break;

      case -4:
        q_status_message(0, 2, 2, "\007Tabs not allowed in address book");
        break;
    }

    return;

create_abook_entry_cancel:
    q_status_message(0, 0, 1, "\007Address book addition cancelled");
    restore_state(savep, stp);
}


/*
 * Add an entry to address book, called from outside address book.
 * It is for capturing addresses off incoming mail.
 * This is a front end for create_abook_entry.
 * It is also used for replacing an existing entry and for adding a single
 * new address to an existing list.
 *
 * Args: addr     -- ADDRESS to be added
 *       strvalue -- ADDRESS in printed out form for adding to lists
 *   command_line -- line to prompt on
 *
 * Result: item is added to one of the address books,
 *       an error message is queued if appropriate.
 */
void
add_abook_entry(addr, strvalue, command_line)
ADDRESS *addr;
char    *strvalue;
int      command_line;
{
    char new_fullname[MAX_FULLNAME + 1],
	 new_address[MAX_ADDRESS + 1];

    dprint(8, (debugfile, "-- add_abook_entry --\n"));

    /*-- rearrange full name (Last, First) ---*/
    new_fullname[0]              = '\0';
    new_fullname[MAX_FULLNAME]   = '\0';
    new_fullname[MAX_FULLNAME-1] = '\0';
    if(addr->personal != NULL){
	if(strindex(addr->personal, ',') != NULL){
	    int add_quotes = 0;
	    char *nf;

	    nf = new_fullname;
	    /*
	     * We'll get this wrong if it is already quoted but the quote
	     * doesn't start right at the beginning.
	     */
	    if(addr->personal[0] != '"'){
		add_quotes++;
		*nf++ = '"';
	    }
	    strncpy(nf, addr->personal, MAX_FULLNAME-2);
	    if(add_quotes)
		strcat(nf, "\"");
	}
	else if(strindex(addr->personal, ' ') == NULL){  /* leave single word */
	    strncpy(new_fullname, addr->personal, MAX_FULLNAME);
	}
	else{
	    char *p, *q, *r;

	    /* switch to Last, First */
	    p = addr->personal;
	    while((q = strindex(p, ' ')) != NULL)
		p = q + 1;
	    for(q = p, r = new_fullname; *q; *r++ = *q++)
		;/* do nothing */
	    *r++ = ',';
	    *r++ = ' ';
	    for(q = addr->personal; q < p; *r++ = *q++)
		;/* do nothing */
	    *r = '\0';
	    for(r--; r >= new_fullname && isspace(*r); *r-- = '\0')
		;/* do nothing */
	}
    }

    /* initial value for new address */
    new_address[MAX_ADDRESS] = '\0';
    strncpy(new_address, addr->mailbox, MAX_ADDRESS);
    if(addr->host && addr->host[0] != '\0'){
	strncat(new_address, "@", MAX_ADDRESS - strlen(new_address));
	strncat(new_address, addr->host, MAX_ADDRESS - strlen(new_address));
    }

    create_abook_entry(NULL, new_fullname, new_address, strvalue, command_line);
}


/*
 * Add a list to address book, called from outside address book.
 * It is also used for adding to an existing list.
 * It is for capturing addresses off incoming mail.
 *
 * Args: new_entries -- a list of addresses to add to a list or to form
 *                      a new list with
 *     create_or_add -- force a create or add to list instead of using ad hoc
 *                      rules
 *      command_line -- line to prompt on
 *
 * Result: item is added to one of the address books,
 *       an error message is queued if appropriate.
 */
void
create_abook_list(new_entries, create_or_add, command_line)
char      **new_entries;
CreateOrAdd create_or_add;
int         command_line;
{
    char         new_nickname[MAX_NICKNAME + 1],
                 new_fullname[MAX_FULLNAME + 1];
    char         prompt[200],
		*fname,
	       **p;
    int          rc,
		 list_add = 0,
		 which_addrbook,
		 ans;
    AdrBk       *abook;
    AddrScrState *savep;
    OpenStatus   *stp;
    PerAddrBook  *pab;
    AdrBk_Entry  *ae = (AdrBk_Entry *)NULL;

    dprint(8, (debugfile, "-- create_abook_list --\n"));

    pab = setup_for_addrbook_add(&savep, &stp, command_line);

    /* check we got it opened ok */
    if(pab == NULL || pab->address_book == NULL)
        goto create_abook_list_cancel;

    abook = pab->address_book;

    /*----- nickname ------*/
    if(create_or_add == AddToList)
	sprintf(prompt, "Enter nickname of list to add to: ");
    else
	sprintf(prompt,
	    "Enter new list nickname (one word and easy to remember): ");
    new_nickname[0] = '\0';
    rc = edit_nickname(abook, (AddrScrn_Disp *)NULL, command_line,
		new_nickname, prompt, h_oe_takenick, 1, 1);
    if(rc == -8){  /* this means an existing nickname was entered */

	ae = adrbk_lookup(abook, new_nickname);
	if(!ae){  /* this shouldn't happen */
	    q_status_message1(0, 1, 3, "Already an entry %s in address book!",
		new_nickname);
	    goto create_abook_list_cancel;
	}

	if(ae->tag == Atom){  /* single address entry */
	    if(create_or_add == AddToList)
		sprintf(prompt, "Entry %s (%s) is not a list, replace",
		    new_nickname, ae->fullname);
	    else
		sprintf(prompt, "Entry %s (%s) exists, replace", new_nickname,
		    ae->fullname);
	    ans = want_to(prompt, 'n', 'n', NO_HELP, 0, 0);
	    if(ans != 'y')
		goto create_abook_list_cancel;
	    /* have to delete Atom and add List below */
            rc = adrbk_delete(abook, ae);
	    if(rc != 0){
		q_status_message1(0, 2, 5, "\007Error writing address book: %s",
                                                   error_description(errno));
		goto create_abook_list_cancel;
	    }
	}
	else{  /* List */
	    if(create_or_add == AddToList)
		list_add++;
	    else{
		sprintf(prompt, "List %s (%s) exists, add addresses to it",
		    new_nickname, ae->fullname);
		ans = want_to(prompt, 'n', 'n', NO_HELP, 0, 0);
		if(ans == 'y')
		    list_add++;
		else
		    goto create_abook_list_cancel;
	    }
	}
    }
    else if(rc != 0)
	goto create_abook_list_cancel;
    else if(create_or_add == AddToList){
	sprintf(prompt, "List %s does not exist in this addrbook, create",
	    new_nickname);
	ans = want_to(prompt, 'n', 'n', NO_HELP, 0, 0);
	if(ans != 'y')
	    goto create_abook_list_cancel;
    }

    if(!list_add){  /* don't edit full name for list additions */
	/*------ full name ------*/
	new_fullname[0] = '\0';
	sprintf(prompt, "Enter new List Full Name: ");
	rc = edit_fullname(command_line, new_fullname, prompt, h_oe_crlst_full);
	if(rc != 0 && rc != -9)
	    goto create_abook_list_cancel;
    }

    /*
     * We may also want to check when list_add is set.  In that
     * case, we would look only in the other addrbooks, not the one we
     * know it is in.  We don't have a routine to do that right now and it
     * may not be worth it.  It isn't as serious since the duplicates already
     * exist in that case, whereas we're adding a new duplicate in the
     * case below.
     */
    if(!list_add &&
	(fname = addr_lookup(new_nickname, &which_addrbook))){
	q_status_message3(0, 7, 7, "Warning! %s exists in %s as %s",
	    new_nickname, as.adrbks[which_addrbook].nickname, fname);
    }

    
    /*---- write it into the file ----*/
    /* we have to create new list head */
    if(!list_add){
	rc = adrbk_add(abook, &ae, new_nickname,
		   new_fullname, NULL, NULL, NULL, List);
    }
    /* add entries to (now) existing list */
    for(p = new_entries; *p; p++)
	if((rc=adrbk_listadd(abook, ae, *p, (char **)NULL)))
	    break;

    restore_state(savep, stp);

    switch(rc){
      case 0:
        q_status_message3(0, 1, 3, "%s added. Address book %s%supdated",
	    new_nickname, (as.n_addrbk>1)?pab->nickname:"",
	    (as.n_addrbk>1)?" ":"");
        dprint(2, (debugfile, "Added \"%s\",\"%s\" to \"%s\"\n",
	    new_nickname, new_fullname, pab->filename));
	    break;

      case -3:
      case -2:
        q_status_message1(1, 2, 4, "\007Error updating address book: %s",
               rc == -2 ? error_description(errno) : "Pine bug");
        dprint(1, (debugfile, "Error adding \"%s\",\"%s\" to %s: %s\n",
	    new_nickname, new_fullname, pab->filename,
		rc == -2 ? error_description(errno) : "Pine bug"));
        break;

      case -4:
        q_status_message(0, 2, 2, "\007Tabs not allowed in address book");
        break;
    }

    return;

create_abook_list_cancel:
    q_status_message(0, 0, 1, "\007Address book addition cancelled");
    restore_state(savep, stp);
}


/*
 * Prep addrbook for TakeAddr add operation.
 *
 * Arg: savep -- Address of a pointer to save addrbook state in.
 *      stp   -- Address of a pointer to save addrbook state in.
 *
 * Returns: a PerAddrBook pointer, or NULL.
 */
PerAddrBook *
setup_for_addrbook_add(savep, stp, command_line)
AddrScrState **savep;
OpenStatus   **stp;
int	       command_line;
{
    PerAddrBook  *pab;

    init_ab_if_needed();
    save_state(savep, stp);

    if(as.n_addrbk == 0){
        q_status_message(0, 1, 3, "Can't open address book!");
        return NULL;
    }
    else
	pab = use_this_addrbook(command_line);
    
    if(!pab)
        return NULL;

    /* initialize addrbook so we can add to it */
    (void)init_disp_list(pab, Open, (AdrBk_Entry *)NULL, NULL, 0);

    if(pab->ostatus != Open){
        q_status_message(0, 1, 3, "Can't open address book!");
        return NULL;
    }
    if(pab->access != ReadWrite){
	if(pab->access == ReadOnly)
	    q_status_message(0, 2, 3, "AddressBook is Read Only");
	else if(pab->access == NoAccess)
	    q_status_message(0, 2, 3,
		"AddressBook not accessible, permission denied");
        return NULL;
    }

    return(pab);
}


/*
 *  Interface to address book lookups for callers outside or inside this file.
 *
 * Args: nickname       -- The nickname to look up
 *       which_addrbook -- If matched, addrbook number it was found in.
 *
 * Result: returns NULL or the corresponding full name
 *
 * This opens the address books if they haven't been opened and closes them.
 */
char *
addr_lookup(nickname, which_addrbook)
char *nickname;
int  *which_addrbook;
{
    AdrBk_Entry *ae;
    AddrScrState *savep;
    OpenStatus   *stp;

    dprint(9, (debugfile, "- addr_lookup -\n"));

    init_ab_if_needed();
    save_state(&savep, &stp);

    ae = adrbk_lookup_with_opens(nickname, 0, which_addrbook);

    restore_state(savep, stp);

    return(ae == NULL ? NULL : ae->fullname);
}


/*
 * Warn about bad characters in address.
 *
 * Args:  string   -- the address to check
 *       spaces_ok -- don't check for spaces
 *
 * Returns: 0 - ok
 *          1 - bad chars, try again
 */
int
warn_bad_addr(string, spaces_ok)
char *string;
int   spaces_ok;
{
    int ic;
    int bad_addr;
    char *error = "";
    char *addr;

    if((ic = illegal_chars(string)) != 0){
	if(!(ic&0x08) && spaces_ok && ic == 0x1)
	    ;/* do nothing */
	else{
	    if(ic&0x8)
		q_status_message(0, 1, 1,
			      "\007unbalanced quotes in addresses not allowed");
	    else
		q_status_message1(0, 1, 1,
			      "\007unquoted %ss not allowed in addresses",
			      ic&0x2 ? "comma" : ic&0x4 ? "paren" : "space");

	    display_message(NO_OP_COMMAND);
	    sleep(1);
	    return 1;
	}
    }

    bad_addr = (our_build_address(string, &addr, &error, NULL, 1) >= 0) ? 0 : 1;
    fs_give((void **)&addr);

    if(*error){
        q_status_message1(0, 1, 1, "%s", error);
	display_message(NO_OP_COMMAND);
	sleep(3);
    }
    return(bad_addr);
}


/*
 * Looks for illegal characters in addresses.  Don't have to worry about
 * \ quoting since addr-specs can't use that kind of quoting.
 * 
 * Args:  string -- the address to check
 *
 * Returns: 0 -- ok
 *       else, bitwise or of
 *          0x1 -- found a space
 *          0x2 -- found a comma
 *          0x4 -- found a (
 *          0x8 -- unbalanced "'s 
 *
 * (Should probably generalize this a little to take an argument telling
 *  what to look for.  I'm going to put that off for now.)
 */
int
illegal_chars(string)
char *string;
{
  register char *p;
  register int   in_quotes = 0,
                 ret = 0;

  for(p=string; p && *p; p++){
    if(*p == '"'){
      in_quotes = !in_quotes;
    }
    else if(!in_quotes){
      switch(*p){
	case SPACE:
	  ret |= 0x1;
	  break;
	case ',':
	  ret |= 0x2;
	  break;
	case '(':
	  ret |= 0x4;
	  break;
      }
    }
  }

  if(in_quotes)
    ret |= 0x8;

  return ret;
}


/*
 * These chars in nicknames will mess up parsing.
 *
 * Returns 0 if ok, 1 if not.
 * Prints an error status message on error.
 */
int
nickname_check(nickname)
char *nickname;
{
    register char *t;

    if((t = strindex(nickname, ' ')) ||
       (t = strindex(nickname, ',')) ||
       (t = strindex(nickname, '"')) ||
       (t = strindex(nickname, ';')) ||
       (t = strindex(nickname, ':')) ||
       (t = strindex(nickname, '@')) ||
       (t = strindex(nickname, '(')) ||
       (t = strindex(nickname, ')')) ||
       (t = strindex(nickname, '\\')) ||
       (t = strindex(nickname, '[')) ||
       (t = strindex(nickname, ']')) ||
       (t = strindex(nickname, '<')) ||
       (t = strindex(nickname, '>'))){
	char s[4];
	s[0] = '"';
	s[1] = *t;
	s[2] = '"';
	s[3] = '\0';
	q_status_message1(0, 1, 2, "\007%s not allowed in nicknames",
	    *t == ' ' ?
		"Blank spaces" :
		*t == ',' ?
		    "Commas" :
		    *t == '"' ?
			"Quotes" :
			s);
	display_message(NO_OP_COMMAND);
	(void)sleep((unsigned)1);
	return 1;
    }
    return 0;
}


/*
 * This is like build_address() only it doesn't close
 * everything down when it is done, and it doesn't open addrbooks that
 * are already open.  Other than that, it has the same functionality.
 * It opens addrbooks that haven't been opened and saves and restores the
 * addrbooks open states (if save_and_restore is set).
 *
 * Args: to                    -- the address to attempt expanding
 *       full_to               -- a pointer to result
 *       error                 -- a pointer to an error message, if non-null
 *       fcc                   -- a pointer to returned fcc, if non-null
 *       save_and_restore      -- restore addrbook states when finished
 *
 * Results:    0 -- address is ok
 *            -1 -- address is not ok
 * full_to will contains the expanded address on success, or a copy to to
 *         on failure
 * *error  will point to an error message on failure it it is non-null
 */
int
our_build_address(to, full_to, error, fcc, save_and_restore)
char *to,
    **full_to,
    **error,
    **fcc;
int   save_and_restore;
{
    int ret;

    dprint(9, (debugfile, "- our_build_address -  (to=%s)\n", to));

    ret = build_address_internal(to, full_to, error, fcc, save_and_restore);

    dprint(9, (debugfile, "   our_build_address says %s address\n",
	ret ? "BAD" : "GOOD"));

    return(ret);
}


/*
 * This is the build_address used by the composer to check for an address
 * in the addrbook.
 *
 * Args: to      -- the passed in line to parse
 *       full_to -- Address of a pointer to return the full address in.
 *		    This will be allocated here and freed by the caller.
 *       error   -- Address of a pointer to return an error message in.
 *                  This is not allocated so should not be free by the caller.
 *       fcc     -- Address of a pointer to return the fcc in.
 *		    This will be allocated here and freed by the caller.
 *
 * Result:  0 is returned if address was OK, 
 *         -1 if address wasn't OK.
 */
int
build_address(to, full_to, error, fcc)
char *to,
    **full_to,
    **error,
    **fcc;
{
    register char *p;

    dprint(9, (debugfile, "- build_address -\n"));

    /* check to see if to string is empty to avoid work */
    for(p = to; *p && isspace(*p); p++)
	;/* do nothing */
    if(!*p){
	*full_to = cpystr(to);  /* because composer does a strcmp() */
	return 0;
    }

    return(build_address_internal(to, full_to, error, fcc, 1));
}


/*
 * Given an address, expand it based on address books, local domain, etc.
 * This will open addrbooks if needed before checking (actually one of
 * its children will open them).
 *
 * Args: to       -- The given address to expand
 *       full_to  -- Returned value after parsing to.
 *       error    -- This gets pointed at error message, if any
 *       fcc      -- Returned value of fcc for first addr in to
 *       save_and_restore -- restore addrbook state when done
 *
 * Result:  0 is returned if address was OK, 
 *         -1 if address wasn't OK.
 * The address is expanded, fully-qualified, and personal name added.
 *
 * Input may have more than one address separated by commas.
 */
int
build_address_internal(to, full_to, error, fcc, save_and_restore)
char *to,
    **full_to,
    **error,
    **fcc;
int   save_and_restore;
{
    ADDRESS *a;
    int      loop;
    int      tried_route_addr_hack = 0;
    int      estimated_size;
    char     tmp[MAX_ADDR_FIELD + 3];
    static char *fcc_last = NULL;
    AddrScrState *savep;
    OpenStatus   *stp;

    dprint(9, (debugfile, "- build_address_internal -\n"));

    init_ab_if_needed();
    if(save_and_restore)
	save_state(&savep, &stp);

start:
    loop = 0;
    ps_global->c_client_error[0] = '\0';

    a = expand_address(to, ps_global->maildomain,
                         !ps_global->do_local_lookup ? NULL :
                        ps_global->use_full_hostname ? ps_global->hostname :
                           ps_global->localdomain, &loop, fcc);

    /*
     * If the address is a route-addr, expand_address() will have rejected
     * it unless it was enclosed in brackets, since route-addrs can't stand
     * alone.  Try it again with brackets.  We should really be checking
     * each address in the list of addresses instead of assuming there is
     * only one address, but we don't want to have this function know
     * all about parsing rfc822 addrs.
     */
    if(!tried_route_addr_hack &&
        ps_global->c_client_error[0] != '\0' &&
	to[0] == '@'){

	tried_route_addr_hack++;

	/* add brackets to whole thing */
	strcat(strcat(strcpy(tmp, "<"), to), ">");

	loop = 0;
	ps_global->c_client_error[0] = '\0';

	/* try it */
	a = expand_address(tmp, ps_global->maildomain,
			     !ps_global->do_local_lookup ? NULL :
			ps_global->use_full_hostname ? ps_global->hostname :
			   ps_global->localdomain, &loop, fcc);

	/* if no error this time, use it (even replace to with it) */
	if(ps_global->c_client_error[0] == '\0'){
	    strcpy(to, tmp);
	    goto ok;
	}
	else  /* go back and use what we had before, so we get the error */
	    goto start;
    }

    if(save_and_restore)
	restore_state(savep, stp);

    if(ps_global->c_client_error[0] != '\0'){
        /* Parse error.  Return string as is and error message */
	*full_to = cpystr(to);
        if(error != NULL)
            *error = ps_global->c_client_error;
        dprint(2, (debugfile,
	    "build_address_internal returning parse error: %s\n",
                   ps_global->c_client_error));
        return -1;
    }

    /*
     * If there's a loop in the addressbook, we modify the address and
     * send an error back, but we still return 0.
     */
ok:
    if(loop){
        if(error != NULL)
            *error = "Loop or Duplicate detected in addressbook!"; 
    }


    estimated_size = est_size(a);
    *full_to       = fs_get((size_t)estimated_size);
    (*full_to)[0]  = '\0';
    rfc822_write_address(*full_to, a);

    /*
     * This first condition means that addressbook fcc's override the
     * fcc-rule.
     */
    if(fcc && !*fcc){
	if(ps_global->fcc_rule == FCC_RULE_LAST){
	    if(fcc_last)
		*fcc = cpystr(fcc_last);
	}
	else{
	    if(ps_global->fcc_rule == FCC_RULE_RECIP &&
		get_uname(a ? a->mailbox : NULL, tmp, MAX_ADDR_FIELD + 3))
		*fcc = cpystr(tmp);
	    else
		*fcc = cpystr(ps_global->VAR_DEFAULT_FCC);

	    if(fcc_last)
		fs_give((void **)&fcc_last);
	    fcc_last = cpystr(*fcc);
	}
    }

    mail_free_address(&a);

    return 0;
}


static char  *fakedomain;
static int    recursing = 0;       
/*
 * Expand an address string against the address books, local names, and domain.
 *
 * Args: a_string -- the address string to parse and expand
 *     userdomain -- domain the user is in
 *    localdomain -- domain of the password file (usually same as userdomain)
 *  loop_detected -- pointer to an int we set if we detect a loop in the
 *		     address books (or a duplicate in a list)
 *       fcc      -- Returned value of fcc for first addr in a_string
 *
 * Result: An adrlist of expanded addresses is returned
 *
 * If the localdomain is NULL, then no lookup against the password file will
 * be done.
 */
ADDRESS *
expand_address(a_string, userdomain, localdomain, loop_detected, fcc)
char   *a_string,
       *userdomain,
       *localdomain;
int    *loop_detected;
char  **fcc;
{
    int          domain_length, length;
    ADDRESS     *adr, *a, *a_tail, *adr2, *a2, *a_temp;
    AdrBk_Entry *abe;
    char        *list, *l1, **l2;
    char        *tmp_a_string;

    dprint(9, (debugfile, "- expand_address -  (%s)\n", a_string));

    /*
     * Create a fake domain name '@@@@', the same length as the real domain
     * name.  Then in the common case where the local domain name is filled in,
     * the right storage is already allocated.  We also use the @@@ to detect
     * an unqualified address.  This is done via a c-client hack.  If an
     * ADDRESS has a host part that begins with @ then rfc822_write_address()
     * will write only the local part and leave off the @domain part.  So
     * we can parse nicknames with this, too.
     */
    if(!recursing){
        domain_length = max(localdomain != NULL ? strlen(localdomain) : 0,
                             userdomain != NULL ? strlen(userdomain)  : 0);
        fakedomain = fs_get(domain_length + 1);
        memset((void *)fakedomain, '@', (size_t)domain_length);
        fakedomain[domain_length] = '\0';
	if(fcc)
	    *fcc = NULL;
    }

    adr = NULL;

    tmp_a_string = cpystr(a_string); /* next func feels free to destroy input */
    rfc822_parse_adrlist(&adr, tmp_a_string, fakedomain);
    fs_give((void **)&tmp_a_string);

    for(a = adr, a_tail = adr; a;){

        if(a->host && a->host[0] == '@'){
            /*
             * Hostname is  '@@@@...' indicating name wasn't qualified.
             * Need to look up in address book, and the password file.
             * If no match then fill in the local domain for host
             */
            abe = adrbk_lookup_with_opens(a->mailbox, !recursing, (int *)NULL);

            if(abe == NULL){
                if(localdomain != NULL && a->personal == NULL){
                    /* lookup in passwd file for local full name */
                    a->personal = local_name_lookup(a->mailbox); 

                    strcpy(a->host, a->personal == NULL ? userdomain :
                                                          localdomain);
                }
		else
                    strcpy(a->host, userdomain);

                /*--- Move to next address in list -----*/
                a_tail = a;
                a = a->next;

            }
	    else{
                /*
                 * There was a match in the address book.  We have to do a lot
                 * here because the item from the address book might be a 
                 * distribution list.  Treat the string just like an address
                 * passed in to parse and recurse on it.  Then combine
                 * the personal names from address book.  Lastly splice
                 * result into address list being processed
                 */

		/*
		 * Easy case for fcc.  This is a nickname that has
		 * an fcc associated with it.
		 */
		if(a == adr && fcc && !*fcc && abe->fcc && abe->fcc[0])
		    *fcc = cpystr(abe->fcc);

                if(recursing && abe->referenced){
                     /*---- caught an address loop! ----*/
                    fs_give(((void **)&a->host));
		    (*loop_detected)++;
                    a->host = cpystr("***address-loop-or_duplicate***");
                    continue;
                }
                abe->referenced++;   /* For address loop detection */
                if(abe->tag == List){
                    length = 0;
                    for(l2 = abe->addr.list; *l2; l2++)
                        length += (strlen(*l2) + 1);
                    list = fs_get(length + 1);
                    l1 = list;
                    for(l2 = abe->addr.list; *l2; l2++){
                        if(l1 != list)
                            *l1++ = ',';
                        strcpy(l1, *l2);
                        l1 += strlen(l1);
                    }
                    recursing++;
                    adr2 = expand_address(list, userdomain, localdomain,
						loop_detected, fcc);
                    recursing--;
                    fs_give((void **)&list);
                }
		else{
                    if(strcmp(abe->addr.addr, a->mailbox)){
                        recursing++;
                        adr2 = expand_address(abe->addr.addr, userdomain,
				    localdomain, loop_detected, fcc);
                        recursing--;
                    }
		    else{
                        /*
			 * A loop within plain single entry is ignored.
			 * Set up so later code thinks we expanded.
			 */
                        adr2          = mail_newaddr();
                        adr2->mailbox = cpystr(a->mailbox);
                        adr2->host    = cpystr(userdomain);
                        adr2->adl     = cpystr(a->adl);
                    }
                }
                if(adr2 == NULL){
                    /* expanded to nothing, hack out of list */
                    a_temp = a;
                    if(a == adr){
                        adr    = a->next;
                        a      = adr;
                        a_tail = adr;
                    }
		    else{
                        a_tail->next = a->next;
                        a            = a->next;
                    }
                    mail_free_address(&a_temp);
                    continue;
                }
                /*
                 * Personal names:  If the expanded address has a personal
                 * name, tack the full name from the address book on in front.
                 * This mainly occurs with a distribution list where the
                 * list has a full name, and the first person in the list also
                 * has a full name.  Also avoid the case where a name might
		 * get doubled by getting expanded against the address book
		 * and against the passwd file.  This happens if the address
		 * in the entry is just the unqualified login in the passwd
		 * file.
                 */
                if(adr2->personal && abe->fullname &&
                   strcmp(adr2->mailbox, abe->addr.addr)){
                    /* combine list name and personal name */
                    char *name = fs_get(strlen(adr2->personal) +
                                      strlen(abe->fullname) + 5);
                    sprintf(name, "%s -- %s", abe->fullname, adr2->personal);
                    fs_give((void **)&adr2->personal);
                    adr2->personal = name;
                }
		else
                    adr2->personal = cpystr(adrbk_formatname(abe->fullname));

                /* splice new list into old list */
                for(a2 = adr2; a2->next != NULL; a2 = a2->next)
		    ;/* do nothing */
                a2->next = a->next;
                if(a == adr){
                    adr    = adr2;
                    a_tail = a2;
                }
		else
                    a_tail->next = adr2;

                /* advance to next item, and free replaced ADDRESS */
                a_tail       = a2;
                a_temp       = a;
                a            = a->next;
                a_temp->next = NULL;  /* So free won't do whole list */
                mail_free_address(&a_temp);
            }

        }
	else{
            /* Already fully qualified hostname -- nothing to do */
            a_tail = a;
            a      = a->next;
        }

	/* if it's first addr in list and fcc hasn't been set yet */
	if(a_tail == adr && fcc && !*fcc){
	    AdrBk_Entry *ae;

	    /*
	     * This looks for the addressbook entry that matches the given
	     * address.  It looks through all the addressbooks
	     * looking for an exact match and then returns that entry.
	     */
	    ae = addr_to_ae(a_tail);
	    if(ae && ae->fcc && ae->fcc[0])
		*fcc = cpystr(ae->fcc);
	}
    }

    if(!recursing)
      fs_give((void **)&fakedomain);

    return(adr);
}


/*
 * Look through addrbooks for nickname, opening addrbooks first
 * if necessary.  It is assumed that the caller will restore the
 * state of the addrbooks if desired.
 *
 * Args: nickname  -- the nickname to lookup
 *       clearrefs -- clear reference bits before lookup
 *      which_addrbook -- If matched, addrbook number it was found in.
 *
 * Results: A pointer to an AdrBk_Entry is returned, or NULL if not found.
 * Stop at first match (so order of addrbooks is important).
 */
AdrBk_Entry *
adrbk_lookup_with_opens(nickname, clearrefs, which_addrbook)
char *nickname;
int   clearrefs;
int  *which_addrbook;
{
    AdrBk_Entry *ae;
    int i;
    PerAddrBook *pab;

    dprint(9, (debugfile, "- adrbk_lookup_with_opens -\n"));

    ae = NULL;

    for(i = 0; i < as.n_addrbk; i++){

	pab = &as.adrbks[i];

	if(pab->ostatus != Open && pab->ostatus != NoDisplay){
	    (void)init_disp_list(pab, NoDisplay, (AdrBk_Entry *)NULL, NULL, 1);
	}
	if(clearrefs)
	    adrbk_clearrefs(pab->address_book);
	ae = adrbk_lookup(pab->address_book, nickname);
	if(ae)
	    break;
    }

    if(ae && which_addrbook)
	*which_addrbook = i;

    return(ae);
}


/*
 * Find the addressbook entry that matches the argument address.
 * Searches through all addressbooks looking for the match.
 * Opens addressbooks if necessary.  It is assumed that the caller
 * will restore the state of the addrbooks if desired.
 *
 * Args: addr -- the address we're trying to match
 *
 * Returns:  NULL -- no match found
 *           ae   -- a pointer to the addrbook entry that matches
 */
AdrBk_Entry *
addr_to_ae(addr)
ADDRESS *addr;
{
    register PerAddrBook *pab;
    int i;
    register int n, n_addrs;
    AdrBk_Entry *ae;

    /* for each addressbook */
    for(i = 0; i < as.n_addrbk; i++){

	pab = &as.adrbks[i];

	if(pab->ostatus != Open && pab->ostatus != NoDisplay)
	    (void)init_disp_list(pab, NoDisplay, (AdrBk_Entry *)NULL, NULL, 1);

	n_addrs = adrbk_count(pab->address_book);

	for(n = 0; n < n_addrs; n++){

	    ae = adrbk_get(pab->address_book, (unsigned)n);

	    if(ae->tag == List)
		continue;

	    if(addresses_match(addr, ae))
		return(ae);
	}
    }
    return NULL;
}


/*
 * Check to see if two addresses are the same user
 *
 * Args: addr -- First address
 *       ae   -- An addressbook entry to test against.
 *
 * There is an assumption that the passed in ae is a Single address.
 *
 * Result:  1 -- they match
 *          0 -- don't match
 */
int
addresses_match(addr, ae)
register ADDRESS     *addr;
register AdrBk_Entry *ae;
{
    char tmp[MAX_ADDR_FIELD];

    if(!(addr && ae && addr->mailbox && ae->addr.addr))
	return 0;

    if(addr->mailbox[0] != ae->addr.addr[0])
	return 0;

    strcpy(tmp, addr->mailbox);

    if(addr->host && addr->host[0])
	strcat(strcat(tmp, "@"), addr->host);

    return(strcmp(tmp, ae->addr.addr) == 0);
}


/*
 * Turn a list of address structures into a formatted string
 *
 * Args: adrlist -- An adrlist
 * Result:  malloced, comma seperated list of addresses
 */
char *
addr_list_string(adrlist)
ADDRESS *adrlist;
{
    int               len;
    char             *string, *s;
    register ADDRESS *a;

    if(!adrlist)
        return(cpystr(""));
    
    len = 0;
    for(a = adrlist; a; a = a->next)
        len += (strlen(addr_string(a)) + 2);

    string = fs_get(len);
    s      = string;
    s[0]   = '\0';

    for(a = adrlist; a; a = a->next){

	strcpy(s, addr_string(a));
	s += strlen(s);

	if(a->next){
	    *s++ = ',';
	    *s++ = ' ';
	}
    }

    return(string);
}


/*
 * Format an address structure into a string
 *
 * Args: addr -- Single ADDRESS structure to turn into a string
 *
 * Result:  Returns pointer to internal static formatted string.
 * Just uses the c-client call to do this.
 */
char *
addr_string(addr)
ADDRESS *addr;
{
    ADDRESS tmp_addr;
    static char string[MAX_ADDR_EXPN+1];

    string[0] = '\0';
    tmp_addr = *addr;
    tmp_addr.next = NULL;
    rfc822_write_address(string, &tmp_addr);
    return(string);
}


/*
 * Check to see if address is that of the current user running pine
 *
 * Args: a  -- Address to check
 *       ps -- The pine_state structure
 *
 * Result: returns 1 if it matches, 0 otherwise.
 *
 * The mailbox must match the user name and (the hostname must match or
 * the full name must match).  In matching the hostname, matches occur if
 * the hostname on the message is blank, or if it matches the local
 * hostname, the full hostname with qualifying domain, or the qualifying
 * domain without a specific host.  Note, there is a very small chance
 * that we will err on the non-conservative side here.  That is, we may
 * decide two addresses are the same even though they are different
 * (because we do case-insensitive compares on the mailbox).  That might
 * cause a reply not to be sent to somebody because they look like they
 * are us.  This should be very, very rare.
 */
int
address_is_us(a, ps)
ADDRESS     *a;
struct pine *ps;
{
    if(!a || a->mailbox == NULL)
        return 0;

    /* at least LHS must match */
    if(strucmp(a->mailbox, ps->VAR_USER_ID) == 0

                      &&
       /* and either personal name matches or hostname matches */

    /* personal name matches if it's not there or if it actually matches */
   ((a->personal == NULL || strucmp(a->personal , ps->VAR_PERSONAL_NAME) == 0)
                              ||
    /* hostname matches if it's not there, */
    (a->host == NULL ||
       /* or if hostname and userdomain (the one user sets) match exactly, */
      ((ps->userdomain && a->host && strucmp(a->host,ps->userdomain) == 0) ||

              /*
               * or if(userdomain is either not set or it is set to be
               * the same as the localdomain or hostname) and (the hostname
               * of the address matches either localdomain or hostname)
               */
             ((ps->userdomain == NULL ||
               strucmp(ps->userdomain, ps->localdomain) == 0 ||
               strucmp(ps->userdomain, ps->hostname) == 0) &&
              (strucmp(a->host, ps->hostname) == 0 ||
               strucmp(a->host, ps->localdomain) == 0))))))
        return 1;

    return 0;
}


/*
 * Compute an upper bound on the size of the array required by
 * rfc822_write_address for this list of addresses.
 *
 * Args: adrlist -- The address list.
 *
 * Returns -- an integer giving the upper bound
 */
int
est_size(adrlist)
ADDRESS *adrlist;
{
    ADDRESS *b;
    register int cnt = 0;

    for(b = adrlist; b; b = b->next){
	cnt   += (b->personal ? strlen(b->personal) : 0);
	cnt   += (b->mailbox  ? strlen(b->mailbox)  : 0);
	cnt   += (b->adl      ? strlen(b->adl)      : 0);
	cnt   += (b->host     ? strlen(b->host)     : 0);
	/*
	 * add room for:
         *   possible single space between fullname and addr
         *   left and right brackets
         *   @ sign
         *   possible : for route addr
         *   , <space>
	 *
	 * So I really think that adding 7 is enough.  Instead, I'll add 10.
	 */
	cnt   += 10;
    }
    return(max(cnt, 200));  /* just making sure */
}


/*
 * Interact with user to figure out which address book they want to add a
 * new entry (TakeAddr) to.
 *
 * Args: command_line -- just the line to prompt on
 *
 * Results: returns a pab pointing to the selected addrbook, or NULL.
 */
PerAddrBook *
use_this_addrbook(command_line)
int command_line;
{
    HelpType   help;
    int        rc;
    ESCKEY_S   ekey[3];
    PerAddrBook  *pab;
#define MAX_ABOOK 1000
    int        i, abook_num, count_read_write;
    char       addrbook[MAX_ABOOK + 1],
               prompt[MAX_ABOOK + 81];

    dprint(9, (debugfile, "\n - use_this_addrbook -\n"));

    /* check for only one ReadWrite addrbook */
    count_read_write = 0;
    for(i = 0; i < as.n_addrbk; i++){
	pab = &as.adrbks[i];
	/*
	 * NoExists is counted, too, so the user can add to an empty
	 * addrbook the first time.
	 */
	if(pab->access == ReadWrite || pab->access == NoExists){
	    count_read_write++;
	    pab = &as.adrbks[i];
	}
    }

    /* only one usable addrbook, use it */
    if(count_read_write == 1)
	return(pab);

    /* no addrbook to write to */
    if(count_read_write == 0){
	q_status_message1(0, 1, 2, "\007No %sAddressbook to Take to!",
	    (as.n_addrbk > 0) ? "writable " : "");
	return NULL;
    }

    /* start with the first addrbook */
    abook_num = 0;
    pab       = &as.adrbks[abook_num];

    /* set up extra command option keys */
    rc = 0;

    ekey[rc].ch      = ctrl('P');
    ekey[rc].rval    = 10;
    ekey[rc].name    = "^P";
    ekey[rc++].label = "Prev AddrBook";

    ekey[rc].ch      = ctrl('N');
    ekey[rc].rval    = 11;
    ekey[rc].name    = "^N";
    ekey[rc++].label = "Next AddrBook";

    ekey[rc].ch = -1;

    strncpy(addrbook, pab->nickname, MAX_ABOOK);
    addrbook[MAX_ABOOK] = '\0';
    sprintf(prompt, "Take to which addrbook : %s",
	(pab->access == ReadOnly || pab->access == NoAccess) ?
	    "[ReadOnly] " : "");
    help = NO_HELP;
    ps_global->mangled_footer = 1;
    do{
	if(!pab){
            q_status_message1(0, 1, 1,
		    "\007No addressbook \"%s\"", addrbook);
            display_message(NO_OP_COMMAND);
            sleep(2);
	}

	if(rc == 3)
            help = (help == NO_HELP ? h_oe_chooseabook : NO_HELP);

	rc = optionally_enter(addrbook, command_line, 0, MAX_ABOOK, 1,
                                  0, prompt, ekey, help, 0);

	if(rc == 1) /* ^C */
	    break;

	if(rc == 10){			/* Previous addrbook */
	    if(--abook_num < 0)
		abook_num = as.n_addrbk - 1;

	    pab = &as.adrbks[abook_num];
	    strncpy(addrbook, pab->nickname, MAX_ABOOK);
	    sprintf(prompt, "Take to which addrbook : %s",
		(pab->access == ReadOnly || pab->access == NoAccess) ?
		    "[ReadOnly] " : "");
	}
	else if(rc == 11){			/* Next addrbook */
	    if(++abook_num > as.n_addrbk - 1)
		abook_num = 0;

	    pab = &as.adrbks[abook_num];
	    strncpy(addrbook, pab->nickname, MAX_ABOOK);
	    sprintf(prompt, "Take to which addrbook : %s",
		(pab->access == ReadOnly || pab->access == NoAccess) ?
		    "[ReadOnly] " : "");
	}

    }while(rc == 2 || rc == 3 || rc == 4 || rc == 10 || rc == 11 || rc == 12 ||
           !(pab = check_for_addrbook(addrbook)));
            
    if(rc != 0)
	return NULL;

    return(pab);
}


/*
 * Return a pab pointer to the addrbook which corresponds to the argument.
 * 
 * Args: addrbook -- the string representing the addrbook.
 *
 * Results: returns a PerAddrBook pointer for the referenced addrbook, NULL
 *          if none.  First the nicknames are checked and then the filenames.
 *          This must be one of the existing addrbooks.
 */
PerAddrBook *
check_for_addrbook(addrbook)
char *addrbook;
{
    register int i;
    register PerAddrBook *pab;

    for(i = 0; i < as.n_addrbk; i++){
	pab = &as.adrbks[i];
	if(strucmp(pab->nickname, addrbook) == 0)
	    break;
    }

    if(i < as.n_addrbk)
	return(pab);

    for(i = 0; i < as.n_addrbk; i++){
	pab = &as.adrbks[i];
	if(strucmp(pab->filename, addrbook) == 0)
	    break;
    }

    if(i < as.n_addrbk)
	return(pab);
    
    return NULL;
}


static struct key takeaddr_keys[] = 
       {{"?","Help",0},	       {NULL,NULL, 0},       {"E","ExitTake",0},
	{"S","[Select]",0},    {"P","Prev", 0},      {"N","Next", 0},
        {"-","PrevPage",0},    {"Spc","NextPage",0}, {"M","Mark/UnMark",0},
	{"A","Mark All",0},    {"U","UnMark All",0}, {"Z","AddToList",0}};
static struct key_menu takeaddr_keymenu = {sizeof(takeaddr_keys)/(sizeof(takeaddr_keys[0])*12), 0, 0, 0, 0, 0, takeaddr_keys};

#define	next_taline(p)	((p) ? (p)->next : NULL)
#define	prev_taline(p)	((p) ? (p)->prev : NULL)


/*
 * Screen for selecting which addresses to Take to address book.  Called
 * from outside of address book.
 *
 * Args:      ps -- Pine state
 *       adrlist -- Screen is formed from this adrlist.
 *
 * Result: an address book may be updated
 */
void
takeaddr_screen(ps, adrlist)
struct pine *ps;
ADDRESS *adrlist;
{
    int		  orig_ch, dline, how_many_selected;
    int           ch = 'x',
                  done = 0;
    TA_S	 *current = NULL, *ctmp = NULL;
    TA_SCREEN_S   screen;
    ADDRESS      *addr;

    screen.current = screen.top_line = NULL;

    /* Allocate and initialize list */
    for(addr = adrlist; addr; addr = addr->next){
	/* This catches the c-client convention for handling group syntax. */
	if(addr->mailbox && !addr->host)
	    continue;
	new_taline(&current);
	/* This is our signal to put in a blank line. */
	if(addr->mailbox == NULL){
	    current->is_blank = 1;
	    /* remove default check from first if multiple messages */
	    first_taline(current)->checked = 0;
	    how_many_selected = 0;
	}
	else{
	    current->checked = 0;
	    if(addr == adrlist){
		current->checked = 1; /* first one checked by default */
		how_many_selected = 1;
	    }
	    current->addr     = addr;
	    current->strvalue = cpystr(addr_string(addr));
	}
    }

    if(current == NULL){
	q_status_message(0, 0, 1, "No addresses to take, cancelled");
	return;
    }

    current	       = first_taline(current);
    ps->mangled_screen = 1;
    ta_screen	       = &screen;

    while(!done){

	ps->redrawer = takeaddr_screen_redrawer;

	if(ps->mangled_screen){
	    ps->mangled_header = 1;
	    ps->mangled_footer = 1;
	    ps->mangled_body   = 1;
	    ps->mangled_screen = 0;
	}

	/*----------- Check for new mail -----------*/
	if(new_mail(NULL, 0,
	    ch==NO_OP_IDLE ? 0 : ch==NO_OP_COMMAND ? 1 : 2) >= 0)
	  ps->mangled_header = 1;

	if(ps->mangled_header){
	    set_titlebar("TAKE ADDRESS SCREEN", ps->context_current, 
			 ps->cur_folder, ps->msgmap, 1, FolderName, 0, 0);
	    ps->mangled_header = 0;
	}

	dline = update_takeaddr_screen(ps, current, &screen);

	/*---- This displays new mail notification, or errors ---*/
        display_message(ch);

	/*---- Redraw footer ----*/
	if(ps->mangled_footer){
	    bitmap_t bitmap;

	    setbitmap(bitmap);
	    ps->mangled_footer = 0;
	    draw_keymenu(&takeaddr_keymenu, bitmap,
		ps->ttyo->screen_cols, -2, 0, FirstMenu, 0);
	}

        /*------ Read the command from the keyboard ----*/
	MoveCursor(max(0, ps->ttyo->screen_rows - 3), 0);
	ch = orig_ch = read_command();

        if(ch <= 0xff && isupper(ch))
          ch = tolower(ch);

	switch(ch){
	  case '?':				/* help! */
	  case PF1:
	  case ctrl('G'):
	    helper(h_takeaddr_screen, "HELP FOR TAKE ADDRESS SCREEN", 1);
	    ps->mangled_screen = 1;
	    break;

	  case 'e':				/* exit takeaddr screen */
	  case PF3:
	  case ctrl('C'):
	    q_status_message(0, 0, 1, "Address book addition cancelled");
	    done++;
	    break;

	  case 's':				/* take marked addrs */
	  case PF4:
	  case ctrl('M'):
	  case ctrl('J'):
	    done = ta_take_marked_addrs(how_many_selected,
		       first_taline(current));
	    break;

	  case 'z':                             /* add to existing list */
	  case PF12:
	    done = ta_add_to_list(how_many_selected, first_taline(current));
	    break;

	  case 'n':				/* next list element */
	  case PF6:
	  case '\t':
	  case ctrl('F'):
	  case KEY_RIGHT:
	  case ctrl('N'):			/* down arrow */
	  case KEY_DOWN:
	    if(ctmp = next_nonblank_taline(current))
	        current = ctmp;
	    break;

	  case 'p':				/* previous list element */
	  case PF5:
	  case ctrl('B'):
	  case KEY_LEFT:
	  case ctrl('P'):			/* up arrow */
	  case KEY_UP:
	    if(ctmp = prev_nonblank_taline(current))
	        current = ctmp;
	    break;

	  case '+':				/* page forward */
	  case ' ':
	  case PF8:
	  case ctrl('V'):
	    while(dline++ < ps->ttyo->screen_rows - FOOTER_LINES){
	        if(ctmp = next_taline(current))
		    current = ctmp;
	        else
		    break;
	    }
	    /* move off of blank line */
	    if(current->is_blank){
		/* try forward */
		if(ctmp = next_nonblank_taline(current))
		    current = ctmp;
		/* else try backwards */
		else if(ctmp = prev_nonblank_taline(current))
		    current = ctmp;
	    }
	    break;

	  case '-':				/* page backward */
	  case PF7:
	  case ctrl('Y'):
	    /* move to top of screen */
	    while(dline-- > HEADER_LINES){
	        if(ctmp = prev_taline(current))
		    current = ctmp;
	        else
		    break;
	    }
	    /* page back one screenful */
	    while(++dline < ps->ttyo->screen_rows - FOOTER_LINES){
	        if(ctmp = prev_taline(current))
		    current = ctmp;
	        else
		    break;
	    }
	    /* move off of blank line */
	    if(current->is_blank){
		/* try forward */
		if(ctmp = next_nonblank_taline(current))
		    current = ctmp;
		/* else try backwards */
		else if(ctmp = prev_nonblank_taline(current))
		    current = ctmp;
	    }
	    break;

	  case ctrl('L'):
          case KEY_RESIZE:
	    ClearScreen();
	    ps->mangled_screen = 1;
	    break;

	  case 'm':  /* mark or unmark this addr */
	  case PF9:
	    current->checked = 1 - current->checked;  /* flip it */
	    how_many_selected += (current->checked ? 1 : -1);
	    break;

	  case 'a':  /* mark all */
	  case PF10:
	    how_many_selected = ta_mark_all(first_taline(current));
	    ps->mangled_body = 1;
	    break;

	  case 'u':  /* unmark all */
	  case PF11:
	    how_many_selected = ta_unmark_all(first_taline(current));
	    ps->mangled_body = 1;
	    break;

	  default:
	    bogus_command(orig_ch, F_ON(F_USE_FK, ps) ? "F1" : "?");

	  case NO_OP_IDLE:			/* simple timeout */
	  case NO_OP_COMMAND:
	    break;
	}
    }

    /* clean up */
    for(current = first_taline(current); current;){
	ctmp = next_taline(current);
	free_taline(&current);
	current = ctmp;
    }

    ps->mangled_screen = 1;
    ClearScreen();
}


/*
 * Call the addrbook functions which add the checked addresses.
 *
 * Args: how_many_selected -- how many addresses are checked
 *                  f_line -- the first ta line
 *
 * Returns: 1 -- we're done, caller should return
 *          0 -- we're not done
 */
int
ta_take_marked_addrs(how_many_selected, f_line)
int how_many_selected;
TA_S *f_line;
{
    TA_S  *ctmp;
    int    done = 0;
    char **new_list;

    if(how_many_selected == 0){
	q_status_message(0, 0, 3,
  "\007No addresses marked for taking. Use ExitTake to leave TakeAddr screen");
    }
    else if(how_many_selected == 1){
	/* set ctmp to the selected address */
	ctmp = first_checked(f_line);
	add_abook_entry(ctmp->addr, ctmp->strvalue, -3);
	done++;
    }
    else{
	new_list = list_of_checked(f_line);
	create_abook_list(new_list, Either, -3);
	free_list_of_checked(&new_list);
	done++;
    }
    return(done);
}


/*
 * Call the addrbook functions to add checked addrs to a list
 *
 * Args: how_many_selected -- how many addresses are checked
 *                  f_line -- the first ta line
 *
 * Returns: 1 -- we're done, caller should return
 *          0 -- we're not done
 */
int
ta_add_to_list(how_many_selected, f_line)
int how_many_selected;
TA_S *f_line;
{
    int    done = 0;
    char **new_list;

    if(how_many_selected == 0){
	q_status_message(0, 0, 3,
  "\007No addresses marked for taking. Use ExitTake to leave TakeAddr screen");
    }
    else{
	new_list = list_of_checked(f_line);
	create_abook_list(new_list, AddToList, -3);
	free_list_of_checked(&new_list);
	done++;
    }
    return(done);
}

/*
 * Mark all of the addresses with a check.
 *
 * Args: f_line -- the first ta line
 *
 * Returns the number of lines checked.
 */
int
ta_mark_all(f_line)
TA_S *f_line;
{
    TA_S *ctmp;
    int   how_many_selected = 0;

    for(ctmp = f_line; ctmp; ctmp = ctmp->next){
	if(!ctmp->is_blank){
	    ctmp->checked = 1;
	    how_many_selected++;
	}
    }
    return(how_many_selected);
}


/*
 * Turn off all of the check marks.
 *
 * Args: f_line -- the first ta line
 *
 * Returns the number of lines checked (0).
 */
int
ta_unmark_all(f_line)
TA_S *f_line;
{
    TA_S *ctmp;

    for(ctmp = f_line; ctmp; ctmp = ctmp->next)
	ctmp->checked = 0;

    return 0;
}


/*
 * Manage display of the Take Address screen.
 *
 * Args:     ps -- pine state
 *      current -- the current TA line
 *       screen -- the TA screen
 */
int
update_takeaddr_screen(ps, current, screen)
struct pine  *ps;
TA_S	     *current;
TA_SCREEN_S  *screen;
{
    int	   dline,
           return_line = HEADER_LINES;
    TA_S  *top_line,
          *ctmp;
    char   box[4];

    /* calculate top line of display */
    dline = 0;
    ctmp = top_line = first_taline(current);
    do
      if(((dline++)%(ps->ttyo->screen_rows - HEADER_LINES - FOOTER_LINES)) == 0)
	  top_line = ctmp;
    while(ctmp != current && (ctmp = next_taline(ctmp)))
	;/* do nothing */

    /* mangled body or new page, force redraw */
    if(ps->mangled_body || screen->top_line != top_line)
        screen->current = NULL;
    
    box[0] = '[';
    box[1] = SPACE;
    box[2] = ']';
    box[3] = '\0';

    /* loop thru painting what's needed */
    for(dline = 0, ctmp = top_line;
	dline < ps->ttyo->screen_rows - FOOTER_LINES - HEADER_LINES;
	dline++, ctmp = next_taline(ctmp)){

	/*
	 * only fall thru painting if something needs painting...
	 */
	if(!(!screen->current || ctmp == screen->current || ctmp == current))
	    continue;

	MoveCursor(dline + HEADER_LINES, 0);
	CleartoEOLN();

	if(ctmp && !ctmp->is_blank && ctmp->strvalue){
	    char *p = tmp_20k_buf;
	    int   i, j;
	    if(ctmp == current){
		return_line = dline + HEADER_LINES;
		StartInverse();
	    }

	    /*
	     * Copy the value to a temp buffer expanding tabs, and
	     * making sure not to write beyond screen right...
	     */
	    for(i = 0, j = 0;
		ctmp->strvalue[i] && j < ps->ttyo->screen_cols;
		i++){
		if(ctmp->strvalue[i] == ctrl('I')){
		    do
		      *p++ = ' ';
		    while(j < ps_global->ttyo->screen_cols && ((++j)&0x07));
			  
		}
		else{
		    *p++ = ctmp->strvalue[i];
		    j++;
		}
	    }

	    *p = '\0';

	    /* mark which lines have check marks */
	    box[1] = ctmp->checked ? 'X' : SPACE;
	    PutLine0(dline + HEADER_LINES, 1, box);
	    PutLine0(dline + HEADER_LINES, 6, tmp_20k_buf);

	    if(ctmp == current)
	        EndInverse();
	}
    }

    ps->mangled_body = 0;
    screen->top_line = top_line;
    screen->current  = current;
    return(return_line);
}


void
takeaddr_screen_redrawer()
{
    bitmap_t	 bitmap;

    ClearScreen();

    set_titlebar("TAKE ADDRESS SCREEN", ps_global->context_current, 
		 ps_global->cur_folder, ps_global->msgmap, 1, FolderName,0,0);

    ps_global->mangled_body = 1;
    (void)update_takeaddr_screen(ps_global, ta_screen->current, ta_screen);

    setbitmap(bitmap);
    draw_keymenu(&takeaddr_keymenu, bitmap,
		 ps_global->ttyo->screen_cols, -2, 0, FirstMenu, 0);
}


/*
 * new_taline - create new TA_S, zero it out, and insert it after current.
 *                NOTE current gets set to the new TA_S, too!
 */
TA_S *
new_taline(current)
    TA_S **current;
{
    TA_S *p;

    p = (TA_S *)fs_get(sizeof(TA_S));
    memset((void *)p, 0, sizeof(TA_S));
    if(current){
	if(*current){
	    p->next	     = (*current)->next;
	    (*current)->next = p;
	    p->prev	     = *current;
	    if(p->next)
	      p->next->prev = p;
	}

	*current = p;
    }

    return(p);
}


void
free_taline(p)
    TA_S **p;
{
    if(p){
	if((*p)->strvalue)
	  fs_give((void **)&(*p)->strvalue);

	if((*p)->prev)
	  (*p)->prev->next = (*p)->next;

	if((*p)->next)
	  (*p)->next->prev = (*p)->prev;

	fs_give((void **)p);
    }
}


/*
 * Return the first TakeAddr line.  There is an assumption (which could be
 * fixed easily if need be) that the first taline is not a blank.
 *
 * Args: p -- any line in the list
 */
TA_S *
first_taline(p)
TA_S *p;
{
    while(p && p->prev)
        p = p->prev;

    return(p);
}


/*
 * Return the prev taline that isn't a blank.
 *
 * Args: p -- a line in the list
 */
TA_S *
prev_nonblank_taline(p)
TA_S *p;
{
    while(p && p->prev && p->prev->is_blank)
        p = p->prev;

    return(p ? p->prev : NULL);
}


/*
 * Return the next taline that isn't a blank.
 *
 * Args: p -- a line in the list
 */
TA_S *
next_nonblank_taline(p)
TA_S *p;
{
    while(p && p->next && p->next->is_blank)
        p = p->next;

    return(p ? p->next : NULL);
}


/*
 * Find the first TakeAddr line which is checked, beginning with the
 * passed in line.
 *
 * Args: head -- A passed in TakeAddr line, usually will be the first
 *
 * Result: returns a pointer to the first checked line.
 *         NULL if no such line
 */
TA_S *
first_checked(head)
TA_S *head;
{
    TA_S *p;

    p = head;

    for(p = head; p; p = p->next){
        if(p->checked)
	    break;
    }

    return(p);
}


/*
 * Form a list of strings which are addresses to go in a list.
 * These are entries in a list, so can be full rfc822 addresses.
 * The strings are allocated here.
 *
 * Args: head -- A passed in TakeAddr line, usually will be the first
 *
 * Result: returns an allocated array of pointers to allocated strings
 */
char **
list_of_checked(head)
TA_S *head;
{
    TA_S  *p;
    int    count;
    char **list, **cur, **q;
    int    add_it;

    /* first count them */
    for(p = head, count = 0; p; p = p->next)
        if(p->checked)
	    count++;
    
    /* allocate pointers */
    list = (char **)fs_get((count + 1) * sizeof(char *));
    memset((void *)list, 0, (count + 1) * sizeof(char *));

    cur = list;

    /* allocate and point to address strings */
    for(p = head; p; p = p->next){
        if(p->checked){
	    /*
	     * Try to throw out dups.  This is only a half-hearted effort.
	     * If there is a dup where one has a fullname and the other
	     * doesn't we certainly won't catch that.  We're just comparing
	     * strings and don't want to parse them.
	     *
	     * Also, we don't check against what's already in the list in
	     * the case where we're adding to an existing list.
	     */
	    add_it = 1;
	    for(q = list; *q; q++){
		if(strucmp(p->strvalue, *q) == 0){
		    add_it = 0;
		    break;
		}
	    }
	    if(add_it){
		*cur = cpystr(p->strvalue);
		cur++;
	    }
	}
    }

    return(list);
}


/*
 * Free the list allocated above.
 *
 * Args: list -- The address of the list that was returned above.
 */
void
free_list_of_checked(list)
char ***list;
{
    char **p;

    for(p = *list; *p; p++)
	fs_give((void **)p);

    fs_give((void **)list);
}
