#if !defined(lint) && !defined(DOS)
static char rcsid[] = "$Id: adrbklib.c,v 4.24 1994/06/21 19:25:12 hubert Exp $";
#endif
/*----------------------------------------------------------------------

            T H E    P I N E    M A I L   S Y S T E M

   Laurence Lundblade and Mike Seibel
   Networks and Distributed Computing
   Computing and Communications
   University of Washington
   Administration Builiding, AG-44
   Seattle, Washington, 98195, USA
   Internet: lgl@CAC.Washington.EDU
             mikes@CAC.Washington.EDU

   Please address all bugs and comments to "pine-bugs@cac.washington.edu"

   Copyright 1989-1994  University of Washington

    Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee to the University of
   Washington is hereby granted, provided that the above copyright notice
   appears in all copies and that both the above copyright notice and this
   permission notice appear in supporting documentation, and that the name
   of the University of Washington not be used in advertising or publicity
   pertaining to distribution of the software without specific, written
   prior permission.  This software is made available "as is", and
   THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
   WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
   WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
   NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
   LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
   (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  
   Pine and Pico are trademarks of the University of Washington.
   No commercial use of these trademarks may be made without prior
   written permission of the University of Washington.

   Pine is in part based on The Elm Mail System:
    ***********************************************************************
    *  The Elm Mail System  -  Revision: 2.13                             *
    *                                                                     *
    * 			Copyright (c) 1986, 1987 Dave Taylor              *
    * 			Copyright (c) 1988, 1989 USENET Community Trust   *
    ***********************************************************************
 

  ----------------------------------------------------------------------*/

#include "headers.h"
#include "adrbklib.h"

#ifdef	DOS
#define	ADRBK_NAME	"addrbook"
#define	TMP_ADRBK_NAME	"addrbook.tmp"
#else
#define	ADRBK_NAME	".addressbook"
#define	TMP_ADRBK_NAME	".addressbook.temp"
#endif


#ifndef MAXPATH
#define MAXPATH 1000    /* Longest file path we can deal with */
#endif
#define MAXLIST 1000    /* Longest address list we can deal with */
#define MAXFULLNAME 256 /* Longest full name we can deal with */


#ifdef ANSI
int   adrbk_write(AdrBk *);
int   cmp_addr(const QSType *, const QSType *);
int   compare_addr_ents(const QSType *, const QSType *);
int   compare_nicks(const QSType *, const QSType *);
void  free_ab_entry(AdrBk *, AdrBk_Entry *);
char *skip_to_next_addr(char *);
char *skip_to_next_entry(char *);
void  sort_addr_list(char **);
#else /* !ANSI */
int   adrbk_write();
int   cmp_addr();
int   compare_nicks();
int   compare_addr_ents();
void  free_ab_entry();
char *skip_to_next_addr();
char *skip_to_next_entry();
void  sort_addr_list();
#endif /* !ANSI */


static char empty[]    = "";
/*
 *  Open, read, and parse an address book.
 *
 * Args: filename -- the filename to open if specified
 *       homedir  -- the user's home directory if specified
 *       warning  -- put a parse error message here
 *
 * The address book is opened, read into memory, closed, and parsed.
 * Returns AdrBk structure or NULL.
 *
 * If filename is NULL, the default will be used in the homedir
 * passed in.  If homedir is NULL, the current dir will be used.
 * If filename is not NULL and is an absolute path, just the filename
 * will be used.  Otherwise, it will be used relative to the homedir, or
 * to the current dir depending on whether or not homedir is NULL.
 *
 * Expected addressbook file format is:
 *  <nickname>\t<fullname>\t<address_field>\t<fcc>\t<comment>
 *
 * The last two fields (\t<fcc>\t<comment>) are optional.
 *
 * Lines that start with SPACE are continuation lines.  Ends of lines are
 * treated as if they were spaces.  The address field is either a single
 * address or a list of comma-separated addresses inside parentheses.
 *
 * Fields missing from the end of an entry are considered blank.
 *
 * Commas in the address field will cause problems, as will tabs in any
 * field.
 *
 * See adrbk_close for comments on storage allocation.
 */
AdrBk *
adrbk_open(filename, homedir, warning)
char *filename,
     *homedir,
     *warning;
{
    register char *p, *q, *next_entry, *cur_entry;
    char           path[MAXPATH], *temp_addr_list[MAXLIST];
    AdrBk         *ab;
    register AdrBk_Entry  **abe, *a, **abe2;
    int            count;
    int            parse_errors = 0;
    char           p_msg[800];


    dprint(7, (debugfile, "- adrbk_open(%s) -\n", filename));

    p_msg[0]           = '\0';
    ab                 = (AdrBk *)fs_get(sizeof(AdrBk));
    ab->book_allocated = 30; /* 30's a nice number. 9 is a measured average */
    ab->book_used      = 0;
    ab->book           = (AdrBk_Entry **)fs_get(ab->book_allocated *
						  sizeof(AdrBk_Entry *));
    ab->book[0]        = NULL;


    /*------------ figure out and save name of file to open ---------*/
    if(filename == NULL){
        if(homedir != NULL){
	    build_path(path, homedir, ADRBK_NAME);
            ab->filename = cpystr(path);
	    build_path(path, homedir, TMP_ADRBK_NAME);
            ab->temp_filename = cpystr(path);
        }
	else{
            ab->filename = cpystr(ADRBK_NAME);
            ab->temp_filename = cpystr(TMP_ADRBK_NAME);
        }
    }
    else{
#ifdef	DOS
        if(*filename == '\\' || (isalpha(*filename) && *(filename+1) == ':')){
#else
        if(*filename == '/'){
#endif
            ab->filename = cpystr(filename);
        }
	else{
            if(homedir != NULL){
		build_path(path, homedir, filename);
                ab->filename = cpystr(path);
            }
	    else
                ab->filename = cpystr(filename);
        }
	q = ".";
#ifdef DOS
	if (p = strrchr(ab->filename, '\\')){
#else
	if (p = strrchr(ab->filename, '/')){
#endif
	  *p = '\0';
	  q = ab->filename;
	}
	ab->temp_filename = temp_nam(q, "adr");
	if(p)
#ifdef DOS
	  *p = '\\';
#else
	  *p = '/';
#endif
    }

    /*------ read the file into one chunk of memory -----*/
    ab->storage = read_file(ab->filename);
    if(ab->storage == NULL){
        /*--- No address book, try creating one ----*/
        FILE *f = fopen(ab->filename, "w");
        if(f == NULL){
            /*--- Create failed, bail out ---*/
	    if(ab->filename)
	        fs_give((void **)&ab->filename);
	    if(ab->temp_filename)
	        fs_give((void **)&ab->temp_filename);
	    if(ab->book)
	        fs_give((void **)&ab->book);
            fs_give((void **)&ab);
            return NULL;
        }
        fclose(f);
        ab->storage     = NULL;
        ab->storage_end = NULL;
        return(ab);
    }
    ab->storage_end = ab->storage + strlen(ab->storage);

    /* skip any leading lines that begin with whitespace */
    for(p = ab->storage; *p && isspace(*p);){
	/* skip to next line and try again */
	for(p++; *p && (*p != '\n' && *p != '\r'); p++)
	    ;/* do nothing */
	while(*p && (*p == '\n' || *p == '\r'))
	    p++;
    }
    if(*p)
	next_entry = p;
    else
	next_entry = NULL;  /* no entries in file */

    /* loop through once per addrbook entry, that is, once per nickname */
    for(cur_entry = next_entry;
        cur_entry;
	cur_entry = next_entry){
	char *addrfield;
	char *addrfield_end;

	/*
	 * This not only skips to the next entry, but it also changes
	 * any continuation line newlines or carriage returns into spaces
	 * and null terminates cur_entry.
	 */
	next_entry = skip_to_next_entry(cur_entry);

	a = adrbk_newentry();
	p = cur_entry;

#define SKIP_SPACE(p) {while(*p && *p == SPACE)p++;}
#define SKIP_TO_TAB(p) {while(*p && *p != TAB)p++;}
#define RM_END_SPACE(start,end) {char *ptr = end; \
			    while(--ptr >= start && *ptr == SPACE)*ptr = '\0';}
	SKIP_SPACE(p);
	a->nickname = p;
	SKIP_TO_TAB(p);
	if(!*p){
	    a->fullname  = empty;
	    a->addr.addr = empty;
	    a->fcc       = empty;
	    a->extra     = empty;
	    a->tag       = Atom;
	}
	else{
	    *p = '\0';
	    RM_END_SPACE(a->nickname, p);
	    p++;
	    SKIP_SPACE(p);
	    a->fullname = p;
	    SKIP_TO_TAB(p);
	    if(!*p){
		a->fcc   = empty;
		a->extra = empty;
		a->tag   = Atom;
	    }
	    else{
		*p = '\0';
		RM_END_SPACE(a->fullname, p);
		p++;
		SKIP_SPACE(p);
		addrfield = p;
		SKIP_TO_TAB(p);
		if(!*p){
		    a->fcc   = empty;
		    a->extra = empty;
		}
		else{
		    *p = '\0';
		    RM_END_SPACE(addrfield, p);
		    p++;
		    SKIP_SPACE(p);
		    a->fcc = p;
		    SKIP_TO_TAB(p);
		    if(!*p){
			a->extra = empty;
		    }
		    else{
			*p = '\0';
			RM_END_SPACE(a->fcc, p);
			p++;
			SKIP_SPACE(p);
			a->extra = p;
			RM_END_SPACE(a->extra, p);
		    }
		}
	    }
	}


	/* parse addrfield */
        if(*addrfield == '('){  /* it's a list */
            char **ad = temp_addr_list;
	    char *next_addr, *cur_addr;

	    p = addrfield;
	    addrfield_end = p + strlen(p);

	    /*
	     * Get rid of the parens.
	     * If this isn't true the input file is messed up.
	     */
	    if(p[strlen(p)-1] == ')'){
		p[strlen(p)-1] = '\0';
		p++;

		/* skip any leading whitespace */
		for(q = p; *q && *q == SPACE; q++)
		    ;/* do nothing */
		next_addr = (*q) ? q : NULL;

		for(cur_addr = next_addr; cur_addr; cur_addr = next_addr){

		    next_addr = skip_to_next_addr(cur_addr);

		    q = cur_addr;
		    SKIP_SPACE(q);

		    *ad++ = q;
		}
		*ad = NULL;
	    }
	    else{
		/* put back what was there to start with */
		*addrfield_end = ')';
		*ad++ = addrfield;
		*ad   = NULL;
		parse_errors++;
		/* just report first error */
		if(!*p_msg)
		    sprintf(p_msg,
			    "nickname %s: %.500s",
			    a->nickname, "addressbook entry is corrupt");
		dprint(1,
		    (debugfile, "parsing error reading addressbook %s: %s %s\n",
			       filename, "missing right paren", addrfield));
	    }

	    a->addr.list = (char **)fs_get((ad - temp_addr_list + 1)
                                           * sizeof(char *));
	    memcpy((void *)a->addr.list, (void *)temp_addr_list,
			   (ad - temp_addr_list + 1) * sizeof(char *));
	    sort_addr_list(a->addr.list);
	    a->tag = List;
        }
	else{  /* A plain, single address */

	    a->addr.addr = addrfield;
	    a->tag       = Atom;
	}

        if(ab->book_used + 1 > ab->book_allocated){
            ab->book_allocated *= 2;
            fs_resize((void **)&(ab->book), ab->book_allocated *
                                           sizeof(AdrBk_Entry *));
        }
        ab->book[ab->book_used++] = a;
	if(ab->book_used % 100 == 0)
	    dprint(9, (debugfile,
		"adrbkopen: have processed %d entries so far\n",
	        ab->book_used));
    }
    dprint(9, (debugfile,
	"adrbkopen: read %d entries for %s\n",
	ab->book_used, ab->filename));

    /*----- sort the entries ----*/
    dprint(9, (debugfile, "adrbkopen: sorting addrbook\n"));
    qsort((QSType *)ab->book,
#ifdef DYN
          (int)ab->book_used, 
#else
          (size_t)ab->book_used, 
#endif
	      sizeof(AdrBk_Entry *), compare_addr_ents);

    {
        /*---- This is faster for large address books ---*/
        AdrBk_Entry **temp_sort_list;
        char         *dupe = "Duplicate";
        
        temp_sort_list = (AdrBk_Entry **)fs_get((ab->book_used + 1) *
                                                     sizeof(AdrBk_Entry *));
        memcpy(temp_sort_list, ab->book, ab->book_used * sizeof(AdrBk_Entry *));
	dprint(9, (debugfile, "adrbkopen: sorting by nickname to de-dupe\n"));
        qsort((QSType *)temp_sort_list,
#ifdef DYN
              (int)ab->book_used,
#else
              (size_t)ab->book_used,
#endif
		  sizeof(AdrBk_Entry *), compare_nicks);
          
        /*----- Hack out duplicates -------*/
        for(abe = temp_sort_list; abe < &temp_sort_list[ab->book_used-1]; abe++)
            if(strucmp((*abe)->nickname, (*(abe+1))->nickname) == 0)
                (*abe)->nickname = dupe;
        count = 0;
        for(abe2 = abe = ab->book; abe < &ab->book[ab->book_used]; abe++)
            if((*abe)->nickname != dupe){
                *abe2++ = *abe;
                count++;
            }
	    else
		free_ab_entry(ab, *abe);
        ab->book_used = count;
	dprint(9, (debugfile, "adrbkopen: after eliminating dups size is %d\n",
	    count));
        fs_give((void **)&temp_sort_list);
    }
                    
    if(parse_errors && warning && *p_msg){
        if(parse_errors == 1)
	    sprintf(warning, "parse error: %s", p_msg);
        else
	    sprintf(warning, "%d parse errors: first: %s", parse_errors, p_msg);
    }

    return(ab);
}


/*
 * Args  cur -- pointer to the start of the current entry.
 *
 * Returns a pointer to the start of the next entry or NULL if there are
 * no more entries.
 *
 * Side effect: current entry has newlines and carriage returns converted
 * to spaces and is null terminated in place.
 */
char *
skip_to_next_entry(cur)
char *cur;
{
    register char *p, *q;

    p = cur;

    while(1){
	/* find end of line */
	while(*p && (*p != '\n' && *p != '\r'))
	    p++;

	/* handle any combination of \r and \n as newlines */
	for(q = p; *q && (*q == '\n' || *q == '\r'); q++)
	    ;/* do nothing */

	/* now q points to start of next line */

	if(!*q){
	    *p = '\0';
	    return NULL;
	}

	/* if continuation line, replace newline chars with spaces */
	if(*q == SPACE || *q == '\t')
	    while(p < q)
		*p++ = SPACE;
	else{
	    *p = '\0';     /* tie off entry */
	    return(q);     /* return next */
	}
    }
}


/*
 * Args  cur -- pointer to the start of the current addr in list.
 *
 * Returns a pointer to the start of the next addr or NULL if there are
 * no more addrs.
 *
 * Side effect: current addr has trailing white space removed
 * and is null terminated.
 */
char *
skip_to_next_addr(cur)
char *cur;
{
    register char *p,
		  *q;
    char *ret_pointer;
    int in_quotes  = 0,
        in_comment = 0;
    char prev_char = '\0';

    /*
     * Find delimiting comma or end.
     * Quoted commas and commented commas don't count.
     */
    for(q = cur; *q; q++){
	switch(*q){
	  case COMMA:
	    if(!in_quotes && !in_comment)
		goto found_comma;
	    break;

	  case LPAREN:
	    if(!in_quotes && !in_comment)
		in_comment = 1;
	    break;

	  case RPAREN:
	    if(in_comment && prev_char != BSLASH)
		in_comment = 0;
	    break;

	  case QUOTE:
	    if(in_quotes && prev_char != BSLASH)
		in_quotes = 0;
	    else if(!in_quotes && !in_comment)
		in_quotes = 1;
	    break;

	  default:
	    break;
	}

	prev_char = *q;
    }
    
found_comma:
    if(*q){  /* trailing comma case */
	*q = '\0';
	ret_pointer = q + 1;
    }
    else
	ret_pointer = NULL;  /* no more addrs after cur */

    /* remove trailing white space from cur */
    for(p = q - 1; p >= cur && isspace(*p); p--)
	*p = '\0';
    
    return(ret_pointer);
}


/*
 * Compare two address book entries for qsort()
 */
int
compare_addr_ents(a, b)
const QSType *a,
	     *b;
{
    AdrBk_Entry **x = (AdrBk_Entry **)a,
                **y = (AdrBk_Entry **)b;
    int result;

    result = strucmp((*x)->fullname, (*y)->fullname);
    if(result == 0)
        result = strucmp((*x)->nickname, (*y)->nickname);
    if(result == 0){
        if((*x)->tag == Atom)
            result = strucmp((*x)->addr.addr, (*y)->addr.addr);
        else
            result = strucmp(*((*x)->addr.list), *((*x)->addr.list));
    }
    if(result == 0 && *x != *y)
        result = *x > *y ? -1 : 1;
      
    return(result);
}


int
compare_nicks(a, b)
const QSType *a,
	     *b;
{
    AdrBk_Entry **x = (AdrBk_Entry **)a,
                **y = (AdrBk_Entry **)b;
    int result;

    result = strucmp((*x)->nickname, (*y)->nickname);
    return(result);
}


/*
 * For sorting a simple list of pointers to strings
 */
int
cmp_addr(a1, a2)
const QSType *a1, *a2;
{
    char *x = *(char **)a1, *y = *(char **)a2;

    return(strucmp(x, y));
}

/*
 * Sort an array of strings
 */
void
sort_addr_list(list)
char **list;
{
    register char **p;

    /* find size of list */
    for(p = list; *p != NULL; p++)
	;/* do nothing */

    qsort((QSType *)list,
#ifdef DYN
          (p - list),
#else          
          (size_t)(p - list),
#endif          
	      sizeof(char *), cmp_addr);
}


/*
 * Return the size of the address book 
 */
unsigned int
adrbk_count(ab)
AdrBk *ab;
{
    return(ab ? ab->book_used : 0);
}


/*
 * Return an indexed entry in the address book. Useful only
 * for sequential access of address book, since indices change when
 * additions or deletions are made.
 *
 * Args: ab  -- the address book
 *       num -- the index number (starting with zero)
 */
AdrBk_Entry *
adrbk_get(ab, num)
AdrBk       *ab;
unsigned int num;
{
    if(!ab || num >= ab->book_used)
        return(NULL);

    return(ab->book[num]);
}


/*
 * Look up an entry in the address book given a nickname
 *
 * Args: ab       -- the address book
 *       nickname -- nickname to match
 *
 * Result: A pointer to an AdrBk_Entry is returned, or NULL if not found.
 *
 * Lookups usually need to be recursive in case the address
 * book references itself.  This is left to the next level up.
 * adrbk_clearrefs() is provided to clear all the reference tags in
 * the address book for loop detetction.
 */
AdrBk_Entry *
adrbk_lookup(ab, nickname)
AdrBk *ab;
char  *nickname;
{
    register AdrBk_Entry **a;
    int                    cmp;

    cmp = 1; /* Not 0 */

    if(!ab)
        return(NULL);

    for(a = ab->book; a < &ab->book[ab->book_used]; a++){
        cmp = strucmp((*a)->nickname, nickname);
        if(cmp == 0)
            break;
    }

    if(cmp == 0)
        return(*a);
    else
        return(NULL);
}


/*
 * Format a full name.
 *
 * Args: fullname -- full name out of address book for formatting
 *
 * Result:  Returns pointer to static internal buffer containing name
 * formatted for a mail header.
 *
 * We need this because we store full names as Last, First.
 * If the name is in quotes, we remove the quotes and leave the part inside
 * the quotes as is.
 * If the name has no comma, then no change is made.
 * Otherwise the text before the first comma is moved to the end and
 * the comma is deleted.
 * We only check the first character for a quote.
 */
char *
adrbk_formatname(fullname)
char *fullname;
{
    char       *comma;
    static char new_name[MAXFULLNAME];

    if(fullname[0] != '"'  && (comma = strindex(fullname, ',')) != NULL){
        int last_name_len = comma - fullname;
        comma++;
        while(*comma && isspace(*comma))
	    comma++;
        strcpy(new_name, comma);
        strcat(new_name, " ");
        strncat(new_name, fullname, last_name_len); 
    }
    /* strip quotes and c-client will put them back */
    else if(fullname[0] == '"' && fullname[strlen(fullname)-1] == '"'){
        strcpy(new_name, fullname);
	new_name[strlen(new_name)-1] = '\0';
	return(new_name+1);
    }
    else
        strcpy(new_name, fullname);

    return(new_name);
}


/*
 * Clear reference flags in preparation for a recursive lookup.
 *
 * For loop detection during address book look up.  This clears all the 
 * referenced flags, then as the lookup proceeds the referenced flags can 
 * be checked and set.
 */
void
adrbk_clearrefs(ab)
AdrBk *ab;
{
    register AdrBk_Entry **a;

    if(!ab)
        return;
    for(a = ab->book; a < &ab->book[ab->book_used]; a++)
        (*a)->referenced = 0;
}


/*
 *  Allocate a new AdrBk_Entry
 */
AdrBk_Entry *
adrbk_newentry()
{
    AdrBk_Entry *a;

    a = (AdrBk_Entry *)fs_get(sizeof(AdrBk_Entry));
    a->nickname = NULL;
    a->fullname = NULL;
    a->addr.addr= NULL;
    a->fcc      = NULL;
    a->extra    = NULL;
    a->tag      = Atom;

    return(a);
}


/*
 * Add an entry to the address book, or modify an existing entry
 *
 * Args: ab       -- address book to add to
 *       new      -- this is a place to return the new AdrBk_Entry
 *       nickname -- the nickname for new entry
 *       fullname -- the fullname for new entry
 *       address  -- the address for new entry
 *       fcc      -- the fcc for new entry
 *       extra    -- the extra field for new entry
 *       tag      -- the type of new entry
 *
 * Result: return code:  0 all went well
 *                      -2 error writing address book, check errno
 *		        -3 no modification, the tag given didn't match
 *                         existing tag
 *                      -4 tabs are in one of the fields passed in
 *
 * If the nickname exists in the address book already, the operation is
 * considered a modification even if the case does not match exactly,
 * otherwise it is an add.  The entry the operation occurs on is returned
 * in new.  All fields are set to those passed in; that is, passing in NULL
 * even on a modification will set those fields to NULL as opposed to leaving
 * them unchanged.  It is acceptable to pass in the current strings
 * in the entry in the case of modification.  For address lists, the
 * structure passed in is what is used, so the storage has to all have
 * come from malloc or fs_get().  If the pointer passed in is the same as
 * the current field, no change is made.
 */
int
adrbk_add(ab, new, nickname, fullname, address, fcc, extra, tag)
AdrBk        *ab;
AdrBk_Entry **new;
char         *nickname,
             *fullname,
	     *address, /* address can be ** too */
             *fcc,
             *extra;
Tag           tag;
{
    register AdrBk_Entry  *l, **a, **b, **c;
    AdrBk_Entry           *foo;
    int                    cmp;

    if(!ab)
        return -2;

    /* ---- Make sure there are no tabs in the stuff to add ------*/
    if((nickname != NULL && strchr(nickname, '\t') != NULL) ||
       (fullname != NULL && strchr(fullname, '\t') != NULL) ||
       (fcc != NULL && strchr(fcc, '\t') != NULL) ||
       (tag == Atom && address != NULL && strchr(address, '\t') != NULL))
        return -4;

    /* Are we adding, or updating ? */
    l = adrbk_lookup(ab, nickname);

    if(l == NULL){  /*----- adding a new entry ----*/

        /*---- Make room for it -----*/
        if(ab->book_used + 1 > ab->book_allocated){
            ab->book_allocated *= 2;
            fs_resize((void **)&(ab->book), ab->book_allocated *
		                                   sizeof(AdrBk_Entry *));
        }
        ab->book_used++;

        /*----- allocate it now so we can use compare_addr_ents() ----*/
        l           = adrbk_newentry();
        l->nickname = cpystr(nickname);  /* has to be a nickname */
        l->fullname = fullname == NULL ? NULL : cpystr(fullname);
        l->fcc      =      fcc == NULL ? NULL : cpystr(fcc);
        l->extra    =    extra == NULL ? NULL : cpystr(extra);
	l->tag      = tag;
	if(tag == Atom)
            l->addr.addr = cpystr(address);
	else
	    l->addr.list = (char **)NULL;

        /*--- Find slot for it ---*/
        foo = l; /* foo is non-register so we can take address of it */
        for(c = ab->book; c < &ab->book[ab->book_used - 1]; c++){
            cmp = compare_addr_ents((QSType *)c, (QSType *)&foo);
            if(cmp >= 0)
              break;
        }

	/* slide things down to make room */
	for(a = &ab->book[ab->book_used - 1]; a > c; a--)
	    *a = *(a-1);

        *c = l;

        if(new != NULL)
            *new = l;
    }
    else{
        /*----- Updating an existing entry ----*/

	if(l->tag != tag)
	    return -3;


        /*
	 * See adrbk_close for explanation of storage management.
	 * Instead of just freeing and reallocating here we attempt to reuse
	 * the space that was already allocated if possible.
	 */
        if(l->nickname != nickname){
            if(l->nickname == NULL || strlen(nickname) > strlen(l->nickname)){
                if(l->nickname > ab->storage_end || l->nickname < ab->storage)
                  fs_give((void **)&l->nickname);
                l->nickname = cpystr(nickname);
            }
	    else
                strcpy(l->nickname, nickname);
        }

        if(fullname == NULL)
            l->fullname = NULL;
	else if(l->fullname != fullname){
            if(l->fullname == NULL || strlen(fullname) > strlen(l->fullname)){
                if(l->fullname > ab->storage_end || l->fullname < ab->storage)
                  fs_give((void **)&l->fullname);
                l->fullname = cpystr(fullname);
            }
	    else
                strcpy(l->fullname, fullname);
        }

        if(fcc == NULL)
            l->fcc = NULL;
	else if(l->fcc != fcc){
            if(l->fcc == NULL || strlen(fcc) > strlen(l->fcc)){
                if(l->fcc > ab->storage_end || l->fcc < ab->storage)
                  fs_give((void **)&l->fcc);
                l->fcc = cpystr(fcc);
            }
	    else
                strcpy(l->fcc, fcc);
        }

        if(extra == NULL)
            l->extra = NULL;
	else if(l->extra != extra){
            if(l->extra == NULL || strlen(extra) > strlen(l->extra)){
                if(l->extra > ab->storage_end || l->extra < ab->storage)
                  fs_give((void **)&l->extra);
                l->extra = cpystr(extra);
            }
	    else
                strcpy(l->extra, extra);
        }

	if(tag == Atom){
            /*---- Atom ----*/
	    if(address == NULL)
		l->addr.addr = NULL;
	    else if(l->addr.addr != address){
		if(l->addr.addr==NULL || strlen(address)>strlen(l->addr.addr)){
		    if(l->addr.addr>ab->storage_end|| l->addr.addr<ab->storage)
		      fs_give((void **)&l->addr.addr);
		    l->addr.addr = cpystr(address);
		}
		else
		    strcpy(l->addr.addr, address);
	    }
	}
	else{
            /*---- List -----*/
            /* we don't mess with lists here */
	}


        /*---------- Make sure it's still in order ---------*/
	for(a = ab->book; a < &ab->book[ab->book_used]; a++)
            if(*a == l) /* find current spot in list */
                break; 

        if(a!=ab->book && compare_addr_ents((QSType *)a,(QSType *)(a-1)) < 0){
            /*--- Out of order, needs to be moved up ----*/
            for(b = a - 1; b >= ab->book; b--)
              if(compare_addr_ents((QSType *)a, (QSType *)b) >= 0)
                break; /* find new position */
            for(c = a; c > b; c--) /* Shuffle things down */
                *c = *(c-1);
            *(c+1) = l;  /* put in it's new place */
        }

        if(a != &ab->book[ab->book_used-1] &&
            compare_addr_ents((QSType *)a, (QSType *)(a + 1)) > 0){
            /*---- Out of order needs, to be moved towards end of list ----*/
            for(b = a + 1; b < &ab->book[ab->book_used]; b++)
              if(compare_addr_ents((QSType *)a, (QSType *)b) <= 0)
                break; /* find new position */
            for(c = a; c+1 < b; c++) /* Slide entries up */
                *c = *(c+1);
            *c = l;                  /* put it in it's new place */
        }


        /*---- return in pointer if requested -----*/
        if(new)
            *new = l;
    }

    return(adrbk_write(ab));
}


/*
 * Delete an entry from the address book
 *
 * Args: ab     -- the address book
 *       to_del -- a pointer to the entry to delete
 *
 * Result: returns:  0 if all went well
 *                  -1 if there is no such entry
 *                  -2 error writing address book, check errno
 *
 * See adrbk_close of explanation of storage management
 */
int
adrbk_delete(ab, to_del)
AdrBk       *ab;
AdrBk_Entry *to_del;
{
    register AdrBk_Entry **a;

    if(!ab)
        return -2;

    /* find this entry */
    for(a = ab->book; a < &ab->book[ab->book_used]; a++) {
        if(*a == to_del)
          break;
    }

    if(a == &ab->book[ab->book_used])
        return -1;   /* entry not found, delete failed */

    /* slide everything up */
    for(; a < &ab->book[ab->book_used - 1]; a++)
	*a = *(a+1);

    ab->book_used--;

    free_ab_entry(ab, to_del);

    return(adrbk_write(ab)); 
}


/*
 * Delete an address out of an address list
 *
 * Args: ab    -- the address book
 *       entry -- the address list we are deleting from
 *       addr  -- address in above list to be deleted
 *
 * Result: 0: Deletion complete, address book written
 *        -1: Address for deletion not found
 *        -2: Error writing address book. Check errno.
 *
 * The address to be deleted is located by matching the string.
 */
int
adrbk_listdel(ab, entry, addr)
AdrBk       *ab;
AdrBk_Entry *entry;
char        *addr;
{
    char **p, *to_free;

    if(!ab)
        return -2;

    for(p = entry->addr.list; *p; p++) 
        if(strcmp(*p, addr) == 0)
            break;

    if(*p == NULL)
        return -1;

    /* free storage */
    to_free = NULL;
    if((*p > ab->storage_end || *p < ab->storage) && *p != empty)
        to_free = *p; /* freeing it now messes up next loop */

    /* slide all the entries below up (including NULL) */
    for(; *p; p++)
        *p = *(p+1);

    if(to_free)
        fs_give((void **)&to_free);

    return(adrbk_write(ab));
}


/*
 * Add one address to an already existing address list
 *
 * Args: ab       -- the address book
 *       entry    -- the address list we are adding to
 *       addr     -- address to be added
 *       new_addr -- return the new address here if non-NULL
 *
 * Result: returns 0 : addition made, address book written
 *                -1 : addition to non-list attempted
 *                -2 : error writing address book -- check errno
 */
int
adrbk_listadd(ab, entry, addr, new_addr)
AdrBk       *ab;
AdrBk_Entry *entry;
char        *addr,
           **new_addr;
{
    char **p;
    int    n;

    if(!ab)
        return -2;

    if(entry->tag != List)
        return -1;

    /*--- count up size of list ------*/    
    for(p = entry->addr.list; p != NULL && *p != NULL; p++)
	;/* do nothing */

    /*----- make room at end of list for it ------*/
    if(entry->addr.list == NULL){
        entry->addr.list = (char **)fs_get(2 * sizeof(char *));
        n = 1;
    }
    else{
        n = p - entry->addr.list;
        n++;
	/* n is size of list, +1 for NULL */
        fs_resize((void **)&entry->addr.list, sizeof(char *) * (n + 1));
    }

    /*----- Put it at the end -------*/
    (entry->addr.list)[n-1] = cpystr(addr);
    (entry->addr.list)[n]   = NULL;

    if(new_addr != NULL)
        *new_addr =(entry->addr.list)[n-1];

    /*---- sort it into the correct place ------*/
    sort_addr_list(entry->addr.list);

    return(adrbk_write(ab));
}


/*
 * Close address book
 *
 * All that is done here is to free the storage, since the address book is 
 * rewritten on every change.
 *
 * Storage management:
 *
 * Initially all the strings in the address book come out of the 
 * large allocated buffer used to read the whole file in.
 * When entries are added or deleted it's not possible
 * to add to this pool, so the strings are allocated individually.
 * When it comes time to free strings, any pointers into the big buffer
 * are left alone and get freed with the big buffer.  Other pointers
 * are deallocated individually.
 *
 * There are some cases where we aren't getting the fs_give feature of
 * setting the pointer to NULL, because we aren't passing in the address
 * of the real pointer.  For example, here we pass in a pointer to an ab.
 * Fs_give(&ab) is going to set the local ab to NULL, not the real one.
 * The real one should be set to NULL in the caller (if necessary).
 */
void
adrbk_close(ab)
AdrBk *ab;
{
    register AdrBk_Entry  **a;
    
    if(!ab)
        return;

    for(a = ab->book; a < &ab->book[ab->book_used]; a++)
	free_ab_entry(ab, *a);

    fs_give((void **)&ab->storage);
      
    fs_give((void **)&ab->book);

    fs_give((void**)&ab->filename);

    fs_give((void**)&ab->temp_filename);

    fs_give((void **)&ab);
}


/*
 * Write out the address book.
 *
 * Format is as in comment in the adrbk_open routine.  Lines are wrapped
 * to be under 80 characters.  This is called on every change to the
 * address book.  SIGINT and SIGHUP are also ignored during writing so the
 * address book won't be messed up.  Write is first to a temporary file,
 * which is then renamed to be the real address book so that we won't
 * destroy the real address book in case of something like a full file
 * system.
 *
 * Writing a temp file and then renaming has the bad side affect of
 * destroying links.  It also overrides any read only permissions on
 * the mail file since rename ignores such permissions.  However, we
 * handle read-only-ness in addrbook.c before we call this.
 * We retain the permissions by doing a stat on the old file and a
 * chmod on the new one.
 *
 * Returns:   0 write was successful
 *           -2 write failed
 */
int
adrbk_write(ab)
AdrBk *ab;
{
    register FILE         *ab_stream;
    register AdrBk_Entry **a;
    register int           len;
    struct stat            sbuf;
#ifdef	DOS
    void                  (*save_sigint)(int);
#else
    void                  (*save_sigint)(), (*save_sighup)();
#endif

    if(!ab)
        return -2;

#define TABWIDTH 8
#define INDENTSTR "   "
#define INDENT 3  /* length of INDENTSTR */

#ifdef	DOS
    save_sigint = signal(SIGINT, SIG_IGN);
#else
    save_sigint = (void (*)())signal(SIGINT, SIG_IGN);
    save_sighup = (void (*)())signal(SIGHUP, SIG_IGN);
#endif

    if((ab_stream = fopen(ab->temp_filename, "w")) == NULL){
        (void)signal(SIGINT, save_sigint);
#ifndef	DOS
        (void)signal(SIGHUP, save_sighup);
#endif
        return -2;
    }

    /*
     * Continuation lines always start with spaces.  Tabs are treated as
     * separators, never as whitespace.  When we output tab separators we
     * always put them on the ends of lines, never on the start of a line
     * after a continuation.  That is, there is always something printable
     * after continuation spaces.
     */
    for(a = ab->book; a < &ab->book[ab->book_used]; a++){
	int tmplen;

        if(fputs((*a)->nickname, ab_stream) == EOF)
            goto io_error;
        tmplen = strlen((*a)->nickname);
        putc('\t', ab_stream);
        len = tmplen + (TABWIDTH - tmplen % TABWIDTH);
        if((*a)->fullname){
            tmplen = strlen((*a)->fullname);
	    len += (tmplen + (TABWIDTH - tmplen % TABWIDTH));
	    if(len > 80){
		if(fprintf(ab_stream, "\n%s", INDENTSTR) == EOF)
		    goto io_error;
		tmplen += INDENT;
		len = tmplen + (TABWIDTH - tmplen % TABWIDTH);
	    }
            if(fputs((*a)->fullname, ab_stream) == EOF)
                goto io_error;
        }
	else
	    len += TABWIDTH;
        putc('\t', ab_stream);
        if((*a)->addr.addr != NULL || (*a)->addr.list != NULL){
	    if((*a)->tag == Atom){
	        /*----- Atom: just one address ----*/
		tmplen = strlen((*a)->addr.addr);
		len += (tmplen + (TABWIDTH - tmplen % TABWIDTH));
		if(len > 80){
		    if(fprintf(ab_stream, "\n%s", INDENTSTR) == EOF)
			goto io_error;
		    tmplen += INDENT;
		    len = tmplen + (TABWIDTH - tmplen % TABWIDTH);
		}
	      	if(fputs((*a)->addr.addr, ab_stream) == EOF)
		    goto io_error;
	    }
	    else{
		register char **a2;

	        /*----- List: a distribution list ------*/
		putc('(', ab_stream);
		len++;
		for(a2 = (*a)->addr.list; *a2 != NULL; a2++){
		    if(a2 != (*a)->addr.list){
		        putc(',', ab_stream);
			len++;
		    }
		    /*
		     * comma or ) also follows, so we're breaking at
		     * no more than 79 chars
		     */
		    if(len + strlen(*a2) > 78 && len != INDENT){
		        /*--- break up long lines ----*/
		        if(fprintf(ab_stream, "\n%s", INDENTSTR) == EOF)
			    goto io_error;
		        len = INDENT;
		      }
		    if(fputs(*a2, ab_stream) == EOF)
		        goto io_error;
		    len += strlen(*a2);
		}
		putc(')', ab_stream);
	    }
        }

	/* If either fcc or extra exists, output both, otherwise, neither */
        if(((*a)->fcc && (*a)->fcc[0]) || ((*a)->extra && (*a)->extra[0])){
            putc('\t', ab_stream);
	    len += (TABWIDTH - len % TABWIDTH);
	    if((*a)->fcc && (*a)->fcc[0]){
		tmplen = strlen((*a)->fcc);
		len += (tmplen + (TABWIDTH - tmplen % TABWIDTH));
		if(len > 80){
		    if(fprintf(ab_stream, "\n%s", INDENTSTR) == EOF)
			goto io_error;
		    tmplen += INDENT;
		    len = tmplen + (TABWIDTH - tmplen % TABWIDTH);
		}
		if(fputs((*a)->fcc, ab_stream) == EOF)
		    goto io_error;
	    }
	    putc('\t', ab_stream);
	    if((*a)->extra && (*a)->extra[0]){
		tmplen = strlen((*a)->extra);
		len += (tmplen + (TABWIDTH - tmplen % TABWIDTH));
		if(len > 80){
		    if(fprintf(ab_stream, "\n%s", INDENTSTR) == EOF)
			goto io_error;
		    tmplen += INDENT;
		    len = tmplen + (TABWIDTH - tmplen % TABWIDTH);
		}
		if(fputs((*a)->extra, ab_stream) == EOF)
		    goto io_error;
	    }
        }
        putc('\n', ab_stream);
    }

    if(fclose(ab_stream) == EOF)
        goto io_error;

    /* if stat failed, don't abort, just don't worry about mode */
    if(stat(ab->filename, &sbuf) == 0)
	(void)chmod(ab->temp_filename, sbuf.st_mode);

    if(rename(ab->temp_filename, ab->filename) < 0)
        goto io_error;

    (void)signal(SIGINT, save_sigint);
#ifndef	DOS
    (void)signal(SIGHUP, save_sighup);
#endif
    return 0;


io_error:
    (void)signal(SIGINT, save_sigint);
#ifndef	DOS
    (void)signal(SIGHUP, save_sighup);
#endif
    return -2;
}


/*
 * Free memory associated with entry abe that was allocated outside of
 * the initial data read in.  See adrbk_close for an explanation.
 *
 * Args:  ab  -- Address book
 *        abe -- Address book entry to be freed.
 */
void
free_ab_entry(ab, abe)
AdrBk       *ab;
AdrBk_Entry *abe;
{
    char **p;

    if(!ab || !abe)
        return;

    if(abe->nickname && abe->nickname != empty &&
	(abe->nickname > ab->storage_end || abe->nickname < ab->storage))
        fs_give((void **)&abe->nickname);

    if(abe->fullname && abe->fullname != empty &&
	(abe->fullname > ab->storage_end || abe->fullname < ab->storage))
        fs_give((void **)&abe->fullname);

    if(abe->tag == Atom ){
        if(abe->addr.addr && abe->addr.addr != empty &&
	    (abe->addr.addr > ab->storage_end || abe->addr.addr < ab->storage))
            fs_give((void **)&abe->addr.addr);
    }
    else{
        if(abe->addr.list){
            for(p = abe->addr.list; *p; p++) 
                if(*p != empty && (*p > ab->storage_end || *p < ab->storage))
		    fs_give((void **)p);
            fs_give((void **)&abe->addr.list);
        }
    }

    if(abe->fcc && abe->fcc != empty &&
	(abe->fcc > ab->storage_end || abe->fcc < ab->storage))
        fs_give((void **)&abe->fcc);

    if(abe->extra && abe->extra != empty &&
	(abe->extra > ab->storage_end || abe->extra < ab->storage))
        fs_give((void **)&abe->extra);

    fs_give((void **)&abe);
}
