/* code to handle constellation boundaries and figures */

/*
Constellation boundaries:

 Primary reference:

  METHOD TO DETERMINE THE CONSTELLATION IN WHICH A POSITION IS LOCATED

        Recently, Mr. Barry N. Rappaport of New  Mexico State University
  transcribed  the constellation  boundaries as  fixed  by the IAU  into
  machine-readable form.  These have  been  transcribed  by Dr. Nancy G.
  Roman to make it possible  to determine by  computer the constellation
  in which a position is located.

 NSSDC catalog description:
 6042   AN     Catalog of Constellation Boundary Data (Delporte, E. 1930, 
               Cambridge Univ. Press)
               Comment(s): includes constellation identification software 
               (ADC 1987; see Roman, N.G. 1987, Publ. Astron. Soc. Pacific 
               99, 695); 23 description, 118 software, 358 data records. 
               3 files: 23x80, 118x80, 358x29 

 Further adapted for xephem by:

    Craig Counterman: conversion from original F77 to C
    Elwood Downey:    incorporation into xephem
    Ernie Wright:     additional speed and time improvments and new interface

Constellation figures:

    Chris Marriott:   original figures
    Elwood Downey:    incorporated into xephem
*/

#include <stdio.h>
#include <math.h>
#include "astro.h"

#if defined(__STDC__) || defined(__cplusplus)
#define P_(s) s
#else
#define P_(s) ()
#endif

extern void cal_mjd P_((int mn, double dy, int yr, double *Mjd));
extern void precess P_((double mjd1, double mjd2, double *ra, double *dec));
extern void range P_((double *v, double r));

char *cns_name P_((int id));
int cns_id P_((char *abbrev));
int cns_pick P_((double ra, double dec, double e));
int cns_list P_((double ra, double dec, double e, double rad, int ids[]));
#ifdef CONOUTLINE
int cns_outline P_((int id, double e, double ra[], double dec[], int path[]));
#endif /* CONOUTLINE */
int cns_figure P_((int id, double e, double ra[], double dec[], int dcodes[]));

#undef P_

/*
======================================================================
Ernie Wright  2 Mar 94

Find the constellation for a given position.

First C version by Craig Counterman and Elwood Downey.  Based on a
FORTRAN program by Nancy G. Roman (Roman, N.G. 1987, Publ. Astron.
Soc. Pacific 99, 695).  IAU constellation boundaries transcribed into
machine-readable form by Barry N. Rappaport, New Mexico State Univ.
======================================================================
*/

#define NNAMES  88
#define NBOUNDS 357

/* constellation ids */
#define And  0
#define Ant  1
#define Aps  2
#define Aql  3
#define Aqr  4
#define Ara  5
#define Ari  6
#define Aur  7
#define Boo  8
#define CMa  9
#define CMi 10
#define CVn 11
#define Cae 12
#define Cam 13
#define Cap 14
#define Car 15
#define Cas 16
#define Cen 17
#define Cep 18
#define Cet 19
#define Cha 20
#define Cir 21
#define Cnc 22
#define Col 23
#define Com 24
#define CrA 25
#define CrB 26
#define Crt 27
#define Cru 28
#define Crv 29
#define Cyg 30
#define Del 31
#define Dor 32
#define Dra 33
#define Equ 34
#define Eri 35
#define For 36
#define Gem 37
#define Gru 38
#define Her 39
#define Hor 40
#define Hya 41
#define Hyi 42
#define Ind 43
#define LMi 44
#define Lac 45
#define Leo 46
#define Lep 47
#define Lib 48
#define Lup 49
#define Lyn 50
#define Lyr 51
#define Men 52
#define Mic 53
#define Mon 54
#define Mus 55
#define Nor 56
#define Oct 57
#define Oph 58
#define Ori 59
#define Pav 60
#define Peg 61
#define Per 62
#define Phe 63
#define Pic 64
#define PsA 65
#define Psc 66
#define Pup 67
#define Pyx 68
#define Ret 69
#define Scl 70
#define Sco 71
#define Sct 72
#define Ser 73
#define Sex 74
#define Sge 75
#define Sgr 76
#define Tau 77
#define Tel 78
#define TrA 79
#define Tri 80
#define Tuc 81
#define UMa 82
#define UMi 83
#define Vel 84
#define Vir 85
#define Vol 86
#define Vul 87

static char *cns_namemap[ NNAMES ] = {
   /*  0 */   "And: Andromeda",
   /*  1 */   "Ant: Antlia",
   /*  2 */   "Aps: Apus",
   /*  3 */   "Aql: Aquila",
   /*  4 */   "Aqr: Aquarius",
   /*  5 */   "Ara: Ara",
   /*  6 */   "Ari: Aries",
   /*  7 */   "Aur: Auriga",
   /*  8 */   "Boo: Bootes",
   /*  9 */   "CMa: Canis Major",
   /* 10 */   "CMi: Canis Minor",
   /* 11 */   "CVn: Canes Venatici",
   /* 12 */   "Cae: Caelum",
   /* 13 */   "Cam: Camelopardalis",
   /* 14 */   "Cap: Capricornus",
   /* 15 */   "Car: Carina",
   /* 16 */   "Cas: Cassiopeia",
   /* 17 */   "Cen: Centaurus",
   /* 18 */   "Cep: Cepheus",
   /* 19 */   "Cet: Cetus",
   /* 20 */   "Cha: Chamaeleon",
   /* 21 */   "Cir: Circinus",
   /* 22 */   "Cnc: Cancer",
   /* 23 */   "Col: Columba",
   /* 24 */   "Com: Coma Berenices",
   /* 25 */   "CrA: Corona Australis",
   /* 26 */   "CrB: Corona Borealis",
   /* 27 */   "Crt: Crater",
   /* 28 */   "Cru: Crux",
   /* 29 */   "Crv: Corvus",
   /* 30 */   "Cyg: Cygnus",
   /* 31 */   "Del: Delphinus",
   /* 32 */   "Dor: Dorado",
   /* 33 */   "Dra: Draco",
   /* 34 */   "Equ: Equuleus",
   /* 35 */   "Eri: Eridanus",
   /* 36 */   "For: Fornax",
   /* 37 */   "Gem: Gemini",
   /* 38 */   "Gru: Grus",
   /* 39 */   "Her: Hercules",
   /* 40 */   "Hor: Horologium",
   /* 41 */   "Hya: Hydra",
   /* 42 */   "Hyi: Hydrus",
   /* 43 */   "Ind: Indus",
   /* 44 */   "LMi: Leo Minor",
   /* 45 */   "Lac: Lacerta",
   /* 46 */   "Leo: Leo",
   /* 47 */   "Lep: Lepus",
   /* 48 */   "Lib: Libra",
   /* 49 */   "Lup: Lupus",
   /* 50 */   "Lyn: Lynx",
   /* 51 */   "Lyr: Lyra",
   /* 52 */   "Men: Mensa",
   /* 53 */   "Mic: Microscopium",
   /* 54 */   "Mon: Monoceros",
   /* 55 */   "Mus: Musca",
   /* 56 */   "Nor: Norma",
   /* 57 */   "Oct: Octans",
   /* 58 */   "Oph: Ophiuchus",
   /* 59 */   "Ori: Orion",
   /* 60 */   "Pav: Pavo",
   /* 61 */   "Peg: Pegasus",
   /* 62 */   "Per: Perseus",
   /* 63 */   "Phe: Phoenix",
   /* 64 */   "Pic: Pictor",
   /* 65 */   "PsA: Piscis Austrinus",
   /* 66 */   "Psc: Pisces",
   /* 67 */   "Pup: Puppis",
   /* 68 */   "Pyx: Pyxis",
   /* 69 */   "Ret: Reticulum",
   /* 70 */   "Scl: Sculptor",
   /* 71 */   "Sco: Scorpius",
   /* 72 */   "Sct: Scutum",
   /* 73 */   "Ser: Serpens",
   /* 74 */   "Sex: Sextans",
   /* 75 */   "Sge: Sagitta",
   /* 76 */   "Sgr: Sagittarius",
   /* 77 */   "Tau: Taurus",
   /* 78 */   "Tel: Telescopium",
   /* 79 */   "TrA: Triangulum Australe",
   /* 80 */   "Tri: Triangulum",
   /* 81 */   "Tuc: Tucana",
   /* 82 */   "UMa: Ursa Major",
   /* 83 */   "UMi: Ursa Minor",
   /* 84 */   "Vel: Vela",
   /* 85 */   "Vir: Virgo",
   /* 86 */   "Vol: Volans",
   /* 87 */   "Vul: Vulpecula"
};

static struct {
   unsigned short lower_ra;      /* hours * 1800 */
   unsigned short upper_ra;      /* hours * 1800 */
   short          lower_dec;     /* degrees * 60 */
   short          index;
} cbound[ NBOUNDS ] = {
   {     0, 43200,  5280, UMi },
   { 14400, 26100,  5190, UMi },
   { 37800, 41400,  5170, UMi },
   { 32400, 37800,  5160, UMi },
   {     0, 14400,  5100, Cep },
   { 16500, 19200,  4920, Cam },
   {     0,  9000,  4800, Cep },
   { 19200, 26100,  4800, Cam },
   { 31500, 32400,  4800, UMi },
   { 36300, 37800,  4800, Dra },
   {     0,  6315,  4620, Cep },
   { 20700, 24450,  4620, Cam },
   { 29760, 31500,  4500, UMi },
   { 36300, 37200,  4500, Cep },
   { 14340, 16500,  4410, Cam },
   { 16500, 20400,  4410, Dra },
   { 23400, 29760,  4200, UMi },
   {  5580,  6150,  4080, Cas },
   { 36750, 37200,  4020, Dra },
   { 20400, 21600,  3990, Dra },
   {     0,   600,  3960, Cep },
   { 25200, 28200,  3960, UMi },
   { 42450, 43200,  3960, Cep },
   { 21600, 24300,  3840, Dra },
   { 24300, 25950,  3780, Dra },
   { 41700, 42450,  3780, Cep },
   { 10980, 12600,  3720, Cam },
   { 36000, 36750,  3690, Dra },
   { 36966, 37080,  3655, Cep },
   { 12600, 14340,  3600, Cam },
   { 14340, 15150,  3600, UMa },
   { 35580, 36000,  3570, Dra },
   { 36000, 36966,  3570, Cep },
   { 41160, 41700,  3545, Cep },
   {     0,  4380,  3510, Cas },
   { 34950, 35580,  3480, Dra },
   {  3060,  3435,  3450, Cas },
   {  4380,  5580,  3420, Cas },
   {  5580,  5700,  3420, Cam },
   { 40170, 41160,  3375, Cep },
   {  9000, 10980,  3360, Cam },
   { 25260, 25950,  3330, UMa },
   { 25950, 34950,  3330, Dra },
   {  5700,  6000,  3300, Cam },
   { 39840, 40170,  3300, Cep },
   { 37080, 39540,  3290, Cep },
   {     0,  3060,  3240, Cas },
   { 10980, 11700,  3240, Lyn },
   { 21750, 24300,  3180, UMa },
   { 27450, 28350,  3180, Dra },
   { 39540, 39840,  3165, Cep },
   {  6000,  9000,  3150, Cam },
   { 41160, 42000,  3150, Cas },
   { 28350, 30600,  3090, Dra },
   {  3675,  4530,  3030, Per },
   { 30600, 32820,  3030, Dra },
   {     0,  2460,  3000, Cas },
   {  2460,  3000,  3000, Per },
   { 11700, 12240,  3000, Lyn },
   { 42000, 43200,  3000, Cas },
   { 24300, 25260,  2910, UMa },
   {     0,  2010,  2880, Cas },
   { 42450, 43200,  2880, Cas },
   { 32715, 32820,  2850, Her },
   { 32820, 34350,  2850, Dra },
   { 34350, 34500,  2850, Cyg },
   {  3000,  3675,  2820, Per },
   { 15150, 16500,  2820, UMa },
   {   300,  1560,  2760, Cas },
   { 21600, 21750,  2700, UMa },
   { 12240, 13260,  2670, Lyn },
   { 39435, 39540,  2640, Cyg },
   { 39375, 39435,  2625, Cyg },
   { 34500, 34920,  2610, Cyg },
   { 16500, 18300,  2520, UMa },
   { 18300, 19410,  2400, UMa },
   { 27780, 28350,  2400, Boo },
   { 28350, 29400,  2400, Her },
   { 16650, 17250,  2385, Lyn },
   {     0,  4530,  2205, And },
   {  4530,  4620,  2205, Per },
   { 34845, 34920,  2190, Lyr },
   {  8100,  8445,  2160, Per },
   { 39120, 39375,  2160, Cyg },
   { 39375, 39600,  2160, Lac },
   { 11760, 13260,  2130, Aur },
   { 13260, 13950,  2130, Lyn },
   {     0,  3600,  2100, And },
   { 39600, 41070,  2100, Lac },
   { 41070, 41160,  2070, Lac },
   { 41160, 42300,  2070, And },
   {  4620,  4890,  2040, Per },
   { 19410, 19800,  2040, UMa },
   { 21600, 22200,  2040, CVn },
   { 13950, 16650,  2010, Lyn },
   { 16650, 17790,  2010, LMi },
   {  1290,  2535,  1980, And },
   { 27330, 27780,  1980, Boo },
   { 42300, 42750,  1925, And },
   { 22200, 23850,  1920, CVn },
   { 42750, 43200,  1880, And },
   { 25125, 25260,  1845, CVn },
   {  4350,  4890,  1840, Tri },
   {  4890,  8100,  1840, Per },
   {  8100,  8550,  1800, Aur },
   { 32715, 34845,  1800, Lyr },
   { 19800, 21600,  1740, UMa },
   { 35400, 37650,  1740, Cyg },
   {  8550, 10590,  1710, Aur },
   { 17790, 18900,  1710, LMi },
   { 23850, 25125,  1710, CVn },
   {     0,   120,  1680, And },
   {  2535,  3000,  1680, Tri },
   { 10590, 11760,  1680, Aur },
   { 14190, 14400,  1680, Gem },
   { 37650, 39120,  1680, Cyg },
   { 34665, 35400,  1650, Cyg },
   {  3450,  4350,  1635, Tri },
   { 29100, 29400,  1620, CrB },
   { 27150, 27330,  1560, Boo },
   { 27330, 29100,  1560, CrB },
   { 33060, 33960,  1560, Lyr },
   { 19350, 19800,  1530, LMi },
   { 33960, 34665,  1530, Lyr },
   {  3000,  3450,  1500, Tri },
   {  1290,  1530,  1425, Psc },
   { 18900, 19350,  1410, LMi },
   { 38250, 38550,  1410, Vul },
   { 10260, 10590,  1370, Tau },
   {   120,   255,  1320, And },
   { 28650, 28860,  1320, Ser },
   { 10590, 11190,  1290, Gem },
   { 35700, 36450,  1275, Vul },
   { 33960, 34650,  1265, Vul },
   {   255,  1530,  1260, And },
   { 36450, 37020,  1230, Vul },
   { 14055, 14190,  1200, Gem },
   { 37020, 38250,  1170, Vul },
   { 34650, 35700,  1150, Vul },
   {  5910,  6060,  1140, Ari },
   { 33960, 34200,  1110, Sge },
   { 10260, 10380,  1080, Ori },
   { 11190, 11355,  1050, Gem },
   { 34200, 35700,   970, Sge },
   {  8940,  9600,   960, Tau },
   { 28650, 28950,   960, Her },
   { 35700, 36450,   945, Sge },
   {  8310,  8940,   930, Tau },
   {  9600, 10080,   930, Tau },
   { 23100, 24300,   900, Com },
   { 31050, 32850,   860, Her },
   { 21360, 23100,   840, Com },
   { 13500, 14055,   810, Gem },
   { 30150, 31050,   770, Her },
   {     0,   255,   750, Peg },
   { 10080, 10380,   750, Tau },
   { 12600, 13500,   750, Gem },
   { 38010, 38400,   750, Peg },
   { 11355, 12480,   720, Gem },
   { 32850, 33960,   720, Her },
   { 37575, 37890,   710, Del },
   { 37890, 38010,   710, Peg },
   { 20730, 21360,   660, Leo },
   { 11235, 11355,   600, Ori },
   { 12480, 12600,   600, Gem },
   { 14055, 14265,   600, Cnc },
   { 42900, 43200,   600, Peg },
   {  3000,  5910,   595, Ari },
   { 36255, 36540,   510, Del },
   { 24300, 27150,   480, Boo },
   { 40950, 42900,   450, Peg },
   { 14265, 16650,   420, Cnc },
   { 16650, 19350,   420, Leo },
   { 32850, 33592,   375, Oph },
   { 33592, 33960,   375, Aql },
   { 37500, 37575,   360, Del },
   { 12600, 12630,   330, CMi },
   { 32850, 33165,   270, Ser },
   { 28950, 30150,   240, Her },
   { 32850, 33165,   180, Oph },
   { 38640, 39000,   165, Peg },
   {     0,  3600,   120, Psc },
   { 33450, 33960,   120, Ser },
   { 36540, 37500,   120, Del },
   { 37500, 38400,   120, Equ },
   { 38400, 38640,   120, Peg },
   { 39600, 40950,   120, Peg },
   { 39000, 39600,   105, Peg },
   { 12630, 12960,    90, CMi },
   {  6450,  8310,     0, Tau },
   {  8310,  8400,     0, Ori },
   { 12960, 14550,     0, CMi },
   { 26400, 27150,     0, Vir },
   { 32100, 32850,     0, Oph },
   {  4770,  5910,  -105, Cet },
   {  5910,  6450,  -105, Tau },
   { 27150, 29280,  -195, Ser },
   {  8400,  9150,  -240, Ori },
   { 10500, 11235,  -240, Ori },
   { 32100, 32340,  -240, Ser },
   { 32850, 33450,  -240, Ser },
   { 33450, 33960,  -240, Aql },
   { 40950, 42900,  -240, Psc },
   { 19350, 20730,  -360, Leo },
   { 20730, 21300,  -360, Vir },
   {     0,   600,  -420, Psc },
   { 42900, 43200,  -420, Psc },
   { 25650, 26400,  -480, Vir },
   { 28650, 29280,  -480, Oph },
   { 36000, 36960,  -540, Aql },
   { 38400, 39360,  -540, Aqr },
   { 30900, 32340,  -600, Oph },
   { 10500, 14550,  -660, Mon },
   {  8850,  9150,  -660, Eri },
   {  9150, 10500,  -660, Ori },
   { 14550, 15060,  -660, Hya },
   { 17250, 19350,  -660, Sex },
   { 21300, 23100,  -660, Vir },
   { 31650, 31800,  -700, Oph },
   { 33960, 36000,  -722, Aql },
   {  8700,  8850,  -870, Eri },
   { 36960, 38400,  -900, Aqr },
   { 30900, 32850,  -960, Ser },
   { 32850, 33960,  -960, Sct },
   { 15060, 15450, -1020, Hya },
   { 29280, 29475, -1095, Oph },
   { 15450, 16350, -1140, Hya },
   { 19350, 19500, -1140, Crt },
   { 29280, 29475, -1155, Oph },
   { 28200, 28650, -1200, Lib },
   { 22650, 23100, -1320, Crv },
   { 23100, 25650, -1320, Vir },
   { 16350, 17550, -1440, Hya },
   {  3000,  4770, -1463, Cet },
   {  4770,  6750, -1463, Eri },
   { 19500, 21300, -1470, Crt },
   { 21300, 22650, -1470, Crv },
   { 25650, 26850, -1470, Lib },
   { 29280, 30150, -1475, Oph },
   {     0,  3000, -1530, Cet },
   { 38400, 39360, -1530, Cap },
   { 39360, 42900, -1530, Aqr },
   { 42900, 43200, -1530, Cet },
   { 17550, 18450, -1590, Hya },
   {  8460,  8700, -1635, Eri },
   {  8700, 11010, -1635, Lep },
   { 36000, 38400, -1680, Cap },
   { 18450, 19050, -1750, Hya },
   { 22650, 26850, -1770, Hya },
   { 26850, 28200, -1770, Lib },
   { 28200, 28800, -1770, Sco },
   {  8250,  8460, -1800, Eri },
   { 30150, 31680, -1800, Oph },
   { 31680, 32100, -1800, Sgr },
   { 19050, 19500, -1870, Hya },
   { 11010, 13260, -1980, CMa },
   { 22050, 22650, -1980, Hya },
   { 19500, 22050, -2100, Hya },
   {  6300,  6750, -2160, For },
   { 15060, 16860, -2205, Pyx },
   {  7680,  8250, -2220, Eri },
   { 32100, 34500, -2220, Sgr },
   { 38400, 41400, -2220, PsA },
   { 41400, 42000, -2220, Scl },
   {  5400,  6300, -2375, For },
   { 16860, 19800, -2385, Ant },
   {     0,  3000, -2400, Scl },
   {  3000,  5400, -2400, For },
   {  6960,  7680, -2400, Eri },
   { 42000, 43200, -2400, Scl },
   { 25500, 26850, -2520, Cen },
   { 28200, 28800, -2520, Lup },
   { 28800, 29557, -2520, Sco },
   {  8700,  9000, -2580, Cae },
   {  9000, 11850, -2580, Col },
   { 14400, 15060, -2580, Pup },
   {  6150,  6960, -2640, Eri },
   { 29557, 32100, -2730, Sco },
   { 32100, 34500, -2730, CrA },
   { 34500, 36600, -2730, Sgr },
   { 36600, 38400, -2730, Mic },
   {  5400,  6150, -2760, Eri },
   {  8100,  8700, -2790, Cae },
   { 27600, 28200, -2880, Lup },
   {     0,  4200, -2890, Phe },
   {  4800,  5400, -2940, Eri },
   {  7350,  7680, -2940, Hor },
   {  7680,  8100, -2940, Cae },
   { 38400, 39600, -3000, Gru },
   { 10800, 14400, -3045, Pup },
   { 14400, 14700, -3045, Vel },
   {  4350,  4800, -3060, Eri },
   {  6900,  7350, -3060, Hor },
   {     0,  3300, -3090, Phe },
   { 10800, 11100, -3150, Car },
   { 14700, 15210, -3180, Vel },
   {  6300,  6900, -3190, Hor },
   {  6900,  7200, -3190, Dor },
   {     0,  2850, -3210, Phe },
   {  3900,  4350, -3240, Eri },
   {  8100,  9000, -3240, Pic },
   { 27090, 27600, -3240, Lup },
   { 15210, 15900, -3270, Vel },
   { 11100, 11700, -3300, Car },
   { 21300, 23100, -3300, Cen },
   { 25500, 27090, -3300, Lup },
   { 27090, 27600, -3300, Nor },
   {  7200,  7800, -3390, Dor },
   { 15900, 19800, -3390, Vel },
   { 19800, 20250, -3390, Cen },
   { 31500, 32400, -3420, Ara },
   { 32400, 36600, -3420, Tel },
   { 39600, 42000, -3420, Gru },
   {  5760,  6300, -3450, Hor },
   {  9000,  9900, -3450, Pic },
   { 11700, 12300, -3480, Car },
   {     0,  2400, -3510, Phe },
   {  2400,  3900, -3510, Eri },
   { 42000, 43200, -3510, Phe },
   {  7800,  8250, -3540, Dor },
   { 27600, 29557, -3600, Nor },
   { 36600, 38400, -3600, Ind },
   {  9900, 10800, -3660, Pic },
   { 27300, 27600, -3660, Cir },
   { 29557, 29850, -3660, Ara },
   { 26850, 27300, -3815, Cir },
   { 29850, 30150, -3815, Ara },
   { 10800, 12300, -3840, Pic },
   { 12300, 16260, -3840, Car },
   { 20250, 21300, -3840, Cen },
   { 21300, 23100, -3840, Cru },
   { 23100, 26160, -3840, Cen },
   { 24300, 24600, -3900, Cir },
   { 30150, 30300, -3900, Ara },
   {  3900,  5760, -4050, Hor },
   {  5760,  8250, -4050, Ret },
   { 26550, 26850, -4050, Cir },
   { 30300, 31500, -4050, Ara },
   { 31500, 32400, -4050, Pav },
   { 39600, 42000, -4050, Tuc },
   {  8250, 11850, -4200, Dor },
   { 24600, 26550, -4200, Cir },
   { 26550, 30600, -4200, TrA },
   {     0,  2400, -4500, Tuc },
   {  6300,  8250, -4500, Hyi },
   { 11850, 16260, -4500, Vol },
   { 16260, 20250, -4500, Car },
   { 20250, 24600, -4500, Mus },
   { 32400, 38400, -4500, Pav },
   { 38400, 42000, -4500, Ind },
   { 42000, 43200, -4500, Tuc },
   {  1350,  2400, -4560, Tuc },
   {     0,  6300, -4950, Hyi },
   { 13800, 24600, -4950, Cha },
   { 24600, 32400, -4950, Aps },
   {  6300, 13800, -5100, Men },
   {     0, 43200, -5400, Oct }
};

static short start[] = {
   355, 352, 343, 340, 332, 320,
   303, 288, 277, 266, 257, 251,
   239, 229, 221, 211, 203, 189,
   177, 163, 149, 136, 124, 104,
    87,  75,  69,  54,  43,  29,
    23,  16,  12,   6,   4,   0,   0
};


/*
======================================================================
constellation_pick()

Do a constellation pick from RA and Dec.

INPUTS
   r        right ascension, radians
   d        declination, radians
   e        epoch to which r and d precessed, as an mjd

RESULTS
   Returns an index for the constellation region that the coordinates
   belong to, or -1 if no constellation pick can be found.

The constellation is identified by linear search.  We look for a
member of cbound[] whose lower_dec is less than the declination of the
pick point and whose lower_ra and upper_ra bracket the pick point's
right ascension.  The data for that cbound[] member describes a sub-
region belonging to the picked constellation.

In geometric terms, the search amounts to starting at the north pole
and traveling south on a line of constant right ascension through the
pick point.  At every declination where a cbound[] member lives, we
wake up, and if (1) the pick point is now behind us, and (2) the
cbound[] segment is crossing our path, we know we've hit the edge of
a constellation subregion that the pick point belongs to.

The cbound[] right ascension and declination values are scaled into
integers; this conserves storage and makes comparisons faster.  The
start[] array, which gives starting points in cbound[] for different
declinations in 5-degree increments, further speeds the search by
skipping early parts of the list for which we know the search can't
succeed--geometrically, we start no more than 5 degrees north of the
pick point, rather than at the north pole.

The data in cbound[] are for epoch 1875.
======================================================================
*/

int
cns_pick(r, d, e)
double r;
double d;
double e;
{
   double mjd;
   unsigned short ra;
   short de, i;

   cal_mjd( 1, 1.0, 1875, &mjd );
   precess( e, mjd, &r, &d );
   ra = ( unsigned short )( radhr( r ) * 1800 );
   de = ( short )( raddeg( d ) * 60 );

   i = ( de + 5400 ) / 300;
   if ( i < 0 || i > 36 ) return -1;
   i = start[ i ];

   for ( ; i < NBOUNDS; i++ )
      if ( cbound[ i ].lower_dec <= de &&
           cbound[ i ].upper_ra   > ra &&
           cbound[ i ].lower_ra  <= ra ) break;

   return ( i == NBOUNDS ) ? -1 : ( int ) cbound[ i ].index;
}

/* given a constellation id (as from cns_pick()), return pointer to static
 * storage containg its name in the form "AAA: Name".
 * return "???: ???" if id is invalid.
 */
char *
cns_name (id)
int id;
{
	if (id < 0 || id >= NNAMES)
	    return ("???: ???");
	return (cns_namemap[id]);
}


/*
======================================================================
constellation_id()

Return constellation id given its name.

INPUTS
   abbrev   3-character constellation abbreviation

RESULTS
   Returns an index used to identify the constellation, or -1 if no
   match to a constellation abbreviation can be found.
======================================================================
*/

#define MKLONG(n) (((long)n[0])<<16 | ((long)n[1])<<8 | ((long)n[2]))

int
cns_id (abbrev)
char *abbrev;
{
   int left = 0, right = NNAMES - 1, k, v;
   long a;

   a = MKLONG( abbrev );
   while ( right >= left ) {
      k = ( left + right ) / 2;
      v = a - MKLONG( cns_namemap[ k ] );
      if ( v == 0 ) return k;
      if ( v < 0 ) right = k - 1; else left = k + 1;
   }
   return -1;
}

#ifdef CONOUTLINE
/*
======================================================================
Ernie Wright  24 May 94

Constellation outlines.

The outline information was reconstructed from a pick correlation
routine for constellations by Nancy G. Roman (Roman, N.G. 1987, Publ.
Astron. Soc. Pacific 99, 695), first C version by Craig Counterman and
Elwood Downey, as modified by EW.
======================================================================
*/

typedef unsigned char  UBYTE;
typedef unsigned short UWORD;
typedef short          WORD;


static struct {
   UWORD    x;       /* first vertex right ascension, hours * 1800 */
   WORD     y;       /* first vertex declination, degrees * 60 */
   WORD     index;   /* offset into the displacement and turn arrays */
} border[ 89 ] = {
   42000,  3150,    0,
   17550, -1440,   36,
   32400, -4050,   48,
   34200,  1110,   54,
   39000,   165,   76,
   32400, -2730,   92,
    6060,  1840,  104,
   10980,  3360,  114,
   27450,  3330,  134,
   13260,  -660,  150,
   14055,   810,  154,
   24300,  3180,  168,
    9000, -1635,  182,
   26100,  5190,  194,
   39360,  -540,  226,
   14700, -3045,  236,
    6150,  4620,  254,
   26850, -1770,  284,
   14400,  5280,  302,
    5910,   595,  338,
   24600, -4500,  350,
   27600, -3300,  354,
   16650,  2010,  368,
   11010, -1635,  378,
   22200,  2040,  384,
   34500, -2220,  396,
   29400,  2400,  400,
   21300,  -360,  408,
   23100, -3300,  414,
   23100,  -660,  418,
   37080,  3655,  424,
   37020,  1230,  454,
    8100, -2940,  468,
   37800,  5160,  488,
   38400,   750,  538,
    8400,     0,  546,
    6750, -1463,  590,
   13950,  2130,  598,
   42000, -2220,  620,
   30600,  3090,  626,
    7680, -2400,  654,
   17250,   420,  676,
    3900, -3510,  708,
   38400, -2730,  720,
   18300,  2520,  730,
   41160,  3375,  746,
   17790,  2010,  762,
   11010,  -660,  778,
   27150,     0,  784,
   28800, -1770,  796,
   12600,  3720,  808,
   34500,  2850,  828,
   11850, -4200,  842,
   38400, -1680,  850,
   12480,   720,  854,
   24300, -3840,  870,
   29557, -2520,  876,
       0, -4500,  886,
   32850,   860,  894,
   10590,  1370,  928,
   36600, -3420,  954,
   39600,  2160,  962,
    4380,  3510, 1000,
   42000, -2400, 1026,
   10800, -2580, 1036,
   41400, -1530, 1054,
    2535,  1980, 1058,
   15060,  -660, 1080,
   15450, -1020, 1090,
    7200, -3190, 1098,
   41400, -1530, 1108,
   29280,  -480, 1114,
   33960,  -240, 1128,
   28860,  1560, 1132,
   19350,   420, 1162,
   36450,  1275, 1166,
   36000,  -722, 1178,
    8100,  1840, 1190,
   36600, -2730, 1214,
   29557, -3600, 1218,
    4620,  2205, 1236,
   42000, -3420, 1250,
   20400,  4410, 1260,
   41400,  5280, 1288,
   16860, -2205, 1310,
   24300,   900, 1324,
   16260, -3840, 1342,
   37650,  1740, 1346,
       0,     0, 1366
};


/* ----- 4 2-bit turn codes per char. */

static UBYTE turn[ 342 ] = {
   0x73,0x33,0x33,0x26,0x22,0x27,0x67,0x76,0x66,0x72,0x66,0x66,
   0x77,0x27,0x73,0x73,0x22,0x66,0x66,0x77,0x23,0x26,0x76,0x73,
   0x33,0x22,0x77,0x72,0x27,0x77,0x33,0x22,0x66,0x67,0x77,0x22,
   0x22,0x67,0x27,0x73,0x32,0x66,0x77,0x33,0x22,0x67,0x77,0x22,
   0x27,0x77,0x77,0x33,0x22,0x32,0x23,0x32,0x27,0x67,0x22,0x73,
   0x33,0x32,0x66,0x67,0x77,0x73,0x33,0x22,0x22,0x23,0x22,0x77,
   0x73,0x23,0x26,0x23,0x77,0x37,0x72,0x33,0x22,0x22,0x22,0x22,
   0x27,0x77,0x22,0x27,0x27,0x73,0x22,0x22,0x77,0x73,0x27,0x26,
   0x77,0x22,0x66,0x72,0x77,0x22,0x73,0x27,0x27,0x22,0x77,0x77,
   0x33,0x77,0x23,0x22,0x22,0x67,0x73,0x22,0x26,0x77,0x33,0x32,
   0x66,0x66,0x77,0x77,0x66,0x66,0x73,0x33,0x33,0x33,0x32,0x22,
   0x22,0x26,0x27,0x77,0x27,0x73,0x77,0x77,0x77,0x22,0x22,0x22,
   0x22,0x22,0x22,0x67,0x22,0x27,0x73,0x33,0x22,0x22,0x26,0x73,
   0x27,0x37,0x77,0x32,0x23,0x26,0x62,0x67,0x77,0x77,0x72,0x22,
   0x22,0x73,0x33,0x33,0x33,0x22,0x26,0x76,0x66,0x76,0x72,0x26,
   0x73,0x26,0x67,0x73,0x32,0x26,0x67,0x77,0x77,0x33,0x27,0x32,
   0x27,0x76,0x67,0x72,0x77,0x32,0x26,0x77,0x22,0x22,0x73,0x33,
   0x32,0x26,0x66,0x73,0x32,0x22,0x67,0x72,0x67,0x27,0x77,0x26,
   0x66,0x67,0x26,0x77,0x73,0x27,0x76,0x67,0x77,0x73,0x32,0x63,
   0x22,0x62,0x26,0x26,0x73,0x66,0x77,0x33,0x22,0x26,0x67,0x32,
   0x67,0x77,0x72,0x32,0x32,0x22,0x22,0x66,0x66,0x66,0x77,0x73,
   0x23,0x33,0x22,0x66,0x6c,0x88,0x89,0x77,0x33,0x32,0x66,0x67,
   0x27,0x37,0x77,0x73,0x22,0x66,0x77,0x72,0x27,0x26,0x67,0x72,
   0x66,0xcc,0x97,0x73,0x32,0x66,0x72,0x72,0x66,0x27,0x26,0x62,
   0x72,0x66,0x27,0x27,0x67,0x33,0x27,0x73,0x32,0x67,0x72,0x22,
   0x33,0x26,0x26,0x67,0x27,0x77,0x72,0x66,0x66,0x77,0x32,0x22,
   0x67,0x33,0x26,0x73,0x33,0x33,0x22,0x32,0x26,0x66,0x73,0x77,
   0x32,0x22,0x22,0x27,0x73,0x33,0x26,0x77,0x73,0x32,0x22,0x67,
   0x27,0x77,0x32,0x33,0x22,0x60
};


/*
--------------------------------------------------
Displacements in right ascension (hours * 1800) or
declination (degrees * 60).  An offset into this
array for each constellation is given in border[].
The direction is determined by the corresponding
turn code.
--------------------------------------------------
*/

static WORD dis[ 1366 ] = {
    -840, -1080,  1140,  -145,   450,   -45,   450,  -200,
     120,  -360,   135,   -60,  1275,   165,  -240,   555,
    1245,   120,  1065,   105,   930,   825,  -855,  -210,
    -675,   180,  -990,  -120,  -450,  -120, -1260,   120,
   -1050,   120,  -450,   150,  -690,  -945,  2940,   285,
    -300,   230,  -450,   120,  -600,   160,  -900,   150,
   -1800,  -150, -6000,  -750,  7800,   900,  -240,  -390,
    -368,  -345,   368,  -255,  -510,  -360,   510,  -482,
    2040,   182,   960,   660,  -420,   390,  -285,   435,
    -555,    25, -1500,   140,  -360,   -45, -1680, -1020,
    1440,   360,   960,  -990,  3540,  1290, -1950,   360,
   -1350,   -15,  -600,    60, -2843,  -930,   293,  -155,
     300,   -85,   150,  -150,  1200,   630,   900,   690,
   -1710,  -205,  -900,  -135,  -450,  -905,  2910,   545,
     150,   700, -1980,  -210,  -555,  -990,  -345,  -360,
     450,   -90,  2040,   -30,  1170,   450,  1500,   540,
   -1020,   330,  -540,   240,  -720,   120, -2190, -1485,
    -135,  -135,  -825, -1230,  2850,  1080,   180,   420,
     450,   420,   570,   780,  -900,   150, -2250, -1320,
    2250,  1320,  -555,   -60,  -900,  -420,    30,  -240,
     330,   -90,  1590,   420,  -285,   180,  -210,   210,
   -2550,  -480,  -150,  -660,   600,  -120,  1650,  -210,
    1275,   135,   135,  1065,  -960,   270,  -540,  -165,
    -210,  -420,  -570,  -720,   420,   150,   600,   210,
     300,   945,-11700,   -90, -5400,  -300, -2685,  -180,
    -165,  -540,  -570,  -660,   120,  -120,   300,  -150,
    3000,   210,  1980,   360,  1620,  -120,  1740,   810,
    2160,   510,  2700,  -120,  1500,  -180,  3750,   180,
    1650,   390,  -960,  -360, -1440,   360,  -960, -1140,
    2400,   150,   960,   990, -3900,  -105,   300,  -150,
     600,  -180,   600,  -360,  3960,  -660,  3990,  1110,
   -4350,   120,  -690,    90,  -510,   135, -5550,  -660,
   -1350,  -180,  -750,  -235,  -540,  -395,   840,  -150,
     450,  -120,  1050,  -120,  1260,   120,   450,   120,
     450,   240,   600,   210,   375,    60,   945,   -90,
    1200,   660,   570,   540, -4200,  -210,  -600,  -120,
   -2250, -1290,   450,  -450,  1050,   540,  1800,  -540,
    3060,   540,  -660,   780,  1350,   750, 27000,  -110,
   -3600,  -370, -1500,  -300,   900,  -480,  -450,  -330,
    -750,  -120,   966,    85,   114,  -365,  2460,  -125,
     300,   135,   330,    75,   990,   170,   540,   235,
     750,   180,  1350,   660,  5715,   180,  2685,   300,
    5400,   180, -2310,  -475, -3000,  -540,  -900, -1110,
    3300,    67,  1770,  1358,  1140,   700,-10800,  -450,
   10800,   450, -1440,  -540, -1860,   -60,   300,  -300,
    1950,   150,   300,   235,   450,   155,   300,   360,
   -2250,  -330,  -210,  -480,  -135,  -600,   210,  -180,
    2385,  1590, -2010,  -945,  2850,   600,  -840,   345,
    -600,  -300,  -240,  -900,  1740,    60,  1200,   810,
    -450,   210, -1650,   120, -2400,  -510,  2400,   510,
   -1620,  -420,  -450,  -420,  1770,    60,   300,   780,
   -1950,  -780,   150,  -330,  1800,  1110, -1800,  -540,
    1800,   540, -1800,  -810,  1350,   150,   450,   660,
    -114,   -85, -1386,   -90,  -630,  -150,  -600,  -480,
     150,  -240,   420,  -420,   -75,  -390,  -180,  -150,
     735,    90,  2250,   -60,  1470,   480,   255,   465,
      60,    15,   105,   650, -2460,   365,  -570,  -285,
    -195,  -435,   285,  -390,   960,   240,    75,   350,
     315,   460,  -870,    60,  -750,  -120,  -450,  -130,
     300,  -200,   600,  -150,   450,  -660,  3600,   360,
   -1050,   180,  -900,   210,  -900,   210,  -900,   300,
   -5400,  -360,  -900,  -300, -1740,  -300, -1560,  -240,
   -3000,   240, -1800,   420, -2700,   180, -1500,   120,
   -2700,  -510,  3900,  -420,  1200,  -150,  2700,   -60,
    1650,  -450,  1500,  -150,   900,   -90,  2250,   -60,
    2220,  -180,  1530,   480,   600,   150,   630,    90,
     420,   120,   750,   330,   450,   480,  -900,   300,
    1500,   360,  -390,   -40,  -435,  -350,   -75,  -240,
     900,   630, -1950,  -105, -1680, -1358,  1980,  -697,
    -450,  -215,  -900,   -25, -1200,  -490,  -900,  -200,
    -450,  -120,  -450,  -300,  1500,   270,   450,   180,
     450,   120,   600,   180,   750,   120,   810,   240,
     720,   180,   570,   420,   210,   165,   240,   765,
     150,   210,   300,   420,  -750,   240, -3750,  -937,
    2400,    25,   900,   215,   450,   697, -2190,  -450,
   -1170,  -390,   600,  -240,   165,  -330,  1125,  -120,
     120,   150,   900,    60,   555,   390,   135,   480,
     210,   330,  -450,   120, -3600,  -780,  1200,  -420,
    2400,  1200, -2250,  -690,  1050,  -780,  -300,   -60,
    -240,  -240,  -210,  -360,   300,  -720,  1200,   530,
     900,    90,  1800,  -140,  1110,   840,  -900,   240,
    -345,  1050,   105,   180, -2220,    60,  -720,  -240,
    -810,  -120,  -750,  -180,  -600,  -120,  -450,  -180,
    -450,  -810,  1860,   600,   540,   260,   600,   130,
     450,   120,   330,   540, -2700, -1080,   510,  -360,
     390,  -120,   900,  -300,  1200,  -150,   900,  -160,
     600,  -120,   450,  -230,  2550,   120,   600,   210,
    4200,   300, -1200,   150, -3000,  -150, -3150,   330,
    -150,   480, -2100,  1080, -1500, -1050, -1050,    60,
   -1350,  -450,  6300,   450,  1950,   450, -4350,   540,
   -1800,  -870,  1800,  -900,  3600,   450, -2400,  1050,
   -1200,   270, -1050,  -135,  -600,  -375,  1140,  -300,
    1110,  -300,   450,   120,   450,   510,  -390,   360,
   -1110,   120,  -990,   -75,  -330,  -135,  -300,  -525,
    -105,   -15,   -60,  -465,   225,   -60,  1470,   -30,
      90,  1305, -1140, -1590,  2700,  -780,  1380,  1020,
     630,  1080, -1560,  -210,  -450,  -120,  -450,   300,
   -1110,   300, -2160,  -210,  -150,  -765,  2310,   975,
    -750,  -480,  -750,  -990,  1200,  -300,  1350,   570,
     450,  1005, -1500,   195, -1950,  -750, -1350,  -780,
    1590,    60,   510,   360,   600,   360,   600,   750,
   -1620,  -480,   720,  -240,   540,  -330,  1020,  -540,
     690,  -120,  2700,   375,   600,   135,  -750,   300,
   -1350,   780, -2550,   120, -1785, -1050,   345,  -240,
     900,   -30,   705,   270,   180,   390,    75,   420,
    -420,   240, -3600,  -300, -1950,  -600,  7500,   600,
   -1950,   300, -1800, -1050,  1800,  1050, -1125,  -120,
    -120,  -840,  -735,  -420,  4050,   660, -1590,    90,
    -330,   240,   -30,   270,  -120,   120, -4050,  -660,
    4350,   600,  -300,    60, -1357,  -360,  -600,  -360,
    -510,   -60,   510,  -300,  1957,  1080,-10800,  -450,
  -18600,  -150, -7500,   150, -6300,   450, -1800,   -90,
    -900,  -530,  -870,  -435,  -630,  -285,   630,  -995,
     870,  -325,  1530,   840,  -780,   360,   750,  -100,
     150,   100,   540,   360,  -240,   240,   750,   180,
     315,    90,  -315,   105,   742,   345,  -742,   140,
    -330,  -290,   120,  -330,  -300,   180,  -480,    30,
    -660,   -30,  -630,  -930,    90,  -240,   750,  -420,
    1350,   420,   735,   840,   120,   450,  -165,   240,
    -600,    80, -5100,  -630,   900,  -450,  6000,   900,
   -1800,   180,  -480,  -480,  -570,  -270,  -300,  -240,
    -360,  -460,   120,    40,   390,  -630,   240,    45,
     360,   -60,   600,    15,  1350,   330,  1950,   150,
     300,   150,   255,   570,  -135,   360,  -120,   200,
    -450,    45,  -450,   145, -1230,    30, -1470,    60,
    -945,   -60,  -375,  -210,  -600,  -240,   540,  -180,
     675,   210,   855,  -825,    90,  -165,   270,  -200,
    3210,   320,   345,   990, -2445,   150,  -300,   120,
   -1320,    90, -1110,  3600,   300,   450,   120,   450,
     200,   900,   490, -5400, -2100,  -210,  -600,  -450,
     900,  -210,   900,  -210,   900,  -180,  1500,   360,
    -600,   180,  -600,   150,  -300,   570, -3000,  -690,
    3000,   690, -1245,  -555,   240,  -165, -1275,  -510,
    -255,  -150,  -300,  -150, -1950,  -690,  1950,  -180,
     900,   540,  3000,   475,  -600,  1085,  -465,   300,
   -1800, -1320, -1410,  -600, -1050,  -465,  3600,   465,
     660,  1920,  -390, -1185,  1800,   765,  -510,   300,
    -900,   120,  -900,  -260,  -540,  -600,  2490,   510,
    -450,   150,  -600,   200,  -690,   600,  -180,  4200,
     870, -4800,  -630,  -720,  -450,  -570,   600,  -750,
     757,  -210,  2543,   930, -1950,   325,  -870,   995,
   -1110,  -720,  1110,   720, -1710, -1755,  2130,   435,
    -330,   720,  -300,   360,   210,   240, -1710, -1755,
    2130,   435,  -330,   720,  -300,   360,   210,   240,
   -1710, -1755,  2130,   435,  -330,   720,  -300,   360,
     210,   240, -2100, -1080,  2100,  1080,  -750,  -125,
   -1050,   115,  -690,  -155,   240,  -140,  1500,   -25,
     750,   330, -2040,  -238, -2280,  -840,   420,  -420,
    2400,  -510,  2100,  1050,  -600,   958, -2040,  -700,
    -150, -1245,   540,   105,  1860,   930,   630,    30,
     660,   -30,   480,  -180,   300,   330,  -120,   290,
     330,   340, -2040,    90,  -450,    40, -4200,  -690,
    4200,   690, -1957,   -60,  -300,  -155,  -450,  -235,
    -300,  -150,  4050,   150,  -300,   150,  -150,    85,
    -300,   155,  -293,    60, -1020,  -105, -1065,  -420,
     465,  -180,   450,   135,   900,   205,   540,   200,
    -270,   165, -2400,  -630,  2400,  -450,  2550,   -60,
    1050,  1050, -3600,    90, -6060,  -810,   810,  -780,
    1350,  -300,  1800,  -120,  1110,  -360,   390,  -300,
    1800,   960,   150,   480,  2550,  -270,   960,   420,
     690,   450, -1650,    60, -2700,   150, -1200,   420,
  -27000,   -90, 11700,  -390, -1650,  -180, -1050,  -420,
    1800,  -240,  3000,   240,  1560,   300,  1740,   300,
     900,   360,  5400,    10,  3600,   110, -1800,  -375,
    -660,  -465,   300,  -135,   510,   -90,   690,  -120,
    3900,  1005, -2940,   180, -1200,   -60, -1740,  -180,
    -630, -1020,   570,  -300,  1800,  -660,  2550,   840,
     750,   480,   750,   480, -2850,   420, -4410,  -660,
    4410,   660, -2250,   -90,  -735,  -120,  -705,  -265,
     690,  -115,  1050,   125,   750,   -45,   570,   -60,
    1230,   240,   300,   270,  -900,    60
};


/*
======================================================================
cns_outline()

Provide a point list describing a constellation's boundaries.

INPUTS
   id       the constellation
   e        date to which to precess coordinates, mjd
   ra       storage, right ascension coordinates, radians
   dec      storage, declination coordinates, radians
   path     storage, turn codes

RESULTS
   Constellation vertices are put in the coordinate arrays, and a
   corresponding sequence of turn codes is put in the path[] array.
   The number of coordinates is returned.  Pass NULL for any of the
   arrays to determine the coordinate count (the required array size)
   without getting the coordinates.  The maximum number of vertices
   is 50.

The id is the same as that returned by constellation_pick() and
constellation_id().  The vertices of the constellation polygon are
listed in clockwise order.  Boundary lines connect the first vertex to
the second, the second to the third, and so on; the last vertex is
connected to the first.  The path array contains the following codes:

   0  left  (increasing r.a.)
   1  right (decreasing r.a.)
   2  up    (increasing dec.)
   3  down  (decreasing dec.)

The code in path[ n ] is the direction of travel from vertex n to
vertex n + 1.  (Note that all boundary segments are lines of constant
r.a. or dec.)  There may be graph-theoretical advantages in having
the path[] array handy, but the information it contains is required
for determining whether segments of changing right ascension cross the
0:00 r.a. line.

Example source for drawing constellation boundaries is given below.
It assumes a display with pixel origin (MINX, MINY) in the upper-left
corner and uses graphics primitives moveto() and lineto(), forms of
which are available on most systems.  It produces a simple rectangular
map.

   #define PX(x) (MAXX-(MAXX*(x))/(2*PI))
   #define PY(y) (MIDY-(MIDY*(y))/(PI/2))

   int id, j, n, x, y, dx, path[ 50 ];
   double e, ra[ 50 ], dec[ 50 ];

   n = cns_outline( id, e, ra, dec, path );
   moveto( PX( ra[ 0 ] ), PY( dec[ 0 ] ));
   for ( j = 1; j <= n; j++ ) {
      x = PX( ra[ j % n ] );
      y = PY( dec[ j % n ] );
      dx = ra[ j % n ] - ra[ j - 1 ];
      if ( path[ j - 1 ] == 0 && dx < 0 ) {
         lineto( MINX, y );
         moveto( MAXX, y );
      }
      else if ( path[ j - 1 ] == 1 && dx > 0 ) {
         lineto( MAXX, y );
         moveto( MINX, y );
      }
      lineto( x, y );
   }
======================================================================
*/

/* ----- convert from internal format */

#define RADX(x) hrrad((double)(x)/1800.0)
#define RADY(y) degrad((double)(y)/60.0)


int
cns_outline (id, e, ra, dec, path)
int id;
double e;
double ra[];
double dec[];
int path[];
{
   UBYTE  mask[ 4 ] = { 0xC0, 0x30, 0x0C, 0x03 };
   int    shift[ 4 ] = { 6, 4, 2, 0 };
   int    i, j, n;
   double x, y, mjd;


   if ( id < 0 || id > 87 ) return 0;
   n = border[ id + 1 ].index - border[ id ].index;
   if ( !ra || !dec || !path ) return n;

   j = border[ id ].index;
   x = RADX( border[ id ].x );
   y = RADY( border[ id ].y );
   cal_mjd( 1, 1.0, 1875, &mjd );

   for ( i = 0; ; i++, j++ ) {
      ra[ i ] = x;
      dec[ i ] = y;
      precess( mjd, e, &ra[i], &dec[i] );
      path[ i ] = ( turn[ j >> 2 ] & mask[ j & 3 ] ) >> shift[ j & 3 ];
      if ( i == n - 1 ) return n;

      if ( path[ i ] < 2 ) {
         x += RADX( dis[ j ] );
         range( &x, 2 * PI );
      }
      else y += RADY( dis[ j ] );
   }
}

#endif /* CONOUTLINE */

/* given an ra, dec and epoch return the list of constellation ids which
 * *may* fall within the given radius of said location.
 * return the number of ids.
 * ids[] need be no larger than 88.
 */
/* ARGSUSED */
int
cns_list (ra, dec, e, rad, ids)
double ra;
double dec;
double e;
double rad;
int ids[];
{
	int i;

	/* TODO: this! */
	for (i = 0; i < NNAMES; i++)
	    ids[i] = i;
	return (NNAMES);
}

/* epoch 2000 RA/Dec of constellation figure end-points.
 * drawcodes: 0=move to; 1=draw to; 2=draw to dashed; 3=end
 */
typedef struct {
    short drawcode;
    unsigned short ra;	/* hours * 1800 */
    short dec;		/* degrees * 60 */
} ConFig;

/* Andromeda */
static ConFig _Andromeda_figs[] = {
    { 0,  3717,  2539},
    { 1,  2904,  2484},
    { 1,  2940,  2917},
    { 0,  2904,  2484},
    { 1,  2092,  2137},
    { 1,  1180,  1851},
    { 1,   251,  1745},
    { 0,  2092,  2137},
    { 1,  1702,  2309},
    { 1,  1494,  2464},
    { 0,  1420,  1456},
    { 1,  1156,  1758},
    { 1,  1180,  1851},
    { 1,   513,  2320},
    { 1, 42544,  2596},
    { 1, 42612,  2660},
    { 1, 42527,  2787},
    { 0, 42544,  2596},
    { 1, 41457,  2539},
    {-1,     0,     0},
};

/* Antlia */
static ConFig _Antlia_figs[] = {
    { 0, 19701, -2228},
    { 1, 18814, -1864},
    { 1, 17526, -1666},
    { 1, 17077, -2157},
    {-1,     0,     0},
};

/* Apus */
static ConFig _Apus_figs[] = {
    { 0, 30092, -4651},
    { 1, 29803, -4733},
    { 1, 29410, -4721},
    { 1, 26636, -4742},
    {-1,     0,     0},
};

/* Aquarius */
static ConFig _Aquarius_figs[] = {
    { 0, 41683, -1270},
    { 1, 41239,  -949},
    { 1, 41088,  -815},
    { 1, 41178,  -454},
    { 1, 41829,  -362},
    { 1, 40465,    -1},
    { 1, 40249,   -83},
    { 1, 39773,   -19},
    { 1, 38747,  -334},
    { 1, 37579,  -539},
    { 1, 37430,  -569},
    { 0, 39773,   -19},
    { 1, 40105,  -467},
    { 1, 39793,  -832},
    {-1,     0,     0},
};

/* Aquila */
static ConFig _Aquila_figs[] = {
    { 0, 34188,   904},
    { 1, 34362,   831},
    { 1, 35588,   636},
    { 1, 35723,   532},
    { 1, 35859,   384},
    { 1, 35774,    60},
    { 1, 35301,   -77},
    { 1, 34387,  -292},
    { 1, 34250,  -344},
    { 0, 36339,   -49},
    { 1, 35774,    60},
    { 1, 34965,   186},
    {-1,     0,     0},
};

/* Ara */
static ConFig _Ara_figs[] = {
    { 0, 31533, -3641},
    { 1, 31361, -3382},
    { 1, 31359, -3331},
    { 1, 31555, -2992},
    { 1, 30587, -3189},
    { 1, 30558, -3359},
    { 1, 30293, -3542},
    { 0, 30558, -3359},
    { 1, 31361, -3382},
    {-1,     0,     0},
};

/* Aries */
static ConFig _Aries_figs[] = {
    { 0,  5749,  1183},
    { 1,  5376,  1280},
    { 1,  5099,  1635},
    { 1,  3815,  1407},
    { 1,  3439,  1248},
    { 1,  3406,  1157},
    {-1,     0,     0},
};

/* Auriga */
static ConFig _Auriga_figs[] = {
    { 0, 10786,  3257},
    { 1, 10786,  2696},
    { 1, 10791,  2232},
    { 1,  8910,  1989},
    { 1,  9195,  2474},
    { 1,  9074,  2464},
    { 1,  9059,  2629},
    { 1,  9195,  2474},
    { 1,  9500,  2759},
    { 1, 10786,  2696},
    {-1,     0,     0},
};

/* Bootes */
static ConFig _Bootes_figs[] = {
    { 0, 25604,  3107},
    { 1, 25691,  2765},
    { 1, 26162,  2298},
    { 1, 27058,  2423},
    { 1, 27465,  1998},
    { 1, 27734,  2242},
    { 0, 27465,  1998},
    { 1, 26549,  1624},
    { 1, 25670,  1150},
    { 1, 26155,  1822},
    { 1, 26162,  2298},
    { 0, 24884,   947},
    { 1, 24818,  1047},
    { 1, 25040,  1103},
    { 1, 25670,  1150},
    { 1, 26422,   985},
    { 1, 26434,   823},
    {-1,     0,     0},
};

/* Caelum */
static ConFig _Caelum_figs[] = {
    { 0,  8417, -2511},
    { 1,  8461, -2228},
    {-1,     0,     0},
};

/* Camelopardalis */
static ConFig _Camelopardalis_figs[] = {
    { 0,  8918,  3225},
    { 1,  9102,  3626},
    { 1,  8821,  3980},
    { 1,  6911,  4279},
    { 1,  6272,  3596},
    {-1,     0,     0},
};

/* Cancer */
static ConFig _Cancer_figs[] = {
    { 0, 16154,   711},
    { 1, 15740,  1089},
    { 1, 15698,  1288},
    { 1, 15800,  1725},
    { 0, 15740,  1089},
    { 1, 14895,   551},
    {-1,     0,     0},
};

/* Canes Venatici */
static ConFig _Canes_Venatici_figs[] = {
    { 0, 23280,  2298},
    { 1, 22612,  2481},
    {-1,     0,     0},
};

/* Canis Major */
static ConFig _Canis_Major_figs[] = {
    { 0, 12425,  -722},
    { 1, 12712,  -938},
    { 1, 12484, -1023},
    { 1, 12154, -1002},
    { 1, 11481, -1077},
    { 0, 12154, -1002},
    { 1, 12690, -1430},
    { 1, 12851, -1583},
    { 1, 12651, -1676},
    { 1, 12559, -1738},
    { 1, 11409, -1803},
    { 0, 12851, -1583},
    { 1, 13044, -1606},
    { 1, 13161, -1497},
    { 0, 13044, -1606},
    { 1, 13323, -1758},
    {-1,     0,     0},
};

/* Canis Minor */
static ConFig _Canis_Minor_figs[] = {
    { 0, 13779,   313},
    { 1, 13414,   497},
    {-1,     0,     0},
};

/* Capricornus */
static ConFig _Capricornus_figs[] = {
    { 0, 36529,  -750},
    { 1, 36630,  -886},
    { 1, 37383, -1516},
    { 1, 37554, -1615},
    { 1, 38014, -1500},
    { 1, 38600, -1344},
    { 1, 38912, -1167},
    { 1, 39211,  -967},
    { 1, 39002,  -999},
    { 1, 38467, -1010},
    { 1, 37978, -1033},
    { 1, 36630,  -886},
    {-1,     0,     0},
};

/* Carina */
static ConFig _Carina_figs[] = {
    { 0, 11518, -3161},
    { 1, 11849, -3178},
    { 1, 12295, -3217},
    { 1, 14303, -3178},
    { 1, 15075, -3570},
    { 1, 16712, -3556},
    { 1, 18512, -3679},
    { 1, 18960, -3701},
    { 1, 19288, -3863},
    { 1, 18412, -4202},
    { 1, 16596, -4183},
    { 1, 17613, -3904},
    {-1,     0,     0},
};

/* Cassiopeia */
static ConFig _Cassiopeia_figs[] = {
    { 0,  3432,  3820},
    { 1,  2574,  3614},
    { 1,  1701,  3643},
    { 1,  1214,  3392},
    { 1,   275,  3548},
    { 0,  1701,  3643},
    { 2,   990,  3775},
    { 2,  4472,  4044},
    { 0,  1214,  3392},
    { 2,  1109,  3233},
    { 2,  2133,  3308},
    {-1,     0,     0},
};

/* Centaurus */
static ConFig _Centaurus_figs[] = {
    { 0, 24018, -2202},
    { 1, 24331, -2364},
    { 1, 24883, -2067},
    { 1, 25400, -2182},
    { 1, 25816, -2273},
    { 1, 26459, -2267},
    { 1, 26509, -2110},
    { 0, 25816, -2273},
    { 1, 26265, -2529},
    { 1, 26975, -2526},
    { 0, 25400, -2182},
    { 1, 24885, -2501},
    { 1, 24888, -2548},
    { 1, 25066, -2837},
    { 1, 24596, -3207},
    { 1, 22845, -2937},
    { 1, 23203, -2410},
    { 1, 24885, -2501},
    { 0, 22845, -2937},
    { 1, 22441, -3013},
    { 1, 21851, -3043},
    { 1, 21949, -3142},
    { 1, 20430, -3269},
    { 1, 21195, -3670},
    { 1, 20873, -3781},
    { 0, 25314, -3622},
    { 1, 24596, -3207},
    { 1, 26388, -3650},
    {-1,     0,     0},
};

/* Cepheus */
static ConFig _Cepheus_figs[] = {
    { 0, 36887,  3779},
    { 1, 37358,  3710},
    { 1, 38357,  3755},
    { 1, 38660,  4233},
    { 1, 42580,  4657},
    { 1, 41090,  3972},
    { 1, 39925,  3492},
    { 1, 38357,  3755},
    { 0, 39925,  3492},
    { 1, 40051,  3422},
    { 1, 40475,  3504},
    {-1,     0,     0},
};

/* Cetus */
static ConFig _Cetus_figs[] = {
    { 0,  4899,   194},
    { 1,  5468,   245},
    { 1,  5391,   534},
    { 1,  4948,   606},
    { 1,  4445,   507},
    { 1,  4676,   335},
    { 1,  4899,   194},
    { 1,  4784,    19},
    { 1,  4180,  -178},
    { 1,  3344,  -620},
    { 1,  2520,  -491},
    { 1,  2057,  -610},
    { 1,  3121,  -956},
    { 1,  3344,  -620},
    { 0,  3121,  -956},
    { 1,  3600, -1264},
    { 1,  4923,  -831},
    { 0,  2057,  -610},
    { 1,  1307, -1079},
    { 1,   583,  -529},
    {-1,     0,     0},
};

/* Chamaeleon */
static ConFig _Chamaeleon_figs[] = {
    { 0, 14956, -4615},
    { 1, 19064, -4716},
    { 1, 22150, -4758},
    { 1, 19373, -4832},
    { 1, 15019, -4649},
    { 1, 14956, -4615},
    {-1,     0,     0},
};

/* Circinus */
static ConFig _Circinus_figs[] = {
    { 0, 27701, -3559},
    { 1, 26475, -3898},
    { 1, 27525, -3528},
    {-1,     0,     0},
};

/* Columba */
static ConFig _Columba_figs[] = {
    { 0,  9936, -2128},
    { 1, 10189, -2044},
    { 1, 10529, -2146},
    { 1, 10726, -2117},
    { 1, 11296, -2108},
    { 1, 11463, -2006},
    { 0, 10529, -2146},
    { 1, 10774, -2568},
    {-1,     0,     0},
};

/* Coma Berenices */
static ConFig _Coma_Berenices_figs[] = {
    { 0, 23699,  1051},
    { 1, 23756,  1672},
    { 1, 22408,  1696},
    {-1,     0,     0},
};

/* Corona Australis */
static ConFig _Corona_Australis_figs[] = {
    { 0, 34161, -2226},
    { 1, 34392, -2223},
    { 1, 34484, -2274},
    { 1, 34501, -2360},
    { 1, 34450, -2429},
    { 1, 34293, -2525},
    { 1, 34088, -2562},
    { 1, 33887, -2606},
    {-1,     0,     0},
};

/* Corona Borealis */
static ConFig _Corona_Borealis_figs[] = {
    { 0, 27988,  1881},
    { 1, 27835,  1746},
    { 1, 28040,  1602},
    { 1, 28282,  1577},
    { 1, 28488,  1564},
    { 1, 28727,  1612},
    { 1, 28843,  1791},
    {-1,     0,     0},
};

/* Corvus */
static ConFig _Corvus_figs[] = {
    { 0, 21852, -1483},
    { 1, 21903, -1357},
    { 1, 22074, -1052},
    { 1, 22496,  -990},
    { 1, 22631, -1403},
    { 1, 21903, -1357},
    {-1,     0,     0},
};

/* Crater */
static ConFig _Crater_figs[] = {
    { 0, 21480, -1029},
    { 1, 21143, -1101},
    { 1, 20546, -1061},
    { 1, 20380,  -886},
    { 1, 20538,  -651},
    { 1, 20900,  -588},
    { 0, 20546, -1061},
    { 1, 20149, -1369},
    { 1, 19793, -1097},
    { 1, 20380,  -886},
    {-1,     0,     0},
};

/* Crux */
static ConFig _Crux_figs[] = {
    { 0, 22398, -3785},
    { 1, 22535, -3426},
    { 0, 23031, -3581},
    { 1, 22054, -3524},
    {-1,     0,     0},
};

/* Cygnus */
static ConFig _Cygnus_figs[] = {
    { 0, 34713,  3202},
    { 2, 35091,  3103},
    { 2, 35293,  3013},
    { 2, 35549,  2707},
    { 1, 36667,  2415},
    { 1, 37386,  2038},
    { 2, 38188,  1813},
    { 2, 39124,  1724},
    { 0, 38188,  1813},
    { 2, 38337,  2093},
    { 2, 38243,  2282},
    { 2, 38322,  2363},
    { 2, 38819,  2735},
    { 2, 39204,  2958},
    { 0, 37243,  2716},
    { 1, 36667,  2415},
    { 1, 35889,  2105},
    { 1, 35717,  1974},
    { 1, 35121,  1677},
    {-1,     0,     0},
};

/* Delphinus */
static ConFig _Delphinus_figs[] = {
    { 0, 36996,   678},
    { 1, 37126,   875},
    { 1, 37304,   904},
    { 1, 37399,   967},
    { 1, 37189,   954},
    { 1, 37126,   875},
    {-1,     0,     0},
};

/* Dorado */
static ConFig _Dorado_figs[] = {
    { 0,  7681, -3089},
    { 1,  8220, -3302},
    { 1,  9165, -3448},
    { 1, 10009, -3749},
    { 1, 10343, -3944},
    { 0, 10009, -3749},
    { 1, 10623, -3785},
    {-1,     0,     0},
};

/* Draco */
static ConFig _Draco_figs[] = {
    { 0, 31565,  3311},
    { 1, 31513,  3138},
    { 1, 32298,  3089},
    { 1, 32206,  3412},
    { 1, 31565,  3311},
    { 0, 32206,  3412},
    { 1, 34576,  4059},
    { 1, 35645,  4216},
    { 1, 33031,  4363},
    { 1, 30863,  3942},
    { 1, 29519,  3690},
    { 1, 28856,  3513},
    { 1, 27748,  3537},
    { 1, 25331,  3862},
    { 1, 22604,  4187},
    { 1, 20742,  4159},
    {-1,     0,     0},
};

/* Equuleus */
static ConFig _Equuleus_figs[] = {
    { 0, 38234,   600},
    { 1, 38487,   408},
    { 1, 38274,   314},
    { 1, 38110,   607},
    {-1,     0,     0},
};

/* Eridanus */
static ConFig _Eridanus_figs[] = {
    { 0,  9274,  -525},
    { 1,  9235,  -305},
    { 1,  8787,  -327},
    { 1,  8565,  -195},
    { 1,  8289,  -201},
    { 1,  7556,  -410},
    { 1,  7141,  -810},
    { 1,  6784,  -726},
    { 1,  6697,  -585},
    { 1,  6388,  -567},
    { 1,  5293,  -533},
    { 1,  4953, -1114},
    { 1,  5130, -1260},
    { 1,  5472, -1417},
    { 1,  5985, -1305},
    { 1,  6413, -1297},
    { 1,  6805, -1394},
    { 1,  7197, -1440},
    { 1,  8205, -1786},
    { 1,  8266, -1833},
    { 1,  7921, -2041},
    { 1,  7737, -2027},
    { 1,  6883, -2172},
    { 1,  6858, -2257},
    { 1,  5998, -2584},
    { 1,  5348, -2418},
    { 1,  4820, -2391},
    { 1,  4794, -2573},
    { 1,  4409, -2862},
    { 1,  4095, -3090},
    { 1,  3478, -3096},
    { 1,  2931, -3434},
    {-1,     0,     0},
};

/* Fornax */
static ConFig _Fornax_figs[] = {
    { 0,  5762, -1739},
    { 1,  5072, -1944},
    { 1,  3734, -1757},
    {-1,     0,     0},
};

/* Gemini */
static ConFig _Gemini_figs[] = {
    { 0, 11246,  1350},
    { 1, 11489,  1350},
    { 1, 12118,  1507},
    { 1, 12934,  1814},
    { 1, 13473,  1907},
    { 1, 13638,  1913},
    { 1, 13899,  1733},
    { 1, 13959,  1681},
    { 1, 13933,  1463},
    { 1, 13203,  1318},
    { 1, 12723,  1234},
    { 1, 11931,   983},
    { 1, 11669,  1212},
    { 1, 11489,  1350},
    { 0, 11931,   983},
    { 1, 12119,   793},
    { 0, 13203,  1318},
    { 1, 13143,   992},
    {-1,     0,     0},
};

/* Grus */
static ConFig _Grus_figs[] = {
    { 0, 39418, -2241},
    { 1, 39783, -2372},
    { 1, 40068, -2480},
    { 1, 40478, -2609},
    { 1, 40880, -2813},
    { 1, 41056, -3079},
    { 1, 41426, -3165},
    { 0, 39847, -2817},
    { 1, 40880, -2813},
    { 1, 41711, -2714},
    { 1, 41606, -2611},
    {-1,     0,     0},
};

/* Hercules */
static ConFig _Hercules_figs[] = {
    { 0, 29063,  2696},
    { 1, 29823,  2546},
    { 1, 30087,  2335},
    { 1, 30038,  1896},
    { 1, 29706,  1289},
    { 1, 29457,  1149},
    { 0, 29706,  1289},
    { 1, 31039,   863},
    { 0, 31784,  2760},
    { 1, 32287,  2235},
    { 1, 31310,  2228},
    { 1, 31051,  2208},
    { 1, 30087,  2335},
    { 0, 31051,  2208},
    { 1, 30608,  1855},
    { 1, 30038,  1896},
    { 0, 30608,  1855},
    { 1, 31050,  1490},
    { 1, 31521,  1566},
    { 1, 31994,  1663},
    { 1, 32333,  1754},
    { 1, 32626,  1725},
    { 1, 32355,  1811},
    { 0, 32626,  1725},
    { 1, 33111,  1306},
    { 1, 32663,  1248},
    { 0, 33111,  1306},
    { 1, 33770,  1232},
    { 1, 33810,  1090},
    {-1,     0,     0},
};

/* Horologium */
static ConFig _Horologium_figs[] = {
    { 0,  7620, -2537},
    { 1,  4876, -3048},
    { 1,  4722, -3152},
    { 1,  4820, -3273},
    { 1,  5508, -3584},
    { 1,  5364, -3844},
    {-1,     0,     0},
};

/* Hydra */
static ConFig _Hydra_figs[] = {
    { 0, 25391, -1600},
    { 1, 23967, -1390},
    { 1, 21387, -2034},
    { 1, 20790, -1911},
    { 1, 19488,  -971},
    { 1, 18782, -1010},
    { 1, 18317,  -741},
    { 1, 18153,  -783},
    { 1, 17744,  -890},
    { 1, 17027,  -519},
    { 1, 17395,   -68},
    { 1, 16631,   138},
    { 1, 16062,   356},
    { 1, 15696,   203},
    { 1, 15562,   200},
    { 1, 15529,   342},
    { 1, 15803,   385},
    { 1, 16062,   356},
    {-1,     0,     0},
};

/* Hydrus */
static ConFig _Hydrus_figs[] = {
    { 0,  6817, -4454},
    { 1,   772, -4635},
    { 1,  4787, -4096},
    { 1,  4252, -4119},
    { 1,  3448, -4058},
    { 1,  3563, -3694},
    {-1,     0,     0},
};

/* Indus */
static ConFig _Indus_figs[] = {
    { 0, 37127, -2837},
    { 1, 38396, -3206},
    { 1, 39537, -3299},
    { 0, 38396, -3206},
    { 1, 37644, -3507},
    {-1,     0,     0},
};

/* Lacerta */
static ConFig _Lacerta_figs[] = {
    { 0, 40079,  2264},
    { 1, 40514,  2587},
    { 1, 40486,  2862},
    { 1, 40335,  2968},
    { 1, 40307,  3133},
    { 1, 40538,  3016},
    { 1, 40486,  2862},
    {-1,     0,     0},
};

/* Leo */
static ConFig _Leo_figs[] = {
    { 0, 16939,  1570},
    { 1, 17151,  1378},
    { 1, 17575,  1426},
    { 1, 17783,  1560},
    { 1, 18500,  1405},
    { 1, 18599,  1190},
    { 1, 18220,  1005},
    { 1, 18251,   718},
    { 1, 17434,   593},
    { 0, 18251,   718},
    { 1, 20227,   925},
    { 1, 21272,   874},
    { 1, 20223,  1231},
    { 1, 18599,  1190},
    { 0, 20227,   925},
    { 1, 20517,   631},
    { 1, 20434,   361},
    {-1,     0,     0},
};

/* Leo Minor */
static ConFig _Leo_Minor_figs[] = {
    { 0, 19599,  2052},
    { 1, 18836,  2202},
    { 1, 18223,  2114},
    {-1,     0,     0},
};

/* Lepus */
static ConFig _Lepus_figs[] = {
    { 0,  9369,  -712},
    { 1,  9397,  -776},
    { 1,  9587,  -790},
    { 1,  9388,  -972},
    { 1,  9982, -1069},
    { 1, 10408,  -889},
    { 1, 10488,  -869},
    { 1, 10692,  -850},
    { 0,  9982, -1069},
    { 1,  9847, -1245},
    { 0, 10539, -1252},
    { 1, 10334, -1346},
    { 1,  9847, -1245},
    { 1,  9613, -1274},
    { 1,  9164, -1342},
    {-1,     0,     0},
};

/* Libra */
static ConFig _Libra_figs[] = {
    { 0, 28745,  -856},
    { 1, 28615, -1003},
    { 1, 28322,  -940},
    { 1, 28066,  -887},
    { 1, 27510,  -562},
    { 1, 26726,  -962},
    { 1, 27122, -1516},
    { 1, 28110, -1688},
    { 1, 28159, -1786},
    { 0, 27122, -1516},
    { 1, 27510,  -562},
    {-1,     0,     0},
};

/* Lupus */
static ConFig _Lupus_figs[] = {
    { 0, 28998, -2208},
    { 1, 28803, -2303},
    { 1, 28054, -2470},
    { 1, 27680, -2681},
    { 1, 27358, -2924},
    { 1, 27368, -3125},
    { 1, 26458, -2843},
    { 1, 26956, -2588},
    { 1, 27641, -2438},
    { 1, 27654, -2175},
    { 1, 28529, -2017},
    { 0, 28054, -2470},
    { 1, 27641, -2438},
    {-1,     0,     0},
};

/* Lynx */
static ConFig _Lynx_figs[] = {
    { 0, 16831,  2063},
    { 1, 16765,  2208},
    { 1, 16219,  2506},
    { 1, 15085,  2591},
    { 1, 13401,  2952},
    { 1, 12518,  3505},
    { 1, 11388,  3540},
    {-1,     0,     0},
};

/* Lyra */
static ConFig _Lyra_figs[] = {
    { 0, 34060,  2636},
    { 1, 34613,  2348},
    { 1, 34691,  2288},
    { 1, 33508,  2327},
    { 1, 33730,  2380},
    { 0, 33508,  2327},
    { 1, 33743,  2256},
    { 1, 33902,  2001},
    { 1, 34168,  1961},
    { 1, 34012,  2218},
    { 1, 33743,  2256},
    { 0, 33508,  2327},
    { 1, 32996,  2163},
    {-1,     0,     0},
};

/* Mensa */
static ConFig _Mensa_figs[] = {
    { 0, 11107, -4485},
    { 1,  9956, -4580},
    { 1,  8855, -4496},
    { 1,  9081, -4278},
    {-1,     0,     0},
};

/* Microscopium */
static ConFig _Microscopium_figs[] = {
    { 0, 38423, -2448},
    { 1, 38338, -1930},
    { 1, 37838, -1935},
    { 1, 37499, -2026},
    {-1,     0,     0},
};

/* Monoceros */
static ConFig _Monoceros_figs[] = {
    { 0, 14657,  -179},
    { 1, 13837,  -573},
    { 1, 12956,   -29},
    { 1, 11664,  -421},
    { 1, 11245,  -376},
    { 1, 11513,   275},
    { 1, 11787,   439},
    {-1,     0,     0},
};

/* Musca */
static ConFig _Musca_figs[] = {
    { 0, 23468, -4292},
    { 1, 22715, -4148},
    { 1, 22574, -4327},
    { 0, 22988, -4086},
    { 1, 22715, -4148},
    { 1, 22127, -4077},
    { 1, 21247, -4008},
    {-1,     0,     0},
};

/* Norma */
static ConFig _Norma_figs[] = {
    { 0, 29615, -2853},
    { 1, 29310, -3004},
    { 1, 29395, -3009},
    { 1, 28896, -2953},
    {-1,     0,     0},
};

/* Octans */
static ConFig _Octans_figs[] = {
    { 0, 26007, -5020},
    { 1, 39044, -4643},
    { 1, 40200, -4826},
    { 1, 40981, -4882},
    { 1, 26007, -5020},
    {-1,     0,     0},
};

/* Ophiuchus */
static ConFig _Ophiuchus_figs[] = {
    { 0, 31260, -1499},
    { 1, 30911,  -943},
    { 1, 29915,  -634},
    { 1, 29734,  -996},
    { 0, 29915,  -634},
    { 1, 29349,  -281},
    { 1, 29230,  -221},
    { 1, 29727,   119},
    { 1, 30530,   562},
    { 1, 30419,   609},
    { 0, 30530,   562},
    { 1, 31647,   753},
    { 1, 32620,   573},
    { 1, 32563,   149},
    { 1, 32371,  -586},
    { 0, 32563,   149},
    { 1, 32419,   175},
    { 1, 32036,   162},
    { 1, 31904,   274},
    { 1, 31395,   248},
    {-1,     0,     0},
};

/* Orion */
static ConFig _Orion_figs[] = {
    { 0, 10917,  1208},
    { 1, 11158,   852},
    { 1, 10871,   578},
    { 1, 10655,   444},
    { 1, 10223,  -116},
    { 1, 10432,  -580},
    { 1,  9436,  -492},
    { 1,  9960,   -17},
    { 1,  9754,   380},
    { 1, 10054,   596},
    { 1, 10655,   444},
    { 0,  9754,   380},
    { 1,  8695,   417},
    { 1,  8736,   336},
    { 1,  8827,   146},
    { 1,  8956,   102},
    { 0,  8695,   417},
    { 1,  8718,   534},
    { 1,  8847,   609},
    {-1,     0,     0},
};

/* Pavo */
static ConFig _Pavo_figs[] = {
    { 0, 36769, -3404},
    { 1, 37348, -3972},
    { 1, 37258, -4005},
    { 1, 36017, -4374},
    { 1, 33691, -4285},
    { 1, 31971, -3883},
    { 1, 32657, -3820},
    { 1, 33097, -3689},
    { 1, 33966, -3731},
    { 1, 36261, -3970},
    { 1, 37348, -3972},
    { 1, 38593, -3921},
    {-1,     0,     0},
};

/* Pegasus */
static ConFig _Pegasus_figs[] = {
    { 0, 39125,   592},
    { 1, 39906,   371},
    { 1, 40844,   649},
    { 1, 41001,   730},
    { 1, 41543,   912},
    { 1,   397,   911},
    { 2,   251,  1745},
    { 2, 41513,  1684},
    { 1, 41543,   912},
    { 0, 39139,  1538},
    { 1, 39810,  1520},
    { 1, 40890,  1813},
    { 1, 41513,  1684},
    { 1, 41100,  1476},
    { 1, 40996,  1413},
    { 1, 39135,  1041},
    {-1,     0,     0},
};

/* Perseus */
static ConFig _Perseus_figs[] = {
    { 0,  7397,  3021},
    { 1,  7646,  2904},
    { 1,  7460,  2862},
    { 1,  6687,  2867},
    { 1,  6494,  2891},
    { 1,  6129,  2991},
    { 1,  5544,  3210},
    { 1,  5121,  3353},
    { 0,  6687,  2867},
    { 1,  6756,  2554},
    { 1,  7135,  2400},
    { 1,  7169,  2147},
    { 1,  7024,  1913},
    { 1,  6729,  1937},
    { 0,  6129,  2991},
    { 1,  5672,  2976},
    { 1,  4925,  2953},
    { 0,  5672,  2976},
    { 1,  5685,  2691},
    { 1,  5645,  2457},
    { 1,  5555,  2330},
    { 1,  5363,  2379},
    {-1,     0,     0},
};

/* Phoenix */
static ConFig _Phoenix_figs[] = {
    { 0,  2737, -2944},
    { 1,  2651, -2599},
    { 1,  2255, -2731},
    { 1,  1982, -2803},
    { 1,   788, -2538},
    { 1,   282, -2744},
    { 1, 42452, -2556},
    { 1, 42535, -2729},
    { 1, 43168, -3164},
    { 0,  1982, -2803},
    { 1,  2051, -3314},
    { 1,  1300, -3447},
    { 1,   282, -2744},
    {-1,     0,     0},
};

/* Pictor */
static ConFig _Pictor_figs[] = {
    { 0, 12245, -3716},
    { 1, 10495, -3370},
    { 1, 10418, -3063},
    {-1,     0,     0},
};

/* Pisces */
static ConFig _Pisces_figs[] = {
    { 0,  2150,  1805},
    { 1,  2384,  1635},
    { 1,  2212,  1475},
    { 1,  2744,   920},
    { 1,  3162,   549},
    { 1,  3661,   165},
    { 1,  3043,   329},
    { 1,  2705,   368},
    { 1,  1888,   473},
    { 1,  1460,   455},
    { 1, 43179,   411},
    { 1, 42598,   337},
    { 1, 42239,   382},
    { 1, 41915,   196},
    { 1, 42208,    75},
    { 1, 42661,   106},
    { 1, 42598,   337},
    {-1,     0,     0},
};

/* Piscis Austrinus */
static ConFig _Piscis_Austrinus_figs[] = {
    { 0, 41329, -1777},
    { 1, 40819, -1622},
    { 1, 40029, -1666},
    { 1, 39232, -1853},
    { 1, 39148, -1981},
    { 1, 39851, -1979},
    { 1, 40545, -1940},
    { 1, 41176, -1972},
    { 1, 41278, -1952},
    { 1, 41329, -1777},
    {-1,     0,     0},
};

/* Puppis */
static ConFig _Puppis_figs[] = {
    { 0, 14671, -1154},
    { 1, 14306, -1372},
    { 1, 14079, -1491},
    { 1, 13764, -1608},
    { 1, 13661, -1702},
    { 1, 13521, -1857},
    { 1, 13114, -2225},
    { 1, 11933, -2591},
    { 1, 12298, -3036},
    { 1, 13006, -2678},
    { 1, 13477, -2598},
    { 1, 14507, -2400},
    { 1, 14626, -1458},
    { 1, 14306, -1372},
    {-1,     0,     0},
};

/* Pyxis */
static ConFig _Pyxis_figs[] = {
    { 0, 15603, -2118},
    { 1, 15708, -1991},
    { 1, 15916, -1662},
    {-1,     0,     0},
};

/* Reticulum */
static ConFig _Reticulum_figs[] = {
    { 0,  6725, -3888},
    { 1,  7633, -3748},
    { 1,  7694, -3558},
    { 1,  7239, -3664},
    { 1,  7162, -3684},
    { 1,  6725, -3888},
    {-1,     0,     0},
};

/* Sagitta */
static ConFig _Sagitta_figs[] = {
    { 0, 36154,  1199},
    { 1, 35962,  1169},
    { 1, 35621,  1112},
    { 1, 35431,  1048},
    { 0, 35621,  1112},
    { 1, 35403,  1080},
    {-1,     0,     0},
};

/* Sagittarius */
static ConFig _Sagittarius_figs[] = {
    { 0, 34896, -2687},
    { 1, 35858, -2512},
    { 1, 34916, -2436},
    { 0, 35858, -2512},
    { 1, 35992, -2116},
    { 1, 35995, -2081},
    { 1, 36080, -1662},
    { 1, 35908, -1630},
    { 1, 35301, -1493},
    { 1, 34666, -1515},
    { 1, 34408, -1660},
    { 1, 34278, -1792},
    { 1, 33769, -1619},
    { 1, 34058, -1577},
    { 1, 34408, -1660},
    { 0, 34058, -1577},
    { 1, 34340, -1304},
    { 1, 34493, -1261},
    { 1, 34729, -1137},
    { 1, 34850, -1070},
    { 0, 34340, -1304},
    { 1, 34132, -1266},
    { 0, 33769, -1619},
    { 1, 33239, -1525},
    { 1, 32813, -1263},
    { 0, 33239, -1525},
    { 1, 33030, -1789},
    { 1, 33125, -2063},
    { 1, 32929, -2205},
    { 0, 33030, -1789},
    { 1, 32574, -1825},
    { 1, 32027, -1669},
    {-1,     0,     0},
};

/* Scorpius */
static ConFig _Scorpius_figs[] = {
    { 0, 28963, -1188},
    { 1, 29004, -1240},
    { 1, 29159, -1166},
    { 1, 28930,  -682},
    { 1, 28963, -1188},
    { 1, 28810, -1357},
    { 1, 28765, -1566},
    { 1, 28706, -1752},
    { 0, 28810, -1357},
    { 1, 29435, -1535},
    { 1, 29681, -1585},
    { 1, 29876, -1692},
    { 1, 30305, -2057},
    { 1, 29741, -2082},
    { 1, 29891, -2115},
    { 1, 30305, -2057},
    { 1, 30356, -2282},
    { 1, 30437, -2541},
    { 1, 30964, -2594},
    { 1, 31719, -2579},
    { 1, 32027, -2407},
    { 1, 31874, -2341},
    { 1, 31608, -2226},
    { 0, 32027, -2407},
    { 1, 32095, -2222},
    {-1,     0,     0},
};

/* Sculptor */
static ConFig _Sculptor_figs[] = {
    { 0,  1758, -1761},
    { 1,   645, -1738},
    { 1, 42868, -1687},
    { 1, 41964, -1951},
    { 1, 42389, -2269},
    {-1,     0,     0},
};

/* Scutum */
static ConFig _Scutum_figs[] = {
    { 0, 33815,  -284},
    { 1, 33456,  -494},
    { 1, 33276,  -873},
    { 0, 33456,  -494},
    { 1, 33110,  -536},
    {-1,     0,     0},
};

/* Serpens */
static ConFig _Serpens_figs[] = {
    { 0, 34086,   252},
    { 1, 33039,  -173},
    { 1, 31842,  -772},
    { 1, 31727,  -923},
    { 1, 31225,  -770},
    { 0, 28488,  -205},
    { 1, 28509,   131},
    { 1, 28524,   268},
    { 1, 28328,   385},
    { 1, 28044,   632},
    { 1, 28385,   925},
    { 1, 28693,   939},
    { 1, 28462,  1088},
    { 1, 28385,   925},
    {-1,     0,     0},
};

/* Sextans */
static ConFig _Sextans_figs[] = {
    { 0, 18908,   -38},
    { 1, 18238,   -22},
    { 1, 17775,  -486},
    {-1,     0,     0},
};

/* Taurus */
static ConFig _Taurus_figs[] = {
    { 0,  9789,  1716},
    { 1,  8467,  1377},
    { 1,  8058,  1150},
    { 1,  7964,  1075},
    { 1,  7888,  1052},
    { 1,  7794,   937},
    { 1,  8060,   952},
    { 1,  8277,   990},
    { 1, 10129,  1268},
    { 0,  7794,   937},
    { 1,  7220,   749},
    { 1,  6326,   776},
    { 1,  6215,   583},
    { 1,  6144,   541},
    { 0,  7220,   749},
    { 1,  7665,   533},
    { 1,  7294,   359},
    {-1,     0,     0},
};

/* Telescopium */
static ConFig _Telescopium_figs[] = {
    { 0, 32737, -2757},
    { 1, 33209, -2758},
    { 1, 33265, -2944},
    {-1,     0,     0},
};

/* Triangulum */
static ConFig _Triangulum_figs[] = {
    { 0,  3392,  1774},
    { 1,  3689,  1997},
    { 1,  3886,  2099},
    { 1,  4119,  2030},
    { 1,  3392,  1774},
    {-1,     0,     0},
};

/* Triangulum Australe */
static ConFig _Triangulum_Australe_figs[] = {
    { 0, 30259, -4141},
    { 1, 28654, -3805},
    { 1, 28101, -3979},
    { 1, 27567, -4120},
    { 1, 30259, -4141},
    {-1,     0,     0},
};

/* Tucana */
static ConFig _Tucana_figs[] = {
    { 0, 40420, -3897},
    { 1, 40155, -3615},
    { 1, 41923, -3494},
    { 1, 43197, -3934},
    { 1,   602, -3892},
    { 1,   946, -3777},
    { 1, 41923, -3494},
    {-1,     0,     0},
};

/* Ursa Major */
static ConFig _Ursa_Major_figs[] = {
    { 0, 24826,  2958},
    { 1, 24118,  3295},
    { 1, 23221,  3357},
    { 1, 22063,  3421},
    { 1, 21415,  3221},
    { 1, 19855,  3382},
    { 1, 19912,  3705},
    { 0, 21415,  3221},
    { 2, 21181,  2866},
    { 2, 20345,  1891},
    { 0, 21181,  2866},
    { 2, 20090,  2669},
    { 2, 18513,  2574},
    { 0, 20090,  2669},
    { 2, 18670,  2489},
    { 0, 19912,  3705},
    { 2, 17146,  3783},
    { 2, 15308,  3643},
    { 0, 19855,  3382},
    { 2, 17763,  3243},
    { 2, 17729,  3542},
    { 0, 17763,  3243},
    { 2, 17185,  3100},
    { 2, 16176,  2882},
    { 2, 16309,  2829},
    { 2, 17185,  3100},
    {-1,     0,     0},
};

/* Ursa Minor */
static ConFig _Ursa_Minor_figs[] = {
    { 0,  4554,  5355},
    { 1, 31566,  5195},
    { 1, 30179,  4922},
    { 1, 28322,  4667},
    { 1, 26721,  4449},
    { 1, 27622,  4310},
    { 1, 29325,  4545},
    { 1, 28322,  4667},
    {-1,     0,     0},
};

/* Vela */
static ConFig _Vela_figs[] = {
    { 0, 19403, -2965},
    { 1, 19119, -2893},
    { 1, 18442, -2527},
    { 1, 17121, -2428},
    { 1, 16440, -2605},
    { 1, 15732, -2558},
    { 1, 15529, -2579},
    { 1, 14686, -2840},
    { 1, 15740, -3282},
    { 1, 16863, -3300},
    { 1, 17906, -3274},
    { 1, 19403, -2965},
    {-1,     0,     0},
};

/* Virgo */
static ConFig _Virgo_figs[] = {
    { 0, 21756,   523},
    { 1, 21176,   391},
    { 1, 21321,   105},
    { 1, 22197,   -40},
    { 1, 22850,   -86},
    { 1, 23268,   203},
    { 1, 23465,   657},
    { 0, 23268,   203},
    { 1, 24441,   -35},
    { 1, 24156,  -669},
    { 1, 23698,  -332},
    { 1, 22850,   -86},
    { 0, 24441,   -35},
    { 1, 25249,    92},
    { 1, 26587,   113},
    { 0, 24156,  -669},
    { 1, 25587,  -616},
    { 1, 25680,  -360},
    { 1, 26492,  -339},
    {-1,     0,     0},
};

/* Volans */
static ConFig _Volans_figs[] = {
    { 0, 16273, -3983},
    { 1, 15172, -3968},
    { 1, 14637, -4117},
    { 1, 13105, -4077},
    { 1, 12861, -4229},
    { 1, 13854, -4356},
    { 1, 14637, -4117},
    {-1,     0,     0},
};

/* Vulpecula */
static ConFig _Vulpecula_figs[] = {
    { 0, 35804,  1444},
    { 1, 35061,  1479},
    { 1, 34686,  1283},
    {-1,     0,     0},
};


static ConFig *figmap[88] = {
    _Andromeda_figs,
    _Antlia_figs,
    _Apus_figs,
    _Aquila_figs,
    _Aquarius_figs,
    _Ara_figs,
    _Aries_figs,
    _Auriga_figs,
    _Bootes_figs,
    _Canis_Major_figs,
    _Canis_Minor_figs,
    _Canes_Venatici_figs,
    _Caelum_figs,
    _Camelopardalis_figs,
    _Capricornus_figs,
    _Carina_figs,
    _Cassiopeia_figs,
    _Centaurus_figs,
    _Cepheus_figs,
    _Cetus_figs,
    _Chamaeleon_figs,
    _Circinus_figs,
    _Cancer_figs,
    _Columba_figs,
    _Coma_Berenices_figs,
    _Corona_Australis_figs,
    _Corona_Borealis_figs,
    _Crater_figs,
    _Crux_figs,
    _Corvus_figs,
    _Cygnus_figs,
    _Delphinus_figs,
    _Dorado_figs,
    _Draco_figs,
    _Equuleus_figs,
    _Eridanus_figs,
    _Fornax_figs,
    _Gemini_figs,
    _Grus_figs,
    _Hercules_figs,
    _Horologium_figs,
    _Hydra_figs,
    _Hydrus_figs,
    _Indus_figs,
    _Leo_Minor_figs,
    _Lacerta_figs,
    _Leo_figs,
    _Lepus_figs,
    _Libra_figs,
    _Lupus_figs,
    _Lynx_figs,
    _Lyra_figs,
    _Mensa_figs,
    _Microscopium_figs,
    _Monoceros_figs,
    _Musca_figs,
    _Norma_figs,
    _Octans_figs,
    _Ophiuchus_figs,
    _Orion_figs,
    _Pavo_figs,
    _Pegasus_figs,
    _Perseus_figs,
    _Phoenix_figs,
    _Pictor_figs,
    _Piscis_Austrinus_figs,
    _Pisces_figs,
    _Puppis_figs,
    _Pyxis_figs,
    _Reticulum_figs,
    _Sculptor_figs,
    _Scorpius_figs,
    _Scutum_figs,
    _Serpens_figs,
    _Sextans_figs,
    _Sagitta_figs,
    _Sagittarius_figs,
    _Taurus_figs,
    _Telescopium_figs,
    _Triangulum_Australe_figs,
    _Triangulum_figs,
    _Tucana_figs,
    _Ursa_Major_figs,
    _Ursa_Minor_figs,
    _Vela_figs,
    _Virgo_figs,
    _Volans_figs,
    _Vulpecula_figs,
};

/* max entries in any one map = 33 */

/* given a constellation id and epoch, return arrays of ra[] and dec[]
 *   end-points precessed to the desired epoch that, if connected, will form the
 *   given constellation figure.
 * dcodes is 0 if the coord is a "move-to", 1 if a "draw-to" or 2 if a "draw-to
 *   as dotted-line".
 * return the total number of tripples or -1 if id is bogus.
 * the arrays need be no larger than 32 entries.
 */
int
cns_figure (id, e, ra, dec, dcodes)
int id;
double e;
double ra[];
double dec[];
int dcodes[];
{
	static double mjd2000 = -1;
	ConFig *cfp;

	if (id < 0 || id > 87)
	    return (-1);

	if (mjd2000 < 0)
	    cal_mjd (1, 1.0, 2000, &mjd2000);

	for (cfp = figmap[id]; cfp->drawcode >= 0; cfp++) {
	    *ra = hrrad((double)cfp->ra/1800.0);
	    *dec = degrad((double)cfp->dec/60.0);
	    precess (mjd2000, e, ra, dec);
	    ra++;
	    dec++;
	    *dcodes++ = cfp->drawcode;
	}

	return (cfp - figmap[id]);
}
