/* code to manage the stuff on the "earth view" display.
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>

#if defined(__STDC__)
#include <stdlib.h>
#else
extern void *malloc(), *realloc();
#endif

#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/DrawingA.h>
#include <Xm/Separator.h>
#include <Xm/ToggleB.h>
#include <Xm/Scale.h>
#include <Xm/CascadeB.h>
#include "astro.h"
#include "circum.h"
#include "map.h"
#include "preferences.h"

#if defined(__STDC__) || defined(__cplusplus)
#define P_(s) s
#else
#define P_(s) ()
#endif

extern Now *mm_get_now P_((void));
extern void mm_movie P_((double stepsz));
extern Obj *db_basic P_((int id));
extern double mjd_day P_((double jd));
extern double mjd_hr P_((double jd));
extern int any_ison P_((void));
extern int get_color_resource P_((Widget w, char *cname, Pixel *p));
extern int obj_cir P_((Now *np, Obj *op));
extern void f_dms_angle P_((Widget w, double a));
extern void fs_date P_((char out[], double jd));
extern void fs_time P_((char out[], double t));
extern void fs_pangle P_((char out[], double a));
extern void f_double P_((Widget w, char *fmt, double f));
extern void get_something P_((Widget w, char *resource, char *value));
extern void set_something P_((Widget w, char *resource, char *value));
extern void get_xmlabel_font P_((Widget w, XFontStruct **f));
extern void hlp_dialog P_((char *tag, char *deflt[], int ndeflt));
extern void precess P_((double mjd1, double mjd2, double *ra, double *dec));
extern void range P_((double *v, double r));
extern void redraw_screen P_((int how_much));
extern void register_selection P_((char *name));
extern void f_showit P_((Widget w, char *s));
extern void fs_dms_angle P_((char out[], double a));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern char *syserrstr P_((void));
extern void solve_sphere P_((double A, double b, double cosc, double sinc,
    double *cosap, double *Bp));
extern void timestamp P_((Now *np, Widget w));
extern void utc_gst P_((double Mjd, double utc, double *gst));
extern void watch_cursor P_((int want));
extern void xe_msg P_((char *msg, int app_modal));
extern FILE *fopenh P_((char *name, char *how));

void e_manage P_((void));
void e_update P_((Now *np, int force));
int e_ison P_((void));
void e_newobj P_((int dbidx));
void e_selection_mode P_((int whether));
void e_cursor P_((Cursor c));
static void e_create_form P_((void));
static void e_create_esform P_((void));
static void e_read_sites P_((void));
static void e_point_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_stat_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_close_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_estats_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_movie_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_set_buttons P_((int whether));
static void e_help_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_exp_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_motion_cb P_((Widget w, XtPointer client, XEvent *ev,
    Boolean *continue_to_dispatch));
static void e_obj_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_objing_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_tb_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_cyl_cb P_((Widget w, XtPointer client, XtPointer call));
static int e_set_dasize P_((int wantcyl));
static void e_setmain_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_latlong_cb P_((Widget w, XtPointer client, XtPointer call));
static void e_track_latlng P_((Widget w, XEvent *ev, unsigned r, unsigned xb, 
    unsigned yb, int x, int y));
static void e_popup P_((Widget w, XEvent *ev, unsigned r, unsigned xb,
    unsigned yb, int x, int y));
static void e_create_popup P_((void));
static void e_init_gcs P_((Display *dsp, Window win));
static void e_copy_pm P_((void));
static void e_setelatlng P_((double lt, double lg));
static void e_subobject P_((Now *np, int dbidx, double *latp, double *longp));
static void e_show_esat_stats P_((void));
static int e_coord P_((unsigned r, unsigned wb, unsigned hb, double pl,
    double pL, short *xp, short *yp));
static void e_drawgrid P_((unsigned r, unsigned wb, unsigned hb));
static void e_drawobjects P_((unsigned r, unsigned wb, unsigned hb));
static void e_drawfootprint P_((unsigned r, unsigned wb, unsigned hb,
    double slat, double slng, double el));
static void e_drawcircle P_((unsigned r, unsigned wb, unsigned hb, double slat,
    double slng, double rad));
static void e_drawcross P_((unsigned r, unsigned wb, unsigned hb, double lt,
    double lg, int style));
static int e_uncoord P_((unsigned r, unsigned xb, unsigned yb, int x, int y,
    double *ltp, double *lgp));
static void e_all P_((Now *np));
static void e_getcircle P_((unsigned int *wp, unsigned int *hp,
    unsigned int *rp, unsigned int *xbp, unsigned int *ybp));
static void e_map P_((Now *np));
static void e_mainmenuloc P_((Now *np, unsigned r, unsigned wb, unsigned hb));
static void e_soleclipse P_((Now *np, unsigned r, unsigned wb, unsigned hb));
static void e_sunlit P_((Now *np, unsigned int r, unsigned int wb,
    unsigned int hb));
static void e_msunlit P_((Now *np, unsigned int r, unsigned int wb,
    unsigned int hb));
static void e_ssunlit P_((Now *np, unsigned int r, unsigned int wb,
    unsigned int hb));
static void e_drawcontinents P_((unsigned r, unsigned wb, unsigned hb));
static void e_drawsites P_((unsigned r, unsigned wb, unsigned hb));
static int add_to_polyline P_((XPoint xp[], int xpsize, int i, int vis, int nxp,
    int max, int w, int x, int y));
static void e_viewrad P_((double height, double alt, double *radp));
static void e_resettrails P_((Now *np));
static void e_addtrail P_((Now *np));

#undef P_

/* these are from earthmap.c */
extern MRegion ereg[];
extern int nereg;

extern Widget	toplevel_w;
extern char	*myclass;
#define XtD     XtDisplay(toplevel_w)

/* this is to form a linked-list of sites */
struct _Site {
    struct _Site *si_nxt;	/* next Site, or NULL */
    float si_lat;	/* lat (+N), rads */
    float si_lng;	/* long (+E), rads */
    float si_elev;	/* elevation above sea level, meters (-1 means ?) */
    char si_name[1];	/* site name, used as a varable-length member */
};
typedef struct _Site Site;
static Site *sites;	/* first Site in list, else NULL */
static char sitefdef[] = "xephem_sites"; /* default name of sites file */

#define	MAXSITELLEN	256	/* maximum site file line length */

#define	MOVIE_STEPSZ	0.5	/* movie step size, hours */

static int e_selecting;	/* set while our fields are being selected */
static Widget eform_w;	/* main form dialog */
static Widget e_da_w;	/* map DrawingArea widget */
static Widget e_dt_w;	/* main date/time stamp label widget */
static Widget e_sdt_w;	/* stats menu date/time stamp label widget */
static Pixmap e_pm;	/* use off screen pixmap for smoother drawing */
static Widget track_w;	/* toggle button to control tracking */
static Widget sites_w;	/* toggle button to control whether to show sites */
static Widget showobj_w;/* toggle button to control whether to show object */
static Widget objX_w, objY_w;	/* pulldown menu widgets for objs X and Y */
static Widget objcb_w;	/* the object name cascade button */
static Widget lat_w;	/* latitude scale widget */
static Widget long_w;	/* longitude scale widget */
static Widget cyl_w;	/* cylindrical view toggle button */
static Widget sph_w;	/* spherical view toggle button */
static Pixel e_fg, e_bg;/* fg and bg colors */
static GC e_strgc;	/* used for text within the drawing area */
static GC e_olgc;	/* used to draw map overlay details */
static GC e_gc;		/* used for all other GXCopy map drawing uses */
static XFontStruct *e_f;/* used to compute string extents */

/* this is to save info about the popup: widgets, and current pointing dir.
 */
#define	MAXPUL	7	/* max label widgets we ever need at once */
typedef struct {
    Widget pu_w;	/* main Popup parent widget */
    Widget pu_labels[MAXPUL];	/* the misc labels */
    double pu_lt;	/* current latitude storage */
    double pu_lg;	/* current longitude storage */
} PopupInfo;
static PopupInfo pu_info;

/* info to get and keep track of the colors.
 * if we can't get all of them, we set them to fg/bg and use xor operations.
 * this is all set up in e_init_gcs().
 * N.B. the order of the entries in the enum must match the stuff in ecolors[].
 */
enum EColors {BORDERC, GRIDC, SITEC, OBJC, ECLIPSEC, SUNC, HEREC};
#define	NCOLORS 7
typedef struct {
    char *name;
    Pixel p;
} EColor;
static EColor ecolors[NCOLORS] = {
    {"EarthBorderColor"},
    {"EarthGridColor"},
    {"EarthSiteColor"},
    {"EarthObjColor"},
    {"EarthEclipseColor"},
    {"EarthSunColor"},
    {"EarthHereColor"}
};

/* info to keep track of all the "wants" toggle buttons. 
 */
enum Wants {		/* client codes for e_tb_cb and indices into wants[] */
    CYLINDRICAL=0, TRACK, TRAILS, GRID, OBJ, SITES, SUNLIGHT,
    NWANTS		/* used to count number of entries -- must be last */
};
static wants[NWANTS];	/* fast state for each of the "wants" toggle buttons */

enum {SET_MAIN, SET_FROM_MAIN};	/* client codes for e_setmain_cb */

static double elat, elng;	/* view center lat/long: +N +E rads */
static double selat, celat;	/* sin and cos of elat */

/* following is to support the table of various statistics.
 */
typedef struct {
    char *selname;	/* name by which it is known for plotting etc */
    char *label;	/* label, else NULL if depends on preference */
    int enumcode;	/* one of the ES_ codes (just a check on right order) */
    Widget lw;		/* stat value label widget */
    Widget pbw;		/* stat value pushbutton widget */
} ESatStatTable;

/* code for each of the ess_ (s_*) fields unique to the Earth satellites.
 * N.B. must be in same order as the estattable entries!!
 */
enum estat {ES_SUBLAT, ES_SUBLONG, ES_ELEV, ES_RANGE, ES_RANGER, ES_ECLIPSED};

/* table to describe each statistic.
 * N.B. must be in same order as the estat entries!!
 */
static ESatStatTable estattable[] = {
    {"ESat.SubLat",	"Sub Latitude:",	ES_SUBLAT},
    {"ESat.SubLong",	"Sub Longitude:",	ES_SUBLONG},
    {"ESat.Elevation",	NULL,			ES_ELEV},
    {"ESat.Range",	NULL,			ES_RANGE},
    {"ESat.RangeRate",	NULL,			ES_RANGER},
    {"ESat.Eclipsed",	"In Sun Shadow:",	ES_ECLIPSED}
};

static Widget esform_w;		/* earth stats form widget */
static Widget esobjname_w;	/* earth stats object name widget */
static int eswasman;		/* whether earth stats was managed */

#define	DEFOBJ	SUN		/* default object to report/track */

/* constants that define details of various drawing actions */
#define	WIDE_LW		2		/* wide line width */
#define	NARROW_LW	0		/* narrow line width */
#define	LINE_ST		LineSolid	/* line style for lines */
#define	CAP_ST		CapRound	/* cap style for lines */
#define	JOIN_ST		JoinRound	/* join style for lines */
#define	CROSSH		1		/* drawcross code for a cross hair */
#define	PLUSS		2		/* drawcross code for a plus sign */
#define	PLLEN		degrad(4)	/* plussign arm radius, rads */
#define	CHLEN		degrad(4)	/* cross hair leg length, rads */
#define	LNGSTEP		15		/* longitude grid spacing, degrees */
#define	LATSTEP		15		/* latitude grid spacing, degrees */
#define	NLATLNG		50		/* max segments per grid line */
#define	MAXDIST		degrad(3)	/* max popup search distance, rads */
#define	CYLASPECT	PI		/* cylndrical view w/h ratio */


/* info to keep track of earlier objects while we are keeping trails.
 */
typedef struct {
    Now t_now;	/* Now when this object was first displayed */
    Obj t_obj;	/* it's Obj when this object was first displayed */
    double t_sublat, t_sublng;	/* sub lat/lng of object at this time */
} Trail;

/* Trails are kept as a variable-length malloc'd array of Trail objects.
 * N.B. even without want_trails on, we keep the current object in here too.
 * the entry corresponding to Now is always the last one on the list.
 * avoid dups because we may be drawing with xor.
 */
static Trail *trails;	/* malloc'ed array of Trails */
static int ntrails;	/* number of entries in the trail[] array */
static int object_dbidx;/* dbidx of current object */

void
e_manage()
{
	if (!eform_w) {
	    /* first call: create and init view to main menu's loc.
	     * rely on an expose to do the drawing.
	     */
	    Now *np = mm_get_now();

	    object_dbidx = DEFOBJ;
	    e_create_form();
	    e_create_esform();
	    e_read_sites();
	    e_setelatlng (lat, lng);
	    e_addtrail (np);
	}

	if (XtIsManaged(eform_w)) {
	    XtUnmanageChild(eform_w);
	    if (eswasman = XtIsManaged(esform_w))
		XtUnmanageChild(esform_w);
	} else {
	    XtManageChild(eform_w);
	    if (eswasman) {
		XtManageChild(esform_w);
		e_set_buttons(e_selecting);
	    }
	}
}

/* update the earth details.
 * remove all trail history if it's off now.
 */
void
e_update(np, force)
Now *np;
int force;
{
	/* don't bother if we've never been created or if we're not
	 * up now and no one cares about our values.
	 */
	if (!eform_w)
	    return;
	if (!XtIsManaged(eform_w) && !any_ison() && !force)
	    return;

	if (!wants[TRAILS])
	    e_resettrails(np);
	e_addtrail (np);

	e_all (np);
}

e_ison()
{
	return (eform_w && XtIsManaged(eform_w));
}

/* called when a user-defined object has changed.
 * we can assume that our update function will also be called subsequently.
 * if the object being changed is the one we are currently displaying
 *   and it is now undefined turn off tracking and trailing and change the
 *   current object to be the Sun.
 */
void
e_newobj (dbidx)
int dbidx;	/* OBJX or OBJY */
{
	Now *np = mm_get_now();
	Obj *op;

	if (dbidx != object_dbidx) 
	    return;	/* doesn't effect us */

	/* change to the new object, or DEFOBJ if it's now undefined */
	op = db_basic (dbidx);
	if (op->type == UNDEFOBJ) {
	    object_dbidx = DEFOBJ;
	    op = db_basic (DEFOBJ);
	} else
	    object_dbidx = dbidx;

	/* set the current object name everywhere */
	set_xmstring (objcb_w, XmNlabelString, op->o_name);
	set_xmstring (esobjname_w, XmNlabelString, op->o_name);

	/* discard the old trail history and add one for new object */
	e_resettrails(np);
	e_all (np);
}

/* called by other menus as they want to hear from our buttons or not.
 * the ons and offs stack.
 */
void
e_selection_mode (whether)
int whether;
{
	e_selecting += whether ? 1 : -1;

	if (e_ison())
	    if (whether && e_selecting == 1	   /* first one to want on */
		|| !whether && e_selecting == 0 /* first one to want off */)
		e_set_buttons (whether);
}

void
e_cursor(c)
Cursor c;
{
	Window win;

	if (eform_w && (win = XtWindow(eform_w))) {
	    Display *dsp = XtDisplay(eform_w);
	    if (c)
		XDefineCursor(dsp, win, c);
	    else
		XUndefineCursor(dsp, win);
	}

	if (esform_w && (win = XtWindow(esform_w))) {
	    Display *dsp = XtDisplay(esform_w);
	    if (c)
		XDefineCursor(dsp, win, c);
	    else
		XUndefineCursor(dsp, win);
	}
}

/* create the main form.
 * the earth stats form is created separately.
 */
static void
e_create_form()
{
	typedef struct {
	    char *name;
	    char *label;
	    void (*cb)();
	    int client;
	    Widget *wp;
	} BtnInfo;
	static BtnInfo ctl[] = {
	    {"SetMain", "Set Main",      e_setmain_cb, SET_MAIN,      NULL},
	    {"SetFrom", "Set From Main", e_setmain_cb, SET_FROM_MAIN, NULL},
	    {"Movie",   "Movie Demo",    e_movie_cb,   0,             NULL},
	    {"Sep1",    NULL,            NULL,         0,             NULL},
	    {"Close",   "Close",         e_close_cb,   0,             NULL},
	};
	static BtnInfo view[] = {
	    {"grid",        "Show Grid",     e_tb_cb,  GRID,          NULL},
	    {"sites",       "Show Sites",    e_tb_cb,  SITES,         NULL},
	    {"sunlight",    "Show Sunlight", e_tb_cb,  SUNLIGHT,      NULL},
	    {"Sep1",        NULL,            NULL,     0,             NULL},
	};
	static BtnInfo obj[] = {
	    {"track",  "Track Object", e_tb_cb,      TRACK,       &track_w},
	    {"trail",  "Leave Trail",  e_tb_cb,      TRAILS,      NULL},
	    {"object", "Show Object",  e_tb_cb,      OBJ,         &showobj_w},
	    {"Sep1",   NULL,           NULL,         0,           NULL},
	};
	Widget pd_w, cas_w, mb_w, pr_w;
	EventMask mask;
	Widget w;
	Arg args[20];
	int n;
	int i;

	/* create form */

	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNdefaultPosition, False); n++;
	XtSetArg (args[n], XmNverticalSpacing, 2); n++;
	XtSetArg (args[n], XmNhorizontalSpacing, 2); n++;
	eform_w = XmCreateFormDialog (toplevel_w, "Earth", args,n);
	XtAddCallback (eform_w, XmNhelpCallback, e_help_cb, 0);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Earth view"); n++;
	XtSetValues (XtParent(eform_w), args, n);

	/* make the menu bar and all its pulldowns */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	mb_w = XmCreateMenuBar (eform_w, "MB", args, n);
	XtManageChild (mb_w);

	/* make the Control pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ControlPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'C'); n++;
	    cas_w = XmCreateCascadeButton (mb_w, "ControlCB", args, n);
	    set_xmstring (cas_w, XmNlabelString, "Control");
	    XtManageChild (cas_w);

	    for (i = 0; i < XtNumber(ctl); i++) {
		BtnInfo *bip = &ctl[i];

		n = 0;
		if (!bip->label)
		    w = XmCreateSeparator (pd_w, bip->name, args, n);
		else {
		    w = XmCreatePushButton (pd_w, bip->name, args, n);
		    set_xmstring (w, XmNlabelString, bip->label);
		    XtAddCallback (w, XmNactivateCallback, bip->cb,
						    (XtPointer)bip->client);
		}
		if (bip->wp)
		    *bip->wp = w;
		XtManageChild (w);
	    }

	/* make the View pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ViewPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'V'); n++;
	    cas_w = XmCreateCascadeButton (mb_w, "ViewCB", args, n);
	    set_xmstring (cas_w, XmNlabelString, "View");
	    XtManageChild (cas_w);

	    /* simulate a radio box for the cylindrical/spherical view selection */

	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    cyl_w = XmCreateToggleButton (pd_w, "cylindrical", args, n);
	    set_xmstring (cyl_w, XmNlabelString, "Cylindrical");
	    XtAddCallback (cyl_w,XmNvalueChangedCallback,e_cyl_cb,(XtPointer)1);
	    XtManageChild (cyl_w);

	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    sph_w = XmCreateToggleButton (pd_w, "spherical", args, n);
	    set_xmstring (sph_w, XmNlabelString, "Spherical");
	    XtAddCallback (sph_w,XmNvalueChangedCallback,e_cyl_cb,(XtPointer)0);
	    XtManageChild (sph_w);

	    if(XmToggleButtonGetState(cyl_w)==XmToggleButtonGetState(sph_w)){
		int cylmode = 0;	/* default to pol if they conflict */
		xe_msg("Earth View display mode conflicts -- using Spherical",0);
		XmToggleButtonSetState (cyl_w, cylmode, False); 
		XmToggleButtonSetState (sph_w, !cylmode, False); 
	    }
	    wants[CYLINDRICAL] = XmToggleButtonGetState(cyl_w);

	    n = 0;
	    w = XmCreateSeparator (pd_w, "Sep", args, n);
	    XtManageChild (w);

	    for (i = 0; i < XtNumber(view); i++) {
		BtnInfo *bip = &view[i];

		n = 0;
		if (!bip->label)
		    w = XmCreateSeparator (pd_w, bip->name, args, n);
		else {
		    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
		    XtSetArg (args[n], XmNmarginHeight, 0); n++;
		    w = XmCreateToggleButton (pd_w, bip->name, args, n);
		    wants[bip->client] = XmToggleButtonGetState(w);
		    set_xmstring (w, XmNlabelString, bip->label);
		    XtAddCallback (w, XmNvalueChangedCallback, bip->cb,
						    (XtPointer)bip->client);
		}

		if (bip->wp)
		    *bip->wp = w;

		XtManageChild (w);
	    }

	    /* add the earth Stats control */

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Stats", args, n);
	    set_xmstring (w, XmNlabelString, "More info...");
	    XtAddCallback (w, XmNactivateCallback, e_estats_cb, NULL);
	    XtManageChild (w);

	/* make the Object pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ObjectPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'O'); n++;
	    cas_w = XmCreateCascadeButton (mb_w, "ObjectCB", args, n);
	    set_xmstring (cas_w, XmNlabelString, "Object");
	    XtManageChild (cas_w);

	    for (i = 0; i < XtNumber(obj); i++) {
		BtnInfo *bip = &obj[i];

		n = 0;
		if (!bip->label)
		    w = XmCreateSeparator (pd_w, bip->name, args, n);
		else {
		    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
		    XtSetArg (args[n], XmNmarginHeight, 0); n++;
		    w = XmCreateToggleButton (pd_w, bip->name, args, n);
		    wants[bip->client] = XmToggleButtonGetState(w);
		    set_xmstring (w, XmNlabelString, bip->label);
		    XtAddCallback (w, XmNvalueChangedCallback, bip->cb,
						    (XtPointer)bip->client);
		}
		if (bip->wp)
		    *bip->wp = w;
		XtManageChild (w);
	    }

	    /* add the object name pull-right */

	    n = 0;
	    pr_w = XmCreatePulldownMenu (pd_w, "ObjectPD", args, n);

		n = 0;
		XtSetArg (args[n], XmNsubMenuId, pr_w); n++;
		objcb_w = XmCreateCascadeButton (pd_w, "ObjNamCB", args, n);
		XtAddCallback (objcb_w, XmNcascadingCallback, e_objing_cb, 0);
		XtManageChild (objcb_w);

		/* create all the pushbuttons forming the cascade menu.
		 * go ahead and fill in and manage the planet names now.
		 * we do the user objects just as we are cascading.
		 */
		for (i = 0; i < NOBJ; i++) {
		    Obj *op;

		    n = 0;
		    w = XmCreatePushButton (pr_w, "ObjectPB", args, n);
		    XtAddCallback (w, XmNactivateCallback, e_obj_cb,
								(XtPointer)i);
		    if (i >= MERCURY && i <= MOON) {
			op = db_basic (i);
			set_xmstring (w, XmNlabelString, op->o_name);
			XtManageChild (w);
		    } else if (i == OBJX)
			objX_w = w;
		    else if (i == OBJY)
			objY_w = w;
		    
		    if (i == DEFOBJ)
			set_xmstring (objcb_w, XmNlabelString, op->o_name);
		}

	/* make the help pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "HelpPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'H'); n++;
	    cas_w = XmCreateCascadeButton (mb_w, "HelpCB", args, n);
	    set_xmstring (cas_w, XmNlabelString, "Help");
	    XtManageChild (cas_w);
	    set_something (mb_w, XmNmenuHelpWidget, (char *)cas_w);

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Help", args, n);
	    XtManageChild (w);
	    XtAddCallback (w, XmNactivateCallback, e_help_cb, NULL);

	/* make the date/time indicator label */

	n = 0;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNrecomputeSize, False); n++;
	e_dt_w = XmCreateLabel (eform_w, "DTstamp", args, n);
	XtManageChild (e_dt_w);

	/* make the long and lat scales */

	n = 0;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, e_dt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNminimum, -180); n++;
	XtSetArg (args[n], XmNmaximum, 179); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_LEFT); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	XtSetArg (args[n], XmNshowValue, True); n++;
	long_w = XmCreateScale (eform_w, "LongS", args, n);
	XtAddCallback (long_w, XmNvalueChangedCallback, e_latlong_cb, NULL);
	XtManageChild (long_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, mb_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, long_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNminimum, -90); n++;
	XtSetArg (args[n], XmNmaximum, 90); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_TOP); n++;
	XtSetArg (args[n], XmNscaleMultiple, 1); n++;
	XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
	XtSetArg (args[n], XmNshowValue, True); n++;
	lat_w = XmCreateScale (eform_w, "LatS", args, n);
	XtAddCallback (lat_w, XmNvalueChangedCallback, e_latlong_cb, NULL);
	XtManageChild (lat_w);

	/* make a drawing area on top */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, mb_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, long_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNrightWidget, lat_w); n++;
	e_da_w = XmCreateDrawingArea (eform_w, "Map", args, n);
	XtAddCallback (e_da_w, XmNexposeCallback, e_exp_cb, NULL);
	mask = Button1MotionMask | ButtonPressMask | ButtonReleaseMask
						    | PointerMotionHintMask,
	XtAddEventHandler (e_da_w, mask, False, e_motion_cb, 0);
	XtManageChild (e_da_w);

	/* init da's size */
	(void) e_set_dasize(wants[CYLINDRICAL]);
}

/* make the stats form dialog */
static void
e_create_esform()
{
	Obj *op;
	Arg args[20];
	Widget w, rc_w, sep_w;
	int n;
	int i;

	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNhorizontalSpacing, 2); n++;
	XtSetArg (args[n], XmNverticalSpacing, 2); n++;
	esform_w = XmCreateFormDialog (toplevel_w, "EarthStats", args, n);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "Earth info"); n++;
	XtSetValues (XtParent(esform_w), args, n);

	/* make a label at the top for the current object name */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	esobjname_w = XmCreateLabel (esform_w, "ObjName", args, n);
	XtManageChild (esobjname_w);

	op = db_basic (object_dbidx);
	set_xmstring (esobjname_w, XmNlabelString, op->o_name);

	/* add a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, esobjname_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sep_w = XmCreateSeparator (esform_w, "Sep1", args, n);
	XtManageChild (sep_w);

	/* make a row/col for everything */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sep_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNpacking, XmPACK_COLUMN); n++;
	XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	XtSetArg (args[n], XmNnumColumns, XtNumber(estattable)); n++;
	rc_w = XmCreateRowColumn (esform_w, "ESRC", args, n);
	XtManageChild(rc_w);

	/* make each earth satellite statistic label/value pair.
	 * only manage now the ones that are always up.
	 */

	for (i = 0; i < XtNumber (estattable); i++) {
	    ESatStatTable *stp = estattable + i;
	    XmString str;

	    /* create the labels for the ones that don't change depending
	     * upon preferences.
	     */
	    n = 0;
	    str = 0;
	    if (stp->label) {
		str = XmStringCreateLtoR (stp->label,
						XmSTRING_DEFAULT_CHARSET);
		XtSetArg (args[n], XmNlabelString, str); n++;
	    }
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	    stp->lw = XmCreateLabel (rc_w, "ESatStatLabel", args, n);
	    if (str)
		XmStringFree (str);

	    n = 0;
	    XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	    XtSetArg (args[n], XmNuserData, stp->selname); n++;
	    stp->pbw = XmCreatePushButton (rc_w, "ESatStatPB", args, n);
	    XtAddCallback (stp->pbw, XmNactivateCallback, e_stat_cb, 0);

	    switch (stp->enumcode) {
	    case ES_SUBLONG:
	    case ES_SUBLAT:
		XtManageChild (stp->lw);
		XtManageChild (stp->pbw);
		break;
	    }
	}

	/* add a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, rc_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sep_w = XmCreateSeparator (esform_w, "Sep2", args, n);
	XtManageChild (sep_w);

	/* add a label for the current date/time stamp */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sep_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
	e_sdt_w = XmCreateLabel (esform_w, "SDTstamp", args, n);
	XtManageChild (e_sdt_w);

	/* add a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, e_sdt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sep_w = XmCreateSeparator (esform_w, "Sep3", args, n);
	XtManageChild (sep_w);

	/* add a close button */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sep_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 20); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 80); n++;
	w = XmCreatePushButton (esform_w, "Close", args, n);
	XtAddCallback (w, XmNactivateCallback, e_estats_cb, NULL);
	XtManageChild (w);
}

/* read in the sites file */
static void
e_read_sites()
{
	char buf[MAXSITELLEN];
	char msg[128];
	char *fn;
	FILE *fp;

	fn = XGetDefault (XtD, myclass, "SITESFILE");
	if (!fn)
	    fn = sitefdef;

	fp = fopenh (fn, "r");
	if (!fp) {
	    (void) sprintf(msg, "Can not open %.75s: %.25s", fn, syserrstr());

	    XtSetSensitive (sites_w, False);
	    XmToggleButtonSetState (sites_w, False, False);
	    xe_msg (msg, 1);
	    return;
	}

	while (fgets (buf, sizeof(buf), fp) != NULL) {
	    char name[MAXSITELLEN];
	    int latd, latm, lats;
	    int lngd, lngm, lngs;
	    char latNS, lngEW;
	    double ele;
	    int nf;

	    if (!isalpha(buf[0]) && !isdigit(buf[0]))
		continue;

	    nf = sscanf (buf, "%[^;]; %3d %2d %2d %c   ; %3d %2d %2d %c   ;%lf",
					    name, &latd, &latm, &lats, &latNS,
					    &lngd, &lngm, &lngs, &lngEW, &ele);
	    if (nf == 10) {
		double lt, lg;
		Site *sp;
		int l;

		/* strip trailing blanks */
		for (l = strlen (name); --l >= 0; )
		    if (isspace(name[l]))
			name[l] = '\0';
		    else
			break;
		lt = degrad (latd + latm/60.0 + lats/3600.0);
		if (latNS == 'S')
		    lt = -lt;
		lg = degrad (lngd + lngm/60.0 + lngs/3600.0);
		if (lngEW == 'W')
		    lg = -lg;
		sp = (Site *) malloc (sizeof(Site) + strlen(name));
		sp->si_lat = lt;
		sp->si_lng = lg;
		sp->si_elev = ele;
		(void) strcpy (sp->si_name, name);
		sp->si_nxt = sites;
		sites = sp;
	    }
	}

	(void) fclose (fp);
}

/* callback when any of the stat buttons are activated. */
/* ARGSUSED */
static void
e_stat_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (e_selecting) {
	    char *userD;
	    get_something (w, XmNuserData, (char *)&userD);
	    register_selection (userD);
	}
}

/* callback for when the main Close button is activated. */
/* ARGSUSED */
static void
e_close_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XtUnmanageChild (eform_w);
	if (eswasman = XtIsManaged(esform_w))
	    XtUnmanageChild(esform_w);

	if (e_pm) {
	    XFreePixmap (XtDisplay(w), e_pm);
	    e_pm = (Pixmap) 0;
	}
}

/* callback for when the stats dialog is closed or activated. */
/* ARGSUSED */
static void
e_estats_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (eswasman = XtIsManaged(esform_w)) {
	    XtUnmanageChild (esform_w);
	} else {
	    XtManageChild (esform_w);
	    e_set_buttons(e_selecting);
	}
}

/* callback for when the Movie button is activated. */
/* ARGSUSED */
static void
e_movie_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	mm_movie (MOVIE_STEPSZ);
}

/* go through all the buttons and set whether they
 * should appear to look like buttons or just flat labels.
 */
static void
e_set_buttons (whether)
int whether;	/* whether setting up for plotting or for not plotting */
{
	static Arg look_like_button[] = {
	    {XmNtopShadowColor, (XtArgVal) 0},
	    {XmNbottomShadowColor, (XtArgVal) 0},
            {XmNtopShadowPixmap, (XtArgVal) 0},
            {XmNbottomShadowPixmap, (XtArgVal) 0},
	    {XmNfillOnArm, (XtArgVal) True},
	    {XmNtraversalOn, (XtArgVal) True},
	};
	static Arg look_like_label[] = {
	    {XmNtopShadowColor, (XtArgVal) 0},
	    {XmNbottomShadowColor, (XtArgVal) 0},
            {XmNtopShadowPixmap, (XtArgVal) 0},
            {XmNbottomShadowPixmap, (XtArgVal) 0},
	    {XmNfillOnArm, (XtArgVal) False},
	    {XmNtraversalOn, (XtArgVal) False},
	};
	static int called;
	Arg *ap;
	int na;
	int i;

	if (!called) {
	    /* get baseline label and shadow appearances.
	     */
            Pixel topshadcol, botshadcol, bgcol;
            Pixmap topshadpm, botshadpm;
	    Arg args[20];
	    int n;

	    n = 0;
            XtSetArg (args[n], XmNtopShadowColor, &topshadcol); n++;
            XtSetArg (args[n], XmNbottomShadowColor, &botshadcol); n++;
            XtSetArg (args[n], XmNtopShadowPixmap, &topshadpm); n++;
            XtSetArg (args[n], XmNbottomShadowPixmap, &botshadpm); n++;
	    XtSetArg (args[n], XmNbackground, &bgcol); n++;
	    XtGetValues (track_w, args, n);

            look_like_button[0].value = topshadcol;
            look_like_button[1].value = botshadcol;
            look_like_button[2].value = topshadpm;
            look_like_button[3].value = botshadpm;
            look_like_label[0].value = bgcol;
            look_like_label[1].value = bgcol;
            look_like_label[2].value = XmUNSPECIFIED_PIXMAP;
            look_like_label[3].value = XmUNSPECIFIED_PIXMAP;

	    called = 1;
	}

	if (whether) {
	    ap = look_like_button;
	    na = XtNumber(look_like_button);
	} else {
	    ap = look_like_label;
	    na = XtNumber(look_like_label);
	}

	for (i = 0; i < XtNumber(estattable); i++)
	    XtSetValues (estattable[i].pbw, ap, na);
}

/* callback from the Help button
 */
/* ARGSUSED */
static void
e_help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	static char *msg[] = {
"This is a simple schematic depiction of the Earth surface at the given time.",
"The view shows the sunlit portion and the location of any object.",
};

	hlp_dialog ("Earth", msg, XtNumber(msg));
}

/* called whenever the earth drawing area gets an expose.
 */
/* ARGSUSED */
static void
e_exp_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	static unsigned int wid_last, hei_last;
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;
	Display *dsp = XtDisplay(w);
	Window win = XtWindow(w);
	Window root;
	int x, y;
	unsigned int bw, d;
	unsigned wid, hei;

	/* filter out a few oddball cases */
	switch (c->reason) {
	case XmCR_EXPOSE: {
	    /* turn off gravity so we get expose events for either shrink or
	     * expand.
	     */
	    static before;
	    XExposeEvent *e = &c->event->xexpose;

	    if (!before) {
		XSetWindowAttributes swa;
                unsigned long mask = CWBitGravity | CWBackingStore;

		swa.bit_gravity = ForgetGravity;
		swa.backing_store = NotUseful;
		XChangeWindowAttributes (e->display, e->window, mask, &swa);
		before = 1;
	    }

	    /* wait for the last in the series */
	    if (e->count != 0)
		return;
	    break;
	    }
	default:
	    printf ("Unexpected eform_w event. type=%d\n", c->reason);
	    exit(1);
	}

	XGetGeometry (dsp, win, &root, &x, &y, &wid, &hei, &bw, &d);

	/* if no pixmap or we're a new size then (re)build everything */
	if (!e_pm || wid != wid_last || hei != hei_last) {
	    if (e_pm)
		XFreePixmap (dsp, e_pm);

	    if (!e_gc)
		e_init_gcs(dsp, win);

	    e_pm = XCreatePixmap (dsp, win, wid, hei, d);
	    XSetForeground (dsp, e_gc, e_bg);
	    XFillRectangle (dsp, e_pm, e_gc, 0, 0, wid, hei);
	    wid_last = wid;
	    hei_last = hei;

	    e_all(mm_get_now());
	} else {
	    /* nothing new so just copy from the pixmap */
	    e_copy_pm();
	}
}

/* called on receipt of a MotionNotify or ButtonPress or ButtonRelease event
 *   while the cursor is wandering over the Earth and button 1 is down or
 *   Button3 is pressed.
 * button 1 causes a report of the current location in world coordinates;
 * button 3 causes a popup.
 */
/* ARGSUSED */
static void
e_motion_cb (w, client, ev, continue_to_dispatch)
Widget w;
XtPointer client;
XEvent *ev;
Boolean *continue_to_dispatch;
{
	int trackmotion, trackbutton, popupbutton;
	int evt = ev->type;


	trackmotion = evt == MotionNotify && ev->xmotion.state == Button1Mask;
	trackbutton = (evt == ButtonPress || evt == ButtonRelease)
					&& ev->xbutton.button == Button1;
	popupbutton = evt == ButtonPress && ev->xbutton.button == Button3;

	if (popupbutton || trackmotion || trackbutton) {
	    unsigned wide, h, r, xb, yb;
	    Window root, child;
	    int rx, ry, wx, wy;
	    unsigned mask;

	    XQueryPointer (XtDisplay(w), XtWindow(w),
				&root, &child, &rx, &ry, &wx, &wy, &mask);

	    e_getcircle (&wide, &h, &r, &xb, &yb);

	    if (popupbutton)
		e_popup (w, ev, r, xb, yb, wx, wy);
	    else
		e_track_latlng (w, ev, r, xb, yb, wx, wy);
	}
}

/* called when the one of the Object pulldown buttons is activated.
 * make that object the one to view now.
 * the object id is client.
 */
/* ARGSUSED */
static void
e_obj_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	Now *np = mm_get_now();
	Obj *op;
	int dbidx;

	dbidx = (int)client;
	if (!(op = db_basic (dbidx)) || op->type == UNDEFOBJ) {
	    printf ("e_obj_cb: bad dbidx: %d\n", dbidx);
	    exit (1);
	}

	object_dbidx = dbidx;
	set_xmstring (objcb_w, XmNlabelString, op->o_name);
	set_xmstring (esobjname_w, XmNlabelString, op->o_name);

	/* restart the trail history based on the new object now */
	e_resettrails(mm_get_now());

	e_all (np);
}

/* called when the pulldown menu from the Object cascade button is cascading.
 * check whether ObjX and Y are defined and if so set their names in the
 * pulldown else turn these selections off.
 */
/* ARGSUSED */
static void
e_objing_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	Obj *op;

	op = db_basic (OBJX);
	if (op->type != UNDEFOBJ) {
	    set_xmstring (objX_w, XmNlabelString, op->o_name);
	    XtManageChild (objX_w);
	} else
	    XtUnmanageChild (objX_w);

	op = db_basic (OBJY);
	if (op->type != UNDEFOBJ) {
	    set_xmstring (objY_w, XmNlabelString, op->o_name);
	    XtManageChild (objY_w);
	} else
	    XtUnmanageChild (objY_w);
}

/* called when any of the toggle buttons changes state.
 * client is one of the Wants enum.
 */
/* ARGSUSED */
static void
e_tb_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	Now *np = mm_get_now();
	int state = XmToggleButtonGetState (w);
	int want = (int) client;

	switch (want) {
	case TRACK:
	    wants[want] = state;
	    if (state) {
		/* if turning on tracking, it's nice to also insure Show Object
		 * want is on too.
		 */
		if (!wants[OBJ]) {
		    /* don't call the callback -- that'd do two e_all()'s */
		    XmToggleButtonSetState (showobj_w, True, False);
		    wants[OBJ] = 1;
		}
		e_all (np);
	    }
	    break;
	case TRAILS:
	    wants[want] = state;
	    if (ntrails > 1)
		e_all (np);
	    break;
	case GRID:	/* FALLTHRU */
	case OBJ:	/* FALLTHRU */
	case SITES:	/* FALLTHRU */
	case SUNLIGHT:	/* FALLTHRU */
	    wants[want] = state;
	    e_all (np);
	    break;
	default:
	    printf ("e_tb_cb: bad client: %d\n", want);
	    exit (1);
	}
}

/* called when either the cylindrical or spherical button changes state.
 * client will be 1 if the former, 0 if the latter.
 */
/* ARGSUSED */
static void
e_cyl_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmToggleButtonCallbackStruct *sp = (XmToggleButtonCallbackStruct *)call;
	int which = (int) client;
	Now *np = mm_get_now();
	int wantcyl;

	wantcyl = which == sp->set;

	if (wantcyl) {
	    XmToggleButtonSetState (cyl_w, True, False);
	    XmToggleButtonSetState (sph_w, False, False);
	} else {
	    XmToggleButtonSetState (cyl_w, False, False);
	    XmToggleButtonSetState (sph_w, True, False);
	}

	wants[CYLINDRICAL] = wantcyl;

	/* recompute the new size.
	 * if no change we call e_all() else leave it up to the expose.
	 */
	if (e_set_dasize(wantcyl) == 0)
	    e_all (np);
}

/* adjust the size of e_da_w for use as CYLINDRICAL or spherical viewing.
 * return 1 if the size really does change, else 0.
 */
static int
e_set_dasize(wantcyl)
int wantcyl;
{
	Dimension wid, hei;
	double a;
	    
	get_something (e_da_w, XmNwidth, (char *)&wid);
	get_something (e_da_w, XmNheight, (char *)&hei);

	a = (double)wid*(double)hei;
	if (wantcyl) {
	    /* spherical to cylindrical */
	    if ((int)wid > DisplayWidth(XtD,0)/2) {
		hei = (Dimension)((double)wid/CYLASPECT + 0.5);;
	    } else {
		wid = (Dimension)(sqrt(a*CYLASPECT) + 0.5);
		hei = (Dimension)(sqrt(a/CYLASPECT) + 0.5);
	    }
	} else {
	    /* cylindrical to spherical */
	    wid = hei = (Dimension)(sqrt(a) + 0.5);
	}

	set_something (e_da_w, XmNwidth, (char *)wid);
	set_something (e_da_w, XmNheight, (char *)hei);

	return (1);
}

/* called when Set Main or Set from Main is activated.
 * client is set to SET_MAIN or SET_FROM_MAIN.
 */
/* ARGSUSED */
static void
e_setmain_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int which = (int)client;
	Now *np = mm_get_now();

	switch (which) {
	case SET_MAIN:
	    lat = elat;
	    lng = elng;
	    redraw_screen (1);
	    break;
	case SET_FROM_MAIN:
	    e_setelatlng (lat, lng);
	    XmToggleButtonSetState (track_w, False, True);
	    e_all (np);
	    break;
	default:
	    printf ("e_setmain_cb: Bad client: %d\n", which);
	    exit (1);
	}
}

/* called when either the longitude or latitude scale moves.
 * doesn't matter which -- just update pointing direction and redraw.
 * as a convenience, also insure tracking is off now.
 */
/* ARGSUSED */
static void
e_latlong_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int lt, lg;

	XmScaleGetValue (lat_w, &lt);
	XmScaleGetValue (long_w, &lg);
	e_setelatlng (degrad(lt), degrad(-lg)); /* want +E but scale is +W */

	XmToggleButtonSetState (track_w, False, True);

	e_all (mm_get_now());
}

/* called when the Point button is selected on the roaming popup.
 * center the point of view to point_lt/point_lg.
 * as a convenience, also turn off tracking.
 */
/* ARGSUSED */
static void
e_point_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	e_setelatlng (pu_info.pu_lt, pu_info.pu_lg);
	XmToggleButtonSetState (track_w, False, True);
	e_all (mm_get_now());
}

/* make all the GCs, define all the e_*pix pixels and set the XFontStruct e_f
 *   from the track_w widget.
 * we get the color names and save the pixels in the global ecolors[] arrary.
 * if we can't get all of the special colors, just use foreground/background.
 * this is because we will use xor if if any color is missing.
 */
static void
e_init_gcs (dsp, win)
Display *dsp;
Window win;
{
	unsigned long gcm;
	XGCValues gcv;
	Pixel p;
	int all;
	int i;

	/* create the main gc. 
	 * it's GCForeground is set each time it's used.
	 */
	gcm = GCFunction;
	gcv.function = GXcopy;
	e_gc = XCreateGC (dsp, win, gcm, &gcv);

	/* get the foreground and background colors */
	get_something (e_da_w, XmNforeground, (char *)&e_fg);
	get_something (e_da_w, XmNbackground, (char *)&e_bg);

	/* collect all the other Earth colors, and check if we have them all.
	 * we also require them to all be unique.
	 */
	all = 1;
	for (i = 0; all && i < XtNumber(ecolors); i++) {
	    EColor *ecp = &ecolors[i];
	    int j;

	    if (get_color_resource (e_da_w, ecp->name, &ecp->p) < 0) {
		all = 0;
		break;
	    } else {
		for (j = i-1; j >= 0; --j) {
		    if (ecp->p == ecolors[j].p) {
			all = 0;
			break;
		    }
		}
	    }
	}

	/* make the overlay gc, e_olgc. it's GCFunction is set once (here)
	 * depending on whether we have all the desired earth colors or to xor
	 * if we must use fg/bg for everything.
	 */

	if (!all) {
	    /* couldn't get at least one color so use fg/bg and xor */
	    Pixel xorp = e_fg ^ e_bg;

	    xe_msg (
  "Can not get all unique Earth colors so using Earth.Map.fg/bg and xor.", 0);
	    for (i = 0; i < XtNumber(ecolors); i++)
		ecolors[i].p = xorp;

	    /* sun is not drawn with xor */
	    ecolors[SUNC].p = e_fg;

	    gcm = GCFunction;
	    gcv.function = GXxor;
	    e_olgc = XCreateGC (dsp, win, gcm, &gcv);
	} else {
	    /* got all the colors -- overlay can just be drawn directly */
	    gcm = GCFunction;
	    gcv.function = GXcopy;
	    e_olgc = XCreateGC (dsp, win, gcm, &gcv);
	}

	/* make the lat/long string gc and get's it's font */

	gcm = GCForeground | GCBackground | GCFont;
	get_something (track_w, XmNforeground, (char *)&p);
	gcv.foreground = p;
	gcv.background = e_bg;
	get_xmlabel_font (track_w, &e_f);
	gcv.font = e_f->fid;
	e_strgc = XCreateGC (dsp, win, gcm, &gcv);
}

/* copy the pixmap e_pm to the window of e_da_w. */
static void
e_copy_pm()
{
	Display *dsp = XtDisplay(e_da_w);
	Window win = XtWindow(e_da_w);
	unsigned int w, h;
	unsigned int wb, hb, r;

	e_getcircle (&w, &h, &r, &wb, &hb);
	XCopyArea (dsp, e_pm, win, e_gc, 0, 0, w, h, 0, 0);
}

/* right button has been hit: find the closest thing to x/y and display what we
 * know about it. if nothing is very close, at least show the lat/long.
 * always give the choice to point to the spot.
 */
/* ARGSUSED */
static void
e_popup (wid, ev, r, xb, yb, x, y)
Widget wid;
XEvent *ev;
unsigned r;
unsigned xb, yb;
int x, y;
{
	double lt, lg;
	double lmt;			/* local mean time: UT - radhr(lng) */
	double gst, lst;		/* Greenich and local sidereal times */
	double A, b, cosc, sinc;	/* see solve_sphere() */
	double cosa, B;			/* see solve_sphere() */
	double mind;
	Site *sip, *minsip;
	Trail *tp, *mintp;
	char buf[128];
	int want_altaz;
	Widget w;
	Now *np = mm_get_now();
	int i, nl;

	/* find the lat/long under the cursor.
	 * if not over the earth, forget it.
	 */
	if (!e_uncoord (r, xb, yb, x, y, &lt, &lg))
	    return;
	cosc = cos(PI/2 - lt);
	sinc = sin(PI/2 - lt);

	watch_cursor (1);

	/* make the popup if this is our first time */
	if (!pu_info.pu_w)
	    e_create_popup();

	/* find the closest site in minsip, if sites are enabled */
	minsip = 0;
	mind = 1e9;
	if (wants[SITES])
	    for (sip = sites; sip; sip = sip->si_nxt) {
		double d;
		A = sip->si_lng - lg;
		b = PI/2 - sip->si_lat;
		solve_sphere (A, b, cosc, sinc, &cosa, &B);
		d = acos (cosa);
		if (d < mind) {
		    minsip = sip;
		    mind = d;
		}
	    }

	/* now see if there are any history entries even closer.
	 * if so, put it in mintp.
	 * don't do this if the object is not currently being shown.
	 */
	mintp = 0;
	if (wants[OBJ])
	    for (tp = trails; tp && tp < &trails[ntrails]; tp++) {
		double d;
		A = tp->t_sublng - lg;
		b = PI/2 - tp->t_sublat;
		solve_sphere (A, b, cosc, sinc, &cosa, &B);
		d = acos (cosa);
		if (d < mind) {
		    mintp = tp;
		    mind = d;
		}
	    }

	/* build the popup */

	/* first unmanage all the labels -- we'll turn on what we want */
	for (i = 0; i < MAXPUL; i++)
	    XtUnmanageChild (pu_info.pu_labels[i]);
	nl = 0;

	want_altaz = 0;
	if (mind <= MAXDIST) {
	    /* found something, question is: a trailed object or a site?
	     * check them in the opposite order than which they were searched.
	     */
	    if (mintp) {
		/* a trail item was closest.
		 * put the name, time, lat and lng in the popup.
		 */
		w = pu_info.pu_labels[nl++];
		set_xmstring (w, XmNlabelString, mintp->t_obj.o_name);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "UT Date: ");
		fs_date (buf+strlen(buf), mjd_day(mintp->t_now.n_mjd));
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "UT Time: ");
		fs_time (buf+strlen(buf), mjd_hr(mintp->t_now.n_mjd));
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "LMT: ");
		lmt = mjd_hr(mintp->t_now.n_mjd) + radhr(mintp->t_sublng);
		range (&lmt, 24.0);
		fs_time (buf+strlen(buf), lmt);
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "LST: ");
		utc_gst (mjd_day(mintp->t_now.n_mjd),
					    mjd_hr(mintp->t_now.n_mjd), &gst);
		lst = gst + radhr(mintp->t_sublng);
		range (&lst, 24.0);
		fs_time (buf+strlen(buf), lst);
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "Sub Lat: ");
		fs_dms_angle (buf+strlen(buf), mintp->t_sublat);
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "Sub Long: ");
		fs_dms_angle (buf+strlen(buf), -mintp->t_sublng); /* want +W */
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		/* save lat/long for Point and if want to display alt/az */
		pu_info.pu_lt = mintp->t_sublat;
		pu_info.pu_lg = mintp->t_sublng;
	    } else if (minsip) {
		/* a site entry was closest.
		 * put it's name, lat/long and alt/az in the popup.
		 */

		want_altaz = 1;

		w = pu_info.pu_labels[nl++];
		set_xmstring (w, XmNlabelString, minsip->si_name);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "LMT: ");
		lmt = mjd_hr(mjd) + radhr(minsip->si_lng);
		range (&lmt, 24.0);
		fs_time (buf+strlen(buf), lmt);
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "LST: ");
		utc_gst (mjd_day(mjd), mjd_hr(mjd), &gst);
		lst = gst + radhr(minsip->si_lng);
		range (&lst, 24.0);
		fs_time (buf+strlen(buf), lst);
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "Lat: ");
		fs_dms_angle (buf+strlen(buf), minsip->si_lat);
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		w = pu_info.pu_labels[nl++];
		(void) sprintf (buf, "Long: ");
		fs_dms_angle (buf+strlen(buf), -minsip->si_lng); /* want +W */
		set_xmstring (w, XmNlabelString, buf);
		XtManageChild (w);

		/* save lat/long for Point and if want to display alt/az */
		pu_info.pu_lt = minsip->si_lat;
		pu_info.pu_lg = minsip->si_lng;
	    } else {
		printf ("e_popup: unknown closest! mind = %g\n", mind);
		exit (1);
	    }
	} else {
	    /* nothing was close enough.
	     * just put the lat/long and alt/az of the cursor loc in the popup.
	     */

	    want_altaz = 1;

	    w = pu_info.pu_labels[nl++];
	    (void) sprintf (buf, "LMT: ");
	    lmt = mjd_hr(mjd) + radhr(lg);
	    range (&lmt, 24.0);
	    fs_time (buf+strlen(buf), lmt);
	    set_xmstring (w, XmNlabelString, buf);
	    XtManageChild (w);

	    w = pu_info.pu_labels[nl++];
	    (void) sprintf (buf, "LST: ");
	    utc_gst (mjd_day(mjd), mjd_hr(mjd), &gst);
	    lst = gst + radhr(lg);
	    range (&lst, 24.0);
	    fs_time (buf+strlen(buf), lst);
	    set_xmstring (w, XmNlabelString, buf);
	    XtManageChild (w);

	    w = pu_info.pu_labels[nl++];
	    (void) sprintf (buf, "Lat: ");
	    fs_dms_angle (buf+strlen(buf), lt);
	    set_xmstring (w, XmNlabelString, buf);
	    XtManageChild (w);

	    w = pu_info.pu_labels[nl++];
	    (void) sprintf (buf, "Long: ");
	    fs_dms_angle (buf+strlen(buf), -lg);	/* want +W */
	    set_xmstring (w, XmNlabelString, buf);
	    XtManageChild (w);

	    /* save lat/long for Point and if want to display alt/az */
	    pu_info.pu_lt = lt;
	    pu_info.pu_lg = lg;
	}

	/* if the Object is enabled also display alt/az of the object from
	 * this location in the popup.
	 */
	if (want_altaz && wants[OBJ]) {
	    Now now_here;
	    Obj obj_here;

	    (void) memcpy (&now_here, &trails[ntrails-1].t_now, sizeof (Now));
	    now_here.n_lat = pu_info.pu_lt;
	    now_here.n_lng = pu_info.pu_lg;
	    (void) memcpy (&obj_here, &trails[ntrails-1].t_obj, sizeof (Obj));
	    (void) obj_cir (&now_here, &obj_here);

	    w = pu_info.pu_labels[nl++];
	    (void) sprintf (buf, "Obj Alt: ");
	    fs_pangle (buf+strlen(buf), obj_here.s_alt);
	    set_xmstring (w, XmNlabelString, buf);
	    XtManageChild (w);

	    w = pu_info.pu_labels[nl++];
	    (void) sprintf (buf, "Obj Az: ");
	    fs_pangle (buf+strlen(buf), obj_here.s_az);
	    set_xmstring (w, XmNlabelString, buf);
	    XtManageChild (w);
	}

	XmMenuPosition (pu_info.pu_w, (XButtonPressedEvent *)ev);
	XtManageChild (pu_info.pu_w);

	watch_cursor(0);
}

/* create the popup
 * save all its widget in the pu_info struct.
 */
static void
e_create_popup()
{
	Widget w;
	Arg args[20];
	int n;
	int i;

	/* make the popup shell which includes a handy row/column */
	n = 0;
	XtSetArg (args[n], XmNisAligned, True); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	pu_info.pu_w = XmCreatePopupMenu (e_da_w, "EPopup", args, n);

	/* add the max labels we'll ever need -- all unmanaged for now. */
	for (i = 0; i < MAXPUL; i++) {
	    n = 0;
	    pu_info.pu_labels[i] = XmCreateLabel (pu_info.pu_w, "EPUL",args,n);
	}

	/* and add the separator and the "Point" button.
	 * we always wants these so manage them now.
	 */
	w = XmCreateSeparator (pu_info.pu_w, "PUSep", args, 0);
	XtManageChild (w);
	w = XmCreatePushButton (pu_info.pu_w, "Point", args, 0);
	XtAddCallback (w, XmNactivateCallback, e_point_cb, 0);
	XtManageChild (w);
}

/* display the lat/long/lmt/lst under the cursor.
 * we are given the radius and x/y of the location, the pointer event and
 * the total window size.
 */
static void
e_track_latlng (w, ev, r, xb, yb, x, y)
Widget w;
XEvent *ev;
unsigned r;
unsigned xb, yb;
{
	static dirty;
	double lt, lg;
	int wide = 2*(r+xb);
	int hi = 2*(r+yb);

	if (ev->type != ButtonRelease && e_uncoord (r, xb, yb, x, y, &lt, &lg)){
	    XCharStruct all;
	    Now *np = mm_get_now();
	    Window win = XtWindow(w);
	    char lst_buf[64], lmt_buf[64], lng_buf[64], lat_buf[64];
	    double lst, gst, lmt;
	    int dir, asc, des;
	    int len;

	    (void) strcpy (lat_buf, "Lat: ");
	    fs_dms_angle (lat_buf+5, lt);

	    (void) strcpy (lng_buf, "Long: ");
	    fs_dms_angle (lng_buf+6, -lg);	/* want +W */

	    lmt = mjd_hr(mjd) + radhr(lg);
	    range (&lmt, 24.0);
	    (void) strcpy (lmt_buf, "LMT: ");
	    fs_time (lmt_buf+5, lmt);

	    utc_gst (mjd_day(mjd), mjd_hr(mjd), &gst);
	    lst = gst + radhr(lg);
	    range (&lst, 24.0);
	    (void) strcpy (lst_buf, "LST: ");
	    fs_time (lst_buf+5, lst);

	    len = strlen (lat_buf);
	    XTextExtents (e_f, lat_buf, len, &dir, &asc, &des, &all);
	    XDrawImageString (XtD, win, e_strgc, 0, asc, lat_buf, len); 

	    len = strlen (lng_buf);
	    XTextExtents (e_f, lng_buf, len, &dir, &asc, &des, &all);
	    XDrawImageString (XtD, win, e_strgc, wide-all.rbearing, asc,
								lng_buf, len);

	    len = strlen (lmt_buf);
	    XTextExtents (e_f, lmt_buf, len, &dir, &asc, &des, &all);
	    XDrawImageString (XtD, win, e_strgc, 0, hi-des, lmt_buf, len);

	    len = strlen (lst_buf);
	    XTextExtents (e_f, lst_buf, len, &dir, &asc, &des, &all);
	    XDrawImageString (XtD, win, e_strgc, wide-all.rbearing, hi-des,
								lst_buf, len);
	    
	    dirty = 1;
	} else if (dirty) {
	    e_copy_pm();
	    dirty = 0;
	}
}

/* set the elat/selat/celat/elng values, taking care to put them into ranges
 * -PI/2 .. elat .. PI/2 and -PI .. elng .. PI.
 */
static void
e_setelatlng (lt, lg)
double lt, lg;
{
	elat = lt;
	selat = sin(elat);
	celat = cos(elat);
	elng = lg;
	range (&elng, 2*PI);
	if (elng > PI)
	    elng -= 2*PI;
}

/* return the long and lat of the subobject point (lat/long of where the object
 * is directly overhead) at np.
 * both are in rads, lat is +N, long is +E.
 */
static void
e_subobject(np, dbidx, latp, longp)
Now *np;
int dbidx;
double *latp, *longp;
{
	Obj *op;

	op = db_basic (dbidx);

	if (op->type == EARTHSAT) {
	    /* these are all ready to go */
	    *latp = op->s_sublat;
	    *longp = op->s_sublng;
	} else {
	    double gst;
	    double ra, dec;
	    ra = op->s_ra;
	    dec = op->s_dec;
	    if (epoch != EOD)
		precess (epoch, mjd, &ra, &dec);

	    *latp = dec;

	    utc_gst (mjd_day(mjd), mjd_hr(mjd), &gst);
	    *longp = ra - hrrad(gst);	/* remember: we want +E */
	    range (longp, 2*PI);
	    if (*longp > PI)
		*longp -= 2*PI;
	}
}

/* show the stats for the latest object in the trails list.
 * if it's of type EARTHSAT then display its s_ fields;
 * otherwise turn off the display of these items.
 */
static void
e_show_esat_stats ()
{
	ESatStatTable *stp;
	int unitspref;
	Obj *op = &trails[ntrails-1].t_obj;

	if (op->type != EARTHSAT) {
	    XtUnmanageChild (estattable[ES_ELEV].lw);
	    XtUnmanageChild (estattable[ES_ELEV].pbw);
	    XtUnmanageChild (estattable[ES_RANGE].lw);
	    XtUnmanageChild (estattable[ES_RANGE].pbw);
	    XtUnmanageChild (estattable[ES_RANGER].lw);
	    XtUnmanageChild (estattable[ES_RANGER].pbw);
	    XtUnmanageChild (estattable[ES_ECLIPSED].lw);
	    XtUnmanageChild (estattable[ES_ECLIPSED].pbw);
	    return;
	} else {
	    XtManageChild (estattable[ES_ELEV].lw);
	    XtManageChild (estattable[ES_ELEV].pbw);
	    XtManageChild (estattable[ES_RANGE].lw);
	    XtManageChild (estattable[ES_RANGE].pbw);
	    XtManageChild (estattable[ES_RANGER].lw);
	    XtManageChild (estattable[ES_RANGER].pbw);
	    XtManageChild (estattable[ES_ECLIPSED].lw);
	    XtManageChild (estattable[ES_ECLIPSED].pbw);
	}

	unitspref = pref_get (PREF_UNITS);

	stp = &estattable[ES_ELEV];
	if (stp->enumcode != ES_ELEV) {
	    printf ("estattable order is wrong\n");
	    exit(1);
	}
	if (unitspref == PREF_ENGLISH) {
	    f_showit (stp->lw, "Elevation(mi):");
	    f_double (stp->pbw, "%6.0f", op->s_elev*FTPM/5280.0);
	} else {
	    f_showit (stp->lw, "Elevation(km):");
	    f_double (stp->pbw, "%6.0f", op->s_elev/1000.0);
	}

	stp = &estattable[ES_RANGE];
	if (stp->enumcode != ES_RANGE) {
	    printf ("estattable order is wrong\n");
	    exit(1);
	}
	if (unitspref == PREF_ENGLISH) {
	    f_showit (stp->lw, "Range(mi):");
	    f_double (stp->pbw, "%6.0f", op->s_range*FTPM/5280.0);
	} else {
	    f_showit (stp->lw, "Range(km):");
	    f_double (stp->pbw, "%6.0f", op->s_range/1000.0);
	}

	stp = &estattable[ES_RANGER];
	if (stp->enumcode != ES_RANGER) {
	    printf ("estattable order is wrong\n");
	    exit(1);
	}
	if (unitspref == PREF_ENGLISH) {
	    f_showit (stp->lw, "Range Rate(f/s):");
	    f_double (stp->pbw, "%6.0f", op->s_rangev*FTPM);
	} else {
	    f_showit (stp->lw, "Range Rate(m/s):");
	    f_double (stp->pbw, "%6.0f", op->s_rangev);
	}

	stp = &estattable[ES_ECLIPSED];
	if (stp->enumcode != ES_ECLIPSED) {
	    printf ("estattable order is wrong\n");
	    exit(1);
	}
	f_double (stp->pbw, "%.1g", op->s_eclipsed ? 1.0 : 0.0);
}

/* given:
 *   the current viewpoint coords (from globals selat/celat/elng)
 *   radius and borders of circle in current window
 *   latitude and longitude of a point p: pl, pL;
 * find:
 *   x and y coords of point on screen
 * return 1 if the point is visible or 0 if it's over the limb.
 */   
static int
e_coord (r, wb, hb, pl, pL, xp, yp)
unsigned r;		/* radius of drawing surface circle */
unsigned wb, hb;	/* width/height borders between circle and edges */
double pl, pL;		/* point p lat, rads, +N; long, rads, +E */
short *xp, *yp;		/* projected location onto drawing surface */
{
	if (wants[CYLINDRICAL]) {
	    unsigned w = 2*(r + wb);
	    unsigned h = 2*(r + hb);
	    double dL;

	    *yp = (int)floor(h/2.0 * (1.0 - sin(pl)) + 0.5);

	    dL = pL - elng;
	    range(&dL, 2*PI);
	    if (dL > PI)  dL -= 2*PI;
	    *xp = (int)floor(w/2.0 * (1.0 + dL/PI) + 0.5);

	    return (1);
	} else {
	    double sR, cR;	/* R is radius from viewpoint to p */
	    double A, sA,cA;/* A is azimuth of p as seen from viewpoint */

	    solve_sphere (pL - elng, PI/2 - pl, selat, celat, &cR, &A);
	    sR = sqrt(1.0 - cR*cR);
	    sA = sin(A);
	    cA = cos(A);

	    *xp = wb + r + (int)floor(r*sR*sA + 0.5);
	    *yp = hb +r - (int)floor(r*sR*cA + 0.5);

	    return (cR > 0);
	}
}

/* draw all the objects in the trails list onto e_pm relative to elat/elng.
 * latest object drawn with full bullseyes, older ones just with crosshair.
 */
static void
e_drawobjects (r, wb, hb)
unsigned r;		/* Earth circle radius, pixels */
unsigned wb, hb;	/* width and height borders, pixels */
{
	Trail *tp, *ftp, *ltp;	/* working, first and last */

	/* only show the last entry if trails is turned off.
	 * (they are not actually forgotten until the next update)
	 */
	ltp = trails + ntrails - 1;
	if (wants[TRAILS])
	    ftp = trails;
	else
	    ftp = ltp;

	XSetForeground (XtD, e_olgc, ecolors[OBJC].p);
	for (tp = ftp; tp <= ltp; tp++) {
	    double el;
	    el = tp->t_obj.type == EARTHSAT ? tp->t_obj.s_elev : 1e9;
	    if (tp == ltp)
		e_drawfootprint (r, wb, hb, tp->t_sublat, tp->t_sublng, el);
	    else
		e_drawcross (r, wb, hb, tp->t_sublat, tp->t_sublng, CROSSH);
	}
}

/* draw circles onto e_pm within circle of radius r and width and height
 * borders wb and hb showing where the satellite is seen as being 0, 30 and 60
 * degrees altitude above horizon as well.
 * also draw a crosshair at the actual location.
 * subobject location.
 */
static void
e_drawfootprint (r, wb, hb, slat, slng, el)
unsigned r, wb, hb;	/* radius and width/height borders of circle to use */
double slat, slng;	/* satellite's lat and lng */
double el;		/* satellite's elevation above surface, m */
{
	double rad;	/* viewing altitude radius, rads */

	e_viewrad (el, degrad(0.0), &rad);
	e_drawcircle (r, wb, hb, slat, slng, rad);
	e_viewrad (el, degrad(30.0), &rad);
	e_drawcircle (r, wb, hb, slat, slng, rad);
	e_viewrad (el, degrad(60.0), &rad);
	e_drawcircle (r, wb, hb, slat, slng, rad);
	e_drawcross (r, wb, hb, slat, slng, CROSSH);
}

/* assuming the current vantage point of elat/elng, draw a circle around the
 *   given lat/long point of given angular radius.
 * all angles in rads.
 * we do it by sweeping an arc from the given location and collecting the end
 *   points. beware going over the horizon or wrapping around the edge.
 */
static void
e_drawcircle (r, wb, hb, slat, slng, rad)
unsigned r, wb, hb;	/* radius and width/height borders of circle to use */
double slat, slng;	/* lat/long of object */
double rad;		/* angular radius of circle to draw */
{
#define	NVCIRSEGS	75
	XPoint xp[NVCIRSEGS+1];
	double cosc = cos(PI/2 - slat);	/* cos/sin of N-pole-to-slat angle */
	double sinc = sin(PI/2 - slat);
	double b = rad;			/* sat-to-arc angle */
	double A;			/* subobject azimuth */
	double cosa;			/* cos of pole-to-arc angle */
	double B;			/* subobj-to-view angle from pole */
	int nxp = 0;
	int w = 2*(r+wb);
	int vis;
	int i;

	/* use the overlay gc, wide lines, the OBJC color */
	XSetLineAttributes (XtD, e_olgc, WIDE_LW, LINE_ST, CAP_ST, JOIN_ST);
	XSetForeground (XtD, e_olgc, ecolors[OBJC].p);

	for (i = 0; i <= NVCIRSEGS; i++) {
	    short x, y;
	    A = 2*PI/NVCIRSEGS * i;
	    solve_sphere (A, b, cosc, sinc, &cosa, &B);
	    vis = e_coord (r, wb, hb, PI/2-acos(cosa), B+slng, &x, &y);

	    nxp = add_to_polyline (xp, XtNumber(xp), i,vis,nxp,NVCIRSEGS,w,x,y);
	}
}

/* draw a little crosshair or plussign at the given location with the e_olgc */
static void
e_drawcross (r, wb, hb, lt, lg, style)
unsigned r, wb, hb;	/* radius and width/height borders of circle to use */
double lt, lg;		/* desired center location */
int style;		/* CROSSH or PLUSS */
{
	double lats[4], lngs[4];	/* lats and longs of endpoints */
	XSegment xs[4];			/* one for each cardinal direction */
	double a, cosa;			/* north-to-cross from view */
	double B;			/* cross-width from north */
	short sx, sy;			/* x and y of the center */
	int w = 2*(r+wb);
	int linew;
	int nxs;
	int vis;
	int xwrap;
	int i;

	/* find location of center of cross-hair */
	vis = e_coord (r, wb, hb, lt, lg, &sx, &sy);
	if (!vis)
	    return;	/* center is not visible so forget the rest */

	/* find longitude sweep to produce given e/w arc */

	switch (style) {
	case CROSSH:
	    solve_sphere (PI/4, CHLEN, sin(lt), cos(lt), &cosa, &B);
	    a = acos(cosa);
	    lats[0] = PI/2-a;		lngs[0] = lg+B;
	    lats[1] = PI/2-a;		lngs[1] = lg-B;
	    solve_sphere (3*PI/4, CHLEN, sin(lt), cos(lt), &cosa, &B);
	    a = acos(cosa);
	    lats[2] = PI/2-a;		lngs[2] = lg+B;
	    lats[3] = PI/2-a;		lngs[3] = lg-B;
	    linew = WIDE_LW;
	    break;
	case PLUSS:
	    solve_sphere (PI/2, PLLEN, sin(lt), cos(lt), &cosa, &B);
	    a = acos(cosa);
	    lats[0] = PI/2-a;		lngs[0] = lg+B;
	    lats[1] = PI/2-a;		lngs[1] = lg-B;
	    lats[2] = lt-PLLEN;		lngs[2] = lg;
	    if (lats[2] < -PI/2) {
		lats[2] = -PI - lats[2];lngs[2] += PI;	/* went under S pole */
	    }
	    lats[3] = lt+PLLEN;		lngs[3] = lg;
	    if (lats[3] > PI/2) {
		lats[3] = PI - lats[3];	lngs[3] += PI;	/* went over N pole */
	    }
	    linew = NARROW_LW;
	    break;
	default:
	    printf ("e_drawcross: bad style: %d\n", style);
	    exit (1);
	}

	nxs = 0;
	for (i = 0; i < 4; i++) {
	    short x, y;
	    vis = e_coord (r, wb, hb, lats[i], lngs[i], &x, &y);

	    if (wants[CYLINDRICAL] && (sx - x > w/2 || x - sx > w/2))
		xwrap = 1;
	    else
		xwrap = 0;

	    if (vis && !xwrap) {
		xs[nxs].x1 = x;
		xs[nxs].y1 = y;
		xs[nxs].x2 = sx;
		xs[nxs].y2 = sy;
		nxs++;
	    }
	}

	if (nxs > 0) {
	    XSetLineAttributes (XtD, e_olgc, linew, LINE_ST, CAP_ST, JOIN_ST);
	    XDrawSegments (XtD, e_pm, e_olgc, xs, nxs);
	}
}

/* draw the lat/long grid lines */
static void
e_drawgrid (r, wb, hb)
unsigned r, wb, hb;	/* radius and width/height borders of circle to use */
{
	XPoint xp[NLATLNG+1];
	int w = 2*(r+wb);
	int i, j;

	/* use the overlay GC, narrow lines, GRIDC color */
	XSetLineAttributes (XtD, e_olgc, NARROW_LW, LINE_ST, CAP_ST, JOIN_ST);
	XSetForeground (XtD, e_olgc, ecolors[GRIDC].p);

	/* draw each line of constant longitude */
	for (i = 0; i < 360/LNGSTEP; i++) {
	    double lg = i * degrad(LNGSTEP);
	    int nxp = 0;
	    for (j = 0; j <= NLATLNG; j++) {
		short x, y;
		double lt = degrad (90 - j*180.0/NLATLNG);
		int vis = e_coord (r, wb, hb, lt, lg, &x, &y);
		nxp = add_to_polyline (xp,XtNumber(xp),j,vis,nxp,NLATLNG,w,x,y);
	    }
	}

	/* draw each line of constant latitude -- beware of x wrap */
	for (i = 1; i < 180/LATSTEP; i++) {
	    double lt = degrad (90 - i*LATSTEP);
	    int nxp = 0;
	    for (j = 0; j <= NLATLNG; j++) {
		short x, y;
		double lg = j * degrad(360.0/NLATLNG);
		int vis = e_coord (r, wb, hb, lt, lg, &x, &y);
		nxp = add_to_polyline (xp,XtNumber(xp),j,vis,nxp,NLATLNG,w,x,y);
	    }
	}
}

/* given radius and size of window and X-coord x/y, return current Earth
 * lat/long.
 * return 1 if inside the circle, else 0.
 */
static int
e_uncoord (r, xb, yb, x, y, ltp, lgp)
unsigned int r;		/* radius of earth limb, pixels */
unsigned xb, yb;	/* borders around circle */
int x, y;		/* X-windows coords of loc */
double *ltp, *lgp;	/* resulting lat/long, rads */
{
	if (wants[CYLINDRICAL]) {
	    int maxx = 2*(r + xb)-1;
	    int maxy = 2*(r + yb)-1;

	    if (x < 0 || x > maxx || y < 0 || y > maxy)
		return (0);

	    *ltp = asin (1.0 - 2.0*y/maxy);

	    *lgp = 2*PI*(x - maxx/2)/maxx + elng;
	    if (*lgp >  PI) *lgp -= 2*PI;
	    if (*lgp < -PI) *lgp += 2*PI;

	    return (1);
	} else {
	    double X, Y;/* x/y but signed from center +x right +y up, pixels*/
	    double R;	/* pixels x/y is from center */
	    double b;	/* angle from viewpoint to x/y */
	    double A;	/* angle between N pole and x/y as seen from viewpt */
	    double a, ca;	/* angle from N pole to x/y */
	    double B;	/* angle from viewpoint to x/y as seen from N pole */

	    X = x - (int)(r + xb);
	    Y = (int)(r + yb) - y;
	    R = sqrt (X*X + Y*Y);
	    if (R >= r)
		return (0);

	    if (R < 1.0) {
		*ltp = elat;
		*lgp = elng;
	    } else {
		b = asin (R/r);
		A = atan2 (X, Y);
		solve_sphere (A, b, selat, celat, &ca, &B);
		a = acos (ca);
		*ltp = PI/2 - a;
		*lgp = elng + B;
		range (lgp, 2*PI);
		if (*lgp > PI)
		    *lgp -= 2*PI;
	    }
	    
	    return (1);
	}
}

/* draw everything: fields (always) and the picture (if managed) */
static void
e_all (np)
Now *np;
{
	double objlat, objlng;

	watch_cursor(1);

	if (!trails) {
	    printf ("e_all(): no trails\n");
	    exit (1);
	}

	/* put up the current time */
	timestamp (np, e_dt_w);
	timestamp (np, e_sdt_w);

	/* get the current location of the object */
	objlat = trails[ntrails-1].t_sublat;
	objlng = trails[ntrails-1].t_sublng;

	/* point at it if tracking is on */
	if (wants[TRACK])
	    e_setelatlng (objlat, objlng);

	/* update the scale -- remember we want +W longitude */
	XmScaleSetValue (lat_w, (int)floor(raddeg(elat)+0.5));
	XmScaleSetValue (long_w, (int)floor(raddeg(-elng)+0.5));

	/* display the lat and long of the current object.
	 * N.B. we want to display longitude as +W in range of -PI..+PI
	 */
	f_dms_angle (estattable[ES_SUBLONG].pbw, -objlng);
	f_dms_angle (estattable[ES_SUBLAT].pbw, objlat);

	/* show the other stats too if applicable */
	e_show_esat_stats ();

	/* display the picture if we are up now for sure now.
	 * N.B. don't just use whether eform_w is managed: if we are looping
	 * and get managed drawing occurs before the first expose makes the pm.
	 */
	if (e_pm) {
	    e_map (np);
	    e_copy_pm();
	}

	watch_cursor(0);
}

/* find size and borders of circle for drawing earth */
static void
e_getcircle (wp, hp, rp, xbp, ybp)
unsigned int *wp, *hp;          /* overall width and height */
unsigned int *rp;               /* circle radius */
unsigned int *xbp, *ybp;        /* x and y border */
{
	Display *dsp = XtDisplay(e_da_w);
	Window win = XtWindow(e_da_w);
	Window root;
	int x, y;
	unsigned int w, h;
	unsigned int bw, d;

	XGetGeometry (dsp, win, &root, &x, &y, wp, hp, &bw, &d);

	w = *wp/2;
	h = *hp/2;
	*rp = w > h ? h : w;    /* radius */
	*xbp = w - *rp;         /* x border */
	*ybp = h - *rp;         /* y border */

}

/* draw everything centered at elat/elng onto e_pm.
 */
static void
e_map (np)
Now *np;
{
	Display *dsp = XtDisplay(e_da_w);
	unsigned int w, h, r, wb, hb;

	/* get size of the rendering area */
	e_getcircle (&w, &h, &r, &wb, &hb);

	/* clear the window */
	XSetForeground (dsp, e_gc, e_bg);
	XFillRectangle (dsp, e_pm, e_gc, 0, 0, w, h);

	/* draw sunlit portion of earth.
	 * not drawn with xor so put it first.
	 */
	if (wants[SUNLIGHT])
	    e_sunlit (np, r, wb, hb);

	/* draw coord grid */
	if (wants[GRID])
	    e_drawgrid (r, wb, hb);

	/* draw each continent border */
	e_drawcontinents (r, wb, hb);

	/* draw each site */
	if (wants[SITES])
	    e_drawsites (r, wb, hb);

	/* draw each object */
	if (wants[OBJ])
	    e_drawobjects (r, wb, hb);

	/* mark the mainmenu location */
	e_mainmenuloc (np, r, wb, hb);

	/* mark any solar eclipse location */
	e_soleclipse (np, r, wb, hb);
}

/* draw the portion of the earth lit by the sun */
static void
e_sunlit (np, r, wb, hb)
Now *np;		/* circumstances */
unsigned int r, wb, hb;	/* circle radius, width and height borders in pixmap */
{
	if (wants[CYLINDRICAL])
	    e_msunlit (np, r, wb, hb);
	else
	    e_ssunlit (np, r, wb, hb);
}

/* draw the sunlit portion of the Earth in cylindrical projection.
 */
static void
e_msunlit (np, r, wb, hb)
Now *np;		/* circumstances */
unsigned int r, wb, hb;	/* circle radius, width and height borders in pixmap */
{
#define NMPTS	59	/* n segments -- larger is finer but slower */
	Display *dsp = XtDisplay(e_da_w);
	XPoint xp[NMPTS+4];	/* extra room for top or bottom box */
	double sslat, sslong;	/* subsolar lat/long, rads, +N +E */
	double az0, daz;	/* initial and delta az so we move +x */
	double ssl, csl;
	int w = 2*(r+wb);
	int h = 2*(r+hb);
	int wrapn, i;

	e_subobject (np, SUN, &sslat, &sslong);
	ssl = sin(sslat);
	csl = cos(sslat);

	if (sslat < 0) {
	    az0 = 0;
	    daz = 2*PI/NMPTS;
	} else {
	    az0 = PI;
	    daz = -2*PI/NMPTS;
	}

	/* fill in the circle of NMPTS points that lie PI/2 from the
	 *   subsolar point and projecting from the viewpoint.
	 * when wrap, shuffle points up so final polygon doesn't wrap.
	 */
	XSetForeground (dsp, e_gc, ecolors[SUNC].p);
	wrapn = 0;
	for (i = 0; i < NMPTS; i++) {
	    double A = az0 + i*daz;		/* azimuth of p from subsol */
	    double sA = sin(A), cA = cos(A);
	    double plat = asin (csl*cA);	/* latitude of p */
	    double plong = atan2(sA, -ssl*cA) + sslong;	/* long of p */
	    XPoint *xpp = &xp[i-wrapn];
	    short x, y;

	    (void) e_coord (r, wb, hb, plat, plong, &x, &y);

	    xpp->x = x;
	    xpp->y = y;

	    if (wrapn == 0 && i > 0 && x < xpp[-1].x) {
		/* when wrap copy points so far to end and this one is first.
		 * N.B. go backwards in case we are over half way.
		 */
		for ( ; wrapn < i; wrapn++)
		    xp[NMPTS-1-wrapn] = xp[i-1-wrapn];
		xp[0].x = x;
		xp[0].y = y;
	    }
	}

	/* complete polygon across bottom if sun is below equator, or vv */
	if (sslat < 0) {
	    xp[i].x = w-1;    xp[i].y = xp[i-1].y;  i++;    /* right */
	    xp[i].x = w-1;    xp[i].y = h-1;        i++;    /* down */
	    xp[i].x = 0;      xp[i].y = h-1;        i++;    /* left */
	    xp[i].x = 0;      xp[i].y = xp[0].y;    i++;    /* up */
	} else {
	    xp[i].x = w-1;    xp[i].y = xp[i-1].y;  i++;    /* right */
	    xp[i].x = w-1;    xp[i].y = 0;          i++;    /* up */
	    xp[i].x = 0;      xp[i].y = 0;          i++;    /* left */
	    xp[i].x = 0;      xp[i].y = xp[0].y;    i++;    /* down */
	}

	XFillPolygon (dsp, e_pm, e_gc, xp, i, Complex, CoordModeOrigin);
}

/* draw the solid gibbous or crescent sun-lit portion of the Earth in spherical
 * projection.
 */
static void
e_ssunlit (np, r, wb, hb)
Now *np;		/* circumstances */
unsigned int r, wb, hb;	/* circle radius, width and height borders in pixmap */
{
#define NSPTS	97	/* number of gibbous/crescent angles.
			 * N.B. be sure loop below never generates 90 or 270
			 */
	Display *dsp = XtDisplay(e_da_w);
	XPoint xp[NSPTS];
	double sslat, sslong;	/* subsolar lat/long, rads, +N +E */
	double ssl, csl;	/* sin/cos of sslat */
	double A;		/* angle from pole to point as seen from vp*/
	short x, y;
	int vis;
	int i;

	/* start with the lit semicircle. */
	e_subobject (np, SUN, &sslat, &sslong);
	ssl = sin(sslat);
	csl = cos(sslat);
	vis = e_coord (r, wb, hb, sslat, sslong, &x, &y);
	x = x - (r + wb);	/* pixels right of center */
	y = (r + hb) - y;	/* pixels up from center */
	if (y == 0)
	    A = x >= 0 ? PI/2 : -PI/2;
	else
	    A = atan2 ((double)x, (double)y);
	range (&A, 2*PI);
	XSetForeground (dsp, e_gc, ecolors[SUNC].p);
	XFillArc (dsp, e_pm, e_gc, wb, hb, 2*r, 2*r, (int)raddeg(2*PI-A)*64,
									180*64);

	/* add or subtract from the semicircle to form gibbous or crescent.
	 * do it by finding the circle of points that lie PI/2 from the
	 *   subsolar point and projecting from the viewpoint.
	 * if subsolar point is over the horizon we draw bg and form a crescent;
	 *   else we draw in fg and form a gibbous.
	 */
	XSetForeground (dsp, e_gc, vis ? ecolors[SUNC].p : e_bg);
	for (i = 0; i < NSPTS; i++) {
	    short x, y;
	    double T = degrad(i*360.0/NSPTS);	/* pole to p as seen frm subso*/
	    double sT = sin(T), cT = cos(T);
	    double plat = asin (csl*cT);	/* latitude of p */
	    double plong = atan2(sT, -ssl*cT) + sslong;	/* long of p */
	    (void) e_coord (r, wb, hb, plat, plong, &x, &y);
	    xp[i].x = x;
	    xp[i].y = y;
	}
	XFillPolygon (dsp, e_pm, e_gc, xp, NSPTS, Convex, CoordModeOrigin);
}

/* draw each continent border */
static void
e_drawcontinents (r, wb, hb)
unsigned r, wb, hb;
{
#define	PTCACHE		512		/* number of XPoints to cache */
	Display *dsp = XtDisplay(e_da_w);
	XPoint xp[PTCACHE];
	MRegion *rp;
	int w = 2*(r+wb);

	/* use the overlay GC, wide lines and the BORDERC color */
	XSetLineAttributes (dsp, e_olgc, WIDE_LW, LINE_ST, CAP_ST, JOIN_ST);
	XSetForeground (dsp, e_olgc, ecolors[BORDERC].p);

	for (rp = ereg; rp < ereg + nereg; rp++) {
	    int n = rp->nmcp;
	    int nxp = 0;
	    int i;

	    /* draw the region -- including closure at the end */
	    for (i = 0; i <= n; i++) {
		MCoord *cp = rp->mcp + (i%n);
		short x, y;
		int vis;

		vis = e_coord (r, wb, hb, degrad(cp->lt/100.0),
						degrad(cp->lg)/100.0, &x, &y);
		nxp = add_to_polyline (xp, XtNumber(xp), i, vis, nxp, n, w,x,y);
	    }
	}
}

/* draw each site */
static void
e_drawsites (r, wb, hb)
unsigned r, wb, hb;
{
#define	NPCACHE		1024		/* number of XPoints to cache */
	Display *dsp = XtDisplay(e_da_w);
	XPoint xps[NPCACHE];
	Site *sip;
	int nxp;

	/* use the overlay GC and the SITEC color */
	XSetForeground (dsp, e_olgc, ecolors[SITEC].p);

	nxp = 0;
	for (sip = sites; sip; sip = sip->si_nxt) {
	    short x, y;
	    int vis;

	    vis = e_coord (r, wb, hb, sip->si_lat, sip->si_lng, &x, &y);
	    if (vis) {
		/* show each site as a little square */
		xps[nxp].x = x;		xps[nxp].y = y;		nxp++;
		xps[nxp].x = x+1;	xps[nxp].y = y;		nxp++;
		xps[nxp].x = x;		xps[nxp].y = y+1;	nxp++;
		xps[nxp].x = x+1;	xps[nxp].y = y+1;	nxp++;
	    }
	    if (nxp > XtNumber(xps)-4 || sip->si_nxt == NULL) {
		if (nxp > 0)
		    XDrawPoints (dsp, e_pm, e_olgc, xps, nxp, CoordModeOrigin);
		nxp = 0;
	    }
	}
}

/* this function just captures the code we found building over and over to draw
 * polylines allowing for wrapping on left and right sides of cyl projection.
 */
static int
add_to_polyline (xp, xpsize, i, vis, nxp, max, w, x, y)
XPoint xp[];	/* working array */
int xpsize;	/* entries in xp[] */
int i;		/* item we are on: 0..max */
int vis;	/* is this point visible */
int nxp;	/* number of items in xp[] in use -- next goes in xp[npx] */
int max;	/* largest item number we will draw */
int w;		/* window width -- used to check top wraps */
int x, y;	/* the point to add to polyline */
{
	int xwrap;

	if (vis) {
	    xp[nxp].x = x;
	    xp[nxp].y = y;
	    nxp++;
	}

	if (wants[CYLINDRICAL] && nxp > 1) {
	    if (xp[nxp-2].x - xp[nxp-1].x >= w/2) {
		xp[nxp-1].x = w-1;
		xwrap = 1;
	    } else if (xp[nxp-1].x - xp[nxp-2].x >= w/2) {
		xp[nxp-1].x = 0;
		xwrap = -1;
	    } else
		xwrap = 0;
	} else
	    xwrap = 0;

	/* draw the line if it just turned invisible, done or wrapped */
	if (!vis || i == max || nxp == xpsize || xwrap) {
	    if (nxp > 1)
		XDrawLines (XtD, e_pm, e_olgc, xp, nxp, CoordModeOrigin);
	    nxp = 0;

	    /* if we wrapped then pick up fragment on opposite end */
	    if (xwrap < 0) {
		/* pick up fragment on right end */
		xp[nxp].x = w-1; xp[nxp].y = y; nxp++;
		xp[nxp].x = x; xp[nxp].y = y; nxp++;
	    } else if (xwrap > 0) {
		/* pick up fragment on left end */
		xp[nxp].x = 0; xp[nxp].y = y; nxp++;
		xp[nxp].x = x; xp[nxp].y = y; nxp++;
	    }
	    /* leave at front of line but draw if this is all */
	    if (i == max)
		XDrawLines (XtD, e_pm, e_olgc, xp, nxp, CoordModeOrigin);
	}

	return (nxp);
}

/* mark the mainmenu location */
static void
e_mainmenuloc (np, r, wb, hb)
Now *np;
unsigned r, wb, hb;
{
	XSetForeground (XtD, e_olgc, ecolors[HEREC].p);
	e_drawcross (r, wb, hb, lat, lng, PLUSS);
}

/* mark the location of a solar ecipse, if any.
 *
 * N.B. this code is geometrically correct for all objects, but in practice
 * only the sun and moon are computed accurately enough by xephem to make this
 * worth while for now. in particular, I have tried tests against several S&T
 * tables of minor planet occultations and while the asteroids are computed
 * well enough for visual identification remember that at even 1 AU the earth
 * only subtends 18 arc seconds and the asteroids are not computed *that*
 * accurately (especially since we do not yet include perturbations).
 *
 * I will try to describe the algorithm: the exective summary is that we
 * are striving for the spherical arc subtended by the intersection of a line
 * from one object and the earth's center and the line from the other object
 * to the earth's center.
 *
 * N.B. I tried just computing the intersection of a line connecting the two
 * objects and a unit sphere but it suffered from terrible numerical instabilty.
 *
 * start in a plane defined by the center of the earth, the north pole and
 * object obj0. label the center of the earth as O, the location of object 0
 * as P0, and place object 1 someplace off the line O-P0 and label it P1.
 * what you have actually placed is the location of P1 as it projected onto
 * this place; ie, we are only working with dec here.  define decA as the
 * angle P1-O-P0; it is so named because it is the amount of declication
 * subtended from P0 to P1. Project the line P0-P1 back to a line
 * perpendicular to the line O-P0 at O. decD is the distance from O to the
 * point where P0-P1 intersects the line. if it is less than the earth radius
 * we have an occultation! now do all this again only this time place
 * yourself in a plane defined by the real locations of O, P0 and P1. and
 * repeat everything except this time use the real angled subtended in the
 * sky between P0 and P1 (not just the dec difference). this angle we define
 * as skyA (and its projection back onto a plane perpendicular to P0-P1
 * through O we call skyD). what we want next is the spherical angle
 * subtended between the point at which O-P0 intersects the earth's surface
 * (which is just the geocentric coords of P0) and the point where a line
 * from the tip of skyD to P1 intersects the earth's surface. we call this
 * skyT (I used tau in my original sketch) and I will let you work out the
 * trig (it's just planar trig since you are working in the O-P0-P1 plane).
 * this gives us the spherical angle between the two lines and the earth
 * surface; now all we need is the angle.image yourself at P0 now looking
 * right at O. we see decD as a vertical line and SkyD as a line going off
 * from O at an angle somewhere. the angle between these lines we define as
 * theta. knowing decD and skyD and knowing that there is a right angle at
 * the tip of decD between O and the tip of skyD we can compute the angle
 * between them. theta.  now just use a little spherical trig to find where
 * our arc ends up, compute the new RA, compute longitude by subtracting
 * gst, set latitude to dec, project and draw!
 */
static void
e_soleclipse (np, r, wb, hb)
Now *np;
unsigned r, wb, hb;
{
	Obj *op0 = db_basic (SUN);	/* op0 must be the further one */
	Obj *op1 = db_basic (MOON);
	Obj obj0, obj1;			/* use copies */
	double r0, r1;			/* dist to objects, in earth radii */
	double theta;			/* angle between projections */
	double decD, decA;		/* dec-only proj dist and angle */
	double skyD, skyA, skyP, skyT;	/* full sky projection */
	Now now;			/* local copy to compute EOD info */
	double lst, gst;		/* local and UTC time */
	double lt, lg;			/* lat/long */
	double sD, dRA;

	now = *np;
	obj0 = *op0;
	obj1 = *op1;

	now.n_epoch = EOD;
	(void) obj_cir (&now, &obj0);
	if (is_ssobj(&obj0))
	    r0 = obj0.s_edist*(MAU/ERAD);	/* au to earth radii */
	else
	    r0 = 1e7;				/* way past pluto */

	(void) obj_cir (&now, &obj1);
	if (is_ssobj(&obj1))
	    r1 = obj1.s_edist*(MAU/ERAD);	/* au to earth radii */
	else
	    r1 = 1e7;				/* way past pluto */

	decA = obj1.s_dec - obj0.s_dec;
	decD = r0*r1*sin(decA)/(r0 - r1);	/* similar triangles */
	if (fabs(decD) >= 1.0)
	    return;

	skyA = acos(sin(obj0.s_dec)*sin(obj1.s_dec) +
		cos(obj0.s_dec)*cos(obj1.s_dec)*cos(obj0.s_ra-obj1.s_ra));
	skyD = r0*r1*sin(skyA)/(r0 - r1);	/* similar triangles */
	if (fabs(skyD) >= 1.0)
	    return;

	/* skyP is angle subtended by skyD as seen from obj0 (I called it psi).
	 * skyT is angle subtended by line from earth center to obj0 to a
	 *   point where the line from obj0 to the tip of skyD intersects the
	 *   earth surface (I called it tau).
	 */
	skyP = atan(skyD/r0);
	skyT = asin(skyD*r0/sqrt(r0*r0+skyD*skyD)) - skyP;

	theta = acos(decD/skyD);
	solve_sphere (theta, skyT, sin(obj0.s_dec), cos(obj0.s_dec), &sD, &dRA);

	lt = asin(sD);

	if (obj1.s_ra > obj0.s_ra)
	    dRA = -dRA;	/* eastward */

	lst = obj0.s_ra - dRA;
	utc_gst (mjd_day(mjd), mjd_hr(mjd), &gst);
	lg = lst - hrrad(gst);
	while (lg < -PI) lg += 2*PI;
	while (lg >  PI) lg -= 2*PI;

	XSetForeground (XtD, e_olgc, ecolors[ECLIPSEC].p);
	e_drawcross (r, wb, hb, lt, lg, CROSSH);
}

/* given a height above the earth, in meters, and an altitude above the
 * horizon, in rads, return the great-circle angular distance from the subpoint
 * to the point at which the given height appears at the given altitude, in
 * rads.
 */
static void
e_viewrad (height, alt, radp)
double height;	/* satellite elevation, m above mean earth */
double alt;	/* viewing altitude, rads above horizon */
double *radp;	/* great-circle distance from subpoint to viewing circle, rads*/
{
	*radp = acos(ERAD/(ERAD+height)*cos(alt)) - alt;
}

/* discard any existing trail history information and add just one */
static void
e_resettrails(np)
Now *np;
{
	if (trails) {
	    XtFree ((char *)trails);
	    trails = 0;
	}
	ntrails = 0;

	e_addtrail (np);
}

/* add an entry to the trails list.
 * avoid dups.
 */
static void
e_addtrail (np)
Now *np;
{
	Trail *tp;
	Obj *op;
	int new;

	op = db_basic (object_dbidx);

	/* prepare for checking dups if there is at least one trail entry.
	 * dups disappear because we may be doing some drawing with xor.
	 */
	if (trails) {
	    tp = &trails[ntrails-1];
	    new = memcmp (np, &tp->t_now, sizeof(Now)) != 0
				|| memcmp (op, &tp->t_obj, sizeof(ObjES)) != 0;
	}

	/* add a trail entry for the new location if we don't have any now or
	 * the new entry would not be a dup.
	 */
	if (!trails || new) {
	    trails = (Trail *)
			XtRealloc ((char *)trails, sizeof(Trail)*(ntrails+1));
	    tp = &trails[ntrails];
	    tp->t_now = *np;
	    tp->t_obj = *op;
	    e_subobject (np, object_dbidx, &tp->t_sublat, &tp->t_sublng);
	    ntrails++;
	}
}
