/* code to manage the stuff on the "jupiter" menu.
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#if defined(__STDC__)
#include <stdlib.h>
#endif
#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/CascadeB.h>
#include <Xm/Separator.h>
#include <Xm/Scale.h>
#include <Xm/RowColumn.h>
#include <Xm/DrawingA.h>
#include <Xm/ToggleB.h>
#include "astro.h"
#include "circum.h"

#if defined(__STDC__) || defined(__cplusplus)
#define P_(s) s
#else
#define P_(s) ()
#endif

extern Now *mm_get_now P_((void));
extern int any_ison P_((void));
extern void f_double P_((Widget w, char *fmt, double f));
extern void get_something P_((Widget w, char *resource, char *value));
extern void range P_((double *v, double r));
extern void register_selection P_((char *name));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern void timestamp P_((Now *np, Widget w));
extern void mm_movie P_((double stepsz));

void jm_manage P_((void));
int jm_ison P_((void));
void jm_selection_mode P_((int whether));
void jm_update P_((Now *np, int how_much));
void jm_cursor P_((Cursor c));
static void jm_create_form_w P_((void));
static void jm_create_jsform_w P_((void));
static void jm_set_buttons P_((int whether));
static void jm_set_a_button P_((Widget pbw, int whether));
static void jm_jstats_cb P_((Widget w, XtPointer client, XtPointer call));
static void jm_bigd_cb P_((Widget w, XtPointer client, XtPointer call));
static void jm_tags_cb P_((Widget w, XtPointer client, XtPointer call));
static void jm_flip_cb P_((Widget w, XtPointer client, XtPointer call));
static void jm_scale_cb P_((Widget w, XtPointer client, XtPointer call));
static void jm_activate_cb P_((Widget w, XtPointer client, XtPointer call));
static void jm_close_cb P_((Widget w, XtPointer client, XtPointer call));
static void jm_movie_cb P_((Widget w, XtPointer client, XtPointer call));
static void jm_da_exp_cb P_((Widget w, XtPointer client, XtPointer call));

extern Widget toplevel_w;
#define	XtD	XtDisplay(toplevel_w)

static Widget jupform_w;	/* main form */
static Widget jsform_w;         /* statistics form */
static Widget jda_w;		/* drawing area */
static Widget cml_w[2];		/* display widgets for CMLs */
static Widget scale_w;		/* size scale */
static Widget dt_w;             /* main date/time stamp widget */
static Widget jdt_w;            /* statistics date/time stamp widget */
enum {CMLI, CMLII};		/* cml indexes */
#define	NM	4		/* number of moons */
static Widget j_w[NM][3];	/* the data display widgets */
enum {X, Y, Z};			/* j_w column index */
static int jm_selecting;	/* set while our fields are being selected */
static int bigdots;		/* whether we want big dots */
static int j_tags;		/* whether we want tags on the drawing */
static int flip_ns;		/* whether we want to flip north/south */
static int flip_ew;		/* whether we want to flip east/west */
static int jswasman;            /* whether statistics form was managed */

#define	MAXSCALE	10.0	/* max scale mag factor */
#define	MOVIE_STEPSZ	0.25	/* movie step size, hours */

static struct MoonNames {
    char *full;
    char *tag;
} mnames[NM] = {
    {"Io", "I"},
    {"Europa", "II"},
    {"Ganymede", "III"},
    {"Callisto", "IV"}
};

/* called when the jupiter menu is activated via the main menu pulldown.
 * if never called before, create and manage all the widgets as a child of a
 * form. otherwise, just toggle whether the form is managed.
 */
void
jm_manage ()
{
	if (!jupform_w) {
	    jm_create_form_w();
	    jm_create_jsform_w();
	}
	
	if (XtIsManaged(jupform_w)) {
	    XtUnmanageChild (jupform_w);
	    if (jswasman = XtIsManaged(jsform_w))
		XtUnmanageChild (jsform_w);
	} else {
	    XtManageChild (jupform_w);
	    if (jswasman) {
		XtManageChild (jsform_w);
		jm_set_buttons(jm_selecting);
	    }
	}
}

jm_ison()
{
	return (jupform_w && XtIsManaged(jupform_w));
}

/* called by other menus as they want to hear from our buttons or not.
 * the "on"s and "off"s stack - only really redo the buttons if it's the
 * first on or the last off.
 */
void
jm_selection_mode (whether)
int whether;	/* whether setting up for plotting or for not plotting */
{
	jm_selecting += whether ? 1 : -1;

	if (jm_ison())
	    if (whether && jm_selecting == 1     /* first one to want on */
		|| !whether && jm_selecting == 0 /* last one to want off */)
		jm_set_buttons (whether);
}

/* called to put up or remove the watch cursor.  */
void
jm_cursor (c)
Cursor c;
{
	Window win;

	if (jupform_w && (win = XtWindow(jupform_w))) {
	    Display *dsp = XtDisplay(jupform_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}

	if (jsform_w && (win = XtWindow(jsform_w))) {
	    Display *dsp = XtDisplay(jsform_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}
}

/* create the main form dialog */
static void
jm_create_form_w()
{
	Widget w;
	Widget frame_w;
	Widget mb_w, pd_w, cb_w;
	XmString str;
	Arg args[20];
	int n;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNdefaultPosition, False); n++;
	XtSetArg (args[n], XmNresizePolicy, XmRESIZE_NONE); n++;
	jupform_w = XmCreateFormDialog (toplevel_w, "Jupiter", args, n);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Jupiter view"); n++;
	XtSetValues (XtParent(jupform_w), args, n);

	/* create the menu bar across the top */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	mb_w = XmCreateMenuBar (jupform_w, "MB", args, n);
	XtManageChild (mb_w);

	/* make the Control pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ControlPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'C'); n++;
	    cb_w = XmCreateCascadeButton (mb_w, "ControlCB", args, n);
	    set_xmstring (cb_w, XmNlabelString, "Control");
	    XtManageChild (cb_w);


	    /* add the "movie" push button */

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Movie", args, n);
	    set_xmstring (w, XmNlabelString, "Movie Demo");
	    XtAddCallback (w, XmNactivateCallback, jm_movie_cb, 0);
	    XtManageChild (w);

	    /* add the "close" push button beneath a separator */

	    n = 0;
	    w = XmCreateSeparator (pd_w, "Sep", args, n);
	    XtManageChild (w);

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Close", args, n);
	    XtAddCallback (w, XmNactivateCallback, jm_close_cb, 0);
	    XtManageChild (w);

	/* make the View pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ViewPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'V'); n++;
	    cb_w = XmCreateCascadeButton (mb_w, "ViewCB", args, n);
	    set_xmstring (cb_w, XmNlabelString, "View");
	    XtManageChild (cb_w);

	    /* make the tags toggle button */

	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    w = XmCreateToggleButton (pd_w, "Tags", args, n);
	    XtAddCallback (w, XmNvalueChangedCallback, jm_tags_cb, 0);
	    XtManageChild (w);
	    j_tags = XmToggleButtonGetState(w);

	    /* "big dots" toggle button */

	    str = XmStringCreate("Big dots", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    XtSetArg (args[n], XmNlabelString, str); n++;
	    w = XmCreateToggleButton(pd_w,"BigDots",args,n);
	    XtAddCallback(w, XmNvalueChangedCallback, jm_bigd_cb, 0);
	    XtManageChild (w);
	    XmStringFree (str);
	    bigdots = XmToggleButtonGetState(w);

	    /* make the flip togglebuttons */

	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    w = XmCreateToggleButton (pd_w, "FlipNS", args, n);
	    set_xmstring (w, XmNlabelString, "Flip N/S");
	    XtAddCallback (w, XmNvalueChangedCallback, jm_flip_cb,
							(XtPointer)&flip_ns);
	    flip_ns = XmToggleButtonGetState (w);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    w = XmCreateToggleButton (pd_w, "FlipEW", args, n);
	    set_xmstring (w, XmNlabelString, "Flip E/W");
	    XtAddCallback (w, XmNvalueChangedCallback, jm_flip_cb, 
							(XtPointer)&flip_ew);
	    flip_ew = XmToggleButtonGetState (w);
	    XtManageChild (w);

	    /* add a separator */

	    n = 0;
	    w = XmCreateSeparator (pd_w, "Sep", args, n);
	    XtManageChild (w);

	    /* add the More Info control */

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Stats", args, n);
	    set_xmstring (w, XmNlabelString, "More info...");
	    XtAddCallback (w, XmNactivateCallback, jm_jstats_cb, NULL);
	    XtManageChild (w);

	/* make the date/time stamp label */

	n = 0;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	dt_w = XmCreateLabel (jupform_w, "DateStamp", args, n);
	timestamp (mm_get_now(), dt_w);	/* establishes size */
	XtManageChild (dt_w);

	/* make the scale widget
	 * attach both top and bottom so it's the one to follow resizing.
	 */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, mb_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, dt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNmaximum, 100); n++;
	XtSetArg (args[n], XmNminimum, 0); n++;
	XtSetArg (args[n], XmNscaleMultiple, 10); n++;
	XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_TOP); n++;
	scale_w = XmCreateScale (jupform_w, "Scale", args, n);
	XtAddCallback (scale_w, XmNdragCallback, jm_scale_cb, 0);
	XtAddCallback (scale_w, XmNvalueChangedCallback, jm_scale_cb, 0);
	XtManageChild (scale_w);

	/* make a frame for the drawing area.
	 * attach both top and bottom so it's the one to follow resizing.
	 */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, mb_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, dt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, scale_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_OUT); n++;
	frame_w = XmCreateFrame (jupform_w, "JupFrame", args, n);
	XtManageChild (frame_w);

	    /* make a drawing area for drawing the little map */

	    n = 0;
	    jda_w = XmCreateDrawingArea (frame_w, "Map", args, n);
	    XtAddCallback (jda_w, XmNexposeCallback, jm_da_exp_cb, 0);
	    XtManageChild (jda_w);
}

/* make the statistics form dialog */
static void
jm_create_jsform_w()
{
	Widget w;
	Widget r_w, title_w, col_w;
	Arg args[20];
	int n;
	int i;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNresizePolicy, XmRESIZE_NONE); n++;
	XtSetArg (args[n], XmNverticalSpacing, 5); n++;
	jsform_w = XmCreateFormDialog (toplevel_w, "JupiterStats", args, n);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Jupiter info"); n++;
	XtSetValues (XtParent(jsform_w), args, n);

	/* make top two title rows */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	r_w = XmCreateLabel (jsform_w, "CMLL", args, n);
	XtManageChild (r_w);
	set_xmstring (r_w, XmNlabelString, 
	    "Central Meridian Longitude (degs):\n(GRS is at about 30-40 degs in System II)");

	/* make row for CMLs */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, r_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNuserData, "Jupiter.CMLI"); n++;
	w = cml_w[CMLI] = XmCreatePushButton(jsform_w, "CMLI", args, n);
	XtAddCallback(w, XmNactivateCallback, jm_activate_cb, 0);
	XtManageChild (w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, r_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, w); n++;
	w = XmCreatePushButton (jsform_w, "CMLIL", args, n);
	XtManageChild (w);
	jm_set_a_button (w, False);
	set_xmstring (w, XmNlabelString, "(Sys I)");

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, r_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreatePushButton (jsform_w, "CMLIIL", args, n);
	XtManageChild (w);
	jm_set_a_button (w, False);
	set_xmstring (w, XmNlabelString, "(Sys II)");

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, r_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNrightWidget, w); n++;
	XtSetArg (args[n], XmNuserData, "Jupiter.CMLII"); n++;
	w = cml_w[CMLII] = XmCreatePushButton(jsform_w, "CMLII", args, n);
	XtAddCallback(w, XmNactivateCallback, jm_activate_cb, 0);
	XtManageChild (w);

	/* make table title label */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	title_w = XmCreateLabel (jsform_w, "TableLabel", args, n);
	XtManageChild (title_w);
	set_xmstring (title_w, XmNlabelString,
				" \nMoon positions -- in Jupiter Radii");

	/* make the moon table, one column at a time */

	/* moon designator column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (jsform_w, "MoonDes", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, " ", args, n);
	    XtManageChild (w);
	    jm_set_a_button (w, False);

	    for (i = 0; i < NM; i++) {
		n = 0;
		w = XmCreatePushButton (col_w, mnames[i].tag, args, n);
		XtManageChild (w);
		jm_set_a_button (w, False);
	    }

	/* moon name column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 10); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_BEGINNING); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (jsform_w, "MoonName", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, " ", args, n);
	    XtManageChild (w);
	    jm_set_a_button (w, False);

	    for (i = 0; i < NM; i++) {
		n = 0;
		w = XmCreatePushButton (col_w, mnames[i].full, args, n);
		XtManageChild (w);
		jm_set_a_button (w, False);
	    }

	/* moon X column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 35); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (jsform_w, "MoonXRC", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, "MoonX", args, n);
	    XtManageChild (w);
	    jm_set_a_button (w, False);
	    set_xmstring (w, XmNlabelString, "X (+E)");

	    for (i = 0; i < NM; i++) {
		char *sel;
		sel = XtMalloc (strlen(mnames[i].full) + 3); /* '.X\0' */
		(void) sprintf (sel, "%s.X", mnames[i].full);
		n = 0;
		XtSetArg (args[n], XmNuserData, sel); n++;
		w = j_w[i][X] = XmCreatePushButton(col_w, "MoonXPB", args, n);
		XtAddCallback(w, XmNactivateCallback, jm_activate_cb, 0);
		XtManageChild (w);
	    }

	/* moon Y column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 55); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (jsform_w, "MoonYRC", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, "MoonYLabel", args, n);
	    XtManageChild (w);
	    set_xmstring (w, XmNlabelString, "Y (+S)");
	    jm_set_a_button (w, False);

	    for (i = 0; i < NM; i++) {
		char *sel;
		sel = XtMalloc (strlen(mnames[i].full) + 3); /* '.Y\0' */
		(void) sprintf (sel, "%s.Y", mnames[i].full);
		n = 0;
		XtSetArg (args[n], XmNuserData, sel); n++;
		w = j_w[i][Y] = XmCreatePushButton(col_w, "MoonYPB", args, n);
		XtAddCallback(w, XmNactivateCallback, jm_activate_cb, 0);
		XtManageChild (w);
	    }

	/* moon Z column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 75); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (jsform_w, "MoonZRC", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, "MoonZ", args, n);
	    XtManageChild (w);
	    set_xmstring (w, XmNlabelString, "Z (+front)");
	    jm_set_a_button (w, False);

	    for (i = 0; i < NM; i++) {
		char *sel;
		sel = XtMalloc (strlen(mnames[i].full) + 3); /* '.Z\0' */
		(void) sprintf (sel, "%s.Z", mnames[i].full);
		n = 0;
		XtSetArg (args[n], XmNuserData, sel); n++;
		w = j_w[i][Z] = XmCreatePushButton(col_w, "MoonZPB", args, n);
		XtAddCallback(w, XmNactivateCallback, jm_activate_cb, 0);
		XtManageChild (w);
	    }

	/* make a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, col_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (jsform_w, "Sep1", args, n);
	XtManageChild (w);

	/* make a date/time stamp */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	jdt_w = XmCreateLabel (jsform_w, "SDateStamp", args, n);
	XtManageChild (jdt_w);

	/* make a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, jdt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (jsform_w, "Sep2", args, n);
	XtManageChild (w);

	/* make the close button */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 30); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 70); n++;
	w = XmCreatePushButton (jsform_w, "Close", args, n);
	XtAddCallback (w, XmNactivateCallback, jm_jstats_cb, 0);
	XtManageChild (w);
}

/* go through all the buttons pickable for plotting and set whether they
 * should appear to look like buttons or just flat labels.
 */
static void
jm_set_buttons (whether)
int whether;	/* whether setting up for plotting or for not plotting */
{
	int i;

	for (i = 0; i < NM; i++) {
	    jm_set_a_button (j_w[i][X], whether);
	    jm_set_a_button (j_w[i][Y], whether);
	    jm_set_a_button (j_w[i][Z], whether);
	}

	jm_set_a_button (cml_w[CMLI], whether);
	jm_set_a_button (cml_w[CMLII], whether);
}

/* set whether the given button looks like a label.
 */
static void
jm_set_a_button(pbw, whether)
Widget pbw;
int whether;
{
	static Arg look_like_button[] = {
	    {XmNtopShadowColor, (XtArgVal) 0},
	    {XmNbottomShadowColor, (XtArgVal) 0},
            {XmNtopShadowPixmap, (XtArgVal) 0},
            {XmNbottomShadowPixmap, (XtArgVal) 0},
	    {XmNfillOnArm, (XtArgVal) True},
	    {XmNtraversalOn, (XtArgVal) True},
	};
	static Arg look_like_label[] = {
	    {XmNtopShadowColor, (XtArgVal) 0},
	    {XmNbottomShadowColor, (XtArgVal) 0},
            {XmNtopShadowPixmap, (XtArgVal) 0},
            {XmNbottomShadowPixmap, (XtArgVal) 0},
	    {XmNfillOnArm, (XtArgVal) False},
	    {XmNtraversalOn, (XtArgVal) False},
	};
	static int called;
	Arg *ap;
	int na;

	if (!called) {
	    /* get baseline label and shadow appearances.
	     */
            Pixel topshadcol, botshadcol, bgcol;
            Pixmap topshadpm, botshadpm;
	    Arg args[20];
	    Widget tmpw;
	    int n;

	    n = 0;
	    tmpw = XmCreatePushButton (jupform_w, "tmp", args, n);

	    n = 0;
            XtSetArg (args[n], XmNtopShadowColor, &topshadcol); n++;
            XtSetArg (args[n], XmNbottomShadowColor, &botshadcol); n++;
            XtSetArg (args[n], XmNtopShadowPixmap, &topshadpm); n++;
            XtSetArg (args[n], XmNbottomShadowPixmap, &botshadpm); n++;
	    XtSetArg (args[n], XmNbackground, &bgcol); n++;
	    XtGetValues (tmpw, args, n);

            look_like_button[0].value = topshadcol;
            look_like_button[1].value = botshadcol;
            look_like_button[2].value = topshadpm;
            look_like_button[3].value = botshadpm;
            look_like_label[0].value = bgcol;
            look_like_label[1].value = bgcol;
            look_like_label[2].value = XmUNSPECIFIED_PIXMAP;
            look_like_label[3].value = XmUNSPECIFIED_PIXMAP;

	    XtDestroyWidget (tmpw);
	     
	    called = 1;
	}

	if (whether) {
	    ap = look_like_button;
	    na = XtNumber(look_like_button);
	} else {
	    ap = look_like_label;
	    na = XtNumber(look_like_label);
	}

	XtSetValues (pbw, ap, na);
}

/* callback from the Close button is activated on the stats menu or when
 * the More Info button is activated. they each do the same thing.
 */
/* ARGSUSED */
static void
jm_jstats_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (jswasman = XtIsManaged(jsform_w))
	    XtUnmanageChild (jsform_w);
	else {
	    XtManageChild (jsform_w);
	    jm_set_buttons(jm_selecting);
	}
}

/* callback from the big dots toggle button
 * TODO: really shouldn't get present time, just redo dots in same location.
 */
/* ARGSUSED */
static void
jm_bigd_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	bigdots = XmToggleButtonGetState(w);
	jm_update (mm_get_now(), 1);
}

/* callback from the tags toggle button
 * TODO: really shouldn't get present time, just redo dots in same location.
 */
/* ARGSUSED */
static void
jm_tags_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	j_tags = XmToggleButtonGetState(w);
	jm_update (mm_get_now(), 1);
}

/* callback from the flip toggle buttons.
 * client points to one of flip_ns or flip_ew.
 * TODO: really shouldn't get present time, just redo at same location
 */
/* ARGSUSED */
static void
jm_flip_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int *p = (int *)client;
	
	*p = XmToggleButtonGetState(w);
	jm_update (mm_get_now(), 1);
}

/* callback from the scale.
 * TODO: really shouldn't get present time, just redo dots in same location.
 */
/* ARGSUSED */
static void
jm_scale_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	jm_update (mm_get_now(), 1);
}

/* callback from any of the data menu buttons being activated.
 */
/* ARGSUSED */
static void
jm_activate_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (jm_selecting) {
	    char *name;
	    get_something (w, XmNuserData, (char *)&name);
	    register_selection (name);
	}
}

/* callback from the Close button
 */
/* ARGSUSED */
static void
jm_close_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XtUnmanageChild (jupform_w);
	if (jswasman = XtIsManaged(jsform_w))
	    XtUnmanageChild (jsform_w);
}

/* callback from the Movie button
 */
/* ARGSUSED */
static void
jm_movie_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	mm_movie (MOVIE_STEPSZ);
}

/* callback from either expose of the drawing area.
 */
/* ARGSUSED */
static void
jm_da_exp_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;

	/* filter out a few oddball cases */
	switch (c->reason) {
	case XmCR_EXPOSE: {
	    /* turn off gravity so we get expose events for either shrink or
	     * expand.
	     */
	    static before;
	    XExposeEvent *e = &c->event->xexpose;

	    if (!before) {
		XSetWindowAttributes swa;
		swa.bit_gravity = ForgetGravity;
		XChangeWindowAttributes (e->display, e->window, 
							    CWBitGravity, &swa);
		before = 1;
	    }
	    /* wait for the last in the series */
	    if (e->count != 0)
		return;
	    break;
	    }
	default:
	    printf ("Unexpected jupform_w event. type=%d\n", c->reason);
	    exit(1);
	}

	jm_update (mm_get_now(), 1);
}

typedef struct {
    double x, y, z;
} MoonData;

static void jm_draw_map ();
static void jupinfo ();

/* called to recompute and fill in values for the jupiter menu.
 * don't bother if it doesn't exist or is unmanaged now and we are not logging.
 */
void
jm_update (np, how_much)
Now *np;
int how_much;
{
	static char fmt[] = "%7.3f";
	int wantstats;
	MoonData moons[NM];
	double sIcml, sIIcml;
	int i;

	if (!jupform_w)
	    return;

	wantstats = XtIsManaged(jsform_w) || any_ison() || how_much;

	if (!XtIsManaged(jupform_w) && !wantstats)
	    return;

	/* compute jupiter info.
	 */
	jupinfo (mjd, moons, &sIcml, &sIIcml);

	if (wantstats) {
	    f_double (cml_w[CMLI], fmt, sIcml);
	    f_double (cml_w[CMLII], fmt, sIIcml);

	    for (i = 0; i < NM; i++) {
		f_double (j_w[i][X], fmt, moons[i].x);
		f_double (j_w[i][Y], fmt, moons[i].y);
		f_double (j_w[i][Z], fmt, moons[i].z);
	    }
	    timestamp (np, jdt_w);
	}

	if (XtIsManaged(jupform_w)) {
	    jm_draw_map (jda_w, moons, sIcml, sIIcml);
	    timestamp (np, dt_w);
	}
}

/* given the loc of the moons, draw a nifty little picture.
 * scale of the locations is in terms of jupiter radii == 1.
 * unflipped view is S up, E right.
 */
/* ARGSUSED */
static void
jm_draw_map (w, moons, sIcml, sIIcml)
Widget w;
MoonData moons[NM];
double sIcml, sIIcml;
{
	static GC j_fgc, j_bgc, j_xgc;
	static XFontStruct *j_fs;
	static last_nx, last_ny;
	static int cw, ch;
	static Pixmap pm;
	Display *dsp = XtDisplay(w);
	Window win = XtWindow(w);
	Window root;
	double scale;
	int sv;
	char c;
	int x, y;
	unsigned int nx, ny, bw, d;
	int ns = flip_ns ? -1 : 1;
	int ew = flip_ew ? -1 : 1;
	int i;

#define	NORM	27.0	/* max callisto orbit radius; used to normalize */
#define	MAPSCALE(v)	((v)*((int)nx)/NORM/2*scale)
#define	XCORD(x)	((int)(((int)nx)/2.0 + ew*MAPSCALE(x) + 0.5))
#define	YCORD(y)	((int)(((int)ny)/2.0 - ns*MAPSCALE(y) + 0.5))

	if (!j_fgc) {
	    XGCValues gcv;
	    unsigned int gcm;
	    Pixel fg, bg;

	    gcm = GCForeground;
	    get_something (w, XmNforeground, (char *)&fg);
	    gcv.foreground = fg;
	    j_fgc = XCreateGC (dsp, win, gcm, &gcv);
	    j_fs = XQueryFont (dsp, XGContextFromGC (j_fgc));
	    cw = j_fs->max_bounds.width;
	    ch = j_fs->max_bounds.ascent + j_fs->max_bounds.descent;

	    gcm = GCForeground;
	    get_something (w, XmNbackground, (char *)&bg);
	    gcv.foreground = bg;
	    j_bgc = XCreateGC (dsp, win, gcm, &gcv);

	    gcm = GCForeground | GCFunction;
	    gcv.foreground = fg ^ bg;
	    gcv.function = GXxor;
	    j_xgc = XCreateGC (dsp, win, gcm, &gcv);
	}

	XmScaleGetValue (scale_w, &sv);
	scale = pow(MAXSCALE, sv/100.0);

	XGetGeometry(dsp, win, &root, &x, &y, &nx, &ny, &bw, &d);
	if (!pm || nx != last_nx || ny != last_ny) {
	    if (pm)
		XFreePixmap (dsp, pm);
	    pm = XCreatePixmap (dsp, win, nx, ny, d);
	    last_nx = nx;
	    last_ny = ny;
	}

	XFillRectangle (dsp, pm, j_bgc, 0, 0, nx, ny);

	/* draw labels */
	c = flip_ew ? 'W' : 'E';
	XDrawString(dsp, pm, j_fgc, nx-cw-1, ny/2-2, &c, 1);
	c = flip_ns ? 'N' : 'S';
	XDrawString(dsp, pm, j_fgc, (nx-cw)/2-1, j_fs->ascent, &c, 1);

	/* draw Jupiter in center with unit radius */
	XFillArc (dsp, pm, j_fgc, nx/2-(int)MAPSCALE(1), ny/2-(int)MAPSCALE(1),
			    2*(int)MAPSCALE(1), 2*(int)MAPSCALE(1), 0, 360*64);

	/* draw each moon that is visible.
	 */
	for (i = 0; i < NM; i++) {
	    double mx = moons[i].x;
	    double my = moons[i].y;
	    double mz = moons[i].z;
	    int outside = mx*mx + my*my > 1.0;
	    int infront = mz > 0.0;

	    if (!outside && !infront)
		continue;	/* behind jupiter */

	    x = XCORD(mx);
	    y = YCORD(my);
	    XDrawPoint (dsp, pm, j_xgc, x, y);
	    if (bigdots) {
		XDrawPoint(dsp, pm, j_xgc, x+1, y);
		XDrawPoint(dsp, pm, j_xgc, x,   y+1);
		XDrawPoint(dsp, pm, j_xgc, x+1, y+1);
	    }
	    if (j_tags)
		XDrawString(dsp, pm, j_xgc, x-cw/2, y+2*ch,
					mnames[i].tag, strlen(mnames[i].tag));
	}

	XCopyArea (dsp, pm, win, j_fgc, 0, 0, nx, ny, 0, 0);
}


#define	dsin(x)	sin(degrad(x))
#define	dcos(x)	cos(degrad(x))

/* given a modified julian date (ie, days since Jan .5 1900), d, return x, y, z
 *   location of each Galilean moon as a multiple of Jupiter's radius. on this
 *   scale, Callisto is never more than 26.5593. +x is easterly, +y is
 *   southerly, +z is towards earth. x and z are relative to the equator
 *   of Jupiter; y is further corrected for earth's position above or below
 *   this plane. also, return the system I and II central meridian longitude,
 *   in degress, relative to the true disk of jupiter and corrected for light
 *   travel time.
 * from "Astronomical Formulae for Calculators", 2nd ed, by Jean Meeus,
 *   Willmann-Bell, Richmond, Va., U.S.A. (c) 1982, chapters 35 and 36.
 */
static void
jupinfo (d, moons, sIcml, sIIcml)
double d;
MoonData moons[NM];	/* Io, Europa, Ganymede, Callisto */
double *sIcml, *sIIcml;
{
	double A, B, Del, J, K, M, N, R, V;
	double cor_u1, cor_u2, cor_u3, cor_u4;
	double solc, tmp, G, H, psi, r, r1, r2, r3, r4;
	double u1, u2, u3, u4;
	double lam, Ds;
	double z1, z2, z3,  z4;
	double De, dsinDe;

	V = 134.63 + 0.00111587 * d;

	M = (358.47583 + 0.98560003*d);
	N = (225.32833 + 0.0830853*d) + 0.33 * dsin (V);

	J = 221.647 + 0.9025179*d - 0.33 * dsin(V);;

	A = 1.916*dsin(M) + 0.02*dsin(2*M);
	B = 5.552*dsin(N) + 0.167*dsin(2*N);
	K = (J+A-B);
	R = 1.00014 - 0.01672 * dcos(M) - 0.00014 * dcos(2*M);
	r = 5.20867 - 0.25192 * dcos(N) - 0.00610 * dcos(2*N);
	Del = sqrt (R*R + r*r - 2*R*r*dcos(K));
	psi = raddeg (asin (R/Del*dsin(K)));

	solc = (d - Del/173.);	/* speed of light correction */
	tmp = psi - B;

	u1 = 84.5506 + 203.4058630 * solc + tmp;
	u2 = 41.5015 + 101.2916323 * solc + tmp;
	u3 = 109.9770 + 50.2345169 * solc + tmp;
	u4 = 176.3586 + 21.4879802 * solc + tmp;

	G = 187.3 + 50.310674 * solc;
	H = 311.1 + 21.569229 * solc;
      
	cor_u1 =  0.472 * dsin (2*(u1-u2));
	cor_u2 =  1.073 * dsin (2*(u2-u3));
	cor_u3 =  0.174 * dsin (G);
	cor_u4 =  0.845 * dsin (H);
      
	r1 = 5.9061 - 0.0244 * dcos (2*(u1-u2));
	r2 = 9.3972 - 0.0889 * dcos (2*(u2-u3));
	r3 = 14.9894 - 0.0227 * dcos (G);
	r4 = 26.3649 - 0.1944 * dcos (H);

	moons[0].x = -r1 * dsin (u1+cor_u1);
	moons[1].x = -r2 * dsin (u2+cor_u2);
	moons[2].x = -r3 * dsin (u3+cor_u3);
	moons[3].x = -r4 * dsin (u4+cor_u4);

	lam = 238.05 + 0.083091*d + 0.33*dsin(V) + B;
	Ds = 3.07*dsin(lam + 44.5);
	De = Ds - 2.15*dsin(psi)*dcos(lam+24.)
		- 1.31*(r-Del)/Del*dsin(lam-99.4);
	dsinDe = dsin(De);

	z1 = r1 * dcos(u1+cor_u1);
	z2 = r2 * dcos(u2+cor_u2);
	z3 = r3 * dcos(u3+cor_u3);
	z4 = r4 * dcos(u4+cor_u4);

	moons[0].y = z1*dsinDe;
	moons[1].y = z2*dsinDe;
	moons[2].y = z3*dsinDe;
	moons[3].y = z4*dsinDe;

	moons[0].z = z1;
	moons[1].z = z2;
	moons[2].z = z3;
	moons[3].z = z4;

	*sIcml  = 268.28 + 877.8169088*(d - Del/173) + psi - B;
	range (sIcml, 360.0);
	*sIIcml = 290.28 + 870.1869088*(d - Del/173) + psi - B;
	range (sIIcml, 360.0);
}
