/* code to manage the stuff on the "saturn" menu.
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#if defined(__STDC__)
#include <stdlib.h>
#endif
#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/CascadeB.h>
#include <Xm/Separator.h>
#include <Xm/Scale.h>
#include <Xm/RowColumn.h>
#include <Xm/DrawingA.h>
#include <Xm/ToggleB.h>
#include "astro.h"
#include "circum.h"

#if defined(__STDC__) || defined(__cplusplus)
#define P_(s) s
#else
#define P_(s) ()
#endif

extern int obj_cir P_((Now *np, Obj *op));
extern Now *mm_get_now P_((void));
extern int any_ison P_((void));
extern void f_double P_((Widget w, char *fmt, double f));
extern void get_something P_((Widget w, char *resource, char *value));
extern void register_selection P_((char *name));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern void timestamp P_((Now *np, Widget w));
extern void mm_movie P_((double stepsz));

void sm_manage P_((void));
int sm_ison P_((void));
void sm_selection_mode P_((int whether));
void sm_cursor P_((Cursor c));
static void sm_create_form_w P_((void));
static void sm_create_ssform_w P_((void));
static void sm_set_buttons P_((int whether));
static void sm_set_a_button P_((Widget pbw, int whether));
static void sm_sstats_cb P_((Widget w, XtPointer client, XtPointer call));
static void sm_bigd_cb P_((Widget w, XtPointer client, XtPointer call));
static void sm_tags_cb P_((Widget w, XtPointer client, XtPointer call));
static void sm_flip_cb P_((Widget w, XtPointer client, XtPointer call));
static void sm_scale_cb P_((Widget w, XtPointer client, XtPointer call));
static void sm_activate_cb P_((Widget w, XtPointer client, XtPointer call));
static void sm_close_cb P_((Widget w, XtPointer client, XtPointer call));
static void sm_movie_cb P_((Widget w, XtPointer client, XtPointer call));
static void sm_da_exp_cb P_((Widget w, XtPointer client, XtPointer call));
void sm_update P_((Now *np, int how_much));

#undef P_

extern Widget toplevel_w;
#define	XtD	XtDisplay(toplevel_w)

static Widget satform_w;	/* main form */
static Widget ssform_w;		/* statistics form */
static Widget sda_w;		/* drawing area */
static Widget ringt_w;		/* widget containing ring tilt */
static Widget scale_w;		/* size scale */
static Widget dt_w;		/* main date/time stamp widget */
static Widget sdt_w;		/* statistics date/time stamp widget */
#define	NM	8		/* number of moons */
static Widget	s_w[NM][4];	/* the data display widgets */
enum {CX, CY, CZ, CMAG};	/* s_w column index */
static int sm_selecting;	/* set while our fields are being selected */
static int bigdots;		/* whether we want big dots */
static int s_tags;		/* whether we want tags on the drawing */
static int flip_ns;		/* whether we want to flip north/south */
static int flip_ew;		/* whether we want to flip east/west */
static int sswasman;		/* whether statistics form was managed */

#define	MAXSCALE	20.0	/* max sclae mag factor */
#define	MOVIE_STEPSZ	0.25	/* movie step size, hours */

static struct MoonNames {
    char *full;
    char *tag;
} mnames[NM] = {
    {"Mimas",	"I"},
    {"Enceladus","II"},
    {"Tethys",	"III"},
    {"Dione",	"IV"},
    {"Rhea",	"V"},
    {"Titan",	"VI"},
    {"Hyperion","VII"},
    {"Iapetus",	"VIII"},
};

/* called when the saturn menu is activated via the main menu pulldown.
 * if never called before, create and manage all the widgets as a child of a
 * form. otherwise, just toggle whether the form is managed.
 */
void
sm_manage ()
{
	if (!satform_w) {
	    sm_create_form_w();
	    sm_create_ssform_w();
	}
	
	if (XtIsManaged(satform_w)) {
	    XtUnmanageChild (satform_w);
	    if (sswasman = XtIsManaged(ssform_w))
		XtUnmanageChild (ssform_w);
	} else {
	    XtManageChild (satform_w);
	    if (sswasman) {
		XtManageChild (ssform_w);
		sm_set_buttons(sm_selecting);
	    }
	    /* rely on expose to do the first draw */
	}
}

sm_ison()
{
	return (satform_w && XtIsManaged(satform_w));
}

/* called by other menus as they want to hear from our buttons or not.
 * the "on"s and "off"s stack - only really redo the buttons if it's the
 * first on or the last off.
 */
void
sm_selection_mode (whether)
int whether;	/* whether setting up for plotting or for not plotting */
{
	sm_selecting += whether ? 1 : -1;

	if (sm_ison())
	    if (whether && sm_selecting == 1     /* first one to want on */
		|| !whether && sm_selecting == 0 /* last one to want off */)
		sm_set_buttons (whether);
}

/* called to put up or remove the watch cursor.  */
void
sm_cursor (c)
Cursor c;
{
	Window win;

	if (satform_w && (win = XtWindow(satform_w))) {
	    Display *dsp = XtDisplay(satform_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}

	if (ssform_w && (win = XtWindow(ssform_w))) {
	    Display *dsp = XtDisplay(ssform_w);
	    if (c)
		XDefineCursor (dsp, win, c);
	    else
		XUndefineCursor (dsp, win);
	}
}

/* create the main form dialog */
static void
sm_create_form_w()
{
	Widget w;
	Widget frame_w;
	Widget mb_w, pd_w, cb_w;
	XmString str;
	Arg args[20];
	int n;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNdefaultPosition, False); n++;
	XtSetArg (args[n], XmNresizePolicy, XmRESIZE_NONE); n++;
	satform_w = XmCreateFormDialog (toplevel_w, "Saturn", args, n);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Saturn view"); n++;
	XtSetValues (XtParent(satform_w), args, n);

	/* create the menu bar across the top */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	mb_w = XmCreateMenuBar (satform_w, "MB", args, n);
	XtManageChild (mb_w);

	/* make the Control pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ControlPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'C'); n++;
	    cb_w = XmCreateCascadeButton (mb_w, "ControlCB", args, n);
	    set_xmstring (cb_w, XmNlabelString, "Control");
	    XtManageChild (cb_w);


	    /* add the "movie" push button */

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Movie", args, n);
	    set_xmstring (w, XmNlabelString, "Movie Demo");
	    XtAddCallback (w, XmNactivateCallback, sm_movie_cb, 0);
	    XtManageChild (w);

	    /* add the "close" push button beneath a separator */

	    n = 0;
	    w = XmCreateSeparator (pd_w, "Sep", args, n);
	    XtManageChild (w);

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Close", args, n);
	    XtAddCallback (w, XmNactivateCallback, sm_close_cb, 0);
	    XtManageChild (w);

	/* make the View pulldown */

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "ViewPD", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
	    XtSetArg (args[n], XmNmnemonic, 'V'); n++;
	    cb_w = XmCreateCascadeButton (mb_w, "ViewCB", args, n);
	    set_xmstring (cb_w, XmNlabelString, "View");
	    XtManageChild (cb_w);

	    /* make the tags toggle button */

	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    w = XmCreateToggleButton (pd_w, "Tags", args, n);
	    XtAddCallback (w, XmNvalueChangedCallback, sm_tags_cb, 0);
	    XtManageChild (w);
	    s_tags = XmToggleButtonGetState(w);

	    /* "big dots" toggle button */

	    str = XmStringCreate("Big dots", XmSTRING_DEFAULT_CHARSET);
	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    XtSetArg (args[n], XmNlabelString, str); n++;
	    w = XmCreateToggleButton(pd_w,"BigDots",args,n);
	    XtAddCallback(w, XmNvalueChangedCallback, sm_bigd_cb, 0);
	    XtManageChild (w);
	    XmStringFree (str);
	    bigdots = XmToggleButtonGetState(w);

	    /* make the flip togglebuttons */

	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    w = XmCreateToggleButton (pd_w, "FlipNS", args, n);
	    set_xmstring (w, XmNlabelString, "Flip N/S");
	    XtAddCallback (w, XmNvalueChangedCallback, sm_flip_cb,
							(XtPointer)&flip_ns);
	    flip_ns = XmToggleButtonGetState (w);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	    XtSetArg (args[n], XmNmarginHeight, 0); n++;
	    XtSetArg (args[n], XmNindicatorType, XmONE_OF_MANY); n++;
	    w = XmCreateToggleButton (pd_w, "FlipEW", args, n);
	    set_xmstring (w, XmNlabelString, "Flip E/W");
	    XtAddCallback (w, XmNvalueChangedCallback, sm_flip_cb, 
							(XtPointer)&flip_ew);
	    flip_ew = XmToggleButtonGetState (w);
	    XtManageChild (w);

	    /* add a separator */

	    n = 0;
	    w = XmCreateSeparator (pd_w, "Sep", args, n);
	    XtManageChild (w);

	    /* add the More Info control */

	    n = 0;
	    w = XmCreatePushButton (pd_w, "Stats", args, n);
	    set_xmstring (w, XmNlabelString, "More info...");
	    XtAddCallback (w, XmNactivateCallback, sm_sstats_cb, NULL);
	    XtManageChild (w);

	/* make the date/time stamp label */

	n = 0;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	dt_w = XmCreateLabel (satform_w, "DateStamp", args, n);
	timestamp (mm_get_now(), dt_w);	/* establishes size */
	XtManageChild (dt_w);

	/* make the scale widget
	 * attach both top and bottom so it's the one to follow resizing.
	 */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, mb_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, dt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNmaximum, 100); n++;
	XtSetArg (args[n], XmNminimum, 0); n++;
	XtSetArg (args[n], XmNscaleMultiple, 10); n++;
	XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
	XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_TOP); n++;
	scale_w = XmCreateScale (satform_w, "Scale", args, n);
	XtAddCallback (scale_w, XmNdragCallback, sm_scale_cb, 0);
	XtAddCallback (scale_w, XmNvalueChangedCallback, sm_scale_cb, 0);
	XtManageChild (scale_w);

	/* make a frame for the drawing area.
	 * attach both top and bottom so it's the one to follow resizing.
	 */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, mb_w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNbottomWidget, dt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, scale_w); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNshadowType, XmSHADOW_ETCHED_OUT); n++;
	frame_w = XmCreateFrame (satform_w, "SatFrame", args, n);
	XtManageChild (frame_w);

	    /* make a drawing area for drawing the little map */

	    n = 0;
	    sda_w = XmCreateDrawingArea (frame_w, "Map", args, n);
	    XtAddCallback (sda_w, XmNexposeCallback, sm_da_exp_cb, 0);
	    XtManageChild (sda_w);
}

/* make the statistics form dialog */
static void
sm_create_ssform_w()
{
	Widget w;
	Widget r_w, title_w, col_w;
	Arg args[20];
	int n;
	int i;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNresizePolicy, XmRESIZE_NONE); n++;
	XtSetArg (args[n], XmNverticalSpacing, 5); n++;
	ssform_w = XmCreateFormDialog (toplevel_w, "SaturnStats", args, n);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Saturn info"); n++;
	XtSetValues (XtParent(ssform_w), args, n);

	/* make top row for ring tilt info */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 50); n++;
	r_w = XmCreatePushButton (ssform_w, "SatTiltMsg", args, n);
	XtManageChild (r_w);
	set_xmstring (r_w, XmNlabelString, "Ring tilt (degs):");
	sm_set_a_button (r_w, False);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 50); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNuserData, "Saturn.Tilt"); n++;
	w = ringt_w = XmCreatePushButton(ssform_w, "SatTilt", args, n);
	XtAddCallback(w, XmNactivateCallback, sm_activate_cb, 0);
	XtManageChild (w);

	/* make table title label */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, r_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	title_w = XmCreateLabel (ssform_w, "SatLab", args, n);
	XtManageChild (title_w);
	set_xmstring (title_w, XmNlabelString,
				" \nMoon positions -- in Saturn Radii");

	/* make the moon table, one column at a time */

	/* moon designator column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_BEGINNING); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (ssform_w, "SatDes", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, " ", args, n);
	    XtManageChild (w);
	    sm_set_a_button (w, False);

	    for (i = 0; i < NM; i++) {
		n = 0;
		w = XmCreatePushButton (col_w, mnames[i].tag, args, n);
		XtManageChild (w);
		sm_set_a_button (w, False);
	    }

	/* moon name column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 8); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_BEGINNING); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (ssform_w, "SatName", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, " ", args, n);
	    XtManageChild (w);
	    sm_set_a_button (w, False);

	    for (i = 0; i < NM; i++) {
		n = 0;
		w = XmCreatePushButton (col_w, mnames[i].full, args, n);
		XtManageChild (w);
		sm_set_a_button (w, False);
	    }

	/* moon X column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 30); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (ssform_w, "SatX", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, "SatLab", args, n);
	    XtManageChild (w);
	    sm_set_a_button (w, False);
	    set_xmstring (w, XmNlabelString, "X (+E)");

	    for (i = 0; i < NM; i++) {
		char *sel;
		sel = XtMalloc (strlen(mnames[i].full) + 3); /* '.X\0' */
		(void) sprintf (sel, "%s.X", mnames[i].full);
		n = 0;
		XtSetArg (args[n], XmNuserData, sel); n++;
		w = s_w[i][CX] = XmCreatePushButton(col_w, "SatPB", args, n);
		XtAddCallback(w, XmNactivateCallback, sm_activate_cb, 0);
		XtManageChild (w);
	    }

	/* moon Y column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 50); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (ssform_w, "SatY", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, "SatLab", args, n);
	    XtManageChild (w);
	    sm_set_a_button (w, False);
	    set_xmstring (w, XmNlabelString, "Y (+S)");

	    for (i = 0; i < NM; i++) {
		char *sel;
		sel = XtMalloc (strlen(mnames[i].full) + 3); /* '.Y\0' */
		(void) sprintf (sel, "%s.Y", mnames[i].full);
		n = 0;
		XtSetArg (args[n], XmNuserData, sel); n++;
		w = s_w[i][CY] = XmCreatePushButton(col_w, "SatPB", args, n);
		XtAddCallback(w, XmNactivateCallback, sm_activate_cb, 0);
		XtManageChild (w);
	    }

	/* moon Z column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 70); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (ssform_w, "SatZ", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, "SatLab", args, n);
	    XtManageChild (w);
	    sm_set_a_button (w, False);
	    set_xmstring (w, XmNlabelString, "Z (+front)");

	    for (i = 0; i < NM; i++) {
		char *sel;
		sel = XtMalloc (strlen(mnames[i].full) + 3); /* '.Z\0' */
		(void) sprintf (sel, "%s.Z", mnames[i].full);
		n = 0;
		XtSetArg (args[n], XmNuserData, sel); n++;
		w = s_w[i][CZ] = XmCreatePushButton(col_w, "SatPB", args, n);
		XtAddCallback(w, XmNactivateCallback, sm_activate_cb, 0);
		XtManageChild (w);
	    }

	/* moon mag column */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, title_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 90); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
	XtSetArg (args[n], XmNisAligned, True); n++;
	col_w = XmCreateRowColumn (ssform_w, "SatMag", args, n);
	XtManageChild (col_w);

	    n = 0;
	    w = XmCreatePushButton (col_w, "SatLab", args, n);
	    XtManageChild (w);
	    sm_set_a_button (w, False);
	    set_xmstring (w, XmNlabelString, "Mag");

	    for (i = 0; i < NM; i++) {
		char *sel;
		sel = XtMalloc (strlen(mnames[i].full) + 5); /* '.Mag\0' */
		(void) sprintf (sel, "%s.Mag", mnames[i].full);
		n = 0;
		XtSetArg (args[n], XmNuserData, sel); n++;
		w = s_w[i][CMAG] = XmCreatePushButton(col_w, "SatPB",args,n);
		XtAddCallback(w, XmNactivateCallback, sm_activate_cb, 0);
		XtManageChild (w);
	    }

	/* make a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, col_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (ssform_w, "Sep1", args, n);
	XtManageChild (w);

	/* make a date/time stamp */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	sdt_w = XmCreateLabel (ssform_w, "SDateStamp", args, n);
	XtManageChild (sdt_w);

	/* make a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, sdt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (ssform_w, "Sep2", args, n);
	XtManageChild (w);

	/* make the close button */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 30); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 70); n++;
	w = XmCreatePushButton (ssform_w, "Close", args, n);
	XtAddCallback (w, XmNactivateCallback, sm_sstats_cb, 0);
	XtManageChild (w);
}

/* go through all the buttons pickable for plotting and set whether they
 * should appear to look like buttons or just flat labels.
 */
static void
sm_set_buttons (whether)
int whether;	/* whether setting up for plotting or for not plotting */
{
	int i;

	for (i = 0; i < NM; i++) {
	    sm_set_a_button (s_w[i][CX], whether);
	    sm_set_a_button (s_w[i][CY], whether);
	    sm_set_a_button (s_w[i][CZ], whether);
	    sm_set_a_button (s_w[i][CMAG], whether);
	}
	sm_set_a_button (ringt_w, whether);
}

/* set whether the given button looks like a label.
 */
static void
sm_set_a_button(pbw, whether)
Widget pbw;
int whether;
{
	static Arg look_like_button[] = {
	    {XmNtopShadowColor, (XtArgVal) 0},
	    {XmNbottomShadowColor, (XtArgVal) 0},
            {XmNtopShadowPixmap, (XtArgVal) 0},
            {XmNbottomShadowPixmap, (XtArgVal) 0},
	    {XmNfillOnArm, (XtArgVal) True},
	    {XmNtraversalOn, (XtArgVal) True},
	};
	static Arg look_like_label[] = {
	    {XmNtopShadowColor, (XtArgVal) 0},
	    {XmNbottomShadowColor, (XtArgVal) 0},
            {XmNtopShadowPixmap, (XtArgVal) 0},
            {XmNbottomShadowPixmap, (XtArgVal) 0},
	    {XmNfillOnArm, (XtArgVal) False},
	    {XmNtraversalOn, (XtArgVal) False},
	};
	static int called;
	Arg *ap;
	int na;

	if (!called) {
	    /* get baseline label and shadow appearances.
	     */
            Pixel topshadcol, botshadcol, bgcol;
            Pixmap topshadpm, botshadpm;
	    Arg args[20];
	    Widget tmpw;
	    int n;

	    n = 0;
	    tmpw = XmCreatePushButton (satform_w, "tmp", args, n);

	    n = 0;
	    XtSetArg (args[n], XmNtopShadowColor, &topshadcol); n++;
	    XtSetArg (args[n], XmNbottomShadowColor, &botshadcol); n++;
            XtSetArg (args[n], XmNtopShadowPixmap, &topshadpm); n++;
            XtSetArg (args[n], XmNbottomShadowPixmap, &botshadpm); n++;
	    XtSetArg (args[n], XmNbackground, &bgcol); n++;
	    XtGetValues (tmpw, args, n);

            look_like_button[0].value = topshadcol;
            look_like_button[1].value = botshadcol;
            look_like_button[2].value = topshadpm;
            look_like_button[3].value = botshadpm;
            look_like_label[0].value = bgcol;
            look_like_label[1].value = bgcol;
            look_like_label[2].value = XmUNSPECIFIED_PIXMAP;
            look_like_label[3].value = XmUNSPECIFIED_PIXMAP;

	    XtDestroyWidget (tmpw);
	     
	    called = 1;
	}

	if (whether) {
	    ap = look_like_button;
	    na = XtNumber(look_like_button);
	} else {
	    ap = look_like_label;
	    na = XtNumber(look_like_label);
	}

	XtSetValues (pbw, ap, na);
}

/* callback from the Close button is activated on the stats menu or when
 * the More Info button is activated. they each do the same thing.
 */
/* ARGSUSED */
static void
sm_sstats_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (sswasman = XtIsManaged(ssform_w))
	    XtUnmanageChild (ssform_w);
	else {
	    XtManageChild (ssform_w);
	    sm_set_buttons(sm_selecting);
	}
}

/* callback from the big dots toggle button
 * TODO: really shouldn't get present time, just redo dots in same location.
 */
/* ARGSUSED */
static void
sm_bigd_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	bigdots = XmToggleButtonGetState(w);
	sm_update (mm_get_now(), 1);
}

/* callback from the tags toggle button
 * TODO: really shouldn't get present time, just redo dots in same location.
 */
/* ARGSUSED */
static void
sm_tags_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	s_tags = XmToggleButtonGetState(w);
	sm_update (mm_get_now(), 1);
}

/* callback from the flip toggle buttons.
 * client points to one of flip_ns or flip_ew.
 * TODO: really shouldn't get present time, just redo at same location
 */
/* ARGSUSED */
static void
sm_flip_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int *p = (int *)client;
	
	*p = XmToggleButtonGetState(w);
	sm_update (mm_get_now(), 1);
}

/* callback from the scale.
 * TODO: really shouldn't get present time, just redo dots in same location.
 */
/* ARGSUSED */
static void
sm_scale_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	sm_update (mm_get_now(), 1);
}

/* callback from any of the data menu buttons being activated.
 */
/* ARGSUSED */
static void
sm_activate_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (sm_selecting) {
	    char *name;
	    get_something (w, XmNuserData, (char *)&name);
	    register_selection (name);
	}
}

/* callback from the main Close button
 */
/* ARGSUSED */
static void
sm_close_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XtUnmanageChild (satform_w);
	if (sswasman = XtIsManaged(ssform_w))
	    XtUnmanageChild (ssform_w);
}

/* callback from the Movie button
 */
/* ARGSUSED */
static void
sm_movie_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	mm_movie (MOVIE_STEPSZ);
}

/* callback from either expose or resize of the drawing area.
 */
/* ARGSUSED */
static void
sm_da_exp_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	XmDrawingAreaCallbackStruct *c = (XmDrawingAreaCallbackStruct *)call;

	/* filter out a few oddball cases */
	switch (c->reason) {
	case XmCR_EXPOSE: {
	    /* turn off gravity so we get expose events for either shrink or
	     * expand.
	     */
	    static before;
	    XExposeEvent *e = &c->event->xexpose;

	    if (!before) {
		XSetWindowAttributes swa;
		swa.bit_gravity = ForgetGravity;
		XChangeWindowAttributes (e->display, e->window, 
							    CWBitGravity, &swa);
		before = 1;
	    }
	    /* wait for the last in the series */
	    if (e->count != 0)
		return;
	    break;
	    }
	default:
	    printf ("Unexpected satform_w event. type=%d\n", c->reason);
	    exit(1);
	}

	sm_update (mm_get_now(), 1);
}

/* functions */
static void saturn_data();
static void sm_draw_map ();

/* called to recompute and fill in values for the saturn menu.
 * don't bother if it doesn't exist or is unmanaged now or no one is logging.
 */
void
sm_update (np, how_much)
Now *np;
int how_much;
{
	static char fmt[] = "%7.3f";
	int wantstats;
        double jd;
	double X[9], Y[9], Z[9], MAG[9];
	double tilt;
	int i;

	if (!satform_w)
	    return;

	wantstats = XtIsManaged(ssform_w) || any_ison() || how_much;

	if (!XtIsManaged(satform_w) && !wantstats)
	    return;

	/* compute saturn info */
        jd = mjd + MJD0;
        saturn_data(jd, X, Y, Z, MAG, &tilt);

	/* convert to our signs and indeces */
	for (i = 0; i < NM; i++) {
	    X[i] =  X[i+1];
	    Y[i] = -Y[i+1];
	    Z[i] = -Z[i+1];
	    MAG[i] = MAG[i+1];
	}
	tilt = -raddeg(tilt);

	if (wantstats) {
	    for (i = 0; i < NM; i++) {
		f_double (s_w[i][CX], fmt, X[i]);
		f_double (s_w[i][CY], fmt, Y[i]);
		f_double (s_w[i][CZ], fmt, Z[i]);
		f_double (s_w[i][CMAG], "%5.1f", MAG[i]);
	    }

	    f_double (ringt_w, fmt, tilt);
	    timestamp (np, sdt_w);
	}

	if (XtIsManaged(satform_w)) {
	    sm_draw_map (sda_w, X, Y, Z, tilt);
	    timestamp (np, dt_w);
	}
}

/* given the loc of the moons, draw a nifty little picture.
 * scale of the locations is in terms of saturn radii == 1.
 * unflipped view is S up, E right.
 * index X/Y/Z from 0 .. 7.
 *     +X: East, sat radii
 *     +Y: South, sat radii
 *     +Z: in front, sat radii
 *  +tilt: southern edge of the rings are in front, degrees
 */
static void
sm_draw_map (w, X, Y, Z, tilt)
Widget w;
double X[], Y[], Z[];
double tilt;	/* degrees */
{
	static GC s_fgc, s_bgc, s_xgc;
	static XFontStruct *s_fs;
	static last_nx, last_ny;
	static int cw, ch;
	static Pixmap pm;
	Display *dsp = XtDisplay(w);
	Window win = XtWindow(w);
	Window root;
	double scale;
	int sv;
	char c;
	int x, y;
	unsigned int nx, ny, bw, d;
	int irw, orw, irh, orh, irx, iry, orx, ory;
	int ns = flip_ns ? -1 : 1;
	int ew = flip_ew ? -1 : 1;
	int i;

#define	RLW	3	/* ring line width, pixels */
#define	NORM	60.0	/* max Iapetus orbit radius; used to normalize */
#define	MAPSCALE(v)	((v)*((int)nx)/NORM/2*scale)
#define	XCORD(x)	((int)(((int)nx)/2.0 + ew*MAPSCALE(x) + 0.5))
#define	YCORD(y)	((int)(((int)ny)/2.0 - ns*MAPSCALE(y) + 0.5))

	if (!s_fgc) {
	    XGCValues gcv;
	    unsigned int gcm;
	    Pixel fg, bg;

	    gcm = GCForeground;
	    get_something (w, XmNforeground, (char *)&fg);
	    gcv.foreground = fg;
	    s_fgc = XCreateGC (dsp, win, gcm, &gcv);
	    s_fs = XQueryFont (dsp, XGContextFromGC (s_fgc));
	    cw = s_fs->max_bounds.width;
	    ch = s_fs->max_bounds.ascent + s_fs->max_bounds.descent;

	    gcm = GCForeground;
	    get_something (w, XmNbackground, (char *)&bg);
	    gcv.foreground = bg;
	    s_bgc = XCreateGC (dsp, win, gcm, &gcv);

	    gcm = GCForeground | GCFunction;
	    gcv.foreground = fg ^ bg;
	    gcv.function = GXxor;
	    s_xgc = XCreateGC (dsp, win, gcm, &gcv);
	}

	XmScaleGetValue (scale_w, &sv);
	scale = pow(MAXSCALE, sv/100.0);

	XGetGeometry(dsp, win, &root, &x, &y, &nx, &ny, &bw, &d);
	if (!pm || nx != last_nx || ny != last_ny) {
	    if (pm)
		XFreePixmap (dsp, pm);
	    pm = XCreatePixmap (dsp, win, nx, ny, d);
	    last_nx = nx;
	    last_ny = ny;
	}

	XFillRectangle (dsp, pm, s_bgc, 0, 0, nx, ny);

	/* draw labels */
	c = flip_ew ? 'W' : 'E';
	XDrawString(dsp, pm, s_fgc, nx-cw-1, ny/2-2, &c, 1);
	c = flip_ns ? 'N' : 'S';
	XDrawString(dsp, pm, s_fgc, (nx-cw)/2-1, s_fs->ascent, &c, 1);

	/* draw Saturn in center with unit radius */
	XFillArc (dsp, pm, s_fgc, nx/2-(int)MAPSCALE(1), ny/2-(int)MAPSCALE(1),
			    2*(int)MAPSCALE(1), 2*(int)MAPSCALE(1), 0, 360*64);
	
	/* rings of radius IRR and ORR.
	 * draw rings in front of planet using xor.
	 * positive tilt means the southern edge of the rings are in front.
	 * always draw the solid s_fgc last in case we are near the ring plane.
	 */
#define	IRR	1.528	/* inner edge of ring system */
#define	ORR	2.267	/* outter edge of A ring */
	irh = MAPSCALE(2*IRR*fabs(sin(degrad(tilt))));
	irw = (int)MAPSCALE(2*IRR);
	orh = MAPSCALE(2*ORR*fabs(sin(degrad(tilt))));
	orw = (int)MAPSCALE(2*ORR);
	irx = nx/2 - MAPSCALE(IRR);
	iry = ny/2 - MAPSCALE(IRR*fabs(sin(degrad(tilt))));
	orx = nx/2 - MAPSCALE(ORR);
	ory = ny/2 - MAPSCALE(ORR*fabs(sin(degrad(tilt))));
	if (irh < RLW || orh < RLW) {
	    /* too near the ring plane to draw a fill ellipse */
	    XDrawLine (dsp, pm, s_fgc, orx, ny/2, nx-orx, ny/2);
	} else {
	    /* near rings are up if tilt is positive and we are not flipping n/s
	     * or if tilt is negative and we are flipping n/s.
	     */
	    int nearup = tilt>0.0 == !flip_ns;

	    XDrawArc (dsp, pm, s_xgc, irx, iry, irw, irh,
					     nearup ? 0 : 180*64, 180*64-1);
	    XDrawArc (dsp, pm, s_xgc, orx, ory, orw, orh,
					     nearup ? 0 : 180*64, 180*64-1);
	    XDrawArc (dsp, pm, s_fgc, irx, iry, irw, irh,
					     nearup ? 180*64 : 0, 180*64-1);
	    XDrawArc (dsp, pm, s_fgc, orx, ory, orw, orh,
					     nearup ? 180*64 : 0, 180*64-1);
	}

	/* draw each moon that is visible.
	 */
	for (i = 0; i < NM; i++) {
	    double mx = X[i];
	    double my = Y[i];
	    double mz = Z[i];
	    int outside = mx*mx + my*my > 1.0;
	    int infront = mz > 0.0;

	    if (!outside && !infront)
		continue;	/* behind saturn */

	    x = XCORD(mx);
	    y = YCORD(my);
	    XDrawPoint (dsp, pm, s_xgc, x, y);
	    if (bigdots) {
		XDrawPoint(dsp,pm, s_xgc, x+1, y);
		XDrawPoint(dsp,pm, s_xgc, x,   y+1);
		XDrawPoint(dsp,pm, s_xgc, x+1, y+1);
	    }
	    if (s_tags)
		XDrawString(dsp, pm, s_xgc, x-cw/2, y+2*ch,
					mnames[i].tag, strlen(mnames[i].tag));
	}

	XCopyArea (dsp, pm, win, s_fgc, 0, 0, nx, ny, 0, 0);
}

/* following is translitered from BASIC of Dan Bruton's satsat2.
 * ECD: comments from Elwood C. Downey
 * ECD: Jan 10, 1995
 */

/*  */
/*     SS2TXT.BAS                     Dan Bruton, astro@tamu.edu */
/*  */
/*       This is a text version of SATSAT2.BAS.  It is smaller, */
/*    making it easier to convert other languages (250 lines */
/*    compared to 850 lines). */
/*  */
/*       This BASIC program computes and displays the locations */
/*    of Saturn's Satellites for a given date and time.  See */
/*    "Practical Astronomy with your Calculator" by Peter */
/*    Duffett-Smith and the Astronomical Almanac for explanations */
/*    of some of the calculations here.  The code is included so */
/*    that users can make changes or convert to other languages. */
/*    This code was made using QBASIC (comes with DOS 5.0). */
/*  */

/* ECD: BASICeze */
#define	FOR	for
#define	IF	if
#define	ELSE	else
#define	COS	cos
#define	SIN	sin
#define	TAN	tan
#define ATN	atan
#define ABS	fabs
#define SQR	sqrt

/* ECD: this returns +X:East +Y:North +Z:behind in [1..8] indeces.
 * ECD: +tilt:north, rads
 */
static void
saturn_data (JD, X, Y, Z, MAG, IP)
double JD;
double X[], Y[], Z[], MAG[];
double *IP;	/* tilt */
{
     /* ECD: we don't use [0] */
     /* why 11? seems like 9 would do */
     double SMA[11], E[11], U[11], U0[11], CS[11], PD[11], RSS[11];

     double P,TP,TE,EP,EE,RE0,RP0,OBL,RS,RS1,RS2,RS3,RS4;
     double JDE,LPE,LPP,OHM,INCS,LEE,LEP;
     double NN,ME,MP,VE,VP;
     double LE,LP,RE,RP,DT,II,F,F1;
     double SINPSI,PSI,NUM,DEN,LPQ,RPQ,LAMB,BETA,RA,SINDEC,DECL;
     double TVA,PVA,TVC,PVC,DOT1,INC,TVB,PVB,DOT2,INCI;
     double TRIP,GAM,TEMPX,TEMPY,TEMPZ,TEMP;
     int N, I;

     /*  ******************************************************************** */
     /*  *                                                                  * */
     /*  *                        Constants                                 * */
     /*  *                                                                  * */
     /*  ******************************************************************** */
     P = PI / 180;
     /*  Orbital Rate of Saturn in Radians per Days */
     TP = 2 * PI / (29.45771 * 365.2422);
     /*  Orbital Rate of Earth in Radians per Day */
     TE = 2 * PI / (1.00004 * 365.2422);
     /*  Eccentricity of Saturn's Orbit */
     EP = .0556155;
     /*  Eccentricity of Earth's Orbit */
     EE = .016718;
     /*  Semimajor axis of Earth's and Saturn's orbit in Astronomical Units */
     RE0 = 1; RP0 = 9.554747;
     /*  Tilt of Earth's Axis - obliquity of the ecliptic */
     OBL = 23.43928 * P;
     /*  Visual Magnitude of the Satellites */
     MAG[1] = 13; MAG[2] = 11.8; MAG[3] = 10.3; MAG[4] = 10.2;
     MAG[5] = 9.8; MAG[6] = 8.4; MAG[7] = 14.3; MAG[8] = 11.2;
     /*  Semimajor Axis of the Satellites' Orbit in Kilometers */
     SMA[1] = 185600; SMA[2] = 238100; SMA[3] = 294700; SMA[4] = 377500;
     SMA[5] = 527200; SMA[6] = 1221600; SMA[7] = 1483000; SMA[8] = 3560100;
     /*  Eccentricity of Satellites' Orbit [Program uses 0] */
     /*  Synodic Orbital Period of Moons in Days */
     PD[1] = .9425049;
     PD[2] = 1.3703731;
     PD[3] = 1.8880926;
     PD[4] = 2.7375218;
     PD[5] = 4.5191631;
     PD[6] = 15.9669028;
     PD[7] = 21.3174647;
     PD[8] = 79.9190206;	/* personal mail 1/14/95 */
     RS = 60330; /*  Radius of planet in kilometers */
     RS1 = 92000; /*  Inner radius of inner ring in kilometers */
     RS2 = 117500; /*  Outer radius of inner ring in kilometers */
     RS3 = 122500; /*  Inner radius of outer ring in kilometers */
     RS4 = 136500; /*  Outer radius of outer ring in kilometers */
    
     /*  ******************************************************************** */
     /*  *                                                                  * */
     /*  *                      Epoch Information                           * */
     /*  *                                                                  * */
     /*  ******************************************************************** */
     JDE = 2444238.5; /*  Epoch Jan 0.0 1980 = December 31,1979 0:0:0 UT */
     LPE = 165.322242 * P; /*  Longitude of Saturn at Epoch */
     LPP = 92.6653974 * P; /*  Longitude of Saturn`s Perihelion */
     OHM = 113.4888341 * P; /*  Longitude of Saturn`s Ascending Node */
     INCS = 2.4893741 * P; /*  Inclination of Saturn`s Orbit */
     LEE = 98.83354 * P; /*  Longitude of Earth at Epoch */
     LEP = 102.596403 * P; /*  Longitude of Earth's Perihelion */
     /*  U0[I] = Angle from inferior geocentric conjuction */
     /*          measured westward along the orbit at epoch */
     U0[1] = 18.2919 * P;
     U0[2] = 174.2135 * P;
     U0[3] = 172.8546 * P;
     U0[4] = 76.8438 * P;
     U0[5] = 37.2555 * P;
     U0[6] = 57.7005 * P;
     U0[7] = 266.6977 * P;
     U0[8] = 195.3513 * P;	/* from personal mail 1/14/195 */
    
     /*  ******************************************************************** */
     /*  *                                                                  * */
     /*  *                    Orbit Calculations                            * */
     /*  *                                                                  * */
     /*  ******************************************************************** */
     /*  ****************** FIND MOON ORBITAL ANGLES ************************ */
     NN = JD - JDE; /*  NN = Number of days since epoch */
     ME = ((TE * NN) + LEE - LEP); /*  Mean Anomoly of Earth */
     MP = ((TP * NN) + LPE - LPP); /*  Mean Anomoly of Saturn */
     VE = ME; VP = MP; /*  True Anomolies - Solve Kepler's Equation */
     FOR (I = 1; I <= 3; I++) {
	 VE = VE - (VE - (EE * SIN(VE)) - ME) / (1 - (EE * COS(VE)));
	 VP = VP - (VP - (EP * SIN(VP)) - MP) / (1 - (EP * COS(VP)));
     }
     VE = 2 * ATN(SQR((1 + EE) / (1 - EE)) * TAN(VE / 2));
     IF (VE < 0) VE = (2 * PI) + VE;
     VP = 2 * ATN(SQR((1 + EP) / (1 - EP)) * TAN(VP / 2));
     IF (VP < 0) VP = (2 * PI) + VP;
     /*   Heliocentric Longitudes of Earth and Saturn */
     LE = VE + LEP; IF (LE > (2 * PI)) LE = LE - (2 * PI);
     LP = VP + LPP; IF (LP > (2 * PI)) LP = LP - (2 * PI);
     /*   Distances of Earth and Saturn from the Sun in AU's */
     RE = RE0 * (1 - EE * EE) / (1 + EE * COS(VE));
     RP = RP0 * (1 - EP * EP) / (1 + EP * COS(VP));
     /*   DT = Distance from Saturn to Earth in AU's - Law of Cosines */
     DT = SQR((RE * RE) + (RP * RP) - (2 * RE * RP * COS(LE - LP)));
     /*   II = Angle between Earth and Sun as seen from Saturn */
     II = RE * SIN(LE - LP) / DT;
     II = ATN(II / SQR(1 - II * II)); /*   ArcSIN and Law of Sines */
     /*    F = NN - (Light Time to Earth in days) */
     F = NN - (DT / 173.83);
     F1 = II + MP - VP;
     /*  U(I) = Angle from inferior geocentric conjuction measured westward */
     FOR (I = 1; I <= NM; I++) {
	U[I] = U0[I] + (F * 2 * PI / PD[I]) + F1;
	U[I] = ((U[I] / (2 * PI)) - (int)(U[I] / (2 * PI))) * 2 * PI;

     }

     /*  ******************** FIND SATURN'S COORDINATES ********************* */
#ifdef USE_BRUTON
     /*  PSI -> Heliocentric Latitude */
     /*  LAMB -> Geocentric Ecliptic Longitude */
     /*  BETA -> Geocentric Ecliptic Latitude */
     SINPSI = SIN(LP - OHM) * SIN(INCS);
     PSI = ATN(SINPSI / SQR(1 - SINPSI * SINPSI)); /*  ArcSIN */
     NUM = SIN(LP - OHM) * COS(INCS);
     DEN = COS(LP - OHM);
     LPQ = ATN(NUM / DEN);
     IF (NUM * DEN < 0) LPQ = LPQ + PI;
     IF (NUM < 0) LPQ = LPQ + PI;
     LPQ = LPQ + OHM;
     RPQ = RP * COS(PSI);
     NUM = RE * SIN(LPQ - LE);
     DEN = RPQ - RE * COS(LPQ - LE);
     LAMB = ATN(NUM / DEN);
     IF (NUM * DEN < 0) LAMB = LAMB + PI;
     IF (NUM < 0) LAMB = LAMB + PI;
     LAMB = LAMB + LPQ;
     BETA = RPQ * TAN(PSI) * SIN(LAMB - LPQ);
     BETA = ATN(BETA / (RE * SIN(LPQ - LE)));
     NUM = (SIN(LAMB) * COS(OBL)) - (TAN(BETA) * SIN(OBL));
     DEN = COS(LAMB);
     RA = ATN(NUM / DEN);
     IF (NUM * DEN < 0) RA = RA + PI;
     IF (NUM < 0) RA = RA + PI;
     SINDEC = SIN(BETA) * COS(OBL);
     SINDEC = SINDEC + COS(BETA) * SIN(OBL) * SIN(LAMB);
     DECL = ATN(SINDEC / SQR(1 - SINDEC * SINDEC)); /*  ArcSIN */

#else /* use xephem -- really helps with ring tilt */
     {
	Obj o;

	o.type = PLANET;
	o.pl.code = SATURN;

	(void) obj_cir (mm_get_now(), &o);
	RA = o.s_ra;
	DECL = o.s_dec;
     }
#endif

     /*  **************** FIND INCLINATION OF RINGS ************************* */
     /*  Use dot product of Earth-Saturn vector and Saturn's rotation axis */
     TVA = (90 - 83.51) * P; /*  Theta coordinate of Saturn's axis */
     PVA = 40.27 * P; /*  Phi coordinate of Saturn's axis */
     TVC = (PI / 2) - DECL;
     PVC = RA;
     DOT1 = SIN(TVA) * COS(PVA) * SIN(TVC) * COS(PVC);
     DOT1 = DOT1 + SIN(TVA) * SIN(PVA) * SIN(TVC) * SIN(PVC);
     DOT1 = DOT1 + COS(TVA) * COS(TVC);
     INC = ATN(SQR(1 - DOT1 * DOT1) / DOT1); /*    ArcCOS */
     IF (INC > 0) INC = (PI / 2) - INC; ELSE INC = -(PI / 2) - INC;
     /*  ************* FIND INCLINATION OF IAPETUS' ORBIT ******************* */
     /*  Use dot product of Earth-Saturn vector and Iapetus' orbit axis */
     /*  Vector B */
     TVB = (90 - 75.6) * P; /*  Theta coordinate of Iapetus' orbit axis (estimate) */
     PVB = 21.34 * 2 * PI / 24; /*  Phi coordinate of Iapetus' orbit axis (estimate) */
     DOT2 = SIN(TVB) * COS(PVB) * SIN(TVC) * COS(PVC);
     DOT2 = DOT2 + SIN(TVB) * SIN(PVB) * SIN(TVC) * SIN(PVC);
     DOT2 = DOT2 + COS(TVB) * COS(TVC);
     INCI = ATN(SQR(1 - DOT2 * DOT2) / DOT2); /*    ArcCOS */
     IF (INCI > 0) INCI = (PI / 2) - INCI; ELSE INCI = -(PI / 2) - INCI;
     /*  ************* FIND ROTATION ANGLE OF IAPETUS' ORBIT **************** */
     /*  Use inclination of Iapetus' orbit with respect to ring plane */
     /*  Triple Product */
     TRIP = SIN(TVC) * COS(PVC) * SIN(TVA) * SIN(PVA) * COS(TVB);
     TRIP = TRIP - SIN(TVC) * COS(PVC) * SIN(TVB) * SIN(PVB) * COS(TVA);
     TRIP = TRIP + SIN(TVC) * SIN(PVC) * SIN(TVB) * COS(PVB) * COS(TVA);
     TRIP = TRIP - SIN(TVC) * SIN(PVC) * SIN(TVA) * COS(PVA) * COS(TVB);
     TRIP = TRIP + COS(TVC) * SIN(TVA) * COS(PVA) * SIN(TVB) * SIN(PVB);
     TRIP = TRIP - COS(TVC) * SIN(TVB) * COS(PVB) * SIN(TVA) * SIN(PVA);
     GAM = -1 * ATN(TRIP / SQR(1 - TRIP * TRIP)); /*  ArcSIN */
    
     /*  ******************************************************************** */
     /*  *                                                                  * */
     /*  *                     Compute Moon Positions                       * */
     /*  *                                                                  * */
     /*  ******************************************************************** */
     FOR (I = 1; I <= NM - 1; I++) {
	 X[I] = -1 * SMA[I] * SIN(U[I]) / RS;
	 Z[I] = -1 * SMA[I] * COS(U[I]) / RS;	/* ECD */
	 Y[I] = SMA[I] * COS(U[I]) * SIN(INC) / RS;
     }
     /*  ************************* Iapetus' Orbit *************************** */
     TEMPX = -1 * SMA[8] * SIN(U[8]) / RS;
     TEMPZ = -1 * SMA[8] * COS(U[8]) / RS;
     TEMPY = SMA[8] * COS(U[8]) * SIN(INCI) / RS;
     X[8] = TEMPX * COS(GAM) + TEMPY * SIN(GAM); /*       Rotation */
     Z[8] = TEMPZ * COS(GAM) + TEMPY * SIN(GAM);
     Y[8] = -1 * TEMPX * SIN(GAM) + TEMPY * COS(GAM);
    
#ifdef SHOWALL
     /*  ******************************************************************** */
     /*  *                                                                  * */
     /*  *                          Show Results                            * */
     /*  *                                                                  * */
     /*  ******************************************************************** */
     printf ("                           Julian Date : %g\n", JD);
     printf ("             Right Ascension of Saturn : %g Hours\n", RA * 24 / (2 * PI));
     printf ("                 Declination of Saturn : %g\n", DECL / P);
     printf ("   Ring Inclination as seen from Earth : %g\n", -1 * INC / P);
     printf ("      Heliocentric Longitude of Saturn : %g\n", LP / P);
     printf ("       Heliocentric Longitude of Earth : %g\n", LE / P);
     printf ("                Sun-Saturn-Earth Angle : %g\n", II / P);
     printf ("     Distance between Saturn and Earth : %g AU = %g million miles\n", DT, (DT * 93));
     printf ("       Light time from Saturn to Earth : %g minutes\n", DT * 8.28);
     TEMP = 2 * ATN(TAN(165.6 * P / (2 * 3600)) / DT) * 3600 / P;
     printf ("                Angular Size of Saturn : %g arcsec\n", TEMP);
     printf ("  Major Angular Size of Saturn's Rings : %g arcsec\n", RS4 * TEMP / RS);
     printf ("  Minor Angular Size of Saturn's Rings : %g arcsec\n", ABS(RS4 * TEMP * SIN(INC) / RS));
#endif

     /* ECD: caller wants INC */
     *IP = INC;
}
