/*
 * callbacks.c -- most of the callbacks for GhostView-VMS.
 * Copyright (C) 1995  Johannes Plass
 *   Author: Johannes Plass
 *           Department of Physics
 *           Johannes-Gutenberg University, Mainz, Germany
 * Internet: plass@dipmza.physik.uni-mainz.de
*/

/*
 * This code is derived from:
*/

/*
 * callbacks.c -- X11 callbacks for ghostview.
 * Copyright (C) 1992  Timothy O. Theisen
 *   Author: Tim Theisen           Systems Programmer
 * Internet: tim@cs.wisc.edu       Department of Computer Sciences
 *     UUCP: uwvax!tim             University of Wisconsin-Madison
 *    Phone: (608)262-0438         1210 West Dayton Street
 *      FAX: (608)262-9777         Madison, WI   53706
*/

/*
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>

#ifndef BUFSIZ
#   define BUFSIZ 1024
#endif

/*
#define MESSAGES
#define MESSAGE_NO_ESC
*/
#include "message.h"

#ifdef VMS
#   include <X11_DIRECTORY/Intrinsic.h>
#   include <X11_DIRECTORY/StringDefs.h>
#   include <X11_DIRECTORY/Shell.h>
#   include <XAW_DIRECTORY/Cardinals.h>
#   include <XAW_DIRECTORY/AsciiText.h>
#   include <XAW_DIRECTORY/Scrollbar.h>
#   include <XAW_DIRECTORY/Form.h>
#   include <XAW_DIRECTORY/Command.h>
#   include <XAW_DIRECTORY/Panner.h>
#   include <XAW_DIRECTORY/ThreeD.h>
#   include <X11_DIRECTORY/IntrinsicP.h>
#   include <XAW_DIRECTORY/ViewportP.h>
#   ifdef _FILESEL_LOCAL_
#      include "Filesel.h"
#   else
#      include <XAW_DIRECTORY/Filesel.h>
#   endif
#else
#   include <X11/Intrinsic.h>
#   include <X11/StringDefs.h>
#   include <X11/Shell.h>
#   include <X11/Xaw/Cardinals.h>
#   include <X11/Xaw/AsciiText.h>
#   include <X11/Xaw/Scrollbar.h>
#   include <X11/Xaw/Form.h>
#   include <X11/Xaw/Command.h>
#   include <X11/Xaw/Panner.h>
#endif

#include "gv.h"
#include "ps.h"
#include "info.h"
#include "popup.h"
#include "dialog.h"
#include "note.h"

static char* save_directory = NULL;
static char* open_directory = NULL;

/*############################################################*/
/* cb_movePanner */
/*############################################################*/

void
cb_movePanner(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    Arg args[10];
    Widget scrollh,scrollv;
    float top,shown;
    XawPannerReport *rep = (XawPannerReport*)call_data;

    BEGINMESSAGE(cb_movePanner)
    if (!show_panner) {INFMESSAGE(panner not used)ENDMESSAGE(cb_movePanner)return;}

    scrollh = ((ViewportWidget)pageview)->viewport.horiz_bar;
    scrollv = ((ViewportWidget)pageview)->viewport.vert_bar;
    if ((!scrollh) && (!scrollv)) {
       INFMESSAGE(nothing to scroll) ENDMESSAGE(cb_movePanner)
       return;
    }
    IIMESSAGE(rep->slider_x,rep->slider_y)
    IIMESSAGE(rep->slider_width,rep->slider_height)
    IIMESSAGE(rep->canvas_width,rep->canvas_height)

    if (scrollh) {
        INFMESSAGE(horizontal scrolling)
	XtSetArg(args[0], XtNshown, &shown);
	XtGetValues(scrollh, args, ONE);
        top = ((float)(rep->canvas_width))-((float)(rep->slider_width));
        if (top > 0.0) top = (1.0-shown)*((float)(rep->slider_x))/top;
        else           top = 0.0;
	if (top>(1.0-shown)) top = (1.0-shown);
	if (top<(0.0)) top = 0.0;
        FMESSAGE(top)
	XtCallCallbacks(scrollh, XtNjumpProc, (XtPointer)&top); 
    }

    if (scrollv) {
        INFMESSAGE(vertical scrolling)
	XtSetArg(args[0], XtNshown, &shown);
	XtGetValues(scrollv, args, ONE);
        top = ((float)(rep->canvas_height))-((float)(rep->slider_height));
        if (top > 0.0) top = (1.0-shown)*((float)(rep->slider_y))/top;
        else           top = 0.0;
	if (top>(1.0-shown)) top = (1.0-shown);
	if (top<(0.0)) top = 0.0;
        FMESSAGE(top)
	XtCallCallbacks(scrollv, XtNjumpProc, (XtPointer)&top);
    }

    ENDMESSAGE(cb_movePanner)
}

/*############################################################*/
/* cb_moveSlider */
/*############################################################*/

void
cb_moveSlider(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    static Dimension opw=0,oph=0,opvw=0,opvh=0;
    static Position opvx=0,opvy=0;
    XawPannerReport *report = (XawPannerReport*) call_data; 
    Dimension pw  = (Dimension) (report->canvas_width);
    Dimension ph  = (Dimension) (report->canvas_height);
    Dimension pvw = (Dimension) (report->slider_width);
    Dimension pvh = (Dimension) (report->slider_height);
    Position  pvx = (Position)  (report->slider_x);
    Position  pvy = (Position)  (report->slider_y);

    BEGINMESSAGE(cb_moveSlider)
    if (!show_panner) {INFMESSAGE(panner not used)ENDMESSAGE(cb_moveSlider)return;}
    
    if ((pw!=opw)||(ph!=oph)||(pvw!=opvw)||(pvh!=opvh)||(pvx!=opvx)||(pvy!=opvy)) {
       Arg args[5];
       Cardinal n;
       Dimension sw,sh,cw,ch;
       Position  sx,sy,ss;
       static Dimension osw=0,osh=0;
       static Position  osx=0,osy=0;

       INFMESSAGE(detected changes)
       XtSetArg(args[0], XtNcanvasWidth,&cw);
       XtSetArg(args[1], XtNcanvasHeight,&ch);
       XtGetValues(panner, args, TWO);

       sw = (Dimension)( ( (float)cw * (float)pvw ) / (float)pw +.5 );
       sh = (Dimension)( ( (float)ch * (float)pvh ) / (float)ph +.5 );
       if (pw>pvw) sx = (Position) ((2+cw-sw)*pvx)/(pw-pvw); else sx = 0;
       if (ph>pvh) sy = (Position) ((2+ch-sh)*pvy)/(ph-pvh); else sy = 0;

       IIMESSAGE(cw,ch)
       IIMESSAGE(sw,sh) IIMESSAGE(sx,sy)
       IIMESSAGE(pw,ph) IIMESSAGE(pvw,pvh) IIMESSAGE(pvx,pvy)

       if ((sw!=osw)||(sh!=osh)) {
          INFMESSAGE(redisplay)
                                                                n=0;
          if (sw!=osw) { XtSetArg(args[n], XtNsliderWidth,sw);  n++; }
          if (sh!=osh) { XtSetArg(args[n], XtNsliderHeight,sh); n++; }
          if (sx!=osx) { XtSetArg(args[n], XtNsliderX,sx);      n++; }
          if (sy!=osy) { XtSetArg(args[n], XtNsliderY,sy);      n++; }
          XtSetValues(panner, args, n);
          osw=sw; osh=sh; osx=sx; osy=sy;
       } else if ((sx!=osx)||(sy!=osy)) {
#if defined(VMS) && (XlibSpecificationRelease<5) /* ###jp### added 1.2.95 */
          XtRemoveCallback(panner, XtNreportCallback,cb_movePanner,NULL);
                                                                n=0;
          if (sx!=osx) { XtSetArg(args[n], XtNsliderX,sx);      n++; }
          if (sy!=osy) { XtSetArg(args[n], XtNsliderY,sy);      n++; }
          XtSetValues(panner, args, n);
          XtAddCallback(panner, XtNreportCallback,cb_movePanner,NULL);
#else
          char xstr[15],ystr[15];
          String par[2];
          sprintf(xstr,"%d",sx); par[0]=xstr;
          sprintf(ystr,"%d",sy); par[1]=ystr;
          SMESSAGE(xstr) SMESSAGE(ystr)
          XtRemoveCallback(panner, XtNreportCallback,cb_movePanner,NULL);
          XtCallActionProc(panner,"page",NULL,par,2);
          XtAddCallback(panner, XtNreportCallback,cb_movePanner,NULL);
#endif
          osx=sx; osy=sy;
       }
       opw=pw; oph=ph; opvw=pvw; opvh=pvh; opvx=pvx; opvy=pvy;
    }
    ENDMESSAGE(cb_moveSlider)
}

/*##################################################################*/
/* cb_pageAdjustNotify */
/*##################################################################*/

void
cb_pageAdjustNotify(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    String params[2];
    Cardinal num_params=2;
    params[0]= "adjusted";
    params[1]= (char*) call_data;
    BEGINMESSAGE(cb_pageAdjustNotify)
    action_movePage(page,(XEvent*)NULL,params,&num_params);
    ENDMESSAGE(cb_pageAdjustNotify)
}

/*##################################################################*/
/* cb_checkFile */
/*##################################################################*/

void
cb_checkFile(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    int changed;

    BEGINMESSAGE(cb_checkFile)
    changed = check_file(((int)client_data));
    if (changed==1) show_page(current_page);
    ENDMESSAGE(cb_checkFile)
}

/*##################################################################*/
/* cb_print */
/*##################################################################*/

Boolean *make_pagelist(mode)
   int mode;
{
   int i=0;
   Boolean mode_valid=False;
   Boolean *pagelist=NULL;

   BEGINMESSAGE(make_pagelist)
   if (toc_text && (mode&(PAGE_MODE_CURRENT|PAGE_MODE_MARKED))) {
      pagelist = (Boolean*) XtMalloc((doc->numpages)*sizeof(Boolean));
      for ( i=0; i < doc->numpages; i++ )  { pagelist[i]=False; }
      if (mode==PAGE_MODE_CURRENT) {
         pagelist[current_page]=True;
         mode_valid=True;
      } else if (mode==PAGE_MODE_MARKED) {
         for (i=0; i < doc->numpages; i++) {
	    if (toc_text[toc_entry_length * i] == '*') {
               pagelist[i]=True;
               mode_valid=True;
            }
         }
      }
   }
   if (!mode_valid) {
      if (pagelist) XtFree(pagelist);
      pagelist=NULL;
   }
   ENDMESSAGE(make_pagelist)
   return pagelist;
}

Boolean *get_pagelist(modep)
   int *modep;
{
    Boolean *pagelist=NULL;
    int mode= *modep;

    BEGINMESSAGE(get_pagelist)
    if (toc_text && (mode&(PAGE_MODE_CURRENT|PAGE_MODE_MARKED))) {
      if (mode&PAGE_MODE_MARKED) {
         pagelist=make_pagelist(PAGE_MODE_MARKED);
         if (pagelist) mode=PAGE_MODE_MARKED;
      }
      if (!pagelist && (mode&PAGE_MODE_CURRENT)) {
         pagelist=make_pagelist(PAGE_MODE_CURRENT);
         if (pagelist) mode=PAGE_MODE_CURRENT;
      }
      if (!pagelist) mode=PAGE_MODE_INVALID;
    } else if (mode==PAGE_MODE_ALL) {
      pagelist=NULL; /* all pages */
    } else {
      mode=PAGE_MODE_INVALID;
    }
    *modep=mode;
    ENDMESSAGE(get_pagelist)
    return pagelist;
}

void
cb_print(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    String name, error;
    int found=0;
    char *prompt=GV_PRINT_MESSAGE;
    char *buttonlabel=GV_PRINT_BUTTON_LABEL;
    char *message;
    Boolean *pagelist=NULL;

    BEGINMESSAGE(cb_print)

    if (!filename) {INFMESSAGE(no file) ENDMESSAGE(cb_print) return;}

    gv_print_mode = (int)client_data;
    pagelist=get_pagelist(&gv_print_mode);
    if (pagelist) XtFree(pagelist);
    if (gv_print_mode==PAGE_MODE_INVALID) {INFMESSAGE(invalid print mode)  ENDMESSAGE(cb_print) return;}

    if (app_res.confirm_print) {
       if        (gv_print_mode==PAGE_MODE_MARKED) {
          message=GV_PRINT_MARKED_MESSAGE; INFMESSAGE(printing marked pages)
       } else if (gv_print_mode == PAGE_MODE_CURRENT) {
          message=GV_PRINT_PAGE_MESSAGE;   INFMESSAGE(printing current page)
       } else {
          message=GV_PRINT_ALL_MESSAGE;    INFMESSAGE(printing document)
       }
       DialogPopupSetPrompt(prompt);
       DialogPopupSetMessage(message);
       DialogPopupSetButton(DIALOG_BUTTON_DONE,buttonlabel,cb_doPrint);
       DialogPopupSetButton(DIALOG_BUTTON_CANCEL,NULL,cb_cancelPrint);
#      ifdef VMS
	  DialogPopupSetText(app_res.print_command);
#      else
	  DialogPopupSetText(app_res.printer_variable);
#      endif
       cb_popupDialogPopup((Widget)NULL,NULL,NULL);
       ENDMESSAGE(cb_print)
       return;
    }   
    cb_doPrint((Widget)NULL,NULL,(XtPointer)app_res.print_command);
    ENDMESSAGE(cb_print)
}

/*##################################################################*/
/* cb_doPrint */
/*##################################################################*/

void
cb_doPrint(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    String print_command, error;
    Boolean *pagelist=NULL;

    BEGINMESSAGE(cb_doPrint)

    if (call_data) print_command = (String)(call_data);  /* dialog was not used */  
    else           print_command = DialogPopupGetText(); /* dialog was used */  
    if (!print_command) print_command="";
    SMESSAGE(print_command)

    cb_popdownNotePopup((Widget)NULL,(XtPointer)NULL,NULL);

    pagelist=get_pagelist(&gv_print_mode);

    if ((gv_print_mode != PAGE_MODE_INVALID) && (error = print_file(print_command,pagelist))) {
       NotePopupShowMessage(error);
       XtFree(error);
    } else {
       cb_popdownDialogPopup((Widget)NULL,(XtPointer)NULL,NULL);
    }

    if (pagelist) XtFree(pagelist);

    ENDMESSAGE(cb_doPrint)
}

/*##################################################################*/
/* cb_cancelPrint */
/*##################################################################*/

void
cb_cancelPrint(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    BEGINMESSAGE(cb_cancelPrint)
    cb_popdownNotePopup((Widget)NULL,(XtPointer)NULL,NULL);
    cb_popdownDialogPopup((Widget)NULL,(XtPointer)NULL,NULL);
    ENDMESSAGE(cb_cancelPrint)
}

/*##################################################################*/
/* cb_save */
/*##################################################################*/

void
cb_save(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    Arg args[10];
    Cardinal n;  
    int i;
    int pages = 0;
    char *title="Save";
    char *buttonlabel;
    Widget button = XtNameToWidget(FileSel,"button1");
    Boolean *pagelist;

    BEGINMESSAGE(cb_save)

    gv_save_mode = (int)client_data;
    pagelist=get_pagelist(&gv_save_mode);
    if (pagelist) XtFree(pagelist);
    if (gv_save_mode==PAGE_MODE_INVALID) {INFMESSAGE(invalid save mode)  ENDMESSAGE(cb_save) return;}

    if        (gv_save_mode==PAGE_MODE_MARKED) {
       buttonlabel="Save Marked Pages";  INFMESSAGE(saving marked pages)
    } else if (gv_save_mode==PAGE_MODE_CURRENT) {
       buttonlabel="Save Current Page";  INFMESSAGE(saving current page)
    } else {
       buttonlabel="Save Document";      INFMESSAGE(saving all pages)
    }

    n=0;
    XtSetArg(args[n], XtNtitle,title); ++n;
    XtSetValues(FileSel_popup, args, n);
    n=0;
    XtSetArg(args[n], XtNlabel, buttonlabel); ++n;
    XtSetValues(button,args,n);
    if (!save_directory) {
       if (app_res.default_save_dir) {
          XawFileSelectionSetSelection(FileSel,app_res.default_save_dir,XawFileSelectionPath);
       } else {
          XawFileSelectionSetSelection(FileSel,".",XawFileSelectionPath);
       }
    } else {
          XawFileSelectionSetSelection(FileSel,save_directory,XawFileSelectionPath);
    }
    XawFileSelectionScan(FileSel,XawFileSelectionRescan);
    XtRemoveAllCallbacks(button, XtNcallback);
    XtAddCallback(button, XtNcallback,cb_doSave,NULL);
    positionPopup(FileSel_popup,2,toplevel,50,50, 1,1);
    cb_popupPopup(w, (XtPointer)FileSel_popup, call_data);
    ENDMESSAGE(cb_save)
}

/*##################################################################*/
/* cb_doSave */
/*##################################################################*/

void
cb_doSave(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    String name,error;
    Boolean *pagelist;

    BEGINMESSAGE(cb_doSave)
    name = XawFileSelectionGetSelection(FileSel,XawFileSelectionBoth);
    cb_popdownNotePopup((Widget)NULL,(XtPointer)NULL,NULL);
    if (save_directory) XtFree(save_directory);
    save_directory=XtNewString(XawFileSelectionGetSelection(FileSel,XawFileSelectionPath));
    SMESSAGE(name)
    pagelist=get_pagelist(&gv_save_mode);
    if ((gv_save_mode != PAGE_MODE_INVALID) && (error = save_file(name,pagelist))) {
       NotePopupShowMessage(error);
       XtFree(error);
    } else {
       XtPopdown(FileSel_popup);
    }
    if (pagelist) XtFree(pagelist);

    ENDMESSAGE(cb_doSave)
}

/*##################################################################*/
/* cb_openFile */
/*##################################################################*/

void
cb_openFile(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    Arg args[1];
    Cardinal num_args;
    Widget button = XtNameToWidget(FileSel,"button1");

    BEGINMESSAGE(cb_openFile)

    num_args=0;
    XtSetArg(args[num_args], XtNtitle, "Open File"); ++num_args;
    XtSetValues(FileSel_popup, args, num_args);
    num_args=0;
    XtSetArg(args[num_args], XtNlabel, "Open File"); ++num_args;
    XtSetValues(button, args, num_args);
    XtRemoveAllCallbacks(button, XtNcallback);
    XtAddCallback(button, XtNcallback,cb_doOpenFile,NULL);
    if (open_directory) {
          XawFileSelectionSetSelection(FileSel,open_directory,XawFileSelectionPath);
    }
    XawFileSelectionScan(FileSel,XawFileSelectionRescan);
    positionPopup(FileSel_popup,2,toplevel,50,50, 1,1);
    cb_popupPopup(w, (XtPointer)FileSel_popup, call_data);
    ENDMESSAGE(cb_openFile)
}   

/*##################################################################*/
/* cb_doOpenFile */
/*##################################################################*/

void
cb_doOpenFile(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    String name,error;

    BEGINMESSAGE(cb_doOpenFile)
    name = XawFileSelectionGetSelection(FileSel,XawFileSelectionBoth);
    if (open_directory) XtFree(open_directory);
    open_directory=XtNewString(XawFileSelectionGetSelection(FileSel,XawFileSelectionPath));
    SMESSAGE(name)
    if (error = open_file(name)) {
       NotePopupShowMessage(error);
       XtFree(error);
    } else {
       cb_popdownNotePopup((Widget)NULL,(XtPointer)NULL,NULL);
       XtPopdown(FileSel_popup);
       show_page(REQUEST_NEW_FILE);
    }
    ENDMESSAGE(cb_doOpenFile)
}

/*##################################################################*/
/* cb_reopen */
/* Explicitly reopen the file. */
/*##################################################################*/

void
cb_reopen(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    BEGINMESSAGE(reopen_file)
    show_page(REQUEST_REOPEN);
    ENDMESSAGE(reopen_file)
}

/*##################################################################*/
/* cb_redisplay */
/*##################################################################*/

void
cb_redisplay(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    BEGINMESSAGE(cb_redisplay)
    show_page(REQUEST_REDISPLAY);
    ENDMESSAGE(cb_redisplay)
}

/*##################################################################*/
/* cb_showPreviousPage */
/* If the new_page is different from the current page show it.  */
/* If not at the first page, show the previous page. */
/*##################################################################*/

void
cb_showPreviousPage(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    XawTextPosition pos;
    int new_page;

    BEGINMESSAGE(cb_showPreviousPage)
    if (!toc_text) {INFMESSAGE(no toc) ENDMESSAGE(cb_showPreviousPage) return;}
    if (gv_pending_page_request>NO_CURRENT_PAGE) {
       new_page=gv_pending_page_request-1;
    } else {
       pos = XawTextGetInsertionPoint(toc);
       if ((new_page = pos/toc_entry_length) == current_page) {
          new_page = current_page - 1;
       }
    }
    if (new_page >= 0) show_page(new_page);
    ENDMESSAGE(cb_showPreviousPage)
}

/*##################################################################*/
/* cb_showThisPage */
/* Show this page.  */
/*##################################################################*/

void
cb_showThisPage(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    BEGINMESSAGE(cb_showThisPage)
    if (toc_text) {
        XawTextPosition pos, end;
        int new_page;

        XawTextGetSelectionPos(toc, &pos, &end);
        IMESSAGE(pos) IMESSAGE(end)
        if (pos == end) {
           pos = XawTextGetInsertionPoint(toc);
        }
        new_page = pos/toc_entry_length;
        IMESSAGE(toc_entry_length) IMESSAGE(new_page)
        show_page(new_page);
    } else {
        cb_redisplay((Widget)NULL,NULL,NULL);
    }
    ENDMESSAGE(cb_showThisPage)
}


/*##################################################################*/
/* cb_showNextPage */
/* If the new_page is different from the current page show it.  */
/* If not at the last page, show the next page. */
/*##################################################################*/

void
cb_showNextPage(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    XawTextPosition pos;
    int new_page = 0;

    BEGINMESSAGE(cb_showNextPage)
    if (toc_text) {
       if (gv_pending_page_request>NO_CURRENT_PAGE) {
          INFMESSAGE(pending page request)
          new_page=gv_pending_page_request+1;
       } else {
          pos = XawTextGetInsertionPoint(toc);
          if ((new_page = pos/toc_entry_length) == current_page) {
             new_page = current_page + 1;
          }
       }
       if (new_page >= doc->numpages) { ENDMESSAGE(cb_showNextPage) return; }
    }
    show_page(new_page);
    ENDMESSAGE(cb_showNextPage)
}
         
/*##################################################################*/
/* cb_centerPage */
/* Center the viewport over the page */
/*##################################################################*/

void
cb_centerPage(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    Arg args[1];
    Widget scroll;
    float top, shown;

    BEGINMESSAGE(cb_centerPage)

    scroll = ((ViewportWidget)pageview)->viewport.vert_bar;
    if (scroll != (Widget)NULL) {
	XtSetArg(args[0], XtNshown, &shown);
	XtGetValues(scroll, args, ONE); INFFMESSAGE(vertical,shown)
	top = (1.0-shown)/2.0; INFFMESSAGE(vertical before ,top)
	XtCallCallbacks(scroll, XtNjumpProc, (XtPointer)&top);
    }

    scroll = ((ViewportWidget)pageview)->viewport.horiz_bar;
    if (scroll != (Widget)NULL) {
	XtSetArg(args[0], XtNshown, &shown);
	XtGetValues(scroll, args, ONE); INFFMESSAGE(horizontal,shown)
	top = (1.0-shown)/2.0; INFFMESSAGE(horizontal before ,top)
	XtCallCallbacks(scroll, XtNjumpProc, (XtPointer)&top);
    }
    ENDMESSAGE(cb_centerPage)
}

/*##################################################################*/
/* cb_setPageMark */
/* Set/unset the 'page marked' property */
/*##################################################################*/

void
cb_setPageMark(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    XawTextPosition begin,end;
    int r=(int)client_data;
    int i;

    BEGINMESSAGE(cb_setPageMark)
    if (!toc_text) {INFMESSAGE(no toc) ENDMESSAGE(cb_setPageMark) return; }
    if (r & SPM_SELECTION) {       INFMESSAGE(operation affects selection)
       XawTextGetSelectionPos(toc, &begin, &end);
       end--;
    } else if (r & SPM_CURRENT) {  INFMESSAGE(opration affects current)
       begin=(XawTextPosition) current_page*toc_entry_length; end=begin+(XawTextPosition)(toc_entry_length-1);
    } else {                       INFMESSAGE(operation affects all)
       begin=(XawTextPosition)0; end=(XawTextPosition)strlen(toc_text);
    }
    if (begin==end) { ENDMESSAGE(cb_setPageMarkSelection) return; }
    for (i = begin/toc_entry_length; i <= end/toc_entry_length; i++) {
        if (!(((r&SPM_EVEN) && i/2==(i+1)/2)  || ((r&SPM_ODD) && i/2!=(i+1)/2))) {
           if      (r & SPM_MARK)   toc_text[i*toc_entry_length] = '*';
           else if (r & SPM_UNMARK) toc_text[i*toc_entry_length] = ' ';
           else if (r & SPM_TOGGLE) toc_text[i*toc_entry_length] = toc_text[i*toc_entry_length]==' ' ? '*' :' ';
	   XawTextInvalidate(toc, i*toc_entry_length, i*toc_entry_length+1);
        }
    }
    ENDMESSAGE(cb_setPageMark)
}

/*##################################################################*/
/* cb_autoResize */
/*##################################################################*/

void
cb_autoResize(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    Arg args[1];
    Pixel top,bottom;

    BEGINMESSAGE(cb_autoResize)
    app_res.auto_resize = !(app_res.auto_resize);
    show_page(REQUEST_TOGGLE_RESIZE);
    ENDMESSAGE(cb_autoResize)
}

/*##################################################################*/
/* cb_setMagstep */
/* Set new magstep.  Reshow the current page if magstep changed. */
/*##################################################################*/

void
cb_setMagstep(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    int i=(int)client_data;
    BEGINMESSAGE(cb_setMagstep)
    i = (i <= app_res.maximum_magstep ? i : app_res.maximum_magstep);
    i = (i >= app_res.minimum_magstep ? i : app_res.minimum_magstep);
    app_res.magstep = i;
    show_page(REQUEST_NEW_MAGSTEP);
    ENDMESSAGE(cb_setMagstep)
}

/*##################################################################*/
/* cb_setOrientation */
/* Set new orientation.  Reshow the current page if orientation changed. */
/*##################################################################*/

void
cb_setOrientation(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    BEGINMESSAGE(cb_setOrientation)
    switch (gv_force) {
       case 2: if (app_res.force_orientation)
                    gv_forced_orientation = (XtPageOrientation) client_data;
               else gv_forced_orientation = XtPageOrientationUnspecified;
               break;
       case 1:      gv_forced_orientation = (XtPageOrientation) client_data;
               break;
       case 0:      gv_forced_orientation = XtPageOrientationUnspecified;
               break;
    }
    gv_orientation = (XtPageOrientation) client_data;
    gv_force = 0;
    show_page(REQUEST_NEW_ORIENTATION);
    ENDMESSAGE(cb_setOrientation)
}

#ifdef USE_SWAP_LANDSCAPE

/*##################################################################*/
/* cb_swapLandscape */
/* Swap the landscape labels and change the flag. */
/*##################################################################*/

void
cb_swapLandscape(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    Arg args[1];
    String s1, s2;

    BEGINMESSAGE(cb_swapLandscape)
    app_res.swap_landscape = !app_res.swap_landscape;

    XtSetArg(args[0], XtNlabel, &s1);
    XtGetValues(landscapebutton, args, ONE);
    s1 = XtNewString(s1);
    XtSetArg(args[0], XtNlabel, &s2);
    XtGetValues(seascapebutton, args, ONE);
    s2 = XtNewString(s2);
    XtSetArg(args[0], XtNlabel, s2);
    XtSetValues(landscapebutton, args, ONE);
    XtSetArg(args[0], XtNlabel, s1);
    XtSetValues(seascapebutton, args, ONE);
    XtFree(s1);
    XtFree(s2);

    show_page(REQUEST_NEW_ORIENTATION);

    ENDMESSAGE(cb_swapLandscape)
}

#endif /*USE_SWAP_LANDSCAPE*/

/*##################################################################*/
/* cb_setPagemedia */
/* Set new page media.  If new page media is different, update app_resources */
/* and redisplay page. */
/*##################################################################*/

void
cb_setPagemedia(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    BEGINMESSAGE(cb_setPagemedia)
    switch (gv_force) {
       case 2: if (app_res.force_pagemedia) gv_forced_pagemedia = (int) client_data;
               else                         gv_forced_pagemedia = -1;
               break;
       case 1:                              gv_forced_pagemedia = (int) client_data;
               break;
       case 0:                              gv_forced_pagemedia = -1;
               break;
    }
    gv_pagemedia = (int) client_data;
    gv_force = 0;
    show_page(REQUEST_NEW_PAGEMEDIA);
    ENDMESSAGE(cb_setPagemedia)
}

/*##################################################################*/
/* cb_trackAndZoom */
/* track mouse pointer and popup zoom window */
/*##################################################################*/

void
cb_trackAndZoom(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    Arg args[20];
    Cardinal num_args;
    Dimension width, height;
    Widget zoom;
    Widget zoomform;
    Widget zoompage;
    Widget zoomdismiss;
    FILE *zoomfile;
    struct stat sbuf;
    GhostviewReturnStruct *p = (GhostviewReturnStruct *)call_data;
    int llx;
    int lly;
    int urx;
    int ury;
    int bottom_margin;
    int left_margin;
    int right_margin;
    int top_margin;
    int i;

    BEGINMESSAGE1(cb_trackAndZoom)
    /* locator events have zero width and height */
    if ((p->width == 0)&&(p->height == 0)) {
        if (show_locator) {
           static char buf[MAX_LOCATOR_LENGTH];
           static int x,y;
           if ((x != p->psx) || (y != p->psy) || (buf[0]='\0')) {
    	      sprintf(buf, app_res.locator_format, p->psx, p->psy);
	      XtSetArg(args[0], XtNlabel, buf);
	      XtSetValues(locator, args, ONE);
           }
           x=p->psx; y=p->psy;
        }
        ENDMESSAGE1(cb_trackAndZoom)
	return;
    }

    if (!psfile) {INFMESSAGE1(no file)ENDMESSAGE1(cb_trackAndZoom)return;}

    /* If the file changed, cannot zoom */
    stat(filename, &sbuf);
    if (mtime != sbuf.st_mtime) {INFMESSAGE1(file has changed)ENDMESSAGE1(cb_trackAndZoom)return;}
    zoom = XtCreatePopupShell("zoom", topLevelShellWidgetClass,
			      toplevel, NULL, ZERO);

    zoomform = XtCreateManagedWidget("form", formWidgetClass,
				     zoom, NULL, ZERO);

    llx = p->psx - p->width/2;
    lly = p->psy - p->height/2;
    urx = p->psx + p->width/2;
    ury = p->psy + p->height/2;

    /* Make sure zoom window doesn't go off the edge of the page */
    if (llx < current_llx) {
	llx = current_llx;
	urx = llx + p->width;
    }
    if (lly < current_lly) {
	lly = current_lly;
	ury = lly + p->height;
    }
    if (urx > current_urx) {
	urx = current_urx;
	llx = urx - p->width;
    }
    if (ury > current_ury) {
	ury = current_ury;
	lly = ury - p->height;
    }
    if (llx < current_llx) {
	llx = current_llx;
    }
    if (lly < current_lly) {
	lly = current_lly;
    }
    bottom_margin = lly - current_lly;
    left_margin = llx - current_llx;
    right_margin = current_urx - urx;
    top_margin = current_ury - ury;

							num_args = 0;
    XtSetArg(args[num_args], XtNtop, XtChainTop);	num_args++;
    XtSetArg(args[num_args], XtNbottom, XtChainBottom);	num_args++;
    XtSetArg(args[num_args], XtNleft, XtChainLeft);	num_args++;
    XtSetArg(args[num_args], XtNright, XtChainRight);	num_args++;
    XtSetArg(args[num_args], XtNorientation, current_orientation);
							num_args++;
    XtSetArg(args[num_args], XtNllx, llx);      	num_args++;
    XtSetArg(args[num_args], XtNlly, lly);      	num_args++;
    XtSetArg(args[num_args], XtNurx, urx);      	num_args++;
    XtSetArg(args[num_args], XtNury, ury);      	num_args++;
    XtSetArg(args[num_args], XtNbottomMargin, bottom_margin);
							num_args++;
    XtSetArg(args[num_args], XtNleftMargin, left_margin);
							num_args++;
    XtSetArg(args[num_args], XtNrightMargin, right_margin);
							num_args++;
    XtSetArg(args[num_args], XtNtopMargin, top_margin); num_args++;
    XtSetArg(args[num_args], XtNbottomMargin, bottom_margin);
							num_args++;
    XtSetFloatArg(args[num_args], XtNxdpi, p->xdpi);	num_args++;
    XtSetFloatArg(args[num_args], XtNydpi, p->ydpi);	num_args++;
    if (!toc_text) {
        XtSetArg(args[num_args], XtNfilename, filename);	num_args++;
    }
    zoompage = XtCreateManagedWidget("page", ghostviewWidgetClass,
				     zoomform, args, num_args);
    num_ghosts++;
    XtAddCallback(zoompage, XtNcallback, cb_trackAndZoom, (XtPointer)0);
    XtAddCallback(zoompage, XtNmessageCallback, cb_message, (XtPointer)zoompage);
    XtAddCallback(zoompage, XtNdestroyCallback, cb_destroyGhost,
		  (XtPointer)zoompage);

							num_args = 0;
    XtSetArg(args[num_args], XtNfromVert, zoompage);	num_args++;
    XtSetArg(args[num_args], XtNtop, XtChainBottom);	num_args++;
    XtSetArg(args[num_args], XtNbottom, XtChainBottom);	num_args++;
    XtSetArg(args[num_args], XtNleft, XtChainLeft);	num_args++;
    XtSetArg(args[num_args], XtNright, XtChainRight);	num_args++;
    zoomdismiss = XtCreateManagedWidget("dismiss", commandWidgetClass,
				       zoomform, args, num_args);
    XtAddCallback(zoomdismiss, XtNcallback, cb_destroy, (XtPointer)zoom);

    XtSetArg(args[0], XtNwidth, &width);
    XtGetValues(zoompage, args, ONE);
    XtSetArg(args[0], XtNwidth, width);
    XtSetValues(zoomdismiss, args, ONE);

    XtRealizeWidget(zoom);
    positionPopup(zoom,1,NULL,50,50, 1,1);

							num_args = 0;
    XtSetArg(args[num_args], XtNwidth, &width);		num_args++;
    XtSetArg(args[num_args], XtNheight, &height);	num_args++;
    XtGetValues(zoom, args, num_args);

							      	num_args = 0;
    XtSetArg(args[num_args], XtNminWidth, width);		num_args++;
    XtSetArg(args[num_args], XtNminHeight, height);		num_args++;
    XtSetArg(args[num_args], XtNmaxWidth, width);		num_args++;
    XtSetArg(args[num_args], XtNmaxHeight, height);		num_args++;
    XtSetValues(zoom, args, num_args);
    XSetWMProtocols(XtDisplay(zoom), XtWindow(zoom), &wm_delete_window, 1);
    XtPopup(zoom, XtGrabNone);

    if (toc_text) {
	zoomfile = fopen(filename, "r");
	if (zoomfile == NULL) return;
	GhostviewSendPS(zoompage, zoomfile, doc->beginprolog,
			doc->lenprolog, False);
	GhostviewSendPS(zoompage, zoomfile, doc->beginsetup,
			doc->lensetup, False);
	if (doc->pageorder == DESCEND)
	    i = (doc->numpages - 1) - current_page;
	else
	    i = current_page;
	GhostviewSendPS(zoompage, zoomfile, doc->pages[i].begin,
			doc->pages[i].len, True);
    }
   ENDMESSAGE1(cb_trackAndZoom)
}

/*##################################################################*/
/* cb_message */
/* Process messages from ghostscript */
/* Refresh occurs when window was resized unexpectedly */
/*##################################################################*/

void
cb_message(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    int i;
    char *error;

    BEGINMESSAGE(cb_message)
    if (!strcmp((char *) call_data, "Failed")) {
        INFMESSAGE(Failed)
	if ((Widget)client_data == page) {
            error = "\nError: PostScript interpreter failed in main window.\n\n";
	} else {
            error = "\nError: PostScript interpreter failed in zoom window.\n\n";
	}
	cb_appendInfoPopup((Widget)NULL,(XtPointer)NULL,(XtPointer)error);
    } else if (!strcmp((char *) call_data, "BadAlloc")) {
        INFMESSAGE(BadAlloc)
	if ((Widget)client_data == page) {
	    error = "\nWarning: Could not allocate backing pixmap in main window.\n\n";
	} else {
	    error = "\nWarning: Could not allocate backing pixmap in zoom window.\n\n";
	}
	cb_appendInfoPopup((Widget)NULL,(XtPointer)NULL,(XtPointer)error);
    } else if (!strcmp((char *) call_data, "Refresh")) {
        INFMESSAGE(Refresh)
	if (toc_text) {
	    GhostviewSendPS(w, psfile, doc->beginprolog,
			    doc->lenprolog, False);
	    GhostviewSendPS(w, psfile, doc->beginsetup,
			    doc->lensetup, False);
	    if (doc->pageorder == DESCEND)
		i = (doc->numpages - 1) - current_page;
	    else
		i = current_page;
	    GhostviewSendPS(w, psfile, doc->pages[i].begin,
			    doc->pages[i].len, False);
	}
    } else if (!strcmp((char *) call_data, "Page")) {
        INFMESSAGE(completed page)
	if ((gv_pending_page_request!=NO_CURRENT_PAGE) && (toc_text) && ((Widget)client_data == page)) {
           INFIMESSAGE(pending request for, gv_pending_page_request)
           show_page(gv_pending_page_request);
	}
    }
    ENDMESSAGE(cb_message)
}

/*##################################################################*/
/* cb_destroy */
/* Destroy popup window */
/*##################################################################*/

void
cb_destroy(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    BEGINMESSAGE(cb_destroy)
    XtDestroyWidget((Widget)client_data);
    ENDMESSAGE(cb_destroy)
}

/*##################################################################*/
/* cb_destroyGhost */
/* destroy callback for Ghostview widgets. */
/* The disable interpreter call ensures that ghostscript is killed. */
/* One the count goes to 0, we are sure that all forked processes have */
/* been killed and that we can safely exit. */
/*##################################################################*/

void
cb_destroyGhost(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
    BEGINMESSAGE(cb_destroyGhost)
    GhostviewDisableInterpreter((Widget) client_data);
    num_ghosts--;
    if (num_ghosts) {ENDMESSAGE(cb_destroyGhost) return; }
    if (dying) old_Xerror(XtDisplay(w), &bomb);
    XtDestroyApplicationContext(app_con);
    ENDMESSAGE(cb_destroyGhost)
    ENDMESSAGE(exiting Ghostview)
    exit(0);
}

/*##################################################################*/
/* cb_quitGhostview */
/* Start application folding up by Destroying the top level widget. */
/* The application exits when the last interpreter is killed during */
/* a destroy callback from ghostview widgets. */
/*##################################################################*/
void
cb_quitGhostview(w, client_data, call_data)
    Widget w;
    XtPointer client_data, call_data;
{
   BEGINMESSAGE(cb_quitGhostview)
   XtUnmapWidget(toplevel);   
   XtDestroyWidget(toplevel);
   ENDMESSAGE(cb_quitGhostview)
}

