/*
 * tabstops.c
 * Copyright (C) 1999 A.J. van Os
 *
 * Description:
 * Read the tab stop information from a MS Word file
 */

#include <stdio.h>
#include "antiword.h"

static int	iDefaultTabWidth = 36000;	/* In millipoints */


/*
 * vSet6DefaultTabWidth -
 */
static void
vSet6DefaultTabWidth(FILE *pFile, int iStartBlock,
	const int *aiBBD, int iBBDLen,
	const unsigned char *aucHeader)
{
	unsigned char	*aucBuffer;
	int		iBeginDocpInfo, iDocpInfoLen;

	iBeginDocpInfo = (int)ulGetLong(0x150, aucHeader);
	DBG_HEX(iBeginDocpInfo);
	iDocpInfoLen = (int)ulGetLong(0x154, aucHeader);
	DBG_DEC(iDocpInfoLen);

	aucBuffer = xmalloc(iDocpInfoLen);
	if (!bReadBuffer(pFile, iStartBlock,
			aiBBD, iBBDLen, BIG_BLOCK_SIZE,
			aucBuffer, iBeginDocpInfo, iDocpInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	iDefaultTabWidth = iTwips2MilliPoints(usGetWord(0x0a, aucBuffer));
	DBG_DEC(iDefaultTabWidth);
	aucBuffer = xfree(aucBuffer);
} /* end of vSet6DefaultTabWidth */

/*
 * vSet8DefaultTabWidth -
 */
static void
vSet8DefaultTabWidth(FILE *pFile, const pps_info_type *pPPS,
	const int *aiBBD, int iBBDLen, const int *aiSBD, int iSBDLen,
	const unsigned char *aucHeader)
{
        const int	*aiBlockDepot;
	unsigned char	*aucBuffer;
	int	iTableStartBlock, iTableSize, iBlockDepotLen, iBlockSize;
	int	iBeginDocpInfo, iDocpInfoLen;
	unsigned short	usDocStatus;

	iBeginDocpInfo = (int)ulGetLong(0x192, aucHeader);
	DBG_HEX(iBeginDocpInfo);
	iDocpInfoLen = (int)ulGetLong(0x196, aucHeader);
	DBG_DEC(iDocpInfoLen);

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		iTableStartBlock = pPPS->t1Table.iSb;
		iTableSize = pPPS->t1Table.iSize;
	} else {
		iTableStartBlock = pPPS->t0Table.iSb;
		iTableSize = pPPS->t0Table.iSize;
	}
	DBG_DEC(iTableStartBlock);
	if (iTableStartBlock < 0) {
		DBG_DEC(iTableStartBlock);
		DBG_MSG("No TAB information");
		return;
	}
	DBG_HEX(iTableSize);
	if (iTableSize < MIN_SIZE_FOR_BBD_USE) {
		/* Use the Small Block Depot */
		aiBlockDepot = aiSBD;
		iBlockDepotLen = iSBDLen;
		iBlockSize = SMALL_BLOCK_SIZE;
	} else {
		/* Use the Big Block Depot */
		aiBlockDepot = aiBBD;
		iBlockDepotLen = iBBDLen;
		iBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(iDocpInfoLen);
	if (!bReadBuffer(pFile, iTableStartBlock,
			aiBlockDepot, iBlockDepotLen, iBlockSize,
			aucBuffer, iBeginDocpInfo, iDocpInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	iDefaultTabWidth = iTwips2MilliPoints(usGetWord(0x0a, aucBuffer));
	DBG_DEC(iDefaultTabWidth);
	aucBuffer = xfree(aucBuffer);
} /* end of vSet8DefaultTabWidth */

/*
 * vSetDefaultTabWidth -
 */
void
vSetDefaultTabWidth(FILE *pFile, const pps_info_type *pPPS,
		const int *aiBBD, int iBBDLen, const int *aiSBD, int iSBDLen,
		const unsigned char *aucHeader, int iWordVersion)
{
	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(iWordVersion < 6 || iWordVersion > 8);
	fail(aiBBD == NULL || iBBDLen < 0);
	fail(aiSBD == NULL || iSBDLen < 0); 

	switch (iWordVersion) {
	case 6:
	case 7:  
		vSet6DefaultTabWidth(pFile, pPPS->tWordDocument.iSb,
				aiBBD, iBBDLen, aucHeader);
		break;
	case 8:
		vSet8DefaultTabWidth(pFile, pPPS,
				aiBBD, iBBDLen, aiSBD, iSBDLen, aucHeader);
		break;
	default:
		werr(0, "Sorry, no TAB information");
		break;
	}
} /* end of vSetDefaultTabWidth */

/*
 * iGetDefaultTabWidth - Get the default tabwidth in millipoints
 */
int
iGetDefaultTabWidth(void)
{
	if (iDefaultTabWidth <= 0) {
		DBG_DEC(iDefaultTabWidth);
		return iTwips2MilliPoints(1);
	}
	return iDefaultTabWidth;
} /* end of iGetDefaultTabWidth */
