# Averist
# Copyright (C) 1999-2000 Henrik Edlund <henrik@edlund.org>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA.
#
# If you like Averist, please express your satisfaction with a donation: send
# me what you feel Averist has been worth to you. If you are glad that I
# developed Averist and distribute it as free software, rather than following
# the obstructive and antisocial practices typical of software developers,
# reward me. If you would like me to develop more free software, contribute.
#
# Documentation can be found at http://www.edlund.org/hacks/averist/index.html.



# login() -> (username, session, time)
# login(query) -> (username, session, time)
# logout() -> ()



package Averist;

use strict;
no strict 'refs';

use vars qw($product_name $product_version);
$product_name = 'Averist';
$product_version = '0.2.9';



#################  Do NOT edit anything outside the Configuration if you don't
# Configuration #  know what you are doing. If you are planning to tweak then
#################  please read the License first.

use vars qw(%config);
%config = (
	   'events' => {
	       'authentication' => 'cgi',  # cgi
	       'reauthentication' => {
		   'remote' => 'cookie',  # cgi | cookie
		   'local'  => 'none',  # sql | none
	       },
	       'verification' => 'file',  # file | sql
	   },
	   'limits' => {
	       'time' => 15 * 60,  # 15 minutes
	   },
	   'methods' => {
	       'cgi' => {
		   'username' => 'username',
		   'password' => 'password',
		   'session'  => 'session',
		   'time'     => 'time',
	       },
	       'file' => {
		   'filename' => '/etc/httpd/conf/auth.conf',
		   'storage'  => 'crypt',  # crypt | clear
	       },
	       'sql' => {
		   'verification' => {
		       'database' => {
			   'host'     => '',
			   'port'     => '',
			   'name'     => 'averist',
			   'user'     => 'henrik',
			   'password' => '',
		       },
		       'tables' => {
			   'users' => 'users',
		       },
		       'fields' => {
			   'users' => {
			       'username' => 'username',
			       'password' => 'password',
			   },
		       },
		   },
		   'reauthentication' => {
		       'database' => {
			   'host'     => '',
			   'port'     => '',
			   'name'     => 'averist',
			   'user'     => 'henrik',
			   'password' => '',
		       },
		       'tables' => {
			   'sessions' => 'sessions',
		       },
		       'fields' => {
			   'sessions' => {
			       'username' => 'username',
			       'session'  => 'session',
			       'time'     => 'time',
			   },
		       },
		   },
		   'storage' => 'crypt',  # crypt | clear
	       },
	       'cookie' => {
		   'username' => 'username',
		   'session'  => 'session',
		   'time'     => 'time',
	       },
	   },
);

######################
# Configuration Ends #
######################



use vars qw($SQL);
$SQL = "PostgreSQL";
if (($config{'events'}{'verification'} eq 'sql') ||
    ($config{'events'}{'reauthentication'}{'local'} eq 'sql')) {
    require PostgreSQL;
}

use vars qw(%cgi_query);

sub login (;$) {
    my $remote_username;
    my $remote_password;
    my $remote_session;
    my $remote_time;
    my $local_password;
    my $local_session;
    my $local_time;
    my $session;
    my $time;
    if (defined(%{$_[0]})) {
	%cgi_query = %{$_[0]};
    }
    else {
	%cgi_query = _parse_cgi_query();
    }
    if ($remote_username = _remote_username()) {
	if ($local_password = _local_password($remote_username)) {
	    ($remote_session, $remote_time) = _remote_session();
	    $session = _session($local_password);
	    $time = time();
	    if (defined($remote_session)) {
		if ($config{'events'}{'reauthentication'}{'local'} eq 'none') {
		    if ((defined($remote_time)) &&
			($remote_session eq
			 crypt($local_password, $remote_session)) &&
			($time <=
			 ($remote_time + $config{'limits'}{'time'}))) {
			_set_session($remote_username, $session, $time);
			return($remote_username, $session, $time);
		    }
		}
		else {
		    ($local_session, $local_time) =
			_local_session($remote_username);
		    if ((defined($local_session)) &&
			(defined($local_time)) &&
			($remote_session eq $local_session) &&
			($time <= ($local_time + $config{'limits'}{'time'}))) {
			_set_session($remote_username, $session, $time);
			return($remote_username, $session, $time);
		    }
		}
	    }
	    else {
		$remote_password = _remote_password();
		if (_verify_password($remote_password, $local_password)) {
		    _set_session($remote_username, $session, $time);
		    return($remote_username, $session, $time);
		}
	    }
	    _set_session($remote_username, '', 0);
	    return($remote_username, undef, undef);
	}
    }
    return(undef, undef, undef);
}

sub logout () {
    my $remote_username;
    if ($remote_username = _remote_username()) {
	_set_session($remote_username, '', 0);
    }
}

sub _remote_username () {
    my %parsed;
    if ($config{'events'}{'reauthentication'}{'remote'} eq 'cgi') {
	if (defined($cgi_query{$config{'methods'}{'cgi'}{'username'}})) {
	    return($cgi_query{$config{'methods'}{'cgi'}{'username'}});
	}
    }
    elsif ($config{'events'}{'reauthentication'}{'remote'} eq 'cookie') {
	%parsed = _parse_cookies();
	if (defined($parsed{$config{'methods'}{'cookie'}{'username'}})) {
	    return($parsed{$config{'methods'}{'cookie'}{'username'}});
	}
    }
    else {
	die "Invalid value in events->reauthentication->remote, stopped";
    }
    if ($config{'events'}{'authentication'} eq 'cgi') {
	if (defined($cgi_query{$config{'methods'}{'cgi'}{'username'}})) {
	    return($cgi_query{$config{'methods'}{'cgi'}{'username'}});
	}
    }
    else {
	die "Invalid value in events->authentication, stopped";
    }
    return(undef);
}

sub _remote_session () {
    my %parsed;
    if ($config{'events'}{'reauthentication'}{'remote'} eq 'cgi') {
	if (defined($cgi_query{$config{'methods'}{'cgi'}{'session'}})) {
	    if (defined($cgi_query{$config{'methods'}{'cgi'}{'time'}})) {
		return($cgi_query{$config{'methods'}{'cgi'}{'session'}},
		       $cgi_query{$config{'methods'}{'cgi'}{'time'}});
	    }
	    else {
		return($cgi_query{$config{'methods'}{'cgi'}{'session'}},
		       undef);
	    }
	}
    }
    elsif ($config{'events'}{'reauthentication'}{'remote'} eq 'cookie') {
	%parsed = _parse_cookies();
	if (defined($parsed{$config{'methods'}{'cookie'}{'session'}})) {
	    if (defined($parsed{$config{'methods'}{'cookie'}{'time'}})) {
		return($parsed{$config{'methods'}{'cookie'}{'session'}},
		       $parsed{$config{'methods'}{'cookie'}{'time'}});
	    }
	    else {
		return($parsed{$config{'methods'}{'cookie'}{'session'}},
		       undef);
	    }
	}
    }
    else {
	die "Invalid value in events->reauthentication->remote, stopped";
    }
    return(undef, undef);
}

sub _remote_password () {
    if ($config{'events'}{'authentication'} eq 'cgi') {
	if (defined($cgi_query{$config{'methods'}{'cgi'}{'password'}})) {
	    return($cgi_query{$config{'methods'}{'cgi'}{'password'}});
	}
    }
    else {
	die "Invalid value in events->authentication, stopped";
    }
    return(undef);
}

sub _local_session ($) {
    (my $username) = @_;
    my $sql;
    my $conn;
    my $q1;
    my @row;
    if ($config{'events'}{'reauthentication'}{'local'} eq 'sql') {
	$sql = $config{'methods'}{'sql'}{'reauthentication'};
	$conn = &{"${SQL}::connect"}($$sql{'database'}{'host'},
				     $$sql{'database'}{'port'},
				     $$sql{'database'}{'name'},
				     $$sql{'database'}{'user'},
				     $$sql{'database'}{'password'});
	$q1 = "select $$sql{'fields'}{'sessions'}{'session'}, ";
	$q1 .= "$$sql{'fields'}{'sessions'}{'time'} from ";
	$q1 .= "$$sql{'tables'}{'sessions'} where ";
	$q1 .= "$$sql{'fields'}{'sessions'}{'username'}=";
	$q1 .= _quote($username);
	if (&{"${SQL}::ntuples"}(&{"${SQL}::query"}($conn, $q1)) == 0) {
	    return(undef, undef);
	}
	else {
	    return(&{"${SQL}::fetchrow"}(&{"${SQL}::query"}($conn, $q1)));
	}
    }
    else {
	die "Invalid value in events->reauthentication->local, stopped";
    }
}

sub _local_password ($) {
    (my $username) = @_;
    my $filename;
    my $read_username;
    my $read_password;
    my $sql;
    my $conn;
    my $q1;
    if ($config{'events'}{'verification'} eq 'file') {
	if (!defined($config{'methods'}{'file'}{'filename'})) {
	    die "Invalid value in methods->file->filename, stopped";
	}
	$filename = $config{'methods'}{'file'}{'filename'};
	die "Can't open $filename for reading, stopped"
	    unless (open(FILE,"<$filename"));
	foreach (<FILE>) {
	    ($read_username, $read_password, undef) = split(/:/, $_, 3);
	    if ($username eq $read_username) {
		$read_password =~ s/\n//;
		return($read_password);
	    }
	}
	return(undef);
    }
    elsif ($config{'events'}{'verification'} eq 'sql') {
	$sql = $config{'methods'}{'sql'}{'verification'};
	$conn = &{"${SQL}::connect"}($$sql{'database'}{'host'},
				     $$sql{'database'}{'port'},
				     $$sql{'database'}{'name'},
				     $$sql{'database'}{'user'},
				     $$sql{'database'}{'password'});
	$q1 = "select $$sql{'fields'}{'users'}{'password'} from ";
	$q1 .= "$$sql{'tables'}{'users'} where ";
	$q1 .= "$$sql{'fields'}{'users'}{'username'}=";
	$q1 .= _quote($username);
	if (&{"${SQL}::ntuples"}(&{"${SQL}::query"}($conn, $q1)) == 0) {
	    return(undef);
	}
	else {
	    return((&{"${SQL}::fetchrow"}(&{"${SQL}::query"}($conn, $q1)))[0]);
	}
    }
    else {
	die "Invalid value in events->verification, stopped";
    }
}    

sub _verify_password ($$) {
    (my $remote_password, my $local_password) = @_;
    my $crypt;
    if ($config{'events'}{'verification'} eq 'file') {
	if ($config{'methods'}{'file'}{'storage'} eq 'crypt') {
	    $crypt = 1;
	}
	elsif ($config{'methods'}{'file'}{'storage'} ne 'clear') {
	    die "Invalid value in methods->file->storage, stopped";
	}
    }
    elsif ($config{'events'}{'verification'} eq 'sql') {
	if ($config{'methods'}{'sql'}{'storage'} eq 'crypt') {
	    $crypt = 1;
	}
	elsif ($config{'methods'}{'sql'}{'storage'} ne 'clear') {
	    die "Invalid value in methods->sql->storage, stopped";
	}
    }
    else {
	die "Invalid value in events->verification, stopped";
    }
    if (defined($crypt)) {
	$remote_password = crypt($remote_password, $local_password);
    }
    if ($remote_password eq $local_password) {
	return(1);
    }
    else {
	return(0);
    }
}

sub _session ($) {
    (my $password) = @_;
    my $salt;
    my $time;
    my $session;
    $salt = join('', ('.', '/', 0..9, 'A'..'Z', 'a'..'z')[rand 64, rand 64]);
    $time = gmtime();
    if ($config{'events'}{'reauthentication'}{'local'} eq 'none') {
	$session = crypt($password, $salt);
    }
    elsif ($config{'events'}{'reauthentication'}{'local'} eq 'sql') {
	$session = crypt($time, $salt);
    }
    else {
	die "Invalid value in events->reauthentication->local, stopped";
    }
    return($session);
}

sub _set_session ($$$) {
    (my $username, my $session, my $time) = @_;
    if ($config{'events'}{'reauthentication'}{'local'} eq 'sql') {
	_set_sql($username, $session, $time);
    }
    if ($config{'events'}{'reauthentication'}{'remote'} eq 'cookie') {
	_set_cookie($username, $session, $time);
    }
}

sub _set_cookie ($$$) {
    (my $username, my $session, my $time) = @_;
    my @weekdays;
    my @months;
    my $second;
    my $minute;
    my $hour;
    my $day;
    my $month;
    my $year;
    my $weekday;
    my $expire;
    @weekdays = ('Sunday',
		 'Monday',
		 'Tuesday',
		 'Wednesday',
		 'Thursday',
		 'Friday',
		 'Saturday');
    @months = ('Jan',
	       'Feb',
	       'Mar',
	       'Apr',
	       'May',
	       'Jun',
	       'Jul',
	       'Aug',
	       'Sep',
	       'Oct',
	       'Nov',
	       'Dec');
    ($second, $minute, $hour, $day, $month, $year, $weekday) =
	gmtime(($time + $config{'limits'}{'time'}));
    $expire = sprintf("%s, %.2u-%s-%u %.2u:%.2u:%.2u GMT",
		      $weekdays[$weekday],
		      $day,
		      $months[$month],
		      $year+1900,
		      $hour,
		      $minute,
		      $second);
    print "Set-Cookie: $config{'methods'}{'cookie'}{'username'}";
    print "=$username; expires=$expire\n";
    print "Set-Cookie: $config{'methods'}{'cookie'}{'session'}";
    print "=$session; expires=$expire\n";
    print "Set-Cookie: $config{'methods'}{'cookie'}{'time'}";
    print "=$time; expires=$expire\n";
}

sub _set_sql ($$$) {
    (my $username, my $session, my $time) = @_;
    my $sql;
    my $conn;
    my $q1;
    my @row;
    $sql = $config{'methods'}{'sql'}{'reauthentication'};
    $conn = &{"${SQL}::connect"}($$sql{'database'}{'host'},
				 $$sql{'database'}{'port'},
				 $$sql{'database'}{'name'},
				 $$sql{'database'}{'user'},
				 $$sql{'database'}{'password'});
    $q1 = "select * from $$sql{'tables'}{'sessions'} where ";
    $q1 .= "$$sql{'fields'}{'sessions'}{'username'}=";
    $q1 .= _quote($username);
    if (&{"${SQL}::ntuples"}(&{"${SQL}::query"}($conn, $q1)) == 0) {
	$q1 = "insert into $$sql{'tables'}{'sessions'} ";
	$q1 .= "($$sql{'fields'}{'sessions'}{'username'}, ";
	$q1 .= "$$sql{'fields'}{'sessions'}{'session'}, ";
	$q1 .= "$$sql{'fields'}{'sessions'}{'time'}) ";
	$q1 .= "values (" . _quote($username) . ", " . _quote($session) .
	    ", $time)";
    }
    else {
	$q1 = "update $$sql{'tables'}{'sessions'} set ";
	$q1 .= "$$sql{'fields'}{'sessions'}{'session'}='$session', ";
	$q1 .= "$$sql{'fields'}{'sessions'}{'time'}=$time ";
	$q1 .= "where $$sql{'fields'}{'sessions'}{'username'}=";
	$q1 .= _quote($username);
    }
    &{"${SQL}::query"}($conn, $q1);
}

sub _quote ($) {
    (my $string) = @_;
    if (!defined($string)) {
	return "NULL";
    }
    else {
	$string =~ s/\'/\'\'/g;
	return "'$string'";
    }
}

sub _parse_cgi_query () {
    my $in;
    my @in;
    my $key;
    my $val;
    my %in;
    if ($ENV{'REQUEST_METHOD'} eq 'GET') {
	$in = $ENV{'QUERY_STRING'};
    }
    elsif ($ENV{'REQUEST_METHOD'} eq 'POST') {
	read(STDIN, $in, $ENV{'CONTENT_LENGTH'});
    }
    @in = split(/[&;]+/, $in);
    foreach (@in) {
	s/\+/ /g;
	($key, $val) = split(/=/);
	$key =~ s/%(..)/pack("c", hex($1))/ge;
	$val =~ s/%(..)/pack("c", hex($1))/ge;
	$key =~ s/\0//g;
	$val =~ s/\0//g;
	$in{$key} = $val;
    }
    return(%in);
}

sub _parse_cookies () {
    my $in;
    my @in;
    my $key;
    my $val;
    my %in;
    $in = $ENV{'HTTP_COOKIE'} || $ENV{'COOKIE'};
    @in = split(/; /, $in);
    foreach (@in) {
	($key, $val) = split(/=/);
	$key =~ s/%(..)/pack("c", hex($1))/ge;
	$val =~ s/%(..)/pack("c", hex($1))/ge;
	$key =~ s/\0//g;
	$val =~ s/\0//g;
	$in{$key} = $val;
    }
    return(%in);
}

1;
