/*
 *    Calc: A Programmable Calculator for GTK+
 *    Copyright (C) 2000  David Hanak (dhanak@inf.bme.hu)
 *
 *    This file can be distributed under GNU GPL.  See file COPYING.
 *
 *    $Id: calc.c,v 2.2 2000/02/21 22:05:07 david Exp $
 */

#include <config.h>

#include <stdio.h>
#ifdef STDC_HEADERS
#  include <stdlib.h>
#  include <string.h>
#endif
#include <ctype.h>
#include <gtk/gtk.h>

#include "memgrp.h"
#include "calc.h"

#define CALC_MAIN_WIDTH  600
#define CALC_HEIGHT      250
#define EXP_WIDTH        320
#define VARNAME_WIDTH    110
#define FUNNAME_WIDTH    140

#define RETURN_ERROR_VALUE(mesg, pos) G_STMT_START {   		        \
  value_t v;                                                            \
  set_err(mesg, pos);                                  		        \
  v.type = V_INVALID;                               		        \
  return v;                                         		        \
} G_STMT_END

#define WARNING_RUNNING_EVAL                                            \
    message("Warning", "Can't start new evaluation while another "      \
	               "one is running!")


/*************************************************************************/
/************ Global widget registration types and variables *************/
/*************************************************************************/

enum WIDGETS {
  W_HISTORY,
  W_MAINENTRY,
  W_ABORT,
  W_VARLIST_WINDOW,
  W_VARLIST,
  W_VARLIST_BUTTONS,
  W_FUNLIST_WINDOW,
  W_FUNLIST,
  W_FUNLIST_T_BUILTIN,
  W_FUNLIST_B_INSERT,
  W_FUNLIST_B_DELETE,
  WIDGET_QTY
};

GtkWidget *widgets[WIDGET_QTY];

/*************************************************************************/
/**** Global types variables for building and evaluating expressions *****/
/*************************************************************************/

/************************* Evaluation variables **************************/
cparam_t cparams;
gint eval_depth, call_depth;
gboolean user_break;
value_t return_value;

void yy_scan_from_file(FILE *);
void yy_scan_from_string(gchar *);
void yy_delete_default_buffer();

/**************************** Variable stack *****************************/
struct stack_item {
  gchar  *name;
  gint    depth;
  value_t value;
  struct stack_item *next;
} *stack;

/**************************** Function stack *****************************/
/* Warning!  The list of args in this case is NULL terminated to allow easy
 * concatenation for printing.
 */
struct user_def {
  gchar  *name;
  gint    argc;
  gchar  *file;
  gchar **args;
  exp_t  *body;
  struct user_def *next;
} *functions;

/****************************** File stack *******************************/
struct file_stack visited_files;

/*************************************************************************/
/************************** Main menu structure **************************/
/*************************************************************************/

/************************** Callback functions ***************************/
void destroy();
void save_history();
void clear_history();
void open_file();
void show_variable_list();
void show_function_list();
void show_about();

/**************************** The menu itself ****************************/
static GtkItemFactoryEntry menu_items[] = {
  { "/_Calc",                 NULL,         NULL, 0, "<Branch>" },
  { "/Calc/Open...",          "<control>O", open_file, 0, NULL  },
  { "/Calc/Save history...",  "<control>S", save_history, 0, NULL },
  { "/Calc/Clear history",    "<control>C", clear_history, 0, NULL },
  { "/Calc/Preferences...",   "<control>P", NULL, 0, NULL },
  { "/Calc/sep1",             NULL,         NULL, 0, "<Separator>" },
  { "/Calc/Exit",             "<control>Q", destroy, 0, NULL },
  { "/_Windows",              NULL,         NULL, 0, "<Branch>" },
  { "/Windows/Variables...",  "<control>V", show_variable_list, 0, NULL },
  { "/Windows/Functions...",  "<control>F", show_function_list, 0, NULL },
  { "/_Help",                 NULL,         NULL, 0, "<LastBranch>" },
  { "/Help/About...",         NULL,         show_about, 0, NULL }
};

/*************************************************************************/
/************ Functions and structures generally used by GUI *************/
/*************************************************************************/

/***************************** Variable List *****************************/
struct stack_item *varlist_selection;

struct varmod {
  struct stack_item *var;
  GtkWidget *set_dialog;
  GtkWidget *set_entry;
};

gchar *varlist_titles[] = { "Name", "Value" };
gint sorted_column, sort_order;

void update_varlist();

/***************************** Function List *****************************/
gint funlist_selection;

void update_funlist();

/***************************** Miscellanious *****************************/
GtkWidget *arrow_button_with_label(gchar *text, GtkJustification just,
				   GtkArrowType arrow_type);

/* Must be static - conflicts with GTK!!! */
static gchar *opendir = NULL;

/*************************************************************************/
/****** Part I: Expression building functions called by the parser *******/
/*************************************************************************/

oper_ptr builtin_lookup(gchar *builtin)
{
  gint i;

  for (i = 0; builtins[i].name != NULL; i++)
    if (strcmp(builtin, builtins[i].name) == 0)
      return builtins[i].func;

  return NULL;
}

arg_t *build_arglist(gint argc, ...)
{
  va_list va;
  arg_t *al;
  gint i;

  g_return_val_if_fail(argc > 0, NULL);

  va_start(va, argc);

  al = expr_alloc(arg_t, sizeof(arg_t) + argc * sizeof(exp_t *));
  al->argc = argc;
  for (i = 0; i < argc; i++)
    al->args[i] = va_arg(va, exp_t *);

  va_end(va);
  return al;
}

arg_t *add_arg(arg_t *al, exp_t *nextarg)
{
  gint newargc;

  if (al) newargc = al->argc + 1;
  else    newargc = 1;

  al = (arg_t *)mg_realloc(al, sizeof(arg_t) + newargc * sizeof(exp_t *));
  al->argc = newargc;
  al->args[al->argc-1] = nextarg;

  return al;
}

exp_t *build_exp(enum EXP_TYPE type, ...)
{
  va_list va;
  exp_t *exp;
  gchar *name;

  va_start(va, type);

  exp = expr_alloc(exp_t, sizeof(exp_t));
  exp->type = type;

  switch (type) {

  case E_LITERAL:
    exp->data.val = va_arg(va, value_t);
    exp->args = NULL;
    break;

  case E_OPER:
    exp->data.ptr = va_arg(va, gpointer);
    exp->args = va_arg(va, arg_t *);
    break;

  case E_VARIABLE:
  case E_USERDEF:
  case E_USERCALL:
    name = va_arg(va, gchar *);
    exp->data.name = expr_alloc(gchar, strlen(name) + 1);
    strcpy(exp->data.name, name);
    switch (type) {
    case E_VARIABLE:
      exp->args = NULL;
      break;
    case E_USERCALL:
      exp->args = va_arg(va, arg_t *);
      break;
    default:
      exp->args = va_arg(va, arg_t *);
      exp->args = add_arg(exp->args, va_arg(va, exp_t *));
    }
    break;
  }

  va_end(va);

  return exp;
}

void yyerror(gchar *error)
{
  if (cparams.error == NULL)
    cparams.error = g_strdup(error);
}

/*************************************************************************/
/******************** Part II: Expression evaluation *********************/
/*************************************************************************/

struct stack_item *lookup_var(gchar *name)
{
  struct stack_item *curr = stack;

  while (curr && strcmp(curr->name, name) != 0)
    curr = curr->next;

  return curr;
}

value_t copy_value(value_t *from, gpointer memgrp)
{
  value_t to;

  switch (to.type = from->type) {
  case V_DOUBLE:
    to.val.d = from->val.d;
    break;
  case V_STRING:
    if (memgrp) {
      to.val.s = mg_malloc(memgrp, strlen(from->val.s) + 1);
      strcpy(to.val.s, from->val.s);
    } else to.val.s = g_strdup(from->val.s);
    break;
  case V_INVALID:
    break;
  }

  return to;
}

void destroy_value(value_t *value)
{
  if (value->type == V_STRING)
    g_free(value->val.s);
  value->type = V_INVALID;
}

void create_var(gchar *name, value_t *value)
{
  struct stack_item *curr;

  curr = g_new(struct stack_item, 1);
  curr->next = stack;
  curr->name = g_strdup(name);
  curr->depth = eval_depth;
  curr->value = copy_value(value, NULL);
  stack = curr;
}

void set_var(gchar *name, value_t *value)
{
  struct stack_item *curr;

  curr = lookup_var(name);
  if (curr) {
    destroy_value(&curr->value);
    curr->value = copy_value(value, NULL);
  }
  else create_var(name, value);
}

void create_frame()
{
  eval_depth++;
}

void drop_frame()
{
  struct stack_item *curr = stack;

  while (curr && curr->depth == eval_depth) {
    curr = curr->next;
    g_free(stack->name);
    destroy_value(&stack->value);
    g_free(stack);
    stack = curr;
  }
  eval_depth--;
}

exp_t *copy_exp(exp_t *exp, gpointer memgrp)
{
  exp_t *newexp;
  gint i;

  if (exp == NULL) return NULL;

  newexp = (exp_t *)mg_malloc(memgrp, sizeof(exp_t));

  newexp->pos = exp->pos;
  switch ((newexp->type = exp->type)) {
  case E_LITERAL:
    newexp->data.val = copy_value(&exp->data.val, newexp);
    break;
  case E_OPER:
    newexp->data.ptr = exp->data.ptr;
    break;
  
  case E_VARIABLE:
  case E_USERDEF:
  case E_USERCALL:
    newexp->data.name = (gchar *)mg_malloc(newexp, strlen(exp->data.name) + 1);
    strcpy(newexp->data.name, exp->data.name);
    break;
  }

  if (exp->args != NULL) {
    newexp->args = (arg_t *)mg_malloc(newexp, sizeof(arg_t) +
				      sizeof(exp_t *) * exp->args->argc);

    newexp->args->argc = exp->args->argc;
    for (i = 0; i < newexp->args->argc; i++)
      newexp->args->args[i] = copy_exp(exp->args->args[i], newexp);
  } else newexp->args = NULL;

  return newexp;
}

struct user_def *lookup_fun(gchar *name)
{
  struct user_def *curr = functions;

  while (curr && strcmp(curr->name, name) != 0)
    curr = curr->next;

  return curr;
}

void visit_file(gchar *file)
{
  gchar *shortname;
  
  g_string_append_c(visited_files.stack, '/');
  if ((shortname = strrchr(file, '/')) != NULL) shortname++;
  else shortname = file;
  g_string_append(visited_files.stack, shortname);
  if ((visited_files.top = strrchr(visited_files.stack->str, '/')) != NULL)
    visited_files.top++;
  
}

void unvisit_file()
{
  g_return_if_fail(visited_files.top != NULL);

  g_string_truncate(visited_files.stack,
		    visited_files.top-1 - visited_files.stack->str);
  if ((visited_files.top = strrchr(visited_files.stack->str, '/')) != NULL)
    visited_files.top++;
}

void create_user_def(YYLTYPE *pos, gchar *name, arg_t *args)
{
  gint i, j;
  struct user_def *fun;
  gchar *argname;
  gpointer memgrp;

  if (args == NULL || args->argc < 1) {
    set_err("fundef error", *pos);
    return;
  }

  for (i = 0; i < args->argc-1; i++) {
    if (args->args[i]->type != E_VARIABLE) {
      set_err("lval required", args->args[i]->pos);
      return;
    }
    for (j = 0; j < i; j++)
      if (strcmp(args->args[i]->data.name, args->args[j]->data.name) == 0) {
	set_err("name clash", args->args[i]->pos);
	return;
      }
  }
  
  fun = lookup_fun(name);
  if (fun) {
    mg_free_group(fun->body);
    mg_free_group(fun->args);
    g_free(fun->file);
    memgrp = MG_NEW_GRP;
  } else {
    fun = g_new(struct user_def, 1);
    fun->name = g_strdup(name);
    fun->next = functions;
    functions = fun;
  }

  fun->file = g_strdup(visited_files.top);
  fun->argc = args->argc-1;
  /* size = argc+1, because this array is NULL terminated. */
  fun->args = (gchar **)mg_malloc(MG_NEW_GRP, sizeof(gchar *) * (fun->argc+1));

  for (i = 0; i < fun->argc; i++) {
    argname = args->args[i]->data.name;
    fun->args[i] = (gchar *)mg_malloc(fun->args, strlen(argname) + 1);
    strcpy(fun->args[i], argname);
  }
  fun->args[fun->argc] = NULL;

  fun->body = copy_exp(args->args[args->argc-1], MG_NEW_GRP);
}

value_t call_user_fun(YYLTYPE *pos, gchar *name, arg_t *args)
{
  struct user_def *fun;
  gint i;
  value_t val, *values;

  if ((fun = lookup_fun(name)) == NULL)
    RETURN_ERROR_VALUE("undefined fun", *pos);
  
  if ((args == NULL && fun->argc != 0) ||
      (args != NULL && fun->argc != args->argc))
    RETURN_ERROR_VALUE("argcnt error", *pos);
  
  values = g_new(value_t, fun->argc);
  for (i = 0; i < fun->argc; i++)
    values[i] = eval_expr(args->args[i]);

  visit_file(fun->file ?: "toplevel");
  create_frame();

  for (i = 0; i < fun->argc; i++)
    create_var(fun->args[i], &values[i]);
  g_free(values);

  val = eval_expr(fun->body);
  if (return_value.type != V_INVALID) {
    val = return_value;
    return_value.type = V_INVALID;
  }

  drop_frame();
  unvisit_file();
  
  return val;
}

value_t eval_expr(exp_t *exp)
{
  struct stack_item *var;
  value_t retval;

  retval.type = V_DOUBLE;
  retval.val.d = 0;

  g_return_val_if_fail(exp != NULL, retval);

  /* If erronous, propagate quickly. */
  if (cparams.error || user_break || return_value.type != V_INVALID)
    return retval;

  /* Do an unblocking (immediately returning) iteration on GUI */
  gtk_main_iteration_do(FALSE);

  switch (exp->type) {

  case E_LITERAL:
    return exp->data.val;
   
  case E_OPER:
    return ((oper_ptr)exp->data.ptr)(&exp->pos, exp->args);

  case E_VARIABLE:
    var = lookup_var(exp->data.name);
    if (var) return copy_value(&var->value, cparams.memgrp);
    return retval;

  case E_USERDEF:
    create_user_def(&exp->pos, exp->data.name, exp->args);
    return retval;

  case E_USERCALL:
    return call_user_fun(&exp->pos, exp->data.name, exp->args);
  }

  return retval;
}

gchar *print_value(value_t *val)
{
  switch (val->type) {
  case V_DOUBLE:
    return g_strdup_printf("%g", val->val.d);
  case V_STRING:
    return g_strdup_printf("\"%s\"", val->val.s);
  case V_INVALID:
    return g_strdup("invalid value");
  }
  return NULL;
}

void init_cparams()
{
  cparams.input       = NULL;
  cparams.error       = NULL;
  cparams.memgrp      = NULL;
  cparams.tree        = NULL;
  cparams.errpos.text = NULL;
  cparams.errpos.first_line = cparams.errpos.last_line = 0;
  cparams.errpos.first_column = cparams.errpos.last_column = 0;
  cparams.result.type = V_INVALID;
}

void reset_cparams()
{
  g_free(cparams.input);
  g_free(cparams.error);
  g_free(cparams.errpos.text);
  mg_free_group(cparams.memgrp);
  destroy_value(&cparams.result);
  init_cparams();
}

void abort_eval(GtkWidget *widget, GtkWidget *entry)
{
  user_break = TRUE;
  gtk_widget_grab_focus(entry);
}

gboolean prepare_eval_str(gchar *str)
{
  g_return_val_if_fail(cparams.input == NULL, FALSE);

  cparams.input = g_strdup(str);
  g_strstrip(cparams.input);
  yy_scan_from_string(cparams.input);
  if (*cparams.input == '\0') return FALSE;

  return TRUE;
}

gchar *eval(gboolean update_lists)
{
#ifndef G_DISABLE_ASSERT
  gint ed = eval_depth;
#endif
  gchar *answer;

  g_assert(widgets[W_ABORT] != NULL &&
	   cparams.error  == NULL   &&
	   cparams.memgrp == NULL   &&
	   cparams.tree   == NULL   &&
	   cparams.result.type == V_INVALID);

  /* yyparse() sets cparams.memgrp and cparams.tree, and
   * may also set cparams.error.
   */
  yyparse();
  yy_delete_default_buffer();

  gtk_widget_set_sensitive(widgets[W_ABORT], ++call_depth);
  if (cparams.error == NULL) {
    user_break = FALSE;
    return_value.type = V_INVALID;
    cparams.result = eval_expr(cparams.tree);
  }
  gtk_widget_set_sensitive(widgets[W_ABORT], --call_depth);
  g_assert(ed == eval_depth);

  if (cparams.error == NULL && !user_break)
    answer = print_value(&cparams.result);
  else {
    /* No need to destroy result, it's not an own copy (part of memgrp). */
    cparams.result.type = V_INVALID;
    if (user_break)
      answer = g_strdup("user break");
    else if (cparams.errpos.text == NULL) /* One line input */
      answer = g_strdup_printf("%s at char %d", cparams.error,
			       cparams.errpos.first_column);
    else /* File input */
      answer = g_strdup_printf("%s in '%s' at %d:%d", cparams.error,
			       cparams.errpos.text,
			       cparams.errpos.first_line,
			       cparams.errpos.first_column);
  }

  /* Create an own copy of the final result. */
  cparams.result = copy_value(&cparams.result, NULL);

  if (update_lists) {
    update_varlist();
    update_funlist();
  }

  /* Return value must be freed after use. */
  return answer;
}

gchar *eval_str(gchar *str, gboolean update_lists)
{
  if (prepare_eval_str(str)) return eval(update_lists);
  else return NULL;
}

value_t sub_eval_str(gchar *str, YYLTYPE strpos)
{
  cparam_t parstack;
  value_t retval;

  parstack = cparams; /* Save params: input, memgrp, tree */
  init_cparams();  /* Pretend to start a new session. */

  g_free(eval_str(str, FALSE));
  retval = copy_value(&cparams.result, parstack.memgrp);

  if ((parstack.error = cparams.error) != NULL) { /* Keep error msg. */
    parstack.errpos = strpos; /* FIXME: how to translate position? */
    parstack.errpos.text = cparams.errpos.text;
  }
  cparams.error = NULL; /* To avoid deletion of the potential message */
  reset_cparams(); /* Restore original evaluation session. */
  cparams = parstack;

  return retval;
}

void eval_entry(GtkWidget *entry, GtkWidget *clist)
{
  gchar *clist_row[] = { NULL, "running..." }, *result;
  gint row = 0; /* Set value to avoid warning about being uninitialized */
  gboolean doeval;

  if (call_depth != 0) {
    WARNING_RUNNING_EVAL;
    return;
  }

  doeval = prepare_eval_str(gtk_entry_get_text(GTK_ENTRY(entry)));
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  if (doeval) {
    clist_row[0] = cparams.input;
    if (clist) {
      row = gtk_clist_append(GTK_CLIST(clist), clist_row);
      gtk_clist_moveto(GTK_CLIST(clist), row, 0, 1.0, 0.0);
    }

    result = eval(TRUE);
    if (clist) gtk_clist_set_text(GTK_CLIST(clist), row, 1, result);
    g_free(result);
  }

  /* Free everything that was allocated during evaluation. */
  reset_cparams();
}

/*************************************************************************/
/********************* Part III: Callback functions **********************/
/*************************************************************************/

void register_widget(gint ID, GtkWidget *widget)
{
  g_return_if_fail(ID < WIDGET_QTY);

  widgets[ID] = widget;
}

void unregister_widget(gint ID)
{
  register_widget(ID, NULL);
}

void insert_text_in_editable(GtkEditable *e, const gchar *text)
{
  gint pos;

  gtk_editable_delete_selection(e);
  pos = gtk_editable_get_position(e);
  gtk_editable_insert_text(e, text, strlen(text), &pos);
  gtk_editable_set_position(e, pos);
}

void destroy()
{
  exit(0);
}

void clear_history()
{
  gtk_clist_clear(GTK_CLIST(widgets[W_HISTORY]));
}

void select_history(GtkWidget *clist, gint row, gint column,
		    GdkEventButton *event, GtkWidget *entry)
{
  gchar *line;

  gtk_clist_get_text(GTK_CLIST(clist), row, column, &line);
  insert_text_in_editable(GTK_EDITABLE(entry), line);
  gtk_clist_unselect_all(GTK_CLIST(clist));
  gtk_widget_grab_focus(entry);
}

void file_dialog(gchar *title, GtkSignalFunc callback)
{
  GtkWidget *filew;
  gchar *dir;

  filew = gtk_file_selection_new(title);
  if (opendir) {
    dir = g_strconcat(opendir, "/", NULL);
    gtk_file_selection_complete(GTK_FILE_SELECTION(filew), dir);
    g_free(dir);
  }
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filew)->ok_button),
		     "clicked", callback, filew);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filew)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(filew));
  gtk_widget_show(filew);
}

gint read_file(gchar *file, GtkWidget *clist, gboolean silent)
{
  FILE *in;
  gchar *clist_row[2];
  gint row = 0; /* Set value to avoid warning about being uninitialized */

  gchar *line = NULL;
  /*  gint linelen = 0; */

  if ((in = fopen(file, "r")) == NULL) {
    if (!silent) {
      line = g_strconcat("File '", file,
			 "' can't be opened for reading!", NULL);
      message("Warning", line);
      g_free(line);
    }
    return 0;
  }
#if 0    
  while (getline(&line, &linelen, in) != -1) {
    g_strdelimit(line, "#", '\0'); /* Drop comments. */
    clist_row[1] = eval_str(line); /* result */
    clist_row[0] = cparams.input;  /* Set by eval_str() call! */

    if (clist_row[1] && clist) {
      row = gtk_clist_append(GTK_CLIST(clist), clist_row);
      gtk_clist_moveto(GTK_CLIST(clist), row, 0, 1.0, 0.0);
    }  
    g_free(clist_row[1]);
    reset_cparams();
  }
#else
  yy_scan_from_file(in);
  visit_file(file);
  clist_row[0] = g_strconcat("Consulting file '", visited_files.top,
			     "'...", NULL);
  clist_row[1] = "running...";
  if (clist) {
    row = gtk_clist_append(GTK_CLIST(clist), clist_row);
    gtk_clist_moveto(GTK_CLIST(clist), row, 0, 1.0, 0.0);
  }  
  line = eval(TRUE);
  if (clist)
    gtk_clist_set_text(GTK_CLIST(clist), row, 1, line ?: "1");
  g_free(line);
  unvisit_file();
  reset_cparams();
#endif

  fclose(in);
  return 1;
}

void open_file_callback(GtkWidget *widget, GtkWidget *filew)
{
  gchar *file;

  file = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filew)));
  gtk_widget_destroy(filew);
  if (read_file(file, widgets[W_HISTORY], FALSE)) {
    g_free(opendir);
    opendir = g_dirname(file);
  }
  g_free(file);
}

void open_file()
{
  if (call_depth != 0) WARNING_RUNNING_EVAL;
  else file_dialog("Open calc file", open_file_callback);
}

void save_history_callback(GtkWidget *widget, GtkWidget *filew)
{
  gchar *fname, *line;
  FILE *out;
  gint i;

  fname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filew));
  out = fopen(fname, "w");
  g_free(opendir);
  opendir = g_dirname(fname);

  if (out == NULL) {
    line = g_strconcat("File '", fname, "' can't be opened for writing!", NULL);
    gtk_widget_destroy(filew);
    message("Warning", line);
    return;
  }

  gtk_widget_destroy(filew);

  fprintf(out, "/* This file was generated by Calc from its history. */");
  i = 0;
  while (gtk_clist_get_text(GTK_CLIST(widgets[W_HISTORY]), i++, 0, &line)) {
    if (strncmp(line, "Consulting", 10) == 0)
      fprintf(out, "\n/* %s */", line);
    else
      fprintf(out, "%s\n%s", (i > 1) ? ";" : "", line);
  }
  fputc('\n', out);
  fclose(out);
}

void save_history()
{
  file_dialog("Save history as", save_history_callback);
}

void modify_variable_callback(GtkWidget *widget, struct varmod *params)
{
  value_t newval;
  gchar *str;
  YYLTYPE dummypos;

  g_return_if_fail(params != NULL &&
		   params->var != NULL &&
		   params->set_dialog != NULL &&
		   params->set_entry != NULL);

  str = gtk_entry_get_text(GTK_ENTRY(params->set_entry));

  if (call_depth != 0) { /* An evaluation is already running */
    g_return_if_fail(cparams.memgrp != NULL);
    newval = sub_eval_str(str, dummypos);
    newval = copy_value(&newval, NULL); /* Create an own copy */
    clear_err(0); /* Drop any errors that were created during sub-eval. */
  } else { /* Call top level evaluation process */
    g_return_if_fail(cparams.memgrp == NULL);
    g_free(eval_str(str, FALSE));
    newval = copy_value(&cparams.result, NULL);
    reset_cparams();
  }

  if (newval.type == V_STRING ||
      (newval.type == V_DOUBLE && finite(newval.val.d))) {
    destroy_value(&params->var->value);
    params->var->value = copy_value(&newval, NULL);
  }

  destroy_value(&newval); /* Own copy => free it */
  gtk_widget_destroy(params->set_dialog);
  g_free(params);

  update_varlist();
  update_funlist();
}

void modify_variable()
{
  GtkWidget *dialog, *label, *entry, *ok, *cancel;
  gchar *mesg;
  struct varmod *params;

  g_return_if_fail(varlist_selection != NULL);

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), "Set Value");
  GTK_WINDOW(dialog)->type = GTK_WINDOW_DIALOG;
  gtk_window_set_position(GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);

  mesg = g_strjoin(NULL, "Set value of '",
		   varlist_selection->name, "' to", NULL);
  label = gtk_label_new(mesg);
  g_free(mesg);

  entry = gtk_entry_new();
  mesg = print_value(&varlist_selection->value);
  gtk_entry_set_text(GTK_ENTRY(entry), mesg);
  g_free(mesg);
  gtk_editable_select_region(GTK_EDITABLE(entry), 0, -1);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		     label, FALSE, FALSE, 5);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		     entry, FALSE, FALSE, 5);

  params = g_new(struct varmod, 1);
  params->var = varlist_selection;
  params->set_dialog = dialog;
  params->set_entry = entry;

  gtk_signal_connect(GTK_OBJECT(entry), "activate",
		     GTK_SIGNAL_FUNC(modify_variable_callback),
		     (gpointer)params);

  ok = gtk_button_new_with_label("Ok");
  gtk_signal_connect(GTK_OBJECT(ok), "clicked",
		     GTK_SIGNAL_FUNC(modify_variable_callback),
		     (gpointer)params);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     ok, TRUE, TRUE, 0);

  cancel = gtk_button_new_with_label("Cancel");
  gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked",
		     GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     cancel, TRUE, TRUE, 0);

  gtk_widget_grab_focus(GTK_WIDGET(entry));
  GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(ok);

  gtk_widget_show_all(dialog);
}

void delete_variable()
{
  struct stack_item **curr = &stack;

  g_return_if_fail(varlist_selection != NULL);

  while (*curr && *curr != varlist_selection) curr = &(*curr)->next;

  g_return_if_fail(*curr != NULL);

  *curr = (*curr)->next;
  g_free(varlist_selection->name);
  g_free(varlist_selection);
  varlist_selection = NULL;

  update_varlist();
}

void insert_function()
{
  GtkWidget *clist = widgets[W_FUNLIST];
  GtkWidget *entry = widgets[W_MAINENTRY];
  gchar *line;
#ifndef G_DISABLE_CHECKS
  gint get_text_ok;
#endif

  g_return_if_fail(clist != NULL && entry != NULL);
#ifndef G_DISABLE_CHECKS
  get_text_ok =
#endif
  gtk_clist_get_text(GTK_CLIST(clist), funlist_selection, 0, &line);
  g_return_if_fail(get_text_ok);
  insert_text_in_editable(GTK_EDITABLE(entry), line);
}

void delete_function()
{
  GtkWidget *clist = widgets[W_FUNLIST];
  struct user_def *this, **curr = &functions;

  g_return_if_fail(clist != NULL);
  this = gtk_clist_get_row_data(GTK_CLIST(clist), funlist_selection);
  g_return_if_fail(this != NULL);

  while (*curr != NULL && *curr != this) curr = &(*curr)->next;

  g_return_if_fail(*curr != NULL);

  *curr = (*curr)->next;
  mg_free_group(this->body);
  mg_free_group(this->args);
  g_free(this->name);
  g_free(this->file);
  g_free(this);

  update_funlist();
}

void select_varlist(GtkWidget *clist, gint row, gint column,
		    GdkEventButton *event, GtkWidget *dummy)
{
  varlist_selection = gtk_clist_get_row_data(GTK_CLIST(clist), row);
}

gint compare_varlist_rows(GtkCList *clist, gconstpointer sa, gconstpointer sb)
{
  value_t *va, *vb;

  va = & ((struct stack_item *)((GtkCListRow *)sa)->data)->value;
  vb = & ((struct stack_item *)((GtkCListRow *)sb)->data)->value;

  switch (va->type) {

  case V_DOUBLE:

    switch (vb->type) {
    case V_DOUBLE:
      if (va->val.d > vb->val.d) return  1;
      if (va->val.d < vb->val.d) return -1;
      return  0;
    default:
      return -1;
    }

  case V_STRING:

    switch (vb->type) {
    case V_DOUBLE: return 1;
    case V_STRING: return strcmp(va->val.s, vb->val.s);
    default:       return -1;
    }
      
  default:
    return 1;
  }
}

void change_varlist_sorting(GtkWidget *clist, gint column)
{
  GtkWidget *head;
  
  /* GTK_JUSTIFY_LEFT = 0, RIGHT = 1
   * GTK_ASCENDING = 0, DESCENDING = 1
   * GTK_ARROW_UP = 0, DOWN = 1
   */

  if (sorted_column == column) {
    sort_order = 1 - sort_order; 
    head = arrow_button_with_label(varlist_titles[sorted_column],
				   sorted_column, 1-sort_order);
    gtk_clist_set_column_widget(GTK_CLIST(clist), sorted_column, head);
    gtk_clist_set_sort_type(GTK_CLIST(clist), sort_order);
    gtk_clist_sort(GTK_CLIST(clist));
 } else {
    sorted_column = 1-sorted_column;
    sort_order = 0;
    head = arrow_button_with_label(varlist_titles[sorted_column],
				   sorted_column, 1-sort_order);
    gtk_clist_set_column_widget(GTK_CLIST(clist), sorted_column, head);

    head = arrow_button_with_label(varlist_titles[1-sorted_column],
				   1-sorted_column, -1);
    gtk_clist_set_column_widget(GTK_CLIST(clist), 1-sorted_column, head);

    gtk_clist_set_sort_column(GTK_CLIST(clist), sorted_column);
    gtk_clist_set_sort_type(GTK_CLIST(clist), sort_order);
    if (sorted_column == 1)
      gtk_clist_set_compare_func(GTK_CLIST(clist), compare_varlist_rows);
    else
      gtk_clist_set_compare_func(GTK_CLIST(clist), NULL);

    gtk_clist_sort(GTK_CLIST(clist));
  }
}

void update_varlist()
{
  GtkWidget *clist = widgets[W_VARLIST];
  GtkWidget *buttons = widgets[W_VARLIST_BUTTONS];
  struct stack_item *selvar, *var = stack;
  gchar *line[2];
  gint row = 0; /* Set value to avoid warning about being uninitialized */
  gboolean empty;

  if (clist == NULL) return;
  g_return_if_fail(buttons != NULL);

  gtk_clist_freeze(GTK_CLIST(clist));

  selvar = varlist_selection;

  gtk_clist_clear(GTK_CLIST(clist));
  empty = TRUE;

  while (var) {
    empty = FALSE;
    line[0] = var->name;
    line[1] = print_value(&var->value);
    row = gtk_clist_append(GTK_CLIST(clist), line);
    gtk_clist_set_row_data(GTK_CLIST(clist), row, (gpointer)var);
    g_free(line[1]);
    var = var->next;
  }

  gtk_clist_sort(GTK_CLIST(clist));

  if (empty) varlist_selection = NULL;
  else if (!selvar || 
	   (row = gtk_clist_find_row_from_data(GTK_CLIST(clist),
					       (gpointer)selvar)) < 0)
    row = 0; /* Default selection */

  gtk_clist_thaw(GTK_CLIST(clist));

  gtk_clist_select_row(GTK_CLIST(clist), row, 0);
  gtk_clist_moveto(GTK_CLIST(clist), row, 0, 0.5, 0.0);

  gtk_widget_set_sensitive(buttons, !empty);
}

void select_funlist(GtkWidget *clist, gint row, gint column,
		    GdkEventButton *event, GtkWidget *dummy)
{
  GtkWidget *delete = widgets[W_FUNLIST_B_DELETE];

  g_return_if_fail(delete != NULL);

  funlist_selection = row;
  if (gtk_clist_get_row_data(GTK_CLIST(clist), row) == NULL)
    gtk_widget_set_sensitive(delete, FALSE);
  else
    gtk_widget_set_sensitive(delete, TRUE);
}

void update_funlist()
{
  GtkWidget *clist = widgets[W_FUNLIST];
  GtkWidget *check = widgets[W_FUNLIST_T_BUILTIN];
  GtkWidget *insert = widgets[W_FUNLIST_B_INSERT];
  GtkWidget *delete = widgets[W_FUNLIST_B_DELETE];
  struct user_def *fun = functions;
  gint row, selrow;
  gchar *tmps, *clist_row[2];
  gchar *selected;

  if (clist == NULL) return;
  g_return_if_fail(insert != NULL && delete != NULL && check != NULL);

  if (gtk_clist_get_text(GTK_CLIST(clist), funlist_selection, 0, &tmps))
    selected = g_strdup(tmps);
  else selected = NULL;

  gtk_clist_freeze(GTK_CLIST(clist));
  gtk_clist_clear(GTK_CLIST(clist));

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check))) {
    clist_row[1] = "builtin";
    for (row = 0; builtins[row].name != NULL; row++) {
      clist_row[0] = g_strconcat(builtins[row].name, builtins[row].arglst, 
				 NULL);
      gtk_clist_append(GTK_CLIST(clist), clist_row);
      g_free(clist_row[0]);
    }
  }

  while (fun) {
    tmps = g_strjoinv(",", fun->args);
    clist_row[0] = g_strconcat(fun->name, "(", tmps, ")", NULL);
    clist_row[1] = fun->file ?: "";
    row = gtk_clist_append(GTK_CLIST(clist), clist_row);
    gtk_clist_set_row_data(GTK_CLIST(clist), row, (gpointer)fun);
    g_free(tmps);
    g_free(clist_row[0]);
    fun = fun->next;
  }

  gtk_clist_sort(GTK_CLIST(clist));

  selrow = row = -1;
  if (selected != NULL) { /* Something was selected, reselect it */
    while (gtk_clist_get_text(GTK_CLIST(clist), ++row, 0, &tmps))
      if (strcmp(tmps, selected) == 0) selrow = row;
    g_free(selected);
  } else row = GTK_CLIST(clist)->rows;
  if (selrow == -1) {
    funlist_selection = -1; /* Perhaps nothing will be selected */
    selrow = 0;
  }
  g_assert(row >= 0); /* row stores the number of rows in clist */

  gtk_clist_thaw(GTK_CLIST(clist));

  if (row > 0) {
    gtk_clist_select_row(GTK_CLIST(clist), selrow, 0);
    gtk_clist_moveto(GTK_CLIST(clist), selrow, 0, 0.5, 0.0);
  } else
    gtk_widget_set_sensitive(delete, FALSE);

  gtk_widget_set_sensitive(insert, row > 0);
}

/*************************************************************************/
/***************************** Part IV: GUI ******************************/
/*************************************************************************/

void query_accept(GtkWidget *widget, gint *done)
{
  *done = 1;
}

void query_refuse(GtkWidget *widget, gint *done)
{
  *done = -1;
}

gchar *query_string(gchar *prompt)
{
  GtkWidget *window, *hbox, *label, *entry;
  gchar *answer;
  gint done = 0;

  window = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_title(GTK_WINDOW(window), "Query");
  gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
  gtk_signal_connect(GTK_OBJECT(window), "destroy",
		     GTK_SIGNAL_FUNC(query_refuse), &done);

  gtk_container_border_width(GTK_CONTAINER(window), 5);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(window), hbox);

  label = gtk_label_new(prompt);
  entry = gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(entry), "activate",
		     GTK_SIGNAL_FUNC(query_accept), &done);

  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

  gtk_widget_grab_focus(GTK_WIDGET(entry));
  gtk_widget_show_all(window);

  while (done == 0 && !user_break)
    gtk_main_iteration();
  
  if (done == 1 || user_break) {
    answer = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
    gtk_widget_destroy(window);
  } else answer = g_strdup("");

  return g_strstrip(answer);
}

gboolean query_yesno(gchar *question, gchar *syes, gchar *sno)
{
  GtkWidget *window, *vbox, *hbox, *label, *yes, *no;
  gint done = 0;

  window = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_title(GTK_WINDOW(window), "Query");
  gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
  gtk_signal_connect(GTK_OBJECT(window), "destroy",
		     GTK_SIGNAL_FUNC(query_refuse), &done);

  gtk_container_border_width(GTK_CONTAINER(window), 5);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(window), vbox);

  label = gtk_label_new(question);
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_FILL);
  gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, FALSE, 0);

  hbox = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  yes = gtk_button_new_with_label(syes ?: "  Yes  ");
  gtk_signal_connect(GTK_OBJECT(yes), "clicked",
		     GTK_SIGNAL_FUNC(query_accept), &done);
  gtk_box_pack_start(GTK_BOX(hbox), yes, TRUE, TRUE, 0);

  no = gtk_button_new_with_label(sno ?: "  No  ");
  gtk_signal_connect_object(GTK_OBJECT(no), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(hbox), no, TRUE, TRUE, 0);

  gtk_widget_show_all(window);

  while (done == 0 && !user_break)
    gtk_main_iteration();

  if (done == 1 || user_break) {
    gtk_widget_destroy(window);
    return TRUE;
  }
  return FALSE;
}

void message(gchar *title, gchar *msg)
{
  GtkWidget *window, *vbox, *label, *ok;

  window = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_title(GTK_WINDOW(window), title);
  gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
  gtk_container_border_width(GTK_CONTAINER(window), 5);

  vbox = gtk_vbox_new(FALSE, 10);
  gtk_container_add(GTK_CONTAINER(window), vbox);

  label = gtk_label_new(msg);
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_FILL);
  gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, FALSE, 0);

  ok = gtk_button_new_with_label("Ok");
  gtk_signal_connect_object(GTK_OBJECT(ok), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(vbox), ok, FALSE, FALSE, 0);
  GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(ok);

  gtk_widget_show_all(window);
}

void close_variable_list()
{ 
  GtkWidget *window = widgets[W_VARLIST_WINDOW];

  g_return_if_fail(window != NULL);
  gtk_widget_destroy(window);
  unregister_widget(W_VARLIST_WINDOW);
  unregister_widget(W_VARLIST);
  unregister_widget(W_VARLIST_BUTTONS);
}

GtkWidget *arrow_button_with_label(gchar *text, GtkJustification just,
				   GtkArrowType arrow_type)
{
  GtkWidget *hbox, *arrow, *label;

  label = gtk_label_new(text);
  if (arrow_type >= 0)
    arrow = gtk_arrow_new(arrow_type, GTK_SHADOW_OUT);
  hbox = gtk_hbox_new(FALSE, 0);

  if (just == GTK_JUSTIFY_RIGHT) {
    gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 2);
    if (arrow_type >= 0) {
      gtk_box_pack_start(GTK_BOX(hbox), arrow, FALSE, FALSE, 2);
      gtk_widget_show(arrow);
    }
  } else {
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
    if (arrow_type >= 0) {
      gtk_box_pack_end(GTK_BOX(hbox), arrow, FALSE, FALSE, 2);
      gtk_widget_show(arrow);
    }
  }

  gtk_widget_show(label);

  return hbox;
}

void show_variable_list()
{
  GtkWidget *window, *vbox, *hbox, *clist, *scrolled;
  GtkWidget *button;

  if (widgets[W_VARLIST_WINDOW] != NULL) return;

  varlist_selection = NULL;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy",
		     GTK_SIGNAL_FUNC(close_variable_list), NULL);
  gtk_widget_set_usize(window, CALC_HEIGHT, CALC_HEIGHT);
  gtk_container_border_width(GTK_CONTAINER(window), 5);
  gtk_window_set_title(GTK_WINDOW(window), "Variable List");
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
  register_widget(W_VARLIST_WINDOW, window);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(window), vbox);


  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrolled),
				  GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  clist = gtk_clist_new(2);
  gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_column_justification(GTK_CLIST(clist), 1, GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_width(GTK_CLIST(clist), 0, VARNAME_WIDTH);
  gtk_clist_column_titles_show(GTK_CLIST(clist));
  sorted_column = sort_order = 1;
  change_varlist_sorting(clist, 0);
  gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		     GTK_SIGNAL_FUNC(select_varlist), NULL);
  gtk_signal_connect(GTK_OBJECT(clist), "click-column",
		     GTK_SIGNAL_FUNC(change_varlist_sorting), NULL);
  register_widget(W_VARLIST, clist);

  gtk_container_add(GTK_CONTAINER(scrolled), clist);
  gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(TRUE, 2);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  register_widget(W_VARLIST_BUTTONS, hbox);

  button = gtk_button_new_with_label("Modify");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(modify_variable), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  button = gtk_button_new_with_label("Delete");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(delete_variable), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(TRUE, 2);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  button = gtk_button_new_with_label("Update");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(update_varlist), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  button = gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(close_variable_list), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

  gtk_widget_show_all(window);

  update_varlist();
}

void close_function_list()
{ 
  GtkWidget *window = widgets[W_FUNLIST_WINDOW];

  g_return_if_fail(window != NULL);
  gtk_widget_destroy(window);
  unregister_widget(W_FUNLIST_WINDOW);
  unregister_widget(W_FUNLIST);
  unregister_widget(W_FUNLIST_B_INSERT);
  unregister_widget(W_FUNLIST_B_DELETE);
}

void show_function_list()
{
  GtkWidget *window, *vbox, *hbox, *check, *clist, *scrolled;
  GtkWidget *button;
  gchar *title[] = { " Function", "In File " };

  if (widgets[W_FUNLIST_WINDOW] != NULL) return;

  funlist_selection = -1;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy",
		     GTK_SIGNAL_FUNC(close_function_list), NULL);
  gtk_widget_set_usize(window, CALC_HEIGHT, CALC_HEIGHT);
  gtk_container_border_width(GTK_CONTAINER(window), 5);
  gtk_window_set_title(GTK_WINDOW(window), "Function List");
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
  register_widget(W_FUNLIST_WINDOW, window);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(window), vbox);

  check = gtk_check_button_new_with_label("List built-in functions too"); 
  register_widget(W_FUNLIST_T_BUILTIN, check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), 1);
  gtk_signal_connect(GTK_OBJECT(check), "toggled",
		     GTK_SIGNAL_FUNC(update_funlist), NULL);
  gtk_box_pack_start(GTK_BOX(vbox), check, FALSE, FALSE, 0);

  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrolled),
				  GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  clist = gtk_clist_new_with_titles(2, title);
  gtk_clist_column_titles_passive(GTK_CLIST(clist));
  gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_column_justification(GTK_CLIST(clist), 1, GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_width(GTK_CLIST(clist), 0, FUNNAME_WIDTH);
  gtk_clist_set_sort_type(GTK_CLIST(clist), GTK_SORT_ASCENDING);
  gtk_clist_set_sort_column(GTK_CLIST(clist), 0);
  gtk_signal_connect(GTK_OBJECT(clist), "select_row",
  		     GTK_SIGNAL_FUNC(select_funlist), NULL);
  register_widget(W_FUNLIST, clist);

  gtk_container_add(GTK_CONTAINER(scrolled), clist);
  gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(TRUE, 2);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  button = gtk_button_new_with_label("Insert");
  register_widget(W_FUNLIST_B_INSERT, button);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(insert_function), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  button = gtk_button_new_with_label("Delete");
  register_widget(W_FUNLIST_B_DELETE, button);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
  		     GTK_SIGNAL_FUNC(delete_function), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(TRUE, 2);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  button = gtk_button_new_with_label("Update");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(update_funlist), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
  button = gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(close_function_list), NULL);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

  gtk_widget_show_all(window);

  update_funlist();
}

void show_about()
{
  GtkWidget *dialog, *box, *label, *close;

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), "About");
  GTK_WINDOW(dialog)->type = GTK_WINDOW_DIALOG;
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);

  box = GTK_DIALOG(dialog)->vbox;

  label = gtk_label_new("C A L C\n\n"
			"A Programmable Calculator for GTK+\n\n"
			"Version " VERSION "\n\n"
			"WWW: http://www.inf.bme.hu/~dhanak/calc\n\n"
			"By David Hanak <dhanak@inf.bme.hu>");
  gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
  gtk_misc_set_padding(GTK_MISC(label), 20, 20);

  box = GTK_DIALOG(dialog)->action_area;

  close = gtk_button_new_with_label("Close");
  gtk_box_pack_start(GTK_BOX(box), close, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS(close, GTK_CAN_DEFAULT);

  gtk_signal_connect_object(GTK_OBJECT(close), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(dialog));

  gtk_widget_grab_default(close);

  gtk_widget_show_all(dialog);
}

GtkWidget *get_main_menu( GtkWidget  *window)
{
  GtkWidget *menubar;
  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);
  
  accel_group = gtk_accel_group_new();
  item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>",
				      accel_group);
  gtk_item_factory_create_items(item_factory, nmenu_items, menu_items, NULL);
  gtk_window_add_accel_group(GTK_WINDOW (window), accel_group);
  menubar = gtk_item_factory_get_widget (item_factory, "<main>");

  return menubar;
}

GtkWidget *create_history(GtkWidget **clist)
{
  GtkWidget *scrolled;
  gchar *title[] = { " Expression", "Result " };

  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrolled),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  *clist = gtk_clist_new_with_titles(2, title);
  gtk_clist_column_titles_passive(GTK_CLIST(*clist));
  gtk_clist_set_selection_mode(GTK_CLIST(*clist), GTK_SELECTION_SINGLE);
  gtk_clist_set_column_justification(GTK_CLIST(*clist), 1, GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_width(GTK_CLIST(*clist), 0, EXP_WIDTH);

  gtk_container_add(GTK_CONTAINER(scrolled), *clist);

  register_widget(W_HISTORY, *clist);

  return scrolled;
}

int main(int argc, char *argv[])
{
  GtkWidget *window, *mainvbox, *vbox, *menu, *history, *clist;
  GtkWidget *hbox, *entry, *abort;
  gchar *conffile;
  gint i;

  gtk_init(&argc, &argv);

  for (i = 0; i < WIDGET_QTY; i++)
    widgets[i] = NULL;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy",
		     GTK_SIGNAL_FUNC(destroy), NULL);
  gtk_widget_set_usize(window, CALC_MAIN_WIDTH, CALC_HEIGHT);
  gtk_window_set_title(GTK_WINDOW(window), "Calc");
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);

  mainvbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(window), mainvbox);

  menu = get_main_menu(window);
  gtk_box_pack_start(GTK_BOX(mainvbox), menu, FALSE, FALSE, 0);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(vbox), 5);
  gtk_box_pack_start(GTK_BOX(mainvbox), vbox, TRUE, TRUE,  0);

  history = create_history(&clist);
  entry = gtk_entry_new();

  gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		     GTK_SIGNAL_FUNC(select_history), (gpointer)entry);
  gtk_box_pack_start(GTK_BOX(vbox), history, TRUE, TRUE, 0);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  register_widget(W_MAINENTRY, entry);
  gtk_signal_connect(GTK_OBJECT(entry), "activate",
		     GTK_SIGNAL_FUNC(eval_entry), (gpointer)clist);
  gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

  abort = gtk_button_new_with_label("  Abort  ");
  register_widget(W_ABORT, abort);
  gtk_widget_set_sensitive(abort, 0);
  gtk_signal_connect(GTK_OBJECT(abort), "clicked",
		     GTK_SIGNAL_FUNC(abort_eval), (gpointer)entry);
  gtk_box_pack_end(GTK_BOX(hbox), abort, FALSE, FALSE, 0);

  gtk_widget_grab_focus(entry);
  gtk_widget_show_all(window);

  init_cparams();
  stack = NULL;
  functions = NULL;
  visited_files.stack = g_string_new("");
  visited_files.top = NULL;
  eval_depth = call_depth = 0;

  conffile = g_strjoin(NULL, getenv("HOME"), "/.calc", NULL);
  read_file(conffile, NULL, TRUE);
  g_free(conffile);

  gtk_main();

  return 0;
}
