/* $Id: calc.h,v 2.2 2000/02/21 22:05:07 david Exp $
 */

#ifndef CALC_H
#define CALC_H

#ifdef HAVE_IEEEFP_H
#  include <ieeefp.h>
#endif
#include <math.h>
#include <glib.h>

#ifndef NAN
#  define NAN (0.0/0.0)
#endif

#ifndef INF
#  define INF (1.0/0.0)
#endif

#if !HAVE_FINITE && !HAVE_ISINF
#  error Missing 'finite' or 'isinf'. Cannot compile.
#else
#  if !HAVE_FINITE
#    define finite(d) (!isnan(d) && !isinf(d))
#  endif
#  if !HAVE_ISINF
#    define isinf(d) (!finite(d) && !isnan(d))
#  endif
#endif

#ifndef YYLTYPE /* From parse.h */
typedef
  struct yyltype
    {
      int timestamp;
      int first_line;
      int first_column;
      int last_line;
      int last_column;
      char *text;
   }
  yyltype;

#define YYLTYPE yyltype
#endif

enum VAL_TYPE {
  V_INVALID,
  V_DOUBLE,
  V_STRING
};

enum EXP_TYPE {
  E_LITERAL,
  E_OPER,
  E_VARIABLE,
  E_USERDEF,
  E_USERCALL
};

struct exp_s;
struct arglist;

typedef struct value_s {
  enum VAL_TYPE type;
  union {
    gdouble  d;
    gchar   *s;
  } val;
} value_t;

typedef struct arglist {
  gint argc;
  struct exp_s *args[0];
} arg_t;

typedef value_t (*oper_ptr)(YYLTYPE *pos, arg_t *args);

typedef struct exp_s {
  YYLTYPE pos;
  enum EXP_TYPE type;
  union {
    value_t val;  /* For E_LITERAL */
    gchar *name;  /* For E_VARIABLE, E_USERDEF and E_USERCALL */
    oper_ptr ptr; /* For E_OPER */
  } data;
  struct arglist *args;
} exp_t;

typedef struct builtin {
  gchar   *name;
  oper_ptr func;
  gchar   *arglst;
} builtin_t;

struct file_stack {
  GString *stack;
  gchar *top;
};

typedef struct cparam {
  gchar  *input, *error;
  YYLTYPE errpos;
  void   *memgrp;
  exp_t  *tree;
  value_t result;
} cparam_t;

extern builtin_t builtins[];
extern struct file_stack visited_files;
extern cparam_t cparams;
extern value_t return_value;

#define expr_alloc(type, size)                                 \
  ((type *)(cparams.memgrp = mg_malloc(cparams.memgrp, size)))

#define set_err(err, pos) G_STMT_START {                       \
  if (cparams.error == NULL) {                                 \
    cparams.error = g_strdup(err);                             \
    cparams.errpos = pos;                                      \
    cparams.errpos.text = g_strdup(visited_files.top);         \
  }                                                            \
} G_STMT_END

#define clear_err(action) G_STMT_START {                       \
  if (cparams.error) {                                         \
    g_free(cparams.error);                                     \
    cparams.error = NULL;                                      \
    action;                                                    \
  }                                                            \
} G_STMT_END

int  yyparse();
int  yylex();
void yyerror (gchar *error);
void parse_input(gchar *buf, gint *result, gint max_size);

void init_cparams();
void reset_cparams();

arg_t *build_arglist(gint argc, ...);
arg_t *add_arg(arg_t *al, exp_t *nextarg);
exp_t *build_exp(enum EXP_TYPE type, ...);
oper_ptr builtin_lookup(gchar *);
gchar *eval_str(gchar *str, gboolean update_lists);
value_t sub_eval_str(gchar *str, YYLTYPE strpos);
value_t eval_expr(exp_t *exp);

value_t copy_value(value_t *from, gpointer memgrp);
void destroy_value(value_t *value);
gchar *print_value(value_t *value);
void create_var(gchar *name, value_t *value);
void set_var(gchar *name, value_t *value);
void create_frame();
void drop_frame();

gchar *query_string(gchar *prompt);
gboolean query_yesno(gchar *question, gchar *syes, gchar *sno);
void message(gchar *title, gchar *msg);

gint todouble(value_t *v);
gint tostring(value_t *v);

#define OPER_FUNC(name) value_t name(YYLTYPE *, arg_t *)

gdouble cot(gdouble a);
gdouble mylog(gdouble a, gdouble b);

OPER_FUNC(negate    );
OPER_FUNC(follow    );
OPER_FUNC(add       );
OPER_FUNC(subtract  );
OPER_FUNC(multiply  );
OPER_FUNC(divide    );
OPER_FUNC(modulo    );
OPER_FUNC(op_pow    );
OPER_FUNC(and       );
OPER_FUNC(or        );
OPER_FUNC(ifthenelse);
OPER_FUNC(dowhile   );
OPER_FUNC(whiledo   );
OPER_FUNC(fordo     );
OPER_FUNC(switchcase);
OPER_FUNC(isequal   );
OPER_FUNC(equals    );
OPER_FUNC(newvar    );
OPER_FUNC(isequal   );
OPER_FUNC(notequal  );
OPER_FUNC(lessthan  );
OPER_FUNC(grtrthan  );
OPER_FUNC(lsthoreq  );
OPER_FUNC(gtthoreq  );
OPER_FUNC(not       );
OPER_FUNC(group     );
OPER_FUNC(postinc   );
OPER_FUNC(postdec   );
OPER_FUNC(funreturn );

#endif
