/*
 *    Calc: A Programmable Calculator for GTK+
 *    Copyright (C) 2000  David Hanak (dhanak@inf.bme.hu)
 *
 *    This file can be distributed under GNU GPL.  See file COPYING.
 *
 *    $Id: expr.c,v 2.2 2000/02/21 22:05:07 david Exp $
 */

#include <config.h>

#include <ctype.h>
#ifdef STDC_HEADERS
#  include <string.h>
#endif

#include "memgrp.h"
#include "calc.h"

gdouble cot(gdouble a)              { return 1/tan(a);      }
gdouble mylog(gdouble a, gdouble b) { return log(b)/log(a); }

#define RETURN_ERROR(mesg, pos) G_STMT_START {         		          \
  set_err(mesg, pos);                                  		          \
  v.type = V_INVALID;                               		          \
  return v;                                         		          \
} G_STMT_END

#define RETURN_ON_ERROR(stmt) G_STMT_START {                              \
  stmt;                                                                   \
  if (cparams.error != NULL) {                                            \
    v.type = V_INVALID;                                                   \
    return v;                                                             \
  }                                                                       \
} G_STMT_END

#define CHECK_ARGS(arg_need) G_STMT_START {                               \
  if (a == NULL || arg_need != a->argc) RETURN_ERROR("argcnt error", *p); \
} G_STMT_END

#define CHECK_ARGS_MIN(arg_min) G_STMT_START {                            \
  if (a == NULL || arg_min > a->argc)                                     \
    RETURN_ERROR("argcnt error", *p);                                     \
} G_STMT_END

#define CHECK_ARGS_MAX(arg_max) G_STMT_START {                            \
  if (a != NULL && arg_max < a->argc)                                     \
    RETURN_ERROR("argcnt error", *p);                                     \
} G_STMT_END

#define CHECK_ARGS_BOUND(arg_min, arg_max) G_STMT_START {                 \
  CHECK_ARGS_MIN(arg_min);                                                \
  CHECK_ARGS_MAX(arg_max);                                                \
} G_STMT_END

#define _EVAL_ARG(arg)       eval_expr(a->args[arg])
#define _EVAL_ARG_TO(v, arg) v = eval_expr(a->args[arg])
#define EVAL_ARG(arg)        RETURN_ON_ERROR(_EVAL_ARG(arg))
#define EVAL_ARG_TO(v, arg)  RETURN_ON_ERROR(_EVAL_ARG_TO(v, arg))

#define NEED_VAR_AS_ARG(argidx) G_STMT_START {                            \
  if (a->args[argidx]->type != E_VARIABLE)                                \
    RETURN_ERROR("lval required", a->args[argidx]->pos);                  \
} G_STMT_END

void copy_string(value_t *v, const gchar *str)
{
  v->type = V_STRING;
  v->val.s = expr_alloc(gchar, strlen(str) + 1);
  strcpy(v->val.s, str);
}

gint todouble(value_t *v)
{
  gchar *endpt;
  gdouble d;

  switch (v->type) {
  case V_DOUBLE:
    break;
  case V_STRING:
    d = g_strtod(v->val.s, &endpt);
    while (isspace((gint)*endpt)) endpt++;
    if (endpt[0] != '\0') return 0;
    v->type = V_DOUBLE;
    v->val.d = d; /* val.s will be freed together with the whole expr. */
    break;
  case V_INVALID:
    return 0;
  }
  return 1;
}

gint tostring(value_t *v)
{
  gchar *tmp;

  switch (v->type) {
  case V_STRING: break;
  default:
    tmp = print_value(v);
    copy_string(v, tmp);
    g_free(tmp);
  }
  return 1;
}

#define NEED_DOUBLE(v, pos) G_STMT_START {                              \
  if (!todouble(&v)) RETURN_ERROR("double expected", pos);              \
} G_STMT_END

#define NEED_STRING(v, pos) tostring(&v)

#define GET_DOUBLE_FROM_ARG(v, arg) G_STMT_START {                      \
  EVAL_ARG_TO(v, arg);                                                  \
  NEED_DOUBLE(v, a->args[arg]->pos);                                    \
} G_STMT_END

#define GET_STRING_FROM_ARG(v, arg) G_STMT_START {                      \
  EVAL_ARG_TO(v, arg);                                                  \
  NEED_STRING(v, a->args[arg]->pos);                                    \
} G_STMT_END 

#define SET_DOUBLE(v, value) G_STMT_START {                             \
  v.type = V_DOUBLE;                                                    \
  v.val.d = value;                                                      \
} G_STMT_END

#define SET_STRING(v, value) copy_string(&v, value)

#define IS_TRUE(tmpv, arg)                                              \
  (GET_DOUBLE_FROM_ARG(tmpv, arg), tmpv.val.d != 0)

#define OP_HEAD(name, argc, decl) value_t name(YYLTYPE *p, arg_t *a)    \
                  { value_t v; decl;                                    \
                    g_return_val_if_fail(cparams.error == NULL, v);     \
                    CHECK_ARGS(argc)
#define OP_TAIL()   return v;                                           \
                  }		                    

#define OP_HEAD_NOCHECK(name, decl) value_t name(YYLTYPE *p, arg_t *a)  \
                  { value_t v; decl;                                    \
                    g_return_val_if_fail(cparams.error == NULL, v)

/* Head for unary operators and for functions with one argument. */
#define STD_UNOP_HEAD(name) 		                                \
  OP_HEAD(name, 1, value_t ev);      					\
    GET_DOUBLE_FROM_ARG(ev, 0)

/* Unary prefix operator called 'name' using C operator 'op'. */
#define STD_UNOP(op, name)                                              \
  STD_UNOP_HEAD(name);                                                  \
    SET_DOUBLE(v, op ev.val.d);                                         \
  OP_TAIL()

/* Unary function called op_'name' using C function called 'name'. */
#define STD_FN1(name)                                                   \
  STD_UNOP_HEAD(op_ ## name);                                           \
    SET_DOUBLE(v, name(ev.val.d));                                      \
  OP_TAIL()

/* Head for binary operators and for functions with two arguments. */
#define STD_BINOP_HEAD(name)                                            \
  OP_HEAD(name, 2, value_t ev1; value_t ev2);            		\
    GET_DOUBLE_FROM_ARG(ev1, 0);                           		\
    GET_DOUBLE_FROM_ARG(ev2, 1)

/* Binary infix operator called 'name' using C operator 'op'. */
#define STD_BINOP(op, name)                                             \
  STD_BINOP_HEAD(name);                                                 \
    SET_DOUBLE(v, ev1.val.d op ev2.val.d);        		        \
  OP_TAIL()

/* Binary function called op_'name' using C function called 'name'. */
#define STD_FN2(name)                                                   \
  STD_BINOP_HEAD(op_ ## name);                                          \
    SET_DOUBLE(v, name(ev1.val.d, ev2.val.d));                          \
  OP_TAIL()

#define STD_CMPOP(op, name)                                             \
  OP_HEAD(name, 2, value_t ev1; value_t ev2);                           \
    EVAL_ARG_TO(ev1, 0);                           		        \
    EVAL_ARG_TO(ev2, 1);                           		        \
    if (ev1.type == V_STRING || ev2.type == V_STRING) {                 \
      NEED_STRING(ev1, a->args[0]->pos);                                \
      NEED_STRING(ev2, a->args[1]->pos);                                \
      SET_DOUBLE(v, strcmp(ev1.val.s, ev2.val.s) op 0);                 \
    } else if (ev1.type == V_DOUBLE || ev2.type == V_DOUBLE) {          \
      NEED_DOUBLE(ev1, a->args[0]->pos);                                \
      NEED_DOUBLE(ev2, a->args[1]->pos);                                \
      SET_DOUBLE(v, ev1.val.d op ev2.val.d);                            \
    } else SET_DOUBLE(v, 1);						\
  OP_TAIL()

STD_FN1(fabs );
STD_FN1(cos  );
STD_FN1(sin  );
STD_FN1(tan  );
STD_FN1(cot  );
STD_FN1(log10);
STD_FN1(log  );
STD_FN1(exp  );
STD_FN1(sqrt );

STD_FN2(pow  );
STD_FN2(mylog);

STD_UNOP(-, negate);
STD_UNOP(!, not);

STD_BINOP(+, add);
STD_BINOP(-, subtract);
STD_BINOP(*, multiply);
STD_BINOP(/, divide);

STD_CMPOP(==, isequal);
STD_CMPOP(!=, notequal);
STD_CMPOP(<, lessthan);
STD_CMPOP(>, grtrthan);
STD_CMPOP(<=, lsthoreq);
STD_CMPOP(>=, gtthoreq);

OP_HEAD(and, 2, value_t ev1; value_t ev2);
  GET_DOUBLE_FROM_ARG(ev1, 0);
  if (ev1.val.d == 0) SET_DOUBLE(v, 0);
  else {
    GET_DOUBLE_FROM_ARG(ev2, 1);
    SET_DOUBLE(v, ev2.val.d != 0);
  }
OP_TAIL();

OP_HEAD(or, 2, value_t ev1; value_t ev2);
  GET_DOUBLE_FROM_ARG(ev1, 0);
  if (ev1.val.d == 1) SET_DOUBLE(v, 1);
  else {
    GET_DOUBLE_FROM_ARG(ev2, 1);
    SET_DOUBLE(v, ev2.val.d != 0);
  }
OP_TAIL();

OP_HEAD(modulo, 2, value_t ev1; value_t ev2);
 GET_DOUBLE_FROM_ARG(ev1, 0);
 GET_DOUBLE_FROM_ARG(ev2, 1);
 SET_DOUBLE(v, ev1.val.d - ((gint)(ev1.val.d / ev2.val.d)) * ev2.val.d);
OP_TAIL();

OP_HEAD(follow, 2,);
  EVAL_ARG(0);
  _EVAL_ARG_TO(v, 1);
OP_TAIL();

OP_HEAD_NOCHECK(ifthenelse, value_t ev);
  CHECK_ARGS_BOUND(2, 3);
  if (IS_TRUE(ev, 0))    _EVAL_ARG_TO(v, 1);
  else if (a->argc == 3) _EVAL_ARG_TO(v, 2);
  else SET_DOUBLE(v, 0);
OP_TAIL();

OP_HEAD(dowhile, 2, value_t ev);
  do EVAL_ARG(0);
  while (IS_TRUE(ev, 1));
  SET_DOUBLE(v, 0);
OP_TAIL();

OP_HEAD(whiledo, 2, value_t ev);
  while (IS_TRUE(ev, 0))
    EVAL_ARG(1);
  SET_DOUBLE(v, 0);
OP_TAIL();

OP_HEAD(fordo, 4, value_t ev);
  for(EVAL_ARG(0);
      IS_TRUE(ev, 1);
      EVAL_ARG(2))
    EVAL_ARG(3);
  SET_DOUBLE(v, 0);
OP_TAIL();

OP_HEAD_NOCHECK(switchcase, value_t ev1; value_t ev2; gint i; gboolean equal);
  CHECK_ARGS_MIN(1);
  if (a->argc % 2 != 1) RETURN_ERROR("argcnt error", *p);
  EVAL_ARG_TO(ev1, a->argc-1);
  if (ev1.type == V_INVALID)
    RETURN_ERROR("invalid test", a->args[a->argc-1]->pos);

  for (i = 0; i < a->argc-1; i += 2) {
    if (a->args[i] != NULL) { /* case specified */
      EVAL_ARG_TO(ev2, i);
      if (ev1.type == V_DOUBLE) {
	NEED_DOUBLE(ev2, a->args[i]->pos);
	equal = (ev1.val.d == ev2.val.d);
      }
      else {
	NEED_STRING(ev2, a->args[i]->pos);
	equal = (ev1.val.s == ev2.val.s);
      }
    } else equal = TRUE; /* default case */

    if (equal) {
      _EVAL_ARG_TO(v, i+1);
      return v;
    }
  }
  SET_DOUBLE(v, 0);
OP_TAIL();

OP_HEAD(equals, 2,);
  NEED_VAR_AS_ARG(0);
  EVAL_ARG_TO(v, 1);
  set_var(a->args[0]->data.name, &v);
OP_TAIL();

OP_HEAD_NOCHECK(newvar, gint i);
  CHECK_ARGS_MIN(1);
  SET_DOUBLE(v, 0);
  for (i = 0; i < a->argc; i++) {
    NEED_VAR_AS_ARG(i);
    create_var(a->args[i]->data.name, &v);
  }
OP_TAIL();

OP_HEAD(group, 1,);
  create_frame();
  _EVAL_ARG_TO(v, 0);
  drop_frame();
OP_TAIL();

OP_HEAD(postinc, 1,);
  NEED_VAR_AS_ARG(0);
  GET_DOUBLE_FROM_ARG(v, 0);
  v.val.d++;
  set_var(a->args[0]->data.name, &v);
  v.val.d--;
OP_TAIL();

OP_HEAD(postdec, 1,);
  NEED_VAR_AS_ARG(0);
  GET_DOUBLE_FROM_ARG(v, 0);
  v.val.d--;
  set_var(a->args[0]->data.name, &v);
  v.val.d++;
OP_TAIL();

OP_HEAD(tostr, 1,);
  GET_STRING_FROM_ARG(v, 0);
OP_TAIL();

OP_HEAD(todbl, 1,);
  EVAL_ARG_TO(v, 0);
  if (!todouble(&v))
    SET_DOUBLE(v, NAN);
OP_TAIL();

OP_HEAD(isstr, 1, value_t ev);
  EVAL_ARG_TO(ev, 0);
  SET_DOUBLE(v, ev.type == V_STRING);
OP_TAIL();

OP_HEAD(isdbl, 1, value_t ev);
  EVAL_ARG_TO(ev, 0);
  SET_DOUBLE(v, ev.type == V_DOUBLE);
OP_TAIL();

OP_HEAD(op_isnan, 1, value_t ev);
  EVAL_ARG_TO(ev, 0);
  SET_DOUBLE(v, ev.type == V_DOUBLE && isnan(ev.val.d));
OP_TAIL();

OP_HEAD(op_isinf, 1, value_t ev);
  EVAL_ARG_TO(ev, 0);
  SET_DOUBLE(v, ev.type == V_DOUBLE && isinf(ev.val.d));
OP_TAIL();

OP_HEAD(onerror, 2,);
  _EVAL_ARG_TO(v, 0);
  clear_err(_EVAL_ARG_TO(v, 1));
OP_TAIL();

OP_HEAD(error, 1, value_t ev);
  GET_STRING_FROM_ARG(ev, 0);
  RETURN_ERROR(ev.val.s, *p);
OP_TAIL();

OP_HEAD(iserr, 1,);
  _EVAL_ARG(0);
  SET_DOUBLE(v, 0);
  clear_err(v.val.d = 1);
OP_TAIL();
  
OP_HEAD_NOCHECK(cat, gint i; gint tlen; gchar *tmp; value_t ev);
  if (a == NULL) SET_STRING(v, "");
  else {
    v.val.s = NULL;
    for(i = 0, tlen = 0; i < a->argc; i++) {
      GET_STRING_FROM_ARG(ev, i);
      tmp = v.val.s;
      if (tmp) {
	v.val.s = g_strconcat(tmp, ev.val.s, NULL);
	g_free(tmp);
      }
      else v.val.s = g_strdup(ev.val.s);
    }
    SET_STRING(v, v.val.s);
  } 
OP_TAIL();

OP_HEAD(readstr, 1, value_t ev; gchar *s);
   GET_STRING_FROM_ARG(ev, 0);
   s = query_string(ev.val.s);
   SET_STRING(v, s);
   g_free(s);
OP_TAIL();

OP_HEAD(wrmsg, 1,);
  GET_STRING_FROM_ARG(v, 0);
  message("User Message", v.val.s);
OP_TAIL();

OP_HEAD(yesno, 1, value_t ev);
  GET_STRING_FROM_ARG(ev, 0);
  SET_DOUBLE(v, query_yesno(ev.val.s, NULL, NULL));
OP_TAIL();

OP_HEAD(op_eval, 1, value_t ev);
  GET_STRING_FROM_ARG(ev, 0);
  v = sub_eval_str(ev.val.s, a->args[0]->pos);
OP_TAIL();

OP_HEAD(funreturn, 1,);
  EVAL_ARG_TO(v, 0);
  return_value = v;
OP_TAIL();

OP_HEAD(slen, 1, value_t ev);
  GET_STRING_FROM_ARG(ev, 0);
  SET_DOUBLE(v, strlen(ev.val.s));
OP_TAIL();

OP_HEAD(strpos, 2, value_t ev1; value_t ev2; gchar *pos);
  GET_STRING_FROM_ARG(ev1, 0);
  GET_STRING_FROM_ARG(ev2, 1);
  pos = strstr(ev1.val.s, ev2.val.s);
  if (pos) SET_DOUBLE(v, pos - ev1.val.s);
  else     SET_DOUBLE(v, -1);
OP_TAIL();

OP_HEAD_NOCHECK(strsub, value_t ev1; value_t ev2; gint from; gint to; gint slen);
  CHECK_ARGS_BOUND(2,3);
  GET_STRING_FROM_ARG(ev1, 0);
  slen = strlen(ev1.val.s);
  GET_DOUBLE_FROM_ARG(ev2, 1);
  from = (gint)ev2.val.d;
  if (a->argc == 3) {
    GET_DOUBLE_FROM_ARG(ev2, 2);
    to = (gint)ev2.val.d;
  } else to = slen-1;

  if (to >= slen) to = slen-1;
  if (from < 0)   from = 0;
  if (from > to) SET_STRING(v, "");
  else {
    SET_STRING(v, ev1.val.s + from);
    v.val.s[to-from+1] = '\0';
  }
OP_TAIL();

/* Array of builtin functions. */
builtin_t builtins[] = {
  { "strcat",  cat,      "(...)"             },

  { "abs",     op_fabs,  "(x)"  	     },
  { "cos",     op_cos,   "(x)"  	     },
  { "sin",     op_sin,   "(x)"  	     },
  { "tg",      op_tan,   "(x)"  	     },
  { "tan",     op_tan,   "(x)"  	     },
  { "ctg",     op_cot,   "(x)"  	     },
  { "cot",     op_cot,   "(x)"  	     },
  { "lg",      op_log10, "(x)"  	     },
  { "ln",      op_log,   "(x)"  	     },
  { "exp",     op_exp,   "(x)"  	     },
  { "sqrt",    op_sqrt,  "(x)"  	     },
  { "tostr",   tostr,    "(val)"  	     },
  { "todbl",   todbl,    "(str)"  	     },
  { "isstr",   isstr,    "(val)"  	     },
  { "isdbl",   isdbl,    "(val)"  	     },
  { "isnan",   op_isnan, "(val)"  	     },
  { "isinf",   op_isinf, "(val)"  	     },
  { "error",   error,    "(msg)"  	     },
  { "iserr",   iserr,    "(exp)"  	     },
  { "read",    readstr,  "(querystr)"        },
  { "msg",     wrmsg,    "(msg)"             },
  { "ask",     yesno,    "(querystr)"        },
  { "eval",    op_eval,  "(expstr)"          },
  { "strlen",  slen,     "(str)"             },

  { "pow",     op_pow,   "(base,power)"      },
  { "log",     op_mylog, "(base,val)"        },
  { "onerror", onerror,  "(exp1,exp2)"       },
  { "strpos",  strpos,   "(str1,str2)"       },

  { "strsub",  strsub,   "(str,start[,end])" },

  { NULL }
};
