/* Scanner for calc -*- C -*- program. 
 *
 *    Calc: A Programmable Calculator for GTK+
 *    Copyright (C) 2000  David Hanak (dhanak@inf.bme.hu)
 *
 *    This file can be distributed under GNU GPL.  See file COPYING.
 *
 *    $Id: lex.l,v 2.1 2000/02/21 22:05:07 david Exp $
 */

/* Options for `lex'. We don't need an interactive scanner. */
%option noyywrap never-interactive batch

%{
#include <config.h>

#include <stdio.h>
#ifdef STDC_HEADERS
#  include <string.h>
#endif
#include <math.h>
#include <glib.h>
#include "memgrp.h"
#include "calc.h"
#include "parse.h"
#define YY_NO_UNPUT

typedef struct reserved_s {
  char *name;
  int   tokenval;
} reserved_t;

static reserved_t reswords[] = {
  { "if",      IF      },
  { "else",    ELSE    },
  { "do",      DO      },
  { "while",   WHILE   },
  { "for",     FOR     },
  { "switch",  SWITCH  },
  { "case",    CASE    },
  { "default", DEFAULT },
  { "var",     VAR     },
  { "fun",     FUN     },
  { "and",     AND     },
  { "or",      OR      },
  { "not",     NOT     },
  { "return",  RETURN  },
  { NULL }
};

int line, col;
GString *parsed_str = NULL;
YYLTYPE string_start;

void yynewline();
int resword_lookup(char *word);

#define DEFSTRLEN 100

#define VALUE_OF(x) G_STMT_START {     \
  yylval->val.type = V_DOUBLE;         \
  yylval->val.val.d = x;               \
  return VAL;                          \
} G_STMT_END

#define YY_DECL int yylex(YYSTYPE *yylval, YYLTYPE *yylloc)

#define YY_USER_INIT line = col = 1;

#define YY_USER_ACTION                 \
  yylloc->first_line = line;           \
  yylloc->last_line = line;            \
  yylloc->first_column = col;          \
  col += yyleng;                       \
  yylloc->last_column = col;

%}

/* Exclusive state for parsing comments. */
%x STRING COMMENT_C COMMENT_CPP

D    [0-9]
WS   [ \t\n]
ID   [A-Za-z][A-Za-z0-9_]*

%%

"=="  return EQ;
"!="  return NEQ;
"<="  return LEQ;
">="  return GEQ;
"+="  return ADDEQ;
"-="  return SUBEQ;
"*="  return MULEQ;
"/="  return DIVEQ;
"%="  return MODEQ;
"^="  return POWEQ;
"++"  return INC;
"--"  return DEC;
"&&"  return AND;
"||"  return OR;
"!"   return NOT;

E  VALUE_OF(exp(1));

PI VALUE_OF(M_PI);

 /*************************************************************************/
 /**************************** Number parsing *****************************/
 /*************************************************************************/

0x[0-9A-Fa-f]+ {
  long n;
  sscanf(yytext, "%lx", &n);
  VALUE_OF(n);
}

({D}+|{D}+"."{D}*|"."{D}+)([eE][+-]?{D}+)?  {
  yylval->val.type = V_DOUBLE;
  sscanf(yytext, "%lf", &yylval->val.val.d);
  return VAL;
}

 /*************************************************************************/
 /****************************** ID parsing *******************************/
 /*************************************************************************/

{ID}/{WS}*"(" {
  /* '/' causes {WS}*'(' to be returned to the input before processing */
  if ((yylval->func = builtin_lookup(yytext)) == NULL)
    goto nonfunction;
  return FUNC;
}

{ID} { 
  int restoken;
nonfunction:
  if ((restoken = resword_lookup(yytext)) >= 0) /* Look for reserved words. */
     return restoken;

  yylval->name = expr_alloc(char, yyleng + 1);
  strcpy(yylval->name, yytext);
  return ATOM;
}

[ \t]+      /* eat up whitespace */
\n          yynewline();

 /*************************************************************************/
 /**************************** String parsing *****************************/
 /*************************************************************************/

\"          {
  BEGIN(STRING);
  parsed_str = g_string_sized_new(DEFSTRLEN);
  string_start = *yylloc;
}

<STRING>\n  yynewline(); g_string_append_c(parsed_str, '\n');

<STRING>\\. {
  char c;
  switch (yytext[1]) {
  case 'n': c = '\n'; break;
  case 't': c = '\t'; break;
  default: c = yytext[1];
  }
  g_string_append_c(parsed_str, c); /* Escaped chars */
}

<STRING>[^\n\\\"]* g_string_append(parsed_str, yytext); /* All other */

<STRING>\"  {
  BEGIN(INITIAL);
  yylval->val.type = V_STRING;
  yylval->val.val.s = expr_alloc(char, strlen(parsed_str->str)+1);
  strcpy(yylval->val.val.s, parsed_str->str);
  g_string_free(parsed_str, TRUE); /* TRUE: Free parsed_str->str also */
  yylloc->first_column = string_start.first_column;
  yylloc->first_line = string_start.first_line;
  return VAL;
}

<STRING><<EOF>> {
  message("Error", "Unterminated string!");
  BEGIN(INITIAL);
  g_string_free(parsed_str, TRUE); /* TRUE: Free parsed_str->str also */
  yyterminate();
}

 /*************************************************************************/
 /**************************** Comment parsing ****************************/
 /*************************************************************************/

"/*"                           BEGIN(COMMENT_C);

<COMMENT_C>"*/"                BEGIN(INITIAL);

"//"                           BEGIN(COMMENT_CPP);

<COMMENT_CPP>\n                BEGIN(INITIAL); yynewline();

<COMMENT_C,COMMENT_CPP><<EOF>> {
  if (YYSTATE == COMMENT_C) message("Warning", "Unterminated comment!");
  BEGIN(INITIAL);
  yyterminate();
}

<COMMENT_C>\n                  yynewline();

<COMMENT_C,COMMENT_CPP>.  /* ignore every character in comment */

 /*************************************************************************/

. return yytext[0]; /* return all other chars */

%%

void yynewline()
{
  line++;
  col = 1;
}

int resword_lookup(char *word)
{
  int i = 0;
  for (i = 0; reswords[i].name != NULL; i++)
    if (strcmp(word, reswords[i].name) == 0)
      return reswords[i].tokenval;

  return -1;
  yy_flex_realloc(NULL, 0); /* To avoid warning about being unused. */
}

void yy_delete_default_buffer()
{
  yy_delete_buffer(YY_CURRENT_BUFFER);
}

void yy_scan_from_file(FILE *f)
{
  line = col = 1;
  yyrestart(f);
}

void yy_scan_from_string(char *str)
{
  line = col = 1;
  yy_scan_string(str);
}
