/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999  Kevin P. Lawton
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifndef __FREEMWARE_H__
#define __FREEMWARE_H__

#include "config.h"


/*
 * the eflags field looks like this:
 * bit:  0  1 2  3 4  5 6  7  8  9  A  B  C/D  E  F 10 11 12 13  14  15 16
 * flg:  CF 1 PF 0 AF 0 ZF SF TF IF DF OF IOPL NT 0 RF VM AC VIF VIP ID 0
 *
 * The following flags are mirrored between monitor and guest (harmless):
 * CF PF AF ZF SF DF OF (AC ID?)
 *
 * The following flags have unrelated values in guest and monitor:
 * TF (RF?)
 *
 * The following flags always have default values, but are emulated in guest:
 * IF IOPL VIF VIP (NT?)
 *
 * The following flags are really weird:
 * (VM?)
 *
 * The 'real' flags (as they are when the guest code is running) consist
 * of the 'default value' flags (FLG_REAL_BASE), together with the
 * 'mirrored' flags (get from context->eflags, masked with FLG_SAFE_MASK).
 * The 'unrelated' flags come from either nexus->guest_flags or
 * nexus->mon_flags, and need special handling.  We use the macro FLG_BUILD
 * to constuct the 'real' flags, like this:
 *
 *     context->eflags = FLG_BUILD(context->eflags,
 *                                 vm_nexus()->guest_eflags,
 *                                 vm_nexus()->mon_eflags);
 *
 * For now, we ignore the guest flags marked with (?).
 *
 * Other macros defined here:  FLG_USER() calculates the virtual flags as
 * experienced by the guest code;  FLG_TRANS() can be used to copy a flag
 * in one flags field to a different flag in the other;  FLG_GET_IOPL()
 * gets the IOPL from the flags.
 */

#define FLG_CF    (1<<0)
#define FLG_PF    (1<<2)
#define FLG_AF    (1<<4)
#define FLG_ZF    (1<<6)
#define FLG_SF    (1<<7)
#define FLG_TF    (1<<8)
#define FLG_IF    (1<<9)
#define FLG_DF    (1<<10)
#define FLG_OF    (1<<11)
#define FLG_IOPL  (3<<12)
#define FLG_NT    (1<<14)
#define FLG_RF    (1<<16)
#define FLG_VM    (1<<17)
#define FLG_AC    (1<<18)
#define FLG_VIF   (1<<19)
#define FLG_VIP   (1<<20)
#define FLG_ID    (1<<21)

#define FLG_REAL_BASE (0x02 | FLG_IF)
#define FLG_SAFE_MASK (FLG_CF | FLG_PF  | FLG_AF | FLG_ZF | FLG_SF | FLG_DF | FLG_OF)
#define FLG_ASYN_MASK (FLG_TF)
#define FLG_EMUL_MASK (FLG_IF | FLG_IOPL | FLG_VIF | FLG_VIP)

#define FLG_ASYN_MON_MASK (FLG_TF)
#define FLG_ASYN_GST_MASK (FLG_TF)

#define FLG_BUILD(src,gst,mon)         \
    ({ Bit32u dst;                     \
       dst  = FLG_REAL_BASE;           \
       dst |= src & FLG_SAFE_MASK;     \
       dst |= mon & FLG_ASYN_MON_MASK; \
       dst |= gst & FLG_ASYN_GST_MASK; \
       dst; })

#define FLG_USER(src,gst)  ((src & FLG_SAFE_MASK) | gst)

#define FLG_GET_IOPL(src)  ((src >> 12) & 0x3)

#define FLG_TRANS(src,sflg,dst,dflg) \
    (( src & sflg ) ? (dst |= dflg) : (dst &= ~dflg))


/*
 * maximum physical memory which can be used for
 * the VM environment.
 */

#define FMW_MAX_PHY_MEGS 8


/*
 *  ioctl() names
 */

#define FMWALLOCVPHYS   0x6b02
#define FMWALLOCINT     0x6b03
#define FMWRELEASEINT   0x6b04
#define FMWRESET        0x6b05
#define FMWRUNGUEST     0x6b06
#define FMWTEARDOWN     0x6b07


/* 
 * Guest context structure used as argument to FMWRUNGUEST
 */

typedef struct 
{
  Bit32u  gs;
  Bit32u  fs;
  Bit32u  ds;
  Bit32u  es;

  Bit32u  edi;
  Bit32u  esi;
  Bit32u  ebp;
  Bit32u  dummy_esp;
  Bit32u  ebx;
  Bit32u  edx;
  Bit32u  ecx;
  Bit32u  eax;

  Bit32u  event_info; // low byte is vector, 2nd byte is event code
  Bit32u  error;      // pushed by int or dummied

  Bit32u  eip;
  Bit32u  cs;
  Bit32u  eflags;
  Bit32u  esp;
  Bit32u  ss;
} guest_context_t;

#define RET_BECAUSE_REDIR     10
#define RET_BECAUSE_EMERR     11
#define RET_BECAUSE_MON_ERROR 12
#define RET_BECAUSE_REMAP     13
#define RET_BECAUSE_USEREMU   14

#define MON_ACTION_RAISE_INT  0x80
#define MON_ACTION_SET_VIP    0x81

#define EMU_INSTR_OUT_8          1
#define EMU_INSTR_OUT_16         2
#define EMU_INSTR_OUT_32         3
#define EMU_INSTR_IN_8           4
#define EMU_INSTR_IN_16          5
#define EMU_INSTR_IN_32          6

#define EMU_INSTR_OUTS_8         7
#define EMU_INSTR_OUTS_16        8
#define EMU_INSTR_OUTS_32        9
#define EMU_INSTR_INS_8         10
#define EMU_INSTR_INS_16        11
#define EMU_INSTR_INS_32        12

#define EMU_INSTR_REP_OUTS_8    13
#define EMU_INSTR_REP_OUTS_16   14
#define EMU_INSTR_REP_OUTS_32   15
#define EMU_INSTR_REP_INS_8     16
#define EMU_INSTR_REP_INS_16    17
#define EMU_INSTR_REP_INS_32    18

#define EMU_INSTR_HLT           19

#define EMU_INSTR_INT           20
#define EMU_INSTR_STI           21

#endif  // #ifndef __FREEMWARE_H__
