/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999 The FreeMWare Team
 *
 *  emulation.c:  This file contains functions to emulate IA32 instructions
 *                
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include "freemware.h"
#include "monitor.h"

/*
 * Prototypes for instructions faulting in ring 3
 */
int emulate_hlt(guest_context_t *context);
int emulate_cli(guest_context_t *context);
int emulate_sti(guest_context_t *context);
int emulate_clts(void);
int emulate_rdmsr(void);
int emulate_wrmsr(void);
int emulate_invd(void);
int emulate_wbinvd(void);
int emulate_invlpg(void);
int emulate_lgdt(guest_context_t *context, Bit32u addr);
int emulate_lidt(Bit32u addr);
int emulate_lldt(Bit16u selector);
int emulate_ltr(Bit16u selector);
int emulate_lmsw(void);
int emulate_mov_from_cr(void);
int emulate_mov_to_cr(void);
int emulate_mov_from_dr(void);
int emulate_mov_to_dr(void);
int emulate_in(guest_context_t *context, int op_size, int port);
int emulate_out(guest_context_t *context, int op_size, int port);
int emulate_ins(guest_context_t *context, int op_size, int address_size, int rep);
int emulate_outs(guest_context_t *context, int op_size, int address_size, int rep);
int emulate_load_ds(guest_context_t *context, Bit16u selector);
int emulate_load_es(guest_context_t *context, Bit16u selector);
int emulate_load_fs(guest_context_t *context, Bit16u selector);
int emulate_load_gs(guest_context_t *context, Bit16u selector);
int emulate_load_ss(guest_context_t *context, Bit16u selector);
int emulate_int(guest_context_t *context, int vector, Bit32u eip);
int emulate_iret(guest_context_t *context);


/*
 * Various utility routines to access guest memory
 *
 * FIXME -- all of these don't take segmentation or paging into account!
 */

static inline Bit8u read_guest_byte(Bit32u *addr)
{
    Bit8u *ptr = (Bit8u *)(*addr - vm_nexus()->mon_base);
    *addr += 1;
    return *ptr;
}
static inline Bit16u read_guest_word(Bit32u *addr)
{
    Bit16u *ptr = (Bit16u *)(*addr - vm_nexus()->mon_base);
    *addr += 2;
    return *ptr;
}
static inline Bit32u read_guest_dword(Bit32u *addr)
{
    Bit32u *ptr = (Bit32u *)(*addr - vm_nexus()->mon_base);
    *addr += 4;
    return *ptr;
}

static inline void write_guest_byte(Bit32u *addr, Bit8u val)
{
    Bit8u *ptr = (Bit8u *)(*addr - vm_nexus()->mon_base);
    *addr += 1;
    *ptr = val;
}
static inline void write_guest_word(Bit32u *addr, Bit16u val)
{
    Bit16u *ptr = (Bit16u *)(*addr - vm_nexus()->mon_base);
    *addr += 2;
    *ptr = val;
}
static inline void write_guest_dword(Bit32u *addr, Bit32u val)
{
    Bit32u *ptr = (Bit32u *)(*addr - vm_nexus()->mon_base);
    *addr += 4;
    *ptr = val;
}

static Bit32u
guest_stack_pop(Bit32u *esp, int operand_size)
{
    switch (operand_size)
    {
    case 16: return read_guest_word(esp);
    case 32: return read_guest_dword(esp);
    }
    return 0;
}

static void
guest_stack_push(Bit32u *esp, int operand_size, Bit32u val)
{
    switch (operand_size)
    {
    case 16: *esp -= 2; *(Bit16u *)(*esp - vm_nexus()->mon_base) = val; break;
    case 32: *esp -= 4; *(Bit32u *)(*esp - vm_nexus()->mon_base) = val; break;
    }
}

static inline void
clear_descriptor(descriptor_t *descr)
{
    ((Bit32u *)descr)[0] = 0;
    ((Bit32u *)descr)[1] = 0;
}

static inline void
clear_gate(gate_t *gate)
{
    ((Bit32u *)gate)[0] = 0;
    ((Bit32u *)gate)[1] = 0;
}

static void 
read_guest_descriptor(descriptor_t *descr, Bit16u selector)
{
    if ( selector & 4 )
    {
        /* LDT selector -- not supported for now */

        clear_descriptor(descr);
    }
    else
    {
        /* GDT selector */

        Bit32u addr = selector & ~3;
        if (addr+7 > vm_nexus()->guest_gdt_info.limit) 
        {
            clear_descriptor(descr);
        }
        else
        {
            addr += vm_nexus()->guest_gdt_info.base;
            ((Bit32u *)descr)[0] = read_guest_dword(&addr);
            ((Bit32u *)descr)[1] = read_guest_dword(&addr);
        }
    }
}

static void 
write_guest_descriptor(descriptor_t *descr, Bit16u selector)
{
    if ( selector & 4 )
    {
        /* LDT selector -- not supported for now */
    }
    else
    {
        /* GDT selector */

        Bit32u addr = selector & ~3;
        if (addr+7 > vm_nexus()->guest_gdt_info.limit) 
            ;
        else
        {
            addr += vm_nexus()->guest_gdt_info.base;
            write_guest_dword(&addr, ((Bit32u *)descr)[0]);
            write_guest_dword(&addr, ((Bit32u *)descr)[1]);
        }
    }
}

static void 
read_guest_gate(gate_t *gate, int vector)
{
    Bit32u addr = 8*vector;
    if (   vector < 0 || vector >= 256 
        || addr+7 > vm_nexus()->guest_idt_info.limit) 
    {
        clear_gate(gate);
    }
    else
    {
        addr += vm_nexus()->guest_idt_info.base;
        ((Bit32u *)gate)[0] = read_guest_dword(&addr);
        ((Bit32u *)gate)[1] = read_guest_dword(&addr);
    }
}


/* 
 * Hack to manage 'activated' code segment
 */

void
codeseg_activate(Bit16u selector, int active)
{
    if ( selector & 4 )
    {
        /* LDT selector -- not supported for now */
    }
    else
    {
        descriptor_t *monitor_gdt = (descriptor_t *)vm_nexus()->gdt_offset;
        descriptor_t *descr = monitor_gdt + (selector >> 3);

        if ( active )
            descr->type |= 0x08;
        else
            descr->type &= ~0x08;
    }
}


/* 
 * Raise a hardware or software interrupt
 */
int
raise_interrupt(guest_context_t *context, int vector, int soft)
{
    Bit32u new_eip, new_esp;
    Bit16u new_cs, new_ss;

    gate_t gate;
    descriptor_t code_descr;
    descriptor_t stack_descr;

    int cpl = vm_nexus()->guest_cpl;

    /* Retrieve interrupt gate */
    read_guest_gate(&gate, vector);
    if (!gate.p) return 0;
    if (gate.type != D_INT && gate.type != D_TRAP) return 0;
    if (soft && gate.dpl < cpl) return 0;
    if (!gate.selector) return 0;

    new_cs     = gate.selector;
    new_eip    = (gate.offset_high << 16) | gate.offset_low;

    /* Build a flags image for the stack */
    context->eflags = FLG_USER(context->eflags,vm_nexus()->guest_eflags);

    /* Update the guest flags */
    if (gate.type == D_INT)
    {
        vm_nexus()->guest_eflags &= ~FLG_IF;
    }

    /* Retrieve code segment descriptor */
    read_guest_descriptor(&code_descr, new_cs);
    if (!code_descr.p) return 0;
    if ((code_descr.type & 0x18) != D_CODE) return 0;
    if (code_descr.dpl > cpl) return 0;
    if (new_eip > Limit(code_descr)) return 0;

    if (!(code_descr.type & D_CONFORM) && code_descr.dpl < cpl)
    {
        /* Inter-privilege-level interrupt */
        descriptor_t *tss = &vm_nexus()->guest_tss_descr;
        Bit32u tss_limit = (tss->limit_high << 16) | tss->limit_low;
        Bit32u tss_base  = (tss->base_high << 24) | (tss->base_med << 16) | tss->base_low;

        Bit32u stack_addr = code_descr.dpl * 8 + 4;
        if (stack_addr+7 > tss_limit) return 0;
        stack_addr += tss_base;
        new_esp = read_guest_dword(&stack_addr);
        new_ss  = read_guest_dword(&stack_addr);

        read_guest_descriptor(&stack_descr, new_ss);
        if (!stack_descr.p) return 0;
        if ((stack_descr.type & 0x18) != D_DATA) return 0;
        if (!(stack_descr.type & D_WRITE)) return 0;
        if (stack_descr.dpl != code_descr.dpl) return 0;

        /* FIXME: stack limits check */

        /* Push old stack info */
        guest_stack_push(&new_esp, 32, (context->ss & ~3) | cpl);
        guest_stack_push(&new_esp, 32, context->esp);

        vm_nexus()->guest_cpl = code_descr.dpl;
    }
    else
    {
        /* Intra-privilege-level interrupt */

        new_ss  = context->ss;
        new_esp = context->esp;

        /* FIXME: stack limits check */
    }

    /* Push return info */
    guest_stack_push(&new_esp, 32, context->eflags);
    guest_stack_push(&new_esp, 32, (context->cs & ~3) | cpl);
    guest_stack_push(&new_esp, 32, context->eip);

    codeseg_activate( context->cs, 0 );
    codeseg_activate( new_cs, 1 );

    context->cs     = new_cs | 3;
    context->eip    = new_eip;
    context->ss     = new_ss | 3;
    context->esp    = new_esp;

    /* FIXME: other flags are also affected */
    context->eflags = FLG_BUILD(context->eflags,
                                vm_nexus()->guest_eflags,
                                vm_nexus()->mon_eflags);

    return 1;
}


/*
 * Operand decoding/evaluation routines
 */

#define FieldMod(modrm)     ( (modrm) >> 6       )
#define FieldReg(modrm)     (((modrm) >> 3) & 0x7)
#define FieldRM(modrm)      ( (modrm)       & 0x7)

#define FieldSIBScale(sib)  ( (sib) >> 6       )
#define FieldSIBIndex(sib)  (((sib) >> 3) & 0x7)
#define FieldSIBBase(sib)   ( (sib)       & 0x7)

static Bit32u
decode_register(guest_context_t *context, int operand_size, int reg)
{
    switch ( operand_size )
    {
    case 8:
        switch ( reg )
        {
        case 0: return context->eax & 0xff;
        case 1: return context->ecx & 0xff;
        case 2: return context->edx & 0xff;
        case 3: return context->ebx & 0xff;
        case 4: return (context->eax >> 8) & 0xff;
        case 5: return (context->ecx >> 8) & 0xff;
        case 6: return (context->edx >> 8) & 0xff;
        case 7: return (context->ebx >> 8) & 0xff;
        }
        break;

    case 16:
        switch ( reg )
        {
        case 0: return context->eax & 0xffff;
        case 1: return context->ecx & 0xffff;
        case 2: return context->edx & 0xffff;
        case 3: return context->ebx & 0xffff;
        case 4: return context->esp & 0xffff;
        case 5: return context->ebp & 0xffff;
        case 6: return context->esi & 0xffff;
        case 7: return context->edi & 0xffff;
        }
        break;

    case 32:
        switch ( reg )
        {
        case 0: return context->eax;
        case 1: return context->ecx;
        case 2: return context->edx;
        case 3: return context->ebx;
        case 4: return context->esp;
        case 5: return context->ebp;
        case 6: return context->esi;
        case 7: return context->edi;
        }
        break;
    }
    return 0;
}

static Bit32u
decode_memory_operand(guest_context_t *context, int operand_size, Bit32u addr)
{
    switch ( operand_size )
    {
    case 8:   return read_guest_byte(&addr);
    case 16:  return read_guest_word(&addr);
    case 32:  return read_guest_dword(&addr);
    }
    return 0;
}

static Bit32u
decode_effective_address(guest_context_t *context, int address_size, 
                         Bit8u ModRegRM, Bit32u *addr)
{
    Bit32u mod = FieldMod(ModRegRM);
    if ( mod == 3 ) return 0;   /* not an effective address - can't happen */

    if ( address_size == 16 )
    { 
        Bit32u base = 0, displ = 0;

        /* Decode Base + Index */
        switch ( FieldRM(ModRegRM) )
        {
        case 0: base = context->ebx + context->esi; break;
        case 1: base = context->ebx + context->edi; break;
        case 2: base = context->ebp + context->esi; break;
        case 3: base = context->ebp + context->edi; break;
        case 4: base = context->esi; break;
        case 5: base = context->edi; break;
        case 6: base = mod? context->ebp : read_guest_word(addr); break;
        case 7: base = context->ebx;
        }

        /* Get Displacement */
        switch ( mod )
        {
        case 1:  displ = (signed char)read_guest_byte(addr); break;
        case 2:  displ = read_guest_word(addr); break;
        }

        return (base + displ) & 0xffff;
    }
    else
    { 
        Bit32u base, index = 0, displ = 0;

        /* Get SIB byte if present */
        if ( (base = FieldRM(ModRegRM)) == 4 )
        {
            Bit8u sib = read_guest_byte(addr);
            base = FieldSIBBase(sib);

            switch ( FieldSIBIndex(sib) )
            {
            case 0: index = context->eax; break;
            case 1: index = context->ecx; break;
            case 2: index = context->edx; break;
            case 3: index = context->ebx; break;
            case 4: index = 0;            break;
            case 5: index = context->ebp; break;
            case 6: index = context->esi; break;
            case 7: index = context->edi; break;
            }

            index <<= FieldSIBScale(sib);
        }

        /* Decode Base */
        switch ( base )
        {
        case 0: base = context->eax; break;
        case 1: base = context->ecx; break;
        case 2: base = context->edx; break;
        case 3: base = context->ebx; break;
        case 4: base = context->esp; break;
        case 5: base = mod? context->ebp : read_guest_dword(addr); break;
        case 6: base = context->esi; break;
        case 7: base = context->edi; break;
        }

        /* Get Displacement */
        switch ( mod )
        {
        case 1:  displ = (signed char)read_guest_byte(addr); break;
        case 2:  displ = read_guest_dword(addr); break;
        }

        return base + index + displ;
    }
}

static Bit32u
decode_rm_operand(guest_context_t *context, int operand_size, int address_size,
                  Bit8u ModRegRM, Bit32u *addr)
{
    Bit32u ptr, val;

    if ( FieldMod(ModRegRM) == 3 )
        val = decode_register( context, operand_size, FieldRM(ModRegRM) );
    else
    {
        ptr = decode_effective_address( context, address_size, ModRegRM, addr );
        val = decode_memory_operand( context, operand_size, ptr );
    }

    return val;
}


int
emulate(guest_context_t *context)
/*
 *  emulation(): find out faulting instruction, jump to emulation function
 *
 *  INPUTS:
 *     none
 *
 *  OUTPUTS:
 *     0 if no success
 *     1 if success
 *     2 if success, but monitor needs to be remapped
 */ 
{
    int action = 0;
    int use32 = 1;       /* FIXME: is CS 16-bit or 32-bit */
    int address_size;
    int operand_size;
    int seg_override = 0;
    int rep = 0;
    Bit8u c;
    Bit32u addr, val;
    Bit32u eip = context->eip;
    Bit32u esp = context->esp;
  
    address_size = use32? 32 : 16;
    operand_size = use32? 32 : 16;

    /*
     * Decode Intel instruction
     */

 prefix:
    c = read_guest_byte(&eip);
    switch (c) 
    {
    case 0x26: /* ES prefix */
        seg_override = context->es;
        goto prefix;
    case 0x2e: /* CS prefix */
        seg_override = context->cs;
        goto prefix;
    case 0x36: /* SS prefix */
        seg_override = context->ss;
        goto prefix;
    case 0x3e: /* DS prefix */
        seg_override = context->ds;
        goto prefix;
    case 0x64: /* FS prefix */
        seg_override = context->fs;
        goto prefix;
    case 0x65: /* GS prefix */
        seg_override = context->gs;
        goto prefix;
    case 0x66: /* Operand size prefix */
        operand_size = use32? 16 : 32;
        goto prefix;
    case 0x67: /* Address size prefix */
        address_size = use32? 16 : 32;
        goto prefix;
    case 0xf3: /* rep prefix */
        rep = 1;
        goto prefix;

    case 0x07: /* pop ES */
        val = guest_stack_pop(&esp, operand_size);
        action = emulate_load_es(context, val);
        break;
    case 0x17: /* pop SS */
        val = guest_stack_pop(&esp, operand_size);
        action = emulate_load_ss(context, val);
        break;
    case 0x1f: /* pop DS */
        val = guest_stack_pop(&esp, operand_size);
        action = emulate_load_ds(context, val);
        break;

    case 0x6c: /* insb == ins dx, m8 */
      action = emulate_ins(context, 8, address_size, rep);
      break;

    case 0x6d: /* insw || insd */
      action = emulate_ins(context, operand_size, address_size, rep);
      break;

    case 0x6e: /* outsb == outs dx, m8 */
      action = emulate_outs(context, 8, address_size, rep);
      break;

    case 0x6f: /* outsw || outsd */
      action = emulate_outs(context, operand_size, address_size, rep);
      break;

    case 0x8e: /* mov Sw, Ew */
        c = read_guest_byte(&eip);  /* RegModRM */
        val = decode_rm_operand(context, 16, address_size, c, &eip);
        switch ( FieldReg(c) )
        {
        case 0: action = emulate_load_es(context, val); break;
        case 1: /* mov CS, ... ??? */; break;
        case 2: action = emulate_load_ss(context, val); break;
        case 3: action = emulate_load_ds(context, val); break;
        case 4: action = emulate_load_fs(context, val); break;
        case 5: action = emulate_load_gs(context, val); break;
        }
        break;

    case 0xcc: /* int3 */
        action = emulate_int(context, 3, eip);
        break;

    case 0xcd: /* int Ib */
        c = read_guest_byte(&eip);
        action = emulate_int(context, c, eip);
        break;

    case 0xcf: /* iret */
        action = emulate_iret(context);
        break;

    case 0xe4: /* in imm8,al */
        c = read_guest_byte(&eip);
        action = emulate_in(context, 8, c);
        break;

    case 0xe5: /* in imm8,[e]ax */
        c = read_guest_byte(&eip);
        action = emulate_in(context, operand_size, c);
        break;

    case 0xe6: /* out imm8,al */
        c = read_guest_byte(&eip);
        action = emulate_out(context, 8, c);
        break;

    case 0xe7: /* out imm8,[e]ax */
        c = read_guest_byte(&eip);
        action = emulate_out(context, operand_size, c);
        break;

    case 0xec: /* in al,dx */
        action = emulate_in(context, 8, context->edx & 0xffff);
        break;

    case 0xed: /* in [e]ax,dx */
        action = emulate_in(context, operand_size, context->edx & 0xffff);
        break;

    case 0xee: /* out dx,al */
        action = emulate_out(context, 8, context->edx & 0xffff);
        break;

    case 0xef: /* out dx,[e]ax */
        action = emulate_out(context, operand_size, context->edx & 0xffff);
        break;

    case 0xf4: /* hlt */
        action = emulate_hlt(context);
        break;

    case 0xfa: /* cli */
        action = emulate_cli(context);
        break;

    case 0xfb: /* sti */
        action = emulate_sti(context);
        break;

    case 0x0f:
        /* there are some instructions beginning with 0x0f
         * we have a look on the second opcode byte
         */
        c = read_guest_byte(&eip);
        switch (c) 
        {
        case 0x00: /* Group 6 */
            c = read_guest_byte(&eip);
            switch (FieldReg(c))
            {
            case 2:  /* lldt */
                val = decode_rm_operand(context, 16, address_size, c, &eip);
                action = emulate_lldt(val);
                break;

            case 3:  /* ltr */
                val = decode_rm_operand(context, 16, address_size, c, &eip);
                action = emulate_ltr(val);
                break;
            }
            break;

        case 0x01: /* Group 7 */
            c = read_guest_byte(&eip);
            switch (FieldReg(c))
            {
            case 2:  /* lgdt */
                addr = decode_effective_address(context, address_size, c, &eip);
                action = emulate_lgdt(context, addr);
                break;

            case 3:  /* lidt */
                addr = decode_effective_address(context, address_size, c, &eip);
                action = emulate_lidt(addr);
                break;
            }
            break;

        case 0x06: /* clts */
            action = emulate_clts();
            break;

	case 0x08: /* invd */
	  action = emulate_invd();
	  break;
	    
	case 0x09: /* wbinvd */
	  action = emulate_wbinvd();
	  break;

        case 0xa1: /* pop FS */
            val = guest_stack_pop(&esp, operand_size);
            action = emulate_load_fs(context, val);
            break;
        case 0xa9: /* pop GS */
            val = guest_stack_pop(&esp, operand_size);
            action = emulate_load_gs(context, val);
            break;
        }
        break;
    }


    /* Check next action */
    switch (action)
    {
    default:
    case 0:   /* Failure */
        return 0;

    case 1:   /* Retry instruction */
        return 1;

    case 2:   /* Skip instruction */
        context->eip = eip;
        context->esp = esp;
        return 1;

    case 3:   /* Remap monitor and skip instruction */
        context->eip = eip;
        context->esp = esp;
        return 2;

    case 4:   /* Try to emulate in user space */
        context->eip = eip;
        context->esp = esp;
        return 3;
    }
}


int
emulate_cli(guest_context_t *context)
     /*
      * Emulate cli instruction
      */
{
    int cpl = vm_nexus()->guest_cpl;

    if (cpl > FLG_GET_IOPL(vm_nexus()->guest_eflags))
    {
        return 0;
    }

    vm_nexus()->guest_eflags &= ~FLG_IF;
    context->eflags = FLG_BUILD(context->eflags,
                                vm_nexus()->guest_eflags,
                                vm_nexus()->mon_eflags);

    return 2;
}

int
emulate_sti(guest_context_t *context) 
     /* 
      * Emulate sti instruction
      */
{
    int cpl = vm_nexus()->guest_cpl;

    if (cpl > FLG_GET_IOPL(vm_nexus()->guest_eflags))
    {
        return 0;
    }

    vm_nexus()->guest_eflags |= FLG_IF;

    /* if the VIP flag is set, we need to trap to the user */
    if (vm_nexus()->mon_eflags & FLG_VIP)
    {
        context->event_info = EMU_INSTR_STI | (RET_BECAUSE_USEREMU << 8);

        vm_nexus()->mon_eflags &= ~FLG_VIP;
        context->eflags = FLG_BUILD(context->eflags,
                                    vm_nexus()->guest_eflags,
                                    vm_nexus()->mon_eflags);
        return 4;
    }
    else
    {
        context->eflags = FLG_BUILD(context->eflags,
                                    vm_nexus()->guest_eflags,
                                    vm_nexus()->mon_eflags);
        return 2;
    }
}

int
emulate_clts(void) 
     /* 
      * Emulate clts instruction
      */
{
    vm_nexus()->debug_msg.msg_code = EMU_CLTS_MSG;
    return 0;
}

int
emulate_lgdt(guest_context_t *context, Bit32u addr) 
     /* 
      * Emulate lgdt instruction
      */
{
    descriptor_t *monitor_gdt = (descriptor_t *)vm_nexus()->gdt_offset;
    Bit16u limit = read_guest_word(&addr);
    Bit32u base  = read_guest_dword(&addr);

    int i, nr_descriptors;

    vm_nexus()->guest_gdt_info.limit = limit;
    vm_nexus()->guest_gdt_info.base  = base;

    if ( limit >= MON_GDT_PAGES*PAGESIZE ) return 0;
    nr_descriptors = (limit+1) / 8;

    for ( i = 1; i < nr_descriptors; i++ )
    {
        descriptor_t descr;
        read_guest_descriptor(&descr, i << 3);
        
        if ( descr.type & 0x10 )
        {
            /* Code or data segment descriptor */

            /* For now, we let everything go through unchanged, 
               only the DPL is forced to 3 */
            descr.dpl = D_DPL3;

            /* Change code segments to data segments; this is to make all
               inter-segment transfers, especially iret, fault. */
            descr.type &= ~0x08;
        }
        else
        {
            /* System segment descriptor */
       
            switch ( descr.type )
            {
            case D_TSS: case D_TSS | 2:  /* available/busy TSS */
                /* Force the DPL to zero to disallow task switches */
                descr.dpl = D_DPL0;
                break;

            /* FIXME: Don't allow LDTs or gates for now ... */
            case D_LDT:
            case D_CALL:
            case D_INT: case D_TRAP:
            default:
                clear_descriptor(&descr);
                break;
            }
        }

        monitor_gdt[i] = descr;
    }
    for ( ; i < MON_GDT_PAGES*PAGESIZE / 8; i++ )
        clear_descriptor(monitor_gdt + i);

    /* Mark current CS active */
    codeseg_activate(context->cs, 1);

    /* We have just clobbered the monitor selectors.  Zero them out in the
       nexus data structure to reflect this.  See unmap_monitor() ... */
    vm_nexus()->mon_jmp_info.selector = 0;
    vm_nexus()->mon_stack_info.selector = 0;
    vm_nexus()->mon_tss_sel = 0;

    /* Return to the host to re-map the monitor */
    return 3;
}

int
emulate_lidt(Bit32u addr) 
     /* 
      * Emulate lidt instruction
      */
{
    Bit16u limit = read_guest_word(&addr);
    Bit32u base  = read_guest_dword(&addr);

    vm_nexus()->guest_idt_info.limit = limit;
    vm_nexus()->guest_idt_info.base  = base;

    return 2;
}

int
emulate_lldt(Bit16u selector) 
     /* 
      * Emulate lldt instruction
      */
{
    descriptor_t descr;

    if (selector & 4) return 0;    /* No LDT allowed */

    if (selector)
    {
        read_guest_descriptor(&descr, selector);
        if (!descr.p) return 0;
        if (descr.type != D_LDT) return 0;
    }
    else
    {
        clear_descriptor(&descr);
    }
    
    vm_nexus()->guest_ldt_sel = selector;
    vm_nexus()->guest_ldt_descr = descr;

    return 2;
}

int
emulate_ltr(Bit16u selector) 
     /* 
      * Emulate ltr instruction
      */
{
    descriptor_t descr;

    if (selector & 4) return 0;    /* No LDT allowed */

    read_guest_descriptor(&descr, selector);
    if (!descr.p) return 0;
    if (descr.type != D_TSS) return 0;
    
    descr.type |= 2;  /* set busy flag */

    write_guest_descriptor(&descr, selector);

    vm_nexus()->guest_tss_sel = selector;
    vm_nexus()->guest_tss_descr = descr;

    return 2;
}

int
emulate_load_ds(guest_context_t *context, Bit16u selector)
     /* 
      * Emulate instruction that loads DS register
      */
{
    vm_nexus()->debug_msg.msg_code = EMU_LOAD_SEGREG_MSG;
    vm_nexus()->debug_msg.para1 = 'D';
    vm_nexus()->debug_msg.para2 = selector;

    return 0;
}

int
emulate_load_es(guest_context_t *context, Bit16u selector)
     /* 
      * Emulate instruction that loads ES register
      */
{
    vm_nexus()->debug_msg.msg_code = EMU_LOAD_SEGREG_MSG;
    vm_nexus()->debug_msg.para1 = 'E';
    vm_nexus()->debug_msg.para2 = selector;

    return 0;
}

int
emulate_load_fs(guest_context_t *context, Bit16u selector)
     /* 
      * Emulate instruction that loads FS register
      */
{
    vm_nexus()->debug_msg.msg_code = EMU_LOAD_SEGREG_MSG;
    vm_nexus()->debug_msg.para1 = 'F';
    vm_nexus()->debug_msg.para2 = selector;

    return 0;
}

int
emulate_load_gs(guest_context_t *context, Bit16u selector)
     /* 
      * Emulate instruction that loads GS register
      */
{
    vm_nexus()->debug_msg.msg_code = EMU_LOAD_SEGREG_MSG;
    vm_nexus()->debug_msg.para1 = 'G';
    vm_nexus()->debug_msg.para2 = selector;

    return 0;
}

int
emulate_load_ss(guest_context_t *context, Bit16u selector)
     /* 
      * Emulate instruction that loads SS register
      */
{
    descriptor_t descr;

    if (!selector) return 0;
    if ((selector & 3) != vm_nexus()->guest_cpl) return 0;

    read_guest_descriptor(&descr, selector);
    if (!descr.p) return 0;
    if ((descr.type & 0x18) != D_DATA) return 0;
    if (!(descr.type & D_WRITE)) return 0;
    if (descr.dpl != vm_nexus()->guest_cpl) return 0;

    context->ss = selector | 3;

    return 2;
}

int
emulate_int(guest_context_t *context, int vector, Bit32u eip)
     /* 
      * Emulate INT instruction
      */
{
    Bit32u old_eip;

    if ( BMAP_GET(vm_nexus()->host_fwd_ints, vector) )
    {
        context->event_info = EMU_INSTR_INT | (RET_BECAUSE_USEREMU << 8) | (vector << 16);
        return 4;
    }

    old_eip = context->eip;
    context->eip = eip;

    if ( !raise_interrupt(context, vector, 1) )
    {
        context->eip = old_eip;
        return 0;
    }

    return 1;
}

int
emulate_iret(guest_context_t *context)
     /* 
      * Emulate IRET instruction
      */
{
    Bit32u esp = context->esp;

    Bit32u new_eip, new_esp, new_eflags;
    Bit16u new_cs, new_ss;

    descriptor_t code_descr;
    descriptor_t stack_descr;

    int cpl = vm_nexus()->guest_cpl;

    /* FIXME: stack limits check */

    new_eip    = guest_stack_pop(&esp, 32);
    new_cs     = guest_stack_pop(&esp, 32);
    new_eflags = guest_stack_pop(&esp, 32);

    if ((new_cs & 3) < cpl) return 0;
    if (!new_cs) return 0;
    read_guest_descriptor(&code_descr, new_cs);
    if (!code_descr.p) return 0;
    if ((code_descr.type & 0x18) != D_CODE) return 0;
    if (   (code_descr.type & D_CONFORM)
         ? (code_descr.dpl < cpl)
         : (code_descr.dpl != (new_cs & 3)) )  return 0;
    if (new_eip > Limit(code_descr)) return 0;

    if (cpl > FLG_GET_IOPL(vm_nexus()->guest_eflags))
    {
        new_eflags &= ~FLG_IF;
        new_eflags |= context->eflags & FLG_IF;
    }

    if (cpl > 0)
    {
        new_eflags &= ~FLG_IOPL;
        new_eflags |= context->eflags & FLG_IOPL;
    }

    vm_nexus()->guest_eflags = new_eflags & (FLG_EMUL_MASK | FLG_ASYN_MASK);

    if ((new_cs & 3) > cpl)
    {
        /* Inter-privilege-level return */

        new_esp = guest_stack_pop(&esp, 32);
        new_ss  = guest_stack_pop(&esp, 32);

        if ((new_ss & 3) != (new_cs & 3)) return 0;
        if (!new_ss) return 0;
        read_guest_descriptor(&stack_descr, new_ss);
        if (!stack_descr.p) return 0;
        if ((stack_descr.type & 0x18) != D_DATA) return 0;
        if (!(stack_descr.type & D_WRITE)) return 0;
        if (stack_descr.dpl != (new_cs & 3)) return 0;

        /* FIXME: clear out invalid DS/ES/FS/GS selectors */

        vm_nexus()->guest_cpl = new_cs & 3;
    }
    else
    {
        /* Intra-privilege-level return */

        new_esp = esp;
        new_ss  = context->ss;
    }

    codeseg_activate( context->cs, 0 );
    codeseg_activate( new_cs, 1 );

    context->cs     = new_cs | 3;
    context->eip    = new_eip;
    context->ss     = new_ss | 3;
    context->esp    = new_esp;
    context->eflags = FLG_BUILD(context->eflags,
                                vm_nexus()->guest_eflags,
                                vm_nexus()->mon_eflags);

    return 1;
}

int
emulate_out(guest_context_t *context, int operand_size, int port)
     /*
      * Emulate out instruction
      */
{
    switch ( operand_size )
    {
    case 8:  context->event_info = EMU_INSTR_OUT_8;  break;
    case 16: context->event_info = EMU_INSTR_OUT_16; break;
    case 32: context->event_info = EMU_INSTR_OUT_32; break;
    }

    context->event_info |= (RET_BECAUSE_USEREMU << 8) | (port << 16);
    return 4;
}

int
emulate_in(guest_context_t *context, int operand_size, int port)
     /*
      * Emulate in instruction
      */
{
    switch ( operand_size )
    {
    case 8:  context->event_info = EMU_INSTR_IN_8;  break;
    case 16: context->event_info = EMU_INSTR_IN_16; break;
    case 32: context->event_info = EMU_INSTR_IN_32; break;
    }

    context->event_info |= (RET_BECAUSE_USEREMU << 8) | (port << 16);
    return 4;
}

int
emulate_outs(guest_context_t *context, int operand_size, int address_size, int rep)
     /*
      * Emulate outs instruction
      */
{
    if (rep)
    {
        switch (operand_size)
        {
        case 8:  context->event_info = EMU_INSTR_REP_OUTS_8;  break;
        case 16: context->event_info = EMU_INSTR_REP_OUTS_16; break;
        case 32: context->event_info = EMU_INSTR_REP_OUTS_32; break;
        }
    }
    else
    {
        switch (operand_size)
        {
        case 8:  context->event_info = EMU_INSTR_OUTS_8;  break;
        case 16: context->event_info = EMU_INSTR_OUTS_16; break;
        case 32: context->event_info = EMU_INSTR_OUTS_32; break;
        }
    }
    context->event_info |= (RET_BECAUSE_USEREMU << 8) | (address_size << 16);
    return 4;
}


int
emulate_ins(guest_context_t *context, int operand_size, int address_size, int rep)
     /*
      * Emulate ins instruction
      */
{
    if (rep)
    {
        switch (operand_size)
        {
        case 8:  context->event_info = EMU_INSTR_REP_INS_8;  break;
        case 16: context->event_info = EMU_INSTR_REP_INS_16; break;
        case 32: context->event_info = EMU_INSTR_REP_INS_32; break;
        }
    }
    else
    {
        switch (operand_size)
        {
        case 8:  context->event_info = EMU_INSTR_INS_8;  break;
        case 16: context->event_info = EMU_INSTR_INS_16; break;
        case 32: context->event_info = EMU_INSTR_INS_32; break;
        }
    }
    context->event_info |= (RET_BECAUSE_USEREMU << 8) | (address_size << 16);
    return 4;
}

int
emulate_invd(void) 
{
    /* nothing to do ;) */
    return 2;
}

int
emulate_wbinvd(void) 
{
    /* nothing to do ;) */
    return 2;
}

int
emulate_hlt(guest_context_t *context) 
{
    context->event_info = EMU_INSTR_HLT | (RET_BECAUSE_USEREMU << 8);
    return 4;
}
