/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999  Ulrich Weigand
 *
 *  fault.c:  Monitor-side fault handlers.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "freemware.h"
#include "monitor.h"

extern int emulate_int(guest_context_t *context, int vector, Bit32u eip);

int
handle_fault( guest_context_t *context )
/*
 *  handle_fault():  main monitor-side fault handler
 *
 *  This fault handler is called from the assembly stub
 *  __handle_fault.
 *
 *  INPUTS:
 *     context: guest context at point of exception
 *
 *  RETURN:
 *     1  if fault handled, and guest execution is to be resumed
 *     0  if we need to switch back to the host to handle fault
 */
{
    int action;

    /*
     * If the context does not point to the address we expect,
     * the fault must have happened inside monitor code.  Abort.
     */
    guest_context_t *expected_context = 
          (guest_context_t *)(((char *)vm_nexus()) + PAGESIZE) - 1;

    if ( context != expected_context )
    {
        expected_context->event_info = RET_BECAUSE_MON_ERROR<<8;
        return 0;
    }

    switch ( context->event_info & 0xff )
    {
    /* FIXME */

    case 13:  /* GPF */
        action = emulate( context );
        break;

    default: 
        action = emulate_int( context, context->event_info & 0xff, context->eip );
        if (action > 1) action--;
        break;
    }

    switch ( action )
    {
    default:
    case 0:   /* unable to emulate */
        context->event_info = 13 | (RET_BECAUSE_EMERR << 8);
        return 0;

    case 1:   /* emulation successful, continue */
        return 1;

    case 2:   /* emulation successful, need to remap monitor */
        context->event_info = RET_BECAUSE_REMAP << 8;
        return 0;
    case 3:   /* emulation in user space */
        return 0;
    }
}

void
monitor_action( guest_context_t *context )
/*
 *  monitor_action():  monitor action handler
 *
 *  This handler is called from within the __host2guest
 *  routine, just before calling the guest, if the host-side
 *  code requested an out-of-band action to be taken on monitor
 *  side (e.g. raise a hardware interrupt).
 *
 *  INPUTS:
 *     context: current guest context
 */
{
    switch ( (context->event_info >> 8) & 0xff )
    {
    case MON_ACTION_RAISE_INT:
    {
        int vector = context->event_info & 0xff;
        raise_interrupt( context, vector, 0 );
        break;
    }

    case MON_ACTION_SET_VIP:
        vm_nexus()->mon_eflags |= FLG_VIP;
        break;

    default:
        /* Hmmm. */
        break;
    }
}
