/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999  Kevin P. Lawton
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifndef __MONITOR_H__
#define __MONITOR_H__

#include "descriptor.h"
#include "tss.h"
#include "paging.h"


/*
 * Passing debug messages between host and guest context.
 */

typedef struct 
{
  // message code
  Bit32u msg_code;

  // parameter 1 and 2
  Bit32u para1, para2;
} debug_msg_t;

#define EMU_CLI_MSG 1
#define EMU_STI_MSG 2
#define EMU_CLTS_MSG 3
#define EMU_LGDT_MSG 4
#define EMU_LIDT_MSG 5
#define EMU_LLDT_MSG 6
#define EMU_LTR_MSG 7
#define EMU_LOAD_SEGREG_MSG 8



// Method1: push event info (CPU pushes error code before)
typedef struct 
{
  Bit8u  pushl;    /* Always 0x68 == pushl            */
  Bit32u vector;   /* Interrupt vector number         */
  Bit8u  jmp;      /* Always 0xe9 == jmp              */
  Bit32u reloc;    /* Relative offset of destination  */
} __attribute__ ((packed)) idt_method1_t;

// Method2: push a dummy error first, then event info
typedef struct 
{
  Bit8u  pushla;   /* Always 0x68 == pushl            */
  Bit32u dummy;    /* Dummy error code                */
  Bit8u  pushlb;   /* Always 0x68 == pushl            */
  Bit32u vector;   /* Interrupt vector number         */
  Bit8u  jmp;      /* Always 0xe9 == jmp              */
  Bit32u reloc;    /* Relative offset of destination  */
} __attribute__ ((packed)) idt_method2_t;

typedef union 
{
  idt_method1_t m1;
  idt_method2_t m2;
} idt_stub_t;


// Bitmap handling macros
#define  BMAP_SZ(y)      (y>>3)
#define  BMAP_GET(x,y)   ((((Bit8u *)x)[y>>3]>>(y&7))&1)
#define  BMAP_SET(x,y)   (((Bit8u *)x)[y>>3]|=(1<<(y&7)))
#define  BMAP_CLR(x,y)   (((Bit8u *)x)[y>>3]&=~(1<<(y&7)))

// State flags
#define  STATE_STI_DELAY    0x1
#define  STATE_SINGLE_STEP  0x2

typedef struct 
{
  // The kernel monitor is a state machine.  state_flags contains
  // the state of the monitor.  When state_flags == 0, the nothing
  // special is going on while the guest code is running.  The
  // following flags are currently recognised:
  //
  //   Bit  Function
  //   0    guest did STI, but interrupts are only enabled after
  //        the next instruction
  //   1    user-level monitor requested single-step debugging
  Bit32u         state_flags;

  // These fields are only used by the transition code.
  // They hold all info necessary to switch back to the host.
  gdt_info_t     host_gdt_info;
  gdt_info_t     host_idt_info;
  far_jmp_info_t host_jmp_info;
  far_jmp_info_t host_stack_info;
  Bit16u         host_ldt_sel;
  Bit16u         host_tss_sel;
  Bit32u         host_cr0;
  Bit32u         host_cr2;
  Bit32u         host_cr3;
  Bit32u         host_cr4;

  // These fields are filled by the host-side code, and used 
  // by the transition code.  They contain all info necessary
  // to switch to the monitor/guest address space.
  // This info changes whenever the monitor migrates.
  gdt_info_t     mon_gdt_info;
  gdt_info_t     mon_idt_info;
  far_jmp_info_t mon_jmp_info;
  far_jmp_info_t mon_stack_info;
  Bit16u         mon_ldt_sel;
  Bit16u         mon_tss_sel;
  Bit32u         mon_base;
  Bit32u         mon_cr3;
  Bit32u         mon_eflags;

  // These fields contain info set by the guest
  gdt_info_t     guest_gdt_info;
  gdt_info_t     guest_idt_info;
  Bit16u         guest_ldt_sel;
  Bit16u         guest_tss_sel;
  descriptor_t   guest_ldt_descr;
  descriptor_t   guest_tss_descr;
  int            guest_cpl;
  Bit32u         guest_eflags;

  // These fields contain info set by the host (forwarding info)
  Bit8u          host_fwd_ints[BMAP_SZ(256)];

  // These fields contain info used by the transition code to
  // create the temporary identity mapping.  They never change.
  pageEntry_t    transition_pde;
  pageEntry_t   *transition_pde_p_host;
  pageEntry_t   *transition_pde_p_mon;
  Bit32u         transition_laddr;

  // These fields contain the offsets relative to the monitor
  // data segment of various structures that make up the nexus.
  // These never change.
  Bit32u module_offset;
  Bit32u nexus_offset;
  Bit32u idt_offset;
  Bit32u gdt_offset;
  Bit32u ldt_offset;
  Bit32u tss_offset;
  Bit32u idt_stubs_offset;
  Bit32u page_dir_offset;

  // Debug messages returned from monitor-side fault handlers
  // to the host-side code.
  debug_msg_t    debug_msg;

} __attribute__ ((packed)) nexus_t;



// I define the 'nexus' as the set of data structures which
// must exist in the current linear guest address space.  The
// host linear address space is not available while the current
// guest code is running, since we are using a completely
// different set of page mappings for the guest.  However,
// at some point an exception/interrupt will occur.  The
// interrupt mechanisms require that several structures exist in
// the current linear address space in order to service such
// an event.  These data structures make up part of our VM,
// a thin layer which exists in the guest.  Following is a
// list of what data structures compose this 'nexus':
//
//     - IDT (max  2048 bytes)
//     - GDT (max 65536 bytes)
//     - LDT (max 65536 bytes)
//     - TSS (max  8328 = 104 + 32 int redir + 8192 I/O permissions)
//     - kernel stack page
//     - transition code (host <--> guest)
//     - interrupt handler stubs
//     - Page Tables; PDE & PTE pages.

//
// Sizes of various nexus data structures used by the monitor
//

#define PAGESIZE 4096

#define IDT_STUB_SIZE 15
#define BYTES2PAGES(b) ( ((b)+4095) >> 12 )

#define MON_IDT_SIZE       (8*256)
#define MON_GDT_SIZE       (8*6)
#define MON_LDT_SIZE       (8*1)
#define MON_IDT_STUBS_SIZE (IDT_STUB_SIZE*256)
#define MON_TSS_SIZE       (104)

#define MON_IDT_PAGES       BYTES2PAGES(MON_IDT_SIZE)
#define MON_GDT_PAGES       BYTES2PAGES(MON_GDT_SIZE)
#define MON_LDT_PAGES       BYTES2PAGES(MON_LDT_SIZE)
#define MON_IDT_STUBS_PAGES BYTES2PAGES(MON_IDT_STUBS_SIZE)
#define MON_TSS_PAGES       BYTES2PAGES(MON_TSS_SIZE)

#define MON_GUEST_PAGES    (FMW_MAX_PHY_MEGS * 256)
#define MON_PAGE_TABLES    ((FMW_MAX_PHY_MEGS+3) >> 2)


/*
 *  Pages allocated for the VM by the host kernel driver.
 *  N Megs of physical memory are allocated, per the user's
 *  request, for the guest OS/application code.
 *  Additionally, some other overhead pages are allocated
 *  for structures such as the page directory, page tables,
 *  and other virtualized facilities.
 */

typedef struct 
{
    // requested size of the guest[] array in megs and pages
    unsigned guest_n_megs;
    unsigned guest_n_pages;

    // for the guest OS/app code
    Bit32u guest[MON_GUEST_PAGES];

    // for the monitor's page directory
    Bit32u page_dir;

    // for the monitor's page table
    Bit32u page_tbl[MON_PAGE_TABLES];

    // for the extra page table that maps our nexus code and structures
    Bit32u nexus_page_tbl;

    // We need a Page Table for identity mapping the transition code
    // between host and monitor spaces.
    Bit32u transition_PT;

    // Physical addresses of host pages which comprise the actual
    // monitor structures.  These will be mapped into the current
    // guest task's linear address space as well.
    Bit32u  nexus;
    Bit32u  idt[MON_IDT_PAGES];
    Bit32u  gdt[MON_GDT_PAGES];
    Bit32u  ldt[MON_LDT_PAGES];
    Bit32u  tss[MON_TSS_PAGES];
    Bit32u  idt_stubs[MON_IDT_STUBS_PAGES];
} vm_pages_t;

typedef struct 
{
    void         *guest;

    pageEntry_t  *page_dir;
    pageEntry_t  *page_tbl[MON_PAGE_TABLES];
    pageEntry_t  *nexus_page_tbl;
    pageEntry_t  *transition_PT;

    nexus_t      *nexus;
    gate_t       *idt;
    descriptor_t *gdt;
    descriptor_t *ldt;
    tss_t        *tss;
    idt_stub_t   *idt_stubs;
} vm_addr_t;

//
// Complete state of the VM (Virtual Machine)
//
typedef struct 
{
    // Is the monitor initialized and OK?
    unsigned initialized;

    // The next three members are filled in by the host
    vm_pages_t   pages;  // memory pages allocated by the host
    vm_addr_t    addr;   // host addresses of data structures
    Bit32u       kernel_offset;

    // Pointer into the monitor stack, so we can easily retrieve the
    // stack snapshot upon interrupt/exception.
    guest_context_t *guest_context;

    // Host to guest nexus entry point
    void (*__host2guest)(void);

    // PDE pointing to nexus page table
    pageEntry_t   nexus_pde;

} vm_t;


extern char __nexus_start, __nexus_end, __mon_cs;
extern char __host2guest, __guest2host, __handle_fault;



/*
 * This structure describes the pages containing the code/data
 * of the monitor itself (inside the kernel module)
 */

#define FMW_MAX_MONITOR_PAGES 32

typedef struct 
{
    // virtual address space occupied by the module
    Bit32u start_addr;
    // number of pages
    unsigned n_pages;
    
    // the pages themselves
    Bit32u page[FMW_MAX_MONITOR_PAGES];
} monitor_pages_t;

extern monitor_pages_t monitor_pages;

/*
 * Access to nexus page of current VM from monitor space
 */

#define vm_nexus() ({           \
    Bit32u __dummy;             \
    __asm__ (                   \
        "movl %%esp,%0  \n"     \
        : "=r" (__dummy)        \
    );                          \
    (nexus_t *)(__dummy & ~0xfff);\
})


/*
 * Generate a software interrupt
 */

#define soft_int(n)                             \
    asm volatile (                              \
        "    movb %b0, __soft_int_vector \n"    \
        "    jmp __soft_int_n            \n"    \
        "__soft_int_n:                   \n"    \
        "    sti                         \n"    \
        "    .byte 0xcd                  \n"    \
        "__soft_int_vector:              \n"    \
        "    .byte 0x00                  \n"    \
        :                                       \
        : "r" ((u8) (n) )                       \
        : "memory"                              \
    )


/*
 * Instruction emulation
 */

int emulate(guest_context_t *context);
int raise_interrupt(guest_context_t *context, int vector, int soft);

int  init_monitor(vm_t *vm);
void host2guest(vm_t *vm);
void set_guest_context(vm_t *vm, guest_context_t *context);
void get_guest_context(vm_t *vm, guest_context_t *context);

#endif  // __GUEST_H__
