/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999  Kevin P. Lawton
 *
 *  monitor.c:  This file contains the main code for initialising
 *              the monitor context, and switching from host to
 *              monitor code.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "freemware.h"
#include "monitor.h"


/************************************************************************/
/* Declarations                                                         */
/************************************************************************/


static int  init_idt_slot(vm_t *vm, unsigned vec, int type);
static void map_nexus_pages(Bit32u *, unsigned, Bit32u *, pageEntry_t *);
static void map_monitor(vm_t *vm);

#define IDT_INTERRUPT          0
#define IDT_EXCEPTION_ERROR    1
#define IDT_EXCEPTION_NOERROR  2

// Access to label offsets in nexus.S...
// From the host address perspective
#define HOST_NEXUS_OFFSET(field) \
  ((Bit32u)vm->addr.nexus + (((Bit32u) &field) - ((Bit32u) &__nexus_start)))

// From the monitor/guest address perspective.
#define MON_NEXUS_OFFSET(field) \
  (vm->addr.nexus->nexus_offset + (((Bit32u) &field) - ((Bit32u) &__nexus_start)))


// We need to set the monitor CS/DS base address so that the module pages,
// which are mapped starting at linear address 'laddr' into the guest address 
// space, reside at the same offset relative to the monitor CS base as they 
// reside relative to the kernel CS base in the host address space.  This way,
// we can execute the (non-relocatable) module code within the guest address 
// space ...
#define MON_BASE_FROM_LADDR(laddr) \
  ((laddr) - (monitor_pages.start_addr & ~0xfff))
#define LADDR_FROM_MON_BASE(base) \
  ((base) + (monitor_pages.start_addr & ~0xfff))



/************************************************************************/
/* Helper routines                                                      */
/************************************************************************/

  void
zero_memory(void *ptr, int size)
{
    char *p = ptr;
    while (size--)
        *p++ = 0;
}

  void
copy_memory(void *dst, void *src, int size)
{
    char *d = dst;
    char *s = src;
    while (size--)
        *d++ = *s++;
}


/************************************************************************/
/* Main monitor code                                                    */
/************************************************************************/

  int
init_monitor(vm_t *vm)
/*
 *  init_monitor():  Initialise the monitor context.
 *
 *  INPUTS:
 *     vm: the virtual machine
 *
 *  OUTPUTS:
 *     none
 */
{
    // These variables are used to set up the monitor pagetables
    unsigned pdi, pti, page_index;
    unsigned int i;
    Bit32u nexus_size;
    pageEntry_t  *pageTable;
    Bit32u laddr, base;
    int r;


    vm->initialized = 0; // monitor not yet initialized.


    //
    // Initialize nexus
    //

    zero_memory(vm->addr.nexus, 4096);

    // Copy transition code (nexus) into code page allocated for this VM.
    nexus_size = ((Bit32u) &__nexus_end) - ((Bit32u) &__nexus_start);
    if (nexus_size > 4096)
      goto error;
    copy_memory(vm->addr.nexus, &__nexus_start, nexus_size);


    //
    // Init the convenience pointers.
    //

    // Pointer to host2guest routine inside nexus page
    vm->__host2guest = (void (*)(void)) HOST_NEXUS_OFFSET(__host2guest);

    // Pointer to guest context on monitor stack
    vm->guest_context = (guest_context_t *)
      ((Bit32u)vm->addr.nexus + PAGESIZE - sizeof(guest_context_t));

    // Zero out various monitor data structures
    zero_memory(vm->addr.page_dir, 4096);
    zero_memory(vm->addr.idt, MON_IDT_SIZE);
    zero_memory(vm->addr.gdt, MON_GDT_SIZE);
    zero_memory(vm->addr.ldt, MON_LDT_SIZE);
    zero_memory(vm->addr.tss, MON_TSS_SIZE);
    zero_memory(vm->addr.idt_stubs, MON_IDT_STUBS_SIZE);


    // ================
    // Nexus Page Table
    // ================

    // All structures needed by the monitor inside the guest environment
    // (code to perform the transition between host<-->guest, fault handler
    // code, various processor data structures like page directory, GDT,
    // IDT, TSS etc.) are mapped into a single Page Table.
    //
    // This allows us to migrate the complete nexus to anywhere in the
    // guest address space by just updating a single (unused) page directory
    // entry in the monitor/guest page directory to point to this nexus
    // page table.
    //
    // To simplify nexus migration, we try to avoid storing guest linear
    // addresses to nexus structures as far as possible.  Instead, we use
    // offsets relative to the monitor code/data segments.  As we update
    // the base of these segments whenever the monitor migrates, the net
    // effect is that those *offsets* remain valid across nexus migration. 

    // Fill in the PDE flags
    vm->nexus_pde.base =vm->pages.nexus_page_tbl;
    vm->nexus_pde.avail = 0;
    vm->nexus_pde.G = 0;      // not global
    vm->nexus_pde.PS = 0;     // 4K pages
    vm->nexus_pde.D = 0;      // (unused in pde)
    vm->nexus_pde.A = 0;      // not accessed
    vm->nexus_pde.PCD = 0;    // normal caching
    vm->nexus_pde.PWT = 0;    // normal write-back
    vm->nexus_pde.US = 0;     // user *cannot* access
    vm->nexus_pde.RW = 1;     // read or write
    vm->nexus_pde.P = 1;      // present in memory

    // clear Page Table
    pageTable = vm->addr.nexus_page_tbl;
    zero_memory(pageTable, 4096);

    // Map pages holding nexus structures into the nexus page table.
    // For calculating the offsets, we pretend we're mapping the nexus
    // page table at linear address zero.  As the offsets are invariant
    // under nexus migration, this address is as good as any ;-)
    laddr = 0;
    base = MON_BASE_FROM_LADDR(laddr);

    vm->addr.nexus->module_offset = laddr - base;
    map_nexus_pages(monitor_pages.page, monitor_pages.n_pages, &laddr, pageTable);

    vm->addr.nexus->nexus_offset = laddr - base;
    map_nexus_pages(&vm->pages.nexus, 1, &laddr, pageTable);

    vm->addr.nexus->idt_offset = laddr - base;
    map_nexus_pages(vm->pages.idt, MON_IDT_PAGES, &laddr, pageTable);

    vm->addr.nexus->gdt_offset = laddr - base;
    map_nexus_pages(vm->pages.gdt, MON_GDT_PAGES, &laddr, pageTable);

    vm->addr.nexus->ldt_offset = laddr - base;
    map_nexus_pages(vm->pages.ldt, MON_LDT_PAGES, &laddr, pageTable);

    vm->addr.nexus->tss_offset = laddr - base;
    map_nexus_pages(vm->pages.tss, MON_TSS_PAGES, &laddr, pageTable);

    vm->addr.nexus->idt_stubs_offset = laddr - base;
    map_nexus_pages(vm->pages.idt_stubs, MON_IDT_STUBS_PAGES, &laddr,
                    pageTable);

    vm->addr.nexus->page_dir_offset = laddr - base;
    map_nexus_pages(&vm->pages.page_dir, 1, &laddr, pageTable);


    // =====================
    // Transition Page Table
    // =====================

    // To aid in the transition between host<-->monitor/guest spaces,
    // we need to have an address identity map situation for at least
    // one page; the page containing the transition code.   As we do
    // not know in advance whether this linear address range is in use
    // by the guest as well, we set aside a complete additional Page
    // Table, which contains only a single PTE pointing to the nexus page.
    //
    // To create the identity map, we simply change the corresponding
    // monitor page directory entry to point to this transition Page Table.
    // This happens transparently inside the host<-->guest transition code; 
    // both the guest/monitor code and the host side code never see this 
    // transition page table entered into the page directory!
    //
    // NOTE: We need to ensure that the nexus page table never spans the
    //       same 4Meg linear address space region as this page table!
    //       As we are free to choose the nexus linear address, this is
    //       not a problem.

    // Get full linear address of nexus code page, as seen in host space.
    laddr = (Bit32u)vm->addr.nexus + vm->kernel_offset;
    pdi = laddr >> 22;
    pti = (laddr >> 12) & 0x3ff;

    // We need to be able to access the PDE in the monitor page directory
    // that corresponds to this linear address from both host and monitor 
    // address spaces.
    vm->addr.nexus->transition_pde_p_host = vm->addr.page_dir + pdi;
    vm->addr.nexus->transition_pde_p_mon  = (pageEntry_t *)
                          (vm->addr.nexus->page_dir_offset + (pdi << 2));
    vm->addr.nexus->transition_laddr = laddr;

    // Fill in the PDE flags
    vm->addr.nexus->transition_pde.base = vm->pages.transition_PT;
    vm->addr.nexus->transition_pde.avail = 0;
    vm->addr.nexus->transition_pde.G = 0;      // not global
    vm->addr.nexus->transition_pde.PS = 0;     // 4K pages
    vm->addr.nexus->transition_pde.D = 0;      // (unused in pde)
    vm->addr.nexus->transition_pde.A = 0;      // not accessed
    vm->addr.nexus->transition_pde.PCD = 0;    // normal caching
    vm->addr.nexus->transition_pde.PWT = 0;    // normal write-back
    vm->addr.nexus->transition_pde.US = 0;     // user can not access
    vm->addr.nexus->transition_pde.RW = 1;     // read or write
    vm->addr.nexus->transition_pde.P = 1;      // present in memory

    // Clear Page Table; only one PTE is used.
    pageTable = vm->addr.transition_PT;
    zero_memory(pageTable, 4096);

    // Fill in the PTE for identity mapping the code page
    pageTable[pti].base = vm->pages.nexus;
    pageTable[pti].avail = 0;
    pageTable[pti].G = 0;      // not global
    pageTable[pti].PS = 0;     // (unused in pte)
    pageTable[pti].D = 0;      // clean
    pageTable[pti].A = 0;      // not accessed
    pageTable[pti].PCD = 0;    // normal caching
    pageTable[pti].PWT = 0;    // normal write-back
    pageTable[pti].US = 0;     // user can not access
    pageTable[pti].RW = 1;     // read or write
    pageTable[pti].P = 1;      // present in memory


    /* 
     *  Setup the TSS for the monitor/guest environment
     *
     *  We don't need to set the pagedir in the TSS, because we don't 
     *  actually jump to it anyway.  The TSS is just used to set the kernel 
     *  stack and in a later stage, perhaps the I/O permission bitmap.
     */

    // No task chain
    vm->addr.tss->back = 0;

    // No debugging or I/O, for now
    vm->addr.tss->trap = 0;
    vm->addr.tss->io = sizeof(tss_t);


    /*
     *  Setup the IDT for the monitor/guest environment
     */

    r = 0;
    r |= init_idt_slot(vm,  0, IDT_EXCEPTION_NOERROR); // Divide error
    r |= init_idt_slot(vm,  1, IDT_EXCEPTION_NOERROR); // Debug exceptions
    r |= init_idt_slot(vm,  2, IDT_INTERRUPT);         // NMI
    r |= init_idt_slot(vm,  3, IDT_EXCEPTION_NOERROR); // Breakpoint
    r |= init_idt_slot(vm,  4, IDT_EXCEPTION_NOERROR); // Overflow
    r |= init_idt_slot(vm,  5, IDT_EXCEPTION_NOERROR); // Bounds check
    r |= init_idt_slot(vm,  6, IDT_EXCEPTION_NOERROR); // Invalid opcode
    r |= init_idt_slot(vm,  7, IDT_EXCEPTION_NOERROR); // FPU not available
    r |= init_idt_slot(vm,  8, IDT_EXCEPTION_ERROR);   // Double fault
    r |= init_idt_slot(vm,  9, IDT_EXCEPTION_NOERROR); // FPU segment overrun
    r |= init_idt_slot(vm, 10, IDT_EXCEPTION_ERROR);   // Invalid TSS
    r |= init_idt_slot(vm, 11, IDT_EXCEPTION_ERROR);   // Segment not present
    r |= init_idt_slot(vm, 12, IDT_EXCEPTION_ERROR);   // Stack exception
    r |= init_idt_slot(vm, 13, IDT_EXCEPTION_ERROR);   // GP fault
    r |= init_idt_slot(vm, 14, IDT_EXCEPTION_ERROR);   // Page fault
    r |= init_idt_slot(vm, 15, IDT_EXCEPTION_NOERROR); // reserved
    r |= init_idt_slot(vm, 16, IDT_EXCEPTION_NOERROR); // Coprocessor error
    r |= init_idt_slot(vm, 17, IDT_EXCEPTION_ERROR);   // Alignment check
    r |= init_idt_slot(vm, 18, IDT_EXCEPTION_NOERROR); // Machine check

    /* Reserved exceptions */
    for (i = 19; i < 32; i++)
        r |= init_idt_slot(vm, i, IDT_EXCEPTION_NOERROR);

    /* Hardware interrupts */
    for (i = 32; i < 256; i++)
        r |= init_idt_slot(vm, i, IDT_INTERRUPT);
    if (r!=0) 
        goto error;



    /*
     *  Setup the initial guest page directory / page tables
     *
     *  For now, the guest is not allowed to use paging.  To simulate
     *  non-paged operation, we identity-map all guest physical memory
     *  at the lower end of the virtual address space.
     */

    page_index = 0;
    for (pdi=0; pdi<(vm->pages.guest_n_megs>>2); pdi++) 
    {
        // Fill in the PDE flags

        // Get the address of the page table
        vm->addr.page_dir[pdi].base = vm->pages.page_tbl[pdi];
        vm->addr.page_dir[pdi].avail = 0;
        vm->addr.page_dir[pdi].G = 0;      // not global
        vm->addr.page_dir[pdi].PS = 0;     // 4K pages
        vm->addr.page_dir[pdi].D = 0;      // (unused in pde)
        vm->addr.page_dir[pdi].A = 0;      // not accessed
        vm->addr.page_dir[pdi].PCD = 0;    // normal caching
        vm->addr.page_dir[pdi].PWT = 0;    // normal write-back
        vm->addr.page_dir[pdi].US = 1;     // user can access
        vm->addr.page_dir[pdi].RW = 1;     // read or write
        vm->addr.page_dir[pdi].P = 1;      // present in memory
  
        // clear Page Table
        pageTable = vm->addr.page_tbl[pdi];
        zero_memory(pageTable, 4096);
  
        // Setup the page table
        for (pti=0; pti<1024; pti++) 
        {
            // Fill in the PTE flags

            // Get the address of the page frame
            pageTable[pti].base = vm->pages.guest[page_index];
            page_index++;
            pageTable[pti].avail = 0;
            pageTable[pti].G = 0;      // not global
            pageTable[pti].PS = 0;     // (unused in pte)
            pageTable[pti].D = 0;      // clean
            pageTable[pti].A = 0;      // not accessed
            pageTable[pti].PCD = 0;    // normal caching
            pageTable[pti].PWT = 0;    // normal write-back
            pageTable[pti].US = 1;     // user can access
            pageTable[pti].RW = 1;     // read or write
            pageTable[pti].P = 1;      // present in memory
        }
    }

    /*
     *  Setup the initial guest GDT segments
     *
     *  For now, we just allocate one flat code and one flat data segment.
     */

    SET_DESCRIPTOR(vm->addr.gdt[1], 0, 0xfffff, 
                   D_PG, D_D32, D_AVL0, D_PRESENT, D_DPL3, D_CODE | D_READ)
    SET_DESCRIPTOR(vm->addr.gdt[2], 0, 0xfffff, 
                   D_PG, D_D32, D_AVL0, D_PRESENT, D_DPL3, D_DATA | D_WRITE)

    /*
     *  Setup the initial guest context
     */

    zero_memory(vm->guest_context, sizeof(guest_context_t));

    vm->guest_context->cs = Selector(1, 0, RPL3);  /* fixed for now */
    vm->guest_context->ss = Selector(2, 0, RPL3);  /* fixed for now */
    vm->guest_context->eflags = FLG_REAL_BASE;


    /*
     *  Map monitor into monitor/guest environment
     */

    map_monitor(vm);


    vm->initialized = 1; // monitor initialized and in OK state.
    return(0); // all OK

error:
    return(-1); // error in init_monitor()
}

  static int
init_idt_slot(vm_t *vm, unsigned vec, int type)
/*
 *  init_idt_slot():  Initialize a monitor IDT slot.
 *
 *  INPUTS:
 *     vec:      IDT vector
 *     type:     IDT_INTERRUPT          reflect interrupt to host
 *               IDT_EXCEPTION_ERROR    handle exception (with error code)
 *               IDT_EXCEPTION_NOERROR  handle exception (without error code)
 *
 *  OUTPUTS:
 *     none
 */
{
    /* IDT slot stubs */

    idt_stub_t *stub = &vm->addr.idt_stubs[vec];
    Bit32u handler_mon, stub_mon;

    if (sizeof(idt_stub_t) != IDT_STUB_SIZE) {
      return(-1);
      }

    // The handler and small stub (which calls the handler) addresses
    // from the monitor's perspective.
    handler_mon = type == IDT_INTERRUPT ? (Bit32u) MON_NEXUS_OFFSET(__guest2host)
                                        : (Bit32u) MON_NEXUS_OFFSET(__handle_fault);
    stub_mon = vm->addr.nexus->idt_stubs_offset + vec*sizeof(idt_stub_t);

    switch (type) {
      case IDT_INTERRUPT:
        stub->m2.pushla = 0x68;
        stub->m2.dummy  = 0;
        stub->m2.pushlb = 0x68;
        stub->m2.vector = vec | (RET_BECAUSE_REDIR << 8);
        stub->m2.jmp    = 0xe9;
        stub->m2.reloc  = handler_mon -
          (stub_mon + sizeof(idt_method2_t));
        break;
      case IDT_EXCEPTION_ERROR:
        stub->m1.pushl  = 0x68;
        stub->m1.vector = vec;
        stub->m1.jmp    = 0xe9;
        stub->m1.reloc  = handler_mon -
          (stub_mon + sizeof(idt_method1_t));
        break;
      case IDT_EXCEPTION_NOERROR:
        stub->m2.pushla = 0x68;
        stub->m2.dummy  = 0;
        stub->m2.pushlb = 0x68;
        stub->m2.vector = vec;
        stub->m2.jmp    = 0xe9;
        stub->m2.reloc  = handler_mon -
          (stub_mon + sizeof(idt_method2_t));
        break;
      default:
        return(-1);
      }

    /* Set the interrupt gate */
    SET_INT_GATE(vm->addr.idt[vec],
                 0, stub_mon, D_PRESENT, D_DPL0, D_D32);
    return(0);
}

  static void
map_monitor(vm_t *vm)
/*
 *  map_monitor():  Map monitor into monitor/guest environment.
 *
 *       This routine searches the guest page directory and GDT for unused
 *       PDEs and GDT selectors, and uses those to map the monitor into the
 *       guest environment.
 * 
 *       All nexus data elements that depend on the choice of nexus linear
 *       address and/or selectors (vm->nexus->mon_*) are updated accordingly.
 *
 *  INPUTS:
 *     vm: the virtual machine
 *
 *  OUTPUTS:
 *     none
 */
{
    Bit16u cs = 0, ss = 0, tss = 0;
    Bit32u laddr, base;
    int i;

    // Search for unused GDT selectors
    for ( i = 1; i < MON_GDT_PAGES*PAGESIZE/8; i++ )
        if (    ((Bit32u *)&vm->addr.gdt[i])[0] == 0
             && ((Bit32u *)&vm->addr.gdt[i])[1] == 0 )
        {
            cs = i << 3;
            break;
        }
    for ( i++; i < MON_GDT_PAGES*PAGESIZE/8; i++ )
        if (    ((Bit32u *)&vm->addr.gdt[i])[0] == 0
             && ((Bit32u *)&vm->addr.gdt[i])[1] == 0 )
        {
            ss = i << 3;
            break;
        }
    for ( i++; i < MON_GDT_PAGES*PAGESIZE/8; i++ )
        if (    ((Bit32u *)&vm->addr.gdt[i])[0] == 0
             && ((Bit32u *)&vm->addr.gdt[i])[1] == 0 )
        {
            tss = i << 3;
            break;
        }
    if ( !cs || !ss || !tss )
    {
        vm->initialized = 0;
        return;
    }

    // Search for unused PDE for nexus PT  (fixed for now)
    laddr = 0x80000000;
    base  = MON_BASE_FROM_LADDR(laddr);

    // Map nexus into monitor/guest address space
    vm->addr.page_dir[laddr >> 22] = vm->nexus_pde;

    // Set up monitor GDT descriptors
    SET_DESCRIPTOR(vm->addr.gdt[cs >> 3], base, 0xfffff, 
                   D_PG, D_D32, D_AVL0, D_PRESENT, D_DPL0, D_CODE | D_READ)
    SET_DESCRIPTOR(vm->addr.gdt[ss >> 3], base, 0xfffff, 
                   D_PG, D_D32, D_AVL0, D_PRESENT, D_DPL0, D_DATA | D_WRITE)
    SET_DESCRIPTOR(vm->addr.gdt[tss >> 3], base + vm->addr.nexus->tss_offset, sizeof(tss_t)-1,
                   D_BG, 0, D_AVL0, D_PRESENT, D_DPL0, D_TSS)

    // Fix up the selectors of all IDT entries
    for ( i = 0; i < 256; i++ )
        vm->addr.idt[i].selector = cs;

    // The monitor GDT/IDT loading info
    vm->addr.nexus->mon_gdt_info.base  = base + vm->addr.nexus->gdt_offset;
    vm->addr.nexus->mon_gdt_info.limit = 0xffff; // MON_GDT_SIZE; // +++
    vm->addr.nexus->mon_idt_info.base  = base + vm->addr.nexus->idt_offset;
    vm->addr.nexus->mon_idt_info.limit = 0xffff; // MON_IDT_SIZE; // +++

    // We don't have a monitor LDT for now
    vm->addr.nexus->mon_ldt_sel = 0;

    // The monitor TSS
    vm->addr.nexus->mon_tss_sel = tss;
    vm->addr.tss->esp0 = vm->addr.nexus->nexus_offset + PAGESIZE;
    vm->addr.tss->ss0  = ss;

    // Monitor code and stack info
    vm->addr.nexus->mon_jmp_info.offset     = MON_NEXUS_OFFSET(__mon_cs);
    vm->addr.nexus->mon_jmp_info.selector   = cs;
    vm->addr.nexus->mon_stack_info.offset   = vm->addr.tss->esp0 - sizeof(guest_context_t);
    vm->addr.nexus->mon_stack_info.selector = vm->addr.tss->ss0;

    // Monitor PDBR
    vm->addr.nexus->mon_cr3 = vm->pages.page_dir << 12;

    // Monitor code/data segment base
    vm->addr.nexus->mon_base = base;
}

  static void
unmap_monitor(vm_t *vm)
/*
 *  unmap_monitor():  Unmap monitor from monitor/guest environment.
 *
 *       This routine frees the PDEs and GDT selectors in the guest environment
 *       that were allocated by the map_monitor() routine.
 * 
 *       To update the guest page directory and/or GDT in a way that might make
 *       nexus migration necessary, call unmap_monitor(), modify the guest 
 *       page directory and/or GDT as needed, and call map_monitor() to map the 
 *       monitor back in (now at a possibly different position).
 *
 *  INPUTS:
 *     vm: the virtual machine
 *
 *  OUTPUTS:
 *     none
 */
{
    // Retrieve monitor parameters
    Bit16u cs    = vm->addr.nexus->mon_jmp_info.selector;
    Bit16u ss    = vm->addr.nexus->mon_stack_info.selector;
    Bit16u tss   = vm->addr.nexus->mon_tss_sel; 
    Bit32u base  = vm->addr.nexus->mon_base;
    Bit32u laddr = LADDR_FROM_MON_BASE(base);

    // Unmap nexus from monitor/guest address space
    *(Bit32u *)&vm->addr.page_dir[laddr >> 22] = 0;

    // Clear out monitor GDT descriptors
    SET_DESCRIPTOR(vm->addr.gdt[cs  >> 3], 0, 0, 0, 0, 0, 0, 0, 0)
    SET_DESCRIPTOR(vm->addr.gdt[ss  >> 3], 0, 0, 0, 0, 0, 0, 0, 0)
    SET_DESCRIPTOR(vm->addr.gdt[tss >> 3], 0, 0, 0, 0, 0, 0, 0, 0)
}


  void
host2guest(vm_t *vm)
{
    for (;;)
    {
        if (!vm->initialized) 
        {
            // Monitor not in an OK state to run.  Either it was not init'd,
            // or an internal monitor error occurred previously.
            vm->guest_context->event_info = RET_BECAUSE_MON_ERROR<<8;
            return;
        }

        // Call assembly routine to effect transition.
        vm->__host2guest();

        switch ( (vm->guest_context->event_info >> 8) & 0xff )
        {
        default:
        case RET_BECAUSE_MON_ERROR:
            // Kill VM if fatal monitor error occurred
            vm->initialized = 0;
            return;

        case RET_BECAUSE_REMAP:
            // Re-map monitor into guest context and restart
            unmap_monitor(vm);
            map_monitor(vm);
            break;

        case RET_BECAUSE_REDIR:
        case RET_BECAUSE_EMERR:
        case RET_BECAUSE_USEREMU:
            // Host needs to handle this situation
            return;
        }
    }
}



// Map pages allocated by host, into the linear address space of
// the monitor/guest, given the Page Table supplied.

  void
map_nexus_pages(Bit32u *pages, unsigned n, Bit32u *laddr_p, pageEntry_t *pageTable)
{
  unsigned i, pti;

  pti = (*laddr_p >> 12) & 0x3ff;
  for (i = 0; i < n; i++, pti++) {
    if (pti > 1024)
        break;  // This should not happen!

    // Fill in the PTE flags
    pageTable[pti].base = pages[i];
    pageTable[pti].avail = 0;
    pageTable[pti].G = 0;      // not global
    pageTable[pti].PS = 0;     // (unused in pte)
    pageTable[pti].D = 0;      // clean
    pageTable[pti].A = 0;      // not accessed
    pageTable[pti].PCD = 0;    // normal caching
    pageTable[pti].PWT = 0;    // normal write-back
    pageTable[pti].US = 0;     // user *cannot* access
    pageTable[pti].RW = 1;     // read or write
    pageTable[pti].P = 1;      // present in memory
    }

  // Advance linear address pointer, for the next set of pages
  // to be mapped.
  *laddr_p += 4096 * n;
}

  void
set_guest_context( vm_t *vm, guest_context_t *context )
/*
 *  set_guest_context():  Set the current guest context
 */
{
    if ( !vm->guest_context || !context ) return;

    /*
     * Pass through monitor action requests
     */
    vm->guest_context->event_info = context->event_info;

    /*
     *  We don't allow to modify segment registers for now;
     *  we don't want user mode to be able to crash the kernel ...
     */

    vm->guest_context->eax = context->eax;
    vm->guest_context->ebx = context->ebx;
    vm->guest_context->ecx = context->ecx;
    vm->guest_context->edx = context->edx;
    vm->guest_context->esi = context->esi;
    vm->guest_context->edi = context->edi;
    vm->guest_context->ebp = context->ebp;
    vm->guest_context->esp = context->esp;
    vm->guest_context->eip = context->eip;

    /*
     * Carefully set flags.
     */

    vm->addr.nexus->guest_eflags = context->eflags & (FLG_EMUL_MASK | FLG_ASYN_MASK);
    vm->guest_context->eflags = FLG_BUILD(context->eflags,
                                          vm->addr.nexus->guest_eflags,
                                          vm->addr.nexus->mon_eflags);
}

  void
get_guest_context( vm_t *vm, guest_context_t *context )
/*
 *  get_guest_context():  Get the current guest context
 */
{
    if ( !vm->guest_context || !context ) return;

    vm->guest_context->eflags = FLG_USER(vm->guest_context->eflags,
                                         vm->addr.nexus->guest_eflags);

    *context = *vm->guest_context;
}
