/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The FreeMWare developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "freemware.h"
#include "user.h"
#include "bin.h"
#include "elf.h"


/************************************************************************/
/* Main code                                                            */
/************************************************************************/

void
bin_load (char *guest_file_name)
{
    enum { TYPE_BIN, TYPE_ELF } type;
    int virtno;
    struct stat stat_buf;
    char *image;


    /* load guest code */

    fprintf (stderr, "Loading guest code: %s ", guest_file_name);

    virtno = open (guest_file_name, O_RDONLY);
    if (virtno < 0)
    {
	perror ("open");
	exit (1);
    }

    if (fstat (virtno, &stat_buf) != 0)
    {
	perror ("fstat");
	exit (1);
    }

    fprintf (stderr, "(0x%lx bytes) ", stat_buf.st_size);

    if ((image = (void *) malloc (stat_buf.st_size)) == NULL)
    {
	perror ("malloc");
	exit (1);
    }

    if (read (virtno, image, stat_buf.st_size) != stat_buf.st_size)
    {
	perror ("read");
	exit (1);
    }

    close (virtno);



    /* Check whether binary is an ELF executable */

    if (image[0] == 0x7f && 
        image[1] == 'E' && image[2] == 'L' && image[3] == 'F')
    {
	fprintf (stderr, "(ELF)\n");
	type = TYPE_ELF;
    }
    else
    {
	fprintf (stderr, "(BIN)\n");
	type = TYPE_BIN;
    }


    /* Load image into guest address space */

    switch (type)
    {
    case TYPE_BIN:
        /* Load complete binary image */
	memcpy (ptr + vm_conf.text_address, image, stat_buf.st_size);

        /* Set up initial context */
        context.eip = vm_conf.text_address;
        context.esp = vm_conf.stack_address;
        break;

    case TYPE_ELF:
    {
        Elf32_Ehdr *eh = (Elf32_Ehdr *) image;
        Elf32_Shdr *sh = (Elf32_Shdr *)(image + eh->e_shoff);
        int i;

	if (eh->e_type != ET_EXEC)
	{
	    fprintf (stderr, "Please, try it with an executable!\n");
	    exit (1);
	}

	if (eh->e_machine != EM_386)
	{
	    fprintf (stderr, "FreeMWare isn't able to run non-x86 binaries\n");
	    fprintf (stderr, "Please use an emulator for your binary!\n");
	    exit (1);
	}

        /* Load all sections that occupy space and have bits */
	for (i = 0; i < eh->e_shnum; i++)
	    if ((sh[i].sh_flags & SHF_ALLOC) && (sh[i].sh_type != SHT_NOBITS))
		memcpy (ptr + sh[i].sh_addr, image + sh[i].sh_offset, sh[i].sh_size);

        /* Retrieve entry point address and set up initial context */
        context.eip = eh->e_entry;
        context.esp = vm_conf.stack_address;
        break;
    }
    }

    free (image);
}


/************************************************************************/
/* Memory dump routine                                                  */
/************************************************************************/

void
bin_dump (char *dump_file_name)
{
    int   dumpno, counter, tmp, i, j;
    char  outbuf[330];

    counter = 0;

    fprintf (stderr, "Dumping memory to %s\n", dump_file_name);

    dumpno = open (dump_file_name, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR);

    for (i = 0; i < vm_conf.max_memory * 1024 * 1024; i++)
    {
        static char hex[] = "0123456789abcdef";

        if (!(i%32)) 
        {
            for (j = 20; j >= 0; j -= 4 )
                outbuf[counter++] = hex[ (i >> j) & 0x0f ];

            outbuf[counter++] = ':';
            outbuf[counter++] = ' ';
        }

        tmp = ((unsigned char *)ptr)[i];
        outbuf[counter++] = hex[ (tmp >> 4) & 0x0f ];
        outbuf[counter++] = hex[  tmp       & 0x0f ];
        
        if (!((i+1)%4)) 
            outbuf[counter++]=' ';
        if (!((i+1)%32)) 
            outbuf[counter++]='\n';
        if (!((i+1)%128))
        {   
            write(dumpno,&outbuf,324);
            counter = 0;
        }
    }

    close (dumpno);
}
