/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  Ulrich Weigand
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifndef __DECODE_H
#define __DECODE_H


/* Assembly Syntax */

enum i386_asm_syntax
{
    SX_INTEL, SX_ATT, SX_NONE
};


/* Register Names */

enum i386_decode_register
{
    REG_unknown = 0,

    /* Byte registers */
    REG_AL,  REG_BL,  REG_CL,  REG_DL,
    REG_AH,  REG_BH,  REG_CH,  REG_DH,
    /* Word registers */
    REG_AX,  REG_BX,  REG_CX,  REG_DX,
    REG_SI,  REG_DI,  REG_BP,  REG_SP,
    /* DWord registers */
    REG_EAX, REG_EBX, REG_ECX, REG_EDX,
    REG_ESI, REG_EDI, REG_EBP, REG_ESP,

    /* Segment registers */
    SEG_CS,  SEG_SS, 
    SEG_DS,  SEG_ES,  SEG_FS,  SEG_GS,

    /* Flags register */
    REG_FL,  REG_EFL,

    /* Instruction pointer */
    REG_IP,  REG_EIP,

    /* Control registers */
    REG_CR0, REG_CR1, REG_CR2, REG_CR3,
    REG_CR4, REG_CR5, REG_CR6, REG_CR7,

    /* Debug registers */
    REG_DR0, REG_DR1, REG_DR2, REG_DR3,
    REG_DR4, REG_DR5, REG_DR6, REG_DR7,

    /* Test registers */
    REG_TR0, REG_TR1, REG_TR2, REG_TR3,
    REG_TR4, REG_TR5, REG_TR6, REG_TR7,

    /* FPU register stack */
    REG_ST0, REG_ST1, REG_ST2, REG_ST3,
    REG_ST4, REG_ST5, REG_ST6, REG_ST7,

    /* MMX registers */
    REG_MM0, REG_MM1, REG_MM2, REG_MM3,
    REG_MM4, REG_MM5, REG_MM6, REG_MM7,

    /* SSE registers */
    REG_XMM0, REG_XMM1, REG_XMM2, REG_XMM3,
    REG_XMM4, REG_XMM5, REG_XMM6, REG_XMM7,

    REG_LAST
};


/* Instruction Names */

enum i386_decode_instruction
{
    INS_unknown = 0,

    /* Data Transfer */
    INS_mov,    
    INS_cmove,   INS_cmovne,  INS_cmovp,   INS_cmovnp,
    INS_cmova,   INS_cmovae,  INS_cmovb,   INS_cmovbe,
    INS_cmovg,   INS_cmovge,  INS_cmovl,   INS_cmovle,
    INS_cmovo,   INS_cmovno,  INS_cmovs,   INS_cmovns,
    INS_xchg,    INS_bswap,   INS_xadd,    INS_cmpxchg, INS_cmpxchg8b,
    INS_push,    INS_pusha,   INS_pushad,  
    INS_pop,     INS_popa,    INS_popad,
    INS_in,      INS_out,
    INS_cwd,     INS_cdq,     INS_cbw,     INS_cwde,
    INS_movsx,   INS_movzx,
    /* Binary Arithmetic */
    INS_add,     INS_adc,     INS_sub,     INS_sbb,
    INS_imul,    INS_mul,     INS_idiv,    INS_div,
    INS_inc,     INS_dec,     INS_neg,     INS_cmp,   
    /* Decimal Arithmetic */
    INS_daa,     INS_das,     
    INS_aaa,     INS_aas,     INS_aam,     INS_aad,	
    /* Logic */
    INS_and,     INS_or,      INS_not,     INS_xor,
    /* Shift and Rotate */
    INS_sar,     INS_shr,     INS_sal,     INS_shl,   INS_shld,  INS_shrd,
    INS_ror,     INS_rol,     INS_rcr,     INS_rcl,
    /* Bit and Byte */
    INS_bt,      INS_bts,     INS_btr,     INS_btc,   INS_bsf,   INS_bsr,
    INS_sete,    INS_setne,   INS_setp,    INS_setnp,
    INS_seta,    INS_setae,   INS_setb,    INS_setbe,
    INS_setg,    INS_setge,   INS_setl,    INS_setle,
    INS_seto,    INS_setno,   INS_sets,    INS_setns,
    INS_test,
    /* Control Transfer */
    INS_jmp,   
    INS_je,      INS_jne,     INS_jp,      INS_jnp,
    INS_ja,      INS_jae,     INS_jb,      INS_jbe,
    INS_jg,      INS_jge,     INS_jl,      INS_jle,
    INS_jo,      INS_jno,     INS_js,      INS_jns,
    INS_jcxz,    INS_jecxz, 
    INS_loop,    INS_loope,   INS_loopne,
    INS_call,    INS_ret,     INS_retf,    INS_iret,
    INS_int,     INS_into,    INS_bound, 
    INS_enter,   INS_leave,
    /* String */
    INS_movs,    INS_cmps,    INS_scas,    INS_lods,    INS_stos,
    INS_rep,     INS_repe,    INS_repne,
    INS_ins,     INS_outs,
    /* Flag Control */
    INS_stc,     INS_clc,     INS_cmc,     INS_std,     INS_cld,
    INS_lahf,    INS_sahf,    
    INS_pushf,   INS_pushfd,  INS_popf,    INS_popfd,
    INS_sti,     INS_cli,
    /* Segment Register Operations */
    INS_lds,     INS_les,     INS_lfs,     INS_lgs,     INS_lss,
    /* Miscellaneous */
    INS_lea,     INS_nop,     INS_ud2,     INS_xlat,
    INS_cpuid,
    /* Systems Instructions */
    INS_lgdt,    INS_sgdt,    INS_lldt,    INS_sldt,  
    INS_ltr,     INS_str,     INS_lidt,    INS_sidt,
    INS_lmsw,    INS_smsw,    INS_clts,
    INS_arpl,    INS_lar,     INS_lsl,     INS_verr,    INS_verw,
    INS_invd,    INS_wbinvd,  INS_invlpg,
    INS_lock,    INS_hlt,     INS_rsm,
    INS_rdmsr,   INS_wrmsr,   INS_rdpmc,   INS_rdtsc,
    INS_sysenter,INS_sysexit,
    /* MMX Data Transfer */
    INS_movd,    INS_movq,
    /* MMX Conversion */
    INS_packsswb,  INS_packssdw,  INS_packuswb,
    INS_punpckhbw, INS_punpckhwd, INS_punpckhdq,
    INS_punpcklbw, INS_punpcklwd, INS_punpckldq,
    /* MMX Packed Arithmetic */
    INS_paddb,   INS_paddw,   INS_paddd,
    INS_paddsb,  INS_paddsw,  INS_paddusb, INS_paddusw,
    INS_psubb,   INS_psubw,   INS_psubd,
    INS_psubsb,  INS_psubsw,  INS_psubusb, INS_psubusw,
    INS_pmulhw,  INS_pmullw,  INS_pmaddwd,
    /* MMX Comparison */
    INS_pcmpeqb, INS_pcmpeqw, INS_pcmpeqd,
    INS_pcmpgtb, INS_pcmpgtw, INS_pcmpgtd,
    /* MMX Logic */
    INS_pand,    INS_pandn,   INS_por,     INS_pxor,
    /* MMX Shift and Rotate */
    INS_psllw,   INS_pslld,   INS_psllq,
    INS_psrlw,   INS_psrld,   INS_psrlq,
    INS_psraw,   INS_psrad,
    /* MMX State Management */
    INS_emms,
    /* SSE Floating Point Data Transfer */
    INS_movss,   INS_movaps,  INS_movups,  INS_movmskps,
    INS_movlps,  INS_movhps,  INS_movhlps, INS_movlhps,
    /* SSE Floating Point Conversion */
    INS_cvtpi2ps,INS_cvtps2pi,INS_cvttps2pi,
    INS_cvtsi2ss,INS_cvtss2si,INS_cvttss2si,
    INS_unpcklps,INS_unpckhps,INS_shufps,
    /* SSE Floating Point Arithmetic */
    INS_addps,   INS_addss,   INS_subps,   INS_subss,
    INS_mulps,   INS_mulss,   INS_divps,   INS_divss,
    INS_maxps,   INS_maxss,   INS_minps,   INS_minss,
    INS_sqrtps,  INS_sqrtss,  INS_rsqrtps, INS_rsqrtss,
    INS_rcpps,   INS_rcpss,
    /* SSE Floating Point Logic */
    INS_andps,   INS_andnps,  INS_orps,    INS_xorps,
    /* SSE Floating Point Comparision */
    INS_cmpps,   INS_cmpordps,INS_cmpunordps,
    INS_cmpeqps, INS_cmpltps, INS_cmpleps,
    INS_cmpneqps,INS_cmpnltps,INS_cmpnleps,
    INS_cmpss,   INS_cmpordss,INS_cmpunordss,
    INS_cmpeqss, INS_cmpltss, INS_cmpless,
    INS_cmpneqss,INS_cmpnltss,INS_cmpnless,
    INS_comiss,  INS_ucomiss,
    /* SSE Integer Data Transfer */
    INS_pinsrw,  INS_pextrw,  INS_pshufw,  INS_pmovmskb,
    /* SSE Integer Arithmetic */
    INS_pmaxub,  INS_pmaxsw,  INS_pminub,  INS_pminsw,
    INS_pavgb,   INS_pavgw,
    INS_pmulhuw, INS_psadbw,
    /* SSE Cachability Control */
    INS_movntq,  INS_movntps, INS_maskmovq,INS_sfence,
    INS_prefetcht0, INS_prefetcht1,
    INS_prefetcht2, INS_prefetchnta,
    /* SSE State Management */
    INS_fxsave,  INS_fxrstor,
    INS_ldmxcsr, INS_stmxcsr,
    /* FPU Data Transfer */
    INS_fld,     INS_fst,     INS_fstp,
    INS_fild,    INS_fist,    INS_fistp,
    INS_fbld,    INS_fbstp,   INS_fxch,
    INS_fcmove,  INS_fcmovne, INS_fcmovu,  INS_fcmovnu,
    INS_fcmovb,  INS_fcmovbe, INS_fcmovnb, INS_fcmovnbe,
    /* FPU Basic Arithmetic */
    INS_fadd,    INS_faddp,   INS_fiadd,
    INS_fsub,    INS_fsubp,   INS_fisub,
    INS_fsubr,   INS_fsubrp,  INS_fisubr,
    INS_fmul,    INS_fmulp,   INS_fimul,
    INS_fdiv,    INS_fdivp,   INS_fidiv,
    INS_fdivr,   INS_fdivrp,  INS_fidivr,
    INS_fprem,   INS_fprem1,
    INS_fabs,    INS_fchs,    INS_frndint, 
    INS_fscale,  INS_fsqrt,   INS_fxtract,
    /* FPU Comparison */
    INS_fcom,    INS_fcomp,   INS_fcompp,
    INS_fucom,   INS_fucomp,  INS_fucompp,
    INS_ficom,   INS_ficomp,  
    INS_fcomi,   INS_fucomi,  INS_fcomip,  INS_fucomip,
    INS_ftst,    INS_fxam,
    /* FPU Transcendental */
    INS_fsin,    INS_fcos,    INS_fsincos,
    INS_fptan,   INS_fpatan,
    INS_f2xm1,   INS_fyl2x,   INS_fyl2xp1,
    /* FPU Load Constants */
    INS_fld1,    INS_fldz,    INS_fldpi,
    INS_fldl2e,  INS_fldln2,  INS_fldl2t,  INS_fldlg2,
    /* FPU Control */
    INS_fincstp, INS_fdecstp, INS_ffree,   
    INS_finit,   INS_fninit,  INS_fclex,   INS_fnclex,
    INS_fstcw,   INS_fnstcw,  INS_fldcw,
    INS_fstenv,  INS_fnstenv, INS_fldenv,
    INS_fsave,   INS_fnsave,  INS_frstor,
    INS_fstsw,   INS_fnstsw,
    INS_fnop,    INS_fwait,    

    INS_LAST
};

/* Operand description */

struct i386_decode_operand
{
    enum { OT_NONE = 0,
           OT_VALUE,
           OT_POINTER,
           OT_DESCRIPTOR,
           OT_VALUEPAIR } type;
    int size;

    enum { OM_REGISTER = 1, 
           OM_IMMEDIATE,
           OM_MEMORY } mode;
    union
    {
        enum i386_decode_register reg;

        struct i386_decode_data
        {
            int segment;
            int offset;
        } immediate;

        struct i386_decode_address
        {
            enum i386_decode_register segment;
            enum i386_decode_register base;
            enum i386_decode_register index;
            int scale;
            int displacement;
        } memory;
    } u;
};

/* Decoded instruction description */

struct i386_decode
{
    int length;

    enum i386_decode_instruction prefix;
    enum i386_decode_instruction instruction;

    int address_size;
    int displacement_size;
    int operand_size;

    enum i386_decode_register segment_override;

    struct i386_decode_operand op[3];
};

/* Decode context */

struct i386_context
{
    enum { CODE16, CODE32, VM86 } mode;

    int segment;
    int offset;

    void *base;
};


/* Function declarations */

int i386_decode( struct i386_context *ctx, struct i386_decode *dc );
int i386_decode_text_intel( struct i386_context *ctx, struct i386_decode *dc, char *string );
int i386_decode_text_att( struct i386_context *ctx, struct i386_decode *dc, char *string );
int i386_decode_text( struct i386_context *ctx, struct i386_decode *dc, char *string, enum i386_asm_syntax syntax );


#endif   /* __DECODE_H */
