/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The FreeMWare developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>

#include "freemware.h"
#include "user.h"
#include "plugin.h"



/************************************************************************/
/* The main event loop                                                  */
/************************************************************************/

#define  FNEXT(x,y) \
    { if (context.eflags & 0x400) x -= y; else x += y; }

#define  FNBUF(x,y) \
    { if (context.eflags & 0x400) { \
      buf = malloc ((context.ecx & mask)*x>>3); if (!buf) { perror ("malloc"); \
      vm_abort (); } } else buf = (ptr + (y & mask)); }

#define  FNFBUF() \
    { if (context.eflags & 0x400) free (buf); }

/* FIXME:  optimize using "rep movs" */
#define  FOUTBUF(x) \
    { int j; if (context.eflags & 0x400) \
          for (j = 0; j < (context.ecx & mask); j++) \
              *((Bit##x##u *) (buf + j)) = *((Bit##x##u *) (ptr + (context.esi & mask)) - j); }

/* FIXME:  optimize using "rep movs" */
#define  FINBUF(x) \
    { int j; if (context.eflags & 0x400) \
          for (j = 0; j < (context.ecx & mask); j++) \
              *((Bit##x##u *) (ptr + (context.edi & mask)) - j) = *((Bit##x##u *) (buf + j)); }


void
emulate (void)
{
    /*
     * context.event_info structure
     * 0x0  byte  EMU_INSTR_ (instruction to emulate)
     * 0x1  byte  RET_BECAUSE_USEREMU
     * 0x2  word  argument(s)
     */

    Bit32u instr = context.event_info & 0xff;
    Bit32u arg   = (context.event_info >> 16) & 0xffff;
    Bit32u value = 0;
    Bit32u mask  = (1 << (arg & 0xff)) - 1;  /* for ins / outs */
    void  *buf;

    switch (instr)
    {
    case EMU_INSTR_IN_8:
	plugin_emulate (EVT_INPORT, arg, 1, 1, &value);
	context.eax = (context.eax & ~0xff) | (value & 0xff);
	break;

    case EMU_INSTR_IN_16:
	plugin_emulate (EVT_INPORT, arg, 2, 1, &value);
	context.eax = (context.eax & ~0xffff) | (value & 0xffff);
	break;

    case EMU_INSTR_IN_32:
	plugin_emulate (EVT_INPORT, arg, 4, 1, &value);
	context.eax = value;
	break;

    case EMU_INSTR_OUT_8:
	value = context.eax & 0xff;
	plugin_emulate (EVT_OUTPORT, arg, 1, 1, &value);
	break;

    case EMU_INSTR_OUT_16:
	value = context.eax & 0xffff;
	plugin_emulate (EVT_OUTPORT, arg, 2, 1, &value);
	break;

    case EMU_INSTR_OUT_32:
	value = context.eax;
	plugin_emulate (EVT_OUTPORT, arg, 4, 1, &value);
	break;

    case EMU_INSTR_INS_8:
	/* FIXME: works only with flat memory models without paging and ES == DS */
	plugin_emulate (EVT_INPORT, context.edx & 0xffff, 1,
                        1, (ptr + (context.edi & mask)));

	FNEXT(context.edi, 1);

	break;

    case EMU_INSTR_INS_16:
	/* FIXME: works only with flat memory models without paging and ES == DS */
	plugin_emulate (EVT_INPORT, context.edx & 0xffff, 2,
                        1, (ptr + (context.edi & mask)));

	FNEXT(context.edi, 2);

	break;

    case EMU_INSTR_INS_32:
	/* FIXME: works only with flat memory models without paging and ES == DS */
	plugin_emulate (EVT_INPORT, context.edx & 0xffff, 4,
                        1, (ptr + (context.edi & mask)));

	FNEXT(context.edi, 4);

	break;

    case EMU_INSTR_OUTS_8:
	/* FIXME: works only with flat memory models without paging and ES == DS */
	plugin_emulate (EVT_OUTPORT, context.edx & 0xffff, 1,
                        1, (ptr + (context.esi & mask)));

	FNEXT(context.esi, 1);

	break;

    case EMU_INSTR_OUTS_16:
	/* FIXME: works only with flat memory models without paging and ES == DS */
	plugin_emulate (EVT_OUTPORT, context.edx & 0xffff, 2,
                        1, (ptr + (context.esi & mask)));

	FNEXT(context.esi, 2);

	break;

    case EMU_INSTR_OUTS_32:
	/* FIXME: works only with flat memory models without paging and ES == DS */
	plugin_emulate (EVT_OUTPORT, context.edx & 0xffff, 4,
                        1, (ptr + (context.esi & mask)));

	FNEXT(context.esi, 4);

	break;

    case EMU_INSTR_REP_INS_8:
        if (!(context.ecx & mask))
            break;

        /* FIXME: works only with flat memory models without paging and ES == DS */
        FNBUF(8, context.edi);

        plugin_emulate (EVT_INPORT, context.edx & 0xffff, 1, (context.ecx & mask), buf);

        FINBUF(8);
        FNFBUF();

        FNEXT(context.edi, (context.ecx & mask));
        context.ecx &= ~mask;

        break;

    case EMU_INSTR_REP_INS_16:
        if (!(context.ecx & mask))
            break;

        /* FIXME: works only with flat memory models without paging and ES == DS */
        FNBUF(16, context.edi);

        plugin_emulate (EVT_INPORT, context.edx & 0xffff, 2, (context.ecx & mask), buf);

        FINBUF(16);
        FNFBUF();

        FNEXT(context.edi, 2*(context.ecx & mask));
        context.ecx &= ~mask;

        break;

    case EMU_INSTR_REP_INS_32:
        if (!(context.ecx & mask))
            break;

        /* FIXME: works only with flat memory models without paging and ES == DS */
        FNBUF(32, context.edi);

        plugin_emulate (EVT_INPORT, context.edx & 0xffff, 4, (context.ecx & mask), buf);

        FINBUF(32);
        FNFBUF();

        FNEXT(context.edi, 4*(context.ecx & mask));
        context.ecx &= ~mask;

        break;

    case EMU_INSTR_REP_OUTS_8:
        if (!(context.ecx & mask))
            break;

        /* FIXME: works only with flat memory models without paging and ES == DS */
        FNBUF(8, context.esi);
        FOUTBUF(8);

        plugin_emulate (EVT_INPORT, context.edx & 0xffff, 1, (context.ecx & mask), buf);

        FNFBUF();

        FNEXT(context.esi, (context.ecx & mask));
        context.ecx &= ~mask;

        break;

    case EMU_INSTR_REP_OUTS_16:
        if (!(context.ecx & mask))
            break;

        /* FIXME: works only with flat memory models without paging and ES == DS */
        FNBUF(16, context.esi);
        FOUTBUF(16);

        plugin_emulate (EVT_INPORT, context.edx & 0xffff, 2, (context.ecx & mask), buf);

        FNFBUF();

        FNEXT(context.esi, 2*(context.ecx & mask));
        context.ecx &= ~mask;

        break;

    case EMU_INSTR_REP_OUTS_32:
        if (!(context.ecx & mask))
            break;

        /* FIXME: works only with flat memory models without paging and ES == DS */
        FNBUF(32, context.esi);
        FOUTBUF(32);

        plugin_emulate (EVT_INPORT, context.edx & 0xffff, 4, (context.ecx & mask), buf);

        FNFBUF();

        FNEXT(context.esi, 4*(context.ecx & mask));
        context.ecx &= ~mask;

        break;

    case EMU_INSTR_HLT:
	/* FIXME: should only shut down if interrupts are off ("cli; hlt") */
	fprintf (stderr, "HLT -- shutting down virtual machine\n");
	vm_abort ();
	break;

    case EMU_INSTR_INT:
        plugin_emulate (EVT_INT, arg & 0xff, 0, 0, NULL);
        break;

    case EMU_INSTR_STI:
        break;

    default:
	fprintf (stderr, "Emulation of instruction 0x%x (0x%04x) not yet implemented\n",
		 instr, arg);
    }
}
