/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The FreeMWare developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "freemware.h"
#include "user.h"
#include "plugin.h"


/************************************************************************/
/* Declarations                                                         */
/************************************************************************/

static void abort_handler (int i);



/************************************************************************/
/* Structures / Variables                                               */
/************************************************************************/

/* Global data */

config_info_t vm_conf;



/************************************************************************/
/* Main code                                                            */
/************************************************************************/

int
main (int argc, char *argv[])
{
    int   config_fd, line_index, i;
    char  line[256], sxname[10], config_file_name[256];
    char  c;
    struct sigaction sg_act;



    /* some inits */

    strcpy (config_file_name, "fmw.conf");
    vm_conf.guest_file_name[0] = '\0';
    vm_conf.dump_file_name[0]  = '\0';
    vm_conf.max_memory    =  0;
    vm_conf.text_address  = -1;
    vm_conf.data_address  = -1;
    vm_conf.stack_address = -1;
    vm_conf.bss_address   = -1;
    vm_conf.verbose       =  0;
    vm_conf.dump_vm       =  0;
    vm_conf.syntax        =  SX_NONE;



    /* parsing command line for arguments */

    fprintf (stderr, "Processing command-line options\n");

    for (i = 0; i < argc; i++)
    {
	if (argv[i][0] != '-')
	    continue;

	switch (argv[i][1])
	{
	case 'h':
	    printf ("FreeMWare user-level monitor version " VERSION "\n");
            printf ("Usage:  user -dfghmsv\n");
            printf ("Options:\n");
	    printf ("-d ... Dump guest state (vm-core) after running\n");
	    printf ("-f ... Config File (fully qualified path)\n");
	    printf ("-g ... Guest code (fully qualified path)\n");
	    printf ("-h ... Print this helpful help\n");
	    printf ("-m ... Megs of memory for VM\n");
	    printf ("-s ... Select disassembly syntax (\"intel\" or \"att\")\n");
	    printf ("-v ... Print debug messages (verbose)\n");
	    exit (0);
	    break;

	case 'm':
	    vm_conf.max_memory = atoi (&argv[i][3]);
	    break;

	case 'f':
	    strncpy (config_file_name, &argv[i][3], 255);
	    break;

	case 'g':
	    strncpy (vm_conf.guest_file_name, &argv[i][3], 255);
	    break;

	case 's':
	    strncpy (sxname, &argv[i][3], 10);
	    if (!strcmp (sxname, "intel"))
		vm_conf.syntax = SX_INTEL;
	    else if (!strcmp (sxname, "att"))
		vm_conf.syntax = SX_ATT;
	    break;

	case 'v':
	    vm_conf.verbose = 1;
	    break;

	case 'd':
	    vm_conf.dump_vm = 1;
	    break;

	default:
	    fprintf (stderr, "Unknown option: %s\n", argv[i]);
	    break;
	}
    }



    /* read config file */

    fprintf (stderr, "Processing configuration script %s\n", config_file_name);

    config_fd = open (config_file_name, O_RDONLY);
    if (config_fd == -1)
    {
	char  error_msg[100];

	sprintf (error_msg, "error while opening %s", config_file_name);
	perror (error_msg);
	exit (1);
    }

    line_index = 0;		/* states: 0 = reading, 1 = ignoring */

    while (read (config_fd, &c, 1))
    {
	line[line_index++] = c;
	if (c == '\n')
	{
	    line[line_index - 1] = '\0';
	    line_index = 0;
	    if ((line[0] == '#') || (line[0] == '\n'))
		continue;	// is a comment or a blank line;

	    while ((line[line_index] != ' ') && (line[line_index] != '='))
		line_index++;
	    if (!strncmp (line, "memory", line_index))
	    {
		if (vm_conf.max_memory != 0)
		{
		    line_index = 0;
		    continue;
		}		// command line overwrites config file settings;

		while ((line[line_index] == ' ') || (line[line_index] == '='))
		    line_index++;
		vm_conf.max_memory = atoi (&line[line_index]);
	    }
	    else if (!strncmp (line, "text_address", line_index))
	    {
		if (vm_conf.text_address != -1)
		{
		    line_index = 0;
		    continue;
		}		// command line overwrites config file settings;

		while ((line[line_index] == ' ') || (line[line_index] == '='))
		    line_index++;
		vm_conf.text_address = atoi (&line[line_index]);
	    }
	    else if (!strncmp (line, "data_address", line_index))
	    {
		if (vm_conf.data_address != -1)
		{
		    line_index = 0;
		    continue;
		}		// command line overwrites config file settings;

		while ((line[line_index] == ' ') || (line[line_index] == '='))
		    line_index++;
		vm_conf.data_address = atoi (&line[line_index]);
	    }
	    else if (!strncmp (line, "stack_address", line_index))
	    {
		if (vm_conf.stack_address != -1)
		{
		    line_index = 0;
		    continue;
		}		// command line overwrites config file settings;

		while ((line[line_index] == ' ') || (line[line_index] == '='))
		    line_index++;
		vm_conf.stack_address = atoi (&line[line_index]);
	    }
	    else if (!strncmp (line, "bss_address", line_index))
	    {
		if (vm_conf.bss_address != -1)
		{
		    line_index = 0;
		    continue;
		}		// command line overwrites config file settings;

		while ((line[line_index] == ' ') || (line[line_index] == '='))
		    line_index++;
		vm_conf.bss_address = atoi (&line[line_index]);
	    }
	    else if (!strncmp (line, "guest", line_index))
	    {
		if (vm_conf.guest_file_name[0] != '\0')
		{
		    line_index = 0;
		    continue;
		}
		while ((line[line_index] == ' ') || (line[line_index] == '='))
		    line_index++;
		strcpy (vm_conf.guest_file_name, &line[line_index]);
	    }
	    else if (!strncmp (line, "db_syntax", line_index))
	    {
		if (vm_conf.syntax != SX_NONE)
		{
		    line_index = 0;
		    continue;
		}

		while ((line[line_index] == ' ') || (line[line_index] == '='))
		    line_index++;
		strcpy (sxname, &line[line_index]);

		if (!strcmp (sxname, "intel"))
		    vm_conf.syntax = SX_INTEL;
		else if (!strcmp (sxname, "at&t"))
		    vm_conf.syntax = SX_ATT;
	    }
	    else if (!strncmp (line, "plugin", line_index))
	    {
		int   parm_index;
		char *plugin_name, *plugin_args;

		while ((line[line_index] == ' ') || (line[line_index] == '='))
		    line_index++;

		parm_index = line_index;
		while ((line[parm_index] != ' ') && (line[parm_index] != '\0'))
		    parm_index++;
		if (line[parm_index] != '\0')
		    line[parm_index++] = '\0';

		plugin_name = strcpy (malloc (strlen (&line[line_index]) + 1), &line[line_index]);
		plugin_args = strcpy (malloc (strlen (&line[parm_index]) + 1), &line[parm_index]);

		plugin_load (plugin_name, plugin_args);
	    }
	    line_index = 0;
	}
    }

    close (config_fd);



    /* set default settings, if not specified */

    if (vm_conf.max_memory == 0)
	vm_conf.max_memory = FMW_MAX_PHY_MEGS;
    if (vm_conf.guest_file_name[0] == '\0')
	strcpy (vm_conf.guest_file_name, "../guest/virtcode/virtcode");
    if (vm_conf.dump_file_name[0] == '\0')
	strcpy (vm_conf.dump_file_name, "vm-core");
    if (vm_conf.syntax == SX_NONE)
	vm_conf.syntax = SX_ATT;
    if (vm_conf.text_address == -1)
	vm_conf.text_address = 0;

    if (vm_conf.verbose)
    {
	fprintf (stderr, "Memory:\t\t%d\n",      vm_conf.max_memory);
	fprintf (stderr, "Text address:\t%d\n",  vm_conf.text_address);
	fprintf (stderr, "Data address:\t%d\n",  vm_conf.data_address);
	fprintf (stderr, "Stack address:\t%d\n", vm_conf.stack_address);
	fprintf (stderr, "BSS address:\t%d\n",   vm_conf.bss_address);
    }



    /* setup to catch SIGINT, SIGABRT and SIGQUIT signals so we can clean up */

    memset (&sg_act, 0, sizeof (sg_act));
    sg_act.sa_handler = abort_handler;
    sg_act.sa_flags = SA_RESETHAND;
    sigaction (SIGINT, &sg_act, NULL);
    sigaction (SIGABRT, &sg_act, NULL);
    sigaction (SIGQUIT, &sg_act, NULL);



    /* kickstart the VM */

    vm_kickstart ();



    /* run the VM */

    fprintf (stderr, "Running VM\n");
    vm_event_loop ();



    /* vm_event_loop() should never return */

    fprintf (stderr, "Fatal: vm_event_loop() returned.  This is a bug. ");
    fprintf (stderr, "Please contact " BUGSMAIL "\n");
    return 1;
}



/************************************************************************/
/* Signal handlers                                                      */
/************************************************************************/

static void
abort_handler (int i)
{
    switch (i)
    {
    case SIGINT:
        printf ("^C pressed. aborting execution\n");
        break;

    case SIGABRT:
    case SIGQUIT:
        printf ("^\\ pressed. dumping VM core and aborting execution\n");
        vm_conf.dump_vm = 1;
        break;

    default:
        printf ("Signal %d caught, aborting execution\n", i);
        break;
    }

    vm_abort ();
    return;
}
