/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The FreeMWare developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "freemware.h"
#include "user.h"
#include "plugin.h"

#define  SUBSET(base1,range1,base2,range2) \
                 ((base2 >= base1) && ((base1+range1) >= (base2+range2)))



/************************************************************************/
/* Declarations                                                         */
/************************************************************************/

static void plugin_emulate_inport (int data, int op_size, int count, void *loc);
static void plugin_emulate_outport (int data, int op_size, int count, void *loc);
static void plugin_emulate_int (int data, int op_size, int count, void *loc);

static int  plugin_alloc_int (event_t event, handler_t handler, int base, int range);
static void plugin_free_int (callback_t *callback);



/************************************************************************/
/* Structures / Variables                                               */
/************************************************************************/

plugin_t *plugins     = NULL;	/* Head of the linked list of plugins  */

callback_t *ioreads   = NULL;   /* Head of I/O read callback list      */
callback_t *iowrites  = NULL;   /* Head of I/O write callback list     */
callback_t *ints      = NULL;   /* Head of interrupt callback list     */



/************************************************************************/
/* Plugin initialization / deinitialization                             */
/************************************************************************/

void
plugin_load (char *name, char *args)
{
    plugin_t *plugin;
    const char *plug_err;

    plugin = malloc (sizeof (plugin_t));
    if (!plugin)
    {
	perror ("malloc");
	exit (1);
    }

    plugin->name = name;
    plugin->args = args;
    plugin->callbacks = NULL;
    plugin->initialized = 0;

    plugin->handle = dlopen (name, RTLD_LAZY);
    if (!plugin->handle)
    {
	fputs (dlerror (), stderr);
	exit (1);
    }

    plugin->init = dlsym (plugin->handle, PLUGIN_INIT);
    if ((plug_err = dlerror ()) != NULL)
    {
	fputs (plug_err, stderr);
	exit (1);
    }

    plugin->fini = dlsym (plugin->handle, PLUGIN_FINI);
    if ((plug_err = dlerror ()) != NULL)
    {
	fputs (plug_err, stderr);
	exit (1);
    }

    plugin->next = plugins;
    plugins = plugin;

    return;
}


void
plugin_init_all (void)
{
    plugin_t *plugin;

    for (plugin = plugins; plugin; plugin = plugin->next)
    {
	char *arg_ptr = plugin->args;

	/* process the command line */
	plugin->argc = 0;
	while (plugin->argc < MAX_ARGC)
	{
	    while (*arg_ptr && isspace (*arg_ptr))
		arg_ptr++;

	    if (!*arg_ptr)
		break;
	    plugin->argv[plugin->argc++] = arg_ptr;

	    while (*arg_ptr && !isspace (*arg_ptr))
		arg_ptr++;

	    if (!*arg_ptr)
		break;
	    *arg_ptr++ = '\0';
	}

	/* initialize the plugin */
	if (plugin->init (plugin, plugin->argc, plugin->argv))
	{
	    fprintf (stderr, "Plugin initialization failed for %s\n", plugin->name);
            plugin_abort();
	}

        plugin->initialized = 1;
    }

    return;
}


plugin_t *
plugin_unload (plugin_t *plugin)
{
    plugin_t *dead_plug;
    callback_t *dead_call;

    if (plugin->initialized)
        plugin->fini ();

    for (dead_call = plugin->callbacks; dead_call; dead_call = dead_call->next_plug)
        plugin_free (dead_call);

    dlclose (plugin->handle);
    free (plugin->name);
    free (plugin->args);

    dead_plug = plugin;
    plugin = plugin->next;
    free (dead_plug);

    return plugin;
}


void
plugin_fini_all (void)
{
    plugin_t *plugin;

    for (plugin = plugins; plugin; plugin = plugin_unload (plugin));

    return;
}



/************************************************************************/
/* Resource allocation                                                  */
/************************************************************************/

callback_t *
plugin_alloc (plugin_t *plugin, event_t event, handler_t handler, int base, int range)
{
    callback_t **list, *callback;
    int (*plugin_evt_alloc) (event_t event, handler_t handler, int base, int range) = NULL;


    switch (event)
    {
    case EVT_INPORT:
        list = &ioreads;
        break;

    case EVT_OUTPORT:
        list = &iowrites;
        break;

    case EVT_INT:
        list = &ints;
        plugin_evt_alloc = plugin_alloc_int;
        break;

    default:
        fprintf (stderr, "Error: plugin %s attempted to allocate non-implemented resource. Ignoring\n", plugin->name);
        return NULL;
    }


    /* look for existing allocation */

    if (conflict[event])
        for (callback = *list; callback; callback = callback->next_type)
            if (callback->base == base && callback->range == range)
            {
                fprintf (stderr, "Conflict: plugin %s attempted to allocate busy resource. Ignoring\n", plugin->name);
                return NULL;
            }


    /* do event-specific allocation */

    if (plugin_evt_alloc && plugin_evt_alloc (event, handler, base, range))
        return NULL;


    /* allocate plugin */

    callback = malloc (sizeof (callback_t));
    callback->type    = event;
    callback->handler = handler;
    callback->base    = base;
    callback->range   = range;
    callback->plugin  = plugin;

    /* put callback in linked list */

    callback->next_type = *list;
    *list = callback;


    /* we're done!! */

    return callback;
}


void
plugin_free (callback_t *callback)
{
    callback_t **list, *next;


    switch (callback->type)
    {
    case EVT_INPORT:
        list = &ioreads;
        break;

    case EVT_OUTPORT:
        list = &iowrites;
        break;

    case EVT_INT:
        list = &ints;
        plugin_free_int (callback);
        break;

    default:
        fprintf (stderr, "Error: attempted to free non-implemented resource. Ignoring\n");
        return;
    }


    /* look for callback in callback list */

    if (*list == callback)
        *list = callback->next_type;
    else
        for (next = *list; next; next = next->next_type)
            if (next->next_type == callback)
                next->next_type = callback->next_type;


    /* look for callback in plugin list */

    if (callback->plugin->callbacks == callback)
        callback->plugin->callbacks = callback->next_plug;
    else
        for (next = callback->plugin->callbacks; next; next = next->next_plug)
            if (next->next_plug == callback)
                next->next_plug = callback->next_type;


    /* clean up callback */

    free (callback);


    /* we're done!! */

    return;
}


int
plugin_alloc_int (event_t event, handler_t handler, int base, int range)
{
    int i;

    for (i = base; i < base+range; i++)
        if (vm_alloc_intr (i))
        {
            int j;

            for (j = base; j < i; j++)
                vm_release_intr (j);

            return 1;
        }

    return 0;
}

void
plugin_free_int (callback_t *callback)
{
    int j;

    for (j = callback->base; j < callback->base+callback->range; j++)
        vm_release_intr (j);

    return;
}



/************************************************************************/
/* Event handling (plugin needs to handle evt / plugin generates evt)   */
/************************************************************************/

void
plugin_emulate (event_t event, int data, int op_size, int count, void *loc)
{
    switch (event)
    {
    case EVT_INPORT:
        plugin_emulate_inport (data, op_size, count, loc);
        break;

    case EVT_OUTPORT:
        plugin_emulate_outport (data, op_size, count, loc);
        break;

    case EVT_INT:
        plugin_emulate_int (data, op_size, count, loc);
        break;

    default:
        fprintf (stderr, "Error: emulation of event %d not supported. Ignoring\n", (int) event);
        return;
    }
}


void
plugin_emulate_inport (int data, int op_size, int count, void *loc)
{
    callback_t *next = ioreads;

    do
    {
        for (; next; next = next->next_type)
            if (SUBSET(next->base,next->range,data,op_size))
                break;

        if (!next)
            break;

        next->handler (EVT_INPORT, data, op_size, count, loc);

        next = next->next_type;
    } while (!conflict[EVT_INPORT]);

    return;
}


void
plugin_emulate_outport (int data, int op_size, int count, void *loc)
{
    callback_t *next = iowrites;

    do
    {
        for (; next; next = next->next_type)
            if (SUBSET(next->base,next->range,data,op_size))
                break;

        if (!next)
            break;

        next->handler (EVT_OUTPORT, data, op_size, count, loc);

        next = next->next_type;
    } while (!conflict[EVT_OUTPORT]);

    return;
}

void
plugin_emulate_int (int data, int op_size, int count, void *loc)
{
    callback_t *next = ints;
    int signal;

    UNUSED(count);
    UNUSED(loc);
    UNUSED(op_size);

    do
    {
        for (; next; next = next->next_type)
            if (SUBSET(next->base,next->range,data,1))
                break;

        if (!next)
            break;

        if ((signal = next->handler (EVT_INT, data, 0, 0, NULL)))
            vm_reflect (signal);

        next = next->next_type;
    } while (!conflict[EVT_INT]);

    return;
}

/************************************************************************/
/* Interrupt handling                                                   */
/************************************************************************/

static iac_handler_t plugin_iac_handler = NULL;

void
plugin_set_intr (int intr)
{
    vm_set_intr (intr);
}

void
plugin_announce_iac_handler (iac_handler_t handler)
{
    if (plugin_iac_handler)
    {
        fprintf (stderr, "Two interrupt sources announced!\n");
        vm_abort();
    }

    plugin_iac_handler = handler;
}

int
plugin_acknowledge_intr (void)
{
    if (!plugin_iac_handler)
    {
        fprintf (stderr, "No interrupt source was announced!\n");
        vm_abort();
    }

    return plugin_iac_handler();
}


/************************************************************************/
/* VM control                                                           */
/************************************************************************/

void
plugin_abort (void)
{
    vm_abort ();
    return;
}

