/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The FreeMWare developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <signal.h>
#include <errno.h>

#include "bochs.h"
#include "elf.h"
#include "decode.h"

bx_options_t bx_options;
bx_debug_t   bx_dbg;
bx_pc_system_c bx_pc_system;

plugin_t *bx_plugin;

static void timer_handler(int i);
static void replay_vga_io_log(void);

#define UPDATE_INTERVAL 10000

extern "C" {

static int bx_freemware_iac(void);

int
plugin_init(plugin_t *plugin, int argc, char *argv[])
{
  struct itimerval timer;
  struct sigaction sg_act;

  UNUSED(argc);
  UNUSED(argv);

  // Announce our INTR acknowledge handler
  plugin_announce_iac_handler(bx_freemware_iac);

  memset(&sg_act, 0, sizeof(sg_act));
  sg_act.sa_handler = timer_handler;
  sg_act.sa_flags = SA_RESTART;
  sigaction(SIGALRM,&sg_act,NULL);

  bx_plugin = plugin;
  bx_options.private_colormap = 0;
  bx_options.mouse_enabled = 0;
  bx_options.keyboard_serial_delay = 200;
  bx_options.vga_update_interval = UPDATE_INTERVAL;
  memset(&bx_dbg, 0, sizeof(bx_dbg));

  // The bochs pc_system framework has a notion of how
  // fast the emulation runs, and timing is derived from that.
  // If I tell it we're running at 1Mip, then we can pass
  // it the number of elapsed microseconds as the number of ticks.
  bx_pc_system.init_ips(1000000);
  bx_devices.init();
  bx_pc_system.start_timers();

  // Since we're not booting or running the BIOS, replay an
  // IO sequence to the VGA adapter, captured from a previous
  // boot of bochs.
  replay_vga_io_log();

  memset(&timer, 0, sizeof(timer));
  timer.it_value.tv_sec  = 0;
  timer.it_value.tv_usec = UPDATE_INTERVAL;
  timer.it_interval.tv_sec  = 0;
  timer.it_interval.tv_usec = UPDATE_INTERVAL;
  setitimer(ITIMER_REAL, &timer, NULL);

  return 0;
}

void plugin_fini(void)
{
  struct sigaction sg_act;

  // Now we should disable our timer handler, or it can try
  // to access nonexistent mapped memory => SIGSEGV
  memset(&sg_act, 0, sizeof(sg_act));
  sg_act.sa_handler = SIG_IGN;
  sigaction(SIGALRM,&sg_act,NULL);

  // Tell the GUI to shut down 
  bx_gui.exit();       
}

int bx_freemware_io(event_t event, int data, int op_size, int count, void *loc)
{
  int i;

  switch (event)
  {
  case EVT_INPORT:
    switch(op_size)
    {
    case 1:
      for (i=0; i<count; i++)
        ((Bit8u *)loc)[i] = bx_pc_system.inp((Bit16u)data, (unsigned)op_size);
      break;
    case 2:
      for (i=0; i<count; i++)
        ((Bit16u *)loc)[i] = bx_pc_system.inp((Bit16u)data, (unsigned)op_size);
      break;
    case 4:
      for (i=0; i<count; i++)
        ((Bit32u *)loc)[i] = bx_pc_system.inp((Bit16u)data, (unsigned)op_size);
      break;
    }
    break;

  case EVT_OUTPORT:
    switch(op_size)
    {
    case 1:
      for (i=0; i<count; i++)
        bx_pc_system.outp((Bit16u)data, ((Bit8u *)loc)[i], (unsigned)op_size);
      break;
    case 2:
      for (i=0; i<count; i++)
        bx_pc_system.outp((Bit16u)data, ((Bit16u *)loc)[i], (unsigned)op_size);
      break;
    case 4:
      for (i=0; i<count; i++)
        bx_pc_system.outp((Bit16u)data, ((Bit32u *)loc)[i], (unsigned)op_size);
      break;
    }
    break;

  default:
    bx_panic("bx_freemware_io: unknown event trigger requested\n");
    break;
  }

  return 0;
}

static int bx_freemware_iac(void)
{
  return bx_pc_system.IAC();
}

} /* extern "C" */

void bx_freemware_set_intr(int intr)
{
  plugin_set_intr(intr);
}

void timer_handler(int i)
{
  UNUSED(i);

  // For now, I just copy the contents of the guest's physical
  // memory representing the video memory area for text mode,
  // into the VGA framebuffer.  Once we have virtualization of
  // memory mapped IO space, we won't need to do this hack.

  memcpy(&bx_devices.vga->s.vga_memory[0x18000], &ptr[0xb8000], 2000*2);
  bx_devices.vga->s.vga_mem_updated = 1;
  bx_pc_system.tickn(UPDATE_INTERVAL);
}


  void
bx_panic(char *fmt, ...)
{
  va_list ap;

#if 0
  if (bx_logfd) {
    fprintf(bx_logfd, "panic, ");

    va_start(ap, fmt);
    vfprintf(bx_logfd, fmt, ap);
    va_end(ap);
    }
#endif
  fprintf(stderr, "bochs: panic, ");
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);

  plugin_abort ();
}

  void
bx_printf(char *fmt, ...)
{
  va_list ap;

#if 0
  if (bx_logfd) {
    va_start(ap, fmt);
    vfprintf(bx_logfd, fmt, ap);
    va_end(ap);
    }
#endif
  fprintf(stderr, "bochs: ");
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
}

  void
replay_vga_io_log(void)
{
  // Replay IO from log to initialize IO devices to
  // a reasonable state needed for the OS.  This is done
  // in lieu of running the 16-bit BIOS to init things,
  // since we want to test straight 32bit stuff for
  // freemware.

  FILE *fp;

  fp = fopen("vga_io.log", "r");

  if (fp == NULL) {
    bx_panic("could not open IO init file.\n");
    }

  while (1) {
    unsigned len, op, port, val;
    int ret;
    ret = fscanf(fp, "%u %u %x %x\n",
      &len, &op, &port, &val);
    if (ret != 4) {
      bx_panic("could not open IO init file.\n");
      }
    if (op == 0) {
      // read
      (void) bx_devices.inp(port, len);
      }
    else if (op == 1) {
      // write
      bx_devices.outp(port, val, len);
      }
    else {
      bx_panic("bad IO op in init filen");
      }
    if (feof(fp)) break;
    }
}
