/*
 * Low level interface to virtual CPU, for the remote server for GDB.
 * Copyright (C) 2000  Ramon van Handel <vhandel@chem.vu.nl>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <signal.h>
#include "server.h"

#ifndef SIGEMT
#define SIGEMT  7
#endif

enum registers                   /* GDB register enum structure            */
{
    EAX, ECX, EDX, EBX,
    ESP, EBP, ESI, EDI,
    EIP, EFL, CS,  SS,
    DS,  ES,  FS,  GS,
};

struct {                         /* GDB register context structure         */
    unsigned int eax;
    unsigned int ecx;
    unsigned int edx;
    unsigned int ebx;
    unsigned int esp;
    unsigned int ebp;
    unsigned int esi;
    unsigned int edi;
    unsigned int eip;
    unsigned int eflags;
    unsigned int cs;
    unsigned int ss;
    unsigned int ds;
    unsigned int es;
    unsigned int fs;
    unsigned int gs;
} _registers;

char *registers = (char *) &_registers;


int fwd_signal;

int
mywait (event_t event, int data, int op_size, int count, void *loc)
{
    unsigned char sigval, status, ret;

    UNUSED(event);
    UNUSED(op_size);
    UNUSED(count);
    UNUSED(loc);

    disable_async_io ();

    fetch_inferior_registers (0);

    status = 'T';

    switch (data)
    {
    case 0  : sigval = SIGFPE;  break; /* divide by zero              */
    case 1  : sigval = SIGTRAP; break; /* debug exception             */
    case 2  : sigval = SIGBUS;  break; /* NMI fault                   */
    case 3  : sigval = SIGTRAP; break; /* breakpoint                  */
    case 4  : sigval = SIGUSR1; break; /* into instruction (overflow) */
    case 5  : sigval = SIGUSR1; break; /* bound instruction           */
    case 6  : sigval = SIGILL;  break; /* Invalid opcode              */
    case 7  : sigval = SIGFPE;  break; /* coprocessor not available   */
    case 8  : sigval = SIGEMT;  break; /* double fault                */
    case 9  : sigval = SIGSEGV; break; /* coprocessor segment overrun */
    case 10 : sigval = SIGSEGV; break; /* Invalid TSS                 */
    case 11 : sigval = SIGSEGV; break; /* Segment not present         */
    case 12 : sigval = SIGSEGV; break; /* stack exception             */
    case 13 : sigval = SIGSEGV; break; /* general protection          */
    case 14 : sigval = SIGSEGV; break; /* page fault                  */
    case 16 : sigval = SIGEMT;  break; /* coprocessor error           */
    case -1 : sigval = SIGINT;  break; /* break command from gdb      */
    default : sigval = SIGEMT;  break; /* "software generated"        */
    }

    ret = gdb_converse (sigval, status);

    switch (ret)
    {
    case 0 : return 0;          break; /* resume execution normally   */
    case 1 : vm_abort ();       break; /* fatal; abort VM             */
    case 2 : return fwd_signal; break; /* fwd signal back to guest    */
    }

    return 0;   /* shouldn't get here */
}

void
myresume (int step, int signal)
{
    if (step)
        context.eflags |= 0x100;
    else
        context.eflags &= ~0x100;

    fwd_signal = signal;

    enable_async_io ();

    return;
}


static void
fetch_register (int regno)
{
    switch (regno)
    {
    case EAX:
        _registers.eax = context.eax;  break;
    case ECX:
        _registers.ecx = context.ecx;  break;
    case EDX:
        _registers.edx = context.edx;  break;
    case EBX:
        _registers.ebx = context.ebx;  break;
    case ESP:
        _registers.esp = context.esp;  break;
    case EBP:
        _registers.ebp = context.ebp;  break;
    case ESI:
        _registers.esi = context.esi;  break;
    case EDI:
        _registers.edi = context.edi;  break;
    case EIP:
        _registers.eip = context.eip;  break;
    case EFL:
        _registers.eflags = context.eflags;  break;
    case CS:
        _registers.cs = context.cs;  break;
    case SS:
        _registers.ss = context.ss;  break;
    case DS:
        _registers.ds = context.ds;  break;
    case ES:
        _registers.es = context.es;  break;
    case FS:
        _registers.fs = context.fs;  break;
    case GS:
        _registers.gs = context.gs;  break;
    }
}

void
fetch_inferior_registers (int regno)
{
    if (regno == -1 || regno == 0)
	for (regno = 0; regno < NUM_REGS - NUM_FREGS; regno++)
	    fetch_register (regno);
    else
	fetch_register (regno);
}

void
store_inferior_registers (int regno)
{
    if (regno >= 0)
    {
        switch (regno)
        {
        case EAX:
            context.eax = _registers.eax;  break;
        case ECX:
            context.ecx = _registers.ecx;  break;
        case EDX:
            context.edx = _registers.edx;  break;
        case EBX:
            context.ebx = _registers.ebx;  break;
        case ESP:
            context.esp = _registers.esp;  break;
        case EBP:
            context.ebp = _registers.ebp;  break;
        case ESI:
            context.esi = _registers.esi;  break;
        case EDI:
            context.edi = _registers.edi;  break;
        case EIP:
            context.eip = _registers.eip;  break;
        case EFL:
            context.eflags = _registers.eflags;  break;
        case CS:
            context.cs = _registers.cs;  break;
        case SS:
            context.ss = _registers.ss;  break;
        case DS:
            context.ds = _registers.ds;  break;
        case ES:
            context.es = _registers.es;  break;
        case FS:
            context.fs = _registers.fs;  break;
        case GS:
            context.gs = _registers.gs;  break;
        }
    }
    else
	for (regno = 0; regno < NUM_REGS - NUM_FREGS; regno++)
	    store_inferior_registers (regno);
}

int
read_inferior_memory (int memaddr, char *myaddr, int len)
{
    if (memaddr+len > vm_conf.max_memory << 20)
        return -1;

    memcpy (myaddr, ptr + memaddr, len);
    return 0;
}

int
write_inferior_memory (int memaddr, char *myaddr, int len)
{
    if (memaddr+len > vm_conf.max_memory << 20)
        return -1;

    memcpy (ptr + memaddr, myaddr, len);
    return 0;
}
