/*
 * Main code for remote server for GDB.
 * Copyright (C) 1989, 1993 Free Software Foundation, Inc.
 * Copyright (C) 2000       Ramon van Handel <vhandel@chem.vu.nl>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include "server.h"
#include "plugin.h"

int   cont_thread;
int   general_thread;
int   extended_protocol;
int   thread_from_wait;
int   old_thread_from_wait;
int   port_num;

callback_t *ice_exceptions;

int
plugin_init (plugin_t *plugin, int argc, char *argv[])
{
    if (argc < 1)
	error ("ice: usage: plugin = plugin-ice.so port");

    port_num = atoi (argv[0]);

    if (!(ice_exceptions = plugin_alloc (plugin, EVT_INT, mywait, 0, 17)))
    {
        fprintf(stderr, "ice: failed allocation of exceptions; aborting\n");
        return 1;
    }

    fetch_inferior_registers (0);

    fprintf (stderr, "ice: please initialize remote debugger on port %d\n", port_num);
    if (gdb_converse (SIGTRAP, 'T'))
    {
        fprintf(stderr, "ice: initialization failed; refraining from further action\n");
        plugin_free (ice_exceptions);
        return 1;
    }

    return 0;
}

void
plugin_fini (void)
{
    plugin_free (ice_exceptions);
    remote_close ();
    return;
}


char  own_buf[2000], mem_buf[2000];
int   resume = 0;

int
gdb_converse (int signal, char status)
{
    char  ch;
    int   i = 0;
    unsigned int mem_addr, len;

    while (1)
    {
	if (resume)
	{
	    prepare_resume_reply (own_buf, status, signal);
	    resume = 0;
	    putpkt (own_buf);
	}
        else
            remote_open (port_num);

	setjmp (toplevel);

	while (getpkt (own_buf) > 0)
	{
	    unsigned char sig;

	    i = 0;
	    ch = own_buf[i++];
	    switch (ch)
	    {
	    case '!':
		extended_protocol = 1;
		prepare_resume_reply (own_buf, status, signal);
		break;

	    case '?':
		prepare_resume_reply (own_buf, status, signal);
		break;

	    case 'H':
		switch (own_buf[1])
		{
		case 'g':
		    general_thread = strtol (&own_buf[2], NULL, 16);
		    write_ok (own_buf);
		    fetch_inferior_registers (0);
		    break;
		case 'c':
		    cont_thread = strtol (&own_buf[2], NULL, 16);
		    write_ok (own_buf);
		    break;
		default:
		    /*
		     * Silently ignore it so that gdb can extend the protocol
		     * without compatibility headaches.
		     */
		    own_buf[0] = '\0';
		    break;
		}
		break;

	    case 'g':
		convert_int_to_ascii (registers, own_buf, REGISTER_BYTES);
		break;

	    case 'G':
		convert_ascii_to_int (&own_buf[1], registers, REGISTER_BYTES);
		store_inferior_registers (-1);
		write_ok (own_buf);
		break;

	    case 'm':
		decode_m_packet (&own_buf[1], &mem_addr, &len);
		if (read_inferior_memory (mem_addr, mem_buf, len) != 0)
		    write_enn (own_buf, 3);
		else
		    convert_int_to_ascii (mem_buf, own_buf, len);
		break;

	    case 'M':
		decode_M_packet (&own_buf[1], &mem_addr, &len, mem_buf);
		if (write_inferior_memory (mem_addr, mem_buf, len) == 0)
		    write_ok (own_buf);
		else
		    write_enn (own_buf, 3);
		break;

	    case 'C':
		convert_ascii_to_int (own_buf + 1, &sig, 1);
		myresume (0, sig);
		resume = 1;
		return 2;

	    case 'S':
		convert_ascii_to_int (own_buf + 1, &sig, 1);
		myresume (1, sig);
		resume = 1;
		return 2;

	    case 'c':
		myresume (0, 0);
		resume = 1;
		return 0;

	    case 's':
		myresume (1, 0);
		resume = 1;
		return 0;

	    case 'k':
		fprintf (stderr, "ice: killing inferior\n");

		/*
		 * When using the extended protocol, we start up a new
		 * debugging session.   The traditional protocol will
		 * exit instead.
		 */
#if 0
		if (extended_protocol)
		{
		    write_ok (own_buf);
		    fprintf (stderr, "ice: restarting system\n");

		    reboot ();

		    /* Wait till we are at 1st instruction in prog.  */
		    signal = mywait (&status);
		    goto restart;
		    break;
		}
		else
		{
#endif
		    return 1;
#if 0
		}
#endif

	    case 'T':
		write_ok (own_buf);
		break;

	    case 'R':
		/*
		 * Restarting the inferior is only supported in the
		 * extended protocol.
		 */
#if 0
		if (extended_protocol)
		{
		    write_ok (own_buf);
		    fprintf (stderr, "ice: restarting system\n");

		    reboot ();

		    /* Wait till we are at 1st instruction in prog.  */
		    signal = mywait (&status);
		    goto restart;
		    break;
		}
		else
		{
#endif
		    /*
		     * It is a request we don't understand.  Respond with an
		     * empty packet so that gdb knows that we don't support this
		     * request.
		     */
		    own_buf[0] = '\0';
		    break;
#if 0
		}
#endif

	    default:
		/*
		 * It is a request we don't understand.  Respond with an
		 * empty packet so that gdb knows that we don't support this
		 * request.
		 */
		own_buf[0] = '\0';
		break;
	    }

	    putpkt (own_buf);
	}

	/*
         * We come here when getpkt fails.
	 * 
	 * For the extended remote protocol we exit (and this is the only
	 * way we gracefully exit!).
	 * 
	 * For the traditional remote protocol close the connection,
	 * and re-open it at the top of the loop.
         */
	if (extended_protocol)
	{
	    remote_close ();
            return 1;
	}
	else
	{
	    fprintf (stderr, "ice: remote side has terminated connection.\n");
            fprintf (stderr, "ice: ICE plugin will reopen the connection.\n");
	    remote_close ();
	}
    }
}
