/*
 *  FreeMWare: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2000  The FreeMWare developers team
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>

#include "freemware.h"
#include "user.h"
#include "plugin.h"
#include "decode.h"
#include "bin.h"
#include "emulation.h"



/************************************************************************/
/* Structures / Variables                                               */
/************************************************************************/

/* Global data */

guest_context_t context;	/* This is the guest context structure   */

char *ptr = NULL;		/* Pointer to the guest virtual phys mem */


/* Local data */

static int filehdl;		/* File handle for /dev/freemware        */

static void (*vm_sig) (void);   /* Signal handler callback               */

static unsigned char int_usage[256];   /* Interrupt usage count table    */


/* Flags */

static int vm_intr = 0;         /* The INTR line has been asserted       */

static int vm_ref  = -1;        /* An interrupt is reflected to guest    */



/************************************************************************/
/* Initialization and abort code                                        */
/************************************************************************/

void
vm_kickstart (void)
{
    /* open a new VM */

    vm_init ();


    /* load guest code into the virtual physical memory and initialize guest context */

    memset (int_usage, 0, 256);
    memset (&context,  0, sizeof (context));
    bin_load (vm_conf.guest_file_name);


    /* initialize all plugins */

    fprintf (stderr, "Initializing plugins\n");
    plugin_init_all ();
}


void
vm_abort (void)
{
    /* the VM has been aborted; if requested, perform a memory dump */

    if (vm_conf.dump_vm)
	bin_dump (vm_conf.dump_file_name);



    /* deinitialize all plugins */

    fprintf (stderr, "Shutting down plugins\n");
    plugin_fini_all ();



    /* shut down the virtual machine */

    vm_fini ();



    /* we're done!! */

    exit (0);
}



/************************************************************************/
/* VM initialization and deinitialization code                          */
/************************************************************************/

void
vm_init (void)
{
    int   request, data, ret;


    /* open a new VM */

    fprintf (stderr, "Opening VM\n");
    filehdl = open ("/dev/freemware0", O_RDWR);
    if (filehdl < 0)
    {
	perror ("open");
	exit (1);
    }


    /* allocate memory from the host OS for the virtual physical memory */

    fprintf (stderr, "Allocating %dMB of physical memory in VM\n", vm_conf.max_memory);
    request = FMWALLOCVPHYS;
    data = vm_conf.max_memory;
    ret = ioctl (filehdl, request, data);
    if (ret < 0)
    {
	perror ("ioctl\n");
	close (filehdl);
        exit (1);
    }


    /* map guest virtual physical memory into user address space and zero it */

    fprintf (stderr, "Mapping virtualized physical memory into monitor\n");
    ptr = mmap (NULL, vm_conf.max_memory * 1024 * 1024, PROT_READ | PROT_WRITE,
		MAP_SHARED, filehdl, 0);
    if (ptr == (void *) -1)
    {
	perror ("mmap");
	close (filehdl);
        exit (1);
    }

    fprintf (stderr, "Zeroing virtualized physical memory\n");
    memset (ptr, 0, vm_conf.max_memory * 1024 * 1024);


    return;
}


void
vm_fini (void)
{
    int   request, data, ret;


    /* unmap the guest's virtual physical memory */

    fprintf (stderr, "Unmapping guest physical memory\n");
    if (munmap (ptr, vm_conf.max_memory * 1024 * 1024) != 0)
    {
	perror ("munmap");
	exit (1);
    }


    /* tell kernel module to clean up the VM */

    fprintf (stderr, "Tearing down VM\n");
    request = FMWTEARDOWN;
    data = 0;
    ret = ioctl (filehdl, request, data);
    if (ret < 0)
    {
	perror ("ioctl\n");
	exit (1);
    }


    /* close the connection to the kernel module */

    fprintf (stderr, "Closing VM\n");
    ret = close (filehdl);
    if (ret < 0)
    {
	perror ("close\n");
	exit (1);
    }

    return;
}




/************************************************************************/
/* The main event loop                                                  */
/************************************************************************/

void
vm_event_loop (void)
{
    int   request, data, ret;

    request = FMWRUNGUEST;
    data = (unsigned long) &context;

    for (;;)
    {
        vm_sig = NULL;

	if (vm_intr && vm_ref < 0)
	{
            if (context.eflags & FLG_IF)
            {
	        int vector = plugin_acknowledge_intr();
                if (int_usage[vector & 0xff])
                    plugin_emulate (EVT_INT, vector & 0xff, 0, 0, NULL);
                else
	            context.event_info = (MON_ACTION_RAISE_INT << 8) | (vector & 0xff);
            }
            else
            {
                /* set "Virtual Interrupt Pending", so we'll get trapped back to on STI */
                context.event_info = (MON_ACTION_SET_VIP << 8);
            }
	}

        if (vm_ref >= 0)
        {
            context.event_info = (MON_ACTION_RAISE_INT << 8) | (vm_ref & 0xff);
            vm_ref = -1;
        }

	ret = ioctl (filehdl, request, data);
	if (ret < 0)
	{
	    perror ("ioctl");
	    vm_abort();
	}

        if (vm_sig)
        {
            vm_sig ();
        }

	switch ( (context.event_info >> 8) & 0xff )
	{
	case RET_BECAUSE_REDIR:
	    break;

	case RET_BECAUSE_USEREMU:
            emulate ();
	    break;

	case RET_BECAUSE_MON_ERROR:
	    fprintf(stderr, "Fatal monitor error\n");
	    if (vm_debug_exception())
		vm_abort();
	    break;

	case RET_BECAUSE_EMERR:
	    fprintf(stderr, "Monitor emulation failure\n");
	    if (vm_debug_exception())
		vm_abort();
	    break;

	default:
	    fprintf (stderr, "Unknown RET_BECAUSE: %08x\n", context.event_info);
	    vm_abort();
	    break;
	}
    }
}



/************************************************************************/
/* Interrupt stuff                                                      */
/************************************************************************/

void
vm_set_intr (int intr)
{
    vm_intr = intr;
}

void
vm_reflect (int signal)
{
    vm_ref = signal;
}

void
vm_set_sig (void *sig)
{
    vm_sig = sig;
}

int
vm_alloc_intr (int intr)
{
    int request, data, ret;

    if (intr < 0 || intr > 255)
    {
        return 1;
    }

    request = FMWALLOCINT;
    data = intr;
    ret = ioctl (filehdl, request, data);
    if (ret < 0)
    {
        return 1;
    }

    int_usage[intr]++;
    return 0;
}

int
vm_release_intr (int intr)
{
    int request, data, ret;

    if (intr < 0 || intr > 255)
    {
        return 1;
    }

    if (--int_usage[intr])
    {
        return 0;
    }

    request = FMWRELEASEINT;
    data = intr;
    ret = ioctl (filehdl, request, data);
    if (ret < 0)
    {
        return 1;
    }

    return 0;
}



/************************************************************************/
/* Debug event handler                                                  */
/************************************************************************/

int
vm_debug_exception ()
{
    struct i386_context ctx;
    struct i386_decode instr;
    int   i, len = 8;
    char  prefix[256], text[256];


    /* dump the guest context */

    memset (&ctx, 0, sizeof (ctx));
    memset (&instr, 0, sizeof (instr));

    ctx.mode = CODE32;		/* FIXME */
    ctx.base = ptr;
    ctx.segment = context.cs;
    ctx.offset = context.eip;

    if (!i386_decode (&ctx, &instr) || !i386_decode_text (&ctx, &instr, text, vm_conf.syntax))
	strcpy (text, "???");
    else
	len = instr.length;

    sprintf (prefix, "%04X.%08X  ", context.cs, context.eip);

    for (i = 0; i < len && i < 12; i++)
	sprintf (prefix + strlen (prefix), "%02X", ((unsigned char *) ptr)[context.eip + i]);
    for (; i < 12; i++)
	strcat (prefix, "  ");


    fprintf (stderr, "\n");
    fprintf (stderr, "Abort due to exception %d at %04x:%08x\n",
	     context.event_info & 0xff, context.cs, context.eip);
    fprintf (stderr, "\n");

    fprintf (stderr, "Register dump:\n");
    fprintf (stderr, " CS:%04x SS:%04x DS:%04x ES:%04x FS:%04x GS:%04x\n",
	     context.cs, context.ss,
	     context.ds, context.es, context.fs, context.gs);
    fprintf (stderr, " EAX:%08x EBX:%08x ECX:%08x EDX:%08x\n",
	     context.eax, context.ebx, context.ecx, context.edx);
    fprintf (stderr, " ESI:%08x EDI:%08x EBP:%08x ESP:%08x EFLAGS:%08x\n",
	context.esi, context.edi, context.ebp, context.esp, context.eflags);
    fprintf (stderr, "\n");

    fprintf (stderr, "Stack dump:\n");
    for (i = 0; i < 4; i++)
	fprintf (stderr, " %08x:  %08lx %08lx %08lx %08lx\n",
		 context.esp + 16 * i,
		 *(unsigned long *) (ptr + context.esp + 16 * i),
		 *(unsigned long *) (ptr + context.esp + 16 * i + 4),
		 *(unsigned long *) (ptr + context.esp + 16 * i + 8),
		 *(unsigned long *) (ptr + context.esp + 16 * i + 12));
    fprintf (stderr, "\n");

    fprintf (stderr, "Current instruction:\n");
    fprintf (stderr, " %s %s\n", prefix, text);
    fprintf (stderr, "\n");

    return 1;
}
