/*
 *
 * JONAMA - Secure TCP Relay with SSLeay
 *
 * ====================================================================
 * Copyright (c) 1999 Henri Gomez. All rights reserved.
 *
 * --------------------------------------------------------------------
 *
 * "This product includes software developed by
 * Ralf S. Engelschall <rse@engelschall.com> for use in the
 * mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * --------------------------------------------------------------------
 */

#include "jonama.h"


void str_tolower(char *str)
{
	while (*str)
	{
        	*str = tolower(*str);
        	++str;
	}
}

/* Code from Harald Hanche-Olsen <hanche@imf.unit.no> */
static int do_double_reverse (char * phname, struct in_addr * paddr)
{
	struct hostent *	hptr;


	if ((phname == NULL) || (phname[0] == '\0'))
	{
		/* single reverse failed, so don't bother */
		return (-1);
	}

	hptr = gethostbyname(phname);

	if (hptr)
	{
		char **haddr;

		for (haddr = hptr->h_addr_list; *haddr; haddr++)
		{
			if (((struct in_addr *)(*haddr))->s_addr == paddr->s_addr)
				return (1);
		}
	}
	else
		ja_logoserr(LOG_ALLWAYS, "Can't resolve remote (gethostbyname)");

	return (-2);
}


char * check_remote(japxprofile * pwp, struct sockaddr_in * pin)
{
	struct in_addr *	laddr;
	struct hostent *	hptr;
	static char		hname[512];

	
	if (! pwp->resolveRemote)
		return (inet_ntoa(pin->sin_addr));

	laddr = &(pin->sin_addr);
        hptr = gethostbyaddr((char *)laddr, sizeof(struct in_addr), AF_INET);

        if (hptr != NULL)
	{
		strcpy(hname, (char *)hptr->h_name);
		str_tolower(hname);
		
		if (do_double_reverse(hname, laddr) >= 0)
			return (hname);
	}
	else
		ja_logoserr(LOG_ALLWAYS, "Can't resolve remote (gethostbyaddr)");

	return(NULL);
}


static int ja_connectremote(japxprofile * pwp)
{
	int			i;
	int			connectsock;
	struct sockaddr_in	connectin;
	remoteInfo *		rip;

	
	if ((connectsock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
                ja_logoserr(LOG_ALLWAYS, "Unable to create socket");
                return (-1);
        }

	for (i = 0, rip = pwp->connectTo; i < pwp->numConnects; i++, rip++)
	{
        	bzero(&connectin, sizeof(connectin));
        	connectin.sin_family = AF_INET;
        	connectin.sin_port   = htons(rip->port);
		memcpy(&connectin.sin_addr, &rip->hostaddr, sizeof(struct in_addr));

		if (connect(connectsock, (struct sockaddr *) &connectin, sizeof(connectin)) >= 0)
		{
			ja_log(LOG_NORMAL, "Connected to %s:%d", rip->hostname, rip->port);
			return (connectsock);
		}	
		
		ja_log(LOG_NORMAL, "Can't connect to %s:%d, trying next host", rip->hostname, rip->port);

	}

	ja_log(LOG_NORMAL, "Can't connect to any hosts in list, aborting...");
	return (-1);
}


static int writeBlock(SSL * pssl, int pstdsocket, char * pbuf, int p2sent, int plen)
{
	int	bpos  = 0;
	int	bsent;
	int	bsize;

	
	for (;;)
	{
		bsize = ((p2sent - bpos) > plen) ? plen : (p2sent - bpos);

		if (pssl)
			bsent = SSL_write(pssl, &pbuf[bpos], bsize);
		else
			bsent = write(pstdsocket, &pbuf[bpos], bsize);

		if (bsent != bsize)
			return(bsent);

		bpos += bsize;

		if (bpos == p2sent)
			return(p2sent);
	}
}


int ja_redirect(japxprofile * pwp, SSL * pssl, int pstdsocket, int prmt, char * pbuf)
{
	fd_set	bakfds;
	fd_set	rdirfds;
	int	cltfd;
	int	bsent;
	int	b2send;	
	int	maxfd;
	long	rmt2clt	= 0;
	long	clt2rmt	= 0;
	int	err;


	if (pssl)
		cltfd = SSL_get_fd(pssl);
	else
		cltfd = pstdsocket;

	FD_ZERO(&bakfds);
	FD_SET(cltfd, &bakfds);
	FD_SET(prmt, &bakfds);

	maxfd = ((cltfd > prmt) ? cltfd : prmt) + 1;

	for (;;)
	{
        	rdirfds = bakfds;

		err = select(maxfd, &rdirfds, NULL, NULL, NULL);

		if (err < 0)
		{
			if (errno == EINTR)
			{
				ja_log(LOG_NORMAL, "select break by signal, continuing...");
				continue;
			}

			ja_logoserr(LOG_ALLWAYS, "select failed");
			break;
		}

		if (FD_ISSET(prmt, &rdirfds))
		{
			b2send = read(prmt, pbuf, pwp->bufferSize);

			if (b2send <= 0)
			{
				if (errno == EINTR)
				{
					ja_log(LOG_NORMAL, "UNS-Remote read break by signal, continuing...");
					continue;
				}

				ja_logoserr(LOG_ALLWAYS, "UNS-Remote closed connection");
				break;
			}

			bsent = writeBlock(pssl, cltfd, pbuf, b2send, pwp->writeSize);
	
			if (bsent != b2send)
			{
				if (pssl)
					ja_logerr(LOG_ALLWAYS, "SSL-Client write error (sent only %d on %d)", bsent, b2send);
				else
					ja_logerr(LOG_ALLWAYS, "UNS-Client write error (sent only %d on %d)", bsent, b2send);

				ERR_print_errors_fp(errlogfile);
				break;
			}

			rmt2clt += b2send;
		}

        	if (FD_ISSET(cltfd, &rdirfds))
        	{
			if (pssl)
            			b2send = SSL_read(pssl, pbuf, pwp->bufferSize);
			else
				b2send = read(cltfd, pbuf, pwp->bufferSize);

			if (b2send <= 0)
			{
				if (errno == EINTR)
				{
					if (pssl)
						ja_log(LOG_NORMAL, "SSL-Client read break by signal, continuing...");
					else
						ja_log(LOG_NORMAL, "UNS-Client read break by signal, continuing...");

					continue;
				}

				if (pssl)
					ja_logerr(LOG_ALLWAYS, "SSL-Client closed connection");
				else
					ja_logerr(LOG_ALLWAYS, "UNS-Client closed connection");

				ERR_print_errors_fp(errlogfile);
				break;
			}

			bsent = write(prmt, pbuf, b2send);

			if (bsent != b2send)
			{
				ja_logerr(LOG_ALLWAYS, "UNS-Remote write error (sent only %d on %d)", bsent, b2send);
				break;
			}

			clt2rmt += b2send;
		}
	}

	ja_log(LOG_NORMAL, "End of redirection: [clt->rmt : %ld b] [rmt->clt : %ld b]", clt2rmt, rmt2clt);
	return (0);
}

int ja_client(japxprofile * pwp, int psocket, struct sockaddr_in * pin)
{
	SSL *	sslsocket;
	int	remotesocket;
	int	err;
	char *	buf;
	char *	hname;


	mode = T_CLIENT;

	if ((hname = check_remote(pwp, pin)) == NULL)
	{
		ja_logerr(LOG_ALLWAYS, "Remote name query failed");
		exit(1);
	}

	ja_log(LOG_NORMAL, "Connection from %s for service %s", hname, pwp->name);

	
	if (pwp->sslProfil)
	{
		sslsocket = SSL_new(pwp->sslProfil->context);
		SSL_set_app_data(sslsocket, pwp->sslProfil);
       		SSL_set_fd(sslsocket, psocket);

		err = SSL_accept(sslsocket);

	       	if (err <= 0)
		{
			ja_logerr(LOG_ALLWAYS, "SSL authentification failed: %d", err);
       			ERR_print_errors_fp(errlogfile);
			exit(1);
		}

		ja_log(LOG_ALLWAYS, "SSL authentification accepted: %d", err);
		ja_DumpSessionInfo(SSL_get_session(sslsocket));
	}
	else
	{
		sslsocket = NULL;
		ja_log(LOG_ALLWAYS, "Unsecure connection accepted");
	}
		
	if ((remotesocket = ja_connectremote(pwp)) < 0)
	{
		ja_logerr(LOG_ALLWAYS, "Connection to remote(s) impossible, aborting");
		SSL_free(sslsocket);
		close(psocket);
		exit(1);
	}

	if ((buf = malloc(pwp->bufferSize)) == NULL)
	{
		ja_logoserr(LOG_ALLWAYS, "can't allocate memory");
		return (-2);
	}

	ja_redirect(pwp, sslsocket, psocket, remotesocket, buf);

	free(buf);

	if (pwp->sslProfil)
		SSL_free(sslsocket);

	close(psocket); 
	close(remotesocket);

	exit(1);
}



