/*
 *
 * JONAMA - Secure TCP Relay with SSLeay
 *
 * ====================================================================
 * Copyright (c) 1999 Henri Gomez. All rights reserved.
 *
 * --------------------------------------------------------------------
 *
 * "This product includes software developed by
 * Ralf S. Engelschall <rse@engelschall.com> for use in the
 * mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * --------------------------------------------------------------------
 */

#include "jonama.h"

static jasslprofile *	curssl;
static japxprofile *	curproxy;

#define SCANOUT		0
#define	SCANINSSL	1
#define	SCANINPROXY	2

/* 0=OUT 1=ssl 2=proxy */

static int		inlocal = SCANOUT;


static uid_t uname2id(const char *name)
{
	struct passwd *ent;

	if (name[0] == '#')
        	return (atoi(&name[1]));

	if (!(ent = getpwnam(name)))
	{
        	fprintf(stderr, "bad user name %s\n", name);
        	exit(1);
    	}

	return (ent->pw_uid);
}


gid_t gname2id(const char *name)
{
	struct group *ent;

	if (name[0] == '#')
		return (atoi(&name[1]));

	if (!(ent = getgrnam(name)))
	{
        	fprintf(stderr, "bad group name %s\n", name);
        	exit(1);
	}

	return (ent->gr_gid);
}


static char * allocString(char * orgstr)
{
	char *	ptr;

	ptr = malloc(strlen(orgstr) + 1);

	if (ptr == NULL)
	{
		fprintf(stderr, "Can't allocate mem for str");
		exit(1);
	}

	return (ptr);
}


static char * copyString(char * orgstr)
{
	char * ptr;

	ptr = allocString(orgstr);
	strcpy(ptr, orgstr);
	return (ptr);
}


static int setString(char ** dststr, char * orgstr)
{
	if (*dststr)
		free(*dststr);

	*dststr = copyString(orgstr);
	return(strlen(orgstr));
}


static int getTokens(char * line, char * tokens[], int maxtokens)
{
		int	i;
		int	pos = 0;
		int	len = strlen(line);
		int	got = 0;

	
	for (i = 0; i < len; i++, line++)
	{
		if (*line <= ' ')
		{
			got = 0;
			*line = 0;
		}
		else
		{
			if (! got)
			{
				tokens[pos] = line;
				pos++;
				got = 1;
			}
		}
	}

/*
	for (i = 0; i < pos; i++)
		printf("%s %d\n", tokens[i], strlen(tokens[i]));
*/

	return (pos);
}


static int buildRemoteInfo(char * str, remoteInfo * rinfo)
{
	struct hostent *	connectent;
	char *			ptr;
	char 			hstr[512];
	int			hlen;


	ptr = rindex(str, ':');

	if (str[0] == ':')
		strcpy(hstr, "localhost");
	else
	{
		hlen = ptr - str;
                strncpy(hstr, str, hlen);
                hstr[hlen] = 0;
	} 			
		
	if ((connectent = gethostbyname(hstr)) == NULL)	
	{
		ja_logoserr(LOG_ALLWAYS, "Can't resolv %s", hstr);
		return (-1);
	}

	setString(&rinfo->hostname, hstr);
	memcpy(&rinfo->hostaddr, connectent->h_addr_list[0], sizeof(struct in_addr));
	rinfo->port = atoi(&ptr[1]);	
	
	return (1);
}


static int decodeTime(japxprofile * ppx, char * ptime)
{
	char *	ptr;


	ptr = index(ptime, '-');

	if (!ptr)
	{
		fprintf(stderr, "acceptTime form is HHMM-HHMM (%s)\n", ptime);
		return(0);
	}

	ppx->acceptTStart = atol(ptime);
	ppx->acceptTEnd   = atol(&ptr[1]);
	return (1);
}


static int day2int(char * pday)
{
	char * 	ldays[] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
	int	i;

	for (i = 0; i < sizeof(ldays); i++)
	{
		if (! strncmp(pday, ldays[i], 3))
			return(i);
	}

	fprintf(stderr, "invalid acceptDays (%s)\n", pday);
	return(-1);
}

static int decodeDays(char * pdate)
{
	int	ldayint = 0;
	int	ldayone;
	char *	lhdl;
	char *	lptr;

	
	lhdl = pdate;

	if (! index(pdate, '-'))
	{
		fprintf(stderr, "acceptDays form is DDD-DDD-DDD-DDD-DDD (%s)\n", pdate);
		return(-1);
	}

	if ((ldayone = day2int(pdate)) < 0)
		return(-1);

	ldayint = 1 << ldayone;

	for (;;)
	{
		lptr = strsep(&lhdl, "-");

		if (lptr)
		{
			if ((ldayone = day2int(lptr)) < 0)
				return(-1);

			ldayint |= 1 << ldayone;
		}
		else
			return(ldayint);
	}
}


static int processLine(char * line, int lnum)
{
	char *		fields[32];
	int		nfields;
	int		greu;
	char *		ptr;


	nfields = getTokens(line, fields, 32);

	if (nfields < 1)
		return(0);

	if (fields[0][0] == '#')
		return (0);
/*
 * Look for global vars
 */

/*
 *	printf("[%s] %d [%s] %d\n", fields[0], strlen(fields[0]), fields[1], strlen(fields[1]));
 */
	if (inlocal == SCANOUT)
	{
		if (nfields < 2)
			return (0);

		if (! strcasecmp(fields[0], "rootpath"))
			return(setString(&chrootPath, fields[1]));
	
		if (! strcasecmp(fields[0], "logpath"))
			return(setString(&logPath, fields[1]));

		if (! strcasecmp(fields[0], "logfile"))
			return(setString(&logFileName, fields[1]));

		if (! strcasecmp(fields[0], "errorlogfile"))
			return(setString(&errorFileName, fields[1]));

		if (! strcasecmp(fields[0], "user"))
		{
			user = uname2id(fields[1]);
			return(1);
		}

		if (! strcasecmp(fields[0], "group"))
		{
			group = gname2id(fields[1]);
			return(1);
		}

		if (! strcasecmp(fields[0], "sslprofile"))
		{
			if ((nfields < 3) || (strcasecmp(fields[2], "{")))
			{
				fprintf(stderr, "bad SSLProfile declaration at line %d\n", lnum);
				return(0);
			}

			inlocal = SCANINSSL;
			curssl = &sslprofiles_pool[numSslp++];
			
			curssl->name = copyString(fields[1]);
			return(1);
		}

		if (! strcasecmp(fields[0], "proxyprofile"))
		{
			if ((nfields < 3) || (strcasecmp(fields[2], "{")))
			{
				fprintf(stderr, "bad Proxyprofile declaration at line %d\n", lnum);
				return(0);
			}

			inlocal = SCANINPROXY;
			curproxy = &pxprofiles_pool[numPxp++];

			curproxy->name = copyString(fields[1]);
			return(1);
		}
	}
	else
	{
		if (! strcasecmp(fields[0], "}"))
		{
			if (inlocal == SCANINSSL)
			{
				if (curssl->certificateFile == NULL)
					curssl->certificateFile = JONAMA_DEF_CERT_PATH;

				if (curssl->privateKeyFile == NULL)
					curssl->privateKeyFile = JONAMA_DEF_KEY_PATH;
			}
			else
			{
				if (curproxy->writeSize > curproxy->bufferSize)
					curproxy->writeSize = curproxy->bufferSize;
			}

			inlocal = SCANOUT;
			return (1);
		}

		if (nfields < 2)
			return (0);

		if (inlocal == SCANINSSL)
		{
			if (! strcasecmp(fields[0], "certificatefile"))
				return(setString(&curssl->certificateFile, fields[1]));

			if (! strcasecmp(fields[0], "privatekeyfile"))
				return(setString(&curssl->privateKeyFile, fields[1]));

			if (! strcasecmp(fields[0], "capath"))
				return(setString(&curssl->CAPath, fields[1]));

			if (! strcasecmp(fields[0], "cafile"))
				return(setString(&curssl->CACertificateFile, fields[1]));

			if (! strcasecmp(fields[0], "cyphers"))
				return(setString(&curssl->cyphers, fields[1]));
	
			if (! strcasecmp(fields[0], "verifydepth"))
			{
				curssl->verifyDepth = atoi(fields[1]);
				return (1);
			}

			if (! strcasecmp(fields[0], "keylen"))
			{
				curssl->keylen = atoi(fields[1]);
				return (1);
			}

			if (! strcasecmp(fields[0], "timeout"))
			{
				curssl->timeOut = atol(fields[1]);
				return (1);
			}

			if (! strcasecmp(fields[0], "verifymode"))
			{
				if (! strcasecmp(fields[1], "none"))
					curssl->verifyMode = SSL_CVERIFY_NONE;
				else if (! strcasecmp(fields[1], "require"))
					curssl->verifyMode = SSL_CVERIFY_REQUIRE;
				else if (! strcasecmp(fields[1], "optional"))
					curssl->verifyMode = SSL_CVERIFY_OPTIONAL;
				else if (! strcasecmp(fields[1], "optionalnoca"))
					curssl->verifyMode = SSL_CVERIFY_OPTIONAL_NO_CA;
				else
				{
					fprintf(stderr, "unknown verify mode %s at line %d\n", fields[1], lnum);
					return (0);
				}	

				return(1);
			}
		}

		if (inlocal == SCANINPROXY)
		{
			if (! strcasecmp(fields[0], "usessl"))
				return(setString(&curproxy->sslName, fields[1]));

			if (! strcasecmp(fields[0], "connectto"))
			{
				if (curproxy->numConnects == MAX_CONNECTS)
				{
					fprintf(stderr, "maxConnects (%d) allready reached", MAX_CONNECTS);
					return (0);
				}

				if ((ptr = rindex(fields[1], ':')) == NULL)
				{
					fprintf(stderr, "connectto: form is host:port, got %s at line %d\n", fields[1], lnum);
					return (0);
				}

				if (atoi(&ptr[1]) <= 0)
				{
					fprintf(stderr, "connectto: port must be numeric and was (%s) at line %d\n", &ptr[1], lnum);
					return(0);
				}

				if (buildRemoteInfo(fields[1], &curproxy->connectTo[curproxy->numConnects]) > 0)
					curproxy->numConnects++;

				return (1);
			}

			if (! strcasecmp(fields[0], "accepttime"))
				return (decodeTime(curproxy, fields[1]));
				
			if (! strcasecmp(fields[0], "acceptdays")) 
			{
				greu = decodeDays(fields[1]);

				if (greu >= 0)
				{
					curproxy->acceptTDays = greu;
					return(1);
				}
				else
					return(0);
			}	

			if (! strcasecmp(fields[0], "acceptport"))
			{
				curproxy->acceptPort = atoi(fields[1]);
				return (1);
			}

			if (! strcasecmp(fields[0], "buffersize"))
			{
				curproxy->bufferSize = atoi(fields[1]);
				return (1);
			}

			if (! strcasecmp(fields[0], "writesize"))
			{
				curproxy->writeSize = atoi(fields[1]);
				return (1);
			}	

			if (! strcasecmp(fields[0], "resolve"))
			{
				curproxy->resolveRemote = atoi(fields[1]);;
				return (1);
			}

			if (! strcasecmp(fields[0], "connectmode"))
			{
				if (! strcasecmp(fields[1], "normal"))
					curproxy->connectMode = C_MODE_NORMAL;
				else if (! strcasecmp(fields[1], "balance"))
					curproxy->connectMode = C_MODE_BALANCE;
				else
				{
					fprintf(stderr, "unknown connectMode %s at line %d\n", fields[1], lnum);
					return (0);
				}

				return (1);
			}
		}
	}

	fprintf(stderr, "unknown command %s at line %d [%d]\n", fields[0], lnum, inlocal);
	return(0);
}


void ja_processConfFile(char * pathconf)
{
	FILE *	fp;
	int	line;
	char 	buf[1024];


	fp = fopen(pathconf, "r");

	if (! fp)
	{
		fprintf(stderr, "cannot locate conf file [%s], aborting...\n", pathconf);
		exit(1);
	}

	for (line = 0;;line++)
	{
		if (!fgets(buf, 1000, fp))
			break;

		if (! strlen(buf))
			continue;

		buf[strlen(buf) - 1] = 0;

		if (processLine(buf, line) < 0)
			break;
	}

	fclose(fp);
}

