/*
 *
 * JONAMA - Secure TCP Relay with SSLeay
 *
 * ====================================================================
 * Copyright (c) 1999 Henri Gomez. All rights reserved.
 *
 * --------------------------------------------------------------------
 *
 * "This product includes software developed by
 * Ralf S. Engelschall <rse@engelschall.com> for use in the
 * mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * --------------------------------------------------------------------
 */

#include "jonama.h"

/*
 * New to OpenSSL
 *
 * Yes, It's also from mod_ssl ;-)
 */

#if SSL_LIBRARY_VERSION >= 0x0920
static RSA *ja_TmpRSA(SSL *pSSL, int nExport, int nKeyLen)
#else
static RSA *ja_TmpRSA(SSL *pSSL, int nExport)
#endif
{
#if SSL_LIBRARY_VERSION >= 0x0920
    if (nExport) {
        /* It's because an export cipher is used */
        if (nKeyLen == 512)
            return(RSATmpKey512);
        else if (nKeyLen == 1024)
            return(RSATmpKey1024);
        else
            /* it's too expensive to generate on-the-fly, so keep 1024bit */
            return(RSATmpKey1024);
    }
    else {
        /* It's because a sign-only certificate situation exists */
        return(RSATmpKey1024);
    }
#else
    if (nExport)
        return(RSATmpKey512);
    else
        return(RSATmpKey1024);
#endif
}
 

void ja_DumpSessionInfo(SSL_SESSION * pSSLS)
{
	int 	i;
	char	lstr[128];
	char	lostr[128];


        if (! pSSLS)
		return;

	ja_log(LOG_NORMAL, "SSL Session Info");
	
/*
 * PROTOCOL
 */

	switch (pSSLS->ssl_version)
	{
		case SSL2_VERSION : strcpy(lstr, "SSLv2");	break;
		case SSL3_VERSION : strcpy(lstr, "SSLv3");	break;
		case TLS1_VERSION : strcpy(lstr, "TLSv1");	break;
		default		  : strcpy(lstr, "unknown");	break;
	}

	ja_log(LOG_NORMAL, "Protocol  : %s", lstr);

/*
 * CYPHERS
 */

        if (! pSSLS->cipher)
	{
                if (((pSSLS->cipher_id) & 0xff000000) == 0x02000000)
                        ja_log(LOG_NORMAL, "Cipher    : %06lX",pSSLS->cipher_id & 0xffffff);
                else
                        ja_log(LOG_NORMAL, "Cipher    : %04lX",pSSLS->cipher_id & 0xffff);
        }
        else
        	ja_log(LOG_NORMAL, "Cipher    : %s", (pSSLS->cipher == NULL) ? "unknown" : pSSLS->cipher->name);


/*
 * SESSION ID
 */

	lstr[0] = 0;

	for (i=0; i<(int)pSSLS->session_id_length; i++)
	{
		sprintf(lostr, "%02X", pSSLS->session_id[i]);
		strcat(lstr, lostr);
	}

	ja_log(LOG_NORMAL, "Session-ID: %s", lstr);

/*
 * MASTER KEY
 */

	lstr[0] = 0; 

        for (i=0; i <(int)pSSLS->master_key_length; i++)
        {
        	sprintf(lostr, "%02X", pSSLS->master_key[i]);
		strcat(lstr, lostr);
	}

	ja_log(LOG_NORMAL, "Master-Key: %s", lstr);

/*
 * KEY ARGS
 */
        if (! pSSLS->key_arg_length)
		strcpy(lstr, "None");
        else
	{
        	for (i=0; i < (int)pSSLS->key_arg_length; i++)
                {
                        sprintf(lostr, "%02X", pSSLS->key_arg[i]);
			strcat(lstr, lostr);
                }
	}

	ja_log(LOG_NORMAL, "Key-Arg   : %s", lstr);

/*
 * START-TIME
 */

        ja_log(LOG_NORMAL, "Start Time: %ld", pSSLS->time);
	
/*
 * TIME OUT
 */

	ja_log(LOG_NORMAL, "Timeout   : %ld (sec)", pSSLS->timeout);
}
 

static int ja_SSLVerify(int pok, X509_STORE_CTX * pctx)
{
	SSL *		lSSL;
	jasslprofile *	lwp;
	X509 *		lxs;
    	int 		lerrnum;
    	int 		lerrdepth;
	char *		lcp;
    	char *		lcp2;


//	ja_logerr(LOG_ALLWAYS, "ja_SSLVerify");

	lSSL  = (SSL *)X509_STORE_CTX_get_app_data(pctx);
    	lwp   = (jasslprofile *)SSL_get_app_data(lSSL);

	if (lwp == NULL)
	{
		ja_logerr(LOG_ALLWAYS, "SSLVerify: Can't get app data"); 
		return (pok);
	}

/*
 *  Get verify ingredients
 */

	lxs       = X509_STORE_CTX_get_current_cert(pctx);
	lerrnum   = X509_STORE_CTX_get_error(pctx);
	lerrdepth = X509_STORE_CTX_get_error_depth(pctx);

/*
 * Log verification information
 */

	lcp  = X509_NAME_oneline(X509_get_subject_name(lxs), NULL, 0);
    	lcp2 = X509_NAME_oneline(X509_get_issuer_name(lxs),  NULL, 0);

	ja_log(LOG_NORMAL, "Certificate Verification: depth: %d, subject: %s, issuer: %s",
				lerrdepth, (lcp != NULL ? lcp : "-unknown-"), (lcp2 != NULL ? lcp2 : "-unknown"));
	free(lcp);
	free(lcp2);

/*
 * Check for optionally acceptable non-verifiable issuer situation
 */

	if ((  lerrnum == X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT
            || lerrnum == X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN
            || lerrnum == X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY
            || lerrnum == X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE)
            && lwp->verifyMode == SSL_CVERIFY_OPTIONAL_NO_CA)
	{
		ja_log(LOG_NORMAL, "Certificate Verification: Verifiable Issuer is configured as "
				       "optional, therefore we're accepting the certificate");				
		pok = 1;
	}

/*
 * If we already know it's not ok, log the real reason
 */

	if (! pok)
	{
        	ja_logerr(LOG_ALLWAYS, "Certificate Verification: Error (%d): %s",
                		lerrnum, X509_verify_cert_error_string(lerrnum));
	}

/*
 *  Finally check the depth of the certificate verification
 */

	if (lerrdepth > lwp->verifyDepth)
	{
        	ja_logerr(LOG_ALLWAYS, "Certificate Verification: Certificate Chain too long "
					   "(chain has %d certificates, but maximum allowed are only %d)",
					   lerrdepth, lwp->verifyDepth);
        	pok = 0;
	}

/*
 * And finally signal SSLeay the (perhaps changed) state
 */
	return (pok);
}

void ja_LogTracingState(SSL * pssl, int pwhere, int prc)
{
	jasslprofile *	lwp;
    	char *		lstr;

/*
 * find corresponding server
 */

	if ((lwp = (jasslprofile *)SSL_get_app_data(pssl)) == NULL)
	{
		ja_logerr(LOG_ALLWAYS, "LogTracingState: Can't get app data");
        	return;
	}
/*
 * create the various trace messages
 */

        if (pwhere & SSL_CB_HANDSHAKE_START)
            ja_log(LOG_NORMAL, "%s: Handshake: start", SSL_LIBRARY_NAME);
        else if (pwhere & SSL_CB_HANDSHAKE_DONE)
            ja_log(LOG_NORMAL, "%s: Handshake: done", SSL_LIBRARY_NAME);
        else if (pwhere & SSL_CB_LOOP)
            ja_log(LOG_NORMAL, "%s: Loop: %s", SSL_LIBRARY_NAME, SSL_state_string_long(pssl));
        else if (pwhere & SSL_CB_READ)
            ja_log(LOG_NORMAL, "%s: Read: %s", SSL_LIBRARY_NAME, SSL_state_string_long(pssl));
        else if (pwhere & SSL_CB_WRITE)
            ja_log(LOG_NORMAL, "%s: Write: %s", SSL_LIBRARY_NAME, SSL_state_string_long(pssl));
        else if (pwhere & SSL_CB_ALERT)
	{
            lstr = (pwhere & SSL_CB_READ) ? "read" : "write";

            ja_log(LOG_NORMAL, "%s: Alert: %s:%s", SSL_LIBRARY_NAME,
                    		SSL_alert_type_string_long(prc),
                    		SSL_alert_desc_string_long(prc));
        }
        else if (pwhere & SSL_CB_EXIT)
	{
            if (! prc)
            	ja_log(LOG_NORMAL, "%s: Exit: failed in %s", SSL_LIBRARY_NAME, SSL_state_string_long(pssl));
            else if (prc < 0)
                ja_log(LOG_NORMAL, "%s: Exit: error in %s", SSL_LIBRARY_NAME, SSL_state_string_long(pssl));
        }
	else if (pwhere & SSL_CB_HANDSHAKE_DONE)
        	ja_log(LOG_NORMAL, "%s : Connection: Handshake Done", SSL_LIBRARY_NAME);

	return;
}


static int ja_FindCAList_X509NameCmp(X509_NAME **a, X509_NAME **b)
{
//	ja_logerr(LOG_ALLWAYS, "ja_FindCAList_X509NameCmp %p %p", a, b); 
    return(X509_NAME_cmp(*a, *b));
}

STACK_OF_X509_NAME * ja_setup_CAs(jasslprofile * pwp)
{
	DIR *			lDIR;
	struct dirent *		ldirentry;
	char 			lbuf[1024];
	STACK_OF_X509_NAME *	lcask;
	STACK_OF_X509_NAME * 	lsk;
	int			i;

/*
 *  CA Stack
 */
	lcask = sk_X509_NAME_new(ja_FindCAList_X509NameCmp);

/*
 *  Process CA certificate bundle file
 */

	if (pwp->CACertificateFile != NULL)
	{
        	lsk = SSL_load_client_CA_file(pwp->CACertificateFile);

        	for (i = 0; ((lsk != NULL) && (i < sk_X509_NAME_num(lsk))); i++)
		{

			ja_log(LOG_NORMAL, "CA certificate: %s",
			X509_NAME_oneline((X509_NAME *)sk_X509_NAME_value(lsk, i), 
			NULL, 0));

            		if (sk_X509_NAME_find(lcask, sk_X509_NAME_value(lsk, i)) < 0)
                		sk_X509_NAME_push(lcask,
				sk_X509_NAME_value(lsk, i));
		}
        }

/*
 *  Process CA certificate path files
 */

	if (pwp->CAPath != NULL)
	{
        	lDIR = opendir(pwp->CAPath);
	
		if (lDIR == NULL)
			ja_logoserr(LOG_ALLWAYS, "Can't access directory %s", pwp->CAPath);
	        else	
		{
			while ((ldirentry = readdir(lDIR)) != NULL)
			{
				strcpy(lbuf, pwp->CAPath);
				strcat(lbuf, "/");
				strcat(lbuf, ldirentry->d_name);

				lsk = SSL_load_client_CA_file(lbuf);

				for (i = 0; ((lsk != NULL) && (i < sk_X509_NAME_num(lsk))); i++)
				{
					ja_log(LOG_NORMAL, "CA certificate: %s",
                        				X509_NAME_oneline((X509_NAME *)sk_X509_NAME_value(lsk, i), NULL, 0));

                			if (sk_X509_NAME_find(lcask, sk_X509_NAME_value(lsk, i)) < 0)
                    				sk_X509_NAME_push(lcask, sk_X509_NAME_value(lsk, i));
            			}
        		}

			closedir(lDIR);
		}
	}

/*
 *  Cleanup
 */
    sk_X509_NAME_set_cmp_func(lcask, NULL);
    return (lcask);
}

void ja_init_ssl()
{
	SSL_load_error_strings();
	SSLeay_add_all_algorithms();
}


int ja_new_context(jasslprofile * pwp)
{
	STACK_OF_X509_NAME *	lcask;
	int			lverify;


	if (pwp == NULL)
		return (-1);

	if (RSATmpKey512 == NULL)
	{
		RSATmpKey512 = RSA_generate_key(512, RSA_F4, NULL, NULL);

		if (RSATmpKey512 == NULL)
		{
			ja_logerr(LOG_ALLWAYS, "Failed to generate temporary (512 bit) RSA private key");
			return(-11);
		}
		
		RSATmpKey1024 = RSA_generate_key(1024, RSA_F4, NULL, NULL);

		if (RSATmpKey1024 == NULL)
		{
			ja_logerr(LOG_ALLWAYS, "Failed to generate temporary (1024 bit) RSA private key");
			return(-11);
		}
	}

/*
 *  Configure SSL Level
 */

	ja_log(LOG_NORMAL, "Configuring SSL level");

	if (pwp->protocolLevel == SSL_PROTOCOL_SSLV2)
		pwp->context = SSL_CTX_new(SSLv2_server_method());
	else
        	pwp->context = SSL_CTX_new(SSLv23_server_method()); /* be more flexible */

	if (pwp->context == NULL)
	{
		ja_logerr(LOG_ALLWAYS, "Can't create context");
                ERR_print_errors_fp(errlogfile);
		return (-2);
	}

	SSL_CTX_set_timeout(pwp->context, pwp->timeOut);

	if (! (pwp->protocolLevel & SSL_PROTOCOL_SSLV2))
        	SSL_CTX_set_options(pwp->context, SSL_OP_NO_SSLv2);

	if ( !(pwp->protocolLevel & SSL_PROTOCOL_SSLV3))
		SSL_CTX_set_options(pwp->context, SSL_OP_NO_SSLv3);

	if (! (pwp->protocolLevel & SSL_PROTOCOL_TLSV1))
		SSL_CTX_set_options(pwp->context, SSL_OP_NO_TLSv1);

/*
 *  Setup Data Area and Callbacks
 */		

	ja_log(LOG_NORMAL, "Setup Data&Callbacks");

    	lverify = SSL_VERIFY_NONE;

	if (pwp->verifyMode == SSL_CVERIFY_REQUIRE)
        	lverify |= SSL_VERIFY_PEER|SSL_VERIFY_FAIL_IF_NO_PEER_CERT;

	if ((pwp->verifyMode == SSL_CVERIFY_OPTIONAL) || (pwp->verifyMode == SSL_CVERIFY_OPTIONAL_NO_CA))
        	lverify |= SSL_VERIFY_PEER;

	SSL_CTX_set_app_data(pwp->context, pwp);
	SSL_CTX_set_verify(pwp->context, lverify, ja_SSLVerify);

/*	SSL_CTX_sess_set_new_cb(pwp->context,      ja_NewSessionCacheEntry);
 *	SSL_CTX_sess_set_get_cb(pwp->context,      ja_GetSessionCacheEntry);
 *	SSL_CTX_sess_set_remove_cb(pwp->context,   ja_DelSessionCacheEntry);
 */

 	SSL_CTX_set_tmp_rsa_callback(pwp->context, ja_TmpRSA);

	SSL_CTX_set_info_callback(pwp->context,    ja_LogTracingState);

/*
 *  Setup Cyphers
 */
	if (pwp->cyphers != NULL)
	{
		ja_log(LOG_NORMAL, "Configuring SSL ciphers");

		if (! SSL_CTX_set_cipher_list(pwp->context, pwp->cyphers))
		{
			ja_logerr(LOG_ALLWAYS, "Unable to configure SSL ciphers (%s)", pwp->cyphers);
	                ERR_print_errors_fp(errlogfile);
			SSL_CTX_free(pwp->context);		
			return (-3);	
		}
	}

/*
 *  Setup Server Certificate
 */

    	if ((pwp->CACertificateFile != NULL) || (pwp->CAPath != NULL))
	{
		ja_log(LOG_NORMAL, "Configuring client authentication");

        	if (! SSL_CTX_load_verify_locations(pwp->context, pwp->CACertificateFile, pwp->CAPath))
		{
            		ja_logerr(LOG_ALLWAYS, "Unable to configure verify"
						   " locations for client authentication %s/%s",
						   pwp->CACertificateFile, pwp->CAPath);

	                ERR_print_errors_fp(errlogfile);
			SSL_CTX_free(pwp->context);
			return (-4);
		}

		if ((lcask = ja_setup_CAs(pwp)) == NULL)
		{
			ja_logerr(LOG_ALLWAYS, "Unable to determine list of available "
                    				   "CA certificates for client authentication %s/%s",
						   pwp->CACertificateFile, pwp->CAPath);

			SSL_CTX_free(pwp->context);
			return (-5);
		}

		SSL_CTX_set_client_CA_list(pwp->context, lcask);
	}	

/*
 *  Give a warning when no CAs were configured but client authentication
 *  should take place. This cannot work.
 */

	if (pwp->verifyMode == SSL_CVERIFY_REQUIRE)
	{
       		lcask = SSL_CTX_get_client_CA_list(pwp->context);

		if (sk_X509_NAME_num(lcask) == 0)
			ja_log(LOG_NORMAL, "Warning, you want to request client authentication "
					       "but no CAs are known for verification");
	}

/*
 *  Configure server certificate
 */

	ja_log(LOG_NORMAL, "Configuring server certificate");

	if (pwp->certificateFile == NULL)
	{
		ja_logerr(LOG_ALLWAYS, "SSL Certificate set");
		SSL_CTX_free(pwp->context);
		return (-6);
	}

	if (SSL_CTX_use_certificate_file(pwp->context, pwp->certificateFile, SSL_FILETYPE_PEM) <= 0)
	{
		ja_logerr(LOG_ALLWAYS, "Can't set Certificate File (%s)", pwp->certificateFile);
		ERR_print_errors_fp(errlogfile);
		SSL_CTX_free(pwp->context);
		return (-7);
	}

	if (pwp->privateKeyFile == NULL)
		pwp->privateKeyFile = pwp->certificateFile;

	if (SSL_CTX_use_PrivateKey_file(pwp->context, pwp->privateKeyFile, SSL_FILETYPE_PEM) <= 0)	
	{
		ja_logerr(LOG_ALLWAYS, "Can't set Private Key File (%s)", pwp->privateKeyFile);
		ERR_print_errors_fp(errlogfile);
		SSL_CTX_free(pwp->context);
		return (-8);
	}

	return (0);
}
