/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    cache.c
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This module manages an LBN cache for the OpenVMS CD-ROM.
 *  See file cache.h for details.
 *
 *
 *  Modification History
 *  --------------------
 *  18 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include "ods2.h"
#include "utils.h"
#include "cache.h"


/*******************************************************************************
 *
 *  Structure of the cache.
 *
 *  The cache is purged on a "least recently used" basis.
 *
 *  Each block is stored in a separate structure (a block_t). This has
 *  the bad consequence of issuing multiple I/O for contiguous blocks
 *  but it prevents the dynamic memory from being too fragmented.
 *
 *  To find a block in the cache, a hash table is used. We try to keep
 *  the depth of each hash entry less than 13. Since the maximum size
 *  of a cdrom is approximately 1,300,000 blocks (650 MB), we need
 *  100,000 entries in the hash table. On a 32-bits machine, the
 *  hash table uses 400 kB.
 *
 *  To find a block in the hash table, use the following method:
 *  - Index in hash table = LBN modulo HASH_TABLE_SIZE
 *  - Blocks are unsorted in this list.
 *
 *******************************************************************************
 */

#define HASH_TABLE_SIZE 100000

typedef struct block_s block_t;
typedef struct list_s list_t;

struct block_s {
    lbn_t lbn;                   /* Logical block number on disk */
    list_t *list;                /* List which the block belongs to */
    block_t *next_in_time;       /* Next block in time of access */
    block_t *prev_in_time;       /* Previous block in time of access */
    block_t *next_in_hash;       /* Next block in hash list */
    block_t *prev_in_hash;       /* Previous block in hash list */
    char data [ODS2_BLOCK_SIZE]; /* Content of the block */
};

struct list_s {
    lbn_t max_blocks;      /* Maximum number of blocks in the cache */
    lbn_t used_blocks;     /* Current number of blocks in the cache */
    block_t *first;        /* Least recently used block */
    block_t *last;         /* Most recently used block */
    int hit_count;         /* Number of cache hits (or statistics) */
    int miss_count;        /* Number of cache misses (or statistics) */
};

struct cache_s {
    int fd;                /* File descriptor of the cdrom device */
    int cdrom_file;        /* Boolean: the cdrom is a plain file (cd image) */
    list_t data;           /* List of files data blocks (VBN cache) */
    list_t meta;           /* List of meta data blocks (dir cache) */
    block_t **hash;        /* Hash table of LBNs */
};


/*******************************************************************************
 *
 *  This procedure creates a new cache.
 *
 *******************************************************************************
 */

cache_t init_cache (int err_flags, int fd, lbn_t max_data, lbn_t max_meta)
{
    cache_t cache;
    struct stat st;
    int err;

    /* Allocate and erase the structure */

    cache = utmalloc (sizeof (struct cache_s));
    memset (cache, 0, sizeof (struct cache_s));
    cache->fd = fd;
    cache->data.max_blocks = max_data;
    cache->meta.max_blocks = max_meta;

    /* Check the type of the cdrom "file" */

    if (fstat (fd, &st) < 0) {
        err = errno;
        error (err_flags | ERR_ERRNO, "cannot stat cdrom device\n");
        free (cache);
        errno = err;
        return NULL;
    }

    /* If it is a regular file (an image of a cdrom), we rely on the */
    /* unix cache to avoid unnecessary memory duplication. */

    cache->cdrom_file = S_ISREG (st.st_mode);

    /* Allocate a hash table if the cache is not disabled */

    if (!cache->cdrom_file && (max_data > 0 || max_meta > 0))
        cache->hash = utmalloc (sizeof (block_t*) * HASH_TABLE_SIZE);

    return cache;
}


/*******************************************************************************
 *
 *  Delete a cache and free all the associated resources.
 *
 *******************************************************************************
 */

void delete_cache (cache_t cache)
{
    block_t *block, *temp;

    /* Free all the allocated blocks */

    for (block = cache->data.first; block != NULL; block = temp) {
        temp = block->next_in_time;
        free (block);
    }

    for (block = cache->meta.first; block != NULL; block = temp) {
        temp = block->next_in_time;
        free (block);
    }

    /* Free the hash table */

    if (cache->hash != NULL)
        free (cache->hash);

    /* Free the cache structure */

    free (cache);
}


/*******************************************************************************
 *
 *  This routine returns the maximum amount of memory for the cache in bytes.
 *  Return zero if the cache is disabled.
 *
 *******************************************************************************
 */

int max_cache_size (cache_t cache)
{
    return cache->hash == NULL ? 0 :
        sizeof (*cache) + HASH_TABLE_SIZE * sizeof (block_t*) +
        (cache->data.max_blocks + cache->meta.max_blocks) * sizeof (block_t);
}


/*******************************************************************************
 *
 *  This internal routine reads one or more LBN directly from the cdrom device.
 *
 *******************************************************************************
 */

static int read_device (
    int err_flags,
    int fd,
    lbn_t lbn,
    int lbn_count,
    void *buffer)
{
    int size;
    char *current = buffer;
    int remain = lbn_count * ODS2_BLOCK_SIZE;

    trace ("cache.read_device: LBN %d, %d blocks\n", lbn, lbn_count);

    if ((int) lseek (fd, lbn * ODS2_BLOCK_SIZE, SEEK_SET) < 0) {
        error (err_flags | ERR_ERRNO, "error positioning to LBN %d\n", lbn);
        return -1;
    }

    while (remain > 0) {
        size = read (fd, current, remain);
        if (size < 0) {
            error (err_flags | ERR_ERRNO, "error reading LBN %d\n",
                lbn + (current - (char*)buffer) / ODS2_BLOCK_SIZE);
            return -1;
        }
        else if (size == 0) {
            error (err_flags, "attempt to read LBN %d after end of disk\n",
                lbn + (current - (char*)buffer) / ODS2_BLOCK_SIZE);
            return -1;
        }
        current += size;
        remain -= size;
    }

    return 0;
}


/*******************************************************************************
 *
 *  This internal routine removes a block from the list it belongs to.
 *
 *******************************************************************************
 */

static void remove_block (cache_t cache, block_t *block)
{
    list_t *list = block->list;

    /* Remove the block from the time list */

    if (block->next_in_time != NULL)
        block->next_in_time->prev_in_time = block->prev_in_time;
    else
        list->last = block->prev_in_time;

    if (block->prev_in_time != NULL)
        block->prev_in_time->next_in_time = block->next_in_time;
    else
        list->first = block->next_in_time;

    list->used_blocks--;

    /* Remove the block from the hash list */

    if (block->next_in_hash != NULL)
        block->next_in_hash->prev_in_hash = block->prev_in_hash;

    if (block->prev_in_hash != NULL)
        block->prev_in_hash->next_in_hash = block->next_in_hash;
    else
        cache->hash [block->lbn % HASH_TABLE_SIZE] = block->next_in_hash;
}


/*******************************************************************************
 *
 *  This internal routine adds a block in a list, as the most recently
 *  used block in this list.
 *
 *******************************************************************************
 */

static void add_block (cache_t cache, list_t *list, block_t *block)
{
    /* Insert the block in the hash list */

    block->prev_in_hash = NULL;
    block->next_in_hash = cache->hash [block->lbn % HASH_TABLE_SIZE];
    cache->hash [block->lbn % HASH_TABLE_SIZE] = block;

    if (block->next_in_hash != NULL)
        block->next_in_hash->prev_in_hash = block;

    /* Insert the block in the time list */

    block->list = list;
    list->used_blocks++;

    block->next_in_time = NULL;
    block->prev_in_time = list->last;
    list->last = block;

    if (block->prev_in_time != NULL)
        block->prev_in_time->next_in_time = block;
    else
        list->first = block;
}


/*******************************************************************************
 *
 *  This routine reads one or more LBN on the cdrom device and update the cache.
 *
 *******************************************************************************
 */

int read_lbn (
    int err_flags,
    cache_t cache,
    lbn_t lbn,
    int lbn_count,
    void *buffer,
    lbn_usage_t usage)
{
    char *current;
    block_t *block;
    list_t *list;

    trace ("cache.read_lbn: LBN %d, %d blocks, %s\n", lbn, lbn_count,
        usage == LBN_META ? "meta-data" : "data");

    /* Determine which part of the cache to use */

    list = usage == LBN_META ? &cache->meta : &cache->data;

    /* If the cache is disabled for this type of LBN, read from cdrom */

    if (cache->cdrom_file || list->max_blocks <= 0)
        return read_device (err_flags, cache->fd, lbn, lbn_count, buffer);

    /* Loop on all blocks to read */

    for (current = buffer;
         lbn_count > 0;
         lbn_count--, lbn++, current += ODS2_BLOCK_SIZE) {

        /* Attempt to locate the block in the cache */

        for (block = cache->hash [lbn % HASH_TABLE_SIZE];
             block != NULL && block->lbn != lbn;
             block = block->next_in_hash);

        if (block != NULL) {

            /* The block is found in the cache, this is a hit. */
            /* Add and remove the block from the list. This serves two */
            /* purposes: First, the block must be placed as "most */
            /* recently used" in the list. Second, it may move from one */
            /* list to another (although not likely). */

            remove_block (cache, block);
            add_block (cache, list, block);

            list->hit_count++;
            memcpy (current, block->data, ODS2_BLOCK_SIZE);
        }
        else {

            /* The block is not found in the cache */
            /* Read the data from the disk */

            if (read_device (err_flags, cache->fd, lbn, 1, current) < 0)
                return -1;

            list->miss_count++;

            /* If there is some room in the cache, allocate a new structure */
            /* Otherwise, remove the least recently used block */

            if (list->used_blocks < list->max_blocks) {
                block = utmalloc (sizeof (block_t));
                list->used_blocks++;
            }
            else {
                /* Remove the least recently used block */
                remove_block (cache, block = list->first);
            }

            /* Copy the data in the cache */

            memcpy (block->data, current, ODS2_BLOCK_SIZE);
            block->lbn = lbn;

            /* Insert the block in the lists */

            add_block (cache, list, block);
        }
    }

    return 0;
}


/*******************************************************************************
 *
 *  This routine displays the statistics of usage of the cache.
 *
 *******************************************************************************
 */

void display_cache_stat (cache_t cache)
{
    list_t *list;
    const char *cache_name;

    printf ("\n");
    if (cache->cdrom_file) {
        printf ("The CD-ROM device is a UNIX file (CD-ROM image).\n"
                "The UNIX cache is implicitely used. "
                "The local LBN cache is not used.\n");
    }
    else {
        list = &cache->data;
        cache_name = "data cache";
        for (;;) {
            printf ("Statistics for %s:\n", cache_name);
            printf ("  Maximum cache size: %d blocks (%.3f MB)\n",
                list->max_blocks,
                (double)(cache->data.max_blocks * ODS2_BLOCK_SIZE) /
                (1024 * 1024));
            printf ("  Current cache size: %d blocks\n", list->used_blocks);
            printf ("  Number of accesses: %d blocks (%d hits, %d %%)\n",
                list->hit_count + list->miss_count, list->hit_count,
                list->hit_count == 0 ? 0 : (list->hit_count * 100) /
                (list->hit_count + list->miss_count));
            if (list == &cache->meta)
                break;
            list = &cache->meta;
            cache_name = "meta-data cache";
        }
    }
    printf ("\n");
}
