/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    cache.h
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This module manages an LBN cache for the OpenVMS CD-ROM.
 *
 *  LBN are Logical Block Numbers. In the OpenVMS terminology,
 *  a disk device is collection of sequential LBN from 0 to the
 *  size of the disk. There is another concept: the VBN or Virtual
 *  Block Number. The VBN always belong to a file.They are numbered
 *  from 1 to the size of the file.
 *
 *  A CD-ROM device is accessed directly by vmscd, without using the
 *  UNIX cache. Since CD-ROM accesses are very slow, especially when
 *  the CD spins up, a cache of LBN is used by vmscd. However, when
 *  the source is a disk file (a CD-ROM image), the LBN cache is
 *  disabled since the UNIX cache is implicitely used.
 *
 *  The LBN cache is divided into two logical areas: The data (content
 *  of the data files) and the meta-data (structure of the file system
 *  and contents of the directories). This artificial distinction is
 *  here to prevent the data part to starve the meta-data part (when
 *  copying a hierarchy of files for instance).
 *
 *
 *  Modification History
 *  --------------------
 *  18 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#ifndef CACHE_H_LOADED
#define CACHE_H_LOADED 1

typedef uint32 lbn_t;
typedef enum {LBN_DATA, LBN_META} lbn_usage_t;
typedef struct cache_s *cache_t;

/*
 *  This procedure creates a new cache.
 *  Specify the maximum number of blocks in the cache.
 *  Two sizes are given: files data and filesystem metadata.
 *  Return NULL in case of error.
 */

cache_t init_cache (int err_flags, int fd, lbn_t max_data, lbn_t max_meta);

/*
 *  Delete a cache and free all the associated resources.
 */

void delete_cache (cache_t cache);

/*
 *  This routine reads one or more LBN on the cdrom device and update
 *  the cache. Return 0 on success. Return -1 in case of error, optionally
 *  print an error message and set errno.
 */

int read_lbn (int err_flags, cache_t cache, lbn_t lbn, int lbn_count,
              void *buffer, lbn_usage_t usage);

/*
 *  This routine displays the statistics of usage of the cache.
 */

void display_cache_stat (cache_t cache);

/*
 *  This routine returns the maximum amount of memory for the cache in bytes.
 *  Return zero if the cache is disabled.
 */

int max_cache_size (cache_t cache);

#endif /* CACHE_H_LOADED */
