/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    copy.c
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This module implements the file copy operations.
 *
 *
 *  Modification History
 *  --------------------
 *  26 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include "ods2.h"
#include "utils.h"
#include "cache.h"
#include "volume.h"
#include "copy.h"

/* The prototype for mkdir(2) is missing from fcntl.h */

int mkdir (const char *pathname, mode_t mode);

/* Blocking factor for file copy */

#define COPY_BLOCK_FACTOR 10

/* Text mode characters */

#define BS  8
#define HT  9
#define LF 10
#define VT 11
#define FF 12
#define CR 13


/*******************************************************************************
 *
 *  The following routine builds the name of a destination file.
 *  Return a mallocated buffer.
 *
 ******************************************************************************* */

static char *build_name (
    const char *dir,       /* unix directory name of destination */
    const char *file)      /* local file name from cdrom */
{
    int dirlen = strlen (dir);
    char *name = utmalloc (dirlen + strlen (file) + 2);
    const char *f;

    strcpy (name, dir);
    if (dirlen > 0 && name [dirlen - 1] != '/')
        name [dirlen++] = '/';

    for (f = file; *f != 0; f++) {
        /* Remove trailing "." or ".DIR" */
        if (*f == '.' && (f[1] == 0 || strcmp (f+1, "DIR") == 0))
            break;
        name [dirlen++] = tolower (*f);
    }
    name [dirlen] = 0;

    trace ("copy.build_name: dir=\"%s\", file=\"%s\", name=\"%s\"\n",
           dir, file, name);

    return name;
}


/*******************************************************************************
 *
 *  The following routine builds the name of a "set file/attributes" file.
 *  Return a mallocated buffer.
 *
 *******************************************************************************
 */

static char *build_sfa_name (const char *path)
{
    int len = strlen (path);
    char *f, *name = utmalloc (len + 10);

    strcpy (name, path);
    for (f = name + len - 1; f >= name && *f != '/'; f--)
        if (*f == '.')
            *f = '_';
    strcpy (name + len, "_sfa.com");
    trace ("copy.build_sfa_name: path = \"%s\", name = \"%s\"\n", path, name);
    return name;
}


/*******************************************************************************
 *
 *  The following routine builds the name of an UCX NFS attribute file.
 *  Return a mallocated buffer.
 *
 *******************************************************************************
 */

static char *build_ucxaf_name (const char *path)
{
    int len = strlen (path);
    char *name = utmalloc (len + 10);
    const char *f;

    for (f = path + len - 1; f >= path && *f != '/'; f--);
    strncpy (name, path, f - path + 1);
    strcpy (name + (f - path + 1), ".$ADF$");
    strcat (name, f + 1);
    strcat (name, ";1");
    trace ("copy.build_ucxaf_name: path = \"%s\", name = \"%s\"\n", path, name);
    return name;
}


/*******************************************************************************
 *
 *  The following routine deletes a file if it exists.
 *
 *******************************************************************************
 */

static void delete_file (int err_flags, copy_opt_t *opt, const char *path)
{
    if (remove (path) == 0) {
        if (opt->verbose)
            fprintf (opt->out, "deleting %s\n", path);
    }
    else if (errno != ENOENT)
        error (err_flags | ERR_ERRNO, "cannot delete %s\n", path);
}


/*******************************************************************************
 *
 *  The following routine creates a DCL command procedure containing
 *  "set file/attribute" commands.
 *
 *
 *  Documentation of the OpenVMS DCL command SET FILE /ATTRIBUTES
 *  -------------------------------------------------------------
 *
 *  SET FILE /ATTRIBUTE=(file-attribute[,...])
 *
 *     Sets the attributes associated with a file. The following table
 *     lists possible keywords and the relationship to both ACP-QIO and
 *     OpenVMS RMS File attributes:
 *                                                  OpenVMS RMS File
 *     Keyword      ACP-QIO File Attribute          Attribute
 *
 *     BKS:{value}  FAT$B_BKTSIZE={byte}            FAB$B_BKS={byte}
 *     DEQ:{value}  FAT$W_DEFEXT={word}             FAB$W_DEQ={word}
 *     EBK:{value}  FAT$L_EFBLK={longword}          XAB$L_EBK={longword}
 *     FFB:{value}  FAT$W_FFBYTE={word}             XAB$W_FFB={word}
 *     FSZ:{value}  FAT$B_VFCSIZE={byte}            FAB$B_FSZ={byte}
 *     GBC:{value}  FAT$W_GBC={word}                FAB$W_GBC={word}
 *     HBK:{value}  FAT$L_HIBLK={longword}          XAB$L_HBK={longword}
 *     LRL:{value}  FAT$W_RSIZE={word}              XAB$W_LRL={word}
 *     MRS:{value}  FAT$W_MAXREC={word}             FAB$W_MRS={word}
 *     ORG:IDX      FAT$V_FILEORG=FAT$C_INDEXED     FAB$B_ORG=FAB$C_IDX
 *     ORG:REL      FAT$V_FILEORG=FAT$C_RELATIVE    FAB$B_ORG=FAB$C_REL
 *     ORG:SEQ      FAT$V_FILEORG=FAT$C_SEQUENTIAL  FAB$B_ORG=FAB$C_SEQ
 *     RAT:BLK      FAT$B_RATTRIB=FAT$M_NOSPAN      FAB$B_RAT=FAB$M_BLK
 *     RAT:CR       FAT$B_RATTRIB=FAT$M_IMPLIEDCC   FAB$B_RAT=FAB$M_CR
 *     RAT:FTN      FAT$B_RATTRIB=FAT$M_FORTRANCC   FAB$B_RAT=FAB$M_FTN
 *     RAT:MSB      FAT$B_RATTRIB=FAT$M_MSBVAR      FAB$B_RAT=FAB$M_MSB
 *     RAT:NONE     FAT$B_RATTRIB=0                 FAB$B_RAT=0
 *     RAT:PRN      FAT$B_RATTRIB=FAT$M_PRINTCC     FAB$B_RAT=FAB$M_PRN
 *     RFM:FIX      FAT$V_RTYPE=FAT$C_FIXED         FAB$B_RFM=FAB$C_FIX
 *     RFM:STM      FAT$V_RTYPE=FAT$C_STREAM        FAB$B_RFM=FAB$C_STM
 *     RFM:STMCR    FAT$V_RTYPE=FAT$C_STREAMCR      FAB$B_RFM=FAB$C_STMCR
 *     RFM:STMLF    FAT$V_RTYPE=FAT$C_STREAMLF      FAB$B_RFM=FAB$C_STMLF
 *     RFM:UDF      FAT$V_RTYPE=FAT$C_UNDEFINED     FAB$B_RFM=FAB$C_UDF
 *     RFM:VAR      FAT$V_RTYPE=FAT$C_VARIABLE      FAB$B_RFM=FAB$C_VAR
 *     RFM:VFC      FAT$V_RTYPE=FAT$C_VFC           FAB$B_RFM=FAB$C_VFC
 *     VRS:{value}  FAT$W_VERSIONS={word}           XAB$W_VERLIMIT={word}
 *
 *******************************************************************************
 */

static void create_dcl_sfa (
    int err_flags,
    copy_opt_t *opt,
    const ods2_file_attr_t *recattr,   /* attributes of the data file */
    const char *path,                  /* name of dcl command file */
    const char *data_file)             /* name of data file */
{
    FILE *fp;
    const char *vms_name;

    if (opt->verbose)
        fprintf (opt->out, "creating %s\n", path);

    if ((fp = fopen (path, "w")) == NULL) {
        error (err_flags | ERR_ERRNO, "cannot create %s\n", path);
        return;
    }

    if ((vms_name = strrchr (data_file, '/')) == NULL)
        vms_name = data_file;
    else
        vms_name++;

    fprintf (fp, "$! Restoration of RMS attributes for %s\n", vms_name);
    fprintf (fp, "$! Automatically generated by vmscd\n$!\n");
    fprintf (fp, "$! set file %s /attributes=( -\n   ", vms_name);

    switch (recattr->fat_v_fileorg) {
    case FAT_C_SEQUENTIAL:
        fprintf (fp, "org:seq,");
        break;
    case FAT_C_RELATIVE:
        fprintf (fp, "org:rel,");
        break;
    case FAT_C_INDEXED:
        fprintf (fp, "org:idx,");
        break;
    }

    switch (recattr->fat_v_rtype) {
    case FAT_C_FIXED:
        fprintf (fp, "rfm:fix,");
        break;
    case FAT_C_STREAM:
        fprintf (fp, "rfm:stm,");
        break;
    case FAT_C_STREAMCR:
        fprintf (fp, "rfm:stmcd,");
        break;
    case FAT_C_STREAMLF:
        fprintf (fp, "rfm:stmlf,");
        break;
    case FAT_C_UNDEFINED:
        fprintf (fp, "rfm:udf,");
        break;
    case FAT_C_VARIABLE:
        fprintf (fp, "rfm:var,");
        break;
    case FAT_C_VFC:
        fprintf (fp, "rfm:vfc,");
        break;
    }

    if (recattr->fat_v_impliedcc)
        fprintf (fp, "rat:cr,");
    else if (recattr->fat_v_printcc)
        fprintf (fp, "rat:prn,");
    else if (recattr->fat_v_fortrancc)
        fprintf (fp, "rat:ftn,");
    else if (recattr->fat_v_nospan)
        fprintf (fp, "rat:blk,");
    else if (recattr->fat_v_msbrcw)
        fprintf (fp, "rat:msb,");
    else
        fprintf (fp, "rat:none,");

    fprintf (fp, "lrl:%d,", recattr->fat_w_rsize);
    fprintf (fp, "mrs:%d,", recattr->fat_w_maxrec);
    fprintf (fp, "deq:%d, -\n   ", recattr->fat_w_defext);
    fprintf (fp, "gbc:%d,", recattr->fat_w_gbc);
    fprintf (fp, "bks:%d,", recattr->fat_b_bktsize);
    fprintf (fp, "fsz:%d,", recattr->fat_b_vfcsize);
    fprintf (fp, "ebk:%d,", recattr->fat_w_efblkl +
             (recattr->fat_w_efblkh << 16));
    fprintf (fp, "ffb:%d,", recattr->fat_w_ffbyte);
    fprintf (fp, "vrs:%d)\n$!\n", recattr->fat_w_versions);

    fclose (fp);
}


/*******************************************************************************
 *
 *  The following routine creates an attribute file for the UCX NFS client.
 *
 *******************************************************************************
 */

static void create_ucx_nfs (
    int err_flags,
    copy_opt_t *opt,
    const ods2_file_header_t *header,    /* header of the data file */
    const char *path)                    /* name of attribute file */
{
    int fd;

    if (opt->verbose)
        fprintf (opt->out, "creating %s\n", path);

    if ((fd = creat (path, 0777)) < 0) {
        error (err_flags | ERR_ERRNO, "cannot create %s\n", path);
        return;
    }

    /* @@@  (Format of ADF file not yet known)  @@@ */

    error (err_flags, "Format of ADF file not yet known, %s is empty\n", path);

    close (fd);
}


/*******************************************************************************
 *
 *  This routine returns non-zero if the record attributes indicate
 *  a possible text file.
 *
 *******************************************************************************
 */

int maybe_text_file (const ods2_file_attr_t *recattr)
{
    /* The file must be sequential */

    if (recattr->fat_v_fileorg != FAT_C_SEQUENTIAL)
        return 0;

    /* Then check the record format and carriage control options */

    switch (recattr->fat_v_rtype) {
    case FAT_C_STREAM:
    case FAT_C_STREAMLF:
    case FAT_C_STREAMCR:
        /* All stream files are potential text files */
        return 1;
    case FAT_C_VARIABLE:
        /* Variable size files must have the CR carriage control option */
        return recattr->fat_v_impliedcc;
    case FAT_C_VFC:
        /* VFC size must match 2 for print and 1 for fortran */
        return (recattr->fat_v_fortrancc && recattr->fat_b_vfcsize == 1) ||
            (recattr->fat_v_printcc && recattr->fat_b_vfcsize == 2);
    default:
        /* Other sequential files are not likely text files */
        return 0;
    }
}


/*******************************************************************************
 *
 *  The following routine copies one file in text mode.
 *  Return -1 in case of error, 0 in case of success.
 *
 *
 *  Record Format
 *  -------------
 *  - Stream: Records are delimited by FF, VT, LF or CR-LF. All leading
 *    zeroes are ignored. Cannot be used with nospan.
 *  - Stream CR: Records are delimited by CR.
 *  - Stream LF: Records are delimited by LF.
 *  - Variable: Each record is preceeded by a 2-bytes size. Next record
 *    always start at an even offset. One zero byte is used to pad odd
 *    sized record. The last record in the file (or in a block with nospan)
 *    is followed by a two bytes 0xFFFF.
 *  - VFC: Same as variable but skip vcf-size bytes at the beginning of
 *    each record.
 *
 *  Record attribute
 *  ----------------
 *  - CR ("implied cc"): Implied that each record should be preceeded by
 *    a LF and followed by a CR. Cannot be used with printcc or fortrancc.
 *  - Fortran CC: The first byte of each record contains a Fortran carriage
 *    control character.
 *  - Print CC: Each record contains a 2-bytes control field. Usually
 *    used for VFC format, vfc-size = 2.
 *  - No span: Record cannot cross block boundary. Cannot be used with
 *    stream formats.
 *  - MSB RCW: For variable format, indicate that the 2-bytes size is
 *    in MSB (big endian) format.
 *
 *  Conversion rules
 *  ----------------
 *  We don't care about complex carriage control attribute. Each record is
 *  simply followed by a \n. The only record attribute we should take care
 *  of are "no span" (to skip the unused ends of blocks) and "msd" (to
 *  correctly interpret the record size).
 *
 *******************************************************************************
 */

static int copy_text_file (
    int err_flags,
    const ods2_file_attr_t *recattr,
    file_t *src,
    FILE *dest)
{
    int c = -1;  /* last input char, initial value is "error" */
    int vfcsize = 0, vfc, atbol, gotcr, recsize, s1, s2, msb, nospan, n;

    trace ("copy.copy_text_file: \"%s\"\n", src->name);

    switch (recattr->fat_v_rtype) {

    case FAT_C_STREAMLF:
        /* Same format as UNIX text files */
        while ((c = read_byte (err_flags, src)) >= 0 && !ferror (dest))
            putc (c, dest);
        break;

    case FAT_C_STREAMCR:
        /* Transform all CR into LF */
        while ((c = read_byte (err_flags, src)) >= 0 && !ferror (dest)) {
            if (c == CR)
                putc (LF, dest);
            else
                putc (c, dest);
        }
        break;

    case FAT_C_STREAM:
        /* Transform all CR-LF in LF. All leading zeroes are ignored. */
        atbol = 1;   /* boolean: at begining of line */
        gotcr = 0;   /* boolean: last character was a CR */
        while ((c = read_byte (err_flags, src)) >= 0 && !ferror (dest)) {
            if (gotcr && c != LF)
                putc (CR, dest); /* was a CR not part of a CR-LF */
            if (c != CR && (!atbol || c != 0)) {
                putc (c, dest);
                atbol = c == LF || c == VT || c == FF;
            }
            gotcr = c == CR;
        }
        break;

    case FAT_C_VFC:
        /* Same as "variable" but use the vfcsize */
        vfcsize = recattr->fat_b_vfcsize;

    case FAT_C_VARIABLE:
        /* Read size + record */
        msb = recattr->fat_v_msbrcw;
        nospan = recattr->fat_v_nospan;
        trace ("copy.copy_text_file: vfcsize=%d, msb=%d, nospan=%d\n",
               vfcsize, msb, nospan);
        /* Loop on all records */
        do {
            /* Read two bytes record size */
            if ((s1 = c = read_byte (err_flags, src)) < 0 ||
                (s2 = c = read_byte (err_flags, src)) < 0)
                break;
            /* Interpret as MSB or LSB */
            recsize = msb ? (s1 << 8) + s2 : s1 + (s2 << 8);
            /* Record size 0xFFFF means end of block if nospan option */
            if (nospan && recsize == 0xFFFF)
                skip_block (src);
            else {
                /* Skip the VFC control field */
                vfc = recsize >= vfcsize ? vfcsize : recsize;
                for (n = vfc; n > 0 && c >= 0; n--)
                    c = read_byte (err_flags, src);
                /* Then read the record */
                for (n = recsize - vfc; n > 0 && c >= 0; n--) {
                    if ((c = read_byte (err_flags, src)) >= 0)
                        putc (c, dest);
                }
                /* If the record size is odd, drop the padding byte */
                if (recsize % 2 == 1 && c >= 0)
                    c = read_byte (err_flags, src);
                /* Terminate the line in the UNIX way */
                putc (LF, dest);
            }
        } while (c >= 0 && !ferror (dest));
        break;
    }

    /* Return -1 if the last read char was -1 (error) */

    return c == -1 ? -1 : 0;
}


/*******************************************************************************
 *
 *  The following routine copies one file in binary mode.
 *  Return -1 in case of error, 0 in case of success.
 *
 *******************************************************************************
 */

static int copy_binary_file (int err_flags, file_t *src, int dest)
{
    vbn_t vbn = 1;
    int vbn_count, size, remain = src->used_size;
    char data [COPY_BLOCK_FACTOR * ODS2_BLOCK_SIZE];

    trace ("copy.copy_binary_file: \"%s\"\n", src->name);

    while (remain > 0) {

        if (remain >= COPY_BLOCK_FACTOR * ODS2_BLOCK_SIZE) {
            vbn_count = COPY_BLOCK_FACTOR;
            size = vbn_count * ODS2_BLOCK_SIZE;
        }
        else {
            vbn_count = (remain + ODS2_BLOCK_SIZE - 1) / ODS2_BLOCK_SIZE;
            size = remain;
        }

        if (read_vbn (err_flags, src, vbn, vbn_count, data) < 0)
            return -1;

        if (write (dest, data, size) < 0) {
            error (err_flags | ERR_ERRNO, "error writing %s\n", src->name);
            return -1;
        }

        vbn += vbn_count;
        remain -= size;
    }

    return 0;
}


/*******************************************************************************
 *
 *  The following routine copies one file (must not be a directory).
 *
 *******************************************************************************
 */

static void copy_one_file (
    int err_flags,
    copy_opt_t *opt,
    const volume_t *vol,
    file_t *src,          /* opened source file */
    const char *dest)     /* unix path name of destination file (or NULL) */
{
    int fd, istext, ok;
    FILE *fp;
    char *sfa_name = NULL, *ucxaf_name = NULL;
    const ods2_file_attr_t *recattr = &src->header.fh2_w_recattr;

    /* Skip reserved files, unless stated otherwise */

    if (is_reserved_file (&src->fid) && !opt->all_files)
        return;

    /* Skip non-text files if requested */

    istext = maybe_text_file (recattr);
    if (opt->text_only && !istext)
        return;

    /* Build the names of attribute files */

    if (dest != NULL) {
        sfa_name = build_sfa_name (dest);
        ucxaf_name = build_ucxaf_name (dest);
    }

    /* Now create the destination file and copy it */

    trace ("copy.copy_one_file: copying \"%s\" into \"%s\"\n", src->name,
        dest != NULL ? dest : "standard output");

    if (opt->verbose) {
        if (dest != NULL)
            fprintf (opt->out, "creating %s\n", dest);
        else
            fprintf (opt->out, "\n---------- %s\n\n", src->name);
    }

    if (istext && opt->text_mode) {

        /* Copy the file in text mode. */

        if (dest == NULL)
            copy_text_file (err_flags, recattr, src, opt->out);
        else if ((fp = fopen (dest, "w")) == NULL)
            error (err_flags | ERR_ERRNO, "cannot create %s\n", dest);
        else {

            /* Copy the content of the file */

            ok = copy_text_file (err_flags, recattr, src, fp) == 0;
            fclose (fp);

            if (!ok)
                delete_file (err_flags, opt, dest);

            /* Delete the "set file/attribute" DCL procedure and the */
            /* attribute file for UCX NFS client if they exist. */
            /* These files may exist from a previous copy in binary mode but */
            /* they must be removed since the file is now in text format. */

            delete_file (err_flags, opt, sfa_name);
            delete_file (err_flags, opt, ucxaf_name);
        }
    }
    else if (dest != NULL) {

        /* Copy the file in raw mode */

        if ((fd = creat (dest, 0666)) < 0)
            error (err_flags | ERR_ERRNO, "cannot create %s\n", dest);
        else {

            /* Copy the content of the file */

            ok = copy_binary_file (err_flags, src, fd) == 0;
            close (fd);

            if (!ok)
                delete_file (err_flags, opt, dest);

            /* Create "set file/attribute" DCL procedure if requested */

            if (ok && opt->create_sfa)
                create_dcl_sfa (err_flags, opt, recattr, sfa_name, dest);

            /* Create attribute file for UCX NFS client if requested */

            if (ok && opt->create_ucx)
                create_ucx_nfs (err_flags, opt, &src->header, ucxaf_name);
        }
    }

    /* Free the names of attribute files */

    if (sfa_name != NULL)
        free (sfa_name);
    if (ucxaf_name != NULL)
        free (ucxaf_name);
}


/*******************************************************************************
 *
 *  The following routine copies one directory.
 *
 *******************************************************************************
 */

static void copy_directory (
    int err_flags,
    copy_opt_t *opt,
    const volume_t *vol,
    file_t *dir,         /* already opened source directory */
    const char *dest)    /* unix path name of destination directory */
{
    fid_t fid;
    file_t *file;
    struct stat st;
    int exists, version, status;
    char *newfile, name [ODS2_MAX_NAME_SIZE];

    /* Create the new destination directory if it does not exist */

    exists = stat (dest, &st) == 0 && S_ISDIR (st.st_mode);

    if (opt->verbose) {
        if (exists)
            fprintf (opt->out, "directory %s already exists\n", dest);
        else
            fprintf (opt->out, "creating directory %s\n", dest);
    }

    if (!exists && mkdir (dest, 0777) < 0)
        error (err_flags | ERR_ERRNO, "cannot create directory %s\n", dest);
    else {

        /* Loop on all files in the directory */

        while ((status = next_file (err_flags, dir, name, sizeof (name),
                                    &version, &fid)) >= 0) {

            /* Skip additional versions of each file */
            if (status == 0)
                continue;

            /* Skip the special case of a directory which contains */
            /* itself (such as 000000.DIR) */
            if (memcmp (&dir->fid, &fid, sizeof (fid)) == 0)
                continue;

            /* Copy the file or the directory */
            if ((file = open_file (err_flags, vol, &fid)) != NULL) {
                newfile = build_name (dest, name);
                if (!is_a_directory (file))
                    copy_one_file (err_flags, opt, vol, file, newfile);
                else if (opt->recurse)
                    copy_directory (err_flags, opt, vol, file, newfile);
                free (newfile);
                close_file (file);
            }
        }
    }
}


/*******************************************************************************
 *
 *  The following routine performs the copy command.
 *
 *******************************************************************************
 */

void copy_files (
    int err_flags,
    copy_opt_t *opt,
    const volume_t *vol,
    const fid_t *curwd,
    const char *src,
    const char *dest)
{
    fid_t fid;
    file_t *file;
    int version;
    char *path, name [ODS2_MAX_NAME_SIZE];

    /*  If copy to command output, forces text_mode and text_only */

    if (dest == NULL)
        opt->text_mode = opt->text_only = 1;

    /* Locate the specified path from the current working directory */
    /* and perform the list. */

    if (locate_file (err_flags, vol, src, curwd, &fid, name, sizeof (name),
                     &version) == 0 &&
        (file = open_file (err_flags, vol, &fid)) != NULL) {

        if (is_a_directory (file)) {
            /* A directory cannot be copied without the recurse flags */
            /* or to the command output */
            if (!opt->recurse || dest == NULL)
                error (err_flags, "%s is a directory\n", src);
            else if (!opt->dest_dir)
                error (err_flags, "%s is not a directory\n", dest);
            else {
                path = build_name (dest, name);
                copy_directory (err_flags, opt, vol, file, path);
                free (path);
            }
        }
        else {
            if (opt->dest_dir) {
                /* Build a new file name in the destination directory */
                path = build_name (dest, name);
                copy_one_file (err_flags, opt, vol, file, path);
                free (path);
            }
            else {
                /* Copy the file using the destination name */
                copy_one_file (err_flags, opt, vol, file, dest);
            }
        }

        close_file (file);
    }
}
