/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    list.c
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This module implements the file listing ("ls") operations.
 *
 *
 *  Modification History
 *  --------------------
 *  26 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include "ods2.h"
#include "utils.h"
#include "cache.h"
#include "volume.h"
#include "copy.h"
#include "list.h"

#define WIDTH_FID     14
#define WIDTH_BLOCKS   5
#define WIDTH_BYTES    8
#define WIDTH_SEP      2


/*******************************************************************************
 *
 *  This internal routine checks if a name ends in ".DIR". If this is 
 *  true, the file may be a directory (must read the header to chack).
 *
 *******************************************************************************
 */

static int ends_in_dir (const char *path)
{
    const char *dir = strstr (path, ".DIR");
    return dir != NULL && (dir[4] == ';' || dir[4] == 0);
}


/*******************************************************************************
 *
 *  The following routine lists one file or directory.
 *
 *******************************************************************************
 */

static void list_one (
    int err_flags,
    list_opt_t *opt,
    const volume_t *vol,
    const fid_t *fid,     /* FID of file to list */
    const char *name,     /* local name of file, without version */
    int version,          /* version of file */
    int max_dirlevel,     /* maximum number of directory recursion remaining */
    int force_nodir,      /* boolean: force the nodir option */
    int dir_only)         /* boolean: skip non-directory file */
{
    file_t *file = NULL;
    int nodir = opt->list_nodir || force_nodir;
    int isdir, size, compact, status, fversion, width, max_width;
    int file_per_line, file_in_line;
    fid_t ffid;
    char fname [ODS2_MAX_NAME_SIZE], path [ODS2_MAX_PATH_SIZE];
    char file_type, tversion [10];

    /* If an error was set on the output file (because of a broken pipe */
    /* for instance), no need to spend time reading the cdrom */

    if (opt->broken_out)
        return;
    if (ferror (opt->out)) {
        error (0, "broken output file, stopping listing\n");
        opt->broken_out = 1;
        return;
    }

    /* If we reached too many level of subdirectories, there is a loop */
    /* in the structure of the disk */

    if (max_dirlevel <= 0) {
        error (err_flags, "too many levels of subdirectories\n");
        return;
    }

    /* Open the file only if we may need more information */

    if ((ends_in_dir (name) && !nodir) || opt->long_list) {
        if ((file = open_file (err_flags, vol, fid)) == NULL)
            return;
    }

    isdir = file != NULL && is_a_directory (file);

    /* If we want directories only, skip it */

    if (dir_only && !isdir) {
        if (file != NULL)
            close_file (file);
        return;
    }

    /* If the file is not a directory or must be treated as a simple file */

    if (!isdir || nodir) {

        /* Determine the file type. Potential text files must be sequential */
        /* with variable, vfc or stream record format. */

        if (opt->long_list) {
            if (is_a_directory (file))
                file_type = 'D'; /* directory */
            else if (maybe_text_file (&file->header.fh2_w_recattr))
                file_type = 'T'; /* text */
            else
                file_type = 'B'; /* binary */
            fprintf (opt->out, "%c%*s", file_type, WIDTH_SEP, "");
        }
        /* List FID if required */

        if (opt->list_fid) {
            size = fprintf (opt->out, "(%d,%d,%d)", fid->number, fid->sequence,
                            fid->volume);
            fprintf (opt->out, "%*s%*s", 
                     (size >= WIDTH_FID ? 0 : WIDTH_FID - size), "",
                     WIDTH_SEP, "");
        }

        /* List long information if required */

        if (opt->long_list) {
            fprintf (opt->out, "%*d/%-*d%*s%*d%*s",
                     WIDTH_BLOCKS, file->last_data_vbn,
                     WIDTH_BLOCKS, file->last_vbn, WIDTH_SEP, "",
                     WIDTH_BYTES, file->used_size, WIDTH_SEP, "");
        }

        /* List name and version */

        fprintf (opt->out, "%s", name);
        if (opt->list_version)
            fprintf (opt->out, ";%d", version);
        fprintf (opt->out, "\n");
    }
    else {

        /* The file is a directory and we must list its content. */
        /* First, display the full path of the directory. */
        /* Add one line between directory listings */

        if (max_dirlevel < ODS2_MAX_DIR_DEPTH)
            fprintf (opt->out, "\n");

        if (get_file_path (err_flags, vol, fid, path, sizeof (path), &fversion)
            == 0)
            fprintf (opt->out, "%s:\n", path);

        /* Check if this is a compact listing */

        compact = !opt->long_list && !opt->list_fid && opt->width > 0;
        max_width = 0;

        /* Loop on all files in the directory. For compact listing, we */
        /* simply record the maximum width of a file name. */

        while (!opt->broken_out &&
               (status = next_file (err_flags, file, fname, sizeof (fname),
                                    &fversion, &ffid)) >= 0) {

            /* Skip reserved files if they must not be displayed */
            if (!opt->list_all && is_reserved_file (&ffid))
                continue;

            /* Keep only first version of each file if version not displayed */
            if (!opt->list_version && status == 0)
                continue;

            if (compact) {
                /* Compute max width of files */
                width = strlen (fname);
                if (opt->list_version) {
                    sprintf (tversion, "%d", fversion);
                    width += strlen (tversion) + 1;
                }
                if (width > max_width)
                    max_width = width;
            }
            else {
                /* Non compact display -> recursion */
                list_one (err_flags, opt, vol, &ffid, fname, fversion,
                          max_dirlevel - 1, /*force_nodir*/ 1, /*dir_only*/ 0);
            }
        }

        /* Compact listing must now be performed */

        if (compact) {

            /* Compute the number of files to display per line */

            file_per_line = (opt->width + WIDTH_SEP - 1) /
                (max_width + WIDTH_SEP);
            if (file_per_line < 1)
                file_per_line = 1;

            /* Read the directory again and display several files per line */

            rewind_file (file);
            file_in_line = 0;

            while ((status = next_file (err_flags, file, fname, sizeof (fname),
                                        &fversion, &ffid)) >= 0) {

                /* Skip reserved files if they must not be displayed */
                if (!opt->list_all && is_reserved_file (&ffid))
                    continue;

                /* Keep only first version of file if version not displayed */
                if (!opt->list_version && status == 0)
                    continue;

                /* List the file */
                if (file_in_line++ > 0)
                    fprintf (opt->out, "%*s", WIDTH_SEP, "");
                size = fprintf (opt->out, "%s", fname);
                if (opt->list_version)
                    size += fprintf (opt->out, ";%d", fversion);
                if (file_in_line >= file_per_line) {
                    file_in_line = 0;
                    fprintf (opt->out, "\n");
                }
                else if (size < max_width)
                    fprintf (opt->out, "%*s", max_width - size, "");
            }

            if (file_in_line > 0)
                fprintf (opt->out, "\n");
        }


        /* If recursion is requested, list the content of each subdirectory */

        if (opt->recurse) {

            /* Read the directory again and select directories only */

            rewind_file (file);

            while (!opt->broken_out &&
                   (status = next_file (err_flags, file, fname, sizeof (fname),
                                        &fversion, &ffid)) >= 0) {

                /* Skip reserved files if they must not be displayed */
                if (!opt->list_all && is_reserved_file (&ffid))
                    continue;

                /* Files not ending in ".DIR" cannot be directories */
                if (!ends_in_dir (fname))
                    continue;

                /* Skip the special case of a directory which contains */
                /* itself (such as 000000.DIR) */
                if (memcmp (&ffid, fid, sizeof (fid_t)) == 0)
                    continue;

                /* Recurse into the subdirectory */
                list_one (err_flags, opt, vol, &ffid, fname, fversion,
                          max_dirlevel - 1, /*force_nodir*/ 0, /*dir_only*/ 1);
            }
        }
    }

    /* Close the file if it was open */

    if (file != NULL)
        close_file (file);
}


/*******************************************************************************
 *
 *  The following routine performs the list command.
 *
 *******************************************************************************
 */

void list_files (
    int err_flags,
    list_opt_t *opt,
    const volume_t *vol,
    const fid_t *curwd,
    const char *path)
{
    fid_t fid;
    int version;
    char name [ODS2_MAX_NAME_SIZE];

    /* Locate the specified path from the current working directory */
    /* and perform the list. */

    if (locate_file (err_flags, vol, path, curwd, &fid, name, sizeof (name),
                     &version) == 0) {

        list_one (err_flags, opt, vol, &fid, name, version,
                  ODS2_MAX_DIR_DEPTH, /*force_nodir*/ 0, /*dir_only*/ 0);
    }
}
