/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    ods2.h
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This header file contains the data structure definition for the ODS-2
 *  file-system. ODS-2 (On Disk Structure, level 2) is the most common
 *  file-system on OpenVMS, including CD-ROMs.
 *
 *  Starting with OpenVMS V7.2, a new variant (ODS-5) has been introduced.
 *  However, all Digital/Compaq CD-ROM remain in ODS-2 format.
 *
 *  Byte order: All processors running th OpenVMS operating (Alpha, VAX)
 *  are little-endian. Consequently, all integer data in the ODS-2 structures
 *  are represented in little-endian order (LSB first).
 *
 *
 *  Credits
 *  -------
 *  These definitions are based on the excellent book "VMS File System
 *  Internals" by Kirby McCoy, Digital Press, 1990. Thanks very much
 *  to the author.
 *
 *
 *  Modification History
 *  --------------------
 *  18 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#ifndef ODS2_H_LOADED
#define ODS2_H_LOADED 1

#include <stddef.h>

#define ODS2_BLOCK_SIZE     512 
#define ODS2_MAX_NAME_SIZE  86
#define ODS2_MAX_DIR_DEPTH  64
#define ODS2_MAX_PATH_SIZE  ((ODS2_MAX_NAME_SIZE + 1) * ODS2_MAX_DIR_DEPTH)


/*
 *  Integer types (assume little-endian representation, Intel, Alpha)
 */ 

typedef unsigned char uint8;
typedef signed char int8;
typedef unsigned short int uint16;
typedef short int int16;
typedef unsigned int uint32;
typedef int int32;
typedef struct {uint32 low; uint32 high;} uint64;
typedef struct {uint32 low; int32 high;} int64;


/*******************************************************************************
 *
 *  Module $FIDDEF : File identification definition.
 *
 *******************************************************************************
 */

typedef struct {
    uint16 fid_w_num;  /* File number */
    uint16 fid_w_seq;  /* File sequence number */
    uint8  fid_b_rvn;  /* Byte form of RVN */
    uint8  fid_b_nmx;  /* File number extension */
} ods2_fid_t;
 
/* Reserved files numbers (with seq = num) */

#define FID_C_INDEXF     1   /* Index file */
#define FID_C_BITMAP     2   /* Storage map file */
#define FID_C_BADBLK     3   /* Bad block file */
#define FID_C_MFD        4   /* Master file directory */
#define FID_C_CORIMG     5   /* Core image file */
#define FID_C_VOLSET     6   /* Volume set list file */
#define FID_C_CONTIN     7   /* Standard continuation file */
#define FID_C_BACKUP     8   /* Backup log file */
#define FID_C_BADLOG     9   /* Bad block log file */
#define FID_C_FREFIL    10   /* Free space file */
#define FID_C_MAXRESFIL 10   /* Maximum value of reserved files id */
        
 
/*******************************************************************************
 *
 *  Module $HM2DEF : Home block definition
 *
 *******************************************************************************
 */

typedef struct _hm2 {
    uint32 hm2_l_homelbn;         /* LBN of home (i.e., this) block */
    uint32 hm2_l_alhomelbn;       /* LBN of alternate home block */
    uint32 hm2_l_altidxlbn;       /* LBN of alternate index file header */
    uint8  hm2_b_strucver;        /* structure version number */
    uint8  hm2_b_struclev;        /* main structure level */
    uint16 hm2_w_cluster;         /* storage bitmap cluster factor */
    uint16 hm2_w_homevbn;         /* VBN of home (i.e., this) block */
    uint16 hm2_w_alhomevbn;       /* VBN of alternate home block */
    uint16 hm2_w_altidxvbn;       /* VBN of alternate index file header */
    uint16 hm2_w_ibmapvbn;        /* VBN of index file bitmap */
    uint32 hm2_l_ibmaplbn;        /* LBN of index file bitmap */
    uint32 hm2_l_maxfiles;        /* maximum ! files on volume */
    uint16 hm2_w_ibmapsize;       /* index file bitmap size, blocks */
    uint16 hm2_w_resfiles;        /* ! reserved files on volume */
    uint16 hm2_w_devtype;         /* disk device type */
    uint16 hm2_w_rvn;             /* relative volume number of this volume */
    uint16 hm2_w_setcount;        /* count of volumes in set */
    uint16 hm2_w_volchar;         /* volume characteristics */
    uint32 hm2_l_volowner;        /* volume owner UIC */
    uint32 hm2_l_sec_mask;        /* volume security mask */
    uint16 hm2_w_protect;         /* volume protection */
    uint16 hm2_w_fileprot;        /* default file protection */
    uint16 hm2_w_recprot;         /* default file record protection */
    uint16 hm2_w_checksum1;       /* first checksum */
    uint64 hm2_q_credate;         /* volume creation date */
    uint8  hm2_b_window;          /* default window size */
    uint8  hm2_b_lru_lim;         /* default LRU limit */
    uint16 hm2_w_extend;          /* default file extend */
    uint64 hm2_q_retainmin;       /* minimum file retention period */
    uint64 hm2_q_retainmax;       /* maximum file retention period */
    uint64 hm2_q_revdate;         /* volume revision date */
    char   hm2_r_min_class [20];  /* volume minimum security class */
    char   hm2_r_max_class [20];  /* volume maximum security class */
    ods2_fid_t hm2_w_filetab_fid; /* file lookup table FID */
    uint16 hm2_w_lowstruclev;     /* lowest struclev on volume */
    uint16 hm2_w_highstruclev;    /* highest struclev on volume */
    uint64 hm2_q_copydate __attribute__ ((packed)); /* volume copy date */
    char   hm2__fill_1 [302];     /* spare */
    uint32 hm2_l_serialnum;       /* pack serial number */
    char   hm2_t_strucname [12];  /* structure (volume set name) */
    char   hm2_t_volname [12];    /* volume name */
    char   hm2_t_ownername [12];  /* volume owner name */
    char   hm2_t_format [12];     /* volume format type */
    char   hm2_fill_2 [2];        /* spare */
    uint16 hm2_w_checksum2;       /* second checksum */
} ods2_home_block_t;
 
/* Flags values for field hm2_w_volchar */

#define HM2_M_READCHECK    0x01
#define HM2_M_WRITCHECK    0x02
#define HM2_M_ERASE        0x04
#define HM2_M_NOHIGHWATER  0x08
#define HM2_M_CLASS_PROT   0x10

 
/*******************************************************************************
 *
 *  Module $FATDEF : File attribute definition
 *
 *******************************************************************************
 */

/* Record format (field fat_v_rtype) */
        
#define FAT_C_UNDEFINED  0    /* undefined record type */
#define FAT_C_FIXED      1    /* fixed record type */
#define FAT_C_VARIABLE   2    /* variable length */
#define FAT_C_VFC        3    /* variable + fixed control */
#define FAT_C_STREAM     4    /* RMS-11 stream format (CR/LF-terminated) */
#define FAT_C_STREAMLF   5    /* LF-terminated stream format */
#define FAT_C_STREAMCR   6    /* CR-terminated stream format */

/* File organization (field fat_v_fileorg) */

#define FAT_C_SEQUENTIAL 0    /* sequential organization */
#define FAT_C_RELATIVE   1    /* relative organization */
#define FAT_C_INDEXED    2    /* indexed organization */
#define FAT_C_DIRECT     3    /* direct organization */

/* Record attributes area as used by FCS and RMS */
        
typedef struct _fat {
    unsigned fat_v_rtype : 4;     /* record type */
    unsigned fat_v_fileorg : 4;   /* file organization */
    unsigned fat_v_fortrancc : 1; /* Fortran carriage control */
    unsigned fat_v_impliedcc : 1; /* implied carriage control */
    unsigned fat_v_printcc : 1;   /* print file carriage control */
    unsigned fat_v_nospan : 1;    /* no spanned records */
    unsigned fat_v_msbrcw : 1;    /* Format of RCW (0=LSB, 1=MSB) */
    unsigned fat_v_fill_1 : 3;    /* MBZ (or should be zero) */
    uint16   fat_w_rsize;         /* record size in bytes */
    uint16   fat_w_hiblkh;        /* highest allocated VBN, high order word */
    uint16   fat_w_hiblkl;        /* highest allocated VBN, low order word */
    uint16   fat_w_efblkh;        /* end of file VBN, high order word */
    uint16   fat_w_efblkl;        /* end of file VBN, low order word */
    uint16   fat_w_ffbyte;        /* first free byte in EFBLK */
    uint8    fat_b_bktsize;       /* bucket size in blocks */
    uint8    fat_b_vfcsize;       /* size in bytes of control for VFC */
    uint16   fat_w_maxrec;        /* maximum record size in bytes */
    uint16   fat_w_defext;        /* default extend quantity */
    uint16   fat_w_gbc;           /* global buffer count */
    uint16   fat_w_fill_2 [4];    /* spare */
    uint16   fat_w_versions;      /* default version limit for directory file */
} ods2_file_attr_t;


/*******************************************************************************
 *
 *  Module $FH2DEF : File header definition
 *
 *******************************************************************************
 */

typedef struct {
    uint8  fh2_b_idoffset;          /* ident area offset in words */
    uint8  fh2_b_mpoffset;          /* map area offset in words */
    uint8  fh2_b_acoffset;          /* access control list offset in words */
    uint8  fh2_b_rsoffset;          /* reserved area offset in words */
    uint16 fh2_w_seg_num;           /* file segment number */
    uint8  fh2_b_strucver;          /* file structure version */
    uint8  fh2_b_struclev;          /* principal file structure level */
    ods2_fid_t fh2_w_fid;           /* file ID */
    ods2_fid_t fh2_w_ext_fid;       /* extension file ID */
    ods2_file_attr_t fh2_w_recattr; /* file record attributes */
    uint32 fh2_l_filechar;          /* file characteristics */
    uint16 fh2_w_recprot;           /* record protection */
    uint8  fh2_b_map_inuse;         /* number of map area words in use */
    uint8  fh2_b_acc_mode;          /* least privileged access mode */
    uint32 fh2_l_fileowner;         /* file owner UIC */
    uint16 fh2_w_fileprot;          /* file protection */
    ods2_fid_t fh2_w_backlink;      /* back link pointer */
    uint8  fh2_b_journal;           /* journal control flags */
    uint8  fh2_b_ru_active;         /* If non-zero, has active recov units */
    uint16 fh2_fill_3;              /* reserved */
    uint32 fh2_l_highwater;         /* high-water mark in file */
    char   fh2_fill_1 [8];          /* reserved */
    char   fh2_r_class_prot [20];   /* security classification mask */
    char   fh2_fill_2 [402];        /* rest of file header */
    uint16 fh2_w_checksum;          /* file header checksum */
} ods2_file_header_t;
 
 
/* Values for field fh2_l_filechar */

#define FH2_M_WASCONTIG         0x000001
#define FH2_M_NOBACKUP          0x000002
#define FH2_M_WRITEBACK         0x000004
#define FH2_M_READCHECK         0x000008
#define FH2_M_WRITCHECK         0x000010
#define FH2_M_CONTIGB           0x000020
#define FH2_M_LOCKED            0x000040
#define FH2_M_CONTIG            0x000080
#define FH2_M_VCC_STATE         0x000700
#define FH2_M_BADACL            0x000800
#define FH2_M_SPOOL             0x001000
#define FH2_M_DIRECTORY         0x002000
#define FH2_M_BADBLOCK          0x004000
#define FH2_M_MARKDEL           0x008000
#define FH2_M_NOCHARGE          0x010000
#define FH2_M_ERASE             0x020000
#define FH2_M_ALM_AIP           0x040000
#define FH2_M_SHELVED           0x080000
#define FH2_M_ALM_ARCHIVED      0x080000
#define FH2_M_ALM_STATE         0x0C0000
#define FH2_M_SCRATCH           0x100000
#define FH2_M_NOMOVE            0x200000
#define FH2_M_NOSHELVABLE       0x400000
#define FH2_M_SHELVING_RESERVED 0x800000

/* Values for field fh2_b_journal */

#define FH2_M_ONLY_RU      0x01
#define FH2_M_RUJNL        0x02
#define FH2_M_BIJNL        0x04
#define FH2_M_AIJNL        0x08
#define FH2_M_ATJNL        0x10
#define FH2_M_NEVER_RU     0x20
#define FH2_M_JOURNAL_FILE 0x40

/* Values for field fh2_b_ru_active */

#define FH2_C_RU_FACILITY_RMS    1   /* RMS */
#define FH2_C_RU_FACILITY_DBMS   2   /* DBMS */
#define FH2_C_RU_FACILITY_RDB    3   /* Rdb/VMS */
#define FH2_C_RU_FACILITY_CHKPNT 4   /* Checkpoint/Restart */

/* Size constants */

#define FH2_C_LENGTH          80   /* length of header area */
#define FH2_C_SUBSET0_LENGTH  88   /* length of header area */
#define FH2_C_FULL_LENGTH    108   /* length of full header */
        

/*******************************************************************************
 *
 *  Module $FI2DEF : File header identification area definition
 *
 *******************************************************************************
 */

typedef struct {
    char   fi2_t_filename [20];    /* file name, type, and version (ASCII) */
    uint16 fi2_w_revision;         /* revision number (binary) */
    uint64 fi2_q_credate __attribute__ ((packed)); /* creation date/time */
    uint64 fi2_q_revdate __attribute__ ((packed)); /* revision date/time */
    uint64 fi2_q_expdate __attribute__ ((packed)); /* expiration date/time */
    uint64 fi2_q_bakdate __attribute__ ((packed)); /* backup date/time */
    char   fi2_t_filenamext [66];  /* extension file name area */
    char   fi2_t_userlabel [80];   /* optional user file label */
} ods2_file_header_ident_t;

 
/*******************************************************************************
 *
 *  Module _FM2DEF : File header map area definition
 *
 *******************************************************************************
 */

/* Retrieval pointer type codes (field fm2_v_format) */

#define FM2_C_PLACEMENT 0  /* 00 = placement control data */
#define FM2_C_FORMAT1   1  /* 01 = format 1 */
#define FM2_C_FORMAT2   2  /* 10 = format 2 */
#define FM2_C_FORMAT3   3  /* 11 = format 3 */

/* Placement retrieval pointer */

typedef struct {
    unsigned fm2_v_exact  : 1;   /* exact placement specified */
    unsigned fm2_v_oncyl  : 1;   /* on cylinder allocation desired */
    unsigned fm2_v_fill   : 10;
    unsigned fm2_v_lbn    : 1;   /* use LBN of next map pointer */
    unsigned fm2_v_rvn    : 1;   /* place on specified RVN */
    unsigned fm2_v_format : 2;
} ods2_fmap_placement_t;

/* Retrieval pointer format 1 */

typedef struct {
    uint8    fm2_b_count1;       /* format 1 count field */
    unsigned fm2_v_highlbn : 6;  /* high order LBN */
    unsigned fm2_v_format  : 2;
    uint16   fm2_w_lowlbn;       /* format 1 low order LBN */
} ods2_fmap1_t;

/* Retrieval pointer format 2 */

typedef struct {
    unsigned fm2_v_count2 : 14;
    unsigned fm2_v_format : 2;
    uint32   fm2_l_lbn2 __attribute__ ((packed));
} ods2_fmap2_t;

/* Retrieval pointer format 3 */

typedef struct {
    unsigned fm2_v_count2 : 14;
    unsigned fm2_v_format : 2;
    uint16   fm2_w_lowcount;
    uint32   fm2_l_lbn3;
} ods2_fmap3_t;
 
/* Maximum number of retrieval pointers per file header */

#define ODS2_MAX_RP_PER_HEADER \
    ((sizeof (ods2_file_header_t) - \
    offsetof (ods2_file_header_t, fh2_l_highwater)) / \
    sizeof (ods2_fmap_placement_t))

 
/*******************************************************************************
 *
 *  Module $DIRDEF : Directory record definition.
 *
 *******************************************************************************
 */

/* Directory entry type code (field dir_v_type) */

#define DIR_C_FID      0        /* normal file ID */
#define DIR_C_LINKNAME 1        /* symbolic name */

/* Directory record structure */

#define DIR_K_LENGTH  6         /* length of directory entry overhead */
#define DIR_C_LENGTH  6         /* length of directory entry overhead */
#define DIR_S_NAME   80         /* maximum length of name string */
        
typedef struct {
    uint16   dir_w_size;        /* size of directory record in bytes */
    uint16   dir_w_verlimit;    /* maximum number of versions */
    unsigned dir_v_type : 3;    /* directory entry type */
    unsigned dir_v_fill_1 : 3;  /* reserved */
    unsigned dir_v_nextrec : 1; /* another record of same name/type follows */
    unsigned dir_v_prevrec : 1; /* another record of same name/type precedes */
    uint8    dir_b_namecount;   /* byte count of name string */
    char     dir_t_name [DIR_S_NAME];  /* file name/type */
} ods2_dir_t;

/* Directory entry structure */

#define DIR_K_VERSION 8         /* size of each version entry */
#define DIR_C_VERSION 8         /* size of each version entry */

typedef struct {
    uint16     dir_w_version;                       /* version number */
    ods2_fid_t dir_w_fid __attribute__ ((packed));  /* file ID */
} ods2_dirent_t;

/* Symbolic name entry definition */
 
typedef struct {
    uint8 dir_b_linkcount;      /* byte count of symbolic link name */
    char  dir_t_linkname [1];   /* symbolic link name */
} ods2_dirlink_t;
        

#endif /* ODS2_H_LOADED */
