/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    utils.c
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This module implements some low-level utility routines, not specific
 *  to vmscd. See file utils.h for details.
 *
 *
 *  Modification History
 *  --------------------
 *  18 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <readline/readline.h>
#include <readline/history.h>
#include "utils.h"

#define UID_ROOT 0

/* Global variables set by init_utils */

static char appname [255] = "";
static onoff_t debug_mode = OFF;


/*******************************************************************************
 *
 *  This routine initializes the module.
 *
 *******************************************************************************
 */

void init_utils (const char *argv0, onoff_t debug, int max_recall)
{
    const char *p, *start = argv0;

    /* Store debug mode flag */

    debug_mode = debug;

    /* Look for last slash (if any) in application name */

    for (p = argv0; *p != 0; p++)
        if (*p == '/')
            start = p + 1;

    /* Store the name */

    strncpy (appname, argv0, sizeof (appname));
    appname [sizeof (appname) - 1] = 0;

    /* Initialize the history routine */

    if (isatty (0)) {
       using_history ();
       stifle_history (max_recall);
    }
}


/*******************************************************************************
 *
 *  Get line routine. Return NULL at EOF, line otherwise.
 *
 *******************************************************************************
 */

char *input (const char *prompt, char *line, int line_size)
{
    char *res;
    int len;

    if (isatty (0)) {
        if ((res = readline (prompt)) == NULL) {
            putchar ('\n');
            return NULL; /* end of file */
        }
        if (strlen (res) < line_size) {
            strcpy (line, res);
            if (*line != 0)
                add_history (line);
        }
        else {
            error (0, "input line too long, ignored\n");
            *line = 0;
        }
        free (res);
    }
    else {
        if (fgets (line, line_size, stdin) == NULL)
            return NULL; /* end of file */
        if ((len = strlen (line)) > 0 && line [len-1] == '\n')
            line [len-1] = 0;
        else {
            while (getchar () != '\n');
            error (0, "input line too long, ignored\n");
            *line = 0;
        }
    }

    return line;
}


/*******************************************************************************
 *
 *  Trace routine. Display only if boolean debug was set in init_utils
 *
 *******************************************************************************
 */

void trace (const char *format, ...)
{
    if (debug_mode) {
        va_list ap;
        fprintf (stderr, "#debug# ");
        va_start (ap, format);
        vfprintf (stderr, format, ap);
        va_end (ap);
    }
}


/*******************************************************************************
 *
 *  Error message routine.
 *
 *******************************************************************************
 */

void error (int err_flags, const char *format, ...)
{
    int err = errno;
    va_list ap;

    if ((err_flags & ERR_SILENT) == 0) {

        fprintf (stderr, "%s: ", appname);
        va_start (ap, format);
        vfprintf (stderr, format, ap);
        va_end (ap);

        if (err_flags & ERR_ERRNO) {
            errno = err;
            perror (appname);
        }
    }

    if (err_flags & ERR_EXIT)
        exit (EXIT_FAILURE);

    errno = err;
}


/*******************************************************************************
 *
 *  This routine dumps a memory area.
 *
 *******************************************************************************
 */

void memdump (FILE *out, const void *data, int size, int display_offset)
{
    const unsigned char *line, *p, *cur = data, *end = cur + size;
    int bytes_per_line = 16;

    for (line = cur; line < end; line += bytes_per_line) {
        fprintf (out, "%08X:   ", display_offset + line - (unsigned char*)data);
        for (p = line + bytes_per_line - 1; p >= line; p--) {
            if (p >= end)
                fprintf (out, "  ");
            else
                fprintf (out, "%02X", *p);
            if ((p - line) % 4 == 0)
                fprintf (out, " ");
        }
        fprintf (out, "  ");
        for (p = line; p < line + bytes_per_line; p++)
            fputc (isprint (*p) ? *p : '.', out);
        fputc ('\n', out);
    }
}


/*******************************************************************************
 *
 *  Variant of malloc. Exit application in case of error.
 *
 *******************************************************************************
 */

void *utmalloc (size_t size)
{
    void *result = malloc (size);
    if (result == NULL)
        error (ERR_EXIT | ERR_ERRNO, "insufficient memory\n");
    return result;     
}


/*******************************************************************************
 *
 *  Allocate a copy of a string.
 *
 *******************************************************************************
 */

char *new_string (const char *s)
{
    char *result = utmalloc (strlen (s) + 1);
    strcpy (result, s);
    return result;
}


/*******************************************************************************
 *
 *  This routine returns the vertical size of a terminal.
 *
 *******************************************************************************
 */

int terminal_rows (int fd)
{
    struct winsize wins;
    memset (&wins, 0, sizeof (wins));
    if (isatty (fd) && ioctl (fd, TIOCGWINSZ, &wins) < 0)
        error (ERR_ERRNO, "ioctl error, cannot get terminal size\n");
    trace ("terminal rows = %d\n", wins.ws_row);
    return wins.ws_row;
}


/*******************************************************************************
 *
 *  This routine returns the horizontal size of a terminal.
 *
 *******************************************************************************
 */

int terminal_columns (int fd)
{
    struct winsize wins;
    memset (&wins, 0, sizeof (wins));
    if (isatty (fd) && ioctl (fd, TIOCGWINSZ, &wins) < 0)
        error (ERR_ERRNO, "ioctl error, cannot get terminal size\n");
    trace ("terminal columns = %d\n", wins.ws_col);
    return wins.ws_col;
}


/*******************************************************************************
 *
 *  Check if a file exists and can be executed by the current process.
 *
 *******************************************************************************
 */

int may_execute (const char *path)
{
    struct stat st;
    uid_t uid = geteuid ();
    gid_t gid = getegid ();

    return stat (path, &st) == 0 &&
        (((st.st_mode & S_IXUSR) && (st.st_uid == uid || uid == UID_ROOT)) ||
         ((st.st_mode & S_IXGRP) && (st.st_gid == gid || uid == UID_ROOT)) ||
         (st.st_mode & S_IXOTH));
}


/*******************************************************************************
 *
 *  Locate an executable file from $PATH.
 *
 *******************************************************************************
 */

char *find_exec (const char *name)
{
    const char *path, *curpath;
    char *buffer, *p;

    /* If the supplied path name is absolute, check this file only */

    if (name == NULL || *name == 0)
        return NULL;
    if (*name == '/')
        return may_execute (name) ? new_string (name) : NULL;

    /* Walk through $PATH to find a match */

    if ((path = getenv ("PATH")) == NULL)
        return NULL;

    buffer = utmalloc (strlen (path) + strlen (name) + 2);
    curpath = path;

    while (*curpath != 0) {
        while (*curpath == ':')
            curpath++;
        if (*curpath == 0)
            break;
        for (p = buffer; *curpath != 0 && *curpath != ':'; *p++ = *curpath++);
        if (p [-1] != '/')
            *p++ = '/';
        strcpy (p, name);
        if (may_execute (buffer)) {
            p = new_string (buffer);
            free (buffer);
            return p;
        }
    }

    free (buffer);
    return NULL;
}
