/*******************************************************************************
 *
 *  Copyright (c) 1999 by Thierry Lelegard
 *
 *  This software is covered by the "GNU GENERAL PUBLIC LICENSE" (GPL),
 *  version 2, June 1991. See the file named COPYING for details.
 *
 *  Project: VMSCD - OpenVMS CD-ROM Utility for Linux
 *  File:    vmscd.c
 *  Author:  Thierry Lelegard
 *
 *
 *  Abstract
 *  --------
 *  This file is the main program for the vmscd utility.
 *
 *
 *  Modification History
 *  --------------------
 *  18 Dec 1999 - Thierry Lelegard (lelegard@club-internet.fr)
 *                Creation of the file.
 *
 *
 *******************************************************************************
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <getopt.h>
#include "ods2.h"
#include "utils.h"
#include "cache.h"
#include "volume.h"
#include "list.h"
#include "copy.h"
#include "version.h"


/*******************************************************************************
 *
 *  Default parameters
 *
 *******************************************************************************
 */

#define DEFAULT_CDROM_DEVICE      "/dev/cdrom"
#define CDROM_DEVICE_VAR          "VMSCDROM"
#define COMMAND_PROMPT            "vmscd> "
#define MAX_COMMAND_SIZE          8192
#define MAX_COMMAND_RECALL        50
#define DEFAULT_DATA_CACHE_SIZE   10000  /* 10_000 blocks = 5 MB */
#define DEFAULT_META_CACHE_SIZE   10000  /* 10_000 blocks = 5 MB */
#define DEFAULT_SCREEN_WIDTH      80

/*
 *  File identification (FID) of the master file directory (MFD)
 *  of any ODS-2 volume. This is a constant.
 */

static const fid_t mfd_fid = {FID_C_MFD, FID_C_MFD, 0};


/*******************************************************************************
 *
 *  This routine displays a command line usage summary and exits.
 *
 *******************************************************************************
 */

static void usage (const char* argv0)
{
    fprintf (stderr, "usage: %s [-d device] [-c data-cache-size] "
        "[-m meta-data-cache-size] [-n] [-s] [-v] [command]\n", argv0);
    fprintf (stderr, "%s: try \"%s help\" for available commands\n",
        argv0, argv0);
    exit (EXIT_FAILURE);
}


/*******************************************************************************
 *
 *  State of the application.
 *
 *******************************************************************************
 */

typedef struct {
    const char *device;  /* CD-ROM device */
    cache_t cache;       /* LBN cache descriptor */
    volume_t *vol;       /* ODS-2 volume descriptor */
    file_t *curwd;       /* Current working directory descriptor */
    FILE *out;           /* Standard output for the command */
} vmscd_state_t;


/*******************************************************************************
 *
 *  Command table.
 *  An action routine returns -1 to exit the application.
 *
 *******************************************************************************
 */

typedef int (*action_t) (int argc, char *argv[], vmscd_state_t *state);

static int cmd_exit (int argc, char *argv[], vmscd_state_t *state);
static int cmd_help (int argc, char *argv[], vmscd_state_t *state);
static int cmd_stat (int argc, char *argv[], vmscd_state_t *state);
static int cmd_cd   (int argc, char *argv[], vmscd_state_t *state);
static int cmd_lcd  (int argc, char *argv[], vmscd_state_t *state);
static int cmd_pwd  (int argc, char *argv[], vmscd_state_t *state);
static int cmd_ls   (int argc, char *argv[], vmscd_state_t *state);
static int cmd_cp   (int argc, char *argv[], vmscd_state_t *state);
static int cmd_cat  (int argc, char *argv[], vmscd_state_t *state);
static int cmd_dump (int argc, char *argv[], vmscd_state_t *state);

typedef struct {
    const char *name;
    action_t action;
    const char *help;
} cmd_desc_t;

static const cmd_desc_t command_table [] = {
    {"cat", cmd_cat,
     "Syntax:\n"
     "  cat [options] cdrom-files ...\n\n"
     "Description:\n"
     "  Copy files from the cdrom to the standard output. These files\n"
     "  must be text files (in any supported OpenVMS text format).\n\n"
     "Options:\n"
     "  -v : verbose, display the name of files\n"},
    {"cd", cmd_cd,
     "Syntax:\n"
     "  cd [options] [path]\n\n"
     "Description:\n"
     "  Change the current directory on the cdrom. The root of the disk\n"
     "  is designated by \"/\". Without argument, move to the root.\n\n"
     "Options:\n"
     "  -n : do not display new working directory.\n"},
    {"cp", cmd_cp,
     "Syntax:\n"
     "  cp [options] cdrom-files ... destination\n\n"
     "Description:\n"
     "  Copy files from the cdrom to the UNIX file system. If multiple\n"
     "  input files are given, the destination must be a directory.\n\n"
     "Options:\n"
     "  -a : copy all files, including system ones.\n"
     "  -d : create DCL command files with \"set file/attributes\" commands.\n"
     "  -n : copy \"text files\" only.\n"
     "  -r : recurse into subdirectories to copy all files.\n"
     "  -t : convert \"text files\" to UNIX text format.\n"
     "  -u : create attribute files for UCX NFS client.\n"
     "  -v : verbose, display the name of created files\n"},
    {"dump", cmd_dump,
     "Syntax:\n"
     "  dump [options] [cdrom-file]\n\n"
     "Description:\n"
     "  Dump the content of a file in hexadecimal and ascii format. If a\n"
     "  file is specified, the VBN of the file are dumped. If no file is\n"
     "  specified, the LBN of the disk are dumped.\n\n"
     "Options:\n"
     "  -c value : Specify the number of blocks to dumps. Optional for\n"
     "             file, required for disk LBN.\n"
     "  -s value : Starting block to dump. Optional for file, required for\n"
     "             disk LBN.\n"},
    {"exit", cmd_exit,
     "Syntax:\n"
     "  exit\n\n"
     "Description:\n"
     "  Exit the application.\n"},
    {"help", cmd_help,
     "Syntax:\n"
     "  help [options] [command-name ...]\n\n"
     "Description:\n"
     "  Without parameter, display the list of commands. Otherwise, display\n"
     "  a description of the specified commands.\n\n"
     "Options:\n"
     "  -a : display the full description of all commands.\n"},
    {"lcd", cmd_lcd,
     "Syntax:\n"
     "  lcd [options] [path]\n\n"
     "Description:\n"
     "  Change the current working directory of the process. Without\n"
     "  argument, move to the home directory of the user.\n\n"
     "Options:\n"
     "  -n : do not display new working directory.\n"},
    {"ls", cmd_ls,
     "Syntax:\n"
     "  ls [options] [path ...]\n\n"
     "Description:\n"
     "  List files on the cdrom.\n\n"
     "Options:\n"
     "  -a : list all files, including system ones.\n"
     "  -d : list directory as a file, do not list directory content.\n"
     "  -f : display FID values with names.\n"
     "  -l : long listing.\n"
     "  -r : recurse into subdirectories.\n"
     "  -v : list OpenVMS version numbers.\n"},
    {"more", cmd_cat,
     "Syntax:\n"
     "  more [options] cdrom-files ...\n\n"
     "Description:\n"
     "  Same command as \"cat\" but use \"less\" or \"more\" as pager.\n\n"
     "Options:\n"
     "  -v : verbose, display the name of files\n"},
    {"pwd", cmd_pwd,
     "Syntax:\n"
     "  pwd [options]\n\n"
     "Description:\n"
     "  Print the current working directory on the cdrom.\n\n"
     "Options:\n"
     "  -f : display the FID of the directory.\n"},
    {"quit", cmd_exit,
     "Syntax:\n"
     "  quit\n\n"
     "Description:\n"
     "  Exit the application.\n"},
    {"stat", cmd_stat,
     "Syntax:\n"
     "  stat\n\n"
     "Description:\n"
     "  Display usage statistics.\n"},
    {"!", NULL,
     "Syntax:\n"
     "  !shell-command\n\n"
     "Description:\n"
     "  Execute a shell command.\n"},
    {NULL, NULL, NULL}
};


/*******************************************************************************
 *
 *  Action routine for the pwd command
 *
 *******************************************************************************
 */

static int cmd_pwd (int argc, char *argv[], vmscd_state_t *state)
{
    int c, version, show_fid = 0;
    char name [ODS2_MAX_PATH_SIZE];

    trace ("command: pwd\n");

    /* Decode the argument list. */

    optind = 0; /* reinitialize getopt */

    while ((c = getopt (argc, argv, "f")) > 0) {
        switch (c) {
        case 'f':
            show_fid = 1;
            break;
        default:
            error (0, "invalid option -%c\n", optopt);
        case '?':
        case ':':
            error (0, "try \"help %s\"\n", argv[0]);
            return 0;
        }
    }

    if (optind < argc) {
        error (0, "no parameter allowed\n");
        return 0;
    }

    /* Print the working directory */

    if (get_file_path (0, state->vol, &state->curwd->fid, name,
                       sizeof (name), &version) == 0) {

        if (show_fid)
            fprintf (state->out, "(%d,%d,%d)  ", state->curwd->fid.number,
                     state->curwd->fid.sequence, state->curwd->fid.volume);

        fprintf (state->out, "%s\n", name);
    }

    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the cd command
 *
 *******************************************************************************
 */

static int cmd_cd (int argc, char *argv[], vmscd_state_t *state)
{
    fid_t fid;
    file_t *dir;
    int c, version, silent = 0;
    char name [ODS2_MAX_PATH_SIZE];

    trace ("command: cd\n");

    /* Decode the argument list. */

    optind = 0; /* reinitialize getopt */

    while ((c = getopt (argc, argv, "n")) > 0) {
        switch (c) {
        case 'n':
            silent = 1;
            break;
        default:
            error (0, "invalid option -%c\n", optopt);
        case '?':
        case ':':
            error (0, "try \"help %s\"\n", argv[0]);
            return 0;
        }
    }

    if (optind >= argc) {
        /* Without argument, move to the master file directory */
        fid = mfd_fid;
    }
    else if (optind == argc - 1) {
        /* Locate the specified directory. If the path starts with a "/", */
        /* use the root as starting point. */
        if (locate_file (0, state->vol, argv[optind],
                         argv[optind][0] == '/' ? &mfd_fid : &state->curwd->fid,
                         &fid, name, sizeof (name), &version) < 0)
            return 0;
    }
    else {
        error (0, "invalid command syntax, try \"help %s\"\n", argv[0]);
        return 0;
    }

    /* Open the file and checks that it is a directory */
        
    if ((dir = open_file (0, state->vol, &fid)) == NULL)
        return 0;

    if (!is_a_directory (dir)) {
        error (0, "%s is not a directory\n", name);
        close_file (dir);
        return 0;
    }

    /* This is now ok, establish it as new working directory */

    close_file (state->curwd);
    state->curwd = dir;

    /* Display the new working directory */

    if (!silent && get_file_path (0, state->vol, &state->curwd->fid, name,
                                  sizeof (name), &version) == 0)
        fprintf (state->out, "New working directory is %s\n", name);

    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the lcd command.
 *
 *******************************************************************************
 */

static int cmd_lcd (int argc, char *argv[], vmscd_state_t *state)
{
    int c, silent = 0;
    const char *path;
    char line [2048];

    trace ("command: lcd\n");

    /* Decode the argument list. */

    optind = 0; /* reinitialize getopt */

    while ((c = getopt (argc, argv, "n")) > 0) {
        switch (c) {
        case 'n':
            silent = 1;
            break;
        default:
            error (0, "invalid option -%c\n", optopt);
        case '?':
        case ':':
            error (0, "try \"help %s\"\n", argv[0]);
            return 0;
        }
    }

    if (optind >= argc) {
        /* Without argument, move to the home directory */
        if ((path = getenv ("HOME")) == NULL) {
            error (0, "environment variable $HOME not set\n");
            return 0;
        }
    }
    else if (optind == argc - 1) {
        path = argv[optind];
    }
    else {
        error (0, "invalid command syntax, try \"help %s\"\n", argv[0]);
        return 0;
    }

    /* This is now ok, establish it as new working directory */

    if (chdir (path) < 0) {
        error (ERR_ERRNO, "cannot chdir to %s\n", path);
        return 0;
    }

    /* Display the new working directory */

    if (!silent) {
        if (getcwd (line, sizeof (line)) == NULL)
            error (ERR_ERRNO, "cannot get resulting working directory\n");
        else
            fprintf (state->out, "New working directory is %s\n", line);
    }

    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the ls command
 *
 *******************************************************************************
 */

static int cmd_ls (int argc, char *argv[], vmscd_state_t *state)
{
    int c, arg;
    list_opt_t opt;

    trace ("command: ls\n");
    memset (&opt, 0, sizeof (opt));

    /* Decode the argument list. */

    optind = 0; /* reinitialize getopt */

    while ((c = getopt (argc, argv, "adflrv")) > 0) {
        switch (c) {
        case 'a':
            opt.list_all = 1;
            break;
        case 'd':
            opt.list_nodir = 1;
            break;
        case 'f':
            opt.list_fid = 1;
            break;
        case 'l':
            opt.long_list = 1;
            break;
        case 'r':
            opt.recurse = 1;
            break;
        case 'v':
            opt.list_version = 1;
            break;
        default:
            error (0, "invalid option -%c\n", optopt);
        case '?':
        case ':':
            error (0, "try \"help %s\"\n", argv[0]);
            return 0;
        }
    }

    opt.out = state->out;
    opt.width = terminal_columns (fileno (opt.out));

    /* List all arguments. Without argument, list current directory. */

    if (optind >= argc)
        list_files (0, &opt, state->vol, &state->curwd->fid, "");
    else {
        for (arg = optind; arg < argc; arg++)
            list_files (0, &opt, state->vol, &state->curwd->fid, argv[arg]);
    }

    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the cp command
 *
 *******************************************************************************
 */

static int cmd_cp (int argc, char *argv[], vmscd_state_t *state)
{
    int c, arg;
    copy_opt_t opt;
    struct stat st;

    trace ("command: cp\n");
    memset (&opt, 0, sizeof (opt));

    /* Decode the argument list. */

    optind = 0; /* reinitialize getopt */

    while ((c = getopt (argc, argv, "adnrtuv")) > 0) {
        switch (c) {
        case 'a':
            opt.all_files = 1;
            break;
        case 'd':
            opt.create_sfa = 1;
            break;
        case 'n':
            opt.text_only = 1;
            break;
        case 'r':
            opt.recurse = 1;
            break;
        case 't':
            opt.text_mode = 1;
            break;
        case 'u':
            opt.create_ucx = 1;
            break;
        case 'v':
            opt.verbose = 1;
            break;
        default:
            error (0, "invalid option -%c\n", optopt);
        case '?':
        case ':':
            error (0, "try \"help %s\"\n", argv[0]);
            return 0;
        }
    }

    if (optind + 1 >= argc) {
        error (0, "not enough parameters, try \"help %s\"\n", argv[0]);
        return 0;
    }

    opt.out = state->out;

    /* Check if destination exists and is a directory */

    opt.dest_dir = stat (argv[argc-1], &st) == 0 && S_ISDIR (st.st_mode);

    /* If more than one source paths are given, the destination must be */
    /* a directory. */

    if (optind < argc - 2 && !opt.dest_dir) {
        error (0, "destination %s must be a directory to copy multiple files\n",
               argv[argc-1]);
        return 0;
    }

    /* Copy all arguments. */

    for (arg = optind; arg < argc - 1; arg++)
        copy_files (0, &opt, state->vol, &state->curwd->fid,
                    argv[arg], argv[argc-1]);

    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the cat and more commands
 *
 *******************************************************************************
 */

static int cmd_cat (int argc, char *argv[], vmscd_state_t *state)
{
    int c, arg;
    copy_opt_t opt;
    char *pager = NULL;
    FILE *pipe_out = NULL;
    void *saved_sigpipe = SIG_ERR;

    trace ("command: cat\n");
    memset (&opt, 0, sizeof (opt));

    /* Decode the argument list. */

    optind = 0; /* reinitialize getopt */

    while ((c = getopt (argc, argv, "v")) > 0) {
        switch (c) {
        case 'v':
            opt.verbose = 1;
            break;
        default:
            error (0, "invalid option -%c\n", optopt);
        case '?':
        case ':':
            error (0, "try \"help %s\"\n", argv[0]);
            return 0;
        }
    }

    if (optind >= argc) {
        error (0, "not enough parameters, try \"help %s\"\n", argv[0]);
        return 0;
    }

    opt.out = state->out;
    opt.text_only = 1;
    opt.text_mode = 1;

    /* A pager is used for the "more" command, not for "cat". */
    /* Use a pager only if the current output is a terminal. */

    if (strcmp (argv[0], "more") == 0 && isatty (fileno (state->out))) {

        /* Find an appropriate pager */

        if ((pager = find_exec ("less")) == NULL &&
            (pager = find_exec ("more")) == NULL) {
            error (0, "cannot find a pager program (\"less\" or \"more\")\n");
            return 0;
        }

        /* Create a pipe to the pager */

        if ((pipe_out = popen (pager, "w")) == NULL) {
            error (ERR_ERRNO, "cannot create pipe to %s\n", pager);
            free (pager);
            return 0;
        }

        /* A piped command has been successfully created */
        opt.out = pipe_out;
        /* Ignore "broken pipe" signals in case the shell command */
        /* fails. In this case, we let all fprintf's silently fail. */
        saved_sigpipe = signal (SIGPIPE, SIG_IGN);
    }

    /* Display all arguments. */

    for (arg = optind; arg < argc; arg++)
        copy_files (0, &opt, state->vol, &state->curwd->fid, argv[arg], NULL);

    /* Close redirected output */

    if (pipe_out != NULL)
        pclose (pipe_out);
    if (saved_sigpipe != SIG_ERR)
        signal (SIGPIPE, saved_sigpipe);
    if (pager != NULL)
        free (pager);

    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the dump command
 *
 *******************************************************************************
 */

static int cmd_dump (int argc, char *argv[], vmscd_state_t *state)
{
    int c, len, start = -1, count = -1, version;
    file_t *file;
    fid_t fid;
    char data [ODS2_BLOCK_SIZE], name [ODS2_MAX_NAME_SIZE];

    trace ("command: dump\n");

    /* Decode the argument list. */

    optind = 0; /* reinitialize getopt */

    while ((c = getopt (argc, argv, "c:s:")) > 0) {
        switch (c) {
        case 'c':
            count = atoi (optarg);
            break;
        case 's':
            start = atoi (optarg);
            break;
        default:
            error (0, "invalid option -%c\n", optopt);
        case '?':
        case ':':
            error (0, "try \"help %s\"\n", argv[0]);
            return 0;
        }
    }

    if (optind < argc - 1) {
        error (0, "too many parameters, try \"help %s\"\n", argv[0]);
        return 0;
    }

    if (optind >= argc) {

        /* No parameter given, we must dump LBN on the disk. */
        /* In this case, both -s and -c are required. */

        if (start < 0 || count < 0) {
            error (0, "both starting LBN and block count must be specified\n");
            return 0;
        }

        /* Dump LBN from the disk */

        while (count-- > 0) {
            putc ('\n', state->out);
            len = fprintf (state->out, "LBN %d (CD label: %s, device: %s)",
                           start, state->vol->volname, state->device);
            putc ('\n', state->out);
            while (len-- > 0)
                putc ('-', state->out);
            putc ('\n', state->out);
            if (read_lbn (0, state->cache, start++, 1, data, LBN_DATA) < 0)
                break;
            memdump (state->out, data, sizeof (data), 0);
        }
    }
    else {

        /* One file was specified, we should dump VBN of this file */

        if (locate_file (0, state->vol, argv [optind], &state->curwd->fid,
                         &fid, name, sizeof (name), &version) < 0 ||
            (file = open_file (0, state->vol, &fid)) == NULL)
            return 0;

        /* Compute default values */

        if (start < 1)
            start = 1;
        if (start > file->last_data_vbn)
            count = 0;
        if (count < 0 || count > file->last_data_vbn - start + 1)
            count = file->last_data_vbn - start + 1;

        /* Dump VBN from the file */

        while (count-- > 0) {
            putc ('\n', state->out);
            len = fprintf (state->out, "VBN %d (File: %s, CD label: %s)",
                           start, file->name, state->vol->volname);
            if (start == file->last_data_vbn)
                len += fprintf (state->out, ", EOF at offset %04X",
                                file->used_size - (start-1) * ODS2_BLOCK_SIZE);
            putc ('\n', state->out);
            while (len-- > 0)
                putc ('-', state->out);
            putc ('\n', state->out);
            if (read_vbn (0, file, start++, 1, data) < 0)
                break;
            memdump (state->out, data, sizeof (data), 0);
        }

        close_file (file);
    }

    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the help command.
 *
 *******************************************************************************
 */

static int cmd_help (int argc, char *argv[], vmscd_state_t *state)
{
    int c, arg, width, n, first = 1, help_all = 0;
    const cmd_desc_t *desc;
    char *sepline;

    trace ("command: help\n");

    /* Decode the argument list. */

    optind = 0; /* reinitialize getopt */

    while ((c = getopt (argc, argv, "a")) > 0) {
        switch (c) {
        case 'a':
            help_all = 1;
            break;
        default:
            error (0, "invalid option -%c\n", optopt);
        case '?':
        case ':':
            error (0, "try \"help %s\"\n", argv[0]);
            return 0;
        }
    }

    /* Build a separator line */

    if ((width = terminal_columns (fileno (state->out))) == 0)
        width = DEFAULT_SCREEN_WIDTH;
    sepline = utmalloc (width + 1);
    for (n = 0; n < width; sepline [n++] = '-');
    sepline [width] = 0;

    /* Display the requested help */

    if (help_all) {

        /* Print help for all commands */

        for (desc = command_table; desc->name != NULL; desc++) {
            if (first)
                first = 0;
            else
                fprintf (state->out, "%s\n", sepline);
            fprintf (state->out, "\n%s\n", desc->help);
        }
    }
    else if (optind >= argc) {

        /* Without arguments, print the list of commands */

        fprintf (state->out, "Available commands:\n");
        for (desc = command_table; desc->name != NULL; desc++)
            fprintf (state->out, "%s%s",
                     desc == command_table ? "  " : ", " , desc->name);
        fprintf (state->out, "\nEnter \"help command-name\" for details\n");
    }
    else {

        /* With arguments, print help for the specified commands */

        for (arg = optind; arg < argc; arg++) {
            for (desc = command_table; desc->name != NULL; desc++) {
                if (strcmp (argv [arg], desc->name) == 0) {
                    if (first)
                        first = 0;
                    else
                        fprintf (state->out, "%s\n", sepline);
                    fprintf (state->out, "\n%s\n", desc->help);
                    break;
               }
            }
            if (desc->name == NULL)
                error (0, "unknown command \"%s\"\n", argv [arg]);
        }
    }

    free (sepline);
    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the stat command.
 *
 *******************************************************************************
 */

static int cmd_stat (int argc, char *argv[], vmscd_state_t *state)
{
    trace ("command: stat\n");

    if (argc > 1)
        error (0, "invalid command syntax, try \"help %s\"\n", argv[0]);
    else
        display_cache_stat (state->cache);

    return 0;
}


/*******************************************************************************
 *
 *  Action routine for the quit and exit commands.
 *  Simply return -1 to indicate "end of session".
 *
 *******************************************************************************
 */

static int cmd_exit (int argc, char *argv[], vmscd_state_t *state)
{
    trace ("command: exit\n");
    return -1;
}


/*******************************************************************************
 *
 *  This routine dispatches a command.
 *  Return the returned value of the command action routine or zero
 *  if there is none.
 *
 *******************************************************************************
 */

static int dispatch_command (int argc, char *argv[], vmscd_state_t *state)
{
    int n;
    const cmd_desc_t *desc;

    for (n = 0; n < argc; n++)
        trace ("arg [%d] = \"%s\"\n", n, argv [n]);

    for (desc = command_table; desc->name != NULL; desc++) {
        if (strcmp (argv [0], desc->name) == 0)
            return desc->action == NULL ? 0 : desc->action (argc, argv, state);
    }

    error (0, "unknown command \"%s\", try command \"help\"\n", argv [0]);
    return 0;
}


/*******************************************************************************
 *
 *  This routine analyzes a command line and dispatches it.
 *  Return -1 to indicate end of session, 0 otherwise.
 *
 *  If the first non-blank character is:
 *    '#' : comment line, ignored.
 *    '!' : shell command, executed.
 *
 *  If a '|' is encountered, this marks the end of the vmscd command.
 *  The vmscd command output is piped into the subsequent command shell.
 *
 *  If a '>' or a '>>' is encountered, this is a redirection of the output
 *  on a file.
 *
 *******************************************************************************
 */

static int analyze_command (char *line, vmscd_state_t *state)
{
    int quoted, arg, status = 0, next_is_redir = 0;
    char *p = NULL, *a = NULL, *pipe_cmd = NULL;
    char *redir_out = NULL, *redir_mode = NULL;
    FILE *pipe_out = NULL, *file_out = NULL, *saved_out;
    char *largv [MAX_COMMAND_SIZE];
    void *saved_sigpipe = SIG_ERR;

    /* Loop on arguments */

    for (arg = 0, p = line; *p != 0 && pipe_cmd == NULL; ) {

        /* Skip leading spaces */

        while (*p != 0 && isspace (*p))
            p++;
        if (*p == 0)
            break;

        /* If the first non blank character is a '#', this is a comment line. */
        /* Similarly, if it is a '!', this is a shell command. */

        if ((*p == '#' || *p == '!') && arg == 0)
            break;

        /* Locate the argument */

        a = NULL;   /* pointer into the current argument */
        quoted = 0; /* boolean: inside a quoted "" string */

        for ( ; *p != 0 && (quoted || !isspace (*p)); p++) {
            if (*p == '"')
                quoted = !quoted;
            else if (*p == '>' && !quoted) {
                /* End of current argument (if there is one). */
                /* Start name of redirected output file. */
                if (redir_out != NULL) {
                    error (0, "double output redirection\n");
                    return 0;
                }
                if (p[1] != '>')
                    redir_mode = "w";
                else {
                    redir_mode = "a";
                    p++;
                }
                if (a == NULL)
                    p++;
                next_is_redir = 1;
                break;
            }
            else if (*p == '|' && !quoted) {
                /* End of vmscd command, start of pipe command */
                pipe_cmd = p + 1;
                break;
            }
            else {
                /* Non-quote character, copy "as is" */
                if (a == NULL) {
                    if (next_is_redir)
                        redir_out = a = p;  /* Start of redirected output */
                    else
                        largv [arg++] = a = p;  /* Start of argument */
                    next_is_redir = 0;
                }
                *a++ = *p;
            }
        }

        /* If an argument was found, terminate it */

        if (a != NULL) {
            if (*p != 0)
                p++;
            *a = 0;
        }
    }

    /* Now execute the command */

    if (arg == 0 && *p == '!') {

        /* Execute a shell command instead of a vmscd command */

        trace ("executing shell command \"%s\"\n", p + 1);

        if (system (p + 1) < 0)
            error (ERR_ERRNO, "cannot execute shell command\n");

    }
    else if (arg > 0) {

        /* Execute a vmscd command */

        saved_out = state->out;

        /* Check for a redirected output for vmscd command output */

        if (redir_out != NULL && pipe_cmd != NULL) {
            error (0, "cannot specify both redirected output and pipe\n");
            return 0;
        }

        if (redir_out != NULL) {
            trace ("redirect output to \"%s\", mode = \"%s\"\n",
                redir_out, redir_mode);
            if ((file_out = fopen (redir_out, redir_mode)) == NULL) {
                error (ERR_ERRNO, "cannot create output file %s\n", redir_out);
                return 0;
            }
            state->out = file_out;
        }

        /* Check for a piped shell command for vmscd command output */

        else if (pipe_cmd != NULL) {
            while (isspace (*pipe_cmd))
                pipe_cmd++;
            trace ("piped command \"%s\"\n", pipe_cmd);
            if (*pipe_cmd == 0)
                pipe_cmd = NULL; /* empty piped command => no command */
            else if ((pipe_out = popen (pipe_cmd, "w")) == NULL) {
                error (ERR_ERRNO, "cannot create pipe to shell command\n");
                return 0;
            }
            else {
                /* A piped command has been successfully created */
                state->out = pipe_out;
                /* Ignore "broken pipe" signals in case the shell command */
                /* fails. In this case, we let all fprintf's silently fail. */
                saved_sigpipe = signal (SIGPIPE, SIG_IGN);
            }
        }

        /* Execute the vmscd command */

        status = dispatch_command (arg, largv, state);

        /* Close redirected output */

        if (file_out != NULL)
            fclose (file_out);
        if (pipe_out != NULL)
            pclose (pipe_out);
        if (saved_sigpipe != SIG_ERR)
            signal (SIGPIPE, saved_sigpipe);

        state->out = saved_out;
    }

    return status;
}


/*******************************************************************************
 *
 *  Application entry point
 *
 *******************************************************************************
 */

int main (int argc, char *argv[])
{
    vmscd_state_t state;
    int opt, fd, cache_size;
    char line [MAX_COMMAND_SIZE];

    /*
     *  Default values for command line options.
     */

    const char* cdrom = NULL;
    int display_initial_info = 1;
    int display_version = 0;
    int display_version_alone = 0;
    int display_stat = 0;
    onoff_t debug = OFF;
    lbn_t data_cache_size = DEFAULT_DATA_CACHE_SIZE;
    lbn_t meta_cache_size = DEFAULT_META_CACHE_SIZE;

    /*
     *  Decode the command line options, before the internal command.
     *  Use "man vmscd" for option description.
     *  Also recognize the following undocumented options:
     *    -D : debug mode (add trace messages).
     *    -V : display the version number alone (used to create kits)
     */

    while ((opt = getopt (argc, argv, "+d:c:m:nsvDV")) > 0) {
        switch (opt) {
        case 'd':
            cdrom = optarg;
            break;
        case 'c':
            data_cache_size = atoi (optarg);
            break;
        case 'm':
            meta_cache_size = atoi (optarg);
            break;
        case 'n':
            display_initial_info = 0;
            break;
        case 's':
            display_stat = 1;
            break;
        case 'v':
            display_version = 1;
            break;
        case 'D':
            debug = ON;
            break;
        case 'V':
            display_version_alone = 1;
            break;
        default:
            usage (argv[0]);
        }
    }

    /*
     *  If "-v" or "-V", simply display the version and exit.
     */

    if (display_version_alone) {
        printf ("%d.%d\n", VMSCD_MAJOR_VERSION, VMSCD_MINOR_VERSION);
        exit (EXIT_SUCCESS);
    }

    if (display_version) {
        printf ("OpenVMS CD-ROM Reader (vmscd) V%d.%d\n",
                VMSCD_MAJOR_VERSION, VMSCD_MINOR_VERSION);
        printf ("(build %s %s)\n", __DATE__, __TIME__);
        exit (EXIT_SUCCESS);
    }

    /*
     *  Initialize the utils module
     */

    init_utils (argv [0], debug, MAX_COMMAND_RECALL);

    /*
     *  Get the CD-ROM device name. Try, in order:
     *  - option "-d cdrom" if present
     *  - environment variable VMSCDROM if exists
     *  - /dev/cdrom
     */

    if (cdrom == NULL && (cdrom = getenv (CDROM_DEVICE_VAR)) == NULL)
        cdrom = DEFAULT_CDROM_DEVICE;

    /*
     *  Open the cdrom device and initialize the cache.
     */

    if ((fd = open (cdrom, 0)) < 0)
        error (ERR_ERRNO | ERR_EXIT, "cannot open %s\n", cdrom);

    state.device = cdrom;
    state.cache = init_cache (ERR_EXIT, fd, data_cache_size, meta_cache_size);
    state.vol = open_volume (ERR_EXIT, state.cache);
    state.curwd = open_file (ERR_EXIT, state.vol, &mfd_fid);
    state.out = stdout;

    /*
     *  Display the CD-ROM label and various info (unless -n is specified).
     *  Do this only in interactive mode.
     */

    if (display_initial_info && optind >= argc) {

        /* Display version of the utility */

        printf ("\nOpenVMS CD-ROM Reader (vmscd) V%d.%d\n",
                VMSCD_MAJOR_VERSION, VMSCD_MINOR_VERSION);

        printf ("Volume name in %s: %s\n", cdrom, state.vol->volname);

        /* Display the cache size */

        if ((cache_size = max_cache_size (state.cache)) == 0)
            printf ("Cache is disabled (implicit usage of UNIX cache)\n");
        else
            printf ("Cache size: %.1f MB\n", (double)cache_size / (1024*1024));

        printf ("\n");
    }

    /*
     *  If an internal command is specified on the command line, execute it.
     *  Otherwise, enter an interactive session.
     */

    if (optind < argc) {
        /* Execute one command from the command line */
        dispatch_command (argc - optind, argv + optind, &state);
    }
    else {
        /* Execute an interactive session */
        while (input (COMMAND_PROMPT, line, sizeof (line)) != NULL &&
               analyze_command (line, &state) >= 0);
    }

    /*
     *  Display usage statistics if required.
     */

    if (display_stat)
        display_cache_stat (state.cache);

    /*
     *  Free all resources.
     */

    close_file (state.curwd);
    close_volume (state.vol);
    delete_cache (state.cache);

    return EXIT_SUCCESS;
}
